/* This is -*- C -*- */
/* vim: set sw=2: */
/* $Id: guppi-memory.c,v 1.7 2001/01/16 23:36:40 trow Exp $ */

/*
 * guppi-memory.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include <gtk/gtkobject.h>

#include "guppi-exit.h"

#include "guppi-memory.h"

/* Undef object auditing for now, since it is all screwed up */
#undef OBJECT_AUDIT

#ifdef USE_MEMORY_DEBUGGING

#ifdef OBJECT_AUDIT

enum {
  OBJECT_NEW,
  OBJECT_REF,
  OBJECT_UNREF,
  OBJECT_SINK,
  OBJECT_FINALIZE
};

typedef struct _ObjectAudit ObjectAudit;
struct _ObjectAudit {
  gint time;
  gint action;
  gint ref_count;
  const gchar *file;
  gint line;
  union {
    ObjectAudit *prev;
    const gchar *type;
  } ptr;
};

static GHashTable *object_audit_table = NULL;
static gint object_audit_clock = 1;

static ObjectAudit *
object_audit_alloc (gint action, const gchar *file, gint line)
{
  ObjectAudit *oa = g_new0 (ObjectAudit, 1);

  oa->time = object_audit_clock;
  ++object_audit_clock;

  oa->action = action;
  oa->file = file;
  oa->line = line;

  return oa;
}

static void
object_audit_free (ObjectAudit *oa)
{
  if (oa == NULL)
    return;

  if (oa->action != OBJECT_NEW) {
    object_audit_free (oa->ptr.prev);
    oa->ptr.prev = NULL;
  }

  g_free (oa);
}

static void
object_audit_add (GtkObject *obj, ObjectAudit *oa)
{
  ObjectAudit *prev = NULL;

  if (object_audit_table)
    prev = (ObjectAudit *)g_hash_table_lookup (object_audit_table, obj);

  if (oa->action == OBJECT_NEW) {

    g_assert (prev == NULL);

    if (object_audit_table == NULL) 
      object_audit_table = g_hash_table_new (NULL, NULL);

    g_hash_table_insert (object_audit_table, obj, oa);

  } else {

    if (object_audit_table == NULL || prev == NULL) {

      object_audit_free (oa);

    } else {
      
      oa->ptr.prev = prev;
      g_hash_table_insert (object_audit_table, obj, oa);

    }
    
  }
}

static void
object_audit_remove_and_free (GtkObject *obj)
{
  ObjectAudit *tail;

  g_return_if_fail (obj);

  if (object_audit_table == NULL)
    return;

  tail = (ObjectAudit *)g_hash_table_lookup (object_audit_table, obj);
  g_return_if_fail (tail);

  g_hash_table_remove (object_audit_table, obj);

  object_audit_free (tail);
}

static void
object_audit_trace (GtkObject *obj)
{
  ObjectAudit *tail;
  ObjectAudit *iter;
  ObjectAudit *last;

  g_return_if_fail (obj);
  g_return_if_fail (object_audit_table);
  
  tail = (ObjectAudit *)g_hash_table_lookup (object_audit_table, obj);
  g_return_if_fail (tail);

  last = NULL;
  while (last != tail) {

    iter = tail;
    while (iter->action != OBJECT_NEW
	   && iter->ptr.prev != last)
      iter = iter->ptr.prev;

    switch (iter->action) {

    case OBJECT_NEW:
      g_print ("%p - %s\n", obj, iter->ptr.type);
      g_print ("NEW %2d %s:%d\n",
	       iter->ref_count, iter->file, iter->line);
      break;

    case OBJECT_REF:
      g_print ("REF %2d %s:%d\n",
	       iter->ref_count, iter->file, iter->line); 
      break;
      
    case OBJECT_UNREF:
      g_print ("UNR %2d %s:%d\n",
	       iter->ref_count, iter->file, iter->line); 
      break;

    case OBJECT_SINK:
      g_print ("SNK %2d %s:%d\n",
	       iter->ref_count, iter->file, iter->line); 
      break;

    case OBJECT_FINALIZE:
      g_print ("FIN %2d %s:%d\n", 
	       iter->ref_count, iter->file, iter->line); 
      break;

    default:
      g_assert_not_reached ();
      
    }

    last = iter;
  }
}

static void
object_audit_log_new (GtkObject *obj, const gchar *file, gint line)
{
  ObjectAudit *oa = object_audit_alloc (OBJECT_NEW, file, line);

  g_assert (obj);

  oa->ptr.type = gtk_type_name (GTK_OBJECT_TYPE (obj));
  oa->ref_count = obj->ref_count;

  object_audit_add (obj, oa);
}

static void
object_audit_log_ref (GtkObject *obj, const gchar *file, gint line)
{
  ObjectAudit *oa;

  oa = object_audit_alloc (OBJECT_REF, file, line);
  oa->ref_count = obj->ref_count;

  object_audit_add (obj, oa);
}

static void
object_audit_log_unref (GtkObject *obj, const gchar *file, gint line)
{
  ObjectAudit *oa;

  oa = object_audit_alloc (OBJECT_UNREF, file, line);
  oa->ref_count = obj->ref_count - 1;

  object_audit_add (obj, oa);

}

static void
object_audit_log_sink (GtkObject *obj, const gchar *file, gint line)
{
  ObjectAudit *oa;

  oa = object_audit_alloc (OBJECT_SINK, file, line);
  oa->ref_count = obj->ref_count;
  if (GTK_OBJECT_FLOATING (obj))
    --oa->ref_count;

  object_audit_add (obj, oa);
}

static void
object_audit_log_finalize (GtkObject *obj, const gchar *file, gint line)
{
  /* If we wanted to do profiling, we could record the finalization,
     remove the trace from the table and log it somewhere. */

  object_audit_remove_and_free (obj);
}

static void
divider (void)
{
  g_print ("-----------------------------------------------------------------------------\n");
}

static void
outstanding_iter_fn (gpointer key, gpointer value, gpointer flag_ptr)
{
  divider ();
  object_audit_trace (GTK_OBJECT (key));
  *(gboolean *)flag_ptr = TRUE;
}

void
guppi_trace_outstanding_objects (void)
{
  gboolean flag = FALSE;

  if (object_audit_table == NULL)
    return;

  g_hash_table_foreach (object_audit_table, outstanding_iter_fn, &flag);

  if (flag)
    divider ();
}

static void
object_audit_init (void)
{

}

#endif
  
/** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** ** **/

static gboolean memtrace_on = FALSE;
static FILE *out = NULL;

#define TRACE_FILE "/tmp/guppi.memtrace"

static void
memtrace (const gchar *file, gint line, gconstpointer ptr, const gchar *op,
	  const gchar *format_str, ...)
{
  va_list args;

  if (!memtrace_on && getenv ("GUPPI_MEMORY_TRACE"))
    memtrace_on = TRUE;

  if (!memtrace_on)
    return;

  if (out == NULL) {
    out = fopen (TRACE_FILE, "w");
    if (out) {
      fputs ("Writing memory trace to " TRACE_FILE "\n", stderr);
    } else {
      fputs ("Could not open " TRACE_FILE, stderr);
      memtrace_on = FALSE;
      return;
    }
  }

  fprintf (out, "%s | %d | %p | %s", file, line, ptr, op);
  
  if (format_str) {
    va_start (args, format_str);
    fputs (" | ", out);
    vfprintf (out, format_str, args);
    va_end (args);
  }
  fputc ('\n', out);
  fflush (out);
}

gpointer
_guppi_malloc (gsize N, const gchar *file, gint line)
{
  gpointer ptr = g_malloc (N);
  memtrace (file, line, ptr, "malloc", "%u", N);
  return ptr;
}

gpointer
_guppi_malloc0 (gsize N, const gchar *file, gint line)
{
  gpointer ptr = g_malloc0 (N);
  memtrace (file, line, ptr, "malloc", "%u", N);
  return ptr;
}

gpointer
_guppi_realloc (gpointer x, gsize N, const gchar *file, gint line)
{
  gpointer ptr = g_realloc (x, N);
  memtrace (file, line, ptr, "realloc", "%p | %u", x, N);
  return ptr;
}

void
_guppi_free (gpointer ptr, const gchar *file, gint line)
{
  if (ptr == NULL)
    return;
  memtrace (file, line, ptr, "free", NULL);
  g_free (ptr);
}

gpointer
_guppi_new (gsize size, gsize count, const gchar *type,
	    const gchar *file, gint line)
{
  gpointer ptr = g_malloc (size * count);
  if (ptr == NULL)
    return NULL;
  memtrace (file, line, ptr, "new",
	    "%s | %u | %u | %u", type, size, count, size * count); 
  return ptr;
}

gpointer
_guppi_new0 (gsize size, gsize count, const gchar *type,
	     const gchar *file, gint line)
{
  gpointer ptr = g_malloc0 (size * count);
  if (ptr == NULL)
    return NULL;
  memtrace (file, line, ptr, "new",
	    "%s | %u | %u | %u", type, size, count, size * count); 
  return ptr;
}

gpointer
_guppi_renew (gsize size, gpointer mem, gsize count, const gchar *type,
	      const gchar *file, gint line)
{
  gpointer ptr = g_realloc (mem, size*count);
  memtrace (file, line, ptr, "renew", "%p | %s | %u | %u | %u",
	    mem, type, size, count, size * count);
  return ptr;
}

void
_guppi_ref (gpointer ptr, const gchar *file, gint line)
{
  gint rc;

  if (ptr == NULL)
    return;
  
  g_return_if_fail (GTK_IS_OBJECT (ptr));

  rc = GTK_OBJECT (ptr)->ref_count;

  memtrace (file, line, ptr, "ref", 
	    "%d | %s", rc, gtk_type_name (GTK_OBJECT_TYPE (ptr)));

  gtk_object_ref (GTK_OBJECT (ptr));
  
#ifdef OBJECT_AUDIT
  object_audit_log_ref (GTK_OBJECT (ptr), file, line);
#endif
}

void
_guppi_unref (gpointer ptr, const gchar *file, gint line)
{
  gint rc;

  if (ptr == NULL)
    return;

  if (!GTK_IS_OBJECT (ptr)) {
    g_warning ("%s:%d _guppi_unref called on non-object (%p)",
	       file, line, ptr);
    return;
  }

  rc = GTK_OBJECT (ptr)->ref_count;
  if (rc <= 0) 
    g_error ("refcount <= 0 at %s: line %d", file, line);

  memtrace (file, line, ptr, "unref",
	    "%d | %s", rc, gtk_type_name (GTK_OBJECT_TYPE (ptr)));

#ifdef OBJECT_AUDIT
  object_audit_log_unref (GTK_OBJECT (ptr), file, line);
#endif

  gtk_object_unref (GTK_OBJECT (ptr));
}

void
_guppi_sink (gpointer ptr, const gchar *file, gint line)
{
  if (ptr == NULL)
    return;

  g_return_if_fail (GTK_IS_OBJECT (ptr));

  memtrace (file, line, ptr, "sink", 
	    "%s", gtk_type_name (GTK_OBJECT_TYPE (ptr)));

#ifdef OBJECT_AUDIT
  object_audit_log_sink (GTK_OBJECT (ptr), file, line);
#endif

  gtk_object_sink (GTK_OBJECT (ptr));
}

gpointer
_guppi_type_new (GtkType t, const gchar *file, gint line)
{
  gpointer ptr;

  g_return_val_if_fail (t, NULL);
  ptr = gtk_type_new (t);
  
  memtrace (file, line, ptr, "type_new", "%s", gtk_type_name (t));

#ifdef OBJECT_AUDIT
  object_audit_log_new (GTK_OBJECT (ptr), file, line);
#endif

  return ptr;
}

gchar *
_guppi_strdup (const gchar *s, const gchar *file, gint line)
{
  gchar *ptr;

  if (s == NULL)
    return NULL;
  ptr = g_strdup (s);
  memtrace (file, line, ptr, "malloc", "%s", ptr);

  return ptr;
}

gchar *
_guppi_strndup (const gchar *s, guint N, const gchar *file, gint line)
{
  gchar *ptr;

  if (s == NULL)
    return NULL;

  ptr = g_strndup (s, N);
  memtrace (file, line, ptr, "malloc", "%s", ptr);

  return ptr;
}

gchar *
_guppi_strdup_printf (const gchar *file, gint line, const gchar *format, ...)
{
  gchar *ptr;
  va_list args;

  if (format == NULL)
    return NULL;

  va_start (args, format);
  ptr  = g_strdup_vprintf (format, args);
  memtrace (file, line, ptr, "malloc", "%s", ptr);
  va_end (args);

  return ptr;
}

gchar *
_guppi_strdup_vprintf (const gchar *file, gint line, const gchar *format, 
		       va_list args)
{
  gchar *ptr;

  if (format == NULL)
    return NULL;

  ptr  = g_strdup_vprintf (format, args);
  memtrace (file, line, ptr, "malloc", "%s", ptr);

  return ptr;
}

void
_guppi_strfreev (gchar **s, const gchar *file, gint line)
{
  gint i=0;

  if (s == NULL)
    return;

  while (s[i]) {
    _guppi_free (s[i], file, line);
    ++i;
  }
  _guppi_free (s, file, line);
}


void
_guppi_outside_alloc (gconstpointer p, const gchar *file, gint line)
{
  memtrace (file, line, p, "outside", NULL);
}

void
_guppi_outside_object (gconstpointer obj, const gchar *file, gint line)
{
  if (obj == NULL)
    return;

  if (!GTK_IS_OBJECT (obj)) {
    g_warning ("guppi_outside_object called on non-object %p at %s, %d",
	       obj, file, line);
    return;
  }
  if (obj) {
    memtrace (file, line, obj, "outside_object", "%s",
	      gtk_type_name (GTK_OBJECT_TYPE (obj)));
  }
}

void
_guppi_permanent_alloc (gconstpointer p, const gchar *file, gint line)
{
  memtrace (file, line, p, "permanent", NULL);
}

void
_guppi_A_contains_B (gconstpointer A, gconstpointer B,
		     const gchar *file, gint line)
{
  memtrace (file, line, A, "contains", "%p", B);
}

void
_guppi_finalized (gconstpointer obj, const gchar *file, gint line)
{
  if (obj == NULL)
    return;

  if (!GTK_IS_OBJECT (obj)) {
    g_warning ("guppi_finalized called on non-object %p at %s, %d",
	       obj, file, line);
    return;
  }

#ifdef OBJECT_AUDIT
  object_audit_log_finalize (GTK_OBJECT (obj), file, line);
#endif

  memtrace (file, line, obj, "finalized",
	    "%s", gtk_type_name (GTK_OBJECT_TYPE (obj)));
}

void
guppi_ref_fn (gpointer x)
{
  _guppi_ref (x, "??", 0);
}

void 
guppi_unref_fn (gpointer x)
{
  _guppi_unref (x, "??", 0);
}

void
guppi_free_fn (gpointer x)
{
  _guppi_free (x, "??", 0);
}

#else

void
guppi_ref (gpointer x)
{
  if (x) {
    gtk_object_ref (GTK_OBJECT (x));
  }
}

void
guppi_unref (gpointer x)
{
  if (x) {
    g_assert (GTK_OBJECT (x)->ref_count > 0);
    gtk_object_unref (GTK_OBJECT (x));
  }
}

void
guppi_sink (gpointer x)
{
  if (x) {
    gtk_object_sink (GTK_OBJECT (x));
  }
}
#endif

void
guppi_memory_trace (gboolean x)
{
#ifdef USE_MEMORY_DEBUGGING
  memtrace_on = x;
#else
  g_warning ("The memory trace feature is not available.");
#endif
}


/* $Id: guppi-memory.c,v 1.7 2001/01/16 23:36:40 trow Exp $ */
