/* This is -*- C -*- */
/* $Id: guppi-element-view.c,v 1.36 2001/05/06 08:26:43 trow Exp $ */

/*
 * guppi-element-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <glade/glade.h>
#include <libart_lgpl/libart.h>
#include <guppi-useful.h>
#include <guppi-unique-id.h>
#include "guppi-style.h"
#include "guppi-shared-hash.h"
#include "guppi-element-view.h"
#include "guppi-canvas-item.h"
#include "guppi-element-print.h"

typedef struct _GuppiElementViewPrivate GuppiElementViewPrivate;
struct _GuppiElementViewPrivate {

  guppi_uniq_t id;

  GuppiElementState *state;
  GuppiGeometry *geometry;

  GuppiShared *x_view_interval;
  GuppiShared *y_view_interval;

  gboolean force_x_preferred;
  gboolean force_y_preferred;

  gint x_axis_marker_type;
  gint y_axis_marker_type;

  GuppiShared *x_axis_markers;
  GuppiShared *y_axis_markers;

  GuppiSharedHash *hash;

  gboolean hide;
  gboolean block_tools;

  gint freeze_count;
  gboolean pending_change;
  guint pending_delayed_change;

  guint pending_structure_change;
};

#define priv(x) ((GuppiElementViewPrivate*)((GUPPI_ELEMENT_VIEW(x))->opaque_internals))

static GtkObjectClass *parent_class = NULL;

enum {
  CHANGED,
  CHANGED_SIZE,
  CHANGED_STRUCTURE,
  LAST_SIGNAL
};

static guint view_signals[LAST_SIGNAL] = { 0 };

static void
guppi_element_view_finalize (GtkObject * obj)
{
  GuppiElementView *view = GUPPI_ELEMENT_VIEW (obj);
  GuppiElementViewPrivate *p = priv (view);

  guppi_finalized (obj);

  gtk_signal_disconnect_by_data (GTK_OBJECT (p->state), view);

  guppi_unref0 (p->state);
  guppi_unref0 (p->geometry);

  guppi_unref0 (p->x_view_interval);
  guppi_unref0 (p->y_view_interval);

  guppi_unref0 (p->x_axis_markers);
  guppi_unref0 (p->y_axis_markers);

  guppi_unref0 (p->hash);

  if (p->pending_delayed_change) {
    gtk_idle_remove (p->pending_delayed_change);
    p->pending_delayed_change = 0;
  }

  if (p->pending_structure_change) {
    gtk_idle_remove (p->pending_structure_change);
    p->pending_structure_change = 0;
  }

  guppi_free (view->opaque_internals);
  view->opaque_internals = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

/**************************************************************************/

static void
print (GuppiElementView * view, GnomePrintContext * pc)
{
  GuppiElementPrint *ep;
  GuppiGeometry *geom;
  double x0, y0, x1, y1;

  geom = guppi_element_view_geometry (view);

  x0 = guppi_geometry_left (geom);
  x1 = guppi_geometry_right (geom);

  y0 = guppi_geometry_bottom (geom);
  y1 = guppi_geometry_top (geom);

  ep = guppi_element_view_make_print (view, pc);

  if (ep) {
    guppi_element_print_set_bbox (ep, x0, y0, x1, y1);
    guppi_element_print_print (ep);
    guppi_unref (ep);
  }
}


/***************************************************************************/

static void
push_state_to_widget (GuppiElementView * view, GladeXML * xml)
{
  GuppiGeometry *geom;
  GtkWidget *w;
  gchar *txt;

  geom = guppi_element_view_geometry (view);

  w = glade_xml_get_widget (xml, "name_label");
  g_return_if_fail (w != NULL);
  gtk_label_set_text (GTK_LABEL (w),
		      guppi_element_state_label (guppi_element_view_state
						 (view)));

  w = glade_xml_get_widget (xml, "size_label");
  g_return_if_fail (w != NULL);
  txt = guppi_strdup_printf ("%.1g\" x %.1g\"",
			 guppi_pt2in (guppi_geometry_width (geom)),
			 guppi_pt2in (guppi_geometry_height (geom)));
  gtk_label_set_text (GTK_LABEL (w), txt);
  guppi_free (txt);

  w = glade_xml_get_widget (xml, "visibility_check");
  g_return_if_fail (w != NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (w),
				guppi_element_view_visible (view));
}

static void
view_toggle_cb (GtkWidget * w, GuppiElementView * view)
{
  gboolean x = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (w));
  guppi_element_view_set_visibility (view, x);
}

static void
destroy_cb (GtkObject * w, GladeXML * xml)
{
  gtk_signal_disconnect_by_func (GTK_OBJECT (gtk_object_get_user_data (w)),
				 push_state_to_widget, xml);
}

static GtkWidget *
top_view_node_cb (gpointer user_data)
{
  GuppiElementView *view;
  const gchar *glade_path;
  GladeXML *xml;
  GtkWidget *w;

  g_return_val_if_fail (user_data != NULL
			&& GUPPI_IS_ELEMENT_VIEW (user_data), NULL);

  view = GUPPI_ELEMENT_VIEW (user_data);

  glade_path = guppi_glade_path ("view-properties.glade");
  g_return_val_if_fail (glade_path != NULL, NULL);

  xml = glade_xml_new (glade_path, "view_properties");
  g_return_val_if_fail (xml != NULL, NULL);

  push_state_to_widget (view, xml);

  w = glade_xml_get_widget (xml, "visibility_check");
  gtk_signal_connect (GTK_OBJECT (w), "toggled",
		      GTK_SIGNAL_FUNC (view_toggle_cb), view);


  gtk_signal_connect (GTK_OBJECT (view), "changed",
		      GTK_SIGNAL_FUNC (push_state_to_widget), xml);
  gtk_signal_connect (GTK_OBJECT (view), "changed_size",
		      GTK_SIGNAL_FUNC (push_state_to_widget), xml);


  w = glade_xml_get_widget (xml, "view_properties");
  g_return_val_if_fail (w != NULL, NULL);
  gtk_object_set_user_data (GTK_OBJECT (w), view);

  gtk_signal_connect (GTK_OBJECT (w), "destroy",
		      GTK_SIGNAL_FUNC (destroy_cb), xml);



  return w;
}

static GuppiConfigItem *
assemble_config_tree (GuppiElementView * view,
		      GuppiConfigItem * state_tree,
		      GuppiConfigItem * view_tree)
{
  GuppiConfigItem *top = NULL;
  const gchar *label =
    guppi_element_state_label (guppi_element_view_state (view));

  if (state_tree || view_tree) {
    top = guppi_config_item_new (label, label, top_view_node_cb, view);
    guppi_config_item_append_below (top, state_tree);
    guppi_config_item_append_below (top, view_tree);
    guppi_unref (state_tree);
    guppi_unref (view_tree);
  }

  return top;
}

/**************************************************************************/

static void
guppi_element_view_class_init (GuppiElementViewClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->finalize = guppi_element_view_finalize;

  klass->print = print;
  klass->assemble_config_tree = assemble_config_tree;

  view_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiElementViewClass, changed),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  view_signals[CHANGED_SIZE] =
    gtk_signal_new ("changed_size",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiElementViewClass, changed_size),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  view_signals[CHANGED_STRUCTURE] =
    gtk_signal_new ("changed_structure",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiElementViewClass,
				       changed_structure),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, view_signals, LAST_SIGNAL);


}

static void
do_x_markers (GuppiElementView *view)
{
  GuppiElementViewClass *klass;
  GuppiElementViewPrivate *p;
  GuppiViewInterval *gvi;
  GuppiAxisMarkers *marks;
  double a, b;

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  p = priv (view);

  if ((marks = guppi_element_view_x_axis_markers (view))) {
    gvi = guppi_element_view_x_view_interval (view);
    guppi_view_interval_range (gvi, &a, &b);
    if (klass->update_x_axis_markers)
      klass->update_x_axis_markers (view, marks, a, b);
    else {
      guppi_axis_markers_populate_generic (marks, p->x_axis_marker_type, a, b);
    }
  }
}

static void
do_y_markers (GuppiElementView *view)
{
  GuppiElementViewClass *klass;
  GuppiElementViewPrivate *p;
  GuppiViewInterval *gvi;
  GuppiAxisMarkers *marks;
  double a, b;

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  p = priv (view);

  if ((marks = guppi_element_view_y_axis_markers (view))) {
    gvi = guppi_element_view_y_view_interval (view);
    guppi_view_interval_range (gvi, &a, &b);
    if (klass->update_y_axis_markers)
      klass->update_y_axis_markers (view, marks, a, b);
    else {
      guppi_axis_markers_populate_generic (marks, p->y_axis_marker_type, a, b);
    }

  }
}

static void
x_vi_changed (GuppiShared * sh_gvi, GuppiElementView * view)
{
  GuppiElementViewClass *klass;
  GuppiElementViewPrivate *p;

  p = priv (view);
  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

  if (p->force_x_preferred) {
    gtk_signal_handler_block_by_data (GTK_OBJECT (sh_gvi), view);
    guppi_element_view_set_preferred_x_view (view);
    gtk_signal_handler_unblock_by_data (GTK_OBJECT (sh_gvi), view);
  }
  
  if (klass->x_interval_changed)
    klass->x_interval_changed (view);

  if (p->x_axis_marker_type != GUPPI_AXIS_NONE)
    do_x_markers (view);	/* Update any x-axis markers */

  guppi_element_view_changed_delayed (view);
}

static void
y_vi_changed (GuppiShared * sh_gvi, GuppiElementView * view)
{
  GuppiElementViewClass *klass;
  GuppiElementViewPrivate *p;

  p = priv (view);
  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

  if (p->force_y_preferred) {
    gtk_signal_handler_block_by_data (GTK_OBJECT (sh_gvi), view);
    guppi_element_view_set_preferred_y_view (view);
    gtk_signal_handler_unblock_by_data (GTK_OBJECT (sh_gvi), view);
  }

  if (klass->y_interval_changed)
    klass->y_interval_changed (view);

  if (p->y_axis_marker_type != GUPPI_AXIS_NONE)
    do_y_markers (view);	/* Update any y-axis markers */

  guppi_element_view_changed_delayed (view);
}

static void
x_vi_preferred (GuppiViewInterval * vi, GuppiElementView * view)
{
  GuppiElementViewClass *klass;
  double a, b;

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  if (klass->preferred_x_range) {
    if (klass->preferred_x_range (view, &a, &b))
      guppi_view_interval_grow_to (vi, a, b);
  }
}

static void
y_vi_preferred (GuppiViewInterval * vi, GuppiElementView * view)
{
  GuppiElementViewClass *klass;
  double a, b;

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  if (klass->preferred_y_range) {
    if (klass->preferred_y_range (view, &a, &b))
      guppi_view_interval_grow_to (vi, a, b);
  }
}

static void
x_vi_changed_held (GuppiShared * sh_gvi, GuppiViewInterval * old_vi,
		   GuppiElementView * view)
{
  if (old_vi)
    gtk_signal_disconnect_by_func (GTK_OBJECT (old_vi),
				   GTK_SIGNAL_FUNC (x_vi_preferred), view);
  gtk_signal_connect (GTK_OBJECT (guppi_shared_get (sh_gvi)),
		      "preferred_range_request",
		      GTK_SIGNAL_FUNC (x_vi_preferred), view);
}

static void
y_vi_changed_held (GuppiShared * sh_gvi, GuppiViewInterval * old_vi,
		   GuppiElementView * view)
{
  if (old_vi)
    gtk_signal_disconnect_by_func (GTK_OBJECT (old_vi),
				   GTK_SIGNAL_FUNC (y_vi_preferred), view);
  gtk_signal_connect (GTK_OBJECT (guppi_shared_get (sh_gvi)),
		      "preferred_range_request",
		      GTK_SIGNAL_FUNC (y_vi_preferred), view);
}

static void
guppi_element_view_init (GuppiElementView * obj,
			 GuppiElementViewClass * klass)
{
  GuppiElementViewPrivate *p;
  GuppiViewInterval *x_gvi;
  GuppiViewInterval *y_gvi;

  p = guppi_new0 (GuppiElementViewPrivate, 1);
  obj->opaque_internals = p;

  p->id = guppi_unique_id ();

  p->geometry = guppi_geometry_new ();
  guppi_geometry_set_user_data (p->geometry, obj);

  p->hash = guppi_shared_hash_new ();

  p->x_axis_marker_type = GUPPI_AXIS_NONE;
  p->y_axis_marker_type = GUPPI_AXIS_NONE;

  p->x_view_interval = guppi_shared_view_interval ();
  x_gvi = guppi_view_interval_new ();

  if (klass->preferred_x_range)
    gtk_signal_connect_while_alive (GTK_OBJECT (p->x_view_interval),
				    "changed_held",
				    GTK_SIGNAL_FUNC (x_vi_changed_held),
				    obj, GTK_OBJECT (obj));

  guppi_shared_set (p->x_view_interval, GTK_OBJECT (x_gvi));
  guppi_unref (x_gvi);

  p->y_view_interval = guppi_shared_view_interval ();
  y_gvi = guppi_view_interval_new ();

  if (klass->preferred_y_range)
    gtk_signal_connect_while_alive (GTK_OBJECT (p->y_view_interval),
				    "changed_held",
				    GTK_SIGNAL_FUNC (y_vi_changed_held),
				    obj, GTK_OBJECT (obj));

  guppi_shared_set (p->y_view_interval, GTK_OBJECT (y_gvi));
  guppi_unref (y_gvi);


  p->x_axis_markers = guppi_shared_axis_markers ();
  p->y_axis_markers = guppi_shared_axis_markers ();

  /* Monitor viewport changes, convert them into view changes. */

  gtk_signal_connect (GTK_OBJECT (p->x_view_interval),
		      "changed", GTK_SIGNAL_FUNC (x_vi_changed), obj);

  gtk_signal_connect (GTK_OBJECT (p->y_view_interval),
		      "changed", GTK_SIGNAL_FUNC (y_vi_changed), obj);

  /* Monitor axis marker changes, convert to view changes. */

  gtk_signal_connect_object (GTK_OBJECT (p->x_axis_markers),
			     "changed",
			     GTK_SIGNAL_FUNC (guppi_element_view_changed_delayed),
			     GTK_OBJECT (obj));
  gtk_signal_connect_object (GTK_OBJECT (p->y_axis_markers),
			     "changed",
			     GTK_SIGNAL_FUNC (guppi_element_view_changed_delayed),
			     GTK_OBJECT (obj));

  /* Monitor geometry changes, convert to view changes. */

  gtk_signal_connect_object (GTK_OBJECT (p->geometry),
			     "changed_position",
			     GTK_SIGNAL_FUNC (guppi_element_view_changed_delayed),
			     GTK_OBJECT (obj));

  gtk_signal_connect_object (GTK_OBJECT (p->geometry),
			     "changed_size",
			     GTK_SIGNAL_FUNC (guppi_element_view_changed_delayed),
			     GTK_OBJECT (obj));
}

GtkType guppi_element_view_get_type (void)
{
  static GtkType guppi_element_view_type = 0;
  if (!guppi_element_view_type) {
    static const GtkTypeInfo guppi_element_view_info = {
      "GuppiElementView",
      sizeof (GuppiElementView),
      sizeof (GuppiElementViewClass),
      (GtkClassInitFunc) guppi_element_view_class_init,
      (GtkObjectInitFunc) guppi_element_view_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_element_view_type = gtk_type_unique (GTK_TYPE_OBJECT,
					       &guppi_element_view_info);
  }
  return guppi_element_view_type;
}

/***************************************************************************/

GuppiElementState *
guppi_element_view_state (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);
  return priv (view)->state;
}

GuppiGeometry *
guppi_element_view_geometry (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);
  return priv (view)->geometry;
}

guppi_uniq_t
guppi_element_view_unique_id (GuppiElementView *view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), 0);
  return priv (view)->id;
}

GuppiViewInterval *
guppi_element_view_x_view_interval (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);

  return
    GUPPI_VIEW_INTERVAL (guppi_shared_get (priv (view)->x_view_interval));
}

GuppiViewInterval *
guppi_element_view_y_view_interval (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);

  return
    GUPPI_VIEW_INTERVAL (guppi_shared_get (priv (view)->y_view_interval));
}

/***************************************************************************/

gint
guppi_element_view_x_axis_marker_type (GuppiElementView *view)
{
  GuppiElementViewPrivate *p;
  g_return_val_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view), GUPPI_AXIS_NONE);
  p = priv (view);
  return p->x_axis_marker_type;
}

gint
guppi_element_view_y_axis_marker_type (GuppiElementView *view)
{
  GuppiElementViewPrivate *p;
  g_return_val_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view), GUPPI_AXIS_NONE);
  p = priv (view);
  return p->y_axis_marker_type;
}

void
guppi_element_view_set_x_axis_marker_type (GuppiElementView *view, gint code)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);
  
  p->x_axis_marker_type = code;

  if (code != GUPPI_AXIS_NONE) {
    if (guppi_shared_get (p->x_axis_markers) == NULL) {
      GuppiAxisMarkers *gam = guppi_axis_markers_new ();
      guppi_shared_set (p->x_axis_markers, GTK_OBJECT (gam));
      guppi_unref (gam);
    }
    do_x_markers (view);
  }
}

void
guppi_element_view_set_y_axis_marker_type (GuppiElementView *view, gint code)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);
  
  p->y_axis_marker_type = code;

  if (code != GUPPI_AXIS_NONE) {
    if (guppi_shared_get (p->y_axis_markers) == NULL) {
      GuppiAxisMarkers *gam = guppi_axis_markers_new ();
      guppi_shared_set (p->y_axis_markers, GTK_OBJECT (gam));
      guppi_unref (gam);
    }
    do_y_markers (view);
  }
}


GuppiAxisMarkers *
guppi_element_view_x_axis_markers (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);

  return GUPPI_AXIS_MARKERS0 (guppi_shared_get (priv (view)->x_axis_markers));
}

GuppiAxisMarkers *
guppi_element_view_y_axis_markers (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);

  return GUPPI_AXIS_MARKERS0 (guppi_shared_get (priv (view)->y_axis_markers));
}

/***************************************************************************/

static void
view_state_change_cb (GuppiElementState * state, GuppiElementView * view)
{
  GuppiElementViewClass *klass;
  GuppiElementViewPrivate *p;

  g_return_if_fail (state != NULL);
  g_return_if_fail (view != NULL);

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  p = priv (view);

  if (p->x_axis_marker_type != GUPPI_AXIS_NONE)
    do_x_markers (view);

  if (p->y_axis_marker_type != GUPPI_AXIS_NONE)
    do_y_markers (view);

  if (klass->state_changed)
    (klass->state_changed) (view);
}

void
guppi_element_view_set_state (GuppiElementView * view,
			      GuppiElementState * state)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  g_return_if_fail (state != NULL && GUPPI_IS_ELEMENT_STATE (state));

  p = priv (view);

  g_return_if_fail (p->state == NULL);

  p->state = state;
  guppi_ref (p->state);

  /* I think that this is unnecessary. */
  /* guppi_shared_hash_chain(view->hash, state->hash); */

  gtk_signal_connect (GTK_OBJECT (state),
		      "changed",
		      GTK_SIGNAL_FUNC (view_state_change_cb), view);

  /* Forward the state's CHANGED_SIZE signals to the view. */
  gtk_signal_connect_object (GTK_OBJECT (state),
			     "changed_size",
			     GTK_SIGNAL_FUNC
			     (guppi_element_view_changed_size),
			     GTK_OBJECT (view));
}

void
guppi_element_view_get_bbox_vp (GuppiElementView * view,
				double *x0, double *y0,
				double *x1, double *y1)
{
  GuppiViewInterval *view_x;
  GuppiViewInterval *view_y;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  view_x = guppi_element_view_x_view_interval (view);
  view_y = guppi_element_view_y_view_interval (view);

  if (x0)
    *x0 = view_x->t0;
  if (x1)
    *x1 = view_x->t1;

  if (y0)
    *y0 = view_y->t0;
  if (y1)
    *y1 = view_y->t1;
}

void
guppi_element_view_get_bbox_pt (GuppiElementView * view,
				double *x0, double *y0,
				double *x1, double *y1)
{
  GuppiGeometry *geom;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  geom = guppi_element_view_geometry (view);

  if (x0)
    *x0 = guppi_geometry_left (geom);
  if (x1)
    *x1 = guppi_geometry_right (geom);
  if (y0)
    *y0 = guppi_geometry_bottom (geom);
  if (y1)
    *y1 = guppi_geometry_top (geom);
}

void
guppi_element_view_vp2pt (GuppiElementView * view,
			  double vp_x, double vp_y, double *pt_x,
			  double *pt_y)
{
  GuppiGeometry *geom;
  GuppiViewInterval *vix;
  GuppiViewInterval *viy;
  double tx, ty;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  geom = guppi_element_view_geometry (view);

  vix = guppi_element_view_x_view_interval (view);
  viy = guppi_element_view_y_view_interval (view);

  tx = guppi_view_interval_conv (vix, vp_x);
  ty = guppi_view_interval_conv (viy, vp_y);

  guppi_geometry_unconv (geom, tx, ty, pt_x, pt_y);
}

void
guppi_element_view_pt2vp (GuppiElementView * view,
			  double pt_x, double pt_y, double *vp_x,
			  double *vp_y)
{
  GuppiGeometry *geom;
  double tx, ty;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  geom = guppi_element_view_geometry (view);

  guppi_geometry_conv (geom, pt_x, pt_y, &tx, &ty);

  if (vp_x) {
    GuppiViewInterval *vi = guppi_element_view_x_view_interval (view);
    *vp_x = guppi_view_interval_unconv (vi, tx);
  }

  if (vp_y) {
    GuppiViewInterval *vi = guppi_element_view_y_view_interval (view);
    *vp_y = guppi_view_interval_unconv (vi, ty);
  }
}

/***************************************************************************/

static void
state_change_cb (GuppiElementState * state, GuppiCanvasItem * item)
{
  GuppiCanvasItemClass *klass;

  g_return_if_fail (state != NULL);
  g_return_if_fail (item != NULL);

  klass = GUPPI_CANVAS_ITEM_CLASS (GTK_OBJECT (item)->klass);
  if (klass->state_changed)
    (klass->state_changed) (item);
}

static void
view_change_cb (GuppiElementView * view, GuppiCanvasItem * item)
{
  GuppiCanvasItemClass *klass;

  g_return_if_fail (view != NULL);
  g_return_if_fail (item != NULL);

  klass = GUPPI_CANVAS_ITEM_CLASS (GTK_OBJECT (item)->klass);
  if (klass->view_changed)
    (klass->view_changed) (item);

}

GuppiCanvasItem *
guppi_element_view_make_canvas_item (GuppiElementView * view,
				     GnomeCanvas * canvas,
				     GnomeCanvasGroup * group)
{
  GuppiElementViewPrivate *p;
  GuppiElementViewClass *klass;
  GuppiCanvasItem *item = NULL;
  GnomeCanvasItem *gnoitem;

  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);
  g_return_val_if_fail (canvas != NULL && GNOME_IS_CANVAS (canvas), NULL);

  p = priv (view);
  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

#if 0
  if (klass->canvas_item_type && klass->make_canvas_item) {
    g_warning
      ("For %s, both an item type and an item constructor are defined.",
       gtk_type_name (GTK_OBJECT_TYPE (view)));
  }
#endif

  /* Default to the root group. */
  if (group == NULL)
    group = gnome_canvas_root (canvas);

  if (klass->make_canvas_item) {

    item = (klass->make_canvas_item) (view, canvas, group);

  } else if (klass->canvas_item_type) {

    gnoitem = gnome_canvas_item_new (group, klass->canvas_item_type, NULL);
    item = GUPPI_CANVAS_ITEM (gnoitem);

  } else {

    g_assert_not_reached ();

  }

  guppi_outside_object (item);

  g_return_val_if_fail (item != NULL, NULL);

  guppi_canvas_item_set_view (item, view);

  gtk_signal_connect (GTK_OBJECT (guppi_element_view_state (view)),
		      "changed", GTK_SIGNAL_FUNC (state_change_cb), item);

  gtk_signal_connect (GTK_OBJECT (view),
		      "changed", GTK_SIGNAL_FUNC (view_change_cb), item);

  guppi_canvas_item_set_ready (item);

  if (GUPPI_CANVAS_ITEM_CLASS (GTK_OBJECT (item)->klass)->post_creation_init)
    GUPPI_CANVAS_ITEM_CLASS (GTK_OBJECT (item)->klass)->
      post_creation_init (item);

  return item;
}

GuppiElementPrint *
guppi_element_view_make_print (GuppiElementView * view,
			       GnomePrintContext * pc)
{
  GuppiElementViewClass *klass;
  GuppiElementPrint *ep = NULL;

  g_return_val_if_fail (view != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_ELEMENT_VIEW (view), NULL);

  g_return_val_if_fail (pc != NULL, NULL);
  g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (pc), NULL);

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

  if (klass->print_type && klass->make_print) {
    g_warning
      ("For %s, both a print type and a print constructor are defined.",
       gtk_type_name (GTK_OBJECT_TYPE (view)));
  }

  if (klass->print_type) {

    ep = GUPPI_ELEMENT_PRINT (guppi_type_new (klass->print_type));

    if (ep)
      guppi_element_print_set_context (ep, pc);

  } else if (klass->make_print != NULL) {

    ep = klass->make_print (view, pc);

  }

  if (ep) {
    ep->view = view;
    guppi_ref (view);
  }

  return ep;
}

/***************************************************************************/

void
guppi_element_view_change_along_with_shared_object (GuppiElementView * view,
						    GuppiShared * shared)
{
  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  g_return_if_fail (shared != NULL);
  g_return_if_fail (GUPPI_IS_SHARED (shared));

  gtk_signal_connect_object (GTK_OBJECT (shared),
			     "changed",
			     GTK_SIGNAL_FUNC (guppi_element_view_changed),
			     GTK_OBJECT (view));
}


void
guppi_element_view_connect (GuppiElementView * v1, const gchar * key1,
			    GuppiElementView * v2, const gchar * key2)
{
  GuppiElementViewPrivate *p1;
  GuppiElementViewPrivate *p2;
  GuppiShared *sh1;
  GuppiShared *sh2;

  g_return_if_fail (v1 != NULL && GUPPI_IS_ELEMENT_VIEW (v1));
  g_return_if_fail (key1 != NULL);

  g_return_if_fail (v2 != NULL && GUPPI_IS_ELEMENT_VIEW (v2));
  g_return_if_fail (key2 != NULL);

  p1 = priv (v1);
  p2 = priv (v2);

  if (!strcmp (key1, SHARED_X_VIEW))
    sh1 = p1->x_view_interval;
  else if (!strcmp (key1, SHARED_Y_VIEW))
    sh1 = p1->y_view_interval;
  else if (!strcmp (key1, SHARED_X_MARKERS))
    sh1 = p1->x_axis_markers;
  else if (!strcmp (key1, SHARED_Y_MARKERS))
    sh1 = p1->y_axis_markers;
  else
    sh1 = guppi_shared_hash_extract (p1->hash, key1);
  g_return_if_fail (sh1 != NULL);

  if (!strcmp (key2, SHARED_X_VIEW))
    sh2 = p2->x_view_interval;
  else if (!strcmp (key2, SHARED_Y_VIEW))
    sh2 = p2->y_view_interval;
  else if (!strcmp (key2, SHARED_X_MARKERS))
    sh2 = p2->x_axis_markers;
  else if (!strcmp (key2, SHARED_Y_MARKERS))
    sh2 = p2->y_axis_markers;
  else
    sh2 = guppi_shared_hash_extract (p2->hash, key2);
  g_return_if_fail (sh2 != NULL);

  if (sh1 != sh2) {
    guppi_shared_connect (sh1, sh2);
  }
}

/***************************************************************************/

void
guppi_element_view_set_preferred_x_view (GuppiElementView * view)
{
  GuppiViewInterval *vi;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  vi = guppi_element_view_x_view_interval (view);
  if (vi)
    guppi_view_interval_request_preferred_range (vi);
}

void
guppi_element_view_set_preferred_y_view (GuppiElementView * view)
{
  GuppiViewInterval *vi;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  vi = guppi_element_view_y_view_interval (view);
  if (vi)
    guppi_view_interval_request_preferred_range (vi);
}

void
guppi_element_view_set_preferred_view (GuppiElementView * view)
{
  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  /* In cases of non-independent preferred values calculations, it might
     be necessary to do something like this to force us into the "true"
     bi-axis preferred state. */

  guppi_element_view_set_preferred_x_view (view);
  guppi_element_view_set_preferred_y_view (view);
  guppi_element_view_set_preferred_x_view (view);
  guppi_element_view_set_preferred_y_view (view);
}

void
guppi_element_view_force_preferred_x_view (GuppiElementView *view, gboolean q)
{
  g_return_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view));

  if (priv (view)->force_x_preferred != q) {
    priv (view)->force_x_preferred = q;
    if (q)
      guppi_element_view_set_preferred_x_view (view);
  }
}

void
guppi_element_view_force_preferred_y_view (GuppiElementView *view, gboolean q)
{
  g_return_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view));

  if (priv (view)->force_y_preferred != q) {
    priv (view)->force_y_preferred = q;
    if (q)
      guppi_element_view_set_preferred_y_view (view);
  }
}

void
guppi_element_view_force_preferred_view (GuppiElementView *view, gboolean q)
{
  g_return_if_fail (view && GUPPI_IS_ELEMENT_VIEW (view));

  guppi_element_view_force_preferred_x_view (view, q);
  guppi_element_view_force_preferred_y_view (view, q);
}

/***************************************************************************/

gboolean guppi_element_view_visible (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), FALSE);

  return !(priv (view)->hide);
}

void
guppi_element_view_show (GuppiElementView * view)
{
  guppi_element_view_set_visibility (view, TRUE);
}

void
guppi_element_view_hide (GuppiElementView * view)
{
  guppi_element_view_set_visibility (view, TRUE);
}

void
guppi_element_view_set_visibility (GuppiElementView * view, gboolean x)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);

  if (p->hide != !x) {
    p->hide = !x;
    guppi_element_view_changed (view);
  }
}

/***************************************************************************/

void
guppi_element_view_style_bounds (GuppiElementView * view, gint * a, gint * b)
{
  GuppiElementViewClass *klass;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

  if (klass->get_style_bounds)
    klass->get_style_bounds (view, a, b);
  else {
    if (a)
      *a = 0;
    if (b)
      *b = -1;
  }
}

gboolean guppi_element_view_style_in_bounds (GuppiElementView * view, gint i)
{
  gint a, b;
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), FALSE);

  guppi_element_view_style_bounds (view, &a, &b);
  return a <= i && i <= b;
}

GuppiStyle *
guppi_element_view_style (GuppiElementView * view, gint i)
{
  GuppiElementViewClass *klass;

  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), NULL);
  g_return_val_if_fail (guppi_element_view_style_in_bounds (view, i), NULL);

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  if (klass->get_style)
    return klass->get_style (view, i);
  else
    return NULL;
}

/***************************************************************************/

gboolean guppi_element_view_tools_are_blocked (GuppiElementView * view)
{
  g_return_val_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view), FALSE);
  return priv (view)->block_tools;
}

void
guppi_element_view_set_tool_blocking (GuppiElementView * view, gboolean x)
{
  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  priv (view)->block_tools = x;
}



GuppiConfigItem *
guppi_element_view_config_tree (GuppiElementView * view)
{
  GuppiElementState *state;
  GuppiElementStateClass *state_class;
  GuppiElementViewClass *view_class;

  GuppiConfigItem *state_tree = NULL;
  GuppiConfigItem *view_tree = NULL;
  GuppiConfigItem *total_tree = NULL;

  g_return_val_if_fail (view != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_ELEMENT_VIEW (view), NULL);

  state = guppi_element_view_state (view);

  state_class = GUPPI_ELEMENT_STATE_CLASS (GTK_OBJECT (state)->klass);
  view_class = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);

  if (view_class->assemble_config_tree == NULL)
    return NULL;

  if (state_class->config_tree) {
    state_tree = state_class->config_tree (state);
    if (state_tree)
      guppi_config_item_set_tag (state_tree, state);
  }

  if (view_class->config_tree)
    view_tree = view_class->config_tree (view);

  total_tree = view_class->assemble_config_tree (view, state_tree, view_tree);
  if (total_tree)
    guppi_config_item_set_tag (total_tree, view);

  return total_tree;
}

/***************************************************************************/

void
guppi_element_view_print (GuppiElementView * view, GnomePrintContext * pc)
{
  GuppiElementViewClass *klass;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  g_return_if_fail (pc != NULL && GNOME_IS_PRINT_CONTEXT (pc));

  klass = GUPPI_ELEMENT_VIEW_CLASS (GTK_OBJECT (view)->klass);
  if (klass->print)
    klass->print (view, pc);
}

void
guppi_element_view_print_to_bbox (GuppiElementView * view,
				  GnomePrintContext * pc,
				  double x0, double y0, double x1, double y1)
{
  GuppiGeometry *geom;
  double scale_w, scale_h, scale;
  double cx, cy;		/* corner x and y */
  double mat_scale[6];
  double mat_translate[6];

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  g_return_if_fail (pc != NULL && GNOME_IS_PRINT_CONTEXT (pc));

  guppi_2sort (&x0, &x1);
  guppi_2sort (&y0, &y1);

  geom = guppi_element_view_geometry (view);

  scale_w = (x1 - x0) / guppi_geometry_width (geom);
  scale_h = (y1 - y0) / guppi_geometry_height (geom);
  scale = MIN (scale_w, scale_h);

  cx = (x0 + x1) / 2 - scale * guppi_geometry_width (geom) / 2;
  cy = (y0 + y1) / 2 - scale * guppi_geometry_height (geom) / 2;

  art_affine_scale (mat_scale, scale, scale);
  art_affine_translate (mat_translate, cx, cy);

  gnome_print_gsave (pc);

  gnome_print_concat (pc, mat_translate);
  gnome_print_concat (pc, mat_scale);

  guppi_element_view_print (view, pc);

  gnome_print_grestore (pc);
}

void
guppi_element_view_print_ps_to_file (GuppiElementView * view,
				     const gchar * filename)
{
  GnomePrinter *gprinter;
  GnomePrintContext *pc;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  g_return_if_fail (filename != NULL);

  gprinter = gnome_printer_new_generic_ps (filename);
  g_return_if_fail (gprinter != NULL);

  pc = gnome_print_context_new (gprinter);
  g_return_if_fail (pc != NULL);

  guppi_element_view_print (view, pc);
  gnome_print_showpage (pc);
  gnome_print_context_close (pc);

  guppi_unref (pc);
  guppi_unref (gprinter);
}

/***************************************************************************/

void
guppi_element_view_changed (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);

  if (p->pending_delayed_change) {
    gtk_idle_remove (p->pending_delayed_change);
    p->pending_delayed_change = 0;
  }

  if (p->freeze_count > 0)
    p->pending_change = 1;
  else {
    gtk_signal_emit (GTK_OBJECT (view), view_signals[CHANGED]);
    p->pending_change = 0;
  }
}

static gint
delayed_changer (gpointer foo)
{
  GuppiElementView *view = GUPPI_ELEMENT_VIEW (foo);
  priv (view)->pending_delayed_change = 0;
  guppi_element_view_changed (view);
  return FALSE;
}

void
guppi_element_view_changed_delayed (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);

  if (p->freeze_count > 0)
    p->pending_change = TRUE;
  else if (p->pending_delayed_change == 0) {

    if (guppi_is_synchronous ()) {

      guppi_element_view_changed (view);

    } else {

      p->pending_delayed_change = gtk_idle_add_priority (G_PRIORITY_HIGH_IDLE,
							 delayed_changer,
							 view);

    }
  }
}

void
guppi_element_view_flush_changes (GuppiElementView * view)
{
  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  if (priv (view)->pending_delayed_change != 0)
    guppi_element_view_changed (view);
}

void
guppi_element_view_changed_size (GuppiElementView * view)
{
  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  guppi_geometry_calc_natural_size (guppi_element_view_geometry (view));
}

void
guppi_element_view_freeze (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL);
  g_return_if_fail (GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);
  ++p->freeze_count;
}

void
guppi_element_view_thaw (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));

  p = priv (view);
  g_return_if_fail (p->freeze_count > 0);

  --p->freeze_count;
  if (p->freeze_count == 0 && p->pending_change)
    guppi_element_view_changed (view);
}

void
guppi_element_view_changed_structure (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  p = priv (view);

  if (p->pending_structure_change) {
    gtk_idle_remove (p->pending_structure_change);
    p->pending_structure_change = 0;
  }

  gtk_signal_emit (GTK_OBJECT (view), view_signals[CHANGED_STRUCTURE]);
}

static gint
struct_change_idle_fn (gpointer ptr)
{
  guppi_element_view_changed_structure (GUPPI_ELEMENT_VIEW (ptr));
  return FALSE;
}

void
guppi_element_view_changed_structure_delayed (GuppiElementView * view)
{
  GuppiElementViewPrivate *p;

  g_return_if_fail (view != NULL && GUPPI_IS_ELEMENT_VIEW (view));
  p = priv (view);

  if (p->pending_structure_change == 0)
    p->pending_structure_change = gtk_idle_add (struct_change_idle_fn, view);
}

/* $Id: guppi-element-view.c,v 1.36 2001/05/06 08:26:43 trow Exp $ */
