/* This is -*- C -*- */
/* $Id: guppi-scatter-style-view.c,v 1.10 2001/05/06 08:26:28 trow Exp $ */

/*
 * guppi-scatter-style-view.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <math.h>
#include <guppi-convenient.h>
#include <guppi-rgb.h>
#include "guppi-scatter-style-view.h"

static GtkObjectClass *parent_class = NULL;

static void
free_alpha_templates (GuppiAlphaTemplate ** at, gint N)
{
  gint i;
  for (i = 0; i < N; ++i)
    guppi_unref (at[i]);
  guppi_free (at);
}

static void
guppi_scatter_style_view_finalize (GtkObject * obj)
{
  GuppiScatterStyleView *sv = GUPPI_SCATTER_STYLE_VIEW (obj);

  guppi_finalized (obj);

  guppi_unref0 (sv->style);

  guppi_scatter_style_view_reset (sv);

  if (sv->sz1_cache)
    free_alpha_templates (sv->sz1_cache, sv->sz1_cache_size);

  if (sv->sz2_cache)
    free_alpha_templates (sv->sz2_cache, sv->sz2_cache_size);

  if (sv->mixed_cache)
    free_alpha_templates (sv->mixed_cache,
			  sv->mixed_sz1_size * sv->mixed_sz2_size);

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_scatter_style_view_class_init (GuppiScatterStyleViewClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->finalize = guppi_scatter_style_view_finalize;

}

static void
guppi_scatter_style_view_init (GuppiScatterStyleView * obj)
{

}

GtkType guppi_scatter_style_view_get_type (void)
{
  static GtkType guppi_scatter_style_view_type = 0;
  if (!guppi_scatter_style_view_type) {
    static const GtkTypeInfo guppi_scatter_style_view_info = {
      "GuppiScatterStyleView",
      sizeof (GuppiScatterStyleView),
      sizeof (GuppiScatterStyleViewClass),
      (GtkClassInitFunc) guppi_scatter_style_view_class_init,
      (GtkObjectInitFunc) guppi_scatter_style_view_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_scatter_style_view_type =
      gtk_type_unique (GTK_TYPE_OBJECT, &guppi_scatter_style_view_info);
  }
  return guppi_scatter_style_view_type;
}

GuppiScatterStyleView *
guppi_scatter_style_view_new (void)
{
  return
    GUPPI_SCATTER_STYLE_VIEW (guppi_type_new
			      (guppi_scatter_style_view_get_type ()));
}

void
guppi_scatter_style_view_set_style (GuppiScatterStyleView * sv,
				    GuppiScatterStyle * sty)
{
  g_return_if_fail (sv != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE_VIEW (sv));
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE0 (sty));

  if (sv->style != sty) {

    guppi_refcounting_assign (sv->style, sty);

  }
}
void
guppi_scatter_style_view_reset (GuppiScatterStyleView * sv)
{
  g_return_if_fail (sv != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE_VIEW (sv));

  guppi_unref0 (sv->marker_template);

  sv->marker_sz1 = sv->marker_sz2 = 0;
}

void
guppi_scatter_style_view_set_scale (GuppiScatterStyleView * sv, double x)
{
  g_return_if_fail (sv != NULL);
  g_return_if_fail (GUPPI_IS_SCATTER_STYLE_VIEW (sv));
  g_return_if_fail (x > 0);

  sv->scale_factor = x;
}

/***************************************************************************/

static GuppiAlphaTemplate *
variable_size1_lookup (GuppiScatterStyleView * sv, double t, double scale)
{

  double sz_min = guppi_scatter_style_size1_gradient_min (sv->style);
  double sz_max = guppi_scatter_style_size1_gradient_max (sv->style);
  double sz2 = guppi_style_marker_size2 (GUPPI_STYLE (sv->style));
  GuppiMarker m = guppi_style_marker (GUPPI_STYLE (sv->style));
  gint i;

  if (sv->sz1_cache != NULL &&
      (sv->sz1_min != sz_min || sv->sz1_max != sz_max ||
       sv->sz1_scale != scale || sv->sz1_sz2 != sz2 || sv->sz1_marker != m)) {
    free_alpha_templates (sv->sz1_cache, sv->sz1_cache_size);
    sv->sz1_cache = NULL;
  }

  if (sv->sz1_cache == NULL) {
    sv->sz1_cache_size = 20;	/* hard-wired for now */
    sv->sz1_cache = guppi_new0 (GuppiAlphaTemplate *, sv->sz1_cache_size);
    sv->sz1_min = sz_min;
    sv->sz1_max = sz_max;
    sv->sz1_sz2 = sz2;
    sv->sz1_scale = scale;
    sv->sz1_marker = m;
  }

  i = (gint) rint (t * (sv->sz1_cache_size - 1));

  if (sv->sz1_cache[i] == NULL) {
    double sz1 = t * (sv->sz1_max - sv->sz1_min) + sv->sz1_min;

    sv->sz1_cache[i] = guppi_marker_alpha_template (sv->sz1_marker,
						    sz1, sz2, scale);

    guppi_ref (sv->sz1_cache[i]);
  }

  return sv->sz1_cache[i];
}

/* code dup from above */
static GuppiAlphaTemplate *
variable_size2_lookup (GuppiScatterStyleView * sv, double t, double scale)
{
  double sz_min = guppi_scatter_style_size2_gradient_min (sv->style);
  double sz_max = guppi_scatter_style_size2_gradient_max (sv->style);
  double sz1 = guppi_style_marker_size1 (GUPPI_STYLE (sv->style));
  GuppiMarker m = guppi_style_marker (GUPPI_STYLE (sv->style));
  gint i;

  if (sv->sz2_cache != NULL &&
      (sv->sz2_min != sz_min || sv->sz2_max != sz_max ||
       sv->sz2_scale != scale || sv->sz2_sz1 != sz1 || sv->sz2_marker != m)) {
    free_alpha_templates (sv->sz2_cache, sv->sz2_cache_size);
    sv->sz2_cache = NULL;
  }

  if (sv->sz2_cache == NULL) {
    sv->sz2_cache_size = 10;	/* hard-wired for now, different than above */
    sv->sz2_cache = guppi_new0 (GuppiAlphaTemplate *, sv->sz2_cache_size);
    sv->sz2_min = sz_min;
    sv->sz2_max = sz_max;
    sv->sz2_sz1 = sz1;
    sv->sz2_scale = scale;
    sv->sz2_marker = m;
  }

  i = (gint) rint (t * (sv->sz2_cache_size - 1));

  if (sv->sz2_cache[i] == NULL) {
    double sz2 = t * (sv->sz2_max - sv->sz2_min) + sv->sz2_min;

    sv->sz2_cache[i] = guppi_marker_alpha_template (sv->sz2_marker,
						    sz1, sz2, scale);
    guppi_ref (sv->sz2_cache[i]);
  }

  return sv->sz2_cache[i];
}

static GuppiAlphaTemplate *
variable_mixed_lookup (GuppiScatterStyleView * sv, double t1, double t2,
		       double scale)
{
  double sz1_min = guppi_scatter_style_size1_gradient_min (sv->style);
  double sz1_max = guppi_scatter_style_size1_gradient_max (sv->style);
  double sz2_min = guppi_scatter_style_size2_gradient_min (sv->style);
  double sz2_max = guppi_scatter_style_size2_gradient_max (sv->style);
  GuppiMarker m = guppi_style_marker (GUPPI_STYLE (sv->style));
  gint i1, i2, i;

  if (sv->mixed_cache != NULL &&
      (sv->mixed_sz1_min != sz1_min || sv->mixed_sz1_max != sz1_max ||
       sv->mixed_sz2_min != sz2_min || sv->mixed_sz2_max != sz2_max ||
       sv->mixed_scale != scale || sv->mixed_marker != m)) {
    free_alpha_templates (sv->mixed_cache,
			  sv->mixed_sz1_size * sv->mixed_sz2_size);
    sv->mixed_cache = NULL;
  }

  if (sv->mixed_cache == NULL) {
    sv->mixed_sz1_size = 20;	/* hard-wired for now */
    sv->mixed_sz2_size = 10;
    sv->mixed_cache = guppi_new0 (GuppiAlphaTemplate *,
			      sv->mixed_sz1_size * sv->mixed_sz2_size);
    sv->mixed_sz1_min = sz1_min;
    sv->mixed_sz1_max = sz1_max;
    sv->mixed_sz2_min = sz2_min;
    sv->mixed_sz2_max = sz2_max;
    sv->mixed_scale = scale;
    sv->mixed_marker = m;
  }

  i1 = (gint) rint (t1 * (sv->mixed_sz1_size - 1));
  i2 = (gint) rint (t2 * (sv->mixed_sz2_size - 1));

  i = i1 + i2 * sv->mixed_sz1_size;

  if (sv->mixed_cache[i] == NULL) {
    double sz1 =
      t1 * (sv->mixed_sz1_max - sv->mixed_sz1_min) + sv->mixed_sz1_min;
    double sz2 =
      t2 * (sv->mixed_sz2_max - sv->mixed_sz2_min) + sv->mixed_sz2_min;

    sv->mixed_cache[i] = guppi_marker_alpha_template (sv->mixed_marker,
						      sz1, sz2, scale);

    guppi_ref (sv->mixed_cache[i]);
  }

  return sv->mixed_cache[i];
}

/***************************************************************************/

GuppiAlphaTemplate *
guppi_scatter_style_view_marker_template (GuppiScatterStyleView * sv,
					  double t_sz1, double t_sz2)
{
  GuppiMarker m;
  gboolean valid_t_sz1, valid_t_sz2;

  g_return_val_if_fail (sv != NULL, NULL);
  g_return_val_if_fail (GUPPI_IS_SCATTER_STYLE_VIEW (sv), NULL);

  if (sv->style == NULL)
    return NULL;

  valid_t_sz1 = (0 <= t_sz1 && t_sz1 <= 1) &&
    guppi_scatter_style_allow_size1_gradient (sv->style);

  valid_t_sz2 = (0 <= t_sz2 && t_sz2 <= 1) &&
    guppi_scatter_style_allow_size2_gradient (sv->style);

  if (valid_t_sz1 && guppi_scatter_style_reverse_size1_gradient (sv->style))
    t_sz1 = 1 - t_sz1;

  if (valid_t_sz2 && guppi_scatter_style_reverse_size2_gradient (sv->style))
    t_sz2 = 1 - t_sz2;

  if (!(valid_t_sz1 || valid_t_sz2)) {

    /* If both gradient terms are out of range, we use a fixed template. */

    double sz1, sz2;

    /* Check that our marker template is up-to-date */
    m = guppi_style_marker (GUPPI_STYLE (sv->style));
    sz1 = guppi_style_marker_size1 (GUPPI_STYLE (sv->style));
    sz2 = guppi_style_marker_size2 (GUPPI_STYLE (sv->style));
    if (sv->marker_template == NULL ||
	m != sv->marker ||
	sz1 * sv->scale_factor != sv->marker_sz1 ||
	sz2 * sv->scale_factor != sv->marker_sz2) {

      guppi_unref0 (sv->marker_template);

      sv->marker_template = guppi_marker_alpha_template (m, sz1, sz2,
							 sv->scale_factor);

      sv->marker = m;
      sv->marker_sz1 = sz1 * sv->scale_factor;
      sv->marker_sz2 = sz2 * sv->scale_factor;

    }

    return sv->marker_template;
  } else if (valid_t_sz1 && !valid_t_sz2) {

    return variable_size1_lookup (sv, t_sz1, sv->scale_factor);


  } else if (!valid_t_sz1 && valid_t_sz2) {

    return variable_size2_lookup (sv, t_sz2, sv->scale_factor);

  } else {			/* both valid */

    return variable_mixed_lookup (sv, t_sz1, t_sz2, sv->scale_factor);

  }
  g_assert_not_reached ();
  return NULL;
}

guint32
guppi_scatter_style_view_marker_color (GuppiScatterStyleView * sv, double t)
{
  return guppi_scatter_style_calc_color_gradient (sv->style, t);
}


/* $Id: guppi-scatter-style-view.c,v 1.10 2001/05/06 08:26:28 trow Exp $ */
