/* $Id: ggraph.c,v 1.4 2000/12/16 03:42:50 trow Exp $ */

/*
 * ggraph.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/*
  This is a very inefficient implementation of graphs, full of
  arbitrary restrictions.  Among other things, multi-graphs should be
  allowed and it should be possible to attach data to edges.  And
  there shouldn't be all of these bad linear searches.

  I've used the glib, rather than the guppi, naming scheme, so that if
  I ever get my shit together here, I could submit this for inclusion
  into glib.
*/


#include <config.h>
#include "guppi-memory.h"
#include "ggraph.h"

GGraph *
g_graph_new (void)
{
  GGraph *gg = guppi_new0 (GGraph, 1);

  return gg;
}

void
g_graph_free (GGraph *gg)
{
  GList *iter;

  if (gg == NULL)
    return;

  iter = gg->vertices;
  while (iter) {
    GVertex *v = (GVertex *)iter->data;
    g_list_free (v->edges);
    guppi_free (v);
    iter = g_list_next (iter);
  }

  g_list_free (gg->vertices);
  guppi_free (gg);
}

GVertex *
g_graph_add_vertex (GGraph * gg, gpointer data)
{
  GVertex *v;

  g_return_val_if_fail (gg != NULL, NULL);

  v = guppi_new0 (GVertex, 1);
  v->data = data;

  gg->vertices = g_list_prepend (gg->vertices, v);

  return v;
}

void
g_graph_add_edge (GGraph * gg, GVertex * a, GVertex * b)
{
  g_return_if_fail (gg != NULL);
  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  /* By default, only allow simple graphs: only one edge between any two
     vertices, and a vertex can't connect to itself. */
  if (a == b || g_list_find (a->edges, b))
    return;

  a->edges = g_list_prepend (a->edges, b);
  b->edges = g_list_prepend (b->edges, a);
}

GVertex *
g_graph_find_vertex_by_data (GGraph * gg, gpointer data)
{
  GList *iter;

  g_return_val_if_fail (gg != NULL, NULL);

  iter = gg->vertices;
  while (iter != NULL) {
    GVertex *v = (GVertex *) iter->data;
    if (v->data == data)
      return v;
    iter = g_list_next (iter);
  }
  return NULL;
}

void
g_graph_remove_vertex (GGraph * gg, GVertex * v)
{
  GList *iter;

  g_return_if_fail (gg != NULL);
  if (v == NULL)
    return;

  iter = v->edges;
  while (iter != NULL) {
    GVertex *vv = (GVertex *) iter->data;
    vv->edges = g_list_remove (vv->edges, v);
    iter = g_list_next (iter);
  }
  gg->vertices = g_list_remove (gg->vertices, v);
  g_list_free (v->edges);
  guppi_free (v);
}

void
g_graph_remove_vertex_and_relink (GGraph * gg, GVertex * v)
{
  GList *iter1;
  GList *iter2;
  GVertex *v1;
  GVertex *v2;

  g_return_if_fail (gg != NULL);
  if (v == NULL)
    return;

  iter1 = v->edges;
  while (iter1 != NULL) {
    v1 = (GVertex *) iter1->data;
    v1->edges = g_list_remove (v1->edges, v);
    iter2 = v->edges;
    while (iter2 != NULL) {
      v2 = (GVertex *) iter2->data;
      if (v2 != v && v2 != v1)
	g_graph_add_edge (gg, v1, v2);
      iter2 = g_list_next (iter2);
    }
    iter1 = g_list_next (iter1);
  }
  gg->vertices = g_list_remove (gg->vertices, v);
  g_list_free (v->edges);
  guppi_free (v);
}

void
g_graph_foreach (GGraph * gg, GFunc func, gpointer user_data)
{
  GList *iter;

  g_return_if_fail (gg != NULL);
  g_return_if_fail (func != NULL);

  iter = gg->vertices;
  while (iter != NULL) {
    func (((GVertex *) iter->data)->data, user_data);
    iter = g_list_next (iter);
  }
}

void
g_vertex_traverse_neighbors (GVertex * v, GFunc func, gpointer user_data)
{
  GList *iter;

  g_return_if_fail (v != NULL);
  g_return_if_fail (func != NULL);

  iter = v->edges;
  while (iter != NULL) {
    func (((GVertex *) iter->data)->data, user_data);
    iter = g_list_next (iter);
  }
}

static void
traverse_cc (GVertex * v, GFunc func, gpointer user_data, GHashTable * seen)
{
  GList *iter;

  func (v->data, user_data);
  g_hash_table_insert (seen, v, seen);

  iter = v->edges;
  while (iter != NULL) {
    v = (GVertex *) iter->data;
    if (g_hash_table_lookup (seen, v) != seen)
      traverse_cc (v, func, user_data, seen);
    iter = g_list_next (iter);
  }
}

void
g_vertex_traverse_connected_component (GVertex * v, GFunc func,
				       gpointer user_data)
{
  GHashTable *seen;

  g_return_if_fail (v != NULL);
  g_return_if_fail (func != NULL);

  seen = g_hash_table_new (NULL, NULL);	/* use default "direct" hash/equals */
  traverse_cc (v, func, user_data, seen);
  g_hash_table_destroy (seen);
}

/* $Id: ggraph.c,v 1.4 2000/12/16 03:42:50 trow Exp $ */
