/* This is -*- C -*- */
/* $Id: guppi-shared.c,v 1.14 2001/05/06 08:26:43 trow Exp $ */

/*
 * guppi-shared.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/* The idea of using a graph of the relationships between different
  objects, rather than signals.  It really does make things a lot
  easier and nicely avoids the "signal cascade" effect.  */

#include <config.h>

#include <gtk/gtksignal.h>

#include <guppi-useful.h>
#include <ggraph.h>
#include <guppi-data.h>
#include "guppi-view-interval.h"
#include "guppi-axis-markers.h"
#include "guppi-shared.h"


static GtkObjectClass *parent_class = NULL;

enum {
  ARG_0
};

enum {
  CHANGED_HELD,
  CHANGED_UNDERLYING,
  CHANGED,
  LAST_SIGNAL
};

static guint gsh_signals[LAST_SIGNAL] = { 0 };

static GGraph *connection_graph = NULL;

static void
guppi_shared_get_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_shared_set_arg (GtkObject * obj, GtkArg * arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_shared_destroy (GtkObject * obj)
{
  if (parent_class->destroy)
    parent_class->destroy (obj);
}

static void
guppi_shared_finalize (GtkObject * obj)
{
  GuppiShared *gsh = GUPPI_SHARED (obj);

  guppi_finalized (obj);

  /* Disconnect any signal handler. */
  if (gsh->changed_handler) {
    gtk_signal_disconnect (gsh->obj, gsh->changed_handler);
    gsh->changed_handler = 0;
  }

  /* Disconnect ourselves from the graph. */
  guppi_shared_disconnect (gsh);

  /* Unref the object we are holding. */
  guppi_unref0 (gsh->obj);

  guppi_free (gsh->changed_signal_name);
  gsh->changed_signal_name = NULL;

  if (parent_class->finalize)
    parent_class->finalize (obj);
}

static void
guppi_shared_class_init (GuppiSharedClass * klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  parent_class = gtk_type_class (GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_shared_get_arg;
  object_class->set_arg = guppi_shared_set_arg;
  object_class->destroy = guppi_shared_destroy;
  object_class->finalize = guppi_shared_finalize;

  gsh_signals[CHANGED_HELD] =
    gtk_signal_new ("changed_held",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSharedClass, changed_held),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

  gsh_signals[CHANGED_UNDERLYING] =
    gtk_signal_new ("changed_underlying",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSharedClass, changed_underlying),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);

  gsh_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_FIRST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GuppiSharedClass, changed),
		    gtk_marshal_NONE__NONE, GTK_TYPE_NONE, 0);


  gtk_object_class_add_signals (object_class, gsh_signals, LAST_SIGNAL);
}

static void
guppi_shared_init (GuppiShared * obj)
{

}

GtkType guppi_shared_get_type (void)
{
  static GtkType guppi_shared_type = 0;
  if (!guppi_shared_type) {
    static const GtkTypeInfo guppi_shared_info = {
      "GuppiShared",
      sizeof (GuppiShared),
      sizeof (GuppiSharedClass),
      (GtkClassInitFunc) guppi_shared_class_init,
      (GtkObjectInitFunc) guppi_shared_init,
      NULL, NULL, (GtkClassInitFunc) NULL
    };
    guppi_shared_type = gtk_type_unique (GTK_TYPE_OBJECT, &guppi_shared_info);

    /* We init our connection graph at the same time as the class. */
    if (connection_graph == NULL) {
      connection_graph = g_graph_new ();
      guppi_permanent_alloc (connection_graph);
    }
  }
  return guppi_shared_type;
}

/**************************************************************************/

void
guppi_shared_construct (GuppiShared * gsh,
			GtkType allowed_type,
			const gchar * changed_signal_name,
			void (*pre_set) (GuppiShared *, GtkObject *),
			void (*post_set) (GuppiShared *))
{
  g_return_if_fail (gsh != NULL);

  gsh->allowed_type = allowed_type;
  guppi_free (gsh->changed_signal_name);
  gsh->changed_signal_name = guppi_strdup (changed_signal_name);
  gsh->pre_set = pre_set;
  gsh->post_set = post_set;
}

GuppiShared *
guppi_shared_new (GtkType allowed_type,
		  const gchar * changed_signal_name,
		  void (*pre_set) (GuppiShared *, GtkObject *),
		  void (*post_set) (GuppiShared *))
{
  GuppiShared *gsh;

  gsh = GUPPI_SHARED (guppi_type_new (guppi_shared_get_type ()));
  guppi_shared_construct (gsh, allowed_type, changed_signal_name,
			  pre_set, post_set);
  return gsh;
}

static void
changed_und (GtkObject * foo, GuppiShared * gsh)
{
  g_return_if_fail (gsh != NULL);
  gtk_signal_emit (GTK_OBJECT (gsh), gsh_signals[CHANGED_UNDERLYING]);
  gtk_signal_emit (GTK_OBJECT (gsh), gsh_signals[CHANGED]);
}

static gboolean
guppi_shared_set_inner (GuppiShared * gsh, GtkObject * obj)
{
  GtkObject *old_obj;

  if (obj != NULL &&
      gsh->allowed_type && !GTK_CHECK_TYPE (obj, gsh->allowed_type)) {
    g_warning ("Attempt to set type %s in a connectable that only accepts %s",
	       gtk_type_name (GTK_OBJECT_TYPE (obj)),
	       guppi_shared_allowed_type (gsh));
    return FALSE;
  }

  if (gsh->obj != obj) {

    /* Call our pre-setting virtual function. */
    if (gsh->pre_set)
      (gsh->pre_set) (gsh, obj);

    /* If we are listening for signals from the current object,
       disconnect the handler. */
    if (gsh->changed_handler) {
      gtk_signal_disconnect (gsh->obj, gsh->changed_handler);
      gsh->changed_handler = 0;
    }

    /* Add a reference to the object we are about to change to. */
    guppi_ref (obj);

    /* Remember our old object. */
    old_obj = gsh->obj;

    /* Do the deed. */
    gsh->obj = obj;

    /* If we have the name of a simple "changed" signal handler for
       the underlying data object, connect to it. */
    if (gsh->obj && gsh->changed_signal_name) {
      gsh->changed_handler = gtk_signal_connect_after (gsh->obj,
						       gsh->changed_signal_name,
						       GTK_SIGNAL_FUNC
						       (changed_und), gsh);
    }



    /* Call our post-setting virtual function. */
    if (gsh->post_set)
      (gsh->post_set) (gsh);

    /* Emit a signal saying that our object has changed. */
    gtk_signal_emit (GTK_OBJECT (gsh), gsh_signals[CHANGED_HELD], old_obj);

    /* Remove a reference to the old object. */
    guppi_unref (old_obj);

    gtk_signal_emit (GTK_OBJECT (gsh), gsh_signals[CHANGED]);

    return TRUE;
  }

  return FALSE;
}

static void
set_traverse_fn (gpointer data, gpointer user_data)
{
  GuppiShared *gsh;
  GtkObject *obj;

  g_return_if_fail (data != NULL);
  g_return_if_fail (GUPPI_IS_SHARED (data));
  g_return_if_fail (user_data == NULL || GTK_IS_OBJECT (user_data));

  gsh = GUPPI_SHARED (data);
  obj = user_data ? GTK_OBJECT (user_data) : NULL;

  guppi_shared_set_inner (gsh, obj);
}

void
guppi_shared_set (GuppiShared * gsh, GtkObject * obj)
{
  GVertex *v;
  g_return_if_fail (gsh != NULL);

  if (guppi_shared_set_inner (gsh, obj)) {
    v = g_graph_find_vertex_by_data (connection_graph, gsh);
    if (v != NULL)
      g_vertex_traverse_connected_component (v, set_traverse_fn, obj);
  }
}


void
guppi_shared_connect (GuppiShared * a, GuppiShared * b)
{
  GVertex *va;
  GVertex *vb;

  g_return_if_fail (a != NULL);
  g_return_if_fail (b != NULL);

  /* We are only allowed to connect two objects of the same type.
     (This rule might be too strict, and might have to be relaxed
     later to accomodate cases I'm not thinking of right now.) */
  g_return_if_fail (GTK_OBJECT_TYPE (a) == GTK_OBJECT_TYPE (b));

  /* Look up our vertices */

  va = g_graph_find_vertex_by_data (connection_graph, a);
  if (va == NULL)
    va = g_graph_add_vertex (connection_graph, a);

  vb = g_graph_find_vertex_by_data (connection_graph, b);
  if (vb == NULL)
    vb = g_graph_add_vertex (connection_graph, b);

  g_graph_add_edge (connection_graph, va, vb);

  /* By convention, if a and b are holding different objects at
     connection time, we sync them by setting b to hold a's object. */
  if (a->obj != b->obj) {
    if (a->obj == NULL && b->obj != NULL)
      guppi_shared_set (a, b->obj);
    else
      guppi_shared_set (b, a->obj);
  }
}

void
guppi_shared_disconnect (GuppiShared * a)
{
  GVertex *va;

  g_return_if_fail (a != NULL);

  va = g_graph_find_vertex_by_data (connection_graph, a);
  if (va == NULL)
    return;

  g_graph_remove_vertex_and_relink (connection_graph, va);
}

const gchar *
guppi_shared_allowed_type (GuppiShared * gsh)
{
  g_return_val_if_fail (gsh != NULL, NULL);

  if (gsh->allowed_type)
    return gtk_type_name (gsh->allowed_type);
  else
    return "*any*";
}

/**************************************************************************/

GuppiShared *
guppi_shared_data (void)
{
  return guppi_shared_new (GUPPI_TYPE_DATA, "changed", NULL, NULL);
}

GuppiShared *
guppi_shared_view_interval (void)
{
  return guppi_shared_new (GUPPI_TYPE_VIEW_INTERVAL, "changed", NULL, NULL);
}

GuppiShared *
guppi_shared_axis_markers (void)
{
  return guppi_shared_new (GUPPI_TYPE_AXIS_MARKERS, "changed", NULL, NULL);
}

/* $Id: guppi-shared.c,v 1.14 2001/05/06 08:26:43 trow Exp $ */
