/*
 * anjuta2.c Copyright (C) 2000  Kh. Naba Kumar Singh
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 59 
 * Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>

#include <sys/wait.h>
#include <gnome.h>

#include "text_editor.h"
#include "anjuta.h"
#include "fileselection.h"
#include "utilities.h"
#include "support.h"
#include "messagebox.h"
#include "launcher.h"
#include "debugger.h"
#include "controls.h"
#include "project_dbase.h"
#include "anjuta_info.h"

extern gboolean closing_state;

static GdkCursor *app_cursor;

void anjuta_child_terminated (int t);

static void anjuta_load_cmdline_files (void);

static void
on_anjuta_window_selected (GtkMenuItem * menuitem, gpointer user_data);

static void anjuta_clear_windows_menu (void);

static void anjuta_fill_windows_menu (void);

static void anjuta_show_text_editor (TextEditor * te);

void
anjuta_new ()
{
  FILE *stream;
  gchar *buffer;
  app = (AnjutaApp *) g_malloc (sizeof (AnjutaApp));
  if (app)
  {
    /* Must declare static, because it will be used forever */
    static FileSelData fsd1 = { N_("Open File"), NULL,
      on_open_filesel_ok_clicked,
      on_open_filesel_cancel_clicked, NULL
    };

    /* Must declare static, because it will be used forever */
    static FileSelData fsd2 = { N_("Save File As"), NULL,
      on_save_as_filesel_ok_clicked,
      on_save_as_filesel_cancel_clicked, NULL
    };
    app->registered_windows = NULL;
    app->registered_child_processes = NULL;
    app->registered_child_processes_cb = NULL;
    app->registered_child_processes_cb_data = NULL;
    app->text_editor_list = NULL;
    app->current_text_editor = NULL;
    app->recent_files = NULL;
    app->recent_projects = NULL;
    app->win_width = gdk_screen_width ();
    app->win_width -= 10;
    app->win_height = gdk_screen_height ();
    app->win_height -= 25;
    app->vpaned_height = app->win_height / 2 + 7;
    app->hpaned_width = app->win_width / 4;
    app->in_progress = FALSE;
    app->auto_gtk_update = TRUE;
    app->busy_count = 0;
    app->prog_list = NULL;
    app->widgets.toolbar.main_toolbar.is_showing = TRUE;
    app->widgets.toolbar.extended_toolbar.is_showing = TRUE;
    app->widgets.toolbar.browser_toolbar.is_showing = TRUE;
    app->widgets.toolbar.debug_toolbar.is_showing = TRUE;

    signal (SIGCHLD, anjuta_child_terminated);
    create_anjuta_gui (app);

    app->dirs = anjuta_dirs_new ();
    app->fileselection = create_fileselection_gui (&fsd1);
    app->save_as_fileselection = create_fileselection_gui (&fsd2);
    app->find_replace = find_replace_new ();
    app->find_in_files = find_in_files_new ();
    app->preferences = preferences_new ();
    app->compiler_options = compiler_options_new ();
    app->src_paths = src_paths_new ();
    app->messages = messages_new ();
    app->executer = executer_new ();
    app->configurer = configurer_new ();
    app->project_dbase = project_dbase_new ();
    app->tags_manager = tags_manager_new ();

    app->widgets.the_client = app->widgets.vpaned;
    app->widgets.hpaned_client = app->widgets.hpaned;
    gtk_container_add (GTK_CONTAINER (app->widgets.hpaned),
		       app->widgets.notebook);
    gtk_notebook_popup_enable (GTK_NOTEBOOK (app->widgets.notebook));
    gtk_box_pack_start (GTK_BOX (app->widgets.mesg_win_container),
			app->messages->client, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX
			(app->widgets.project_dbase_win_container),
			app->project_dbase->widgets.client, TRUE, TRUE, 0);
    project_dbase_hide (app->project_dbase);
    messages_hide (app->messages);

    gtk_paned_set_position (GTK_PANED (app->widgets.vpaned),
			    app->vpaned_height);
    gtk_paned_set_position (GTK_PANED (app->widgets.hpaned),
			    app->hpaned_width);
    anjuta_update_title ();
    launcher_init ();
    debugger_init ();

    buffer = g_strconcat (app->dirs->settings, "/anjuta.set", NULL);
    stream = fopen (buffer, "r");
    g_free (buffer);
    if (stream)
    {
      if (anjuta_load_yourself (stream))
	gtk_widget_set_uposition (app->widgets.window, app->win_pos_x,
				  app->win_pos_y);
      fclose (stream);
    }
    gtk_window_set_default_size (GTK_WINDOW (app->widgets.window),
				 app->win_width, app->win_height);
    main_toolbar_update ();
    extended_toolbar_update ();
    debug_toolbar_update ();
  }
  else
  {
    app = NULL;
    g_error (_("Cannot create application...exiting\n"));
  }
}

TextEditor *
anjuta_append_text_editor (gchar * filename)
{
  GtkWidget *label, *eventbox;
  TextEditor *te, *cur_page;
  gchar *buff;

  cur_page = anjuta_get_current_text_editor ();
  if (cur_page)
  {
    if (cur_page->mode == TEXT_EDITOR_PAGED)
    {
      te = text_editor_new (filename, cur_page->mode, NULL, app->preferences);
    }
    else
    {
      te =
	text_editor_new (filename, cur_page->mode,
			 &cur_page->allocation, app->preferences);
    }
  }
  else
  {
    te =
      text_editor_new (filename, TEXT_EDITOR_PAGED, NULL, app->preferences);
  }
  if (te)
  {
    anjuta_clear_windows_menu ();
    app->text_editor_list = g_list_append (app->text_editor_list, te);
    switch (te->mode)
    {
    case TEXT_EDITOR_PAGED:
      label = gtk_label_new (te->filename);
      gtk_widget_show (label);
      eventbox = gtk_event_box_new ();
      gtk_container_add (GTK_CONTAINER (eventbox), te->widgets.client);
      gtk_widget_show (eventbox);
      gtk_notebook_prepend_page (GTK_NOTEBOOK (app->widgets.notebook),
				 eventbox, label);
      buff = g_strdup_printf (_("Anjuta: %s"), te->full_filename);
      gtk_window_set_title (GTK_WINDOW (app->widgets.window), buff);
      g_free (buff);
      gtk_notebook_set_page (GTK_NOTEBOOK (app->widgets.notebook), 0);
      main_toolbar_update ();
      extended_toolbar_update ();
      break;

    case TEXT_EDITOR_WINDOWED:
      gtk_widget_show (te->widgets.window);
      break;
    }
    anjuta_fill_windows_menu ();
  }
  return te;
}

static void
on_anjuta_window_selected (GtkMenuItem * menuitem, gpointer user_data)
{
  anjuta_show_text_editor ((TextEditor *) user_data);
}

void
anjuta_remove_current_text_editor ()
{
  TextEditor *te;
  GtkWidget *submenu;
  gint nb_cur_page_num;

  te = anjuta_get_current_text_editor ();
  if (te == NULL)
    return;

  anjuta_clear_windows_menu ();
  if (te->full_filename != NULL)
  {
    app->recent_files =
      update_string_list (app->recent_files, te->full_filename,
			  app->preferences->max_recent_files);
    submenu =
      create_submenu (_("Recent Files "), app->recent_files,
		      GTK_SIGNAL_FUNC (on_recent_files_menu_item_activate));
    gtk_menu_item_set_submenu (GTK_MENU_ITEM
			       (app->widgets.menubar.file.recent_files),
			       submenu);
  }
  switch (te->mode)
  {
  case TEXT_EDITOR_PAGED:
    gtk_signal_disconnect_by_func (GTK_OBJECT (app->widgets.notebook),
				   GTK_SIGNAL_FUNC
				   (on_anjuta_notebook_switch_page), NULL);
    nb_cur_page_num =
      gtk_notebook_get_current_page (GTK_NOTEBOOK (app->widgets.notebook));
    gtk_container_remove (GTK_CONTAINER
			  (gtk_notebook_get_nth_page
			   (GTK_NOTEBOOK (app->widgets.notebook),
			    nb_cur_page_num)), te->widgets.client);
    gtk_notebook_remove_page (GTK_NOTEBOOK (app->widgets.notebook),
			      nb_cur_page_num);
    gtk_container_add (GTK_CONTAINER (te->widgets.client_area),
		       te->widgets.client);
    on_anjuta_window_focus_in_event (NULL, NULL, NULL);
    app->text_editor_list = g_list_remove (app->text_editor_list, te);
    main_toolbar_update ();
    extended_toolbar_update ();
    anjuta_update_title ();
    gtk_signal_connect (GTK_OBJECT (app->widgets.notebook), "switch_page",
			GTK_SIGNAL_FUNC (on_anjuta_notebook_switch_page),
			NULL);
    break;

  case TEXT_EDITOR_WINDOWED:
    app->text_editor_list = g_list_remove (app->text_editor_list, te);
    break;
  }
  if (te->full_filename)
  {
    if (app->project_dbase->project_is_open == FALSE)
      tags_manager_remove (app->tags_manager, te->full_filename);
  }
  anjuta_fill_windows_menu ();
  text_editor_destroy (te);
}

TextEditor *
anjuta_get_current_text_editor ()
{
  return app->current_text_editor;
}

TextEditor *
anjuta_get_notebook_text_editor (gint page_num)
{
  gint i;
  GtkWidget *page;
  GtkWidget *client;
  TextEditor *te;

  page =
    gtk_notebook_get_nth_page (GTK_NOTEBOOK (app->widgets.notebook),
			       page_num);
  client = GTK_BIN (page)->child;
  for (i = 0; i < g_list_length (app->text_editor_list); i++)
  {
    te = g_list_nth_data (app->text_editor_list, i);
    if (te->widgets.client == client)
      return te;
  }
  return NULL;
}

GtkAnText *
anjuta_get_current_text ()
{
  TextEditor *t = anjuta_get_current_text_editor ();
  if (t)
    return GTK_ANTEXT (t->widgets.editor);
  return NULL;
}

gchar *
anjuta_get_current_selection ()
{
  GtkAnText *text;
  GtkEditable *ed;
  gchar *chars;

  text = anjuta_get_current_text ();
  if (text == NULL)
    return NULL;

  ed = GTK_EDITABLE (text);
  if (ed->has_selection)
  {
    chars =
      gtk_editable_get_chars (ed, ed->selection_start_pos,
			      ed->selection_end_pos);
    return chars;
  }
  return NULL;
}

void
anjuta_goto_file_line (gchar * fname, guint lineno)
{
  gchar *fn;
  gint i;
  TextEditor *te;

  if (!fname)
    return;
  fn = anjuta_get_full_filename (fname);
  if (!fn)
    return;
  for (i = 0; i < g_list_length (app->text_editor_list); i++)
  {
    te = (TextEditor *) (g_list_nth (app->text_editor_list, i)->data);
    if (te->full_filename == NULL)
      continue;
    if (strcmp (fn, te->full_filename) == 0)
    {
      if (lineno >= 0)
	text_editor_goto_line (te, lineno);
      anjuta_show_text_editor (te);
      g_free (fn);
      anjuta_grab_text_focus (-1);
      return;
    }
  }
  te = anjuta_append_text_editor (fn);
  if (lineno >= 0 && te)
    text_editor_goto_line (te, lineno);
  g_free (fn);
  return;
}

void
anjuta_show_text_editor (TextEditor * te)
{
  GtkWidget *page;
  gint i;

  if (te == NULL)
    return;
  if (te->mode == TEXT_EDITOR_WINDOWED)
  {
    gdk_window_raise (te->widgets.window->window);
    return;
  }
  for (i = 0;
       i <
       g_list_length (GTK_NOTEBOOK (app->widgets.notebook)->children); i++)
  {
    page =
      gtk_notebook_get_nth_page (GTK_NOTEBOOK (app->widgets.notebook), i);
    if (GTK_BIN (page)->child == te->widgets.client)
    {
      gtk_notebook_set_page (GTK_NOTEBOOK (app->widgets.notebook), i);
      return;
    }
  }
}

void
anjuta_show ()
{
  gtk_widget_show (app->widgets.window);

  app->widgets.toolbar.main_toolbar.dock_item =
    gnome_dock_get_item_by_name (GNOME_DOCK (GNOME_APP (app->widgets.window)->dock),
				 "toolbar1", NULL, NULL, NULL, NULL);
  app->widgets.toolbar.extended_toolbar.dock_item =
    gnome_dock_get_item_by_name (GNOME_DOCK (GNOME_APP (app->widgets.window)->dock),
				 "toolbar2", NULL, NULL, NULL, NULL);
  app->widgets.toolbar.browser_toolbar.dock_item =
    gnome_dock_get_item_by_name (GNOME_DOCK (GNOME_APP (app->widgets.window)->dock),
				 "toolbar3", NULL, NULL, NULL, NULL);
  app->widgets.toolbar.debug_toolbar.dock_item =
    gnome_dock_get_item_by_name (GNOME_DOCK (GNOME_APP (app->widgets.window)->dock),
				 "toolbar4", NULL, NULL, NULL, NULL);

  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (app->widgets.menubar.view.main_toolbar),
				  app->widgets.toolbar.main_toolbar.is_showing);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (app->widgets.menubar.view.extended_toolbar),
                                   app->widgets.toolbar.extended_toolbar.is_showing);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (app->widgets.menubar.view.debug_toolbar),
				  app->widgets.toolbar.debug_toolbar.is_showing);
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (app->widgets.menubar.view.browser_toolbar),
				  app->widgets.toolbar.browser_toolbar.is_showing);

  if (app->widgets.toolbar.main_toolbar.is_showing == FALSE)
    gtk_widget_hide(GTK_WIDGET(app->widgets.toolbar.main_toolbar.dock_item));

  if (app->widgets.toolbar.browser_toolbar.is_showing == FALSE)
    gtk_widget_hide(GTK_WIDGET(app->widgets.toolbar.browser_toolbar.dock_item));

  if (app->widgets.toolbar.extended_toolbar.is_showing == FALSE)
    gtk_widget_hide(GTK_WIDGET(app->widgets.toolbar.extended_toolbar.dock_item));

  if (app->widgets.toolbar.debug_toolbar.is_showing == FALSE)
    gtk_widget_hide(GTK_WIDGET(app->widgets.toolbar.debug_toolbar.dock_item));

  update_gtk ();
  messages_append (app->messages,
		   _("CVS is not yet implemented in this version: Wait for the next release. :-(\n"),
		   MESSAGE_CVS);
  anjuta_load_cmdline_files ();
  anjuta_find_missings (TRUE);
  if (app->dirs->first_time)
  {
    gchar *file;
    app->dirs->first_time = FALSE;
    file = g_strconcat (app->dirs->data, "/welcome.txt", NULL);
    anjuta_info_show_file (file, 500, 435);
    g_free (file);
  }
}

void
anjuta_save_settings ()
{
  gchar *buffer;
  FILE *stream;

  if (!app)
    return;
  buffer = g_strconcat (app->dirs->settings, "/anjuta.set", NULL);
  stream = fopen (buffer, "w");
  g_free (buffer);
  if (stream)
  {
    anjuta_save_yourself (stream);
    fclose (stream);
  }
}

gboolean anjuta_save_yourself (FILE * stream)
{
  gint i;
  fprintf (stream,
	   "# **************************************************************************\n");
  fprintf (stream,
	   _
	   ("# *********** DO NOT EDIT OR RENAME THIS FILE *****************\n"));
  fprintf (stream,
	   _
	   ("# ******************* Created by  Anjuta ********************************\n"));
  fprintf (stream,
	   _
	   ("# **************** Anjuta global settings file ****************************\n"));
  fprintf (stream,
	   "# **************************************************************************\n\n");
  fprintf (stream, "Anjuta %s\n", VERSION);
  gdk_window_get_root_origin (app->widgets.window->window,
			      &app->win_pos_x, &app->win_pos_y);
  gdk_window_get_size (app->widgets.window->window, &app->win_width,
		       &app->win_height);
  fprintf (stream, "%d %d %d %d\n", app->win_pos_x, app->win_pos_y,
	   app->win_width, app->win_height);

  fprintf (stream, "%d\n", GTK_PANED (app->widgets.vpaned)->child1_size);
  fprintf (stream, "%d\n", GTK_PANED (app->widgets.hpaned)->child1_size);

  fprintf (stream, "%d\n", g_list_length (app->recent_files));
  for (i = 0; i < g_list_length (app->recent_files); i++)
  {
    fprintf (stream, "%s\n",
	     (gchar *) (g_list_nth (app->recent_files, i)->data));
  }
  fprintf (stream, "%d\n", g_list_length (app->recent_projects));
  for (i = 0; i < g_list_length (app->recent_projects); i++)
  {
    fprintf (stream, "%s\n",
	     (gchar *) (g_list_nth (app->recent_projects, i)->data));
  }
  if (!messages_save_yourself (app->messages, stream))
    return FALSE;
  if (!project_dbase_save_yourself (app->project_dbase, stream))
    return FALSE;
  preferences_save_yourself (app->preferences);

  if (!compiler_options_save_yourself (app->compiler_options, stream))
    return FALSE;
  if (app->project_dbase->project_is_open == FALSE)
    compiler_options_save (app->compiler_options, NULL);

  if (!src_paths_save_yourself (app->src_paths, stream))
    return FALSE;
  if (app->project_dbase->project_is_open == FALSE)
    src_paths_save (app->src_paths, NULL);

  if (!find_replace_save_yourself (app->find_replace, stream))
    return FALSE;
  if (!debugger_save_yourself (stream))
    return FALSE;
  return TRUE;
}

gboolean anjuta_load_yourself (FILE * stream)
{
  gint i = 0;
  gint ch, length;
  gchar buffer[256];
  while (i < 6)
  {
    if ((ch = fgetc (stream)) == EOF)
      return FALSE;
    if (ch == '\n')
      i++;
  }
  if (fscanf (stream, "Anjuta %s\n", buffer) < 1)
    return FALSE;
  if (strcmp (buffer, VERSION) != 0)
    return FALSE;

  if (fscanf
      (stream, "%d %d %d %d\n", &app->win_pos_x, &app->win_pos_y,
       &app->win_width, &app->win_height) < 4)
    return FALSE;

  if (fscanf (stream, "%d\n", &length) < 1)
    return FALSE;
  gtk_paned_set_position (GTK_PANED (app->widgets.vpaned), length);
  if (fscanf (stream, "%d\n", &length) < 1)
    return FALSE;
  gtk_paned_set_position (GTK_PANED (app->widgets.hpaned), length);

  if (app->recent_files != NULL)
  {
    for (i = 0; i < g_list_length (app->recent_files); i++)
    {
      g_free (g_list_nth (app->recent_files, i)->data);
    }
    g_list_free (app->recent_files);
  }
  if (fscanf (stream, "%d\n", &length) < 1)
    return FALSE;
  for (i = 0; i < length; i++)
  {
    if (fscanf (stream, "%s\n", buffer) < 1)
      return FALSE;
    app->recent_files = g_list_append (app->recent_files, g_strdup (buffer));
  }
  if (app->recent_projects != NULL)
  {
    for (i = 0; i < g_list_length (app->recent_projects); i++)
    {
      g_free (g_list_nth (app->recent_projects, i)->data);
    }
    g_list_free (app->recent_projects);
  }
  if (fscanf (stream, "%d\n", &length) < 1)
    return FALSE;
  for (i = 0; i < length; i++)
  {
    if (fscanf (stream, "%s\n", buffer) < 1)
      return FALSE;
    app->recent_projects =
      g_list_append (app->recent_projects, g_strdup (buffer));
  }
  if (!messages_load_yourself (app->messages, stream))
    return FALSE;
  if (!project_dbase_load_yourself (app->project_dbase, stream))
    return FALSE;
  preferences_load_yourself (app->preferences);

  if (!compiler_options_load_yourself (app->compiler_options, stream))
    return FALSE;
  if (!compiler_options_load (app->compiler_options, NULL))
    return FALSE;

  if (!src_paths_load_yourself (app->src_paths, stream))
    return FALSE;
  if (!src_paths_load (app->src_paths, NULL))
    return FALSE;

  if (!find_replace_load_yourself (app->find_replace, stream))
    return FALSE;
  if (!debugger_load_yourself (stream))
    return FALSE;
  return TRUE;
}

void
anjuta_update_title ()
{
  gchar *buff1, *buff2;
  GtkWidget *window;
  TextEditor *te;
  te = anjuta_get_current_text_editor ();
  if (te)
  {
    if (te->mode == TEXT_EDITOR_PAGED)
    {
      window = app->widgets.window;
      main_toolbar_update ();
      extended_toolbar_update ();
    }
    else
      window = te->widgets.window;

    if (te->status == TEXT_EDITOR_SAVED)
    {
      buff1 = g_strdup_printf (_("Anjuta: %s(Saved)"), te->full_filename);
      buff2 = g_strdup_printf (_("Anjuta: %s"), te->filename);
    }
    else
    {
      buff1 = g_strdup_printf (_("Anjuta: %s(Unsaved)"), te->full_filename);
      buff2 = g_strdup_printf (_("Anjuta: %s"), te->filename);
    }
    if (te->full_filename)
    {
      gtk_window_set_title (GTK_WINDOW (window), buff1);
      tags_manager_set_filename (app->tags_manager, te->full_filename);
    }
    else
      gtk_window_set_title (GTK_WINDOW (window), buff2);
    g_free (buff1);
    g_free (buff2);
  }
  else
    gtk_window_set_title (GTK_WINDOW (app->widgets.window),
			  _("Anjuta: No file"));
}

void
anjuta_apply_preferences (void)
{
  TextEditor *te;
  gint i;
  Preferences *pr = app->preferences;
  if (pr->no_tag == TRUE)
  {
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (app->widgets.notebook), FALSE);
  }
  else
  {
    switch (pr->tag_pos)
    {
    case TAG_POS_BOTTOM:
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (app->widgets.notebook),
				GTK_POS_BOTTOM);
      break;
    case TAG_POS_LEFT:
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (app->widgets.notebook),
				GTK_POS_LEFT);
      break;
    case TAG_POS_RIGHT:
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (app->widgets.notebook),
				GTK_POS_RIGHT);
      break;
    case TAG_POS_TOP:
    default:
      gtk_notebook_set_tab_pos (GTK_NOTEBOOK (app->widgets.notebook),
				GTK_POS_TOP);
    }
    gtk_notebook_set_show_tabs (GTK_NOTEBOOK (app->widgets.notebook), TRUE);
  }
  messages_update (app->messages);
  for (i = 0; i < g_list_length (app->text_editor_list); i++)
  {
    te = (TextEditor *) (g_list_nth (app->text_editor_list, i)->data);
    text_editor_update_preferences (te);
  }
}

void
anjuta_clean_exit ()
{
  TextEditor *te;
  gint i;
  gchar *tmp;

  if (app)
  {
    GList *list;
    pid_t pid;

    list = app->prog_list;
    while (list)
    {
      if (list->data)
	g_free (list->data);
      list = g_list_next (list);
    }
    g_list_free (app->prog_list);

    /* This will remove all tmp files */
    tmp =
      g_strdup_printf ("rm -f %s/anjuta_*.%ld", app->dirs->tmp,
		       (long) getpid ()); if ((pid = fork ()) == 0)
    {
      execlp (app->preferences->commands.shell,
	      app->preferences->commands.shell, "-c", tmp, NULL);
      g_error (_("Cannot execute command shell"));
    }
    waitpid (pid, NULL, 0);

    gtk_widget_hide (app->widgets.notebook);
    for (i = 0;
	 i <
	 g_list_length (GTK_NOTEBOOK (app->widgets.notebook)->children); i++)
    {
      te = anjuta_get_notebook_text_editor (i);
      gtk_container_remove (GTK_CONTAINER
			    (gtk_notebook_get_nth_page
			     (GTK_NOTEBOOK (app->widgets.notebook), i)),
			    te->widgets.client);
      gtk_container_add (GTK_CONTAINER (te->widgets.client_area),
			 te->widgets.client);}
    for (i = 0; i < g_list_length (app->text_editor_list); i++)
    {
      te = (TextEditor *) (g_list_nth_data (app->text_editor_list, i));
      text_editor_destroy (te);
    }
    if (app->text_editor_list)
      g_list_free (app->text_editor_list);
    for (i = 0; i < g_list_length (app->recent_files); i++)
    {
      g_free (g_list_nth_data (app->recent_files, i));
    }
    if (app->recent_files)
      g_list_free (app->recent_files);
    for (i = 0; i < g_list_length (app->recent_projects); i++)
    {
      g_free (g_list_nth_data (app->recent_projects, i));
    }
    if (app->recent_projects)
      g_list_free (app->recent_projects);
    if (app->fileselection)
      gtk_widget_destroy (app->fileselection);
    if (app->save_as_fileselection)
      gtk_widget_destroy (app->save_as_fileselection);
    if (app->preferences)
      preferences_destroy (app->preferences);
    if (app->compiler_options)
      compiler_options_destroy (app->compiler_options);
    if (app->src_paths)
      src_paths_destroy (app->src_paths);
    if (app->messages)
      messages_destroy (app->messages);
    if (app->project_dbase)
      project_dbase_destroy (app->project_dbase);
    if (app->find_replace)
      find_replace_destroy (app->find_replace);
    if (app->find_in_files)
      find_in_files_destroy (app->find_in_files);
    if (app->dirs)
      anjuta_dirs_destroy (app->dirs);
    if (app->executer)
      executer_destroy (app->executer);
    if (app->configurer)
      configurer_destroy (app->configurer);
    if (app->tags_manager)
      tags_manager_destroy (app->tags_manager);
    debugger_shutdown ();
    app->text_editor_list = NULL;
    app->fileselection = NULL;
    app->save_as_fileselection = NULL;
    app->preferences = NULL;
    main_menu_unref ();
    app = NULL;
  }
  gtk_main_quit ();
}


void
anjuta_status (gchar * mesg)
{
  gnome_app_flash (GNOME_APP (app->widgets.window), mesg);
}

void
anjuta_warning (gchar * mesg)
{
  gnome_app_warning (GNOME_APP (app->widgets.window), mesg);
}


void
anjuta_error (gchar * mesg)
{
  gnome_app_error (GNOME_APP (app->widgets.window), mesg);
}

void
anjuta_open_pixmap (gchar * filename)
{
  Preferences *pref;
  pid_t pid;
  if (!filename)
    return;
  pref = app->preferences;

  if (anjuta_is_installed (pref->commands.pixmap_editor, TRUE))
  {
    if ((pid = fork ()) == 0)
    {
      execlp (pref->commands.pixmap_editor,
	      pref->commands.pixmap_editor, filename, NULL);
      g_error (_("Cannot execute pixmap editor"));
    }
    if (pid > 0)
      anjuta_register_child_process (pid, NULL, NULL);
  }
}

gchar *
anjuta_get_full_filename (gchar * fn)
{
  gchar *cur_dir, *dummy;
  GList *list;
  gchar *text;
  gint i;
  if (!fn)
    return NULL;
  cur_dir = g_get_current_dir ();
  if (fn[0] == '/')
    return g_strdup (fn);
  if (app->project_dbase->project_is_open)
  {
    dummy = g_strconcat (app->project_dbase->src_dir, "/", fn, NULL);
    if (file_is_regular (dummy) == TRUE)
    {
      g_free (cur_dir);
      return dummy;
    }
    g_free (dummy);
  }
  else
  {
    dummy = g_strconcat (cur_dir, "/", fn, NULL);
    if (file_is_regular (dummy) == TRUE)
    {
      g_free (cur_dir);
      return dummy;
    }
    g_free (dummy);
  }
  list = GTK_CLIST (app->src_paths->widgets.src_clist)->row_list;
  for (i = 0; i < g_list_length (list); i++)
  {
    gtk_clist_get_text (GTK_CLIST (app->src_paths->widgets.src_clist), i,
			0, &text);
    if (app->project_dbase->project_is_open)
      dummy =
	g_strconcat (app->project_dbase->top_proj_dir, "/", text, "/", fn,
		     NULL);
    else
      dummy = g_strconcat (cur_dir, "/", text, "/", fn, NULL);
    if (file_is_regular (dummy) == TRUE)
    {
      g_free (cur_dir);
      return dummy;
    }
    g_free (dummy);
  }
  if (app->project_dbase->project_is_open)
  {

    dummy = g_strconcat (app->project_dbase->doc_dir, "/", fn, NULL);
    if (file_is_regular (dummy) == TRUE)
    {
      g_free (cur_dir);
      return dummy;
    }
    g_free (dummy);
    dummy = g_strconcat (app->project_dbase->pix_dir, "/", fn, NULL);
    if (file_is_regular (dummy) == TRUE)
    {
      g_free (cur_dir);
      return dummy;
    }
    g_free (dummy);
  }
  dummy = g_strconcat (cur_dir, "/", fn, NULL);

  g_free (cur_dir);
  return dummy;
}

gboolean
anjuta_init_progress (gchar * description, gdouble full_value,
		      GnomeAppProgressCancelFunc cancel_cb, gpointer data)
{
  if (app->in_progress)
    return FALSE;
  app->in_progress = TRUE;
  app->progress_value = 0.0;
  app->progress_full_value = full_value;
  app->progress_cancel_cb = cancel_cb;
  app->progress_key =
    gnome_app_progress_timeout (GNOME_APP (app->widgets.window),
				description,
				100,
				on_anjuta_progress_cb,
				on_anjuta_progress_cancel, data);
  return TRUE;
}

void
anjuta_set_progress (gdouble value)
{
  if (app->in_progress)

    app->progress_value = value / app->progress_full_value;
}

void
anjuta_done_progress (gchar * end_mesg)
{
  if (app->in_progress)
    gnome_app_progress_done (app->progress_key);
  anjuta_status (end_mesg);
  app->in_progress = FALSE;
}

void
anjuta_not_implemented (char *file, guint line)
{
  gchar *mesg =
    g_strdup_printf (N_("Not yet implemented.\nInsert code at \"%s:%u\""),
		     file, line);
  messagebox (GNOME_MESSAGE_BOX_INFO, _(mesg));
  g_free (mesg);
}

void
anjuta_set_busy ()
{
  GnomeAnimator *led;
  led = GNOME_ANIMATOR (app->widgets.toolbar.main_toolbar.led);
  app->busy_count++;
  if (app->busy_count > 1)
    return;
  if (app_cursor)
    gdk_cursor_destroy (app_cursor);
  gnome_animator_stop (led);
  gnome_animator_goto_frame (led, 1);
  app_cursor = gdk_cursor_new (GDK_WATCH);
  gdk_window_set_cursor (app->widgets.window->window, app_cursor);
  gdk_flush ();
}

void
anjuta_set_active ()
{
  app->busy_count--;
  if (app->busy_count > 0)
    return;
  app->busy_count = 0;
  if (app_cursor)
    gdk_cursor_destroy (app_cursor);
  app_cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
  gdk_window_set_cursor (app->widgets.window->window, app_cursor);
  update_led_animator ();
}

void
anjuta_grab_text_focus (gint page)
{
  GtkWidget *text;
  if (page >= 0
      && page <
      g_list_length (GTK_NOTEBOOK (app->widgets.notebook)->children))
    app->current_text_editor = anjuta_get_notebook_text_editor (page);
  text = GTK_WIDGET (anjuta_get_current_text ());
  if (text == NULL)
    return;
  gtk_widget_grab_focus (text);
  main_toolbar_update ();

  extended_toolbar_update ();
  anjuta_update_title ();
}

void
anjuta_register_window (GtkWidget * win)
{
  app->registered_windows = g_list_append (app->registered_windows, win);
}

void
anjuta_unregister_window (GtkWidget * win)
{
  app->registered_windows = g_list_remove (app->registered_windows, win);
}

void
anjuta_foreach_windows (GFunc cb, gpointer data)
{
  g_list_foreach (app->registered_windows, cb, data);
}

void
anjuta_register_child_process (pid_t pid,
			       void (*ch_terminated) (int s,
						      gpointer d),
			       gpointer data)
{
  if (pid < 1)
    return;
  app->registered_child_processes =
    g_list_append (app->registered_child_processes, (int *) pid);
  app->registered_child_processes_cb =
    g_list_append (app->registered_child_processes_cb, ch_terminated);
  app->registered_child_processes_cb_data =
    g_list_append (app->registered_child_processes_cb_data, data);
}

void
anjuta_unregister_child_process (pid_t pid)
{
  gint index;
  index = g_list_index (app->registered_child_processes, (int *) pid);
  app->registered_child_processes =
    g_list_remove (app->registered_child_processes, (int *) pid);
  app->registered_child_processes_cb =
    g_list_remove (app->registered_child_processes_cb,
		   g_list_nth_data (app->registered_child_processes_cb,
				    index));
  app->registered_child_processes_cb_data =
    g_list_remove (app->registered_child_processes_cb_data,
		   g_list_nth_data (app->registered_child_processes_cb_data,

				    index));}

void
anjuta_foreach_child_processes (GFunc cb, gpointer data)
{
  g_list_foreach (app->registered_child_processes, cb, data);
}

void
anjuta_child_terminated (int t)
{
  int status;
  gint index;
  pid_t pid;
  int (*callback) (int st, gpointer d);
  gpointer cb_data;
  pid = waitpid (0, &status, WNOHANG);
  if (pid < 1)
    return;
  index = g_list_index (app->registered_child_processes, (int *) pid);
  if (index < 0)
    return;
  callback = g_list_nth_data (app->registered_child_processes_cb, index);
  cb_data = g_list_nth_data (app->registered_child_processes_cb, index);
  if (callback)
    (*callback) (status, cb_data);
  anjuta_unregister_child_process (pid);
}

void
anjuta_load_cmdline_files ()
{
  int i;
  for (i = 1; i < arg_c; i++)
  {
    switch (get_file_ext_type (arg_v[i]))
    {
    case FILE_EXT_TYPE_XPM:
    case FILE_EXT_TYPE_JPG:
    case FILE_EXT_TYPE_BMP:
    case FILE_EXT_TYPE_TIFF:
    case FILE_EXT_TYPE_GIF:
    case FILE_EXT_TYPE_PNG:
      continue;
    case FILE_EXT_TYPE_PRJ:
      if (!app->project_dbase->project_is_open)
      {
	gtk_file_selection_set_filename (GTK_FILE_SELECTION
					 (app->project_dbase->
					  fileselection_open), arg_v[i]);
	project_dbase_load_project (app->project_dbase);
      }
      continue;
    default:
      anjuta_goto_file_line (arg_v[i], -1);
      continue;
    }
  }
}

void
anjuta_clear_windows_menu ()
{
  guint count;
  count = g_list_length (app->text_editor_list);
  if (count < 1)
    return;
  gnome_app_remove_menus (GNOME_APP (app->widgets.window),
			  "_Windows/<Separator>", count + 1);
}

void
anjuta_fill_windows_menu ()
{
  gint count, i;
  GnomeUIInfo wininfo[] = {
    {
     GNOME_APP_UI_ITEM, NULL,
     N_("Activate this to select this window"),
     on_anjuta_window_selected, NULL, NULL,
     GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_FORWARD, 0, 0, NULL}
    , GNOMEUIINFO_END
  };
  GnomeUIInfo sep[] = {
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_END
  };
  count = g_list_length (app->text_editor_list);
  if (count > 0)
  {
    gnome_app_insert_menus (GNOME_APP (app->widgets.window),
			    "_Windows/Cl_ose Current window", sep);
    for (i = (count - 1); i >= 0; i--)
    {
      TextEditor *te;
      te = g_list_nth_data (app->text_editor_list, i);
      wininfo[0].label = te->filename;
      if (te == NULL)
	continue;
      wininfo[0].user_data = te;
      gnome_app_insert_menus (GNOME_APP (app->widgets.window),
			      "_Windows/<Separator>", wininfo);
    }
  }
}

gboolean
anjuta_set_auto_gtk_update (gboolean auto_flag)
{
  gboolean save;
  save = app->auto_gtk_update;

  app->auto_gtk_update = auto_flag;
  return save;
}

void
anjuta_find_missings (gboolean show)
{
  GList *list;
  gchar *i_file, *o_file;
  int status;
  pid_t pid;
  FILE *fp;
  gchar buffer[1024];
  i_file = g_strconcat (app->dirs->data, "/programs", NULL);
  o_file = get_a_tmp_file ();
  list = app->prog_list;
  while (list)
  {
    if (list->data)
      g_free (list->data);
    list = g_list_next (list);
  }
  g_list_free (app->prog_list);
  app->prog_list = NULL;
  if ((pid = fork ()) == 0)
  {
    execlp ("anjuta_missing", "anjuta_missing", i_file, o_file, NULL);
    g_error ("Fatal: Unable to execute anjuta_missing");
  }
  waitpid (pid, &status, 0);
  if (WEXITSTATUS (status) != 0)
  {
    if (show)
      anjuta_error (_("Cannot execute anjuta_missing\n"
		      "Make sure you have installed anjuta properly."));
    g_free (i_file);
    g_free (o_file);
    return;
  }
  fp = fopen (o_file, "r");
  if (fp == NULL)
  {
    if (show)
      anjuta_error (_("Cannot execute anjuta_missing\n"
		      "Make sure you have installed anjuta properly."));
    g_free (i_file);
    g_free (o_file);
    return;
  }
  while (fscanf (fp, "%s\n", buffer) == 1)
  {
    app->prog_list = g_list_append (app->prog_list, g_strdup (buffer));
  }
  fclose (fp);
}

gboolean
anjuta_is_installed (gchar * prog, gboolean show)
{
  GList *list;
  list = app->prog_list;
  while (list)
  {
    if (strcmp (list->data, prog) == 0)
      return TRUE;
    list = g_list_next (list);
  }
  if (show)
  {
    gchar *msg;
    msg =
      g_strdup_printf (_
		       ("You do not have \"%s\" installed in your box.\n"
			"Install it and then restart Anjuta."), prog);
    anjuta_error (msg);
    g_free (msg);
  }
  return FALSE;
}
