/*
    stack_trace.c
    Copyright (C) 2000  Kh. Naba Kumar Singh

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <gnome.h>
#include "support.h"
#include "debugger.h"
#include "messagebox.h"
#include "utilities.h"
#include "../pixmaps/pointer.xpm"

GdkPixmap *pointer_pix;
GdkBitmap *pointer_pix_mask;

StackTrace *
stack_trace_new ()
{
  StackTrace *st;
  GtkStyle *style;

  st = g_malloc (sizeof (StackTrace));
  if (st)
  {
    create_stack_trace_gui (st);
    st->current_index = -1;
    st->current_frame = 0;
    st->is_showing = FALSE;
    st->is_docked = FALSE;
    st->win_pos_x = 30;
    st->win_pos_y = 200;
    st->win_width = 400;
    st->win_height = 150;

    style = gtk_widget_get_style (st->widgets.window);
    pointer_pix =
      gdk_pixmap_create_from_xpm_d (st->widgets.window->window,
				    &pointer_pix_mask,
				    &style->bg[GTK_STATE_NORMAL],
				    pointer_xpm);
  }
  return st;
}

void
stack_trace_clear (StackTrace * st)
{
  gtk_clist_unselect_all(GTK_CLIST (st->widgets.clist));
  gtk_clist_clear (GTK_CLIST (st->widgets.clist));
  st->current_index = -1;
}

void
stack_trace_update (GList * outputs, gpointer data)
{
  StackTrace *st;
  gchar *ptr;
  gfloat adj_value;
  GtkAdjustment *adj;
  GList *list, *node;

  st = (StackTrace *) data;
  list = remove_blank_lines(outputs);
  adj = gtk_clist_get_vadjustment (GTK_CLIST (st->widgets.clist));
  if(st->current_frame > 0) adj_value = adj->value;
  else adj_value = 0;

  stack_trace_clear (debugger.stack);
  if (g_list_length (list) < 1) {g_list_free(list); return;}
  node = list->next;
  ptr = g_strdup ((gchar *) list->data);
  while(node)
  {
    gchar *line = (gchar *) node->data;
    node = g_list_next(node);
    if (isspace (line[0]))	/* line break */
    {
      gchar *tmp;
      tmp = ptr;
      ptr = g_strconcat (tmp, line, NULL);
      g_free (tmp);
    }
    else
    {
      add_frame (st, ptr);
      g_free (ptr);
      ptr = g_strdup (line);
    }
  }
  if (ptr)
  {
    add_frame (st, ptr);
    g_free (ptr);
    ptr = NULL;
  }
  gtk_adjustment_set_value (adj, adj_value);
  g_list_free(list);
}

void
add_frame (StackTrace * st, gchar * line)
{
  gchar frame[10], *dummy_fn;
  gchar *row[3];
  gint count, last_row, dummy_int;
  GdkColor blue = { 16, 0, 0, -1 };
  GdkColor red = { 16, -1, 0, 0 };

  count = sscanf (line, "#%s", frame);
  if (count != 1)
    return;
  while (!isspace (*line))
    line++;
  while (isspace (*line))
    line++;
  row[0] = NULL;
  row[1] = frame;
  row[2] = line;
  gtk_clist_append (GTK_CLIST (st->widgets.clist), row);
  last_row = g_list_length (GTK_CLIST (st->widgets.clist)->row_list) - 1;
  if (last_row == st->current_frame)
  {
    gtk_clist_set_pixmap (GTK_CLIST (st->widgets.clist), last_row, 0,
			  pointer_pix, pointer_pix_mask);
  }
  if (parse_error_line (line, &dummy_fn, &dummy_int))
  {
    gtk_clist_set_foreground (GTK_CLIST (st->widgets.clist), last_row, &red);
    g_free (dummy_fn);
  }
  else
  {
    gtk_clist_set_foreground (GTK_CLIST (st->widgets.clist), last_row, &blue);
  }
}

void
stack_trace_show (StackTrace * st)
{
  if (st)
  {
    if (st->is_showing)
    {
      if (st->is_docked == FALSE)
	gdk_window_raise (st->widgets.window->window);
      return;
    }
    if (st->is_docked)
    {
      stack_trace_attach (st);
    }
    else			/* Is not docked */
    {
      gtk_widget_set_uposition (st->widgets.window, st->win_pos_x,
				st->win_pos_y);
      gtk_window_set_default_size (GTK_WINDOW (st->widgets.window),
				   st->win_width, st->win_height);
      gtk_widget_show (st->widgets.window);
    }
    st->is_showing = TRUE;
  }
}

void
stack_trace_hide (StackTrace * st)
{
  if (st)
  {
    if (st->is_showing == FALSE)
      return;
    if (st->is_docked == TRUE)
    {
      stack_trace_detach (st);
    }
    else			/* Is not docked */
    {
      gdk_window_get_root_origin (st->widgets.window->window, &st->win_pos_x,
				  &st->win_pos_y);
      gdk_window_get_size (st->widgets.window->window, &st->win_width,
			   &st->win_height);
      gtk_widget_hide (st->widgets.window);
    }
    st->is_showing = FALSE;
  }
}

void
stack_trace_attach (StackTrace * st)
{

}

void
stack_trace_detach (StackTrace * st)
{

}

void
stack_trace_dock (StackTrace * st)
{

}

void
stack_trace_undock (StackTrace * st)
{

}

gboolean stack_trace_save_yourself (StackTrace * st, FILE * stream)
{
  if (!st)
    return FALSE;

  if ( fprintf (stream, "Stack trace:\n") < 0) return FALSE;
  if (st->is_docked)
  {
    if ( fprintf (stream, "%d\n", 1) < 1)
      return FALSE;
  }
  else if ( fprintf (stream, "%d\n", 0) < 1)
    return FALSE;
  if (st->is_showing)
  {
    if (!st->is_docked)
    {
      gdk_window_get_root_origin (st->widgets.window->window, &st->win_pos_x,
				  &st->win_pos_y);
      gdk_window_get_size (st->widgets.window->window, &st->win_width,
			   &st->win_height);
    }
  }
  if ( fprintf
      (stream, "%d %d %d %d\n", st->win_pos_x, st->win_pos_y, st->win_width,
       st->win_height) < 4) return FALSE;
  return TRUE;
}

gboolean stack_trace_load_yourself (StackTrace * st, FILE * stream)
{
  gboolean dock_flag;
  if (!st)
    return FALSE;

  if ( fscanf (stream, "Stack trace:\n") < 0) return FALSE;
  if ( fscanf (stream, "%d\n", &dock_flag) < 1)
    return FALSE;
  if ( fscanf
      (stream, "%d %d %d %d\n", &st->win_pos_x, &st->win_pos_y,
       &st->win_width, &st->win_height) < 4) return FALSE;
  if (dock_flag)
    stack_trace_dock (st);
  else
    stack_trace_undock (st);
  return TRUE;
}

void
stack_trace_destroy (StackTrace * st)
{
  if (st)
  {
    stack_trace_clear (st);
    gtk_widget_unref (st->widgets.window);
    gtk_widget_unref (st->widgets.clist);
    gtk_widget_unref (st->widgets.menu);
    gtk_widget_unref (st->widgets.menu_set);
    gtk_widget_unref (st->widgets.menu_info);
    gtk_widget_unref (st->widgets.menu_update);
    gtk_widget_unref (st->widgets.menu_view);

    if (GTK_IS_WIDGET (st->widgets.window))
      gtk_widget_destroy (st->widgets.window);
    g_free (st);
  }
}

void
stack_trace_update_controls (StackTrace * st)
{
  gboolean A, R, S;

  A = debugger_is_active ();
  R = debugger_is_ready ();
  S = (st->current_index >= 0);
  gtk_widget_set_sensitive (st->widgets.menu_set, A && R && S);
  gtk_widget_set_sensitive (st->widgets.menu_info, A && R);
  gtk_widget_set_sensitive (st->widgets.menu_update, A && R);
  gtk_widget_set_sensitive (st->widgets.menu_view, A && R && S);
}
