/*
 * text_editor.c
 * Copyright (C) 2000  Kh. Naba Kumar Singh
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <gnome.h>
#include "gtkaneditor.h"

#include "global.h"
#include "support.h"
#include "anjuta.h"

#include "gtkaneditor.h"
#include "text_editor.h"
#include "text_editor_gui.h"
#include "text_editor_cbs.h"
#include "text_editor_menu.h"
#include "utilities.h"
#include "syntax.h"
#include "launcher.h"

#define    TEST_STRING   "X X X X"

TextEditor *
text_editor_new (gchar * filename, TextEditorMode mode,
		 GtkAllocation * allocation, Preferences * eo)
{
  gchar *buff;
  TextEditor *te;
  static guint new_file_count;

  te = (TextEditor *) g_malloc (sizeof (TextEditor));
  if (te == NULL)
    return NULL;
  te->undo_list = NULL;
  te->undo_index = 0;
  te->filename = g_strdup_printf ("Newfile#%d", ++new_file_count);
  te->full_filename = NULL;
  te->modified_time = time (NULL);
  te->mode = mode;
  te->status = TEXT_EDITOR_SAVED;
  te->preferences = eo;
  te->hilite_type = HILITE_NONE;
  te->menu = NULL;
  te->autosave_on = FALSE;
  te->autosave_it = 10;
  te->font = NULL;
  te->style = NULL;
  te->org_style = NULL;

  if (allocation)
  {
    te->allocation.x = allocation->x + 25;
    te->allocation.y = allocation->y + 80;
    te->allocation.width = allocation->width;
    te->allocation.height = allocation->height;
  }
  else
  {
    te->allocation.x = 50;
    te->allocation.y = 50;
    te->allocation.width = 600;
    te->allocation.height = 500;
  }
  create_text_editor_gui (te);
  te->org_style = gtk_style_copy (te->widgets.editor->style);
  te->style = gtk_style_copy (te->org_style);
  if (mode == TEXT_EDITOR_PAGED)
  {
    gtk_container_remove (GTK_CONTAINER (te->widgets.client_area),
			  te->widgets.client);
  }
  buff = g_strdup_printf ("Anjuta: %s", te->filename);
  gtk_window_set_title (GTK_WINDOW (te->widgets.window), buff);
  g_free (buff);
  if (filename)
  {
    new_file_count--;
    if (te->filename)
      g_free (te->filename);
    if (te->full_filename)
      g_free (te->full_filename);
    te->filename = g_strdup (extract_filename (filename));
    te->full_filename = g_strdup (filename);
    buff = g_strdup_printf ("Anjuta: %s", te->full_filename);
    gtk_window_set_title (GTK_WINDOW (te->widgets.window), buff);
    g_free (buff);
    if (text_editor_load_file (te) == FALSE)
    {
      buff =
	g_strdup_printf (_("Error: Could not load the file \"%s\".\n"
			   "Make sure the file exists with read permission."),
			 te->full_filename);
      anjuta_error (buff);
      g_free (buff);
      text_editor_destroy (te);
      return NULL;
    }
  }
  te->menu = text_editor_menu_new ();
  text_editor_update_preferences (te);
  text_editor_update_controls (te);
  return te;
}

void
text_editor_set_hilite_type (TextEditor * te)
{
  FileExtType fet;
  TextEditorHiliteType pre_hilite;

  if (!te)
    return;
  pre_hilite = te->hilite_type;
  fet = get_file_ext_type (te->filename);
  switch (fet)
  {
  case FILE_EXT_TYPE_C:
  case FILE_EXT_TYPE_CPP:
  case FILE_EXT_TYPE_IC:
  case FILE_EXT_TYPE_ICPP:
  case FILE_EXT_TYPE_H:
    te->hilite_type = HILITE_C;
    GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle = FALSE;
    if (pre_hilite != te->hilite_type && te->preferences->auto_hilite)
    {
      gtk_aneditor_hilite_buffer (GTK_ANEDITOR (te->widgets.editor));
    }
    GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle =
      te->preferences->auto_hilite;
    break;
  default:
    te->hilite_type = HILITE_NONE;
    GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle = FALSE;
  }
}

void
text_editor_destroy (TextEditor * te)
{
  gint i;
  if (te)
  {
    gtk_widget_hide (te->widgets.window);
    gtk_widget_hide (te->widgets.client);
    if (te->autosave_on)
       gtk_timeout_remove (te->autosave_id);
    gtk_widget_unref (te->widgets.window);
    gtk_widget_unref (te->widgets.client_area);
    gtk_widget_unref (te->widgets.client);
    gtk_widget_unref (te->widgets.line_label);
    gtk_widget_unref (te->widgets.editor);
    gtk_widget_unref (te->buttons.new);
    gtk_widget_unref (te->buttons.new);
    gtk_widget_unref (te->buttons.open);
    gtk_widget_unref (te->buttons.save);
    gtk_widget_unref (te->buttons.reload);
    gtk_widget_unref (te->buttons.cut);
    gtk_widget_unref (te->buttons.copy);
    gtk_widget_unref (te->buttons.paste);
    gtk_widget_unref (te->buttons.find);
    gtk_widget_unref (te->buttons.replace);
    gtk_widget_unref (te->buttons.compile);
    gtk_widget_unref (te->buttons.build);
    gtk_widget_unref (te->buttons.print);
    gtk_widget_unref (te->buttons.attach);
    gtk_style_unref (te->style);
    gtk_style_unref (te->org_style);

    if (te->widgets.window)
      gtk_widget_destroy (te->widgets.window);
    if (te->filename)
      g_free (te->filename);
    if (te->full_filename)
      g_free (te->full_filename);
    if (te->menu)
      text_editor_menu_destroy (te->menu);
    for (i = 0; i < g_list_length (te->undo_list); i++)
      text_editor_undo_item_destroy ((TextEditorUndoItem *)
				     g_list_nth_data (te->undo_list, i));
    g_list_free (te->undo_list);
    g_free (te);
    te = NULL;
  }
}

void
text_editor_undock (TextEditor * te, GtkWidget * container)
{
  if (!te)
    return;
  if (te->mode == TEXT_EDITOR_WINDOWED)
    return;
  te->mode = TEXT_EDITOR_WINDOWED;
  gtk_container_remove (GTK_CONTAINER (container), te->widgets.client);
  gtk_container_add (GTK_CONTAINER (te->widgets.client_area),
		     te->widgets.client);
  gtk_widget_show (te->widgets.client);
  gtk_widget_show (te->widgets.window);
}

void
text_editor_dock (TextEditor * te, GtkWidget * container)
{
  if (!te)
    return;
  if (te->mode == TEXT_EDITOR_PAGED)
    return;
  te->mode = TEXT_EDITOR_PAGED;
  gtk_widget_hide (te->widgets.window);
  gtk_container_remove (GTK_CONTAINER (te->widgets.client_area),
			te->widgets.client);
  gtk_container_add (GTK_CONTAINER (container), te->widgets.client);
  gtk_widget_show (te->widgets.client);
}

gint
text_editor_find (TextEditor * te, gchar * str, gboolean forward,
		  gboolean regexp, gboolean ignore_case)
{
  gint ret;
  GtkAnEditor *editor;

  if (!te)
    return FALSE;
  editor = GTK_ANEDITOR (te->widgets.editor);
  if (forward)			/* forward search */
  {
    if (regexp)			/* regexp search  */
      ret = gtk_aneditor_regex_search_from_point (editor, str);
    else			/* string search */
      ret = gtk_aneditor_search_from_point (editor, str, ignore_case);
  }
  else				/* backward search */
  {
    if (regexp)			/* regexp search */
      ret = gtk_aneditor_regex_search_back_from_point (editor, str);
    else			/* string search */
      ret = gtk_aneditor_search_back_from_point (editor, str, ignore_case);
  }
  return ret;
}

void
text_editor_replace (TextEditor * te)
{
  if (!te)
    return;
}

gboolean
text_editor_find_string (TextEditor * te, char *s)
{
  gchar *text, *found;
  guint cur_loc;
  guint index;
  if (strlen (s) == 0)
    return TRUE;
  if (!te)
    return FALSE;
  cur_loc = gtk_antext_get_point (GTK_ANTEXT (te->widgets.editor));
  text =
    gtk_editable_get_chars (GTK_EDITABLE (te->widgets.editor), cur_loc, -1);
  found = strstr (text, s);
  if (found)
  {
    index = (found - text) + cur_loc;
    text_editor_goto_point (te, index + strlen (s));
    gtk_editable_select_region (GTK_EDITABLE (te->widgets.editor), index,
				index + strlen (s));
    g_free (text);
    gtk_widget_grab_focus (te->widgets.editor);
    return TRUE;
  }
  else
  {
    g_free (text);
    gtk_widget_grab_focus (te->widgets.editor);
    return FALSE;
  }
}

guint
text_editor_get_total_lines (TextEditor * te)
{
  guint i;
  guint count = 0;
  if (te == NULL)
    return 0;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return 0;
  for (i = 0; i < gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor));
       i++)
  {
    if (GTK_ANTEXT_INDEX ((GTK_ANTEXT (te->widgets.editor)), i) == '\n')
      count++;
  }
  return count;
}

guint
text_editor_get_current_lineno (TextEditor * te)
{
  guint i;
  guint count = 0;
  if (te == NULL)
    return 0;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return 0;
  for (i = 0; i < gtk_antext_get_point (GTK_ANTEXT (te->widgets.editor)); i++)
  {
    if (GTK_ANTEXT_INDEX ((GTK_ANTEXT (te->widgets.editor)), i) == '\n')
      count++;
  }
  return count;
}

gboolean
text_editor_goto_point (TextEditor * te, guint point)
{
  float index;
  GtkAdjustment *adj;
  guint cur_line;
  if (te == NULL)
    return FALSE;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return FALSE;
  if (point < 0
      || point >
      gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor))) return FALSE;

  gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), point);
  cur_line = text_editor_get_current_lineno (te);
  adj = GTK_ANTEXT (te->widgets.editor)->vadj;
  index = adj->lower
    + (adj->upper - adj->lower) * cur_line / text_editor_get_total_lines (te)
    - adj->page_size / 2;
  index = (index > adj->lower) ? index : adj->lower;
  index =
    (index <
     (adj->upper) - (adj->page_size)) ? index : (adj->upper) -
    (adj->page_size);
  gtk_adjustment_set_value (adj, index);
  return TRUE;
}

gboolean
text_editor_goto_line (TextEditor * te, guint line)
{
  guint start_index, end_index;
  guint count = 1;
  if (te == NULL)
    return FALSE;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return FALSE;

  for (start_index = 0;
       start_index < gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor));
       start_index++)
  {
    if (count >= line)
      break;
    if (GTK_ANTEXT_INDEX ((GTK_ANTEXT (te->widgets.editor)), start_index) ==
	'\n') count++;
  }
  if (count == line)
  {
    gtk_antext_set_line_wrap (GTK_ANTEXT (te->widgets.editor), FALSE);
    gtk_antext_set_word_wrap (GTK_ANTEXT (te->widgets.editor), FALSE);

    for (end_index = start_index;
	 end_index <
	 gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor)); end_index++)
    {
      if (GTK_ANTEXT_INDEX ((GTK_ANTEXT (te->widgets.editor)), end_index)
	  == '\n')
	break;
    }
    text_editor_goto_point (te, start_index);
    gtk_editable_select_region (GTK_EDITABLE (te->widgets.editor),
				start_index, ++end_index);
    return TRUE;
  }
  return FALSE;
}

gboolean
text_editor_load_file (TextEditor * te)
{
  TextEditorHiliteType pre_hilite;
  FILE *fp;

  if (te == NULL)
    return FALSE;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return FALSE;


  fp = fopen (te->full_filename, "r");
  if (fp)
  {
    gchar buffer[FILE_BUFFER_SIZE];
    gint nchars;
    gfloat size;
    gulong count;
    struct stat st;

    anjuta_set_busy ();
    stat (te->full_filename, &st);
    size = (float) st.st_size / FILE_BUFFER_SIZE;
    count = 0;
    te->modified_time = time (NULL);
    GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle = FALSE;
    gtk_antext_freeze (GTK_ANTEXT (te->widgets.editor));
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), 0);
    gtk_antext_forward_delete (GTK_ANTEXT (te->widgets.editor),
			       gtk_antext_get_length (GTK_ANTEXT
						      (te->widgets.editor)));
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), 0);
    anjuta_status (_("Loading File ..."));
    while (1)
    {
      nchars = fread (buffer, 1, FILE_BUFFER_SIZE, fp);
      gtk_antext_insert (GTK_ANTEXT (te->widgets.editor),
			 te->style->font, NULL, NULL, buffer, nchars);
      count++;
      if (feof (fp))
	break;
    }
    fclose (fp);
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), 0);
    gtk_antext_thaw (GTK_ANTEXT (te->widgets.editor));
    pre_hilite = te->hilite_type;
    text_editor_set_hilite_type (te);

    /*
     * text_editor_set_patterns(te); 
     */

    if (te->preferences->auto_hilite
	&& te->hilite_type == pre_hilite && te->hilite_type != HILITE_NONE)
    {
      GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle = FALSE;
      gtk_aneditor_hilite_buffer (GTK_ANEDITOR (te->widgets.editor));
      GTK_ANEDITOR (te->widgets.editor)->hilite_when_idle = TRUE;
    }
    gtk_widget_grab_focus (te->widgets.editor);
    if (app->project_dbase->project_is_open == FALSE
	&& app->preferences->tags_update)
      tags_manager_update (app->tags_manager, te->full_filename);
    else
      if (project_dbase_file_is_src (app->project_dbase, te->full_filename)
	  && te->preferences->tags_update)
      tags_manager_update (app->tags_manager, te->full_filename);
    anjuta_status (_("File Loaded Successfully"));
    anjuta_set_active ();
    text_editor_update_controls (te);
    return TRUE;
  }
  return FALSE;
}

gboolean
text_editor_save_file (TextEditor * te)
{
  FILE *fp;

  if (te == NULL)
    return FALSE;
  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return FALSE;

  fp = fopen (te->full_filename, "w");
  if (fp)
  {
    gchar ch;
    guint nchars, i;

    anjuta_set_busy ();
    nchars = gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor));

    anjuta_status (_("Saving File ..."));
    for (i = 0; i < nchars; i++)
    {
      ch = (GTK_ANTEXT_INDEX (GTK_ANTEXT (te->widgets.editor), i));
      if (fwrite (&ch, 1, 1, fp) != 1)
      {
	fclose (fp);
	anjuta_warning (_("Error in Saving File ..."));
	anjuta_set_active ();
	text_editor_update_controls (te);
	return FALSE;
      }
    }
    fclose (fp);
    gtk_widget_grab_focus (te->widgets.editor);
    if (te->preferences->tags_update)
      tags_manager_update (app->tags_manager, te->full_filename);
    anjuta_status (_("File Saved successfully"));
    te->status = TEXT_EDITOR_SAVED;
    te->modified_time = time (NULL);
    anjuta_update_title ();
    text_editor_set_hilite_type (te);
    anjuta_set_active ();
    text_editor_update_controls (te);
    return TRUE;
  }
  anjuta_error (_("Error in Saving File ..."));
  return FALSE;
}

gboolean
text_editor_save_yourself (TextEditor * te, FILE * stream)
{
  return TRUE;
}

gboolean
text_editor_recover_yourself (TextEditor * te, FILE * stream)
{
  return TRUE;
}

void
text_editor_update_preferences (TextEditor * te)
{
  Preferences *pr;
  guint len;

  if (te == NULL)
    return;
  pr = te->preferences;

  if (GTK_IS_ANTEXT (te->widgets.editor) == FALSE)
    return;

  GTK_ANTEXT (te->widgets.editor)->default_tab_width
    = te->preferences->tab_size;
  text_editor_set_hilite_type (te);
  if (te->preferences->auto_save)
  {
    if (te->autosave_on == TRUE)
    {
      if (te->preferences->auto_save_timer != te->autosave_it)
      {
	gtk_timeout_remove (te->autosave_id);
	te->autosave_id =
	  gtk_timeout_add (te->preferences->auto_save_timer * 60000,
			   on_text_editor_auto_save, te);
      }
    }
    else
    {
      te->autosave_id =
	gtk_timeout_add (te->preferences->auto_save_timer * 60000,
			 on_text_editor_auto_save, te);
    }
    te->autosave_it = te->preferences->auto_save_timer;
    te->autosave_on = TRUE;
  }
  else
  {
    if (te->autosave_on == TRUE)
      gtk_timeout_remove (te->autosave_id);
    te->autosave_on = FALSE;
  }

  gtk_aneditor_set_colors (GTK_ANEDITOR (te->widgets.editor),
			   &pr->plain_color, &pr->comment_color,
			   &pr->string_color, &pr->keywords_color);
  if (pr->font != te->style->font)
  {
    if (pr->font)
    {
      if (pr->font)
	gdk_font_ref (pr->font);
      if (te->style->font)
	gdk_font_unref (te->style->font);
      te->style->font = pr->font;
    }
    else
    {
      if (te->org_style->font)
	gdk_font_ref (te->org_style->font);
      if (te->style->font)
	gdk_font_unref (te->style->font);
      te->style->font = te->org_style->font;
    }
    gtk_aneditor_set_fonts (GTK_ANEDITOR (te->widgets.editor),
			    te->style->font, te->style->font,
			    te->style->font, te->style->font);
    len = gtk_antext_get_length (GTK_ANTEXT (te->widgets.editor));
    if (len > 0)
    {
      gtk_aneditor_flush_properties (GTK_ANEDITOR (te->widgets.editor));
    }
    else
    {
      gtk_antext_insert (GTK_ANTEXT (te->widgets.editor),
			 te->preferences->font, NULL, NULL, " ", -1);
      gtk_antext_backward_delete (GTK_ANTEXT (te->widgets.editor), 1);
    }
  }
}

gboolean
text_editor_check_disk_status (TextEditor * te)
{
  struct stat status;
  gchar *buff;
  if (!te)
    return FALSE;
  if (stat (te->full_filename, &status))
    return TRUE;
  if (te->modified_time < status.st_mtime)
  {
    buff =
      g_strdup_printf (_
		       ("WARNING: The file \"%s\" in the disk is more recent than\nthe current buffer.\nDo you want to reload it."),
		       te->filename);
    messagebox2 (GNOME_MESSAGE_BOX_WARNING, buff, GNOME_STOCK_BUTTON_YES,
		 GNOME_STOCK_BUTTON_NO,
		 GTK_SIGNAL_FUNC (on_text_editor_check_yes_clicked),
		 GTK_SIGNAL_FUNC (on_text_editor_check_no_clicked), te);
    return FALSE;
  }
  return TRUE;
}

void
text_editor_undo_item_add (TextEditor * te, TextEditorUndoItem * ui)
{
  gint i;
  if (!te)
    return;
  if (!ui)
    return;
  for (i = 0; i < te->undo_index; i++)
  {
    TextEditorUndoItem *tui = g_list_nth_data (te->undo_list, 0);
    te->undo_list = g_list_remove (te->undo_list, tui);
    text_editor_undo_item_destroy (tui);
  }
  te->undo_list = g_list_prepend (te->undo_list, ui);
  te->undo_index = 0;
  if (g_list_length (te->undo_list) > app->preferences->max_undo_histories)
  {
    for (i = app->preferences->max_undo_histories;
	 i < g_list_length (te->undo_list); i++)
    {
      TextEditorUndoItem *tui = g_list_nth_data (te->undo_list,
						 app->preferences->
						 max_undo_histories);
      te->undo_list = g_list_remove (te->undo_list, tui);
      text_editor_undo_item_destroy (tui);
    }
  }
  gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.undo, TRUE);
  gtk_widget_set_sensitive (app->widgets.menubar.edit.undo, TRUE);
  gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.redo, FALSE);
  gtk_widget_set_sensitive (app->widgets.menubar.edit.redo, FALSE);
}

TextEditorUndoItem *
text_editor_undo_item_new (const gchar * text, gulong length, gulong pos,
			   TextEditorUndoItemType type)
{
  TextEditorUndoItem *ui;
  ui = g_malloc (sizeof (TextEditorUndoItem));
  if (ui)
  {
    ui->text = g_strdup (text);
    ui->length = length;
    ui->position = pos;
    ui->type = type;
  }
  return ui;
}

void
text_editor_undo_item_destroy (TextEditorUndoItem * ui)
{
  if (ui)
  {
    if (ui->text)
      g_free (ui->text);
    g_free (ui);
  }
}

void
text_editor_undo (TextEditor * te)
{
  TextEditorUndoItem *ui;
  gulong start, end;
  if (!te)
    return;
  if (te->undo_index >= g_list_length (te->undo_list))
    return;
  if ((te->undo_index + 1) >= g_list_length (te->undo_list))
  {
    gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.undo, FALSE);
    gtk_widget_set_sensitive (app->widgets.menubar.edit.undo, FALSE);
  }
  gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.redo, TRUE);
  gtk_widget_set_sensitive (app->widgets.menubar.edit.redo, TRUE);

  ui = g_list_nth_data (te->undo_list, te->undo_index);
  gtk_signal_disconnect (GTK_OBJECT (te->widgets.editor),
			 te->insert_signal_id);
  gtk_signal_disconnect (GTK_OBJECT (te->widgets.editor),
			 te->delete_signal_id);

  start = ui->position;
  end = ui->position + ui->length;


  switch (ui->type)
  {
  case INSERT:
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), start);
    gtk_editable_select_region (GTK_EDITABLE (te->widgets.editor),
				start, end);
    gtk_editable_delete_selection (GTK_EDITABLE (te->widgets.editor));
    gtk_editable_set_position (GTK_EDITABLE (te->widgets.editor), start);
    break;
  case DELETE:
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), start);
    gtk_antext_insert (GTK_ANTEXT (te->widgets.editor),
		       te->preferences->font, NULL, NULL, ui->text,
		       ui->length);
    gtk_editable_set_position (GTK_EDITABLE (te->widgets.editor), end);
    break;
  default:
    g_warning (_("Unknown Undo type\n"));
    break;
  }
  te->insert_signal_id =
    gtk_signal_connect (GTK_OBJECT (te->widgets.editor), "insert_text",
			GTK_SIGNAL_FUNC (on_text_editor_insert_text), te);
  te->delete_signal_id =
    gtk_signal_connect (GTK_OBJECT (te->widgets.editor), "delete_text",
			GTK_SIGNAL_FUNC (on_text_editor_delete_text), te);
  te->undo_index++;
}

void
text_editor_redo (TextEditor * te)
{
  TextEditorUndoItem *ui;
  gulong start, end;
  if (!te)
    return;
  if (te->undo_index <= 0)
    return;
  te->undo_index--;
  if (te->undo_index <= 0)
  {
    gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.redo, FALSE);
    gtk_widget_set_sensitive (app->widgets.menubar.edit.redo, FALSE);
  }
  gtk_widget_set_sensitive (app->widgets.toolbar.main_toolbar.undo, TRUE);
  gtk_widget_set_sensitive (app->widgets.menubar.edit.undo, TRUE);

  ui = g_list_nth_data (te->undo_list, te->undo_index);
  gtk_signal_disconnect (GTK_OBJECT (te->widgets.editor),
			 te->insert_signal_id);
  gtk_signal_disconnect (GTK_OBJECT (te->widgets.editor),
			 te->delete_signal_id);

  start = ui->position;
  end = ui->position + ui->length;


  switch (ui->type)
  {
  case INSERT:
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), start);
    gtk_antext_insert (GTK_ANTEXT (te->widgets.editor),
		       te->preferences->font, NULL, NULL, ui->text,
		       ui->length);
    gtk_editable_set_position (GTK_EDITABLE (te->widgets.editor), end);
    break;
  case DELETE:
    gtk_antext_set_point (GTK_ANTEXT (te->widgets.editor), start);
    gtk_editable_select_region (GTK_EDITABLE (te->widgets.editor),
				start, end);
    gtk_editable_delete_selection (GTK_EDITABLE (te->widgets.editor));
    gtk_editable_set_position (GTK_EDITABLE (te->widgets.editor), start);
    break;
  default:
    g_warning (_("Unknown Undo type\n"));
    break;
  }
  te->insert_signal_id =
    gtk_signal_connect (GTK_OBJECT (te->widgets.editor), "insert_text",
			GTK_SIGNAL_FUNC (on_text_editor_insert_text), te);
  te->delete_signal_id =
    gtk_signal_connect (GTK_OBJECT (te->widgets.editor), "delete_text",
			GTK_SIGNAL_FUNC (on_text_editor_delete_text), te);
}

void
text_editor_hilite (TextEditor * te)
{
  if (!te)
    return;
  if (te->hilite_type == HILITE_NONE)
    return;
  anjuta_set_busy ();
  gtk_aneditor_hilite_buffer (GTK_ANEDITOR (te->widgets.editor));
  anjuta_set_active ();
}

void
text_editor_update_controls (TextEditor * te)
{
  gboolean F, P, L, C, S;

  if (te == NULL)
    return;
  S = (te->status == TEXT_EDITOR_SAVED);
  L = launcher_is_busy ();
  P = app->project_dbase->project_is_open;

  switch (get_file_ext_type (te->filename))
  {
  case FILE_EXT_TYPE_C:
  case FILE_EXT_TYPE_CPP:
  case FILE_EXT_TYPE_ASM:
  case FILE_EXT_TYPE_IC:
  case FILE_EXT_TYPE_ICPP:
    F = TRUE;
    break;
  case FILE_EXT_TYPE_H:
  case FILE_EXT_TYPE_XPM:
  default:
    F = FALSE;
  }
  switch (get_file_ext_type (te->filename))
  {
  case FILE_EXT_TYPE_C:
  case FILE_EXT_TYPE_CPP:
  case FILE_EXT_TYPE_IC:
  case FILE_EXT_TYPE_ICPP:
  case FILE_EXT_TYPE_H:
    C = TRUE;
    break;
  case FILE_EXT_TYPE_XPM:
  default:
    C = FALSE;
  }
  gtk_widget_set_sensitive (te->buttons.save, !S);
  gtk_widget_set_sensitive (te->buttons.reload, (te->full_filename != NULL));
  gtk_widget_set_sensitive (te->buttons.compile, F && !L);
  gtk_widget_set_sensitive (te->buttons.build, (F || P) && !L);
}
