/*
    utilities.c
    Copyright (C) 2000  Kh. Naba Kumar Singh

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdlib.h>
#include <glib.h>

#include <gtk/gtk.h>
#include "anjuta.h"
#include "utilities.h"
#include "launcher.h"

gchar *
extract_filename (gchar * full_filename)
{

  gint i;
  gint length;

  length = strlen (full_filename);
  if (length == 0)
    return full_filename;
  if (full_filename[length - 1] == '/')
    return &full_filename[length];
  for (i = strlen (full_filename) - 1; i >= 0; i--)
    if (full_filename[i] == '/')
      break;
  if (i == 0 && full_filename[0] != '/')
    return full_filename;
  return &full_filename[++i];
}

GList *
update_string_list (GList * list, gchar * string, gint length)
{
  gint i;
  gchar *str;
  if (!string)
    return list;
  for (i = 0; i < g_list_length (list); i++)
  {
    str = (gchar *) g_list_nth_data (list, i);
    if (!str)
      continue;
    if (strcmp (string, str) == 0)
    {
      list = g_list_remove (list, str);
      list = g_list_prepend (list, str);
      return list;
    }
  }
  list = g_list_prepend (list, g_strdup (string));
  while (g_list_length (list) > length)
  {
    str = g_list_nth_data (list, g_list_length (list) - 1);
    list = g_list_remove (list, str);
    g_free (str);
  }
  return list;
}

gboolean parse_error_line (gchar * line, gchar ** filename, int *lineno)
{
  gint i = 0;
  gint j = 0;
  gint k = 0;
  gchar *dummy;

  while (line[i++] != ':')
  {
    if (i >= strlen (line) || i >= 512 || line[i - 1] == ' ')
    {
      goto down;
    }
  }
  if (isdigit (line[i]))
  {
    j = i;
    while (isdigit (line[i++])) ;
    dummy = g_strndup (&line[j], i - j - 1);
    *lineno = atoi (dummy);
    if (dummy)
      g_free (dummy);
    dummy = g_strndup (line, j - 1);
    *filename = g_strdup (g_strstrip (dummy));
    if (dummy)
      g_free (dummy);
    return TRUE;
  }

down:
  i = strlen (line) - 1;
  while (isspace (line[i]) == FALSE)
  {
    i--;
    if (i < 0)
    {
      *filename = NULL;
      *lineno = 0;
      return FALSE;
    }
  }
  k = i++;
  while (line[i++] != ':')
  {
    if (i >= strlen (line) || i >= 512 || line[i - 1] == ' ')
    {
      *filename = NULL;
      *lineno = 0;
      return FALSE;
    }
  }
  if (isdigit (line[i]))
  {
    j = i;
    while (isdigit (line[i++])) ;
    dummy = g_strndup (&line[j], i - j - 1);
    *lineno = atoi (dummy);
    if (dummy)
      g_free (dummy);
    dummy = g_strndup (&line[k], j - k - 1);
    *filename = g_strdup (g_strstrip (dummy));
    if (dummy)
      g_free (dummy);
    return TRUE;
  }
  *lineno = 0;
  *filename = NULL;
  return FALSE;
}

gchar *
get_file_extension (gchar * file)
{
  gchar *pos;
  if (!file)
    return NULL;
  pos = strrchr (file, '.');
  if (pos)
    return ++pos;
  else
    return NULL;
}

FileExtType get_file_ext_type (gchar * file)
{
  gchar *pos;
  if (!file)
    return FILE_EXT_TYPE_UNKNOWN;
  pos = get_file_extension (file);
  if (pos == NULL)
    return FILE_EXT_TYPE_UNKNOWN;
  if (strcmp (pos, "cc") == 0)
    return FILE_EXT_TYPE_CPP;
  if (strcmp (pos, "cxx") == 0)
    return FILE_EXT_TYPE_CPP;
  if (strcmp (pos, "cpp") == 0)
    return FILE_EXT_TYPE_CPP;
  if (strcmp (pos, "ii") == 0)
    return FILE_EXT_TYPE_ICPP;
  if (strcmp (pos, "xpm") == 0)
    return FILE_EXT_TYPE_XPM;
  if (strcmp (pos, "jpg") == 0)
    return FILE_EXT_TYPE_JPG;
  if (strcmp (pos, "bmp") == 0)
    return FILE_EXT_TYPE_BMP;
  if (strcmp (pos, "gif") == 0)
    return FILE_EXT_TYPE_GIF;
  if (strcmp (pos, "png") == 0)
    return FILE_EXT_TYPE_PNG;
  if (strcmp (pos, "tiff") == 0)
    return FILE_EXT_TYPE_TIFF;
  if (strcmp (pos, "java") == 0)
    return FILE_EXT_TYPE_JAVA;
  if (strcmp (pos, "scm") == 0)
    return FILE_EXT_TYPE_SCM;
  if (strcmp (pos, "sh") == 0)
    return FILE_EXT_TYPE_SH;
  if (strcmp (pos, "pl") == 0)
    return FILE_EXT_TYPE_PERL;
  if (strcmp (pos, "prj") == 0)
    return FILE_EXT_TYPE_PRJ;

  if (strlen (pos) == 1)
  {
    if (pos[0] == 'h')
      return FILE_EXT_TYPE_H;
    if (pos[0] == 'c')
      return FILE_EXT_TYPE_C;
    if (pos[0] == 'C')
      return FILE_EXT_TYPE_CPP;
    if (pos[0] == 's')
      return FILE_EXT_TYPE_ASM;
    if (pos[0] == 'S')
      return FILE_EXT_TYPE_ASM;
    if (pos[0] == 'i')
      return FILE_EXT_TYPE_IC;
    if (pos[0] == 'm')
      return FILE_EXT_TYPE_OC;
  }
  return FILE_EXT_TYPE_UNKNOWN;
}

gchar *
get_a_tmp_file ()
{
  static gint count;
  gchar *filename;
  filename =
    g_strdup_printf ("%s/anjuta_%d.%d", app->dirs->tmp, count++, getpid ());
  return filename;
}

void
fast_widget_repaint (GtkWidget * w)
{
  GdkRectangle rect;
  if (GTK_IS_WIDGET (w) && GTK_WIDGET_VISIBLE (w))
  {
    rect.x = w->allocation.x;
    rect.y = w->allocation.y;
    rect.width = w->allocation.width;
    rect.height = w->allocation.height;
    gtk_widget_draw (w, &rect);
  }
}

gboolean
write_line (FILE * stream, gchar * str)
{
  unsigned long len;

  if (!str || !stream)
    return FALSE;
  len = strlen (str);
  if (len != 0)
    if (fwrite (str, len, sizeof (gchar), stream) < 0)
      return FALSE;
  if (fwrite ("\n", 1, sizeof (gchar), stream) < 1)
    return FALSE;
  return TRUE;
}

gboolean
read_line (FILE * stream, gchar ** str)
{
  unsigned long count;
  gchar buffer[1024];
  gint ch;

  if (stream == NULL)
    return FALSE;

  count = 0;
  while (1)
  {
    ch = fgetc (stream);
    if (ch < 0)
      break;
    if (ch == '\n')
      break;
    buffer[count] = (char) ch;
    count++;
    if (count >= 1024 - 1)
      break;
  }
  buffer[count] = '\0';
  *str = g_strdup (buffer);
  return TRUE;
}

gboolean
write_string (FILE * stream, gchar * t, gchar * str)
{
  unsigned long len;
  if (!str || !stream)
    return FALSE;
  len = strlen (str);
  if (fprintf (stream, "%s :%lu: ", t, len) < 2)
    return FALSE;
  if (len != 0)
    if (fwrite (str, len, sizeof (gchar), stream) < 0)
      return FALSE;
  if (fwrite ("\n", 1, sizeof (gchar), stream) < 1)
    return FALSE;
  return TRUE;
}

gboolean
read_string (FILE * stream, gchar * t, gchar ** str)
{
  unsigned long tmp;
  gchar *buff, bb[3], token[256];

  tmp = 0;
  strcpy (token, "");
  if (stream == NULL)
    return FALSE;
  if (fscanf (stream, "%s :%lu: ", token, &tmp) < 2)
    return FALSE;
  if (strcmp (token, t) != 0)
    return FALSE;
  if (tmp == 0)
  {
    if (str)
      (*str) = g_strdup ("");
    return TRUE;
  }
  buff = g_malloc ((tmp + 1) * sizeof (char));
  if (fread (buff, tmp, sizeof (gchar), stream) < 0)
  {
    g_free (buff);
    return FALSE;
  }
  if (fread (bb, 1, sizeof (gchar), stream) < 1)
  {
    g_free (buff);
    return FALSE;
  }
  buff[tmp] = '\0';
  if (str)
    (*str) = buff;
  else
    g_free (buff);
  return TRUE;
}

gint
compare_string_func (gconstpointer a, gconstpointer b)
{
  if (!a && !b)
    return 0;
  if (!a || !b)
    return -1;
  return (strcmp ((char *) a, (char *) b));
}

gboolean
summon_glade ()
{
  gchar *filename;
  gint pid;
  FILE *fp;

  if (app->project_dbase->project_is_open == FALSE)
    return FALSE;
  if (anjuta_is_installed ("glade", TRUE) == FALSE)
    return FALSE;

  filename = g_strdup_printf ("%s/%s.glade",
			      app->project_dbase->top_proj_dir,
			      app->project_dbase->proj_name);
  if ((fp = fopen (filename, "r")) == NULL)
  {
    g_free (filename);
    anjuta_error (_
		  ("ERROR: The Project glade file does not exist in the top level directory."));
    return FALSE;
  }
  fclose (fp);
  if ((pid = fork ()) == 0)
  {
    execlp ("glade", "glade", filename, NULL);
    g_error (_("Cannot launch glade"));
  }
  g_free (filename);
  if (pid < 0)
  {
    anjuta_error (_
		  ("ERROR: Unable to execute glade.\nMake sure it is installed."));
    return FALSE;
  }
  else
  {
    return TRUE;
    anjuta_register_child_process (pid, NULL, NULL);
  }
}

gboolean file_is_regular (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISREG (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_directory (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISDIR (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_link (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISLNK (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_char_device (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISCHR (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_block_device (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISBLK (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_fifo (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISFIFO (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_socket (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISSOCK (st.st_mode))
    return TRUE;
  return FALSE;
}

gboolean file_is_readable (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_IRUSR & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean file_is_readonly (const gchar * fn)
{
  return file_is_readable (fn) && !file_is_writable (fn);
}

gboolean file_is_readwrite (const gchar * fn)
{
  return file_is_readable (fn) && file_is_writable (fn);
}


gboolean file_is_writable (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_IWUSR & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean file_is_executable (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_IXUSR & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean file_is_suid (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISUID & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean file_is_sgid (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISGID & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean file_is_sticky (const gchar * fn)
{
  struct stat st;
  int ret;
  if (!fn)
    return FALSE;
  ret = stat (fn, &st);
  if (ret)
    return FALSE;
  if (S_ISVTX & st.st_mode)
    return TRUE;
  return FALSE;
}

gboolean copy_file (gchar * src, gchar * dest)
{
  FILE *s, *d;
  gchar *mesg;
  char buff[2];
  int n;

  if ((s = fopen (src, "r")) == NULL)
  {
    mesg =
      g_strdup_printf (_
		       ("Could not open \"%s\" to read.\nMay be the file does not"
			\
			"exist.\n Or you do not have the read permission for this file."),
src);
    anjuta_error (mesg);
    g_free (mesg);
    return FALSE;
  }
  if ((d = fopen (dest, "w")) == NULL)
  {
    mesg = g_strdup_printf (_("Could not open \"%s\" to write.\n" \
			      "May be there is no space left in the file system.\n"
			      \
			      "Or you do not have the write permission for this file system."),
			    dest);
    anjuta_error (mesg);
    fclose (s);
    g_free (mesg);
    return FALSE;
  }
  clearerr (s);
  n = fread (&buff, sizeof (char), 1, s);
  while (feof (s) == 0)
  {
    n = fwrite (buff, sizeof (char), 1, d);
    if (n == 0)
    {
      anjuta_error (_("Could not complete the file copy operation.\n" \
		      "May be there is no space left in the file system.\n"));
      fclose (d);
      fclose (s);
      return FALSE;
    }
    n = fread (&buff, sizeof (char), 1, s);
  }
  fclose (d);
  fclose (s);
  return TRUE;
}

inline void
update_gtk ()
{
  if (launcher_is_busy () == TRUE)
    return;
  if (app->auto_gtk_update == FALSE)
    return;
  while (gtk_events_pending ())
  {
    gtk_main_iteration ();
  }
}

void
entry_set_text_n_select (GtkWidget * entry, gchar * text,
			 gboolean use_selection)
{
  gchar *chars;

  if (!entry)
    return;
  if (GTK_IS_ENTRY (entry) == FALSE)
    return;

  chars = anjuta_get_current_selection ();
  if (use_selection)
  {
    if (chars)
    {
      gtk_entry_set_text (GTK_ENTRY (entry), chars);
      gtk_editable_select_region (GTK_EDITABLE (entry), 0,
				  strlen (gtk_entry_get_text
					  (GTK_ENTRY (entry))));
    }
    else
    {
      if (text)
	gtk_entry_set_text (GTK_ENTRY (entry), text);
      gtk_editable_select_region (GTK_EDITABLE (entry), 0,
				  strlen (gtk_entry_get_text
					  (GTK_ENTRY (entry))));
    }
  }
  else
  {
    if (text)
      gtk_entry_set_text (GTK_ENTRY (entry), text);
    gtk_editable_select_region (GTK_EDITABLE (entry), 0,
				strlen (gtk_entry_get_text
					(GTK_ENTRY (entry))));
  }
  if (chars)
    g_free (chars);
}

gboolean force_create_dir (gchar * d)
{
  if (file_is_directory (d))
    return TRUE;
  if (mkdir (d, 0755))
    return FALSE;
  return TRUE;
}

GdkFont *
get_fixed_font ()
{
  static GdkFont *font;
  static gint done;

  if (done)
    return font;
  font =
    gdk_font_load
    ("-misc-fixed-medium-r-semicondensed-*-*-120-*-*-c-*-iso8859-9");
  done = 1;
  if (font)
  {
    gdk_font_ref (font);
    return font;
  }
  g_warning ("Cannot load fixed(misc) font. Using default font.");
  return NULL;
}

gchar *
remove_white_spaces (gchar * text)
{
  guint src_count, dest_count, tab_count;
  gchar buff[2048];		/* Let us hope that it does not overflow */

  tab_count = app->preferences->tab_size;
  dest_count = 0;
  for (src_count = 0; src_count < strlen (text); src_count++)
  {
    if (text[src_count] == '\t')
    {
      gint j;
      for (j = 0; j < tab_count; j++)
	buff[dest_count++] = ' ';
    }
    else if (isspace (text[src_count]))
    {
      buff[dest_count++] = ' ';
    }
    else
    {
      buff[dest_count++] = text[src_count];
    }
  }
  buff[dest_count] = '\0';
  return g_strdup (buff);
}

GList *
remove_blank_lines (GList * lines)
{
  GList *list, *node;
  gchar *str;

  if (lines)
    list = g_list_copy (lines);
  else
    list = NULL;

  node = list;
  while (node)
  {
    str = node->data;
    node = g_list_next (node);
    if (!str)
    {
      list = g_list_remove (list, str);
      continue;
    }
    if (strlen (g_strchomp (str)) < 1)
      list = g_list_remove (list, str);
  }
  return list;
}
