/*
 * text_editor_cbs.c
 * Copyright (C) 2000  Kh. Naba Kumar Singh
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <ctype.h>
#include "gtkantext.h"

#include "anjuta.h"
#include "gtkaneditor.h"
#include "controls.h"
#include "text_editor.h"
#include "text_editor_cbs.h"
#include "text_editor_gui.h"
#include "mainmenu_callbacks.h"
#include "support.h"


void
on_text_editor_window_realize (GtkWidget * widget, gpointer user_data)
{
  TextEditor *te = user_data;
  te->widgets.window = widget;
}

void
on_text_editor_window_size_allocate (GtkWidget * widget,
				     GtkAllocation * allocation,
				     gpointer user_data)
{
  TextEditor *te = user_data;
  te->allocation.x = allocation->x;
  te->allocation.y = allocation->y;
  te->allocation.width = allocation->width;
  te->allocation.height = allocation->height;
}

void
on_text_editor_client_realize (GtkWidget * widget, gpointer user_data)
{
  TextEditor *te = user_data;
  te->widgets.client = widget;
}

gboolean
on_text_editor_text_buttonpress_event (GtkWidget * widget,
				       GdkEventButton * event,
				       gpointer user_data)
{
  TextEditor *te = user_data;
  text_editor_check_disk_status (te);
  return FALSE;
}

gboolean
on_text_editor_text_event (GtkWidget * widget,
			   GdkEvent * event, gpointer user_data)
{
  GdkEventButton *bevent;
  if (event->type != GDK_BUTTON_PRESS)
    return FALSE;
  if (((GdkEventButton *) event)->button != 3)
    return FALSE;
  bevent = (GdkEventButton *) event;
  bevent->button = 1;
  text_editor_menu_popup (((TextEditor *) user_data)->menu, bevent);
  return TRUE;
}

gboolean
on_text_editor_window_delete (GtkWidget * widget,
			      GdkEventFocus * event, gpointer user_data)
{
  on_close_file1_activate (NULL, NULL);
  return TRUE;
}

gboolean
on_text_editor_window_focus_in_event (GtkWidget * widget,
				      GdkEventFocus * event,
				      gpointer user_data)
{
  TextEditor *te = user_data;
  app->current_text_editor = te;
  update_main_menubar ();
  return FALSE;
}



void
on_text_editor_dock_activate (GtkButton * button, gpointer user_data)
{
  GtkWidget *label, *eventbox;
  TextEditor *te = anjuta_get_current_text_editor ();
  label = gtk_label_new (te->filename);
  gtk_widget_show (label);
  eventbox = gtk_event_box_new ();
  gtk_widget_show (eventbox);
  text_editor_dock (te, eventbox);
  gtk_notebook_prepend_page (GTK_NOTEBOOK (app->widgets.notebook),
			     eventbox, label);
  gtk_window_set_title (GTK_WINDOW (app->widgets.window), te->full_filename);
  gtk_notebook_set_page (GTK_NOTEBOOK (app->widgets.notebook), 0);
}

void
on_text_editor_text_changed (GtkEditable * editable, gpointer user_data)
{
  TextEditor *te = (TextEditor *) user_data;
  if (te->status == TEXT_EDITOR_SAVED)
  {
    te->status = TEXT_EDITOR_NOT_SAVED;
    anjuta_update_title ();
    update_main_menubar ();
    text_editor_update_controls (te);
  }
}

void
on_text_editor_insert_text (GtkEditable * text,
			    const gchar * insertion_text,
			    gint length, gint * pos, TextEditor * te)
{
  TextEditorUndoItem *ui;

  if (te == NULL) return;
  ui = text_editor_undo_item_new (insertion_text, length, *pos, INSERT);
  text_editor_undo_item_add (te, ui);
}

gint
on_text_editor_insert_text_after (GtkEditable * text,
			    const gchar * insertion_text,
			    gint length, gint * pos, TextEditor * te)
{
  int i, newlines, newline_1;
  gchar *buffer, *whitespace;

  if (te == NULL)
     return FALSE;
  if ((length != 1) || (insertion_text[0] != '\n'))
    return FALSE;
  if (gtk_antext_get_length (GTK_ANTEXT (text)) <= 1)
    return FALSE;
  if (te->preferences->auto_indent == FALSE)
    return FALSE;

  newline_1 = 0;
  newlines = 0;
  for (i = *pos; i > 0; i--)
  {
    buffer = gtk_editable_get_chars (text, i - 1, i);
    if (buffer == NULL)
      continue;
    if (buffer[0] == 10)
    {
      if (newlines > 0)
      {
	g_free (buffer);
	buffer = NULL;
	break;
      }
      else
      {
	newlines++;
	newline_1 = i;
	g_free (buffer);
	buffer = NULL;
      }
    }
    g_free (buffer);
    buffer = NULL;
  }

  whitespace = g_malloc0 (newline_1 - i + 2);

  for (i = i; i <= newline_1; i++)
  {
    buffer = gtk_editable_get_chars (text, i, i + 1);
    if ((buffer[0] != 32) && (buffer[0] != 9))
    {
      g_free (buffer);
      buffer = NULL;
      break;
    }
    strncat (whitespace, buffer, 1);
    g_free (buffer);
  }

  if (strlen (whitespace) > 0)
  {
    i = *pos;
    gtk_antext_set_point (GTK_ANTEXT (text), i);
    gtk_antext_insert (GTK_ANTEXT (text), te->preferences->font, NULL, NULL,
		       whitespace, strlen (whitespace));
  }
  g_free (whitespace);
  return TRUE;
}

void
on_text_editor_delete_text (GtkEditable * editable,
			    gint start_pos, gint end_pos, TextEditor * te)
{
  TextEditorUndoItem *ui;
  gchar *text;
  text = gtk_editable_get_chars (editable, start_pos, end_pos);
  ui =
    text_editor_undo_item_new (text, end_pos - start_pos, start_pos, DELETE);
  g_free (text);
  text_editor_undo_item_add (te, ui);
}

void
on_text_editor_check_yes_clicked (GtkButton * button, TextEditor * te)
{
  text_editor_load_file (te);
}

void
on_text_editor_check_no_clicked (GtkButton * button, TextEditor * te)
{
  te->modified_time = time (NULL);
}

gboolean
on_text_editor_auto_save (gpointer data)
{
  gchar *mesg;
  TextEditor *te = data;

  if (!te)
    return FALSE;
  if (te->preferences->auto_save == FALSE)
  {
    te->autosave_on = FALSE;
    return FALSE;
  }
  if (te->full_filename == NULL)
    return TRUE;
  if (te->status == TEXT_EDITOR_SAVED)
    return TRUE;
  if (text_editor_save_file (te))
  {
    mesg = g_strdup_printf (_("Autosaved \"%s\""), te->filename);
    anjuta_status (mesg);
  }
  else
  {
    mesg = g_strdup_printf (_("Autosave \"%s\" -- Failed"), te->filename);
    anjuta_warning (mesg);
  }
  g_free (mesg);
  return TRUE;
}
