//***********************************************************************
// *  $RCSfile$ - Conversation.cs
// *
// *  Copyright (C) 2007 Novell, Inc.
// *
// *  This program is free software; you can redistribute it and/or
// *  modify it under the terms of the GNU General Public
// *  License as published by the Free Software Foundation; either
// *  version 2 of the License, or (at your option) any later version.
// *
// *  This program is distributed in the hope that it will be useful,
// *  but WITHOUT ANY WARRANTY; without even the implied warranty of
// *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// *  General Public License for more details.
// *
// *  You should have received a copy of the GNU General Public
// *  License along with this program; if not, write to the Free
// *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
// *
// **********************************************************************

using System;
using System.Collections;
using System.Collections.Generic;
using System.Net;
using System.Text;

using NDesk.DBus;
using org.freedesktop.DBus;
using org.freedesktop.Telepathy;

namespace Banter
{
	public delegate void MessageSentHandler (Conversation conversation, Message message);
	public delegate void MessageReceivedHandler (Conversation conversation, Message message);
	public delegate void VideoChannelInitializedHandler (Conversation conversation);
	public delegate void VideoChannelConnectedHandler (Conversation conversation, uint streamId);
	public delegate void AudioChannelInitializedHandler (Conversation conversation);
	
	public class Conversation : IDisposable
	{
		public event MessageSentHandler MessageSent;
		public event MessageReceivedHandler MessageReceived;
		public event VideoChannelInitializedHandler VideoChannelInitialized;
		public event VideoChannelConnectedHandler VideoChannelConnected;
		public event AudioChannelInitializedHandler AudioChannelInitialized;

		private bool initiatedChat;
		private uint previewWindowID;
		private uint peerWindowID;
		private List<Message> messages;
		private Account account;
		
		// Telepathy connection and channels		
		private org.freedesktop.Telepathy.IConnection tlpConnection;
		private IChannelHandler channelHandler;
		private ObjectPath videoChannelObjectPath;
		private ObjectPath audioChannelObjectPath;
		private org.freedesktop.Telepathy.IChannelText txtChannel;
		private IChannelStreamedMedia videoChannel;
		private IChannelStreamedMedia audioChannel;
		private IStreamEngine streamEngine;
		private uint videoInputStreamId = 0;
		
		private ProviderUser peerUser;
		private Presence lastPeerPresence;
		
		private uint current;
		private uint last;
	
		public ProviderUser PeerUser
		{
			get {return peerUser;}
		}
		
		public bool CurrentMessageSameAsLast
		{
			get { return (current == last) ? true : false;}
		}
	
		#region Constructors	
		internal Conversation (Account account, Person peer, ProviderUser peerUser, bool initiate)
		{
			this.account = account;
			this.tlpConnection = account.TlpConnection;
			this.peerUser = peerUser;
			this.messages = new List<Message> ();
			last = 999;

			peerUser.PresenceUpdated += OnPeerPresenceUpdated;
			lastPeerPresence = peerUser.Presence;
			
			if (initiate == true)
				this.CreateTextChannel ();
		}
		#endregion
		
		#region Private Methods
		/// <summary>
		/// Signal called when presence changes for the peer user.
		/// </summary>
		private void OnPeerPresenceUpdated (ProviderUser user)
		{
			string msg;
			Banter.SystemMessage systemMessage = null;
			
			// no handlers?  exit
			if (MessageReceived == null) return;
			
			string displayName = (user.Alias != null) ? user.Alias.Split (' ')[0] : user.Uri;
			if (user.Presence.Type == Banter.PresenceType.Offline) {
				msg = String.Format("{0} has gone {1}", displayName, user.Presence.Name); 
				systemMessage = new Banter.SystemMessage (msg);
			} else {
				if (user.Presence.Message != null && 
					user.Presence.Message != String.Empty &&
					user.Presence.Message != lastPeerPresence.Message) {
				
					msg = String.Format(
							"{0} is {1} \"{2}\"", 
							displayName, 
							user.Presence.Name,
							user.Presence.Message);
					systemMessage = new Banter.SystemMessage (msg);
					
				} else {
					msg = String.Format("{0} is {1}", displayName, user.Presence.Name); 
					systemMessage = new Banter.SystemMessage (msg);
				}
			}

			lastPeerPresence = user.Presence;
			
			// Indicate the message to registered handlers
			if (systemMessage != null)
				MessageReceived (this, systemMessage);
		}
		
		/// <summary>
		/// Message receive indication called from telepathy
		/// </summary>
		private void OnReceiveMessageHandler (
			uint id,
			uint timeStamp,
			uint sender,
			org.freedesktop.Telepathy.MessageType messageType,
			org.freedesktop.Telepathy.MessageFlag messageFlag,
			string text)
		{
			if (this.peerUser != null)
			{
				Logger.Debug ("Conversation::OnReceiveMessageHandler - called");
				Logger.Debug ("  received message from: {0}", peerUser.Uri);
				Logger.Debug ("  peer id: {0}  incoming id: {1}", peerUser.ID, id);
				
				TextMessage txtMessage = new TextMessage (text);
				messages.Add (txtMessage);
				
				if (current != 0) last = current;
				current = this.peerUser.ID;
			
				// Indicate the message to registered handlers
				if (MessageReceived != null){
					MessageReceived (this, txtMessage);
				}
			}
		}
		
		private void OnTextChannelClosed()
		{
			this.txtChannel = null;
		}
		
		#endregion
		
		#region Public Methods
		public void Dispose()
		{
			if (txtChannel != null)	{
				txtChannel.Close();
				txtChannel = null;
			}
		}
		
		public Banter.Message[] GetReceivedMessages()
		{
			return messages.ToArray();
		}
		
		public void SendMessage (Message message)
		{
			// FIXME::Throw exception
			if (tlpConnection == null) return;
			if (txtChannel == null) return;
			if (message == null) return;
			
			this.txtChannel.Send (org.freedesktop.Telepathy.MessageType.Normal, message.Text);

			if (current != 0)
				last = current;
				
			current = tlpConnection.SelfHandle;
			
			if (MessageSent != null)
				MessageSent (this, message);
		}
		
		public void SetTextChannel (IChannelText channel)
		{
			if (txtChannel != null) return;
			txtChannel = channel;
			
			txtChannel.Received += OnReceiveMessageHandler;
			txtChannel.Closed += OnTextChannelClosed;
			
			// Check for any pending messages and add them to our list
			TextMessage txtMessage;
			PendingMessageInfo[] msgInfos = txtChannel.ListPendingMessages (true);
			foreach (PendingMessageInfo info in msgInfos) {
				Logger.Debug ("Pending Message: {0}", info.Message);
				txtMessage = new TextMessage (info.Message);
				txtMessage.From = peerUser.Uri;
				messages.Add (txtMessage);
			}
		}
		
		public void SetVideoWindows (uint meID, uint peerID)
		{
			this.previewWindowID = meID;
			this.peerWindowID = peerID;
		}
		
		private void CreateTextChannel ()
		{
			if (txtChannel != null) return;
			
			ObjectPath op = 
				tlpConnection.RequestChannel (
					org.freedesktop.Telepathy.ChannelType.Text, 
					HandleType.Contact,
					//target.Handle.Type, 
					this.peerUser.ID, 
					true);
				
			txtChannel = Bus.Session.GetObject<IChannelText> (account.BusName, op);
			txtChannel.Received += OnReceiveMessageHandler;
		}
		
		private bool SetupVideoChannel ()
		{
			if (videoChannel != null) return true;
			
			Logger.Debug ("SetupVideoChannel entered");
			uint[] handles = new uint [] {peerUser.ID};
			try
			{
				if (this.initiatedChat == true) {
					videoChannelObjectPath = 
						tlpConnection.RequestChannel (
							org.freedesktop.Telepathy.ChannelType.StreamedMedia,
							HandleType.Contact,
							handles[0],
							true);
					Logger.Debug("Have the Video Channel Object Path");
					videoChannel = 
						Bus.Session.GetObject<IChannelStreamedMedia> (
							account.BusName, videoChannelObjectPath);
					
					if (videoChannel == null) {
						Logger.Debug ("videoChannel is null!");
						throw new ApplicationException ("Cannot proceed with a null video channel");
					}
				}

				Logger.Debug("Have a Video Channel");
				Logger.Debug("Initializing video channel with ChannelHandler");
				
				channelHandler = 
		       	Bus.Session.GetObject<IChannelHandler> (
		       		"org.freedesktop.Telepathy.StreamEngine",
		           	new ObjectPath ("/org/freedesktop/Telepathy/StreamEngine"));
		           
				if (channelHandler != null) 
				{
					Logger.Debug("Have the channelHandler... telling it to handle the channel");
					channelHandler.HandleChannel (
						account.BusName,
						account.BusPath,
						//connectionInfo.BusName, 
						//connectionInfo.ObjectPath,
					    videoChannel.ChannelType, 
					    this.videoChannelObjectPath,
					    0,
					    0);
				}
				else
		       		Logger.Debug("Didn't get a channel handler");

				StreamInfo[] lst = ((IChannelStreamedMedia) videoChannel).ListStreams();
				Logger.Debug("StreamInfo List Length: {0}", lst.Length);
				
		       	foreach (StreamInfo info in lst)
		       	{
		       		Logger.Debug(
		       			"Stream Info: Id:{0}, Type:{1}, ContactHandle:{2}, Direction: {3}",
		       			info.Id,
		       			info.Type,
		       			info.ContactHandle,
		       			info.Direction);
		       				
		       		// Save the type of the stream so we can reference it later
		       		//SaveStream (info.Type, info.Id);
		       }
		       
				videoChannel.StreamAdded += OnStreamAdded;
				videoChannel.StreamDirectionChanged += OnStreamDirectionChanged;
				videoChannel.StreamError += OnStreamError;
				videoChannel.StreamRemoved += OnStreamRemoved;
				videoChannel.StreamStateChanged += OnStreamStateChanged;
		       
				videoChannel.MembersChanged += OnMembersChanged;

				Logger.Debug("Getting the stream_engine");
				
				streamEngine = 
					Bus.Session.GetObject<IStreamEngine> (
						"org.freedesktop.Telepathy.StreamEngine",
		           		new ObjectPath ("/org/freedesktop/Telepathy/StreamEngine"));

		        Logger.Debug("have the stream engine");
		        
				Logger.Debug("Adding Preview Window");
			    streamEngine.AddPreviewWindow(previewWindowID);
				streamEngine.Receiving += OnStreamEngineReceiving;

				Logger.Debug("The numder of members is: {0}", videoChannel.Members.Length);

				if (this.initiatedChat == true) {
	//				uint[] stream_type = new uint[2];
					uint[] streamtype = new uint[1];
					
	//				stream_type[0] = (uint) StreamType.Audio;
					streamtype[0] = (uint) StreamType.Video;

					Logger.Debug("Requesting streams from video channel");
					StreamInfo[] infos = videoChannel.RequestStreams (handles[0], streamtype);
					
					Logger.Debug("Number of Streams Received: {0}", infos.Length);
					Logger.Debug("Stream Info: Id{0} State{1} Direction{2} ContactHandle{3}", infos[0].Id, infos[0].State, infos[0].Direction, infos[0].ContactHandle);
				}
				
				if (VideoChannelInitialized != null)
					VideoChannelInitialized (this);
			}
			catch(Exception e)
			{
				Logger.Debug("Exception in SetupMediaChannel: {0}\n{1}", e.Message, e.StackTrace);
			}

			return true;
		}	
		
		
        private void OnStreamAdded (uint streamid, uint contacthandle, org.freedesktop.Telepathy.StreamType streamtype)
		{
			Logger.Debug(
				"OnStreamAdded of type: {0} with id:{1}, for contact {2}", 
				streamtype, 
				streamid, 
				contacthandle);
			//SaveStream (stream_type, stream_id);
		}

		private void OnStreamDirectionChanged (uint streamid, StreamDirection streamdirection, StreamPendingFlags pendingflags)
        {
			Logger.Debug("OnStreamDirectionChanged called");
        }

        private void OnStreamError (uint streamid, uint errno, string message)
        {
 			Logger.Debug("OnStreamError called with message: {0}:{1}", errno, message);
        }

        private void OnStreamRemoved (uint streamid)
        {
			Logger.Debug("OnStreamRemoved called on stream {0}", streamid);
			//RemoveStream (stream_id);
        }

        private void OnStreamStateChanged (uint streamid, org.freedesktop.Telepathy.StreamState streamstate)
        {
            Logger.Debug ("OnStreamStateChanged called - ID: {0} State: {1}", streamid, streamstate);
            
            // Audio or Video
            videoInputStreamId = streamid;
            
            // Make sure that this stream is a Video stream
            if (videoChannel.Handle.Id == streamid && streamstate == StreamState.Connected)
			{
				if (VideoChannelConnected != null)
					VideoChannelConnected (this, streamid);
			}
        }

        private void OnStreamEngineReceiving (ObjectPath channelpath, uint streamid, bool state)      
        {   
        	Logger.Debug("OnStreamEngineReceiving: stream id: {0}", streamid);
        }
        
        private void OnMembersChanged (string message, uint[] added, uint[] removed, uint[] localpending, uint[] remotepending, uint actor, uint reason)
        {
        	Logger.Debug ("OnMembersChanged: {0}", message);
        	
        	Logger.Debug ("\tAdded {0}: {1}", added.Length, PrintHandles (added));
        	Logger.Debug ("\tRemoved {0}: {1}", removed.Length, PrintHandles (removed));
        	Logger.Debug ("\tLocal Pending {0}: {1}", localpending.Length, PrintHandles (localpending));
        	Logger.Debug ("\tRemote Pending {0}: {1}", remotepending.Length, PrintHandles (remotepending));
        	Logger.Debug ("\tActor: {0}", actor);
        	Logger.Debug ("\tReason: {0}", reason);
        }
		
        private string PrintHandles (uint[] handles)
        {
        	string str = string.Empty;
        	foreach (uint handle in handles) {
        		if (str.Length > 0)
        			str += ", ";
        		str += string.Format ("0", handle);
        	}
        	
        	return str;
        }
		
		#endregion
		
#region Public Methods

		public void SetMediaChannel (IChannelStreamedMedia channel, ObjectPath op)
		{
			videoChannelObjectPath = op;
			videoChannel = channel;
		}
		
		public void SetPreviewWindow (uint windowId)
		{
			previewWindowID = windowId;
		}
		
		public void SetPeerWindow (uint windowId, uint streamId)
		{
			this.peerWindowID = windowId;
			
			if (streamEngine != null && 
				videoChannelObjectPath != null &&
				videoChannel != null) {
				streamEngine.SetOutputWindow (
					videoChannelObjectPath, 
					streamId,
					windowId);
			}
		}
		
		public void StartVideo (bool initiatedChat)
		{
			this.initiatedChat = initiatedChat;
			if (tlpConnection == null) 
			{
		    	throw new ApplicationException (String.Format ("No telepathy connection exists"));
			}
			
			// Create the video channel
			SetupVideoChannel ();
		}
#endregion
	}
}	
