//
// GoogleDriver.cs
//
// Copyright (C) 2004 Novell, Inc.
//
// Google is a trademark of Google.  But you already knew that.
//

//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
//


using System;
using System.Collections;
using Beagle.Util;

namespace Beagle {

	public class GoogleDriver : IQueryable {

		int maxResults = 5;

		GoogleSearchService gss = new GoogleSearchService ();
		String googleKey;

		public GoogleDriver ()
		{
			googleKey = Environment.GetEnvironmentVariable ("GOOGLE_WEB_API_KEY");
		}

		Hit FromGoogleResultElement (ResultElement res, int rank)
		{
			Hit hit = new Hit ();

			hit.Uri      = res.URL;
			hit.Type     = "WebLink";
			hit.MimeType = "text/html"; // FIXME
			hit.Source   = "Google";

			// FIXME: We can't really compare scores if the Hits
			// come from different sources.  This is a hack.
			hit.Score    = 0.2f / (1 + rank);

			hit ["Summary"]        = res.summary;
			hit ["Snippet"]        = res.snippet;
			hit ["Title"]          = res.title;
			hit ["CachedSize"]     = res.cachedSize;
			hit ["HostName"]       = res.hostName;
			hit ["DirectoryTitle"] = res.directoryTitle;

			return hit;
		}

		static bool showNoKeyMessage = true;

		public String Name {
			get { return "Google"; }
		}

		public bool AcceptQuery (Query query)
		{
			// Reject queries if the key isn't set.
			if (googleKey == null || googleKey == "") {
				if (showNoKeyMessage) {
					Console.WriteLine ("To query Google, put your Google key into the GOOGLE_WEB_API_KEY environment variable.");
					Console.WriteLine ("To get a Google key, go to http://api.google.com/createkey");
					showNoKeyMessage = false;
				}
				return false;
			}
			return true;
		}


		public void Query (Query query, IQueryResult result)
		{
			// FIXME: stop using the abusive query api
			GoogleSearchResult gsr = gss.doGoogleSearch (googleKey,
								     query.AbusivePeekInsideQuery,
								     0, maxResults,
								     false, "", false, "", "", "");

			ArrayList hits = new ArrayList ();
			int rank = 0;
			foreach (ResultElement elt in gsr.resultElements) {
				Hit hit = FromGoogleResultElement (elt, rank);
				++rank;
				hits.Add (hit);
			}

			result.Add (hits);
		}

	}

}
