/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 *  Copyright © 2010 Igalia S.L.
 *
 *  This file is part of Epiphany.
 *
 *  Epiphany is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Epiphany is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Epiphany.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "config.h"

#include "ephy-settings.h"

#include "ephy-debug.h"

#include <glib.h>
#include <gio/gio.h>

static GHashTable *settings = NULL;

static void
ensure_settings (void)
{
  if (settings)
    return;

  settings = g_hash_table_new_full (g_str_hash,
                                    g_str_equal, g_free,
                                    g_object_unref);
}

void
ephy_settings_shutdown (void)
{
  if (settings != NULL) {
    g_hash_table_remove_all (settings);
    g_hash_table_unref (settings);
  }
}

GSettings *
ephy_settings_get (const char *schema)
{
  GSettings *gsettings = NULL;

  ensure_settings ();

  gsettings = g_hash_table_lookup (settings, schema);

  if (gsettings == NULL) {
    gsettings = g_settings_new (schema);
    if (gsettings == NULL)
      g_warning ("Invalid schema %s requested", schema);
    else
      g_hash_table_insert (settings, g_strdup (schema), gsettings);
  }

  return gsettings;
}

void
ephy_settings_ensure_schema_for_path (const char *schema,
                                      const char *path)
{
  GSettings *gsettings;

  ensure_settings ();

  if (g_hash_table_lookup (settings, schema))
    return;

  gsettings = g_settings_new_with_path (schema, path);
  if (gsettings == NULL)
    g_warning ("Invalid schema %s requested for path %s", schema, path);
  else
    g_hash_table_insert (settings, g_strdup (schema), gsettings);
}
