/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            kolab-util-calendar.h
 *
 *  2011
 *  Copyright  2011  Silvan Marco Fin
 *  <silvan@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#include <libecal/e-cal.h>

#include <libekolabutil/camel-system-headers.h>
#include <libekolabutil/kolab-util-http.h>
#include <libekolabutil/kolab-http-job.h>
#include <libekolabutil/kolab-util-types.h>
#include <libekolab/kolab-settings-handler.h>
#include <libekolab/kolab-mail-access.h>

#include "kolab-util-calendar.h"

/*----------------------------------------------------------------------------*/
/* internal statics */

static KolabHttpJob*
kolab_util_calendar_create_http_request (KolabSettingsHandler *ksettings,
                                         const gchar *path,
                                         GError **error)
{
	CamelURL *uri = NULL;
	const gchar *url = NULL;
	const gchar *user = NULL;
	const gchar *passwd = NULL;
	guint tls_variant = KOLAB_TLS_VARIANT_DEFAULT;
	GError *tmp_error = NULL;
	KolabHttpJob *job = NULL;

	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (path != NULL);
	g_return_val_if_fail (error == NULL || *error == NULL, NULL);

	url = kolab_settings_handler_get_char_field (ksettings,
	                                             KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_URI,
	                                             &tmp_error);
	if (url == NULL) {
		g_propagate_error (error, tmp_error);
		return NULL;
	}
	tls_variant = kolab_settings_handler_get_uint_field (ksettings,
	                                                     KOLAB_SETTINGS_HANDLER_UINT_FIELD_TLS_VARIANT,
	                                                     &tmp_error);
	if (error != NULL) {
		g_propagate_error (error, tmp_error);
		return NULL;
	}
	uri = camel_url_new (url, &tmp_error);
	if (tmp_error != NULL) {
		g_propagate_error (error, tmp_error);
		return NULL;
	}
	if (uri == NULL) {
		/* FIXME is this an error condition? */
		return NULL;
	}
	switch (tls_variant) {
	case KOLAB_TLS_VARIANT_NONE:
		camel_url_set_protocol (uri, "http");
		break;
	default:
		camel_url_set_protocol (uri, "https");
	}
	camel_url_set_path (uri, path);
	user = kolab_settings_handler_get_char_field (ksettings,
	                                              KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_NAME,
	                                              &tmp_error);
	if (user == NULL) {
		g_propagate_error (error, tmp_error);
		camel_url_free (uri);
		return NULL;
	}
	camel_url_set_user (uri, user);
	passwd = kolab_settings_handler_get_char_field (ksettings,
	                                                KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_PASSWORD,
	                                                &tmp_error);
	if (passwd == NULL) {
		g_propagate_error (error, tmp_error);
		camel_url_free (uri);
		return NULL;
	}

	/* FIXME CamelURL does no longer carry auth details */
	g_error ("%s: FIXME CamelURL does no longer carry auth details",
	         __func__);
	/* camel_url_set_passwd (uri, passwd); */

	job = kolab_http_job_new ();
	job->url = uri;

	return job;
} /* kolab_util_calendar_create_http_request () */

static KolabHttpJob*
kolab_util_calendar_create_pfb_trigger (KolabSettingsHandler *ksettings,
                                        const gchar *sourcename,
                                        GError **error)
{
	KolabHttpJob *job = NULL;
	gchar *path = NULL;
	const gchar *username = NULL;
	gint source_offset = 0;
	GError *tmp_error = NULL;

	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (sourcename != NULL);
	g_return_val_if_fail (error == NULL || *error == NULL, NULL);

	username = kolab_settings_handler_get_char_field (ksettings,
	                                                  KOLAB_SETTINGS_HANDLER_CHAR_FIELD_KOLAB_USER_NAME,
	                                                  &tmp_error);
	if (tmp_error != NULL) {
		g_propagate_error (error, tmp_error);
		return NULL;
	}

	if (strncmp (username, "INBOX/", 6))
		source_offset = 6;
	path = g_strdup_printf ("/freebusy/trigger/%s/%s.pfb",
	                        username, sourcename+source_offset);

	job = kolab_util_calendar_create_http_request (ksettings,
	                                               path,
	                                               &tmp_error);
	g_free (path);
	if (job == NULL) {
		g_propagate_error (error, tmp_error);
		return NULL;
	}
	return job;
} /* kolab_util_calendar_create_pfb_trigger () */

static KolabHttpJob*
kolab_util_calendar_create_xfb_request (KolabSettingsHandler *ksettings,
                                        gchar *query,
                                        GError **error)
{
	KolabHttpJob *job = NULL;
	gchar *path = NULL;
	GError *tmp_error = NULL;

	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (query != NULL);
	g_return_val_if_fail (error == NULL || *error == NULL, NULL);

	path = g_strdup_printf ("/freebusy/%s.ifb", query);

	job = kolab_util_calendar_create_http_request (ksettings,
	                                               path,
	                                               &tmp_error);
	g_free (path);
	if (job == NULL) {
		g_propagate_error (error, tmp_error);
		g_warning ("%s()[%u] call to kolab_util_calendar_create_http_request() returned NULL!",
		           __func__, __LINE__);
		return NULL;
	}
	return job;
} /* kolab_util_calendar_create_xfb_request () */

/*----------------------------------------------------------------------------*/
/* public API */

/**
 * kolab_util_calendar_get_tzid:
 * @comp: An ECalComponent to derive the tzid from.
 * @from: field to get the tzid from.
 *
 * The ECalComponentDateTime Struct from an ECalComponents dtstart field or
 * dtend field is queried for its tzid.
 *
 * Returns: A newly allocated tzid string (g_free() after use)
 *          or NULL if no ID was found.
 */
gchar*
kolab_util_calendar_get_tzid (ECalComponent *comp,
                              ECalComponentField from)
{
	ECalComponentDateTime *dt = NULL;
	gchar *tzid = NULL;

	g_assert (E_IS_CAL_COMPONENT (comp));

	dt = g_new0 (ECalComponentDateTime, 1);
	switch (from) {
	case E_CAL_COMPONENT_FIELD_DTSTART:
		e_cal_component_get_dtstart (comp, dt);
		break;
	case E_CAL_COMPONENT_FIELD_DTEND:
		e_cal_component_get_dtend (comp, dt);
		break;
	case E_CAL_COMPONENT_FIELD_DUE:
		e_cal_component_get_due (comp, dt);
		break;
	default:
		g_warning ("%s()[%u]: TZID from %u not supported.",
		           __func__, __LINE__, from);
		return NULL;
	}

	if (dt == NULL) {
		g_warning ("%s()[%u]: Found ECalComponent without ECalComponentField %u.",
		           __func__, __LINE__, from);
		g_assert_not_reached();
	}

	tzid = g_strdup (dt->tzid);
	e_cal_component_free_datetime (dt);
	g_free (dt);
	return tzid;
} /* kolab_util_calendar_get_tzid () */

gboolean
kolab_util_calendar_toggle_pfb_trigger (KolabSettingsHandler *ksettings,
                                        const gchar *sourcename,
                                        GError **error)
{
	/* Keep running on errors.
	 * If an error occured, internal_state is set to false as indicator.
	 */
	KolabHttpJob *job = NULL;
	GError *tmp_error = NULL;

	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (sourcename != NULL);
	g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

	job = kolab_util_calendar_create_pfb_trigger (ksettings,
	                                              sourcename,
	                                              &tmp_error);
	if (job == NULL) {
		g_propagate_error (error, tmp_error);
		g_warning ("%s()[%u] error: could not create F/B toggle, giving up on %s.",
		           __func__, __LINE__, sourcename);
		return FALSE;
	}
	job->buffer = g_byte_array_new ();
	(void)kolab_util_http_get (job, &tmp_error);
	kolab_http_job_free (job);

	if (tmp_error != NULL) {
		g_propagate_error (error, tmp_error);
		return FALSE;
	}
	return TRUE;
} /* kolab_util_calendar_toggle_pfb_trigger () */

KolabHttpJob*
kolab_util_calendar_retrieve_xfb (KolabSettingsHandler *ksettings,
                                  gchar *query,
                                  GError **error)
{
	KolabHttpJob *job = NULL;
	GError *tmp_error = NULL;

	g_assert (KOLAB_IS_SETTINGS_HANDLER (ksettings));
	g_assert (query != NULL);
	g_return_val_if_fail (error == NULL || *error == NULL, NULL);

	job = kolab_util_calendar_create_xfb_request (ksettings,
	                                              (gchar *) query,
	                                              &tmp_error);
	if (job == NULL) {
		g_propagate_error (error, tmp_error);
		g_warning ("%s()[%u] error creating job, giving up on %s.",
		           __func__, __LINE__, (gchar *) query);
		return NULL;
	}

	job->buffer = g_byte_array_new ();
	(void)kolab_util_http_get (job, &tmp_error);
	if (tmp_error != NULL) {
		/* might happen, perhaps the server did not now about the user. */
		kolab_http_job_free (job);
		g_propagate_error (error, tmp_error);
		return NULL;
	}

	return job;
} /* kolab_util_calendar_retrieve_xfb () */

gboolean
kolab_util_calendar_store (ECalComponent *ecalcomp,
                           ECalComponent *ecaltz,
                           ECalComponent *default_tz,
                           KolabMailAccess *koma,
                           const gchar *uri,
                           GCancellable *cancellable,
                           GError **error)
{
	gchar *sourcename = NULL;
	KolabMailHandle *kmh = NULL;
	KolabSettingsHandler *ksettings = NULL;
	KolabMailAccessOpmodeID mode = KOLAB_MAIL_ACCESS_OPMODE_INVAL;
	gboolean do_trigger = FALSE;
	gboolean ok = FALSE;
	GError *tmp_error = NULL;

	g_assert (E_IS_CAL_COMPONENT (ecalcomp));
	/* ecaltz may be NULL */
	g_assert (E_IS_CAL_COMPONENT (default_tz));
	g_assert (KOLAB_IS_MAIL_ACCESS (koma));
	g_assert (uri != NULL);
	/* cancellable may be NULL */
	g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

	ksettings = kolab_mail_access_get_settings_handler (koma);
	if (ksettings == NULL) {
		/* FIXME: set GError here */
		return FALSE;
	}

	sourcename = kolab_util_backend_get_relative_path_from_uri (uri);

	kolab_util_backend_modtime_set_on_ecalcomp (ecalcomp);
	kmh = kolab_mail_handle_new_from_ecalcomponent (ecalcomp,
	                                                ecaltz);
	ok =  kolab_mail_access_store_handle (koma,
	                                      kmh,
	                                      sourcename,
	                                      cancellable,
	                                      &tmp_error);
	if (! ok)
		goto cleanup;

	mode = kolab_mail_access_get_opmode (koma,
	                                     &tmp_error);
	if (tmp_error != NULL)
		goto cleanup;
	if (mode < KOLAB_MAIL_ACCESS_OPMODE_ONLINE) {
		/* In OFFLINE state, the triggers are generated during
		 * changing to ONLINE state in e_cal_backend_set_mode ().
		 */
		goto cleanup;
	}

	do_trigger = kolab_mail_access_source_fbtrigger_needed (koma,
	                                                        sourcename,
	                                                        &tmp_error);
	if (tmp_error != NULL)
		goto cleanup;

	if (do_trigger == TRUE)
		(void) kolab_util_calendar_toggle_pfb_trigger (ksettings,
		                                               sourcename,
		                                               &tmp_error);
 cleanup:
	g_object_unref (ksettings);
	g_free (sourcename);
	if (tmp_error != NULL) {
		g_propagate_error (error, tmp_error);
		return FALSE;
	}
	return TRUE;
} /* kolab_util_calendar_store () */

/*----------------------------------------------------------------------------*/

#if 0 /* FIXME old */
ECalBackendSyncStatus
kolab_util_calendar_map_error (GError *error)
{
	ECalBackendSyncStatus status = GNOME_Evolution_Calendar_Success;

	if (error == NULL)
		return GNOME_Evolution_Calendar_Success;

	switch (error->code) {
	case KOLAB_BACKEND_ERROR_SYNC_NOTSTORED:
		status = GNOME_Evolution_Calendar_Success;
		break;
	case KOLAB_BACKEND_ERROR_NOTFOUND:
		status = GNOME_Evolution_Calendar_ObjectNotFound;
		break;
	case KOLAB_BACKEND_ERROR_CONTEXT_MISUSE:
	case KOLAB_BACKEND_ERROR_INFODB_NOFOLDER:
		status = GNOME_Evolution_Calendar_NoSuchCal;
		break;
	case KOLAB_BACKEND_ERROR_DATATYPE_EVOLUTION:
	case KOLAB_BACKEND_ERROR_DATATYPE_KOLAB:
		status = GNOME_Evolution_Calendar_InvalidObject;
	default:
		status = GNOME_Evolution_Calendar_OtherError;
	}

	g_debug ("%s()[%u] ECalBackendSyncStatus: %i", __func__, __LINE__, status);
	return status;
}
#endif

#if 0 /* FIXME old */
/**
 * kolab_util_calendar_extract:
 * @icalcomp: An icalcomponent.
 * @source_type: The data type to process.
 * @ecalcomp: An ECalComponent to return the extracted values to.
 * @tzcomp: An ECaComponent to return contained timezone data.
 *
 * Extract supported information from icalcomponent and return them through
 * ecalcomp and tzcomp.
 *
 * Returns: if the given component type could be extracted successfully into ecalcomp + tzcomp;
 */
ECalBackendSyncStatus
kolab_util_calendar_extract (icalcomponent *icalcomp,
                             ECalSourceType source_type,
                             ECalComponent **ecalcomp,
                             ECalComponent **tzcomp)
{
	ECalComponent *comp = NULL;
	ECalComponent *tz = NULL;
	icalcomponent *icalsub = NULL;
	icalcomponent *icaltz = NULL;
	icalcomponent_kind icalkind;

	switch (source_type) {
	case E_CAL_SOURCE_TYPE_EVENT:
		icalkind = ICAL_VEVENT_COMPONENT;
		break;
	case E_CAL_SOURCE_TYPE_TODO:
		icalkind = ICAL_VTODO_COMPONENT;
		break;
	case E_CAL_SOURCE_TYPE_JOURNAL:
		icalkind = ICAL_VJOURNAL_COMPONENT;
		break;
	default:
		g_error ("%s()[%u] Got an unknown value for ECalSourceType, this should not happen.",
		         __func__, __LINE__);
	}

	icalsub = icalcomponent_get_first_component (icalcomp,
	                                             icalkind);
	if (icalsub == NULL) {
		return GNOME_Evolution_Calendar_InvalidObject;
	}

	comp = e_cal_component_new ();
	e_cal_component_set_icalcomponent (comp, icalsub);
	*ecalcomp = e_cal_component_clone (comp);
	g_object_unref (comp);

	icaltz = icalcomponent_get_first_component (icalcomp,
	                                            ICAL_VTIMEZONE_COMPONENT);
	if (icaltz == NULL) {
		/* no TZ - no problem */
		tzcomp = NULL;
	}
	else {
		/* Create a new ECalComponent from the
		 * ical timezone component.
		 */
		tz = e_cal_component_new ();
		e_cal_component_set_icalcomponent (tz,
		                                   icaltz);
		*tzcomp = e_cal_component_clone (tz);
		g_object_unref (tz);
	}

	return GNOME_Evolution_Calendar_Success;
} /* kolab_util_calendar_extract () */
#endif
