/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * test-kolab-contact.c
 *
 *  Created on: 02.09.2010
 *      Author: Hendrik Helwich <h.helwich@tarent.de>,
 *      		Peter Neuhaus <p.neuhaus@tarent.de>,
 *      		Andreas Grau <a.grau@tarent.de>
 */

/*
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor Boston, MA 02110-1301, USA
 */

#include <glib.h>
#include "../../main/src/kolab-conv.h"
#include "../../main/src/kolab-conv.h"
#include "../../main/src/kolab/kolab-constants.h"
#include "test-convert-setup.h"
#include "test-util.h"
#include "testbase.h"

/**
 * contact xml generated by toltec connector/outlook
 */
void
validate_kolab_contact_100 (const I_contact *icontact, gint stage)
{
	GList *list = NULL;

	log_bitmask(stage);

	/*
	 * Common Fields
	 */
	validate_icommon(
		icontact->common,
		"Toltec Connector Open Format (2.0)",
		"TOLTEC.91CCE9951F664909BC8E3F245FAA1DCB",
		IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "Notizen\n123" : "Notizen\r\n123\r\n", /* TODO: handle/remove/convert "\r" ?? */
		NULL,
		ICOMMON_SENSITIVITY_NULL,	/* in contacts not available in evolution, in contacts always set to NULL and real value added to kolab store */
		"2010-08-19 12:50:08",
		"2010-08-19 15:11:27",
		stage);

	/*
	 * Name Fields
	 */
	assert_equal("Nachname", icontact->last_name);
	assert_equal("Vorname", icontact->given_name);
	assert_equal("Weitere Vorn.", icontact->middle_names);
	assert_equal("Anrede", icontact->prefix);
	assert_equal("Namenszusatz", icontact->suffix);
	assert_equal("Anrede Vorname Weitere Vorn. Nachname Namenszusatz", icontact->full_name);
	assert_equal(NULL, icontact->nick_name);

	/*
	 * Misc Fields
	 */
	assert_equal(NULL, icontact->spouse_name);
	assert_equal_gdate(NULL, icontact->birthday);
	assert_equal_gdate(NULL, icontact->anniversary);

	/*
	 * Emails
	 */
	/* XXX: (Toltec) Writes E-Mail address to field display_name. Evolution only provides smtp_address
	 * Check when reading if smtp_address is empty and display_name is valid E-Mail address -> write to smtp.
	 */
#if 0
	assert_email_in_list (icontact->emails, NULL, "em@il1");
	assert_email_in_list (icontact->emails, NULL, "em@il2");
	assert_email_in_list (icontact->emails, NULL, "em@il3");
#endif
	/*  <email>
	 *	<display-name>em@il1</display-name>
	 *  </email>
	 *  <email>
	 *	<display-name>em@il2</display-name>
	 *  </email>
	 *  <email>
	 *	<display-name>em@il3</display-name>
	 *  </email>
	 */


	/*
	 * Addresses
	 */
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_HOME,
			"P Straße", IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL, "P Ort", "P Bundesland", "53123", "Albanien");
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_BUSINESS,
			"G Starße", IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL, "G Ort", "G Bundesland/Kanton", "53111", "Ägypten");
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_OTHER,
			"W Straße", IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL, "W Ort", "W Bundesland", "43212", "Afghanistan");

	/*
	 * Phone Numbers
	 */
	/* FIXME: Order of home/business phone gets mixed up when converting to and from econtact/vcard. */
	assert_phone_in_list (icontact->phone_numbers, "+246 (0228) 123456783423", ICONTACT_PHONE_ASSISTANT);
	assert_phone_in_list (icontact->phone_numbers, "123", ICONTACT_PHONE_BUSINESS_1);
	assert_phone_in_list (icontact->phone_numbers, "45645645", ICONTACT_PHONE_BUSINESS_1);  /* business 2 */
	assert_phone_in_list (icontact->phone_numbers, "45645645", ICONTACT_PHONE_PRIMARY); 	/* business 2 */
	assert_phone_in_list (icontact->phone_numbers, "+503 (8888) 5546 - 56", ICONTACT_PHONE_BUSINESS_FAX);
	assert_phone_in_list (icontact->phone_numbers, "+246 3453453453", ICONTACT_PHONE_CALLBACK);
	assert_phone_in_list (icontact->phone_numbers, "+246 3489678", ICONTACT_PHONE_CAR);
	assert_phone_in_list (icontact->phone_numbers, "454", ICONTACT_PHONE_COMPANY);

	/*
	 * Cooperation Fields
	 */
	assert_equal("Firma", icontact->organization); 	/* ORG[1] */
	assert_equal(NULL, icontact->department); 	/* ORG[2] */
	assert_equal(NULL, icontact->office_location); 	/* ORG[3] */
	assert_equal("Position", icontact->job_title);
	assert_equal(NULL, icontact->profession);
	assert_equal(NULL, icontact->manager_name);
	assert_equal(NULL, icontact->assistant);

	/*
	 * Other Internet Communication Fields
	 */
	assert_equal("http://www.webseite.de", icontact->web_page);
	assert_equal(NULL, icontact->free_busy_url);

	/*
	 * Attached Multimedia Files
	 */
	assert(icontact->photo == NULL);

	/*
	 * Geolocation Fields
	 */
	assert_equal_double(DEGREE_NOT_SET, icontact->latitude);
	assert_equal_double(DEGREE_NOT_SET, icontact->longitude);

	assert_binary_attachment_store_equal(
			"src/libekolabconv/test/resources/imap/Contacts/100.tnef.attach",
			NULL, "application/x-outlook-tnef",
			icontact->common, 0);

	assert_binary_attachment_store_equal(
			"src/libekolabconv/test/resources/imap/Contacts/100-contact-photo.jpg",
			"ContactPicture.jpg", "image/jpeg",
			icontact->common, 1);

	/*
	 * Kolab Store
	 */
	g_assert(icontact->common->kolab_store != NULL);
	list = kolab_store_get_element_list(icontact->common, icontact->common);
	g_assert_cmpint(g_list_length(list), ==, 13);
	validate_kolab_store_xml(list, 0, "sensitivity");
	validate_kolab_store_xml(list, 1, "im-address");
	validate_kolab_store_xml(list, 2, "phone");
	validate_kolab_store_xml(list, 3, "phone");
	validate_kolab_store_xml(list, 4, "phone");
	validate_kolab_store_xml(list, 5, "phone");
	validate_kolab_store_xml(list, 6, "phone");
	validate_kolab_store_xml(list, 7, "phone");
	validate_kolab_store_xml(list, 8, "phone");
	validate_kolab_store_xml(list, 9, "phone");
	validate_kolab_store_xml(list, 10, "phone");
	validate_kolab_store_xml(list, 11, "phone");
	validate_kolab_store_xml(list, 12, "phone");

	/* With restriction of 8 numbers now 11 numbers in kolab store */
#if 0
	assert_phone_in_list (icontact->phone_numbers, "+1 (12121) 34234456456456", ICONTACT_PHONE_HOME_1);
	assert_phone_in_list (icontact->phone_numbers, "4562", ICONTACT_PHONE_HOME_2); 						/* in kolab store not splitted in two numbers */
	assert_phone_in_list (icontact->phone_numbers, "+1 (5675) 7567567 - 566", ICONTACT_PHONE_HOME_FAX);
	assert_phone_in_list (icontact->phone_numbers, "78078", ICONTACT_PHONE_ISDN);
	assert_phone_in_list (icontact->phone_numbers, "+509 (5675) 56756756567567", ICONTACT_PHONE_MOBILE);
	assert_phone_in_list (icontact->phone_numbers, "45634578", ICONTACT_PHONE_OTHER);
	assert_phone_in_list (icontact->phone_numbers, "89789 45", ICONTACT_PHONE_PAGER);
	assert_phone_in_list (icontact->phone_numbers, "456456456", ICONTACT_PHONE_PRIMARY);
	assert_phone_in_list (icontact->phone_numbers, "+246 4444444", ICONTACT_PHONE_RADIO);
	assert_phone_in_list (icontact->phone_numbers, "+246 79789789789", ICONTACT_PHONE_TELEX);
	assert_phone_in_list (icontact->phone_numbers, "5675675533333", ICONTACT_PHONE_TTYTDD);
#endif
	 list = kolab_store_get_element_list(icontact->common, (gpointer) KOLAB_STORE_PTR_CONTACT_NAME);
	 g_assert_cmpint(g_list_length(list), ==, 1);
	 validate_kolab_store_xml(list, 0, "initials");
}

void
validate_kolab_contact_200 (const I_contact *icontact, gint stage)
{
	GList *list = NULL;

	log_bitmask(stage);

	/*
	 * Common Fields
	 */
	validate_icommon(
		icontact->common,
		"KAddressBook 3.3, Kolab resource",
		"uUzgnRqsSd",
		"Bemerkungen\n123\ntest",
		"Familie,Freund,Geschäftliches,Hotel,Kunde,LinuxTag,Partner,Presse,Schule,test",
		ICOMMON_SENSITIVITY_NULL,	/* in contacts not available in evolution, in contacts always set to NULL and real value added to kolab store */
		"2010-06-15 17:33:37",
		"2010-06-30 13:52:32",
		stage);

	/*
	 * Name Fields
	 */
	assert_equal("Nachname", icontact->last_name);
	assert_equal("Vorname", icontact->given_name);
	assert_equal("Zusätzliche Namen", icontact->middle_names);
	assert_equal("Titel", icontact->prefix);
	assert_equal("Ehrentitel nach dem Namen", icontact->suffix);
	assert_equal("Vorname Nachname", icontact->full_name);
	assert_equal("Spitzname", icontact->nick_name);

	/*
	 * Misc Fields
	 */
	assert_equal("Name des Partners", icontact->spouse_name);
	assert_equal_gdate("2010-06-15", icontact->birthday);
	assert_equal_gdate("2010-06-16", icontact->anniversary);

	/*
	 * Emails
	 * TODO: handle incongruent fields
	 */
	assert_email_in_list (icontact->emails, "name@host.de", NULL);
	assert_email_in_list (icontact->emails, "name2@host2.com", NULL);

	/*
	 * Addresses
	 */
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_HOME,
			"Privat Straße\n123", "Privat Postfach",
			"Privat Ort", "Privat Region", "Privat Postleitzahl", "Deutschland");
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_BUSINESS,
			"Geschäft Straße", "Geschäft Postfach",
			"Geschäft Ort", "Geschäft Region", "Geschäft Postleitzahl", "Djibouti");
	assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_OTHER,
			"Inland Straße",
			IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL,
			IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL,
			IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL,
			IS_TEST_STAGE_EVO_TO_INTERN(stage) ? "" : NULL,
			"Deutschland");
	/* 3 following addresses are in kolab store */

	/*
	 * Phone Numbers
	 */
	assert_phone_in_list (icontact->phone_numbers, "123", ICONTACT_PHONE_HOME_1);
	assert_phone_in_list (icontact->phone_numbers, "234", ICONTACT_PHONE_BUSINESS_1);
	assert_phone_in_list (icontact->phone_numbers, "345", ICONTACT_PHONE_COMPANY);
	assert_phone_in_list (icontact->phone_numbers, "456", ICONTACT_PHONE_CALLBACK);
	assert_phone_in_list (icontact->phone_numbers, "567", ICONTACT_PHONE_BUSINESS_FAX);
	assert_phone_in_list (icontact->phone_numbers, "678", ICONTACT_PHONE_MOBILE);
	assert_phone_in_list (icontact->phone_numbers, "789", ICONTACT_PHONE_RADIO);
	assert_phone_in_list (icontact->phone_numbers, "890", ICONTACT_PHONE_TTYTDD);

	/*
	 * Cooperation Fields
	 */
	assert_equal("Organisation", icontact->organization); 	/* ORG[1] */
	assert_equal("Abteilung", icontact->department); 	/* ORG[2] */
	assert_equal("Büro", icontact->office_location); 	/* ORG[3] */
	assert_equal("Details Titel", icontact->job_title);
	assert_equal("Beruf", icontact->profession);
	assert_equal("Name des Managers", icontact->manager_name);
	assert_equal("Name des Assistenten", icontact->assistant);

	/*
	 * Other Internet Communication Fields
	 */
	assert_equal("http://www.google.de/", icontact->web_page);
	assert_equal(NULL, icontact->free_busy_url);

	/*
	 * Attached Multimedia Files
	 */
	assert(icontact->common->kolab_attachment_store == NULL);

	assert_binary_attachment_equal(
		"./src/libekolabconv/test/resources/imap/Contacts/200-contact-photo.png",
		"kolab-picture.png",
		"image/png",
		icontact->photo);

	/*
	 * Geolocation Fields
	 */
	assert_equal_double(49.5, icontact->latitude);
	assert_equal_double(-126.900001525879, icontact->longitude);

	/*
	 * X-Custom (Instant Messenger & Blog URL)
	 */
	assert_custom_field_in_list(icontact->custom_list, KLBX_CNT_X_CUSTOM_APP_AIM, "All", "aim");
	assert_custom_field_in_list(icontact->custom_list, KLBX_CNT_X_CUSTOM_APP_GADUGADU, "All", "gadu");
	assert_custom_field_in_list(icontact->custom_list, KLBX_CNT_X_CUSTOM_APP_GROUPWISE, "All", "groupwise");
	assert_custom_field_in_list(icontact->custom_list, KLBX_CNT_X_CUSTOM_APP_ICQ, "All", "icq");
	/* evolution can save maximum 4 im's, rest is saved in kolab store! */
	assert_custom_field_in_list(icontact->custom_list, "", KLBX_CNT_X_CUSTOM_NAME_BLOGFEED, "http://www.blog.de");

	/*
	 * Kolab Store
	 */
	g_assert(icontact->common->kolab_store != NULL);

	list = kolab_store_get_element_list(icontact->common, icontact->common);
	g_assert_cmpint(g_list_length(list), ==, IS_TEST_STAGE_EVO_TO_INTERN(stage) ? 18:11);	/* x-custom fields can not be set before conversion to evolution */
	validate_kolab_store_xml(list, 0, "sensitivity");
	validate_kolab_store_xml(list, 1, "im-address");
	validate_kolab_store_xml(list, 2, "role");
	validate_kolab_store_xml(list, 3, "phone");
	validate_kolab_store_xml(list, 4, "phone");
	validate_kolab_store_xml(list, 5, "phone");
	validate_kolab_store_xml(list, 6, "phone");
	validate_kolab_store_xml(list, 7, "address");
	validate_kolab_store_xml(list, 8, "address");
	validate_kolab_store_xml(list, 9, "address");
	validate_kolab_store_xml(list, 10, "preferred-address");

	/* With restriction of 8 numbers now 4 numbers in Kolab store */
#if 0
	assert_phone_in_list (icontact->phone_numbers, "901", ICONTACT_PHONE_TELEX);
	assert_phone_in_list (icontact->phone_numbers, "109", ICONTACT_PHONE_ISDN);
	assert_phone_in_list (icontact->phone_numbers, "210", ICONTACT_PHONE_ASSISTANT);
	assert_phone_in_list (icontact->phone_numbers, "321", ICONTACT_PHONE_PAGER);
#endif

	/* The x-custom fields are written manually to kolab store while writing evolution */
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 11, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 12, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 13, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 14, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 15, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 16, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 17, "x-custom") : (void)NULL ;
}

void
validate_kolab_contact_201 (const I_contact *icontact, gint stage)
{
	(void)stage;

	assert(g_list_length(icontact->common->kolab_attachment_store) == 2);

	assert_binary_attachment_equal(
		"./src/libekolabconv/test/resources/imap/Contacts/201-kolab-logo.png",
		"kolab-logo.png",
		"image/png",
		icontact->common->kolab_attachment_store->data);

	assert_binary_attachment_equal(
		"./src/libekolabconv/test/resources/imap/Contacts/201-sound",
		"sound",
		"audio/unknown",
		icontact->common->kolab_attachment_store->next->data);
}

void
validate_kolab_contact_300 (const I_contact *icontact, gint stage)
{
	GList *list = NULL;
	Phone_number* pn;
	Email *eml;

	log_bitmask(stage);

	/*
	 * Common Fields
	 */
	validate_icommon(
		icontact->common,
		"KAddressBook 3.3, Kolab resource",
		"VYa3AT5NtK",
		"Blablub\n1234",
		"Freund,Hotel",
		ICOMMON_SENSITIVITY_NULL,	/* in contacts not available in evolution, in contacts always set to NULL and real value added to kolab store */
		"2010-12-14 14:48:15",
		"2010-12-14 14:48:15",
		stage);

	/*
	 * Name Fields
	 */
	assert_equal("Müller", icontact->last_name);
	assert_equal("Peter", icontact->given_name);
	assert_equal("Klaus", icontact->middle_names);
	assert_equal("Prof.", icontact->prefix);
	assert_equal("von Pinkepinke", icontact->suffix);
	assert_equal("Peter Müller", icontact->full_name);
	assert_equal("Kläuschen", icontact->nick_name);

	/*
	 * Misc Fields
	 */
	assert_equal("Gabi", icontact->spouse_name);
	assert_equal_gdate("1913-12-11", icontact->birthday);
	assert_equal_gdate("1901-12-23", icontact->anniversary);

	/*
	 * Emails
	 * TODO: handle incongruent fields
	 */
	assert_email_in_list (icontact->emails, "peter@uni.de", NULL);
	assert_email_in_list (icontact->emails, "klaus@gmx.net", NULL);

	/*
	 * Addresses
	 */
	/* Nothing */

	/*
	 * Phone Numbers
	 */
	assert_phone_in_list (icontact->phone_numbers, "110", ICONTACT_PHONE_HOME_1);
	assert_phone_in_list (icontact->phone_numbers, "112", ICONTACT_PHONE_BUSINESS_1);
	assert_phone_in_list (icontact->phone_numbers, "01801234567", ICONTACT_PHONE_MOBILE);


	/*
	 * Cooperation Fields
	 */
	assert_equal("Die Firma", icontact->organization); 	/* ORG[1] */
	assert_equal("Abteilung 42", icontact->department); 	/* ORG[2] */
	assert_equal("3. Stock", icontact->office_location); 	/* ORG[3] */
	assert_equal("MC Checker", icontact->job_title);
	assert_equal("Prof", icontact->profession);
	/* assert_equal("", icontact->role); */ /* not available in evolution => profession */
	assert_equal("Dr Boss", icontact->manager_name);
	assert_equal("Mr Asi", icontact->assistant);

	/*
	 * Other Internet Communication Fields
	 */
	assert_equal("http://www.peter.de", icontact->web_page);
	assert_equal(NULL, icontact->free_busy_url);

	/*
	 * Attached Multimedia Files
	 */
	/* TODO: handle attachments! */
	assert(icontact->photo == NULL);

	/*
	 * Geolocation Fields
	 */
	assert_equal_double(-12.6000003814697, icontact->latitude);
	assert_equal_double(-55.2000007629395, icontact->longitude);

	/*
	 * X-Custom (Instant Messenger & Blog URL)
	 */
	assert_custom_field_in_list(icontact->custom_list, "", KLBX_CNT_X_CUSTOM_NAME_BLOGFEED, "http://p.blog.de");

	/*
	 * Kolab Store
	 */
	g_assert(icontact->common->kolab_store != NULL);
	/* validate number of locations with unknown elements */
	/* g_assert_cmpint(g_hash_table_size(icontact->common->kolab_store), ==, 2); */
	/* get top level unknown elements */
	 list = kolab_store_get_element_list(icontact->common, (gpointer) KOLAB_STORE_PTR_CONTACT_NAME);
	 g_assert_cmpint(g_list_length(list), ==, 2);
	 validate_kolab_store_xml(list, 0, "unknown42");
	 validate_kolab_store_xml(list, 1, "unknown43");

	list = kolab_store_get_element_list(icontact->common, icontact->common);
	g_assert_cmpint(g_list_length(list), ==, IS_TEST_STAGE_EVO_TO_INTERN(stage) ? 12:7);
	validate_kolab_store_xml(list, 0, "unknown1");
	validate_kolab_store_xml(list, 1, "sensitivity");
	validate_kolab_store_xml(list, 2, "unknown2");
	validate_kolab_store_xml(list, 3, "role");
	validate_kolab_store_xml(list, 4, "unknown3");
	validate_kolab_store_xml(list, 5, "preferred-address");
	validate_kolab_store_xml(list, 6, "unknown4");

	/* The x-custom fields are written manually to kolab store while writing evolution */
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 7, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 8, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 9, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 10, "x-custom") : (void)NULL ;
	IS_TEST_STAGE_EVO_TO_INTERN(stage) ? validate_kolab_store_xml(list, 11, "x-custom") : (void)NULL ;

	/* test elements below phone number elements */

	pn = assert_phone_in_list (icontact->phone_numbers, "110", ICONTACT_PHONE_HOME_1);
	list = kolab_store_get_element_list(icontact->common, pn);
	g_assert_cmpint(g_list_length(list), ==, 3);
	validate_kolab_store_xml(list, 0, "a");
	validate_kolab_store_xml(list, 1, "b");
	validate_kolab_store_xml(list, 2, "c");

	pn = assert_phone_in_list (icontact->phone_numbers, "112", ICONTACT_PHONE_BUSINESS_1);
	list = kolab_store_get_element_list(icontact->common, pn);
	g_assert_cmpint(g_list_length(list), ==, 0);

	pn = assert_phone_in_list (icontact->phone_numbers, "01801234567", ICONTACT_PHONE_MOBILE);
	list = kolab_store_get_element_list(icontact->common, pn);
	g_assert_cmpint(g_list_length(list), ==, 1);
	validate_kolab_store_xml(list, 0, "unknown5");


	eml = assert_email_in_list(icontact->emails, "peter@uni.de", NULL);
	list = kolab_store_get_element_list(icontact->common, eml);
	g_assert_cmpint(g_list_length(list), ==, 2);
	validate_kolab_store_xml(list, 1, "unknown6");
	validate_kolab_store_xml(list, 0, "display-name");

	eml = assert_email_in_list(icontact->emails, "klaus@gmx.net", NULL);
	list = kolab_store_get_element_list(icontact->common, eml);
	g_assert_cmpint(g_list_length(list), ==, 1);
	validate_kolab_store_xml(list, 0, "display-name");

#if 0
	Address *addr;

	addr = assert_address_in_list(icontact->addresses, ICONTACT_ADDR_TYPE_HOME,
				"street", "Privat Postfach", "Privat Ort", "Privat Region", "Privat Postleitzahl", "Deutschland");

	list = kolab_store_get_element_list(icontact->common, addr);
	g_assert_cmpint(g_list_length(list), ==, 2);
	validate_kolab_store_xml(list, 1, "unknown7");
	validate_kolab_store_xml(list, 0, "x-kde-type");
#endif
}



void
validate_kolab_contact_302 (const I_contact *icontact, gint stage)
{
	/*
	 * Attached Multimedia Files
	 */
	assert(icontact->common->kolab_attachment_store == NULL);

	assert_binary_attachment_equal(
		"./src/libekolabconv/test/resources/imap/Contacts/100-contact-photo.jpg",
		IS_TEST_STAGE_INTERN_TO_EVO(stage) ? "kolab-picture.jpg" : "ContactPicture.jpg",
		"image/jpeg",
		icontact->photo);
}


void
validate_kolab_contact_500 (const I_contact *icontact, gint stage)
{
	GList *list = NULL;

	(void)stage;

	assert_email_in_list (icontact->emails, "a1@b.d", NULL);
	assert_email_in_list (icontact->emails, "a2@b.d", NULL);
	assert_email_in_list (icontact->emails, "a3@b.d", NULL);
	assert_email_in_list (icontact->emails, "a4@b.d", NULL);
	g_assert_cmpint(g_list_length(icontact->emails), ==, 4);

	list = kolab_store_get_element_list(icontact->common, icontact->common);
	validate_kolab_store_xml(list, 0, "sensitivity");
	validate_kolab_store_xml(list, 1, "email");
	validate_kolab_store_xml(list, 2, "email");
	validate_kolab_store_xml(list, 3, "preferred-address");
}
