/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/***************************************************************************
 *            kolab-util-folder.c
 *
 *  Mon Feb 28 17:57:21 2011
 *  Copyright  2011  Christian Hilberg
 *  <hilberg@kernelconcepts.de>
 ****************************************************************************/

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with main.c; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

/*----------------------------------------------------------------------------*/

#include <glib.h>

#include "kolab-util-folder.h"

/*----------------------------------------------------------------------------*/

static gchar *kolab_folder_type_inv_map[] = {
	"---INVALID---",	/* KOLAB_FOLDER_TYPE_INVAL */
	"---UNKNOWN---",	/* KOLAB_FOLDER_TYPE_UNKNOWN */

	"mail",			/* KOLAB_FOLDER_TYPE_EMAIL */
	"mail.inbox",		/* KOLAB_FOLDER_TYPE_EMAIL_INBOX */
	"mail.drafts",		/* KOLAB_FOLDER_TYPE_EMAIL_DRAFTS */
	"mail.sentitems",	/* KOLAB_FOLDER_TYPE_EMAIL_SENTITEMS */
	"mail.junkemail",	/* KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL */

	"event",		/* KOLAB_FOLDER_TYPE_EVENT */
	"event.default",	/* KOLAB_FOLDER_TYPE_EVENT_DEFAULT */
	"journal",		/* KOLAB_FOLDER_TYPE_JOURNAL */
	"journal.default",	/* KOLAB_FOLDER_TYPE_JOURNAL_DEFAULT */
	"task",			/* KOLAB_FOLDER_TYPE_TASK */
	"task.default",		/* KOLAB_FOLDER_TYPE_TASK_DEFAULT */
	"note",			/* KOLAB_FOLDER_TYPE_NOTE */
	"note.default",		/* KOLAB_FOLDER_TYPE_NOTE_DEFAULT */

	"contact",		/* KOLAB_FOLDER_TYPE_CONTACT */
	"contact.default"	/* KOLAB_FOLDER_TYPE_CONTACT_DEFAULT */
};

static KolabFolderTypeID kolab_folder_type_nums[KOLAB_FOLDER_LAST_TYPE];
static GHashTable *kolab_folder_type_map = NULL;
static gboolean kolab_util_folder_is_initialized = FALSE;

/*----------------------------------------------------------------------------*/

static void
kolab_util_folder_type_map_init (void)
{
	gint ii = 0;
	kolab_folder_type_map = g_hash_table_new (g_str_hash, g_str_equal);
	for (ii = 0; ii < KOLAB_FOLDER_LAST_TYPE; ii++) {
		kolab_folder_type_nums[ii] = ii;
		g_hash_table_insert (kolab_folder_type_map,
		                     kolab_folder_type_inv_map[ii],
		                     &(kolab_folder_type_nums[ii]));
	}
}

static void
kolab_util_folder_type_map_uninit (void)
{
	g_hash_table_destroy (kolab_folder_type_map);
	kolab_folder_type_map = NULL;
}

/*----------------------------------------------------------------------------*/

void
kolab_util_folder_init (void)
{
	if (kolab_util_folder_is_initialized)
		return;
	
	kolab_util_folder_type_map_init ();

	kolab_util_folder_is_initialized = TRUE;
}

void
kolab_util_folder_shutdown (void)
{
	if (! kolab_util_folder_is_initialized)
		return;
	
	kolab_util_folder_type_map_uninit ();

	kolab_util_folder_is_initialized = FALSE;
}

KolabFolderTypeID
kolab_util_folder_type_get_id (const gchar *typestring)
{
	/* when looking up a value from kolab_folder_type_map, store
	 * it in gpointer, check for NULL, then dereference and cast
	 * to KolabFolderTypeID
	 */
	gpointer map_entry = NULL;
	KolabFolderTypeID id = KOLAB_FOLDER_TYPE_INVAL;

	g_assert (kolab_util_folder_is_initialized);
	g_assert (typestring != NULL);

	map_entry = g_hash_table_lookup (kolab_folder_type_map, typestring);

	if (map_entry == NULL)
		return KOLAB_FOLDER_TYPE_INVAL;

	id = *((KolabFolderTypeID*)map_entry);
	return id;
}

const gchar*
kolab_util_folder_type_get_string (KolabFolderTypeID foldertype)
{
	g_assert (kolab_util_folder_is_initialized);
	g_assert (foldertype < KOLAB_FOLDER_LAST_TYPE);
	return kolab_folder_type_inv_map[foldertype];
}

KolabFolderContextID
kolab_util_folder_type_map_to_context_id (KolabFolderTypeID type_id)
{
	g_assert (kolab_util_folder_is_initialized);

	/* TODO better handling here */
	g_assert ((type_id > KOLAB_FOLDER_TYPE_INVAL) &&
	          (type_id < KOLAB_FOLDER_LAST_TYPE));

	if ((type_id >= KOLAB_FOLDER_TYPE_UNKNOWN) &&
	    (type_id <= KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL))
		return KOLAB_FOLDER_CONTEXT_EMAIL;

	if ((type_id >= KOLAB_FOLDER_TYPE_EVENT) &&
	    (type_id <= KOLAB_FOLDER_TYPE_NOTE_DEFAULT))
		return KOLAB_FOLDER_CONTEXT_CALENDAR;

	return KOLAB_FOLDER_CONTEXT_CONTACT;
}

gboolean
kolab_util_folder_type_match_with_context_id (KolabFolderTypeID type_id,
                                              KolabFolderContextID context_id)
{
	g_assert (kolab_util_folder_is_initialized);

	/* TODO better handling here */
	g_assert ((type_id > KOLAB_FOLDER_TYPE_INVAL) &&
	          (type_id < KOLAB_FOLDER_LAST_TYPE));
	g_assert ((context_id > KOLAB_FOLDER_CONTEXT_INVAL) &&
	          (context_id < KOLAB_FOLDER_LAST_CONTEXT));

	switch (context_id) {
		case KOLAB_FOLDER_CONTEXT_EMAIL:
			if ((type_id < KOLAB_FOLDER_TYPE_UNKNOWN) ||
			    (type_id > KOLAB_FOLDER_TYPE_EMAIL_JUNKEMAIL))
				return FALSE;
			break;
		case KOLAB_FOLDER_CONTEXT_CALENDAR:
			if ((type_id < KOLAB_FOLDER_TYPE_EVENT) ||
			    (type_id > KOLAB_FOLDER_TYPE_NOTE_DEFAULT))
				return FALSE;
			break;
		case KOLAB_FOLDER_CONTEXT_CONTACT:
			if ((type_id < KOLAB_FOLDER_TYPE_CONTACT) ||
			    (type_id > KOLAB_FOLDER_TYPE_CONTACT_DEFAULT))
				return FALSE;
			break;
		default:
			/* can't happen */
			g_assert_not_reached ();

	}
	return TRUE;
}

/*----------------------------------------------------------------------------*/
