/*---[ netfilter-script.c ]-------------------------------------------
 * Copyright (C) 2000 Tomas Junnonen (majix@sci.fi)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Functions to write the netfilter shell scripts
 *--------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>   
#include <errno.h>

#include "druid.h"
#include "netfilter-script.h"

/* [ write_netfilter_script ]
 * Creates the netfilter shell script
 */
void
write_netfilter_script (Druid *data)
{
	gboolean masq = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->masq));
	gboolean tos = GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->tos));

	gchar *scriptpath = FIRESTARTER_RULES_DIR "/firestarter/firewall.sh";

	FILE *script = fopen (scriptpath, "w");
	if (script == NULL) {
	/* use perror for sane error messages */
		perror(scriptpath);
		return;
	}

	chmod (scriptpath, 00700);

/* Init variables and external files */
	fprintf (script, "#!/bin/sh\n");
	fprintf (script, "#Generated by Firestarter " VERSION ", NETFILTER in use\n\n");

	if (gnome_config_get_bool ("/firestarter/Druid/locatesbins=FALSE"))
		fprintf (script, "IPT=`which iptables`\nMPB=`which modprobe`\nLSM=`which lsmod`\n\n");
	else
		fprintf (script, "IPT=/sbin/iptables\nMPB=/sbin/modprobe\nLSM=/sbin/lsmod\n\n");

	fprintf (script, "#Some distributions still load ipchains\n");
	fprintf (script, "$LSM | grep ipchains -q -s && rmmod ipchains\n\n");

	/* Hack for autoloading of netfilter modules (such as the reject chain or ToS) - must be done before
	chains are flushed.*/

	fprintf (script, "#Loading Requested Kernel Modules\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ip_conntrack > /dev/null ); then\n");
	fprintf (script, "$MPB ip_conntrack\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_REJECT > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_REJECT\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_REDIRECT > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_REDIRECT\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_TOS > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_TOS\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_MASQUERADE > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_MASQUERADE\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_MIRROR > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_MIRROR\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep ipt_LOG > /dev/null ); then\n");
	fprintf (script, "$MPB ipt_LOG\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep iptable_mangle > /dev/null ); then\n");
	fprintf (script, "$MPB iptable_mangle\nfi\n");
	fprintf (script, "if ! ( $LSM | /bin/grep iptable_nat > /dev/null ); then\n");
	fprintf (script, "$MPB iptable_nat\nfi\n\n");

	/* The network interface */
	fprintf (script, "IF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->extdevice)));
	/* The internal network interface */
	if (masq)
		fprintf (script, "INIF=%s\n", gtk_entry_get_text (GTK_ENTRY (data->intdevice)));

	/* IP address of interface */
	fprintf (script, "IP=`/sbin/ifconfig $IF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
	/* Netmask/net of external interface */
	fprintf (script, "MASK=`/sbin/ifconfig $IF | grep Mas | cut -d : -f 4`\n");
	fprintf (script, "NET=$IP/$MASK\n\n");
	/* Internal network*/
	if (masq) {
		if (GTK_WIDGET_IS_SENSITIVE (GTK_WIDGET (data->auto_int_ip)))
			fprintf (script, "INNET=%s\n\n", gtk_entry_get_text (GTK_ENTRY (data->intrange)));
		else {
			fprintf (script, "INIP=`/sbin/ifconfig $INIF | grep inet | cut -d : -f 2 | cut -d \\  -f 1`\n");
			fprintf (script, "INMASK=`/sbin/ifconfig $INIF | grep Mas | cut -d : -f 4`\n");
			fprintf (script, "INNET=$INIP/$INMASK\n");
		}
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
		fprintf (script, "#Modules: Loading bsdcomp module\n");
        fprintf (script, "if ! ( $LSM | /bin/grep bsd_comp > /dev/null ); then\n");
		fprintf (script, "$MPB bsd_comp\nfi\n");
        fprintf (script, "if ! ( $LSM | /bin/grep _deflate > /dev/null ); then\n");
		fprintf (script, "$MPB ppp_deflate\nfi\n\n");
	}


	fprintf (script, "#Delete user made chains. Flush and zero the chains.\n");
	fprintf (script, "$IPT -F\n$IPT -X\n$IPT -Z\n\n");

/*	fprintf (script, "#Set default chain policy to DROP.\n");
	fprintf (script, "$IPT -P input DROP\n$IPT -P OUTPUT DROP\n$IPT -P forward DROP\n\n");
*/
	fprintf (script, "#Delete `nat' and `mangle' chains.\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_mangle > /dev/null ); then\n");
	fprintf (script, "$IPT -t mangle -F\nfi\n");
	fprintf (script, "if ( $LSM | /bin/grep iptable_nat > /dev/null ); then\n");
	fprintf (script, "$IPT -t nat -F\nfi\n\n");

	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "#Create a new log and drop (LD) convenience chain.\n");
	fprintf (script, "$IPT -N LD\n$IPT -A LD -j LOG\n$IPT -A LD -j DROP\n\n");
	fprintf (script, "STOP=LD\n\n");
	}
	else {
	fprintf (script, "#Create a new log and reject (LR) convenience chain.\n");
	fprintf (script, "$IPT -N LR\n$IPT -A LR -j LOG\n$IPT -A LR -j REJECT\n\n");
	fprintf (script, "STOP=LR\n\n");
	}

	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_throughput)))
			fprintf (script, "TOSOPT=8\n\n");
	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_reliability)))
			fprintf (script, "TOSOPT=4\n\n");
	if (gtk_toggle_button_get_active (GTK_RADIO_BUTTON (data->tos_option_delay)))
			fprintf (script, "TOSOPT=16\n\n");

	/* Pipe in the modrules files */
	fprintf (script, "#Deny all traffic on these ports, without logging\n");
	fprintf (script, "if [ -e "FIRESTARTER_RULES_DIR "/firestarter/do-not-log-ports ]\n then\n"
		"source " FIRESTARTER_RULES_DIR "/firestarter/do-not-log-ports\nfi\n\n");

	fprintf (script, "#Deny all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/deny-all\n\n");

	fprintf (script, "#Portforwarding rules\n");
	fprintf (script, "if [ -e "FIRESTARTER_RULES_DIR "/firestarter/portfw ]\n then\n"
		"source " FIRESTARTER_RULES_DIR "/firestarter/portfw\nfi\n\n");

	fprintf (script, "#Allow all traffic from these machines\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-all\n\n");
	fprintf (script, "#Allow a specific service to a specific machine\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-machine\n\n");
	fprintf (script, "#Allow a specific service to everyone\n");
	fprintf (script, "source " FIRESTARTER_RULES_DIR "/firestarter/allow-service-all\n\n");

/* GOOD IDEA(tm) RULES */
	fprintf (script, "#Allow all traffic on the loopback interface\n");
	fprintf (script, "$IPT -t filter -A INPUT -i lo -s 0/0 -d 0/0 -j ACCEPT\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -o lo -s 0/0 -d 0/0 -j ACCEPT\n\n");

	fprintf (script, "#Turn on source address verification in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/conf/all/rp_filter ]; then\n"
	"  for f in /proc/sys/net/ipv4/conf/*/rp_filter\n  do\n   echo 2 > $f\n  done\nfi\n\n");

	fprintf (script, "#Turn on syn cookies protection in kernel\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/tcp_syncookies ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/tcp_syncookies\nfi\n\n");

	fprintf (script, "#ICMP Dead Error Messages protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_ignore_bogus_error_responses\nfi\n\n");

	fprintf (script, "#ICMP Broadcasting protection\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts ]; then\n"
	"  echo 1 > /proc/sys/net/ipv4/icmp_echo_ignore_broadcasts\nfi\n\n");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pppcheck))) {
		fprintf (script, "#Turn on dynamic TCP/IP address hacking\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n");
	} else {
		fprintf (script, "#Turn off dynamic TCP/IP address hacking\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_dynaddr ]; then\n"
		"  echo 0 > /proc/sys/net/ipv4/ip_dynaddr\nfi\n\n");
	}

	fprintf (script, "#Doubling current limit for ip_conntrack\n");
	fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_conntrack_max ]; then\n"	
	"  echo 16384 > /proc/sys/net/ipv4/ip_conntrack_max\nfi\n\n");


/* ICMP RULES */

if GTK_WIDGET_IS_SENSITIVE (data->icmp)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_echo))) {
		fprintf (script, "#ICMP: Ping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-request -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Ping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-request -m limit --limit 1/s -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type echo-reply -m limit --limit 1/s -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_traceroute))) {
		fprintf (script, "#ICMP: Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33434 -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33434 -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_tracert))) {
		fprintf (script, "#ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: MS Traceroute Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type destination-unreachable -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_unreach))) {
		fprintf (script, "#ICMP: Unreachable Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Unreachable Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type host-unreachable -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_timestamp))) {
		fprintf (script, "#ICMP: Timestamping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Timestamping Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-request -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type timestamp-reply -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_masking))) {
		fprintf (script, "#ICMP: Address Masking\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Address Masking\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-request -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type address-mask-reply -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_redir))) {
		fprintf (script, "#ICMP: Redirection Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type redirect -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Redirection Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type redirect -m limit --limit 2/s -j ACCEPT\n");
	}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->icmp_quench))) {
		fprintf (script, "#ICMP: Source Quench Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type source-quench -j $STOP\n");
	}
		else {
		fprintf (script, "#ICMP: Source Quench Requests\n");
		fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET --icmp-type source-quench -m limit --limit 2/s -j ACCEPT\n\n");
	}
}
else {
	fprintf (script, "#Allowing all ICMP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p icmp -s 0/0 -d $NET -m limit --limit 1/s -j ACCEPT\n\n");
}

/* MASQ RULES */
	if (masq) {
		fprintf (script, "#FTP fix for masqed machines\n");
		fprintf (script, "#$MPB ip_nat_ftp\n\n");

		fprintf (script, "#Turn on IP forwarding\n");
		fprintf (script, "if [ -e /proc/sys/net/ipv4/ip_forward ]\n then\n"
		"  echo 1 > /proc/sys/net/ipv4/ip_forward\nfi\n\n");

		fprintf (script, "#Forward Int/Ext & Ext/Int Traffic before Masquerading\n");
		fprintf (script, "$IPT -t filter -A FORWARD -d 0/0 -s $INNET -o $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A FORWARD -d $INNET -j ACCEPT\n");

		fprintf (script, "#Masquerade outgoing traffic\n");
		fprintf (script, "$IPT -t nat -A POSTROUTING -o $IF -j MASQUERADE\n\n");

		fprintf (script, "#Don't masq external interface traffic\n");
		fprintf (script, "$IPT -t nat -A POSTROUTING -s $NET -d 0/0 -j ACCEPT\n\n");

		fprintf (script, "#Allow traffic from internal network going anywhere\n");
		fprintf (script, "$IPT -t filter -A INPUT -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A OUTPUT -s $INNET -d 0/0 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A OUTPUT -p icmp -s $INNET -d 0/0 -j ACCEPT\n\n");

		fprintf (script, "#Setting default forwarding rule\n");
		fprintf (script, "$IPT -t filter -P FORWARD DROP\n\n\n");

	}

/* TOS RULES */
if (tos) {

	fprintf (script, "#Altering Type of Service (ToS) flags\n\n");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_client))) {
	fprintf (script, "#ToS: Client Applications\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 20:21 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 68 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 80 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 443 --set-tos $TOSOPT\n\n");
}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_server))) {
	fprintf (script, "#ToS: Server Applications\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 20:21 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 25 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 53 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 67 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 80 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 110 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 143 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 443 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 1812 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 1813 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 2401 --set-tos $TOSOPT\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 8080 --set-tos $TOSOPT\n\n");
}

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->tos_X))) {
	fprintf (script, "#ToS: The X Window System\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 22 --set-tos 0x10\n");
	fprintf (script, "$IPT -t mangle -A OUTPUT -p tcp -j TOS --dport 6000:6015 --set-tos 0x08\n\n");
 }		
}

	fprintf (script, "#Block nonroutable IPs\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 1.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 2.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 7.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 23.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 27.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 31.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 41.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 45.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 60.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 68.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 69.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 70.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 71.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 80.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 88.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 90.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 91.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 92.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 100.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 111.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 112.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 127.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 127.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 128.66.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 172.16.0.0/12 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 192.168.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 197.0.0.0/16 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 201.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 220.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 222.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 240.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 242.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 244.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 251.0.0.0/8 -d $NET -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 254.0.0.0/8 -d $NET -i $IF -j $STOP\n\n");

	fprintf (script, "#Block Back Orifice\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 31337 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 31337 -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31337 -m limit --limit 2/minute -j $STOP\n\n");	
	}
	fprintf (script, "#Block Trinity v3\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 33270 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 33270 -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 33270 -m limit --limit 2/minute -j $STOP\n\n");	
	}
	fprintf (script, "#Block Subseven (1.7/1.9)\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 1234 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 6711 -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1234 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 6711 -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1234 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 6711 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n\n");	
	}
	fprintf (script, "#Block Stacheldraht\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 16660 --syn -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 60001 --syn -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 16660 --syn -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 60001 --syn -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 16660 --syn -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 60001 --syn -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n\n");	
	}
	fprintf (script, "#Block NetBus\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 12345:12346 -m limit --limit 2/minute -j $STOP\n\n");	
	}
	fprintf (script, "#Block Trin00\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 1524 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 27665 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 27444 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 31335 -m limit --limit 2/minute -j $STOP\n\n");
	if (gnome_config_get_bool ("/firestarter/Druid/selectdeny=TRUE")) {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1524 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 27665 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 27444 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31335 -m limit --limit 2/minute -j $STOP\n\n");
	}
	else {
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1524 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p tcp -s $NET -d 0/0 --dport 27665 -m limit --limit 2/minute -j $STOP --reject-with tcp-reset\n");	
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 27444 -m limit --limit 2/minute -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -p udp -s $NET -d 0/0 --dport 31335 -m limit --limit 2/minute -j $STOP\n\n");
	}
	fprintf (script, "#Block Multicast\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 0/0 -d 224.0.0.0/8 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 224.0.0.0/8 -d 0/0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 0/0 -d 224.0.0.0/8 -j $STOP\n\n");

	fprintf (script, "#Block Packets with Stuffed Routing\n");
	fprintf (script, "$IPT -t filter -A INPUT -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -d 0.0.0.0 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -s 255.255.255.255 -j $STOP\n");
	fprintf (script, "$IPT -t filter -A OUTPUT -d 0.0.0.0 -j $STOP\n\n");
	
	fprintf (script, "#Block Fragmented Packets\n");
	fprintf (script, "$IPT -t filter -A INPUT -f -m limit --limit 10/minute -j $STOP\n\n");

/* RULES BASED ON WIZARD SELECTIONS */

	/* Allow DHCP */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcpcheck))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 67:68 -i $IF -j ACCEPT\n\n");
	}

if GTK_WIDGET_IS_SENSITIVE (data->services)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ftp))) {
		fprintf (script, "#FTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 20  ! --syn -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 21 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ssh))) {
		fprintf (script, "#SSH\n");		
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 22 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->telnet))) {
		fprintf (script, "#Telnet\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 23 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->smtp))) {
		fprintf (script, "#SMTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 25 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dns))) {
		fprintf (script, "#DNS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 53 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 53 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->finger))) {
		fprintf (script, "#Finger\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 79 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->www))) {
		fprintf (script, "#HTTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 80 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->sslweb))) {
		fprintf (script, "#SSL HTTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 443 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->pop))) {
		fprintf (script, "#POP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 110 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->imap))) {
		fprintf (script, "#IMAP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 143 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ident))) {
		fprintf (script, "#IDENT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 113 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 113 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nntp))) {
		fprintf (script, "#NNTP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 119 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ntp))) {
		fprintf (script, "#NTP\n");		
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d $NET --dport 123 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d $NET --dport 123 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->samba))) {
		fprintf (script, "#SAMBA\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 445 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 445 -i $IF -j ACCEPT\n\n");
	}
 	else {
		fprintf (script, "#Block SAMBA\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 445 -i $IF -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 445 -i $IF -j $STOP\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->ipsec))) {
		fprintf (script, "#IPSec / KLIPS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 500 -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p 51 -s 0/0 -d 0/0 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->routed))) {
		fprintf (script, "#Routed\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 520 -j ACCEPT\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->nfs))) {
		fprintf (script, "#NFS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 2049 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 2049 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block NFS\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 2049 -i $IF -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 2049 -i $IF -j $STOP\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->x))) {
		fprintf (script, "#Xwindows\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j ACCEPT\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "#Block Xwindows\n");
		fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j $STOP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 6000:6015 -i $IF -j $STOP\n\n");
	}
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (data->dhcp))) {
		fprintf (script, "#DHCP\n");
		fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 67:68 -i $IF -j ACCEPT\n\n");
	}
}

else {
	fprintf (script, "#Block NFS\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 2049 -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 2049 -i $IF -j $STOP\n\n");
	
	fprintf (script, "#Block SAMBA\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 137:139 -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p tcp -s 0/0 -d 0/0 --dport 445 -i $IF -j $STOP\n");
	fprintf (script, "$IPT -t filter -A INPUT -p udp -s 0/0 -d 0/0 --dport 445 -i $IF -j $STOP\n\n");
}


/* REQUIRED RULES */

	fprintf (script, "#Allow ICMP Output\n");
	fprintf (script, "$IPT  -A OUTPUT -p icmp -s $NET -d 0/0 -j ACCEPT\n\n");

	fprintf (script, "#Open ports for inbound established connections\n");
	fprintf (script, "#SSH fix\n");
	fprintf (script, "$IPT  -A INPUT -p tcp --sport 22 --dport 513:65535 ! --syn -m state --state RELATED -j ACCEPT\n\n");
	fprintf (script, "#FTP Data fix\n");
	fprintf (script, "$IPT  -A INPUT -p tcp --sport 20 --dport 1023:65535 ! --syn -m state --state RELATED -j ACCEPT\n");
	fprintf (script, "$IPT  -A INPUT -p tcp -m state --state ESTABLISHED -j ACCEPT\n");
	fprintf (script, "$IPT  -A INPUT -p udp -s 0/0 -d $NET --dport 1023:65535 -j ACCEPT\n\n");

	if (gnome_config_get_bool ("/firestarter/Druid/outfilter=TRUE")) {
		fprintf (script, "#Open ports for outbound established connections\n");
		fprintf (script, "$IPT -A OUTPUT -p tcp -s $NET -d 0/0 --dport 1023:65535 -j ACCEPT\n");
		fprintf (script, "$IPT -A OUTPUT -p udp -s $NET -d 0/0 --dport 1023:65535 -j ACCEPT\n\n");
	}
	else {
		fprintf (script, "$IPT -A OUTPUT -j ACCEPT\n\n");
	}

	fprintf (script, "#Deny everything not let through earlier\n");
	fprintf (script, "$IPT -A INPUT -j $STOP\n");

	fclose (script);

	g_print (_("Firewall script saved as %s\n"), scriptpath);
}
