/* mainpagequery.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "mainpagequery.h"
#include "sqlquery.h"
#include "sqlqueryedit.h"
#include "form.xpm"
#include "query.xpm"
#include "grid.xpm"
#include "sqlqueryexec.h"

static void main_page_query_class_init (MainPageQueryClass * class);
static void main_page_query_init (MainPageQuery * wid);
static void main_page_query_initialize (MainPageQuery * wid);

typedef struct
{
	gboolean is_query;
	SqlQuery *query;	/* used only if (is_query==TRUE) */
	GtkWidget *edit_dlg;	/* used only if (is_query==TRUE) */
	SqlQueryEnv *env;	/* used only if (is_query==FALSE) */
}
Row_Data;

/*
 * static functions 
 */
static gint press_handler_event (GtkWidget * widget,
				 GdkEventButton * event, MainPageQuery * mpq);
static void selection_made (GtkCTree * ctree,
			    GtkCTreeNode * row,
			    gint column, MainPageQuery * mpq);
static void selection_unmade (GtkCTree * ctree,
			      GtkCTreeNode * row,
			      gint column, MainPageQuery * mpq);
static void drop_query_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void edit_query_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void exec_query_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void create_query_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void copy_query_button_cb (GtkWidget * button, MainPageQuery * mpq);

static void drop_form_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void edit_form_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void exec_form_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void real_exec_form (SqlQueryEnv * qe);
static void create_form_button_cb (GtkWidget * button, MainPageQuery * mpq);
static void copy_form_button_cb (GtkWidget * button, MainPageQuery * mpq);

static void conn_closed_cb (GtkWidget * serveur, MainPageQuery * mpq);
static void ctree_remove_a_query (MainPageQuery * mpq, GtkCTreeNode * tnode);
static void edit_query_query_changed_cb (SqlQuery * q, MainPageQuery * mpq);

/* static variables */
static GdkPixmap *query_icon = NULL;
static GdkBitmap *query_mask = NULL;
static GdkPixmap *form_icon = NULL;
static GdkBitmap *form_mask = NULL;
static GdkPixmap *grid_icon = NULL;
static GdkBitmap *grid_mask = NULL;

guint
main_page_query_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Main_Page_Query",
			sizeof (MainPageQuery),
			sizeof (MainPageQueryClass),
			(GtkClassInitFunc) main_page_query_class_init,
			(GtkObjectInitFunc) main_page_query_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_vbox_get_type (), &f_info);
	}

	return f_type;
}

static void
main_page_query_class_init (MainPageQueryClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;
}

static void
main_page_query_init (MainPageQuery * wid)
{
	GtkWidget *sw, *bb;

	/* setting spaces,... */
	gtk_container_set_border_width (GTK_CONTAINER (wid), GNOME_PAD / 2);

	/* Scrolled Window for CTree */
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start (GTK_BOX (wid), sw, TRUE, TRUE, GNOME_PAD);
	gtk_widget_show (sw);

	/* CTree */
	wid->ctree = gtk_ctree_new (2, 0);
	gtk_clist_set_row_height (GTK_CLIST (wid->ctree), 22);
	gtk_clist_set_column_title (GTK_CLIST (wid->ctree), 0,
				    _("Query/Form"));
	gtk_clist_set_column_title (GTK_CLIST (wid->ctree), 1,
				    _("Description"));
	gtk_clist_set_selection_mode (GTK_CLIST (wid->ctree),
				      GTK_SELECTION_SINGLE);
	gtk_clist_column_titles_show (GTK_CLIST (wid->ctree));
	gtk_clist_column_titles_passive (GTK_CLIST (wid->ctree));
	gtk_clist_set_column_auto_resize (GTK_CLIST (wid->ctree), 0, TRUE);
	gtk_clist_set_column_auto_resize (GTK_CLIST (wid->ctree), 1, TRUE);
	gtk_container_add (GTK_CONTAINER (sw), wid->ctree);
	gtk_widget_show (wid->ctree);
	gtk_signal_connect (GTK_OBJECT (wid->ctree), "tree_select_row",
			    GTK_SIGNAL_FUNC (selection_made), wid);
	gtk_signal_connect (GTK_OBJECT (wid->ctree), "tree_unselect_row",
			    GTK_SIGNAL_FUNC (selection_unmade), wid);
	gtk_signal_connect (GTK_OBJECT (wid->ctree),
			    "button_press_event",
			    GTK_SIGNAL_FUNC (press_handler_event), wid);

	/*
	 * buttons for the queries
	 */

	/* Button Box */
	bb = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_end (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD);
	gtk_widget_show (bb);
	wid->queries_bbox = bb;

	/* New Query Button */
	wid->new_query = gtk_button_new_with_label (_("Create Query"));
	gtk_container_add (GTK_CONTAINER (bb), wid->new_query);
	gtk_widget_show (wid->new_query);
	gtk_signal_connect (GTK_OBJECT (wid->new_query), "clicked",
			    GTK_SIGNAL_FUNC (create_query_button_cb), wid);

	/* Edit Query Button */
	wid->edit_query = gtk_button_new_with_label (_("Properties"));
	gtk_container_add (GTK_CONTAINER (bb), wid->edit_query);
	gtk_widget_show (wid->edit_query);
	gtk_signal_connect (GTK_OBJECT (wid->edit_query), "clicked",
			    GTK_SIGNAL_FUNC (edit_query_button_cb), wid);

	/* View Query Button */
	wid->exec_query = gtk_button_new_with_label (_("Execute"));
	gtk_container_add (GTK_CONTAINER (bb), wid->exec_query);
	gtk_signal_connect (GTK_OBJECT (wid->exec_query), "clicked",
			    GTK_SIGNAL_FUNC (exec_query_button_cb), wid);
	gtk_widget_show (wid->exec_query);

	/* Copy Query Button */
	wid->copy_query = gtk_button_new_with_label (_("Copy"));
	gtk_container_add (GTK_CONTAINER (bb), wid->copy_query);
	gtk_signal_connect (GTK_OBJECT (wid->copy_query), "clicked",
			    GTK_SIGNAL_FUNC (copy_query_button_cb), wid);
	/*gtk_widget_show(wid->copy_query); */

	/* Remove Query Button */
	wid->remove_query = gtk_button_new_with_label (_("Delete"));
	gtk_container_add (GTK_CONTAINER (bb), wid->remove_query);
	gtk_widget_show (wid->remove_query);
	gtk_signal_connect (GTK_OBJECT (wid->remove_query), "clicked",
			    GTK_SIGNAL_FUNC (drop_query_button_cb), wid);

	/*
	 * buttons for the ENV
	 */

	/* Button Box */
	bb = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_end (GTK_BOX (wid), bb, FALSE, TRUE, GNOME_PAD);
	wid->env_bbox = bb;

	/* New Form Button */
	wid->new_form = gtk_button_new_with_label (_("Create Form"));
	gtk_container_add (GTK_CONTAINER (bb), wid->new_form);
	gtk_widget_show (wid->new_form);
	gtk_signal_connect (GTK_OBJECT (wid->new_form), "clicked",
			    GTK_SIGNAL_FUNC (create_form_button_cb), wid);

	/* Edit Form Button */
	wid->edit_form = gtk_button_new_with_label (_("Properties"));
	gtk_container_add (GTK_CONTAINER (bb), wid->edit_form);
	gtk_widget_show (wid->edit_form);
	gtk_signal_connect (GTK_OBJECT (wid->edit_form), "clicked",
			    GTK_SIGNAL_FUNC (edit_form_button_cb), wid);

	/* Exec Form Button */
	wid->exec_form = gtk_button_new_with_label (_("Execute"));
	gtk_container_add (GTK_CONTAINER (bb), wid->exec_form);
	gtk_widget_show (wid->exec_form);
	gtk_signal_connect (GTK_OBJECT (wid->exec_form), "clicked",
			    GTK_SIGNAL_FUNC (exec_form_button_cb), wid);

	/* Copy Form Button */
	wid->copy_form = gtk_button_new_with_label (_("Copy"));
	gtk_container_add (GTK_CONTAINER (bb), wid->copy_form);
	/*gtk_widget_show(wid->copy_form); */

	/* Remove Form Button */
	wid->remove_form = gtk_button_new_with_label (_("Delete"));
	gtk_container_add (GTK_CONTAINER (bb), wid->remove_form);
	gtk_widget_show (wid->remove_form);
	gtk_signal_connect (GTK_OBJECT (wid->remove_form), "clicked",
			    GTK_SIGNAL_FUNC (drop_form_button_cb), wid);


	wid->sel_node = NULL;
	wid->new_dlg = NULL;
}

GtkWidget *
main_page_query_new (gASQL_Main_Config * conf)
{
	GtkObject *obj;
	MainPageQuery *wid;

	obj = gtk_type_new (main_page_query_get_type ());
	wid = MAIN_PAGE_QUERY (obj);
	wid->conf = conf;

	main_page_query_initialize (wid);

	gtk_signal_connect (GTK_OBJECT (conf->srv), "conn_closed",
			    GTK_SIGNAL_FUNC (conn_closed_cb), wid);
	return GTK_WIDGET (obj);
}

static void
main_page_query_initialize (MainPageQuery * mpq)
{

	gtk_widget_set_sensitive (mpq->remove_query, FALSE);
	gtk_widget_set_sensitive (mpq->exec_query, FALSE);
	gtk_widget_set_sensitive (mpq->edit_query, FALSE);
	gtk_widget_set_sensitive (mpq->new_query, FALSE);
	gtk_widget_set_sensitive (mpq->copy_query, FALSE);
	gasql_conf_register_sensitive_on_connect (mpq->conf,
						  GTK_WIDGET (mpq->
							      new_query));

	gtk_widget_set_sensitive (mpq->new_form, FALSE);
	gtk_widget_set_sensitive (mpq->exec_form, FALSE);
	gtk_widget_set_sensitive (mpq->remove_form, FALSE);
	gtk_widget_set_sensitive (mpq->copy_form, FALSE);
}


/*
 * Widget specific static functions 
 */

/* Callback to handle the button presses over the CTree */
static gint
press_handler_event (GtkWidget * widget,
		     GdkEventButton * event, MainPageQuery * mpq)
{
	gint row, col;
	GtkCTreeNode *tnode;
	Row_Data *rd;

	if (GTK_IS_CTREE (widget)) {
		/* setting the right selection */
		gtk_clist_get_selection_info (GTK_CLIST (widget),
					      event->x, event->y, &row, &col);
		tnode = gtk_ctree_node_nth (GTK_CTREE (widget), row);
		if ((mpq->sel_node != tnode) && (event->button != 1))
			gtk_ctree_select (GTK_CTREE (widget), tnode);

		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (widget),
						     tnode);

		/* depending on what the clicks were */
		if (event->type == GDK_2BUTTON_PRESS) {
			if (rd && !rd->is_query)
				exec_form_button_cb (NULL, mpq);
			return TRUE;
		}

		if (event->button == 3) {
			GtkWidget *menu, *wid;

			if (rd && rd->is_query) {
				menu = gtk_menu_new ();
				wid = gtk_menu_item_new_with_label (_
								    ("Properties"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (edit_query_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				wid = gtk_menu_item_new_with_label (_
								    ("Execute"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (exec_query_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				wid = gtk_menu_item_new_with_label (_
								    ("Copy"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (copy_query_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				wid = gtk_menu_item_new_with_label (_
								    ("Delete"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (drop_query_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
						NULL, NULL, event->button,
						event->time);
				/* Tell calling code that we have handled this event */
				return TRUE;
			}
			else {
				menu = gtk_menu_new ();
				wid = gtk_menu_item_new_with_label (_
								    ("Properties"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (edit_form_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				wid = gtk_menu_item_new_with_label (_
								    ("Execute"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (exec_form_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				wid = gtk_menu_item_new_with_label (_
								    ("Copy"));
				gtk_signal_connect (GTK_OBJECT (wid),
						    "activate",
						    GTK_SIGNAL_FUNC
						    (copy_form_button_cb),
						    mpq);
				gtk_menu_append (GTK_MENU (menu), wid);
				gtk_widget_show (wid);

				if (rd->env->q->envs->data != rd->env) {	/* (default cannot be removed) */
					wid = gtk_menu_item_new_with_label (_
									    ("Delete"));
					gtk_signal_connect (GTK_OBJECT (wid),
							    "activate",
							    GTK_SIGNAL_FUNC
							    (drop_form_button_cb),
							    mpq);
					gtk_menu_append (GTK_MENU (menu),
							 wid);
					gtk_widget_show (wid);
				}

				gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
						NULL, NULL, event->button,
						event->time);
				/* Tell calling code that we have handled this event */
				return TRUE;
			}
		}
	}

	return FALSE;
}


static void
selection_made (GtkCTree * ctree,
		GtkCTreeNode * row, gint column, MainPageQuery * mpq)
{
	Row_Data *rd;

	mpq->sel_node = row;
	rd = (Row_Data *) gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						       row);
	if (rd->is_query) {
		gtk_widget_show (mpq->queries_bbox);
		gtk_widget_hide (mpq->env_bbox);
		gtk_widget_set_sensitive (mpq->remove_query, TRUE);
		gtk_widget_set_sensitive (mpq->edit_query, TRUE);
		gtk_widget_set_sensitive (mpq->exec_query, TRUE);
		gtk_widget_set_sensitive (mpq->copy_query, TRUE);
	}
	else {
		gtk_widget_hide (mpq->queries_bbox);
		gtk_widget_show (mpq->env_bbox);
		gtk_widget_set_sensitive (mpq->new_form, TRUE);
		gtk_widget_set_sensitive (mpq->exec_form, TRUE);
		if (rd->env->q->envs->data == rd->env)	/* => default cannot be removed */
			gtk_widget_set_sensitive (mpq->remove_form, FALSE);
		else
			gtk_widget_set_sensitive (mpq->remove_form, TRUE);
		gtk_widget_set_sensitive (mpq->copy_form, TRUE);
		gtk_widget_set_sensitive (mpq->edit_form, TRUE);
	}
}

static void
selection_unmade (GtkCTree * ctree,
		  GtkCTreeNode * row, gint column, MainPageQuery * mpq)
{
	if (mpq->sel_node == row) {
		mpq->sel_node = NULL;
		gtk_widget_set_sensitive (mpq->remove_query, FALSE);
		gtk_widget_set_sensitive (mpq->edit_query, FALSE);
		gtk_widget_set_sensitive (mpq->exec_query, FALSE);
		gtk_widget_set_sensitive (mpq->copy_query, FALSE);

		gtk_widget_set_sensitive (mpq->new_form, FALSE);
		gtk_widget_set_sensitive (mpq->exec_form, FALSE);
		gtk_widget_set_sensitive (mpq->remove_form, FALSE);
		gtk_widget_set_sensitive (mpq->copy_form, FALSE);
		gtk_widget_set_sensitive (mpq->edit_form, FALSE);
	}
}

static void drop_query_question_cb (gint reply, MainPageQuery * mpq);
static void
drop_query_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	GtkWidget *dlg;
	gchar *que;
	Row_Data *rd;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);

		que = g_strdup_printf (_("Remove the query %s?"),
				       rd->query->name);
		dlg = gnome_app_question_modal (GNOME_APP (mpq->conf->app),
						que,
						(GnomeReplyCallback)
						drop_query_question_cb, mpq);
		g_free (que);
		gtk_widget_show (dlg);
	}
}

static void
drop_query_question_cb (gint reply, MainPageQuery * mpq)
{
	if (reply == GNOME_YES)
		ctree_remove_a_query (mpq, mpq->sel_node);
}

static void query_edit_dlg_destroy_cb (GtkObject * obj, Row_Data * rd);
static void
edit_query_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;
	GtkWidget *dlg;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		if (!rd->edit_dlg) {
			/* when the DLG is closed, query_edit_dlg_destroy_cb is called */
			dlg = sql_query_edit_new_dlg (rd->query);
			gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
					    GTK_SIGNAL_FUNC
					    (query_edit_dlg_destroy_cb), rd);
			rd->edit_dlg = dlg;
			gtk_widget_show (dlg);
		}
		else
			gdk_window_raise (rd->edit_dlg->window);
	}
}

static void
query_edit_dlg_destroy_cb (GtkObject * obj, Row_Data * rd)
{
	rd->edit_dlg = NULL;
}


static void
exec_query_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;
	gchar *name;
	GtkWidget *dlg;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		if (rd->is_query) {	/* should not be otherwise! */
			SqlQuery *q = rd->query;
			if (!q->envs) {	/* program error, should not happen */
				gnome_app_message (GNOME_APP (q->conf->app),
						   _
						   ("A program error has occured:\n"
						    "This query has no QueryEnv, please make "
						    "a bug report."));
				return;
			}

			/* NOTE: the default form is the first one (linked to the delete action) */
			real_exec_form ((SqlQueryEnv *) (q->envs->data));
		}
	}
}

static void create_dialog_entry_changed_cb (GtkWidget * wid, gpointer data);
static void create_dialog_clicked_cb (GnomeDialog * dialog,
				      gint button_number,
				      gASQL_Main_Config * conf);
static void
create_query_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	GtkWidget *wid, *dlg;

	if (!mpq->new_dlg) {
		dlg = gnome_dialog_new (_("Creation of a new query"),
					GNOME_STOCK_BUTTON_OK,
					GNOME_STOCK_BUTTON_CANCEL, NULL);

		wid = gtk_label_new (_("Query Name:"));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
				    wid, TRUE, TRUE, GNOME_PAD);
		gtk_widget_show (wid);

		wid = gtk_entry_new ();
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox),
				    wid, TRUE, TRUE, GNOME_PAD);
		gtk_widget_show (wid);
		gtk_object_set_user_data (GTK_OBJECT (dlg), wid);
		gtk_object_set_data (GTK_OBJECT (dlg), "mpq", mpq);
		gtk_signal_connect (GTK_OBJECT (wid), "changed",
				    GTK_SIGNAL_FUNC
				    (create_dialog_entry_changed_cb), dlg);

		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC
				    (create_dialog_clicked_cb), mpq->conf);

		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, FALSE);
		mpq->new_dlg = dlg;
		gtk_widget_show (dlg);
	}
	else
		gdk_window_raise (mpq->new_dlg->window);
}


static gboolean name_already_exists (MainPageQuery * mpq,
				     SqlQuery * q, gchar * name);
static void
create_dialog_clicked_cb (GnomeDialog * dialog, gint button_number,
			  gASQL_Main_Config * conf)
{
	GtkWidget *entry;
	gchar *str, *query;
	SqlQuery *q;
	MainPageQuery *mpq;
	gboolean leave = TRUE;

	mpq = gtk_object_get_data (GTK_OBJECT (dialog), "mpq");
	entry = GTK_WIDGET (gtk_object_get_user_data (GTK_OBJECT (dialog)));

	switch (button_number) {
	case 0:		/* OK button */
		/* is the name OK (not already used) ? */
		if (name_already_exists
		    (mpq, NULL, gtk_entry_get_text (GTK_ENTRY (entry)))) {
			leave = FALSE;
			query = g_strdup_printf (_
						 ("A query named \"%s\" already exists,\n"
						  "choose another name"),
						 gtk_entry_get_text (GTK_ENTRY
								     (entry)));
			gnome_app_error (GNOME_APP (mpq->conf->app), query);
			g_free (query);
		}
		else {
			/* new structure */
			str = g_strdup (gtk_entry_get_text
					(GTK_ENTRY (entry)));
			q = SQL_QUERY (sql_query_new (str, conf));	/* automatically add query to the
									   conf->queries list */
			main_page_query_add_query (mpq, q);
			g_free (str);
			break;
		}
	case 1:		/* Cancel Button */
		break;
	default:
	};

	if (leave) {
		gnome_dialog_close (dialog);
		mpq->new_dlg = NULL;
	}
}

static gboolean
name_already_exists (MainPageQuery * mpq, SqlQuery * q, gchar * name)
{
	gboolean result = FALSE;
	gchar *str;
	Row_Data *rd;
	SqlQuery *qu;
	GtkCTreeNode *tnode;
	GList *list;

	list = GTK_CLIST (mpq->ctree)->row_list;

	while (list && !result) {
		tnode = GTK_CTREE_NODE (list);
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     tnode);
		if (rd && rd->is_query) {
			qu = rd->query;	/* qu can be NULL! */
			if (qu)
				str = qu->name;
			else
				str = NULL;
			if (str && !strcmp (str, name))
				if ((q && (qu != q)) || !q)
					result = TRUE;
		}
		list = g_list_next (list);
	}

	return result;
}

static void
copy_query_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;
	SqlQuery *q;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		if (rd->query) {
			q = SQL_QUERY (sql_query_copy (rd->query));
			main_page_query_add_query (mpq, q);
		}
	}
}


static void
create_dialog_entry_changed_cb (GtkWidget * wid, gpointer data)
{
	GnomeDialog *dlg;
	gchar *str;
	GtkWidget *entry;

	dlg = GNOME_DIALOG (data);
	entry = GTK_WIDGET (gtk_object_get_user_data (GTK_OBJECT (data)));
	str = gtk_entry_get_text (GTK_ENTRY (entry));
	if (str && (*str != '\0'))
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, TRUE);
	else
		gnome_dialog_set_sensitive (GNOME_DIALOG (dlg), 0, FALSE);
}


static void
conn_closed_cb (GtkWidget * serveur, MainPageQuery * mpq)
{
	GtkCTreeNode *tnode;
	GList *list;
	Row_Data *rd;

	list = GTK_CLIST (mpq->ctree)->row_list;

	while (list) {
		tnode = GTK_CTREE_NODE (list);
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     tnode);
		if (rd && rd->is_query) {
			ctree_remove_a_query (mpq, tnode);
			list = GTK_CLIST (mpq->ctree)->row_list;
		}
		else
			list = g_list_next (list);
	}
	/* close any add dialog */
	if (mpq->new_dlg) {
		gnome_dialog_close (GNOME_DIALOG (mpq->new_dlg));
		mpq->new_dlg = NULL;
	}
}

static void ctree_remove_a_env (MainPageQuery * mpq, GtkCTreeNode * tnode);	/*impl later */
static void
ctree_remove_a_query (MainPageQuery * mpq, GtkCTreeNode * tnode)
{
	gchar *str;
	Row_Data *rd;
	GSList *list;
	WidRelData *reldata;

	g_return_if_fail (mpq != NULL);
	g_return_if_fail (tnode != NULL);

	rd = (Row_Data *) gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						       tnode);

	/* sensitiveness of the buttons */
	if (mpq->sel_node && (mpq->sel_node == tnode)) {
		gtk_widget_set_sensitive (mpq->remove_query, FALSE);
		gtk_widget_set_sensitive (mpq->edit_query, FALSE);
		gtk_widget_set_sensitive (mpq->exec_query, FALSE);
		gtk_widget_set_sensitive (mpq->copy_query, FALSE);
	}

	/* the work needs to be saved */
	mpq->conf->save_up_to_date = FALSE;

	if (rd) {
		/* if there is a SqlQueryEdit, it destroys itself before */
		if (rd->edit_dlg) {
			gnome_dialog_close (GNOME_DIALOG (rd->edit_dlg));
			rd->edit_dlg = NULL;
		}
		if (rd->query) {
			mpq->conf->queries =
				g_slist_remove (mpq->conf->queries,
						rd->query);

			/* if any SqlWidDbRel was used for that query, it is removed as well */
			list = mpq->conf->widdbrel_list;
			str = g_strdup_printf ("Q%d", rd->query->id);
			while (list) {
				if (!strcmp (((WidRelData *) (list->data))->name, str)) {
					reldata = (WidRelData *) (list->data);
					mpq->conf->widdbrel_list =
						g_slist_remove_link (mpq->conf->widdbrel_list,
								     list);
					g_slist_free_1 (list);
					g_free (reldata->name);
					list = reldata->nodes;
					while (list) {
						GSList *hold = list;
						g_free (list->data);
						list = g_slist_remove_link (list, list);
						g_slist_free_1 (hold);
					}
					g_free (reldata);
				}
				else
					list = g_slist_next (list);
			}
			g_free (str);

			/* free the SqlQueryEnv objects associated to this query */
			list = rd->query->envs;
			if (!GTK_CTREE_ROW (tnode)->expanded)
				while (list) {
					/*sql_query_env_free(SQL_QUERY_ENV(list->data)); */
					gtk_object_destroy (GTK_OBJECT
							    (list->data));
					list = rd->query->envs;
				}
			else
				while (list) {
					GtkCTreeNode *tnode2 = NULL, *iter;
					Row_Data *rd2;

					iter = GTK_CTREE_NODE (GTK_CLIST
							       (mpq->ctree)->
							       row_list);
					while (iter && !tnode2) {
						rd2 = (Row_Data *)
							gtk_ctree_node_get_row_data
							(GTK_CTREE
							 (mpq->ctree), iter);
						if ((!rd2->is_query)
						    && (rd2->env ==
							SQL_QUERY_ENV (list->
								       data)))
							tnode2 = iter;
						iter = GTK_CTREE_NODE_NEXT
							(iter);
					}

					if (tnode2)
						ctree_remove_a_env (mpq,
								    tnode2);
					else
						g_error ("tnode2 == NULL");

					list = rd->query->envs;
				}

			/* the end of the query ... */
			gtk_object_destroy (GTK_OBJECT (rd->query));
			rd->query = NULL;
		}
		g_free (rd);
	}

	/* in the end: remove entry from CTree (will also remove the sub nodes
	   for the SqlQueryEnv */

	gtk_ctree_remove_node (GTK_CTREE (mpq->ctree), tnode);
}

static void
edit_query_query_changed_cb (SqlQuery * q, MainPageQuery * mpq)
{
	gboolean found;
	Row_Data *rd;
	GtkCTreeNode *tnode;
	GList *list;

	found = FALSE;
	list = GTK_CLIST (mpq->ctree)->row_list;

	while (list && !found) {
		tnode = GTK_CTREE_NODE (list);
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     tnode);
		if (rd && (rd->is_query) && (rd->query == q)) {
			found = TRUE;
			/* updating names and description */
			if (name_already_exists (mpq, q, q->name)) {	/* there is a duplicate */
				gchar *str = NULL;
				gint i = 0;

				do {
					if (str)
						g_free (str);
					str = g_strdup_printf ("%s_%d",
							       q->name, i);
					i++;
				} while (name_already_exists
					 (mpq, NULL, str));
				g_free (q->name);
				q->name = str;
			}
			if (*(q->name) == '\0') {	/* name = '\0' */
				gchar *str = NULL;
				gint i = 0;

				do {
					if (str)
						g_free (str);
					str = g_strdup_printf (_("Query_%d"),
							       i);
					i++;
				} while (name_already_exists
					 (mpq, NULL, str));
				g_free (q->name);
				q->name = str;
			}
			gtk_ctree_node_set_pixtext (GTK_CTREE (mpq->ctree),
						    tnode, 0, q->name,
						    GNOME_PAD / 2.,
						    query_icon, query_mask);
			if (q->descr)
				gtk_ctree_node_set_text (GTK_CTREE
							 (mpq->ctree), tnode,
							 1, q->descr);
			else
				gtk_ctree_node_set_text (GTK_CTREE
							 (mpq->ctree), tnode,
							 1, "");
		}
		list = g_list_next (list);
	}
}

void
main_page_query_add_query (MainPageQuery * mpq, SqlQuery * q)
{
	Row_Data *rd;
	GtkCTreeNode *tnode;
	SqlQueryEnv *env;
	GSList *envs;
	gchar *row[2];

	g_return_if_fail (mpq != NULL);
	g_return_if_fail (q != NULL);

	if (!query_icon)
		query_icon =
			gdk_pixmap_create_from_xpm_d (GTK_WIDGET
						      (mpq->conf->app)->
						      window, &query_mask,
						      NULL, query_xpm);

	rd = g_new0 (Row_Data, 1);
	rd->is_query = TRUE;
	rd->query = q;
	rd->edit_dlg = NULL;
	row[0] = g_strdup (q->name);
	if (q->descr)
		row[1] = g_strdup (q->descr);
	else
		row[1] = g_strdup ("");
	tnode = gtk_ctree_insert_node (GTK_CTREE (mpq->ctree), NULL, NULL,
				       row, GNOME_PAD / 2., query_icon,
				       query_mask, query_icon, query_mask,
				       FALSE, FALSE);
	g_free (row[0]);
	g_free (row[1]);
	gtk_ctree_node_set_row_data (GTK_CTREE (mpq->ctree), tnode, rd);
	gtk_signal_connect (GTK_OBJECT (rd->query), "name_changed",
			    GTK_SIGNAL_FUNC (edit_query_query_changed_cb),
			    mpq);

	/* if the query does not have any SqlQueryEnv, we create a default one */
	if (!q->envs) {
		env = SQL_QUERY_ENV (sql_query_env_new (q));
		sql_query_env_set_descr (env, _("Default"));
	}

	envs = q->envs;
	while (envs) {
		env = SQL_QUERY_ENV (envs->data);
		main_page_query_add_env (mpq, env);
		envs = g_slist_next (envs);
	}
}

/*
 * Callback functions for the forms/grid buttons 
 */
static Row_Data *get_query_row_data (MainPageQuery * mpq, SqlQuery * q);
static Row_Data *get_env_row_data (MainPageQuery * mpq, SqlQueryEnv * env);
static void env_descr_changed_cb (SqlQueryEnv * env, MainPageQuery * mpq);
void
main_page_query_add_env (MainPageQuery * mpq, SqlQueryEnv * qe)
{
	GtkCTreeNode *tnode;
	Row_Data *qrd, *rd;
	gchar *row[2];

	g_return_if_fail (mpq != NULL);
	g_return_if_fail (qe != NULL);

	if (!form_icon)
		form_icon =
			gdk_pixmap_create_from_xpm_d (GTK_WIDGET
						      (mpq->conf->app)->
						      window, &form_mask,
						      NULL, form_xpm);
	if (!grid_icon)
		grid_icon =
			gdk_pixmap_create_from_xpm_d (GTK_WIDGET
						      (mpq->conf->app)->
						      window, &grid_mask,
						      NULL, grid_xpm);


	qrd = get_query_row_data (mpq, qe->q);
	tnode = gtk_ctree_find_by_row_data (GTK_CTREE (mpq->ctree), NULL,
					    qrd);
	if (qe->name)
		row[0] = qe->name;
	else
		row[0] = _("Unnamed");
	if (qe->descr)
		row[1] = qe->descr;
	else
		row[1] = "";

	rd = g_new0 (Row_Data, 1);
	rd->is_query = FALSE;
	rd->env = qe;
	if (qe->form_is_default)
		tnode = gtk_ctree_insert_node (GTK_CTREE (mpq->ctree), tnode,
					       NULL, row, GNOME_PAD / 2.,
					       form_icon, form_mask,
					       form_icon, form_mask, TRUE,
					       FALSE);
	else
		tnode = gtk_ctree_insert_node (GTK_CTREE (mpq->ctree), tnode,
					       NULL, row, GNOME_PAD / 2.,
					       grid_icon, grid_mask,
					       grid_icon, grid_mask, TRUE,
					       FALSE);
	gtk_ctree_node_set_row_data (GTK_CTREE (mpq->ctree), tnode, rd);
	gtk_signal_connect (GTK_OBJECT (qe), "descr_changed",
			    GTK_SIGNAL_FUNC (env_descr_changed_cb), mpq);
	gtk_signal_connect (GTK_OBJECT (qe), "type_changed",
			    GTK_SIGNAL_FUNC (env_descr_changed_cb), mpq);
}

static Row_Data *
get_query_row_data (MainPageQuery * mpq, SqlQuery * q)
{
	GtkCTreeNode *tnode;
	Row_Data *retval = NULL;

	tnode = GTK_CTREE_NODE (GTK_CLIST (mpq->ctree)->row_list);
	while (tnode && !retval) {
		if (((Row_Data *) gtk_ctree_node_get_row_data
		     (GTK_CTREE (mpq->ctree), tnode))->query == q)
			retval = (Row_Data *) gtk_ctree_node_get_row_data
				(GTK_CTREE (mpq->ctree), tnode);

		tnode = GTK_CTREE_ROW (tnode)->sibling;
	}

	return retval;
}

static Row_Data *
get_env_row_data (MainPageQuery * mpq, SqlQueryEnv * env)
{
	GtkCTreeNode *tnode;
	Row_Data *retval = NULL, *rd;

	tnode = GTK_CTREE_NODE (GTK_CLIST (mpq->ctree)->row_list);
	while (tnode && !retval) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     tnode);
		if ((!rd->is_query) && (rd->env == env))
			retval = rd;

		tnode = GTK_CTREE_NODE_NEXT (tnode);
	}

	return retval;
}

static void
env_descr_changed_cb (SqlQueryEnv * env, MainPageQuery * mpq)
{
	Row_Data *rd;
	GtkCTreeNode *tnode;

	static GdkPixmap *icon = NULL;
	static GdkBitmap *mask = NULL;

	if (env->form_is_default) {
		icon = form_icon;
		mask = form_mask;
	}
	else {
		icon = grid_icon;
		mask = grid_mask;
	}

	rd = get_env_row_data (mpq, env);

	tnode = gtk_ctree_find_by_row_data (GTK_CTREE (mpq->ctree), NULL, rd);
	if (tnode) {
		if (env->name)
			gtk_ctree_node_set_pixtext (GTK_CTREE (mpq->ctree),
						    tnode, 0, env->name,
						    GNOME_PAD / 2., icon,
						    mask);
		else
			gtk_ctree_node_set_pixtext (GTK_CTREE (mpq->ctree),
						    tnode, 0, _("Unnamed"),
						    GNOME_PAD / 2., icon,
						    mask);
		if (env->descr)
			gtk_ctree_node_set_text (GTK_CTREE (mpq->ctree),
						 tnode, 1, env->descr);
		else
			gtk_ctree_node_set_text (GTK_CTREE (mpq->ctree),
						 tnode, 1, "");
	}
}

static void
drop_form_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		if (!rd->env->q->envs)
			g_error ("mainpagequery->drop_form_button_cb: rd->env->q->envs is NULL!");
		if (rd->env != rd->env->q->envs->data) {
			GtkCTreeNode *tnode;
			tnode = gtk_ctree_find_by_row_data (GTK_CTREE
							    (mpq->ctree),
							    NULL, rd);
			ctree_remove_a_env (mpq, tnode);
		}
	}
}

static void
ctree_remove_a_env (MainPageQuery * mpq, GtkCTreeNode * tnode)
{
	gchar *str;
	Row_Data *rd;
	GSList *list;
	WidRelData *reldata;

	g_return_if_fail (mpq != NULL);
	g_return_if_fail (tnode != NULL);

	rd = (Row_Data *) gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						       tnode);

	/* sensitiveness of the buttons */
	if (mpq->sel_node && (mpq->sel_node == tnode)) {
		gtk_widget_set_sensitive (mpq->remove_form, FALSE);
		gtk_widget_set_sensitive (mpq->edit_form, FALSE);
		gtk_widget_set_sensitive (mpq->exec_form, FALSE);
		gtk_widget_set_sensitive (mpq->copy_form, FALSE);
	}

	/* the work needs to be saved */
	mpq->conf->save_up_to_date = FALSE;

	gtk_ctree_remove_node (GTK_CTREE (mpq->ctree), tnode);
	/*sql_query_env_free(rd->env); */
	gtk_object_destroy (GTK_OBJECT (rd->env));
	g_free (rd);
}


static void
edit_form_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		sql_query_env_show_props_dialog (rd->env);
	}
}

static void
exec_form_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	Row_Data *rd;

	if (mpq->sel_node) {
		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		if (!rd->is_query)	/* should not be otherwise */
			real_exec_form (rd->env);
	}
}

static void
real_exec_form (SqlQueryEnv * qe)
{
	SqlQueryExec *qx;
	qx = SQL_QUERY_EXEC (sql_query_exec_new (qe));
	sql_query_env_register_exec_obj (qe, GTK_OBJECT (qx));
	sql_query_exec_run (qx);
}

static void
create_form_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	if (mpq->sel_node) {
		SqlQueryEnv *env;
		SqlQuery *q;
		GtkCTreeNode *tnode;
		Row_Data *rd;

		rd = (Row_Data *)
			gtk_ctree_node_get_row_data (GTK_CTREE (mpq->ctree),
						     mpq->sel_node);
		tnode = gtk_ctree_find_by_row_data (GTK_CTREE (mpq->ctree),
						    NULL, rd);
		while (tnode && (GTK_CTREE_ROW (tnode)->level != 1))
			tnode = GTK_CTREE_ROW (tnode)->parent;
		if (tnode) {
			rd = gtk_ctree_node_get_row_data (GTK_CTREE
							  (mpq->ctree),
							  tnode);
			env = SQL_QUERY_ENV (sql_query_env_new (rd->query));
			main_page_query_add_env (mpq, env);
		}
	}
}

static void
copy_form_button_cb (GtkWidget * button, MainPageQuery * mpq)
{
	/* FIXME: TODO */
	gnome_app_message (GNOME_APP (mpq->conf->app),
			   _("Not Yet Implemented"));
}
