/* sqlqueryexec.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlqueryenv.h"
#include "sqlqueryexec.h"
#include "datadisplay_common.h"
#include "sqlquerygrid.h"
#include "sqlqueryform.h"

static void sql_query_exec_class_init (SqlQueryExecClass * class);
static void sql_query_exec_init (SqlQueryExec * qx);
static void sql_query_exec_destroy (GtkObject * object);

static void run_the_query (SqlQueryExec * qx);

typedef struct
{
	gchar *descr;
	SqlDataDisplayFns *fns;	/* to get the SQL value from the widget */
	GtkWidget *widget;
	QueryWhereRightOperandType type;
	gpointer main;
}
MissingDataNode;


/*
 *
 * Main functions for the object
 *
 */

/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

enum
{
	NOT_VALID,
	LAST_SIGNAL
};

static gint sql_query_exec_signals[LAST_SIGNAL] = { 0 };

guint
sql_query_exec_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query_Exec",
			sizeof (SqlQueryExec),
			sizeof (SqlQueryExecClass),
			(GtkClassInitFunc) sql_query_exec_class_init,
			(GtkObjectInitFunc) sql_query_exec_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_exec_class_init (SqlQueryExecClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_query_exec_signals[NOT_VALID] =
		gtk_signal_new ("not_valid",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryExecClass,
						   not_valid),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	gtk_object_class_add_signals (object_class, sql_query_exec_signals,
				      LAST_SIGNAL);
	class->not_valid = NULL;

	object_class->destroy = sql_query_exec_destroy;
	parent_class = gtk_type_class (gtk_object_get_type ());
}

static void
sql_query_exec_init (SqlQueryExec * qx)
{
	qx->q = NULL;
	qx->env = NULL;
	qx->qres = NULL;
	qx->missing_data_dlg = NULL;
	qx->missing_data = NULL;
	qx->query = NULL;
	qx->exec_entities = NULL;
}

static void query_changed (SqlQuery * q, SqlQueryExec * qx);
GtkObject *
sql_query_exec_new (SqlQueryEnv * env)
{
	GtkObject *obj;
	SqlQueryExec *qx;

	g_return_val_if_fail (env != NULL, NULL);
	g_return_val_if_fail (IS_SQL_QUERY_ENV (env), NULL);

	obj = gtk_type_new (sql_query_exec_get_type ());
	qx = SQL_QUERY_EXEC (obj);
	/* working on a copy of the query, so the user can work on the query at the same time as
	   the query is executed */
	qx->q = SQL_QUERY (sql_query_copy_noref (env->q));
	qx->env = env;

	/* connection to signals from SqlDb */
	gtk_signal_connect_while_alive (GTK_OBJECT (qx->q), "changed",
					GTK_SIGNAL_FUNC (query_changed),
					qx, obj);
	return obj;
}

static void
query_changed (SqlQuery * q, SqlQueryExec * qx)
{
	/* FIXME: emit the "not_valid" signal and freeze the dialogs related to this Exec */
}

static void
sql_query_exec_destroy (GtkObject * object)
{
	SqlQueryExec *qx;
	GSList *hold;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY_EXEC (object));

	qx = SQL_QUERY_EXEC (object);

	/* destroy all the dialogs for this SqlQueryExec */
	if (qx->missing_data_dlg) {
		gtk_object_destroy (GTK_OBJECT (qx->missing_data_dlg));
		qx->missing_data_dlg = NULL;
	}

	while (qx->missing_data) {
		MissingDataNode *mdn;

		mdn = (MissingDataNode *) (qx->missing_data->data);
		if (mdn->descr)
			g_free (mdn->descr);
		if (mdn->widget)
			gtk_widget_destroy (mdn->widget);
		g_free (mdn);
		hold = qx->missing_data;
		qx->missing_data = g_slist_remove_link (qx->missing_data, qx->missing_data);
		g_slist_free_1 (hold);
	}

	while (qx->exec_entities) {
		ExecEntity *ee;

		ee = (ExecEntity *) (qx->exec_entities->data);
		gtk_object_destroy (GTK_OBJECT (ee->dlg));	/* => destroys the form or grid as well */
		hold = qx->exec_entities;
		qx->exec_entities = g_slist_remove_link (qx->exec_entities, qx->exec_entities);
		g_slist_free_1 (hold);
	}

	if (qx->qres) {
		sql_query_res_free (SQL_QUERY_RES (qx->qres));
		qx->qres = NULL;
	}

	if (qx->q) {
		gtk_object_destroy (GTK_OBJECT (qx->q));
		qx->q = NULL;
	}

	if (qx->query)
		g_free (qx->query);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

typedef struct
{
	GSList *missing_list;
	gASQL_Main_Config *conf;
}
ForeachStruct;

/* returns TRUE if an element has a role to play in the execution of the query */
static gboolean sql_query_is_elt_present (SqlQuery * q, QueryElement * qe);
static gboolean sql_query_is_elt_in_where (SqlQuery * q, QueryElement * qe);
static gboolean missing_foreach_func (GNode * node, ForeachStruct * fes);
static void missing_data_dlg_closed_cb (GtkObject * dlg, SqlQueryExec * qx);
static void missing_data_dlg_clicked_cb (GtkObject * dlg, gint button_number,
					 SqlQueryExec * qx);

/* The object runs the query (and asks for values before if applicable) and displays
   what is supposed to be displayed as a default from the qx->env */
void
sql_query_exec_run (SqlQueryExec * qx)
{
	GSList *list, *missing_list = NULL;
	MissingDataNode *mdn;
	QueryElement *qe;
	QueryElementValue *qev;
	GdaField *gfield;
	ForeachStruct fes;

	/* if we have a SQL text only query, then run it, we don't need any values */
	if (qx->q->text_only) {
		run_the_query (qx);
		return;
	}

	/* values from QueryElementValue */
	list = qx->q->objects;
	while (list) {
		qe = (QueryElement *) (list->data);
		if (qe->type == QUERY_ELT_VALUE) {
			qev = (QueryElementValue *) (qe->main);
			if (qev->ask_at_exec) {
				/* does this value have an effect on the query at all, and the the user needs to be
				   asked, or not, and then don't ask for anything */
				gboolean needed = FALSE;

				/* is this value printed ? */
				if (!needed && qe->print_name)
					needed = TRUE;

				/* is it in an argument of an element which has a printed value printed? */
				if (!needed)
					needed = sql_query_is_elt_present
						(qx->q, qe);

				/* is it somewhere in the WHERE clauses? */
				if (!needed)
					needed = sql_query_is_elt_in_where
						(qx->q, qe);

				if (needed) {
					mdn = g_new0 (MissingDataNode, 1);
					mdn->main = qe;
					mdn->type = QUERY_WHERE_ELT;
					mdn->descr =
						g_strdup_printf ("%s:",
								 qe->name);
					mdn->fns =
						sql_access_get_object_display_fns
						(qx->q->conf->srv,
						 GTK_OBJECT (qev->data_type));
					mdn->widget =
						GTK_WIDGET ((mdn->fns->
							     sql_to_widget)
							    (qev->
							     default_val));
					missing_list =
						g_slist_append (missing_list,
								mdn);
				}
			}
		}
		list = g_slist_next (list);
	}

	/* values from QueryWhereNode values to be asked at query time */
	fes.missing_list = missing_list;
	fes.conf = qx->q->conf;
	g_node_traverse (qx->q->where_tree, G_IN_ORDER, G_TRAVERSE_LEAFS, -1,
			 (GNodeTraverseFunc) missing_foreach_func, &fes);
	missing_list = fes.missing_list;

	/* building the dialog to fill in values */
	if (missing_list == NULL)
		run_the_query (qx);
	else {
		GtkWidget *dlg, *table, *label;
		guint i = 0;
		gchar *str;

		dlg = gnome_dialog_new (_("Enter necessary values"),
					GNOME_STOCK_BUTTON_OK,
					GNOME_STOCK_BUTTON_CANCEL, NULL);
		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC
				    (missing_data_dlg_clicked_cb), qx);
		gtk_signal_connect (GTK_OBJECT (dlg), "close",
				    GTK_SIGNAL_FUNC
				    (missing_data_dlg_closed_cb), qx);

		str = g_strdup_printf (_
				       ("Fill the following parameters to execute the\n"
					"'%s' query"), qx->q->name);
		label = gtk_label_new (str);
		g_free (str);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), label,
				    FALSE, TRUE, GNOME_PAD / 2.);

		table = gtk_table_new (g_slist_length (missing_list), 2,
				       FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (table),
						GNOME_PAD / 2.);
		gtk_table_set_col_spacings (GTK_TABLE (table),
					    GNOME_PAD / 2.);
		gtk_table_set_row_spacings (GTK_TABLE (table),
					    GNOME_PAD / 2.);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), table,
				    TRUE, TRUE, 0);
		list = missing_list;
		while (list) {
			mdn = (MissingDataNode *) (list->data);
			label = gtk_label_new (mdn->descr);
			gtk_table_attach_defaults (GTK_TABLE (table), label,
						   0, 1, i, i + 1);
			gtk_table_attach_defaults (GTK_TABLE (table),
						   mdn->widget, 1, 2, i,
						   i + 1);
			i++;
			list = g_slist_next (list);
		}
		gtk_widget_show_all (dlg);
		qx->missing_data_dlg = dlg;
		qx->missing_data = missing_list;
	}
}

/* recursive function */
static gboolean
sql_query_is_elt_present (SqlQuery * q, QueryElement * qe)
{
	if (qe->print_name)
		return TRUE;
	else {
		GSList *objects = q->objects;
		QueryElement *elt;
		gboolean present = FALSE;
		while (objects && !present) {
			elt = (QueryElement *) (objects->data);
			if (g_slist_find (elt->args, qe))
				present = sql_query_is_elt_present (q, elt);
			objects = g_slist_next (objects);
		}

		return present;
	}
}

static gboolean elt_is_in_where (SqlQuery * q, QueryElement * qe);
static gboolean
sql_query_is_elt_in_where (SqlQuery * q, QueryElement * qe)
{
	if (elt_is_in_where (q, qe))
		return TRUE;
	else {
		GSList *objects = q->objects;
		QueryElement *elt;
		gboolean present = FALSE;
		while (objects && !present) {
			elt = (QueryElement *) (objects->data);
			if (g_slist_find (elt->args, qe))
				present = sql_query_is_elt_in_where (q, elt);
			objects = g_slist_next (objects);
		}

		return present;
	}
}
struct _InWhereTraverse
{
	QueryElement *qe;
	gboolean found;
};
static gboolean elt_is_in_where_traverse (GNode * node, gpointer data);
static gboolean
elt_is_in_where (SqlQuery * q, QueryElement * qe)
{
	struct _InWhereTraverse trav;
	trav.found = FALSE;
	trav.qe = qe;
	g_node_traverse (q->where_tree, G_LEVEL_ORDER, G_TRAVERSE_LEAFS, -1,
			 elt_is_in_where_traverse, &trav);
	return trav.found;
}
static gboolean
elt_is_in_where_traverse (GNode * node, gpointer data)
{
	struct _InWhereTraverse *trav;
	QueryWhereNode *qwh;

	trav = (struct _InWhereTraverse *) data;
	qwh = (QueryWhereNode *) (node->data);
	if (qwh->is_node)
		return FALSE;
	if ((qwh->val.leaf.l_elt == trav->qe) ||
	    ((qwh->val.leaf.r_type == QUERY_WHERE_ELT)
	     && (qwh->val.leaf.r_object == trav->qe))) {
		trav->found = TRUE;
		return TRUE;
	}
	return FALSE;
}

static gchar *render_operand (gASQL_Main_Config * conf, QueryElement * elt);
static gboolean
missing_foreach_func (GNode * node, ForeachStruct * fes)
{
	QueryWhereNode *qwh;
	GtkObject *obj;

	qwh = (QueryWhereNode *) (node->data);
	if (!(qwh->is_node) && (qwh->val.leaf.r_type == QUERY_WHERE_ASK)) {
		MissingDataNode *mdn;
		mdn = g_new0 (MissingDataNode, 1);
		mdn->main = qwh;
		mdn->type = qwh->val.leaf.r_type;
		mdn->descr = g_strdup_printf (_("%s %s"),
					      /*render_operand(fes->conf, qwh->val.leaf.l_elt), */
					      qwh->val.leaf.l_elt->name,
					      sql_query_render_operator (qwh->
									 val.
									 leaf.
									 op));
		obj = NULL;
		switch (((QueryElement *) (qwh->val.leaf.l_elt))->type) {
		case QUERY_ELT_FIELD:
		case QUERY_ELT_AGGREGATE:
		case QUERY_ELT_FUNCTION:
			obj = ((QueryElement *) (qwh->val.leaf.l_elt))->main;
			break;
		case QUERY_ELT_VALUE:
			obj = GTK_OBJECT (((QueryElementValue
					    *) ((QueryElement *) (qwh->val.
								  leaf.
								  l_elt))->
					   main)->data_type);
		}
		mdn->fns =
			sql_access_get_object_display_fns (fes->conf->srv,
							   obj);
		mdn->widget =
			GTK_WIDGET ((mdn->fns->gdafield_to_widget) (NULL));
		fes->missing_list = g_slist_append (fes->missing_list, mdn);
	}
	return FALSE;
}

static gchar *
render_operand (gASQL_Main_Config * conf, QueryElement * elt)
{
	gchar *str = NULL;

	switch (elt->type) {
	case QUERY_ELT_FIELD:
		str = g_strdup_printf ("%s.%s",
				       sql_db_find_table_from_field (conf->db,
								     SQL_MEM_FIELD
								     (elt->
								      main))->
				       name, SQL_MEM_FIELD (elt->main)->name);
		break;
	case QUERY_ELT_FUNCTION:
		{
			GSList *list;
			GString *gstr;
			gboolean first = TRUE;

			list = elt->args;
			gstr = g_string_new ("");
			g_string_sprintf (gstr, "%s (",
					  SQL_DATA_FUNCTION (elt->main)->
					  sqlname);
			while (list) {
				if (first) {
					g_string_sprintfa (gstr, "%s",
							   ((QueryElement
							     *) (list->
								 data))->
							   name);
					first = FALSE;
				}
				else
					g_string_sprintfa (gstr, ", %s",
							   ((QueryElement
							     *) (list->
								 data))->
							   name);
				list = g_slist_next (list);
			}
			g_string_sprintfa (gstr, ")");
			str = gstr->str;
			g_string_free (gstr, FALSE);
			break;
		}
	case QUERY_ELT_AGGREGATE:
		str = g_strdup_printf ("%s (%s)",
				       SQL_DATA_AGGREGATE (elt->main)->
				       sqlname,
				       ((QueryElement *) (elt->args->data))->
				       name);
		break;
	case QUERY_ELT_VALUE:
		str = g_strdup_printf ("%s",
				       ((QueryElementValue *) (elt->main))->
				       default_val);
		break;
	}

	if (!str)
		str = g_strdup ("???");
	return str;
}

/* callbacks for the missing data dialog */
static void
missing_data_dlg_closed_cb (GtkObject * dlg, SqlQueryExec * qx)
{
	gtk_object_destroy (GTK_OBJECT (dlg));
	qx->missing_data_dlg = NULL;
	gtk_object_destroy (GTK_OBJECT (qx));
}

static void
missing_data_dlg_clicked_cb (GtkObject * dlg, gint button_number,
			     SqlQueryExec * qx)
{
	switch (button_number) {
	case 0:		/* OK */
		gtk_widget_hide (GTK_WIDGET (dlg));
		run_the_query (qx);
		gtk_object_destroy (GTK_OBJECT (dlg));
		qx->missing_data_dlg = NULL;
		break;
	case 1:		/* Cancel */
		gnome_dialog_close (GNOME_DIALOG (dlg));
		break;
	}
}


/* 
 * Now really run the query
 */
static GtkWidget *get_grid_widget (SqlQueryExec * qx, gchar * sql_query);
static GtkWidget *get_form_widget (SqlQueryExec * qx, gchar * sql_query,
				   gulong row);
static void entity_destroyed_cb (GtkObject * obj, SqlQueryExec * qx);
static void exec_dlg_clicked_cb (GnomeDialog * dialog, gint button_number,
				 SqlQueryExec * qx);
static void exec_dlg_close_cb (GnomeDialog * dialog, SqlQueryExec * qx);
static void sub_action_refresh (GtkObject * ent, SqlQueryExec * qx);
static void sub_action_insert (GtkObject * ent, SqlQueryExec * qx);
static void sub_action_edit (GtkObject * ent, gulong row, SqlQueryExec * qx);
static void sub_action_delete (GtkObject * ent, gulong row,
			       SqlQueryExec * qx);
static void sub_action_choose (GtkObject * ent, gulong row,
			       SqlQueryExec * qx);
static void sub_action_viewall (GtkObject * ent, SqlQueryExec * qx);
static void
run_the_query (SqlQueryExec * qx)
{
	GSList *hold, *mlist, *query_missing_values = NULL;
	QueryMissingValue *qmv;
	gchar *query;
	GtkWidget *dlg, *wid;
	gchar *str;
	gboolean error = FALSE;
	ExecEntity *ee;

	mlist = qx->missing_data;
	while (mlist) {
		MissingDataNode *mdn;
		gchar *sql;

		mdn = (MissingDataNode *) (mlist->data);
		sql = (mdn->fns->widget_to_sql) (DATA_DISPLAY (mdn->widget));
		mdn->widget = NULL;	/* FIXME: the widgets are destroyed when the dialog is closed,
					   so we set them to NULL here to avoid trying to destroy them
					   again. */
		/*g_print("Value type:%d and SQL=%s\n", mdn->type, sql); */

		qmv = g_new0 (QueryMissingValue, 1);
		qmv->type = mdn->type;
		qmv->node = mdn->main;
		qmv->sql = sql;
		query_missing_values =
			g_slist_append (query_missing_values, qmv);
		mlist = g_slist_next (mlist);
	}
	query = sql_query_get_select_query (qx->q, query_missing_values);

	/* free the list of missing values */
	while (query_missing_values) {
		qmv = (QueryMissingValue *) (query_missing_values->data);
		g_free (qmv->sql);
		g_free (qmv);
		hold = query_missing_values;
		query_missing_values = g_slist_remove_link (query_missing_values, query_missing_values);
		g_slist_free_1 (hold);
	}

	/* Actual preparation of the Dialog */
	str = g_strdup_printf (_("Execution of query '%s'"), qx->q->name);
	dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
	g_free (str);

	/* entity to hold this grid or form */
	ee = g_new0 (ExecEntity, 1);
	ee->dlg = dlg;
	if (qx->env->form_is_default) {
		if (!qx->q->text_only)
			wid = get_form_widget (qx, query, 1);
		else {
			gnome_app_message (GNOME_APP (qx->q->conf->app),
					   _
					   ("This query is a SQL only query, and it is\n"
					    "impossible for now to have a form for this\n"
					    "kind of queries, so a grid is presented here instead."));
			wid = get_grid_widget (qx, query);
			gtk_widget_set_usize (GTK_WIDGET (wid), 380, 380);
		}
	}
	else
		wid = get_grid_widget (qx, query);
	if (!wid)
		error = TRUE;
	else {
		ee->entity = wid;
		gtk_box_pack_start_defaults (GTK_BOX
					     (GNOME_DIALOG (dlg)->vbox), wid);
		gtk_widget_show (wid);
	}

	if (!error) {
		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC (exec_dlg_clicked_cb),
				    qx);
		gtk_signal_connect (GTK_OBJECT (dlg), "close",
				    GTK_SIGNAL_FUNC (exec_dlg_close_cb), qx);
		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, TRUE);
		gtk_widget_show (dlg);
		qx->exec_entities = g_slist_append (qx->exec_entities, ee);

		/* signals from the form or the grid */
		if (IS_SQL_QUERY_GRID (ee->entity)) {
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_refresh",
					    GTK_SIGNAL_FUNC
					    (sub_action_refresh), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_insert",
					    GTK_SIGNAL_FUNC
					    (sub_action_insert), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_edit",
					    GTK_SIGNAL_FUNC (sub_action_edit),
					    qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_delete",
					    GTK_SIGNAL_FUNC
					    (sub_action_delete), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_choose",
					    GTK_SIGNAL_FUNC
					    (sub_action_choose), qx);
		}
		else {		/* IS_SQL_QUERY_FORM(ee->entity) == TRUE */
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_refresh",
					    GTK_SIGNAL_FUNC
					    (sub_action_refresh), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_insert",
					    GTK_SIGNAL_FUNC
					    (sub_action_insert), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_viewall",
					    GTK_SIGNAL_FUNC
					    (sub_action_viewall), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_choose",
					    GTK_SIGNAL_FUNC
					    (sub_action_choose), qx);
		}
		gtk_signal_connect (GTK_OBJECT (ee->entity), "destroy",
				    GTK_SIGNAL_FUNC (entity_destroyed_cb),
				    qx);
	}
	else {
		g_free (ee);
		gtk_object_destroy (GTK_OBJECT (dlg));
		/* test if we should destroy the SqlQueryExec widget now */
		if (!qx->exec_entities)
			gtk_object_destroy (GTK_OBJECT (qx));
	}
	qx->query = query;
}

static void
exec_dlg_clicked_cb (GnomeDialog * dialog, gint button_number,
		     SqlQueryExec * qx)
{
	/* there is only a "close" button, so no need to test the button_number */
	GSList *list;
	gboolean found = FALSE;
	ExecEntity *ee = NULL;

	list = qx->exec_entities;
	while (list && !found) {
		ee = (ExecEntity *) (list->data);
		if (GNOME_DIALOG (ee->dlg) == dialog)
			found = TRUE;
		else
			list = g_slist_next (list);
	}

	if (found) {
		qx->exec_entities = g_slist_remove_link (qx->exec_entities, list);
		g_slist_free_1 (list);
		gtk_object_destroy (GTK_OBJECT (ee->dlg));	/* entity destroyed at the same time as its dialog */
	}

	/* test if we should destroy the SqlQueryExec widget now */
	if (!qx->exec_entities)
		gtk_object_destroy (GTK_OBJECT (qx));
}

static void
exec_dlg_close_cb (GnomeDialog * dialog, SqlQueryExec * qx)
{
	exec_dlg_clicked_cb (dialog, 0, qx);
}

static GtkWidget *
get_grid_widget (SqlQueryExec * qx, gchar * sql_query)
{
	GtkWidget *grid;
	SqlQueryRes *qres;

	if (!qx->qres) {
		qres = sql_access_do_query (qx->q->conf->srv, sql_query);
		if (!qres) {
			gchar *str;
			str = g_strdup_printf (_("Can't execute query:\n%s"),
					       sql_query);
			gnome_app_error (GNOME_APP (qx->q->conf->app), str);
			g_free (str);
			return NULL;
		}
		qx->qres = qres;
	}

	grid = GTK_WIDGET (sql_query_grid_new (qx));
	gtk_widget_show_all (grid);

	return grid;
}

/* if row == 0, the we want an insert dialog, otherwise row needs be >= 1 */
static GtkWidget *
get_form_widget (SqlQueryExec * qx, gchar * sql_query, gulong row)
{
	GtkWidget *form;
	SqlQueryRes *qres;

	if (!qx->qres && row) {
		qres = sql_access_do_query (qx->q->conf->srv, sql_query);
		if (!qres) {
			gchar *str;
			str = g_strdup_printf (_("Can't execute query:\n%s"),
					       sql_query);
			gnome_app_error (GNOME_APP (qx->q->conf->app), str);
			g_free (str);
			return NULL;
		}
		qx->qres = qres;
	}

	form = sql_query_form_new (qx, row);

	if (form)		/* form can be NULL if no modif_table is selected */
		gtk_widget_show_all (form);

	return form;
}

static void
entity_destroyed_cb (GtkObject * obj, SqlQueryExec * qx)
{
	GSList *list;
	gboolean found = FALSE;
	ExecEntity *ee;

	g_print ("entity_destroyed_cb\n");
	list = qx->exec_entities;
	while (list && !found) {
		ee = (ExecEntity *) (list->data);
		if (ee->entity == GTK_WIDGET (obj))
			found = TRUE;
		else
			list = g_slist_next (list);
	}
	if (found)
		exec_dlg_clicked_cb (GNOME_DIALOG (ee->dlg), 0, qx);
}

static void
sub_action_refresh (GtkObject * ent, SqlQueryExec * qx)
{
	SqlQueryRes *qres;
	GSList *list;
	ExecEntity *ee;

	if (qx->qres) {
		sql_query_res_free (SQL_QUERY_RES (qx->qres));
		qx->qres = NULL;
	}

	/* getting a new SqlQueryRes */
	qres = sql_access_do_query (qx->q->conf->srv, qx->query);
	if (!qres) {
		gchar *str;
		str = g_strdup_printf (_("Can't execute query:\n%s"),
				       qx->query);
		gnome_app_error (GNOME_APP (qx->q->conf->app), str);
		g_free (str);
	}
	qx->qres = qres;

	/* Updating everything */
	list = qx->exec_entities;
	while (list) {
		ee = (ExecEntity *) (list->data);
		if (IS_SQL_QUERY_GRID (ee->entity))
			sql_query_grid_refresh (SQL_QUERY_GRID (ee->entity));
		else if (IS_SQL_QUERY_FORM (ee->entity))
			sql_query_form_refresh (SQL_QUERY_FORM (ee->entity));
		list = g_slist_next (list);
	}
}

static void
sub_action_insert (GtkObject * ent, SqlQueryExec * qx)
{
	GtkWidget *dlg, *wid;
	ExecEntity *ee;
	gchar *str;

	if (!qx->env->modif_table)
		g_error ("env->modif_table = NULL in %s at %d", __FILE__,
			 __LINE__);
	str = g_strdup_printf (_("Data insertion in '%s' (query '%s')"),
			       qx->env->modif_table->name, qx->q->name);
	dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
	g_free (str);

	ee = g_new0 (ExecEntity, 1);
	ee->dlg = dlg;
	wid = get_form_widget (qx, NULL, 0);

	if (!wid) {
		g_free (ee);
		gtk_object_destroy (GTK_OBJECT (dlg));
		/* test if we should destroy the SqlQueryExec widget now */
		if (!qx->exec_entities)
			gtk_object_destroy (GTK_OBJECT (qx));
	}
	else {
		ee->entity = wid;
		gtk_box_pack_start_defaults (GTK_BOX
					     (GNOME_DIALOG (dlg)->vbox), wid);
		gtk_widget_show (wid);

		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC (exec_dlg_clicked_cb),
				    qx);
		gtk_signal_connect (GTK_OBJECT (dlg), "close",
				    GTK_SIGNAL_FUNC (exec_dlg_close_cb), qx);
		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, TRUE);
		gtk_widget_show (dlg);
		qx->exec_entities = g_slist_append (qx->exec_entities, ee);

		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_refresh",
				    GTK_SIGNAL_FUNC (sub_action_refresh), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_insert",
				    GTK_SIGNAL_FUNC (sub_action_insert), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_viewall",
				    GTK_SIGNAL_FUNC (sub_action_viewall), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_choose",
				    GTK_SIGNAL_FUNC (sub_action_choose), qx);
	}
}

static void
sub_action_edit (GtkObject * ent, gulong row, SqlQueryExec * qx)
{
	GtkWidget *dlg, *wid;
	ExecEntity *ee;
	gchar *str;

	str = g_strdup_printf (_("Recordset Edition (query '%s')"),
			       qx->q->name);
	dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
	g_free (str);

	ee = g_new0 (ExecEntity, 1);
	ee->dlg = dlg;
	wid = get_form_widget (qx, NULL, row);

	if (!wid) {
		g_free (ee);
		gtk_object_destroy (GTK_OBJECT (dlg));
		/* test if we should destroy the SqlQueryExec widget now */
		if (!qx->exec_entities)
			gtk_object_destroy (GTK_OBJECT (qx));
	}
	else {
		ee->entity = wid;
		gtk_box_pack_start_defaults (GTK_BOX
					     (GNOME_DIALOG (dlg)->vbox), wid);
		gtk_widget_show (wid);

		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC (exec_dlg_clicked_cb),
				    qx);
		gtk_signal_connect (GTK_OBJECT (dlg), "close",
				    GTK_SIGNAL_FUNC (exec_dlg_close_cb), qx);
		gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE, TRUE);
		gtk_widget_show (dlg);
		qx->exec_entities = g_slist_append (qx->exec_entities, ee);

		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_refresh",
				    GTK_SIGNAL_FUNC (sub_action_refresh), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_insert",
				    GTK_SIGNAL_FUNC (sub_action_insert), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_viewall",
				    GTK_SIGNAL_FUNC (sub_action_viewall), qx);
		gtk_signal_connect (GTK_OBJECT (ee->entity), "action_choose",
				    GTK_SIGNAL_FUNC (sub_action_choose), qx);
	}
}

static void
sub_action_delete (GtkObject * ent, gulong row, SqlQueryExec * qx)
{
	gchar *str;
	str = g_strdup_printf (_
			       ("Removal of row %d:\nto be implemented in %s, line %d"),
			       row, __FILE__, __LINE__);
	gnome_app_message (GNOME_APP (qx->q->conf->app), str);
	g_free (str);
}

static void
sub_action_choose (GtkObject * ent, gulong row, SqlQueryExec * qx)
{
	gchar *str;
	str = g_strdup_printf (_
			       ("Choosing row %d:\nto be implemented in %s, line %d"),
			       row, __FILE__, __LINE__);
	gnome_app_message (GNOME_APP (qx->q->conf->app), str);
	g_free (str);
}

static void
sub_action_viewall (GtkObject * ent, SqlQueryExec * qx)
{
	GtkWidget *dlg, *wid;
	ExecEntity *ee = NULL;
	gchar *str;
	GSList *list;
	gboolean found = FALSE;

	list = qx->exec_entities;
	while (list && !found) {
		ee = (ExecEntity *) (list->data);
		if (IS_SQL_QUERY_GRID (ee->entity))
			found = TRUE;
		else
			list = g_slist_next (list);
	}

	if (found)
		gdk_window_raise (GTK_WIDGET (ee->dlg)->window);
	else {
		str = g_strdup_printf (_("Execution of query '%s'"),
				       qx->q->name);
		dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
		g_free (str);

		ee = g_new0 (ExecEntity, 1);
		ee->dlg = dlg;
		wid = get_grid_widget (qx, NULL);

		if (!wid) {
			g_free (ee);
			gtk_object_destroy (GTK_OBJECT (dlg));
			/* test if we should destroy the SqlQueryExec widget now */
			if (!qx->exec_entities)
				gtk_object_destroy (GTK_OBJECT (qx));
		}
		else {
			ee->entity = wid;
			gtk_box_pack_start_defaults (GTK_BOX
						     (GNOME_DIALOG (dlg)->
						      vbox), wid);
			gtk_widget_show (wid);

			gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
					    GTK_SIGNAL_FUNC
					    (exec_dlg_clicked_cb), qx);
			gtk_signal_connect (GTK_OBJECT (dlg), "close",
					    GTK_SIGNAL_FUNC
					    (exec_dlg_close_cb), qx);
			gtk_window_set_policy (GTK_WINDOW (dlg), TRUE, TRUE,
					       TRUE);
			gtk_widget_show (dlg);
			qx->exec_entities =
				g_slist_append (qx->exec_entities, ee);

			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_refresh",
					    GTK_SIGNAL_FUNC
					    (sub_action_refresh), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_insert",
					    GTK_SIGNAL_FUNC
					    (sub_action_insert), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_edit",
					    GTK_SIGNAL_FUNC (sub_action_edit),
					    qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_delete",
					    GTK_SIGNAL_FUNC
					    (sub_action_delete), qx);
			gtk_signal_connect (GTK_OBJECT (ee->entity),
					    "action_choose",
					    GTK_SIGNAL_FUNC
					    (sub_action_choose), qx);
		}
	}
}






static void add_stock_pixmap (GtkWidget * button_box, gchar * action,
			      gchar * tip, GtkObject * collector,
			      SqlQueryActions action_val);
/* helper function. Could as well be in SqlQueryEnv */
GtkWidget *
sql_query_exec_get_action_buttons (SqlQueryExec * exec)
{
	GtkWidget *vbox, *bbox, *button, *pix;
	gint actions = exec->env->actions;

	vbox = gtk_vbox_new (FALSE, GNOME_PAD / 4.);
	/* first row */
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vbox), bbox, TRUE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox),
				   GTK_BUTTONBOX_SPREAD);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), GNOME_PAD / 4.);

	if (actions & QUERY_ACTION_FIRST)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_FIRST,
				  _("Move to the first record"),
				  GTK_OBJECT (vbox), QUERY_ACTION_FIRST);

	if (actions & QUERY_ACTION_PREV)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_BACK,
				  _("Move to the previous record"),
				  GTK_OBJECT (vbox), QUERY_ACTION_PREV);

	if (actions & QUERY_ACTION_NEXT)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_FORWARD,
				  _("Move to the next record"),
				  GTK_OBJECT (vbox), QUERY_ACTION_NEXT);

	if (actions & QUERY_ACTION_LAST)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_LAST,
				  _("Move to the last record"),
				  GTK_OBJECT (vbox), QUERY_ACTION_LAST);


	/* second row */
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vbox), bbox, TRUE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox),
				   GTK_BUTTONBOX_SPREAD);
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), GNOME_PAD / 4.);

	if (actions & QUERY_ACTION_EDIT)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_INDEX,
				  _("Edit data"), GTK_OBJECT (vbox),
				  QUERY_ACTION_EDIT);

	if (actions & QUERY_ACTION_VIEWALL)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_INDEX,
				  _("View all as grid"), GTK_OBJECT (vbox),
				  QUERY_ACTION_VIEWALL);

	if (actions & QUERY_ACTION_INSERT)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_NEW,
				  _("Insert new data"), GTK_OBJECT (vbox),
				  QUERY_ACTION_INSERT);

	if (actions & QUERY_ACTION_COMMIT)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_SAVE,
				  _("Commit changes"), GTK_OBJECT (vbox),
				  QUERY_ACTION_COMMIT);

	if (actions & QUERY_ACTION_DELETE)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_TRASH,
				  _("Delete selected data"),
				  GTK_OBJECT (vbox), QUERY_ACTION_DELETE);

	if (actions & QUERY_ACTION_REFRESH)
		add_stock_pixmap (bbox, GNOME_STOCK_PIXMAP_REFRESH,
				  _("Refresh display"), GTK_OBJECT (vbox),
				  QUERY_ACTION_REFRESH);

	/* third row Not Yet necessary */
	/*  bbox = gtk_hbutton_box_new(); */
/*   gtk_box_pack_start(GTK_BOX(vbox), bbox, TRUE, FALSE, 0); */
/*   gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_SPREAD); */
/*   gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), GNOME_PAD/4.); */

/*   if (actions & QUERY_ACTION_CHOOSE)  */
/*     add_stock_pixmap(bbox, GNOME_STOCK_PIXMAP_JUMP_TO, _("Validate selection"), */
/* 		     GTK_OBJECT(vbox), QUERY_ACTION_CHOOSE); */

	return vbox;
}

static void
add_stock_pixmap (GtkWidget * button_box, gchar * action,
		  gchar * tip, GtkObject * collector,
		  SqlQueryActions action_val)
{
	GtkWidget *button, *pix;
	static GtkTooltips *tips = NULL;
	gchar *str;

	button = gtk_button_new ();
	pix = gnome_stock_pixmap_widget_new (NULL, action);
	gtk_container_add (GTK_CONTAINER (button), pix);
	gtk_container_add (GTK_CONTAINER (button_box), button);

	if (!tips)
		tips = gtk_tooltips_new ();
	gtk_tooltips_set_tip (GTK_TOOLTIPS (tips), button, tip, NULL);
	gtk_tooltips_enable (GTK_TOOLTIPS (tips));

	str = g_strdup_printf ("%ld", action_val);
	gtk_object_set_data (GTK_OBJECT (collector), str, button);
	g_free (str);
}
