#define _E_FONT_C_

/*
 * e-font
 *
 * Temporary wrappers around GdkFonts to get unicode displaying
 *
 * Author: Lauris Kaplinski <lauris@helixcode.com>
 *
 * Copyright (C) 2000 Helix Code, Inc.
 *
 * TODO: LRU Cache
 *
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <unicode.h>
#include <iconv.h>
#include "gal/util/e-cache.h"
#include "e-font.h"

#define noE_FONT_VERBOSE
#define E_FONT_CACHE_SIZE 32

struct _EFont {
	gint refcount;
	GdkFont *font;
	GdkFont *bold;
	gboolean twobyte;
	gboolean nbsp_zero_width;
	iconv_t to;
	iconv_t from;
};

static gchar * get_font_name (const GdkFont * font);
static void split_name (gchar * c[], gchar * name);
static gboolean find_variants (gchar **namelist, gint length,
			       gchar *base_weight, gchar **light,
			       gchar **bold);
static const gchar *get_locale_charset (void);
static gint         e_font_to_native   (EFont *font, gchar *native, const gchar *utf, gint bytes);
#if 0
static const gchar *locale_charset_to_encoding(void);
#endif
#ifdef E_FONT_VERBOSE
static void e_font_print_gdk_font_name (const GdkFont * font);
#endif

EFont *
e_font_from_gdk_name (const gchar *name)
{
	EFont * font;
	GdkFont *gdkfont;

	gdkfont = gdk_font_load (name);
	if (!gdkfont) gdkfont = gdk_font_load ("fixed");
	g_return_val_if_fail (gdkfont != NULL, NULL);
	font = e_font_from_gdk_font (gdkfont);
	gdk_font_unref (gdkfont);

	return font;
}

static void
set_nbsp_zero_width_flag (EFont *efont)
{
	guchar *nbsp = "\xc2\xa0";
	guchar native_nbsp [8];

	efont->nbsp_zero_width = FALSE;
	efont->nbsp_zero_width = gdk_text_width (efont->font, native_nbsp, e_font_to_native (efont, native_nbsp, nbsp, 2))
		? FALSE : TRUE;
}

EFont *
e_font_from_gdk_font (GdkFont *gdkfont)
{
	static ECache * cache = NULL;
	EFont *font;
	GdkFont *keyfont, *boldfont, *lightfont;
	gchar * name;
	XFontStruct *xfs;

	/* Try cache */
	if (!cache) cache = e_cache_new (NULL, /* Key hash */
					 NULL, /* Key compare */
					 (ECacheDupFunc) gdk_font_ref, /* Key dup func */
					 (ECacheFreeFunc) gdk_font_unref, /* Key free func */
					 (ECacheFreeFunc) e_font_unref, /* Object free func */
					 E_FONT_CACHE_SIZE, /* Soft limit (fake) */
					 E_FONT_CACHE_SIZE); /* Hard limit */

	font = e_cache_lookup (cache, gdkfont);
	if (font) {
		e_font_ref (font);
		return font;
	} else {
		keyfont = gdkfont;
	}

	/* Not in cache */
	lightfont = gdkfont;
	boldfont = NULL;

	gdk_font_ref (gdkfont);

	name = get_font_name (gdkfont);

	if (name) {
		gchar *c[14], *p;
		const gchar * encoding;
		gchar *boldname, *lightname;
		gchar **namelist;
		GdkFont *newfont;
		gint numfonts, len;

		/* Allocate space for experimenting */

		len = strlen (name) + 64; /* Hope that's sufficent */
		p = alloca (len);

		/* Split name into components */

		split_name (c, name);

		/* Try to find iso-10646-1 encoded font with same name */
		/* Compose name for unicode encoding */
		encoding = "iso10646-1";
		g_snprintf (p, len, "%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s",
			    "*", c[1], c[2], c[3], c[4], c[5], c[6], c[7],
			    c[8], c[9], c[10], c[11], c[12], encoding);
		/* Try to load unicode font */
#ifdef E_FONT_VERBOSE
		g_print ("Trying font: %s\n", p);
#endif
		newfont = gdk_font_load (p);
		if (newfont) {
#ifdef E_FONT_VERBOSE
			e_font_print_gdk_font_name (newfont);
#endif
			/* OK, use that */
			gdk_font_unref (gdkfont);
			gdkfont = newfont;
		} else {
			/* Try to load fontset */
			g_snprintf (p, len, "%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s,*",
				    "*", c[1], c[2], c[3], c[4], c[5], c[6], c[7],
				    c[8], c[9], c[10], c[11], c[12], c[13]);
#ifdef E_FONT_VERBOSE
			g_print ("Trying fontset: %s\n", p);
#endif
			newfont = gdk_fontset_load (p);

			if (newfont) {
#ifdef E_FONT_VERBOSE
				e_font_print_gdk_font_name (newfont);
#endif
				/* OK, use fontset */
				gdk_font_unref (gdkfont);
				gdkfont = newfont;
				encoding = "*";
			} else {
				/* Nope, use original font */
				encoding = c[13];
			}
		}

		/* Here we are:
		 * gdkfont points to font(set) to be used, with right refcount
		 * encoding is the X encoding string for that font
		 */

		/* List all weight variants of given font(set) */

		g_snprintf (p, len, "%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s",
			    "*", c[1], c[2], "*", c[4], c[5], "*", "*",
			    c[8], c[9], c[10], c[11], "*", encoding);
		namelist = XListFonts (GDK_FONT_XDISPLAY (gdkfont), p, 32, &numfonts);

		if (find_variants (namelist, numfonts, c[3], &lightname, &boldname)) {
			gboolean is_fontset;

			is_fontset = (gdkfont->type == GDK_FONT_FONTSET);

			/* There are usable variants - their weights are returned in lightname and boldname */
			lightfont = NULL;
			boldfont  = NULL;
			if (!g_strcasecmp (c[3], lightname))
				/* Our name maps to light font */
				lightfont = gdkfont;
			else if (!g_strcasecmp (c[3], boldname))
				/* Our name maps to bold font */
				boldfont = gdkfont;
			else
				/* Our name maps neither to light nor to bold font */
				gdk_font_unref (gdkfont);

			if (!lightfont) {
				g_snprintf (p, len, "%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s%s",
					    "*", c[1], c[2], lightname, c[4],
					    c[5], "*", "*", c[8], c[9], c[10],
					    c[11], "*", encoding, is_fontset ? ",*" : "");
#ifdef E_FONT_VERBOSE
				g_print ("Trying light: %s\n", p);
#endif
				if (is_fontset) {
					lightfont = gdk_fontset_load (p);
				} else {
					lightfont = gdk_font_load (p);
				}
#ifdef E_FONT_VERBOSE
				e_font_print_gdk_font_name (lightfont);
#endif
			}
			if (!boldfont) {
				g_snprintf (p, len, "%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s-%s%s",
					    "*", c[1], c[2], boldname, c[4],
					    c[5], "*", "*", c[8], c[9], c[10],
					    c[11], "*", encoding, is_fontset ? ",*" : "");
#ifdef E_FONT_VERBOSE
				g_print ("Trying bold: %s\n", p);
#endif
				if (is_fontset) {
					boldfont = gdk_fontset_load (p);
				} else {
					boldfont = gdk_font_load (p);
				}
#ifdef E_FONT_VERBOSE
				e_font_print_gdk_font_name (boldfont);
#endif
			}
		} else {
			/* We fall back to double drawing */
			lightfont = gdkfont;
			boldfont = NULL;
		}

		XFreeFontNames (namelist);

		g_free (name);
	}

#ifdef E_FONT_VERBOSE
	g_print ("******** Creating EFont with following fonts ********\n");
	e_font_print_gdk_font_name (lightfont);
	e_font_print_gdk_font_name (boldfont);
#endif

	/* Here we are:
	 * lightfont is light GdkFont with correct refcount
	 * boldfont is either bold GdkFont or NULL
	 */

	font = g_new (EFont, 1);

	xfs = GDK_FONT_XFONT (lightfont);

	font->refcount = 1;
	font->font = lightfont;
	font->bold = boldfont;
	font->twobyte = (lightfont->type == GDK_FONT_FONTSET || ((lightfont->type == GDK_FONT_FONT) && ((xfs->min_byte1 != 0) || (xfs->max_byte1 != 0))));
	font->to = e_iconv_to_gdk_font (font->font);
	font->from = e_iconv_from_gdk_font (font->font);
	set_nbsp_zero_width_flag (font);

	/* Insert into cache */
	if (e_cache_insert (cache, keyfont, font, 1)) {
		/* Inserted, so add cache ref manually */
		e_font_ref (font);
	}

	return font;

}

gchar *
e_font_get_name (EFont *font)
{
	return get_font_name (font->font);
}

void
e_font_ref (EFont *font)
{
	font->refcount++;
}

void
e_font_unref (EFont *font)
{
	font->refcount--;

	if (font->refcount < 1) {
		gdk_font_unref (font->font);
		if (font->bold) gdk_font_unref (font->bold);
		g_free (font);
	}
}

gint
e_font_ascent (EFont * font)
{
	return font->font->ascent;
}

gint
e_font_descent (EFont * font)
{
	return font->font->descent;
}

static gint
no_conv_wrapper (EFont *font, gchar *native, const gchar *utf, gint bytes)
{
	gint len;
	const gchar *u;
	unicode_char_t uc;

	u   = utf;
	len = 0;

	while (u && u - utf < bytes) {
		u = unicode_get_utf8 (u, &uc);
		if (font->twobyte) {
			native [len] = (uc & 0xff00) >> 8; len++;
		}
		native [len] = uc & 0xff; len++;
	}

	return len;
}

static guchar *
replace_nbsp_with_spaces (const guchar *orig, gint *bytes)
{
	guchar *rv, *p;
	gboolean nbsp1;
	gint remains = *bytes;

	if (!orig)
		return NULL;

	/* printf ("bytes: %d orig\n%s\n", *bytes, orig); */

	p = rv = (guchar *) g_malloc (strlen (orig) + 1);
	nbsp1  = FALSE;

	while (remains) {
		if (*orig == 0xc2)
			nbsp1 = TRUE;
		else if (nbsp1) {
			if (*orig == 0xa0) {
				*p = ' ';   p ++;
				(*bytes) --;
			} else {
				*p = 0xc2;  p ++;
				*p = *orig; p ++;
			}
			nbsp1 = FALSE;
		} else {
			*p = *orig;
			p ++;
		}
		orig ++;
		remains --;
	}

	if (nbsp1) {
		*p = 0xc2; p ++;
	}
	*p = 0;

	/* printf ("rv: %s\n", rv); */

	return rv;
}

static gint
e_font_to_native (EFont *font, gchar *native, const gchar *utf, gint bytes)
{
	const char *ib;
	char *ob;
	size_t ibl, obl;
	gint rv;

	if (font->nbsp_zero_width)
		utf = replace_nbsp_with_spaces (utf, &bytes);

	if (font->to == (iconv_t) -1)
		rv = no_conv_wrapper (font, native, utf, bytes);
	else {
		ib = utf;
		ibl = bytes;
		ob = native;
		obl = bytes * 4;

		while (ibl > 0) {
			iconv (font->to, &ib, &ibl, &ob, &obl);
			if (ibl > 0) {
				gint len;
				if ((*ib & 0x80) == 0x00) len = 1;
				else if ((*ib &0xe0) == 0xc0) len = 2;
				else if ((*ib &0xf0) == 0xe0) len = 3;
				else if ((*ib &0xf80) == 0xf0) len = 4;
				else {
					g_warning ("Invalid UTF-8 sequence");
					break;
				}
				ib += len;
				ibl = bytes - (ib - utf);
				if (ibl > bytes) ibl = 0;
				if (!font->twobyte) {
					*ob++ = '_';
					obl--;
				} else {
					*((guint16 *) ob) = '_';
					ob += 2;
					obl -= 2;
				}
			}
		}
		rv = ob - native;
	}

	if (font->nbsp_zero_width)
		g_free ((void *) utf);

	return rv;
}

void
e_font_draw_utf8_text (GdkDrawable *drawable, EFont *font, EFontStyle style, GdkGC *gc, gint x, gint y, const gchar *text, gint numbytes)
{
	gchar *native;
	gint native_bytes;

	g_return_if_fail (drawable != NULL);
	g_return_if_fail (font != NULL);
	g_return_if_fail (gc != NULL);
	g_return_if_fail (text != NULL);

	if (numbytes < 1) return;

	native = alloca (numbytes * 4);

	native_bytes = e_font_to_native (font, native, text, numbytes);

	if ((style & E_FONT_BOLD) && (font->bold)) {
		gdk_draw_text (drawable, font->bold, gc, x, y, native, native_bytes);
	} else {
		gdk_draw_text (drawable, font->font, gc, x, y, native, native_bytes);
		if (style & E_FONT_BOLD)
			gdk_draw_text (drawable, font->font, gc, x + 1, y, native, native_bytes);
	}
}

gint
e_font_utf8_text_width (EFont *font, EFontStyle style, const char *text, int numbytes)
{
	gchar *native;
	gint native_bytes;
	gint width;

	g_return_val_if_fail (font != NULL, 0);
	g_return_val_if_fail (text != NULL, 0);

	if (numbytes < 1) return 0;

	native = alloca (numbytes * 4);

	native_bytes = e_font_to_native (font, native, text, numbytes);

	if ((style & E_FONT_BOLD) && (font->bold)) {
		width = gdk_text_width (font->bold, native, native_bytes);
	} else {
		width = gdk_text_width (font->font, native, native_bytes);
	}

	return width;
}

gint
e_font_utf8_char_width (EFont *font, EFontStyle style, char *text)
{
	gint len;

	g_return_val_if_fail (font != NULL, 0);
	g_return_val_if_fail (text != NULL, 0);

	if ((*text & 0x80) == 0x00) len = 1;
	else if ((*text &0xe0) == 0xc0) len = 2;
	else if ((*text &0xf0) == 0xe0) len = 3;
	else if ((*text &0xf80) == 0xf0) len = 4;
	else {
		g_warning ("Invalid UTF-8 sequence");
		return 0;
	}

	return e_font_utf8_text_width (font, style, text, len);
}

static const gchar *
translate_encoding (const gchar *encoding)
{
	static GHashTable *eh = NULL;
	gchar e[64];

#if 0
	static const gchar *charset;

	charset = get_locale_charset();
	if (charset) return charset;
#endif

	if (!eh) {
		eh = g_hash_table_new (g_str_hash, g_str_equal);

		g_hash_table_insert (eh, "iso8859-1", "iso-8859-1");
		g_hash_table_insert (eh, "iso_8859-1", "iso-8859-1");
		g_hash_table_insert (eh, "iso8859-2", "iso-8859-2");
		g_hash_table_insert (eh, "iso_8859-2", "iso-8859-2");
		g_hash_table_insert (eh, "iso8859-3", "iso-8859-3");
		g_hash_table_insert (eh, "iso_8859-3", "iso-8859-3");
		g_hash_table_insert (eh, "iso8859-4", "iso-8859-4");
		g_hash_table_insert (eh, "iso_8859-4", "iso-8859-4");
		g_hash_table_insert (eh, "iso8859-5", "iso-8859-5");
		g_hash_table_insert (eh, "iso_8859-5", "iso-8859-5");
		g_hash_table_insert (eh, "iso8859-6", "iso-8859-6");
		g_hash_table_insert (eh, "iso_8859-6", "iso-8859-6");
		g_hash_table_insert (eh, "iso8859-7", "iso-8859-7");
		g_hash_table_insert (eh, "iso_8859-7", "iso-8859-7");
		g_hash_table_insert (eh, "iso8859-8", "iso-8859-8");
		g_hash_table_insert (eh, "iso_8859-8", "iso-8859-8");
		g_hash_table_insert (eh, "iso8859-9", "iso-8859-9");
		g_hash_table_insert (eh, "iso_8859-9", "iso-8859-9");
		g_hash_table_insert (eh, "iso8859-10", "iso-8859-10");
		g_hash_table_insert (eh, "iso_8859-10", "iso-8859-10");
		g_hash_table_insert (eh, "iso8859-13", "iso-8859-13");
		g_hash_table_insert (eh, "iso_8859-13", "iso-8859-13");
		g_hash_table_insert (eh, "iso8859-14", "iso-8859-14");
		g_hash_table_insert (eh, "iso_8859-14", "iso-8859-14");
		g_hash_table_insert (eh, "iso8859-15", "iso-8859-15");
		g_hash_table_insert (eh, "iso_8859-15", "iso-8859-15");
		g_hash_table_insert (eh, "iso10646-1", "UTF-16");
		g_hash_table_insert (eh, "iso_10646-1", "UTF-16");
		g_hash_table_insert (eh, "koi8-r", "koi8-r");
	}

	strncpy (e, encoding, 64);
	g_strdown (e);

	return g_hash_table_lookup (eh, e);
}

static const gchar *
get_locale_charset (void)
{
	gchar *locale;
	static gchar *charset = NULL;
	static gboolean initialized = FALSE;

	if (!initialized) {
		locale = gtk_set_locale ();
		if ( strcmp (locale, "C") == 0 )
			locale = NULL;
		if ( locale != NULL ) {
			gchar * p;
			size_t len;

			p = strchr (locale, '@');
			len = p ? (p - locale) : strlen(locale);
			p = strchr (locale, '.');
			if (p) {
				charset = g_strndup(p+1, len - (p - locale) + 1);
				g_strdown (charset);
			}
		}
		initialized = TRUE;
	}

	return charset;
}

#if 0
static const gchar *
locale_charset_to_encoding(void)
{
	static GHashTable *lh = NULL;
	const gchar *charset;

	if (!lh) {
		lh = g_hash_table_new (g_str_hash, g_str_equal);

		g_hash_table_insert (lh, "eucjp", "iso8859-1");
		g_hash_table_insert (lh, "ujis", "iso8859-1");
	}
	charset = get_locale_charset ();

	return g_hash_table_lookup (lh, charset);
}
#endif

const gchar *
e_gdk_font_encoding (GdkFont *font)
{
	static ECache * cache = NULL;
	Atom font_atom, atom;
	Bool status;
	char *name, *p;
	const gchar *encoding;
	gint i;

	if (!font) return NULL;

	if (!cache) cache = e_cache_new (NULL, /* Key hash (GdkFont) */
					 NULL, /* Key compare (GdkFont) */
					 (ECacheDupFunc) gdk_font_ref, /* Key dup func */
					 (ECacheFreeFunc) gdk_font_unref, /* Key free func */
					 NULL, /* Object free func (const string) */
					 E_FONT_CACHE_SIZE, /* Soft limit (fake) */
					 E_FONT_CACHE_SIZE); /* Hard limit */

	encoding = e_cache_lookup (cache, font);
	if (encoding) return encoding;

#ifdef E_FONT_VERBOSE
	g_print ("Extracting X font info\n");
#endif

	if (font->type == GDK_FONT_FONTSET) {
		/* We are fontset, so try to find right encoding from locale */
		encoding = get_locale_charset ();
		if (encoding) {
		  const gchar *trans_encoding = translate_encoding (encoding);

		  if (trans_encoding) return trans_encoding;
		}
	}

	font_atom = gdk_atom_intern ("FONT", FALSE);

	if (font->type == GDK_FONT_FONTSET) {
		XFontStruct **font_structs;
		gint num_fonts;
		gchar **font_names;

		num_fonts = XFontsOfFontSet (GDK_FONT_XFONT (font),
					     &font_structs,
					     &font_names);
		status = XGetFontProperty (font_structs[0],
					   font_atom,
					   &atom);
	} else {
		status = XGetFontProperty (GDK_FONT_XFONT (font),
					   font_atom,
					   &atom);
	}

	if (!status) {
		/* Negative cache */
		e_cache_insert (cache, font, NULL, 1);
		return NULL;
	}

	name = p = gdk_atom_name (atom);

	for (i = 0; i < 13; i++) {
		/* Skip hyphen */
		while (*p && (*p != '-')) p++;
		if (*p) p++;
	}

	if (!*p) {
		/* Negative cache */
		e_cache_insert (cache, font, NULL, 1);
		return NULL;
	}

	encoding = translate_encoding (p);
	e_cache_insert (cache, font, (gpointer) encoding, 1);
	g_free (name);


	return encoding;
}

iconv_t
e_iconv_from_gdk_font (GdkFont *font)
{
	static GHashTable *ih = NULL;
	const gchar *enc;
	iconv_t ic;

	if (!font) return (iconv_t) -1;

	enc = e_gdk_font_encoding (font);

	if (!enc) return (iconv_t) -1;

	if (!ih) ih = g_hash_table_new (g_str_hash, g_str_equal);

	ic = g_hash_table_lookup (ih, enc);

	if (!ic) {
		ic = iconv_open ("UTF-8", enc);
		if (ic == (iconv_t) -1) return ic;
		g_hash_table_insert (ih, (gpointer) enc, ic);
	}

	return ic;
}

iconv_t
e_iconv_to_gdk_font (GdkFont *font)
{
	static GHashTable *ih = NULL;
	const gchar *enc;
	iconv_t ic;

	if (!font) return (iconv_t) -1;

	enc = e_gdk_font_encoding (font);

	if (!enc) return (iconv_t) -1;

	if (!ih) ih = g_hash_table_new (g_str_hash, g_str_equal);

	ic = g_hash_table_lookup (ih, enc);

	if (!ic) {
		ic = iconv_open (enc, "UTF-8");
		if (ic == (iconv_t) -1) return ic;
		g_hash_table_insert (ih, (gpointer) enc, ic);
	}

	return ic;
}

/*
 * Return newly allocated full name
 */

static gchar *
get_font_name (const GdkFont * font)
{
	Atom font_atom, atom;
	Bool status;

#ifdef E_FONT_VERBOSE
	g_print ("Extracting X font info\n");
#endif

	font_atom = gdk_atom_intern ("FONT", FALSE);

	if (font->type == GDK_FONT_FONTSET) {
		XFontStruct **font_structs;
		gint num_fonts;
		gchar **font_names;

		num_fonts = XFontsOfFontSet (GDK_FONT_XFONT (font), &font_structs, &font_names);
		status = XGetFontProperty (font_structs[0], font_atom, &atom);
	} else {
		status = XGetFontProperty (GDK_FONT_XFONT (font), font_atom, &atom);
	}

	if (status) {
		return gdk_atom_name (atom);
	}

	return NULL;
}

/*
 * Splits full X font name into pieces, overwriting hyphens
 */

static void
split_name (gchar * c[], gchar * name)
{
	gchar *p;
	gint i;

	p = name;

	for (i = 0; i < 13; i++) {
		c[i] = p;
		/* Skip text */
		while (*p && (*p != '-')) p++;
		/* Replace hyphen with '\0' */
		if (*p) *p++ = '\0';
	}

	c[i] = p;
}

/* Find light and bold variants of a font, ideally using the provided
 * weight for the light variant, and a weight 2 shades darker than it
 * for the bold variant. If there isn't something 2 shades darker, use
 * something 3 or more shades darker if it exists, or 1 shade darker
 * if that's all there is. If there is nothing darker than the provided
 * weight, but there are lighter fonts, then use the darker one for
 * bold and a lighter one for light.
 */
static gboolean
find_variants (gchar **namelist, gint length, gchar *weight,
	       gchar **lightname, gchar **boldname)
{
	static GHashTable *wh = NULL;
	/* Standard, Found, Bold, Light */
	gint sw, fw, bw, lw;
	gchar *s, *f, *b, *l;
	gchar *p;
	gint i;

	if (!wh) {
		wh = g_hash_table_new (g_str_hash, g_str_equal);
		g_hash_table_insert (wh, "light", GINT_TO_POINTER (1));
		g_hash_table_insert (wh, "book", GINT_TO_POINTER (2));
		g_hash_table_insert (wh, "regular", GINT_TO_POINTER (2));
		g_hash_table_insert (wh, "medium", GINT_TO_POINTER (3));
		g_hash_table_insert (wh, "demibold", GINT_TO_POINTER (5));
		g_hash_table_insert (wh, "bold", GINT_TO_POINTER (6));
		g_hash_table_insert (wh, "black", GINT_TO_POINTER (8));
	}

	s = alloca (strlen (weight) + 1);
	strcpy (s, weight);
	g_strdown (s);
	sw = GPOINTER_TO_INT (g_hash_table_lookup (wh, s));
	if (sw == 0) return FALSE;

	fw = 0; lw = 0; bw = 32;
	f = NULL; l = NULL; b = NULL;
	*lightname = NULL; *boldname = NULL;

	for (i = 0; i < length; i++) {
		p = namelist[i];
		if (*p) p++;
		while (*p && (*p != '-')) p++;
		if (*p) p++;
		while (*p && (*p != '-')) p++;
		if (*p) p++;
		f = p;
		while (*p && (*p != '-')) p++;
		if (*p) *p = '\0';
		g_strdown (f);
		fw = GPOINTER_TO_INT (g_hash_table_lookup (wh, f));
		if (fw) {
			if (fw > sw) {
				if ((fw - 2 == sw) ||
				    ((fw > bw) && (bw == sw + 1)) ||
				    ((fw < bw) && (fw - 2 > sw))) {
					bw = fw;
					b = f;
				}
			} else if (fw < sw) {
				if ((fw + 2 == sw) ||
				    ((fw < lw) && (lw == sw - 1)) ||
				    ((fw > lw) && (fw + 2 < sw))) {
					lw = fw;
					l = f;
				}
			}
		}
	}

	if (b) {
		*lightname = weight;
		*boldname = b;
		return TRUE;
	} else if (l) {
		*lightname = l;
		*boldname = weight;
		return TRUE;
	}
	return FALSE;
}

#ifdef E_FONT_VERBOSE
/*
 * Return newly allocated full name
 */

static void
e_font_print_gdk_font_name (const GdkFont * font)
{
	Atom font_atom, atom;
	Bool status;

	font_atom = gdk_atom_intern ("FONT", FALSE);

	g_print ("-------- start of font description --------\n");

	if (font == NULL) {
		g_print ("GdkFont is NULL\n");
	} else if (font->type == GDK_FONT_FONTSET) {
		XFontStruct **font_structs;
		gint num_fonts;
		gchar **font_names;
		gint i;

		num_fonts = XFontsOfFontSet (GDK_FONT_XFONT (font), &font_structs, &font_names);

		g_print ("Gdk Fontset:\n");
		for (i = 0; i < num_fonts; i++) {
			g_print ("    %s\n", font_names[i]);
		}
	} else {
		gchar * name;
		status = XGetFontProperty (GDK_FONT_XFONT (font), font_atom, &atom);
		name = gdk_atom_name (atom);
		g_print ("GdkFont: %s\n", name);
		if (name) g_free (name);
	}

	g_print ("-------- end of font description --------\n");
}
#endif
