/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "embed.h"
#include "window.h"
#include "misc.h"
#include "find.h"
#include "context.h"

#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkbox.h>
#include <gdk/gdkkeysyms.h>

extern const GtkTargetEntry bookmarks_dnd_targets[];

/* local function prototypes */
static void bookmarks_toolbars_create_from_root (GaleonWindow *window, 
						 BookmarkItem *bi);
static void  bookmarks_toolbar_create_from_folder (GaleonWindow *window,
						     BookmarkItem *bi);
static void bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, 
						    GtkToolbar *tb,
						    BookmarkItem *b,
						    ToolbarStyle style);
gboolean bookmarks_nick_entry_key_press_event_cb (GtkEntry *entry,
						  GdkEventKey *event,
						  BookmarkItem *bi);
static void bookmarks_toolbar_drag_data_received_cb 
                                       (GtkWidget *widget, 
					GdkDragContext *drag_context,
					gint x, gint y,
					GtkSelectionData *selection_data,
					guint info, guint time,
					BookmarkItem *bm);
static gboolean bookmarks_toolbar_button_press_event_cb (GtkWidget *item,
							GdkEventButton *event,
							BookmarkItem *bi);
static gboolean folder_button_press_event_cb (GtkWidget *item, 
						GdkEventButton *event,
						BookmarkItem *bi);
static gboolean folder_button_release_event_cb (GtkWidget *item, 
						  GdkEventButton *event,
						  BookmarkItem *bi);
static gboolean bookmark_button_press_event_cb (GtkWidget *item,
						GdkEventButton *event,
						BookmarkItem *bi);
static gboolean bookmark_button_release_event_cb (GtkWidget *item, 
						  GdkEventButton *event,
						  BookmarkItem *bi);
static gboolean smartbookmark_button_release_event_cb (GtkWidget *item, 
						       GdkEventButton *event,
						       BookmarkItem *bi);
static void bookmarks_toolbar_item_menu_hide_cb (GtkMenuItem *menuitem, 
						 BookmarkItem *bm);
						
/* this magic number is used to hack around a problem with
 * gnome-libs (I think) which causes multiple toolbars with
 * the same number not to display properly -- MattA
 * This number will also be appended to toolbar names to make sure no two
 * toolbars have the same name.  This keeps the gnome layout routines from
 * getting confused when restoring toolbar positions  --Josh */
static gint magic_number;

/**
 * bookmarks_toolbars_create: create personal toolbars
 */
void 
bookmarks_toolbars_create (GaleonWindow *window)
{
	gboolean visible;

	/* if no toolbars, skip this */
	if (!(window->toolBarOn))
	{
		return;
	}

	/* hack, we need probably to unify not created / not visible case */
	/* FIXME: shouldnt need menu bar on to know visiblity of toolbar */
	if (window->menuBarOn && window->view_toolbar)
	{
		visible = GTK_CHECK_MENU_ITEM (window->view_toolbar)->active;
	}
	else
	{
		visible = TRUE;
	}

	/* reset the magic number */
	magic_number = 99;

	/* build all the toolbars */
	bookmarks_toolbars_create_from_root (window, bookmarks_root);

	/* setup visibility of them all */
	bookmarks_toolbars_set_visibility (window, visible);
}

/**
 * bookmarks_toolbars_recreate: recreate personal toolbars
 */
void 
bookmarks_toolbars_recreate (GaleonWindow *window)
{
	GList *l;

	/* destroy all the toolbars */
	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkToolbar *toolbar = GTK_TOOLBAR (l->data);
		gtk_widget_destroy (GTK_WIDGET (toolbar)->parent);
	}

	/* list is now empty */
	g_list_free (window->bookmarks_toolbars);
	window->bookmarks_toolbars = NULL;

	/* now create toolbars for all of these */
	bookmarks_toolbars_create (window);
}

/**
 * bookmarks_toolbars_create_from_root: create toolbars from a given
 * root bookmark
 */
static void
bookmarks_toolbars_create_from_root (GaleonWindow *window, BookmarkItem *bi)
{
	GList *l;

	switch (bi->type)
	{
	case BM_SITE:
	case BM_SEPARATOR:
		/* can't create a toolbar from these */
		return;

	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		/* continue... */
		break;
	}

	/* if we're meant to create a toolbar... */
	if (bi->create_toolbar)
	{
		/* do so */
		bookmarks_toolbar_create_from_folder (window, bi);
	}
	
	/* now iterate to find any marked subfolders, 
	 * unless this is an alias */
	if (!(bi->alias_of))
	{
		for (l = bi->list; l != NULL; l = g_list_next (l))
		{
			bookmarks_toolbars_create_from_root (window, l->data);
		}
	}
}

/**
 * bookmarks_toolbar_create_from_folder: create a toolbar from a folder
 */
static void 
bookmarks_toolbar_create_from_folder (GaleonWindow *window, BookmarkItem *bi)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	GnomeDockItemBehavior props;
	GtkWidget *toolbar;
	gchar *unique_name;
	GList *l;

	/* read gnome prefs */
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();

	/* make the toolbar */
	toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
				   GTK_TOOLBAR_TEXT);

	/* check whether it should be detachable or not */
	props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
		 (toolbar_detachable ? 0 : GNOME_DOCK_ITEM_BEH_LOCKED));

	/* make sure each toolbar has a unique name */
	unique_name = g_strdup_printf("%ld", bi->id);

	/* add the toolbar it to the main window */
	gnome_app_add_docked (GNOME_APP (window->WMain), toolbar, unique_name,
			      props, GNOME_DOCK_TOP, magic_number++, 0, 0);
	gtk_container_set_border_width (GTK_CONTAINER (toolbar->parent),
					toolbar_relief ? 2 : 1);
	gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (toolbar->parent), 
					 toolbar_relief ? 
					 GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	g_free (unique_name);

	/* override global setting */
	gtk_toolbar_set_space_style (GTK_TOOLBAR (toolbar),
				     GTK_TOOLBAR_SPACE_EMPTY);
	gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 6);

	/* iterate over the contents of the folder */
	for (l = bi->list; l != NULL; l = g_list_next (l))
	{
		BookmarkItem *bm = l->data;

		/* build the appropriate element */
		if (bm->type != BM_SEPARATOR)
		{
			bookmarks_toolbar_create_from_bookmark 
						(window, GTK_TOOLBAR (toolbar), 
						 bm, bi->toolbar_style);
		} 
		else 
		{
			gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
		}
	}

	gtk_signal_connect (GTK_OBJECT (toolbar), "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_data_received_cb), bi);
	gtk_drag_dest_set (toolbar,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (toolbar->parent), "button-press-event",
			    GTK_SIGNAL_FUNC 
				(bookmarks_toolbar_button_press_event_cb), bi);

	/* show it */
	gtk_widget_show_all (GTK_WIDGET (toolbar));

	/* add it to the per-window list of toolbars */
	window->bookmarks_toolbars = 
		g_list_append (window->bookmarks_toolbars,
			       GTK_TOOLBAR (toolbar));

}

/**
 * bookmarks_toolbar_create_from_bookmark:
 */
static void
bookmarks_toolbar_create_from_bookmark (GaleonWindow *window, 
					GtkToolbar *tb,
					BookmarkItem *b, 
					ToolbarStyle style)
{
        GtkWidget *button, *box, *pixmap, *entry;
	const PixmapData *bm_icon;
	GtkWidget *dnd_dest;
	gchar *name = NULL;
	gchar *tip = NULL;
	gchar *short_url = NULL;

	/* build a box, and pack the pixmap and label inside */
	if (style == TOOLBAR_STYLE_VERTICAL)
		box = gtk_vbox_new (FALSE, 2);
	else
		box = gtk_hbox_new (FALSE, 2);
	if (style != TOOLBAR_STYLE_TEXT_ONLY)
	{
		if (b->pixmap_data == NULL)
		{
			if (b->type == BM_FOLDER || 
			    b->type == BM_AUTOBOOKMARKS)
			{
				if (default_bookmarks_root == b)
					pixmap = gtk_pixmap_new 
						(default_folder_pixmap_data->pixmap,
					 	 default_folder_pixmap_data->mask);
				else
					pixmap = gtk_pixmap_new 
						(folder_pixmap_data->pixmap,
					 	 folder_pixmap_data->mask);
			}
			else
			{
				bm_icon = bookmarks_get_siteicon (b->url);
				pixmap = gtk_pixmap_new (bm_icon->pixmap, 
							 bm_icon->mask);
			}
			gtk_box_pack_start (GTK_BOX (box), pixmap, 
					    FALSE, FALSE, 0);
		}
		else
		{
			pixmap = gtk_pixmap_new (b->pixmap_data->pixmap, 
						 b->pixmap_data->mask);
			gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET (pixmap),
					    TRUE, FALSE, 0);
		}
	}

	if (b->pixmap_data == NULL || style == TOOLBAR_STYLE_TEXT_ONLY)
	{
		name = strip_uline_accel (b->name);
		gtk_box_pack_start (GTK_BOX (box), gtk_label_new (name),
				    TRUE, TRUE, 0);
		g_free (name);
	}

	/* is this a normal (i.e. non-smart) bookmark */
	if ((b->url == NULL) || strstr(b->url, "%s") == NULL)
	{
		/* yes, build a button and pack it */
		dnd_dest = button = gtk_button_new ();
		gtk_container_add(GTK_CONTAINER(button), box);

		if ((b->url != NULL) && strlen (b->url) > BOOKMARKS_TOOLTIP_MAX_LENGTH) {
			gchar *s = g_strndup (b->url, BOOKMARKS_TOOLTIP_MAX_LENGTH);
			short_url = g_strconcat (s, "...", NULL);
			g_free (s);
		} 
		else 
		{
			short_url = b->url;
		}

		if (b->notes == NULL || b->notes[0] == '\0')
		{
			tip = short_url;
		}
		else
		{
			tip = g_strdup_printf ("%s\n%s", short_url, b->notes);
		}
		
		
		gtk_toolbar_append_widget (tb, button, tip, NULL);
		if (short_url != b->url) g_free (short_url);
		if (tip != short_url) g_free (tip);

		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);

		/* the click handlers */
		if (b->type == BM_SITE)
		{
			gtk_signal_connect (GTK_OBJECT (dnd_dest), 
					    "button-press-event",
					    GTK_SIGNAL_FUNC
					    (bookmark_button_press_event_cb),
					    b);
			gtk_signal_connect (GTK_OBJECT (dnd_dest), 
					    "button-release-event",
					    GTK_SIGNAL_FUNC
					    (bookmark_button_release_event_cb),
					    b);
		}
		else
		{
			gtk_signal_connect (GTK_OBJECT (dnd_dest), 
					    "button-press-event",
					    GTK_SIGNAL_FUNC
					    (folder_button_press_event_cb),
					    b);
			gtk_signal_connect (GTK_OBJECT (dnd_dest), 
					    "button-release-event",
					    GTK_SIGNAL_FUNC
					    (folder_button_release_event_cb),
					    b);
		}
	}
	else
	{
		GtkWidget *button = gtk_button_new ();
		gint entry_width;

		dnd_dest = button;
		entry_width = gnome_config_get_int
			("/galeon/Appearance/smart_bm_entry_width=100");
		gtk_container_add (GTK_CONTAINER (button), 
				   GTK_WIDGET (box));
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
		gtk_container_set_border_width (GTK_CONTAINER (box), 0);
		/* otherwise build an entry field and pack it */
		entry = gtk_entry_new ();
		gtk_widget_set_usize(entry, entry_width, -2);
		gtk_toolbar_append_widget (tb, button, 
					   _("Search for entered phrase"),
					   NULL);
		gtk_toolbar_append_widget (tb, entry, b->url, NULL);
		gtk_toolbar_append_space (tb);
		gtk_object_set_data (GTK_OBJECT (button), "entry", entry);
		gtk_object_set_data (GTK_OBJECT (entry), "GaleonWindow", 
				     window);
		gtk_signal_connect (GTK_OBJECT (entry), "key-press-event",
		       (GtkSignalFunc) bookmarks_nick_entry_key_press_event_cb,
		       b);

		/* the click handlers */
		gtk_signal_connect (GTK_OBJECT (dnd_dest), 
				    "button-press-event",
				    GTK_SIGNAL_FUNC
				    (bookmark_button_press_event_cb), b);
		gtk_signal_connect (GTK_OBJECT (dnd_dest), 
				    "button-release-event",
				    GTK_SIGNAL_FUNC
				    (smartbookmark_button_release_event_cb), b);
	}

	gtk_signal_connect (GTK_OBJECT (dnd_dest), "drag_data_received",
			    GTK_SIGNAL_FUNC 
			    (bookmarks_toolbar_drag_data_received_cb), b);
	gtk_drag_dest_set (dnd_dest,
			   GTK_DEST_DEFAULT_ALL,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);
}

#if 0
/**
 * bookmarks_toolbars_update_entries_global: search the bookmarks toolbars for
 *	smart bookmark entry widgets and sync their sizes with the
 *	preferences setting
 */
void
bookmarks_toolbars_update_entries_global (void)
{
	gint entry_width;
	GList *toolbars;
	GList *l;

	entry_width = gnome_config_get_int
		("/galeon/Appearance/smart_bm_entry_width=100");

	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *)(l->data);

		for (toolbars = window->bookmarks_toolbars; toolbars != NULL;
		     toolbars = g_list_next (toolbars))
		{
			GtkToolbar *toolbar = GTK_TOOLBAR (toolbars->data);
			GList *children = toolbar->children;

			while (children)
			{
				GtkToolbarChild *child = children->data;

				if (GTK_IS_ENTRY (child->widget))
				{
					gtk_widget_set_usize (child->widget,
							      entry_width, -2);
				}
				children = children->next;
			}
		}
	}			
}
#endif

void 
bookmarks_toolbars_set_visibility (GaleonWindow *window, gboolean visible)
{
	GList *l;

	for (l = window->bookmarks_toolbars; l != NULL; l = g_list_next (l))
	{
		GtkToolbar *toolbar = GTK_TOOLBAR (l->data);

		if (GTK_IS_WIDGET (GTK_WIDGET (toolbar)->parent))
		{
			GtkWidget *w = GTK_WIDGET (toolbar)->parent;

			if (visible)
			{
				gtk_widget_show (w);
			}
			else
			{
				gtk_widget_hide (w);
			}
		}
	}
}

/** 
 * FIXME: comments
 */
static gboolean
folder_button_press_event_cb (GtkWidget *item, GdkEventButton *event,
				BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	/* create the folder menu */
	if (event->button == 1)
	{
		GtkWidget *menu = gtk_menu_new();
		GtkTooltips *tooltips = gtk_tooltips_new ();
		
		bookmarks_menu_create_recursively
			(bi, GTK_MENU (menu), NULL, tooltips, 
			 TRUE, TRUE);
		
		/* attach "GaleonWindow" to the menu so it can be looked up */
		gtk_object_set_data (GTK_OBJECT (menu), "GaleonWindow",
				     window);
		
		gtk_signal_emit_stop_by_name (GTK_OBJECT (item),
					      "button-press-event");
		
		gnome_popup_menu_do_popup_modal (menu, 
						 menu_position_under_widget,
						 item, event, NULL);
		gtk_widget_unref (GTK_WIDGET(menu));

		if (tooltips)
		{
			gtk_object_destroy (GTK_OBJECT (tooltips));
		}
	}
	else if (event->button == 3)
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());

		bookmarks_menu_create_extra_items (popup,
				bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);

		context_menu_add_seperator (popup);
		context_show_appearance_menu (window, popup, bi, TRUE, TRUE);
		context_menu_add_seperator (popup);

		context_menu_add_item (popup, _("Hide toolbar"),
				bookmarks_toolbar_item_menu_hide_cb, bi, 
				NULL, TRUE);

		/* warning: ugly code */
		gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
				    window);

		gtk_signal_emit_stop_by_name (GTK_OBJECT(item),
				"button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET (popup));
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
folder_button_release_event_cb (GtkWidget *item, GdkEventButton *event,
				  BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);
	
	if (item->window != event->window)
		return FALSE;

	/* Load the bookmark(s) */
	if (event->button == 2)
	{
		gboolean tabbed_mode;

		tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);
		if (event->state & GDK_SHIFT_MASK)
			tabbed_mode = !tabbed_mode;

		/* Load the bookmark in a new tab/window */
		bookmarks_folder_open_all_items (window->active_embed,
						 bi, !tabbed_mode, FALSE);
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmark_button_press_event_cb (GtkWidget *item, GdkEventButton *event,
				BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	if (event->button == 3)
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());

		bookmarks_item_create_extra_items (popup, bi);
		
		context_menu_add_seperator (popup);
		context_show_appearance_menu (window, popup, bi, TRUE, TRUE);
		context_menu_add_seperator (popup);

		context_menu_add_item (popup, _("Hide toolbar"),
				bookmarks_toolbar_item_menu_hide_cb, bi,
				NULL, TRUE);

		/* warning: ugly code */
		gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
				    window);

		gtk_signal_emit_stop_by_name (GTK_OBJECT(item),
				"button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET (popup));
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmark_button_release_event_cb (GtkWidget *item, GdkEventButton *event,
				  BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);
	
	if (item->window != event->window)
		return FALSE;

	/* Load the bookmark(s) */
	if (!GTK_IS_EVENT_BOX(item))
	{
		/* normal bookmark */
		bookmarks_set_visited (bi);
		embed_activate_link_mouse (window->active_embed, NULL, 
					   bi->url, event);
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
bookmarks_toolbar_button_press_event_cb (GtkWidget *item,
					 GdkEventButton *event,
					 BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);

	if (event->button == 3)
	{
		GtkMenu *popup = GTK_MENU (gtk_menu_new ());

		bookmarks_menu_create_extra_items (popup,
				bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);

		context_menu_add_seperator (popup);
		context_show_appearance_menu (window, popup, bi, FALSE, TRUE);

		/* warning: ugly code */
		gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
				    window);

		gtk_signal_emit_stop_by_name (GTK_OBJECT(item),
				"button-press-event");

		gnome_popup_menu_do_popup_modal (GTK_WIDGET (popup), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET (popup));
	}
	return TRUE;
}

/**
 * FIXME: comments
 */
static gboolean
smartbookmark_button_release_event_cb (GtkWidget *item, GdkEventButton *event,
				       BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (item);
	GaleonEmbed *embed = window->active_embed;
	gchar *text, *translated_text, *url;
	GtkWidget *entry;
	
	if (item->window != event->window)
		return FALSE;

	if (GTK_IS_EVENT_BOX (item))
	{
		return FALSE;
	}

	/* smart bookmark, do search for phrase */
	entry = gtk_object_get_data (GTK_OBJECT (item), "entry");
	text = gtk_entry_get_text (GTK_ENTRY (entry));
	if (text != NULL && strlen (text) > 0)
	{
		/* translate non-alphanumeric characters+into %{} format */
		translated_text = bookmarks_translate_string (text);
		
                /* get a completed url */
		url = bookmarks_substitute_argument (bi, translated_text);

		/* if the search was already performed... */
		if (g_strcasecmp (url, embed->site_location) == 0 &&
		    event->button == 1)
		{
			/* search in the current page for the entered text */
			find_next (embed, text);
		}
		else
		{
			/* otherwise activate normal behaviour */
			bookmarks_set_visited (bi);
			embed_activate_link_mouse (window->active_embed, NULL,
						   url, event);
		}
		g_free (url);
	}
	else
	{
		/* if search field is empty, go to a normal bookmark
		 * with matching nick, if one exists. */
		BookmarkItem *b = bookmarks_find_by_nick (bi->nick,
							bookmarks_root, FALSE);
		if (b != NULL && b->url != NULL)
		{
			bookmarks_set_visited (b);
			embed_activate_link_mouse (window->active_embed, NULL,
						   b->url, event);
		}

	}

	return TRUE;
}

/**
 * bookmarks_toolbar_menu_hide_cb: called by the hide context menu on the
 * toolbar items. Hide the toolbar.
 */
static void
bookmarks_toolbar_item_menu_hide_cb (GtkMenuItem *menuitem, 
				BookmarkItem *bm)
{
	bm->parent->create_toolbar = FALSE;
	bookmarks_updated ();
}

/**
 * bookmarks_nick_entry_key_press_event_cb: called when the user hits
 * return on an entry field on the toolbar (i.e. one created by a bookmark
 * which has %s in the url)
 */
gboolean
bookmarks_nick_entry_key_press_event_cb (GtkEntry *entry, GdkEventKey *event,
					 BookmarkItem *bi)
{
	gchar *text, *url, *translated_text;
	GaleonWindow *window;
	if (event->keyval != GDK_Return)
		return FALSE;

	/* find the window */
	window = gtk_object_get_data (GTK_OBJECT (entry),
				      "GaleonWindow");

	/* get the entry text: DON'T free the returned text! */
	text = gtk_entry_get_text (entry);

	/* translate non-alphanumeric characters into %{} format */
	translated_text = bookmarks_translate_string (text);

	/* get a completed url */
	url = bookmarks_substitute_argument (bi, translated_text);

	/* load it in correct location*/
	bookmarks_set_visited (bi);
	embed_activate_link_keyboard (window->active_embed, NULL, url,
				      event->state);

	/* free all allocated strings */
	g_free (url);
	g_free (translated_text);
	return TRUE;
}

/** 
 * bookmarks_toolbar_drag_data_received_cb:
 */
static void
bookmarks_toolbar_drag_data_received_cb (GtkWidget *widget, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 BookmarkItem *drag_bm)
{
	BookmarkItem *b = NULL; 
	gchar *mem = selection_data->data;
	
	g_return_if_fail (drag_bm != NULL);

	switch (info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_item_from_string (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, NULL, mem, 
					    NULL, NULL, NULL);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	if (b)
	{
		bookmarks_insert_bookmark (b, drag_bm, GTK_CLIST_DRAG_INTO);
		bookmarks_editor_place_tree_items (b);
		bookmarks_updated ();
	 }
}

