/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "embed.h"
#include "window.h"
#include "misc.h"
#include "history.h"
#include "prefs.h"
#include "mozilla_prefs.h"
#include "mozilla.h"
#include "mozcallbacks.h"
#include "bookmarks.h"
#include "downloader.h"
#include "MozRegisterComponents.h"
#include "filepicker.h"
#include "dialog.h"

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/* local function prototypes */
static GaleonEmbed *embed_create_no_window (void);
static void embed_connect_signals (GaleonEmbed *embed);
static void embed_single_init (void);
static void embed_view_source_external (GaleonEmbed *embed, const gchar *url);
static void add_bf_menu (GtkWidget *menu, char* label, int index, 
			 GaleonEmbed *embed, int level);
static GtkWidget * new_num_accel_menu_item (gint num, gchar *origtext, 
					    gboolean lettersok, 
					    GtkWidget *menu);
static void save_url (GaleonEmbed *embed, char *target, 
		      char* default_dir_pref, gboolean ask);
static void save_url_message (GaleonEmbed *embed, char *path);
static void embed_select_new_page (GaleonEmbed *embed);

/** The global list of all GaleonEmbed structures */
GList *all_embeds = NULL;

/* global character set hash and sorted title list*/
GHashTable *charsets = NULL;
GList *sorted_charset_titles = NULL;

/* tab close button global pixmap data */
PixmapData *close_pix = NULL;

/* global var to indicate whether xpcom has been initialised */
gboolean pushed_startup = FALSE;

/* DnD drop types for embeds and tabs -- NOTE NOTE NOTE dont mess around 
 * with the order of these unless you know what you're doing! -- MattA */
const GtkTargetEntry embed_drop_types[] =
{
	{ "GALEON_EMBED",    0, DND_TARGET_GALEON_EMBED    },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "text/uri-list",   0, DND_TARGET_TEXT_URI_LIST   },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint embed_drop_types_num_items = (sizeof (embed_drop_types) / 
				         sizeof (GtkTargetEntry));

/**
 * embed_create: create a GaleonEmbed structure based on a given Embed
 * @previous can be NULL (orphan popups)
 */
GaleonEmbed *
embed_create (GaleonEmbed *previous, gboolean new_window, 
	      gboolean force_jump, guint32 chrome_mask)
{
	GaleonWindow *window = NULL;
	GaleonEmbed *embed;

	/* get the parent window if we're using it */
	if (previous != NULL && !new_window)
	{
		window = previous->parent_window;
	}
	else
	{
		/* create our own */
		window = window_create (chrome_mask);
	}

	/* create embed */
	embed = embed_create_no_window ();

	/* setup chromedness */
	if ((chrome_mask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
	{
		embed->is_chrome = TRUE;
	}

	/* add to window */
	window_add_embed (window, embed, force_jump);

	/* return completed embed */
	return embed;
}

/**
 * embed_create_in_window: create embed in a given window
 */
GaleonEmbed *
embed_create_in_window (GaleonWindow *window)
{
	GaleonEmbed *embed;

	/* create embed */
	embed = embed_create_no_window ();

	/* add to window */
	window_add_embed (window, embed, FALSE);

	/* return completed embed */
	return embed;
}

/**
 * embed_create_from_url: create a browser from a given url string
 */
GaleonEmbed *
embed_create_from_url (GaleonEmbed *previous, const gchar *url, 
		       gboolean force_jump, gboolean new_window)
{
	GaleonEmbed *embed;

	/* check argument */
	g_assert(url != NULL);

	/* create a window */
	embed = embed_create (previous, new_window, force_jump, 
			      GTK_MOZ_EMBED_FLAG_ALLCHROME);

	/* show the window */
	embed_set_visibility (embed, TRUE);

	/* load the url */
	embed_load_url (embed, url);

	/* return completed embed */
	return embed;
}

/**
 * embed_create_default: create a new browser pointing at the page
 * specified by configuration
 */
GaleonEmbed *
embed_create_default (GaleonEmbed *previous_embed, gboolean new_window)
{
	StartPageType start_page;
	GaleonEmbed *embed;
	gchar *url;

	/* get default URL */
	url = embed_get_default_url (previous_embed);

	/* load into a new window */
	embed = embed_create_from_url (previous_embed, url, TRUE, new_window);

	/* copy across session history (the wrapper must be initialized) */
	start_page = gnome_config_get_int (CONF_GENERAL_NEWPAGE_TYPE);
	if (previous_embed != NULL && start_page == STARTPAGE_LAST)
	{
		mozilla_copy_session_history (previous_embed, embed);
	}

	/* free and return */
	g_free (url);
	return embed;
}

/**
 * embed_create_from_url_view_source: create a browser from a given 
 * url string  in view source mode
 */
GaleonEmbed *
embed_create_from_url_view_source (GaleonEmbed *previous, const gchar *url,
				   gboolean new_window)
{
	GaleonEmbed *embed;
	gchar *vs_url;

	/* prepend view-source: protocol */
	vs_url = g_strconcat ("view-source:", url, NULL);

	/* get completed browser */
	embed = embed_create_from_url (previous, vs_url, TRUE, new_window);

	/* free allocated string */
	g_free (vs_url);
	
	/* return completed browser */
	return embed;
}

/**
 * embed_create_no_window: create a GaleonEmbed without reference to
 * a parent window
 */
static GaleonEmbed *
embed_create_no_window (void)
{
	GaleonEmbed *embed;

	/* build an embed structure */
	embed = g_new0 (GaleonEmbed, 1);

	/* add to global list of all embeds */
	all_embeds = g_list_prepend (all_embeds, embed);

	/* not closing */
	embed->is_closing = FALSE;

	/* no content for the location or title */
	embed->site_location = NULL;
	embed->site_title = g_strdup (_("Untitled"));
	embed->site_title_utf8 = locale_to_utf8 (embed->site_title);

	/* initial zoom level */
	embed->zoom = 100;
	
	/* make an embedding widget and check */
	embed->mozEmbed = gtk_moz_embed_new ();
	g_assert (GTK_IS_MOZ_EMBED (embed->mozEmbed));

	/* ref/unref widget -- is this really needed? */
	/* NB: apparently it is to get rid of some warnings, but I'm
	 * still not clear why -- MattA 13/4/2001 */
	/* as of 1/6/2001, it still doens't work... ! */
	gtk_widget_ref (GTK_WIDGET (embed->mozEmbed));
	gtk_object_set_data_full (GTK_OBJECT (embed->mozEmbed), "mozEmbed",
				  GTK_WIDGET (embed->mozEmbed),
				  (GtkDestroyNotify) gtk_widget_unref);

	/* setup data so we can always find GaleonEmbed */
	gtk_object_set_data (GTK_OBJECT (embed->mozEmbed), 
			     "GaleonEmbed", embed);

	/* connect appropriate signals */
	embed_connect_signals (embed);
	
	/* set magic as this is now ready to go */
	embed->magic = GALEON_EMBED_MAGIC;

	/* done! */
	return embed;
}

/**
 * embed_get_default_url: get the default URL on new page creation
 */
gchar *
embed_get_default_url (GaleonEmbed *previous_embed)
{
	const gchar *last_page_url;
	gchar *home_page_url;
	gint page_type;

	/* find out where we're supposed to start */
	if (previous_embed == NULL)
	{
		page_type = gnome_config_get_int (CONF_GENERAL_STARTPAGE_TYPE);
	}
	else
	{
		page_type = gnome_config_get_int (CONF_GENERAL_NEWPAGE_TYPE);
	}

       	/* return the appropriate page */
	if (page_type == STARTPAGE_HOME)
	{
		/* get location of home page */
		home_page_url = gnome_config_get_string(CONF_GENERAL_HOMEPAGE);

		/* load home page, if set */
		if (home_page_url != NULL)
		{
			return home_page_url;
		}
	}

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_embed != NULL && previous_embed->site_location != NULL)
	{
		last_page_url = previous_embed->site_location;
	}
	else
	{
		last_page_url = history_get_last_url ();
	}
	
	if (page_type == STARTPAGE_LAST && last_page_url != NULL)
	{
		/* return last page pointed at */
		return g_strdup (last_page_url);
	}

	/* even in case of error, it's a good default */
	return g_strdup ("about:blank");
}

/**
 * embed_set_visibility: set the visiblity of a single embed
 */
void
embed_set_visibility (GaleonEmbed *embed, gboolean visibility)
{
	GaleonWindow *window;

	/* check args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* test for hiding */
	if (!visibility && embed->is_visible)
	{
		window_hide_one_embed (window);
	}

	/* test for showing */
	if (visibility && !embed->is_visible)
	{
		window_show_one_embed (window);

		/* show the widget */
		/* don't ask me why this has to be done after showing
		 * the main window, it just does... */
		gtk_widget_show (embed->mozEmbed);
	}

	/* new status */
	embed->is_visible = visibility;
}

/**
 * embed_load_url: interpret and load a URL into a given GaleonEmbed
 */
void
embed_load_url (GaleonEmbed *embed, const gchar *url)
{
	/* check arguments */
	return_if_not_embed (embed);
	g_assert(url != NULL);

	/* update location bar */
	if (embed->site_location != NULL)
	{
		g_free (embed->site_location);
	}
	embed->site_location = g_strdup (url);
	embed_update_page_location (embed);

	/* reset title */
	g_free (embed->site_title);
	g_free (embed->site_title_utf8);
	embed->site_title = g_strdup (_("Untitled"));
	embed->site_title_utf8 = locale_to_utf8 (embed->site_title);
	embed_update_page_title (embed);

	/* load the URL */
	gtk_moz_embed_load_url (GTK_MOZ_EMBED(embed->mozEmbed), url);

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
	{
		embed_wrapper_init (embed);
	}
}

/**
 * embed_activate_link_mouse: handle a link click, using a mouse state
 */
gboolean
embed_activate_link_mouse (GaleonEmbed *previous, GaleonEmbed **new_embed,
			   const gchar *url, GdkEventButton *event)
{
	LinkState link_state;

	/* get link state from mouse state */
	link_state = mouse_state_to_link_state (event->button, event->state);

	/* do the activation */
	return embed_activate_link (previous, new_embed, url, link_state);
}

/**
 * embed_activate_link_keyboard: handle a link click, using a keyboard state
 */
gboolean
embed_activate_link_keyboard (GaleonEmbed *previous, GaleonEmbed **new_embed,
			      const gchar *url, guint evstate)
{
	LinkState link_state;

	/* get link state from mouse state */
	link_state = key_state_to_link_state (evstate);

	/* do the activation */
	return embed_activate_link (previous, new_embed, url, link_state);
}

/**
 * embed_activate_link: generalized URL activation function
 */
gboolean
embed_activate_link (GaleonEmbed *previous, GaleonEmbed **new_embed,
		     const gchar *url, LinkState link_state)
{
	gboolean tabbed_mode, handled;
	gboolean shifted, ctrled, alted;
	GaleonEmbed *new = NULL;
	guint button;

	/* check URL is valid */
	if (url == NULL)
	{
		return FALSE;
	}

	/* parse link state */
	handled = FALSE;
	button = (link_state & LINKSTATE_BUTTON_MASK) + 1; 
	shifted = link_state & LINKSTATE_SHIFTED;
	ctrled = link_state & LINKSTATE_CTRLED;
	alted = link_state & LINKSTATE_ALTED;
	
        /* parse button */
	switch (button)
	{
	case 1:
		/* left button (or no button) */
		if (!previous)
		{
			new = previous = embed_create_from_url 
				(NULL, "about:blank", FALSE, TRUE);
		}
		if (shifted)
		{
			downloader_save_link (previous, url);
		}
		else
		{
			embed_load_url (previous, url);
		}
		handled = TRUE;
		break;

	case 2:
		/* middle button */
		if (link_state == LINKSTATE_NEWTAB)
		{
			tabbed_mode = TRUE;
		}
		else if (link_state == LINKSTATE_NEWWIN)
		{
			tabbed_mode = FALSE;
		}
		else
		{
			tabbed_mode = gnome_config_get_bool 
				(CONF_APPEARANCE_TABBED);
		}
		if (shifted && !(alted || ctrled))
		{
			new = embed_create_from_url (previous, url,
						     FALSE, tabbed_mode);
		}
		else
		{
			new = embed_create_from_url (previous, url,
						     FALSE, !tabbed_mode);
		}
		handled = TRUE;
		break;

	case 3:
		/* context menu shown by other stuff */
		handled = FALSE;
		break;

	default:
		g_warning ("link selected with unusual button");
		handled = FALSE;
		break;
	}

	/* store return value (NULL if not handled) */
	if (new_embed != NULL)
	{
		*new_embed = new;
	}

	/* done */
	return handled;
}

/**
 * embed_switch_to_page: switch to the embed notebook page
 */
void 
embed_switch_to_page (GaleonEmbed *embed)
{
	GaleonWindow *window;
	gint page;

	/* check */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* lookup the page number */
	page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
				      GTK_WIDGET (embed->mozEmbed));

	/* jump to that page */
	gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
}

/**
 * embed_startup_init: initialitation that need to occur only one time
 * before any gtkmozembed object is created
 */
void
embed_startup_init (void)
{
	/* general mozilla stuff that needs doing */
	embed_single_init ();
	mozilla_register_components ();
	mozilla_load_prefs ();
	
	/* only register the external ftp handler if it isn't galeon */
	if (gnome_config_get_int(CONF_HANDLER_EXTERNAL_FTP) == 1 &&
	    !(preferences_galeon_used_by_gnome_for_ftp ()))
	{
		mozilla_register_FtpProtocolHandler ();
	}
}

/**
 * embed_connect_signals: connect the necessary signals to the embed
 */
static void 
embed_connect_signals (GaleonEmbed *embed)
{
	gint i;

        /* signals to connect on each embed widget */
	static const struct
	{ 
		char *event; 
		void *func; /* should be a GtkSignalFunc or similar */
	}
	signal_connections[] =
	{
		{ "location",        mozembed_location_changed_cb  },
		{ "title",           mozembed_title_changed_cb     },
		{ "net_start",       mozembed_load_started_cb      },
		{ "net_stop",        mozembed_load_finished_cb     },
		{ "net_state",       mozembed_net_status_change_cb },
		{ "progress",        mozembed_progress_change_cb   },
		{ "link_message",    mozembed_link_message_cb      },
		{ "js_status",       mozembed_js_status_cb         },
		{ "open_uri",        mozembed_open_uri_cb          },
		{ "visibility",      mozembed_visibility_cb        },
		{ "destroy_browser", mozembed_destroy_brsr_cb      },
		{ "dom_mouse_down",  mozembed_dom_mouse_down_cb    },	
		{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
		{ "dom_key_press",   mozembed_dom_key_press_cb     },
		{ "size_to",         mozembed_size_to_cb           },
		{ "new_window",      mozembed_new_window_cb        },

		/* terminator -- must be last in the list! */
		{ NULL, NULL } 
	};
	
	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(embed->mozEmbed),
						signal_connections[i].event,
						signal_connections[i].func, 
						embed,
						GTK_OBJECT(embed->mozEmbed));
	}
	
	/* also connect destroy (but not with "while_alive") */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), embed);

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(embed->mozEmbed), GTK_DEST_DEFAULT_ALL,
			   embed_drop_types, embed_drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtkmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_received",
			    GTK_SIGNAL_FUNC (embed_drag_data_received_cb),
			    embed);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT (embed->mozEmbed), "drag_data_get",
			    GTK_SIGNAL_FUNC (window_drag_data_get_cb),
			    embed);
}

/**
 * embed_single_init: init the gtkmozembed Single object
 */
static void
embed_single_init (void)
{
	GtkMozEmbedSingle *single;

	/* startup done */
	gtk_moz_embed_push_startup ();
	pushed_startup = TRUE;

	/* get single */
	single = gtk_moz_embed_single_get ();
	if (single == NULL)
	{
		g_warning ("Failed to get singleton embed object!\n");
	}

	/* allow creation of orphan windows */
	gtk_signal_connect (GTK_OBJECT (single), "new_window_orphan",
			    GTK_SIGNAL_FUNC (new_window_orphan_cb),  NULL);
}

/**
 * embed_wrapper_init: call it after the first page is loaded
 */
void 
embed_wrapper_init (GaleonEmbed *embed)
{
	gboolean event_listener = gnome_config_get_bool(CONF_MOUSE_LINKS_DRAG);
	embed->wrapper = mozilla_wrapper_init (embed, event_listener);
}

/**
 * embed_progress_clear: clear all infos about download progress
 */
void
embed_progress_clear(GaleonEmbed *embed)
{
	/* set all progress values to 0 */ 
	embed->loadPercent = 0;
	embed->bytesLoaded = 0;
	embed->maxBytesLoaded = 0;
}

/**
 * embed_save_image: save an image from an embed
 */
void 
embed_save_image (GaleonEmbed *embed, gchar *url, gboolean ask)
{
	save_url (embed, url, CONF_STATE_SAVE_IMAGE_DIR, ask);
}

/**
 * embed_set_image_as_background: set an image as the desktop background
 */
void embed_set_image_as_background (GaleonEmbed *embed, gchar *url)
{
	gchar *file, *path;

	/* get the filename and check */
	file = g_basename (url);
	g_return_if_fail (file != NULL);
	g_return_if_fail (strlen (file) != 0);

	/* build a path to save it in */
	path = g_strconcat (g_get_home_dir (), "/.galeon/", file, NULL);

	/* save the image */
	mozilla_save_url (embed, url, path, ACTION_SETBACKGROUND);

	/* free */
	g_free (path);
}

/**
 * embed_save_document: save this document (or frame)
 */
void embed_save_document (GaleonEmbed *embed, gboolean main)
{
	gchar *target = NULL;

	/* get either frame or main document */
	if (main)
	{
		target = mozilla_get_main_document_url (embed);
	}
	else 
	{
		target = mozilla_get_document_url (embed);
	}

	/* save it */
	save_url (embed, target,  CONF_STATE_SAVE_DIR, TRUE);

	/* free */
	g_free (target);
}

/**
 * save_url: save a given URL
 */
static void 
save_url (GaleonEmbed *embed, gchar *target, 
	  gchar *default_dir_pref, gboolean ask)
{
	GnomeVFSURI *uri;
	gchar *retPath = NULL;
	gchar *fileName = NULL;
	gchar *dirName;
	gchar *retDir;
	gboolean ret;

	/* Get a filename from the target url */
	uri = gnome_vfs_uri_new (target);
	if (uri)
	{
		fileName = gnome_vfs_uri_extract_short_name (uri);
		gnome_vfs_uri_unref (uri);
	}

	dirName = gnome_config_get_string (default_dir_pref);

	if (ask)
	{
		ret = show_file_picker (NULL, N_("Select the file to save"),
					dirName, fileName, modeSave, &retPath);                
	}
	else
	{
		retPath = g_strconcat (dirName, fileName, NULL);
		ret = TRUE;
	}

	if (ret == TRUE)
	{
		uri = gnome_vfs_uri_new (retPath);
		if (uri)
		{
			retDir = gnome_vfs_uri_extract_dirname (uri);
			mozilla_save_url (embed, target, 
					  retPath, ACTION_NONE);
			save_url_message (embed,retPath);

			/* set default save dir */
			gnome_config_set_string (default_dir_pref, 
						 retDir);
			gnome_config_sync ();

			g_free (retDir);
			gnome_vfs_uri_unref (uri);
		}
	}

	g_free (dirName);
	g_free (fileName);
	g_free (retPath);
}

static 
void save_url_message (GaleonEmbed *embed, char *path)
{
	char *temp_message;
	temp_message = g_strdup_printf(_("Saved as %s"), path);
	window_update_temp_message (embed->parent_window,
				    temp_message);
	g_free (temp_message);
}

/**
 * embed_view_source: view web page source 
 */
void embed_view_source (GaleonEmbed *embed, gboolean main, 
			gboolean new_window)
{
	gchar *url;
	
	if (main) {
		url = mozilla_get_main_document_url (embed);
	} else {
		url = mozilla_get_document_url (embed);
	}
	g_return_if_fail (url);

	if (gnome_config_get_bool (CONF_HANDLERS_USE_EXTERNAL_SOURCE_VIEWER))
	{
		embed_view_source_external (embed, url);
	}
	else
	{
		embed_create_from_url_view_source (embed, url, new_window);
	}
	g_free (url);
}

/**
 * embed_view_source_external: view web page source with an external viewer 
 */
static void embed_view_source_external (GaleonEmbed *embed, const gchar *url)
{
	/* FIXME is use /tmp ok ? */
	char *filename = g_strdup("/tmp/galeon-viewsource-XXXXXX");
	char *htmlfile;
	int result;

	/* get a name for the temporary file */
	result = mkstemp (filename);

	if (result == -1)
	{
		gnome_error_dialog ( _("Could not create a "
				       "temporary file"));
		return;
	}
	close (result);

	htmlfile = g_strconcat (filename, ".html", NULL);
	rename (filename, htmlfile);
	mozilla_save_url (embed, url, htmlfile, ACTION_VIEWSOURCE);

	g_free (htmlfile);
	g_free (filename);
}

/**
 * embed_close: close a GaleonEmbed
 */
void
embed_close (GaleonEmbed *embed)
{
	GaleonWindow *window;

	/* already closing */
	if (embed->is_closing)
	{
		return;
	}
	embed->is_closing = TRUE;

	/* check args */
	window = embed->parent_window;
	return_if_not_window (window);
	
	/* select the new page before destroying the current one */
	embed_select_new_page (embed);

	/* stop any pending loads */
	gtk_moz_embed_stop_load (GTK_MOZ_EMBED (embed->mozEmbed));

	/* destroy the embedding widget -- this will also destroy 
	 * the notebook tab and its label, and remove it from 
	 * the relevant lists */
	gtk_widget_destroy (GTK_WIDGET (embed->mozEmbed));
}

/**
 * embed_open_frame: open the frame pointed by the event target 
 */
void 
embed_open_frame (GaleonEmbed *embed, gboolean same_embed, gboolean new_window)
{
	gchar *url;
	
	/* get document URL */
	url = mozilla_get_document_url (embed);
	if (url == NULL)
	{
		return;
	}

	/* load it */
	if (same_embed)
	{
		embed_load_url (embed, url);
	} 
	else 
	{
		embed_create_from_url (embed, url, TRUE, new_window);
	}

	/* free */
	g_free (url);
}

/*
 * embed_reload: call gtk_moz_embed_reload.
 */
void
embed_reload (GaleonEmbed *embed)
{
	gtk_moz_embed_reload (GTK_MOZ_EMBED (embed->mozEmbed), 
			      GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

/**
 * embed_update_page_location: called if the page location changes, or to
 * bring the GaleonEmbed.site_location field in sync with the currently 
 * viewed page
 */
void
embed_update_page_location (GaleonEmbed *embed)
{
	const PixmapData *drag_pixmap;
	GaleonWindow *window;
	gchar *new_location;

	/* check we're currently being viewed */
	if (!embed->is_active)
		return;

	/* get the parent window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the location string */
	new_location = window->active_embed->site_location;

	/* clear text */
	window_clear_url_entry (window);
	if (new_location != NULL && strcmp (new_location, "about:blank") != 0)
	{
		/* change the url entry text */
		window_set_url_entry (window, new_location);
	
		/* update the drag location pixmap */
		if (gnome_config_get_bool (CONF_GENERAL_FAVICONS_ENABLED) &&
		    embed->parent_window->toolBarOn && 
		    window->drag_pixmap != NULL)
		{
			drag_pixmap = bookmarks_get_siteicon (new_location);
			gtk_pixmap_set (GTK_PIXMAP (window->drag_pixmap), 
					drag_pixmap->pixmap, 
					drag_pixmap->mask);
		}
	}
}

/**
 * embed_update_page_title: called if the page title changes, or to bring
 * the main window title in sync with the currently viewed page
 */
void
embed_update_page_title (GaleonEmbed *embed)
{
	gchar *full_title;
	gchar *title_string;
	GaleonWindow *window;

	/* set notebook label (although this might not be visible) */
	embed_set_notebook_label (embed);

	/* if this page isn't being viewed, get out now */
	if (!embed->is_active)
	{
		return;
	}

	/* get the window */
	window = embed->parent_window;
	return_if_not_window (window);

	/* get the format string */
	title_string = gnome_config_get_string (CONF_APPEARANCE_WINDOWS_TITLE);

	/* format the full title */
	full_title = g_strdup_printf (title_string, embed->site_title);

	/* set the toplevel window title to the document title */
	gtk_window_set_title (GTK_WINDOW (window->WMain), full_title);

	/* free allocated strings */
	if (full_title) g_free (full_title);
	if (title_string) g_free (title_string);  
}

/**
 * embed_set_notebook_label: sets the notebook tab label belonging to embed
 * to the string contained in text.
 */
void
embed_set_notebook_label (GaleonEmbed *embed)
{
	GaleonWindow *window;
	gchar *shortened;
	GtkWidget *tab;
	gint length;
	gint style;

	/* get the parent window */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

        /* shorten notebook's label */
	tab = embed->notebook_event_box;
	style = gnome_config_get_int (CONF_APPEARANCE_TABBED_SHORTEN_STYLE);
        switch (style)
        {
        case 0:
		/* abbreviate the text label */
                gtk_widget_set_usize (GTK_WIDGET (tab), -1, -1);
                length = gnome_config_get_int 
			(CONF_APPEARANCE_TABBED_SHORTEN_CHARS);   
                shortened = shorten_name (embed->site_title, length);
                break;

        case 1: 
		/* set fixed size of the label widget */
		length = gnome_config_get_int 
			(CONF_APPEARANCE_TABBED_SHORTEN_POINTS);
                gtk_widget_set_usize (GTK_WIDGET (tab), length, -2);
               	shortened = embed->site_title;
                break;

        default:
                g_assert_not_reached ();
                return;
        }

	/* if it's different than the new shortened text, change it */
	if (strcmp (GTK_LABEL (embed->notebook_label)->label, shortened) != 0)
	{
		gtk_label_set_text (GTK_LABEL (embed->notebook_label),
				    shortened);
	}

	/* the menu text is the full text */
	gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (window->notebook),
					  GTK_WIDGET (embed->mozEmbed),
					  embed->site_title);
	gtk_widget_queue_resize (window->notebook);

	/* free allocated strings */
	if (style == 0)
	{
		g_free (shortened);
	}
}

/**
 * embed_set_notebook_label_status: sets the status for the notebook tab
 * label belonging to embed.
 */
void
embed_set_notebook_label_status (GaleonEmbed *embed)
{
	GtkWidget *label;

	return_if_not_embed (embed);

	/* get the label */
	label = GTK_WIDGET (embed->notebook_label);
	g_return_if_fail (GTK_IS_LABEL (label));

	/* set the right colour */
	if (embed->load_started > 0)
	{
		/* loading... */
		if (embed->is_active)
		{
			/* active embed */
			gtk_widget_set_style (label,
				active_loading_text_style);
		}
		else
		{
			/* inactive */
			gtk_widget_set_style (label,
				inactive_loading_text_style);
		}
	}
	else if (!(embed->has_been_viewed))
	{
		/* loaded, with new content */
		gtk_widget_set_style (label, inactive_new_text_style);
	}
	else if (embed->is_active)
	{
		/* loaded, viewed */
		gtk_widget_set_rc_style (label);
	}
	else
	{
		/* loaded, seen, but currently not at the front */
		gtk_widget_set_style (label, inactive_text_style);
	}
}

/**
 * embed_notebook_tab_update_closebutton (internal function): tab, the
 * notebook tab belonging to embed, is updated (if necessary) to reflect
 * the user's preference in the display of a close button.  if tab is
 * null, it will be found using embed.
 */
void
embed_update_tab_closebutton (GaleonEmbed *embed)
{
	GaleonWindow *window;
	GtkWidget *button;
	GtkWidget *pixmap;
	gboolean show;

	/* check args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* check prefs */
	show = gnome_config_get_bool (CONF_APPEARANCE_TABBED_CLOSEBUTTON);

	/* check for case when this is being destroyed */
	if (!show && embed->notebook_close_button != NULL)
	{
		gtk_widget_destroy (embed->notebook_close_button);
		embed->notebook_close_button = NULL;
	}
	/* check for case where button is being created */
	else if (show && embed->notebook_close_button == NULL)
	{
		/* load the close pixmap if it's not already loaded */
		if (close_pix == NULL)
		{
			/* load it */
			close_pix = pixmap_data_new_from_file
				(SHARE_DIR "/small-close.xpm");

			/* cope with uninstalled case cleanly */
			if (close_pix == NULL)
			{
				close_pix = g_new0 (PixmapData, 1);
			}
		}

		/* build and setup the button */
		button = embed->notebook_close_button = gtk_button_new ();
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		
		/* put the close pixmap into the button */
		pixmap = gtk_pixmap_new (close_pix->pixmap, close_pix->mask);
		gtk_container_add (GTK_CONTAINER (button), pixmap);

		/* pack button into tab */
		gtk_box_pack_start (GTK_BOX (embed->notebook_hbox), button,
				    FALSE, FALSE, 0);
		gtk_widget_show_all (button);

		/* when the close button is clicked, close the embed */
		gtk_signal_connect (GTK_OBJECT (button), "clicked", 
				    embed_tab_close_clicked_cb, embed);
	}
}

/**
 * embed_go_up: go to the nth parent directory
 */
void
embed_go_up (GaleonEmbed *embed, gint levels, LinkState state)
{
	GnomeVFSURI *uri, *up_uri;
	gchar *location;

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return;
	}

	/* go upwards to find the nth level up */
	do
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;

		/* this can happen if Up is selected from the menu */
		if (uri == NULL)
		{
			return;
		}
	}
	while (levels--);

	/* get the location */
	location = gnome_vfs_uri_to_string (uri, 0);
	gnome_vfs_uri_unref (uri);

	/* visit it */
	embed_activate_link (embed, NULL, location, state);

	/* free */
	g_free (location);
}

/**
 * embed_can_go_up: test to see if we can go to a parent directory
 */
gboolean
embed_can_go_up (GaleonEmbed *embed)
{
	GnomeVFSURI *uri;
	gboolean result;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return FALSE;
	}

	/* use gnome-vfs to find parent */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return FALSE;
	}

	result = gnome_vfs_uri_has_parent (uri);

	gnome_vfs_uri_unref (uri);

	return result;
}

/**
 * embed_set_zoom: set the zoom level for a given embed
 */
void
embed_set_zoom (GaleonEmbed *embed, gint zoom)
{
	/* sanity check */
	if (zoom == 0)
	{
		g_warning ("ignoring request to set zoom to 0");
		return;
	}

	/* check we're not already set */
	if (zoom == embed->zoom || embed->wrapper == NULL)
	{
		return;
	}

	/* set in mozilla */
	mozilla_set_zoom (embed, (float)zoom / 100.0);
	embed->zoom_auto_set = FALSE;
	embed->zoom = zoom;

	/* set in window */
	window_update_zoom (embed->parent_window);
}

/** 
 * Creates the back history menu 
 */
GtkMenu *
embed_create_back_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history_get_all_titles (embed, &titles,
						     &count, &index))
	{
		return NULL;
	}

	for (i = index - 1, level = 0; i >= 0; i--, level++) 
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}

	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/**
 * Creates the forward history menu
 */
GtkMenu *
embed_create_forward_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history_get_all_titles (embed, &titles,
						     &count, &index))
	{
		return NULL;
	}	

	for (i = index + 1, level = 0; i < count; i++, level++)
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	free_string_array (titles, count);
	return GTK_MENU(menu);
}

/** 
 * Creates the multi-level up menu 
 */
GtkMenu *
embed_create_up_menu (GaleonEmbed *embed)
{
	GnomeVFSURI *uri, *up_uri;
	GtkWidget *menu, *item;
	gint level;

	/* check embed location is valid */
	if (embed->site_location == NULL || strlen (embed->site_location) == 0)
	{
		return NULL;
	}

	/* create a vfs entry for this level */
	uri = gnome_vfs_uri_new (embed->site_location);
	if (uri == NULL)
	{
		return NULL;
	}

	/* create the menu */
	menu = gtk_menu_new ();

	/* create each possible up entry */
	for (level = 0;; level++)
	{
		up_uri = gnome_vfs_uri_get_parent (uri);
		gnome_vfs_uri_unref (uri);
		uri = up_uri;
		
		/* get out of loop if no parent */
		if (uri == NULL)
		{
			break;
		}

		/* create the menu entry */
		item = new_num_accel_menu_item (level, uri->text, TRUE, menu);
		gtk_widget_show (GTK_WIDGET (item));
		gtk_object_set_user_data (GTK_OBJECT (item), embed);
		gtk_menu_append (GTK_MENU (menu), GTK_WIDGET (item));
		gtk_signal_connect (GTK_OBJECT (item), "activate",
				    up_menu_menuitem_activate_cb, 
				    GINT_TO_POINTER (level));
		gtk_signal_connect (GTK_OBJECT (item), "button_release_event",
				(GtkSignalFunc) 
					(up_menu_menuitem_button_release_cb),
				GINT_TO_POINTER (level));

		gtk_widget_lock_accelerators (item);
	}

	/* the menu is completed */
	return GTK_MENU (menu);
}

/**
 * embed_shift_tab: shifts embed by amount in the notebook
 */
void
embed_shift_tab (GaleonEmbed *embed, gint amount)
{
	GaleonWindow *window;
	GtkNotebook *notebook;
	gint page;

	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);
	notebook = GTK_NOTEBOOK (window->notebook);

	/* reorder the tab */
	page = gtk_notebook_page_num (notebook, GTK_WIDGET (embed->mozEmbed));
	gtk_notebook_reorder_child (notebook, GTK_WIDGET (embed->mozEmbed),
				    page + amount);

	/* and update the menuitems */
	window_update_tab_controls (window);
}

/* embed_move_tab - moves an embed.  dest_embed can be an embed in the
 * same window or in another window, or NULL if embed should be moved to a
 * new window (not yet implemented, since i don't have a way to easily test
 * it) -Dan */
void
embed_move_tab (GaleonEmbed *embed, GaleonEmbed *dest_embed)
{
	GaleonWindow *orig_window, *dest_window;
	GtkNotebook *orig_notebook, *dest_notebook;
	guint orig_page, dest_page;

	return_if_not_embed (embed);

	/* if we've only received a single embed, exit without doing
	 * anything */
	if (embed == dest_embed) return;

	/* check to see if we're moving to a new window */
	if (dest_embed == NULL)
	{
		dest_window = window_create (GTK_MOZ_EMBED_FLAG_ALLCHROME);
	}
	else
	{
		return_if_not_embed (dest_embed);
		dest_window = dest_embed->parent_window;
	}

	/* check args */
	orig_window = embed->parent_window;
	return_if_not_window (orig_window);
	return_if_not_window (dest_window);

	/* find where we're putting it */
	dest_notebook = GTK_NOTEBOOK (dest_window->notebook);
	if (dest_embed != NULL)
	{
		dest_page = gtk_notebook_page_num (dest_notebook,
						   dest_embed->mozEmbed);
	}
	else
	{
		dest_page = 0;
	}

	/* simple case -- both embeds are in the same window, just reorder
	 * the pages */
	if (orig_window == dest_window)
	{
		gtk_notebook_reorder_child (dest_notebook, embed->mozEmbed,
					    dest_page);
		window_update_tab_controls (orig_window);
	}
	/* otherwise, move the dropped embed to the other notebook */
	else
	{
		/* get original notebook */
		orig_notebook = GTK_NOTEBOOK (orig_window->notebook);
		orig_page = gtk_notebook_page_num (orig_notebook,
						   embed->mozEmbed);

		/* add a reference to the tab so it won't be destroyed when
		 * we remove the page (the embed has already been
		 * referenced when it was created, so it doesn't need
		 * another reference) */
		gtk_object_ref (GTK_OBJECT (embed->notebook_hbox));

		/* select the new page before moving the current one */
		embed_select_new_page (embed);

		/* remove the embed from its current page */
		orig_window->embed_list =
			g_list_remove (orig_window->embed_list, embed);
		if (embed->is_visible)
		{
			window_hide_one_embed (orig_window);
		}
		gtk_notebook_remove_page (orig_notebook, orig_page);
		window_update_tab_controls (orig_window);

		/* if that was the only embed in the window,
		 * delete the window */
		if (orig_window->embed_list == NULL)
		{
			window_close (orig_window);
		}

		/* add the embed to the new window's list */
		dest_window->embed_list =
			g_list_append (dest_window->embed_list, embed);
		if (embed->is_visible)
		{
			window_show_one_embed (dest_window);
		}
		embed->parent_window = dest_window;

		/* add it into the destination notebook */
		gtk_notebook_insert_page (dest_notebook, embed->mozEmbed, 
					  embed->notebook_hbox, dest_page);

		/* if the embed was dropped on top of the active embed,
		 * switch to the dropped embed's page */
		if (dest_embed != NULL && dest_embed->is_active)
		{
			gtk_notebook_set_page (dest_notebook, dest_page);
		}

		/* update tab state */
		embed_set_notebook_label_status (embed);

		/* and remove the tab's reference */
		gtk_object_unref (GTK_OBJECT (embed->notebook_hbox));
	}
}

/**
 * Creates a menu item with a numbered/lettered accel
 */
static GtkWidget *
new_num_accel_menu_item (gint num, gchar *origtext, gboolean lettersok, 
			 GtkWidget *menu)
{
	gchar *text = new_num_accel_str(num, origtext, lettersok);
	if (text == NULL)
		return gtk_menu_item_new_with_label (origtext);
	else
	{
		GtkWidget *item = gtk_menu_item_new_with_label ("");
		label_set_accel_text (text, GTK_BIN (item)->child, menu, item);
		g_free (text);
		return item;
	}
}

/**
 * Adds a menuitem to a back/forward history menu
 */
static void
add_bf_menu (GtkWidget *menu, char* label, int index, GaleonEmbed *embed, int level)
{
	GtkWidget *item = new_num_accel_menu_item (level, label, TRUE, menu);

	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), embed);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    history_menu_menuitem_activate_cb, 
			    GINT_TO_POINTER (index));
	gtk_signal_connect (GTK_OBJECT (item), "button_release_event",
			(GtkSignalFunc)	
				(history_menu_menuitem_button_release_cb),
			GINT_TO_POINTER (index));
	gtk_widget_lock_accelerators (item);
}


static void
embed_select_new_page (GaleonEmbed *embed)
{
	GaleonWindow *window;
	GaleonEmbed *new_embed;
	gboolean last_embed = FALSE;

	/* check args */
	return_if_not_embed (embed);
	window = embed->parent_window;
	return_if_not_window (window);

	/* check if this is the last embed in the window */
	last_embed = (g_list_length (window->embed_list) == 1);

	if (!last_embed && embed->is_active)
	{
		/* this logic implements the galeon Smart Tab Selection
		 * Technology (tm) as suggested by Matthew Aubury -- Nate */
		switch (embed->focus_type)
		{
		case FOCUS_ON_CREATE:
			/* 
			 * embed_list is in order of tabs most recently
			 * focused.  select the second element in list since
			 * the first still hasn't been destroyed
			 */
			new_embed = g_list_nth_data (window->embed_list, 1);
			embed_switch_to_page (new_embed);
			break;

		case FOCUS_ON_REQUEST:
			gtk_notebook_next_page 
				(GTK_NOTEBOOK (window->notebook));
			break;

		default:
			g_assert_not_reached ();
			break;
		}
	} 
}

