/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "session.h"
#include "embed.h"
#include "misc.h"
#include "mime.h"
#include "window.h"
#include "prefs.h"
#include "history.h"
#include "mozilla.h"

#include <gtkmozembed.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomevfs/gnome-vfs.h>

/*
 *  Session code 
 *
 *  Copyright (C) 2000 Matthew Aubury
 *
 *  This code implements the idea of "sessions", meaning the state of
 *  all the open browsers with respect to the URLs they're pointed at.
 *
 *  This should be useful in two ways:
 *    -- You need to quit X but have got lots of windows open
 *       with stuff you want to read, but you don't want to
 *       bookmark it all, just come back and see it later.
 *    -- By autosaving sessions we should offer a degree of 
 *       automatic crash recovery. This should mitigate some
 *       of the Mozilla/Galeon instability problems.
 *
 *  NOTE: this code won't do the right thing with frames yet. This is
 *  an ongoing problem with frames (they can't easily be referenced by
 *  URL) which is shared with bookmarks.
 *
 *  This code is quite derived from the history code,
 *  kudos to whoever did that...
 *
 *  At the moment we only do crash recovery saves, as I haven't yet
 *  gotten around to doing the dialogs for generic session saving.
 *
 *  -- MattA
 */

/*
 *  Hmm, I think we should really be using the gnome-client /
 *  gnome-session stuff to handle this properly. -- MattA 16/4/2001
 */

/* local function prototypes */
static xmlDocPtr session_load (const gchar *filename);
static void session_resume (xmlDocPtr doc, gboolean override_behaviour);
static void galeon_exit (void);
static void session_close_all_windows (void);

/* local variables */
static GtkWidget *dialog = NULL;

/** The global list of all GaleonWindow structures */
GList *all_windows = NULL;

/**
 * session_autosave: save to the "crash recovery" session file
 */
void
session_autosave (void)
{
	gchar *filename;
	gint autosave;

	/* check we're configured to autosave */
	autosave = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autosave == FALSE)
		return;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session_crashed.xml", NULL);

	/* save it out */
	session_save_to (filename);

	/* free allocated string */
	g_free (filename);
}

/**
 * session_autoresume: check to see if there's a crashed session and load it
 */
gint
session_autoresume (void)
{
	gchar *filename;
	xmlDocPtr session;
	AutoResume autoresume;
	gint crashed, saved, choice;

	/* see if there's a crashed or saved session to resume */
	crashed = gnome_config_get_int ("/galeon/General/crashed=0");
	saved = gnome_config_get_int ("/galeon/General/session_saved=0");
	gnome_config_set_int ("/galeon/General/session_saved", 0);
	gnome_config_sync ();
	if (crashed == FALSE && saved == FALSE)
	{
		return FALSE;
	}

	/* see if we're configured to autoresume */
	autoresume = gnome_config_get_int("/galeon/Advanced/crash_recovery=1");
	if (autoresume == RESUME_NEVER && !saved)
	{
		return FALSE;
	}

	/* see if there's a session file to resume from */
	if (crashed)
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_crashed.xml", NULL);
	}
	else
	{
		filename = g_strconcat (g_get_home_dir (),
					"/.galeon/session_saved.xml", NULL);
	}
	if (access (filename, F_OK) != 0)
	{
		g_free(filename);
		return FALSE;
	}
	
	/* load it */
	session = session_load (filename);
	g_free (filename);
	if (session == NULL)
	{
		return FALSE;
	}

	/* abort if no windows in session */
	if (session->root->childs == NULL)
	{
		xmlFreeDoc (session);
		return FALSE;
	}

	/* resume saved session */
	if (saved)
	{
		/* resume and free */
		session_resume (session, TRUE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* do the appropriate thing */
	switch (autoresume)
	{
	case RESUME_NEVER:
		/* never reached: this is here to stop a warning */
		g_assert_not_reached ();
		return FALSE;

	case RESUME_ASK:
		/* build question dialog */
		dialog = gnome_question_dialog_modal
			(_("Galeon appears to have crashed or "
			   "been killed last time it was run\n\n"
			   "Would you like to recover the state "
			   "of your last browsing session?"), NULL, NULL);

		/* run it */
		choice = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

		/* check response */
		switch (choice)
		{
		case -1: /* dialog was killed */
		case 1: /* opted not to recover */
			xmlFreeDoc (session);
			return FALSE;
			
		case 0: /* do recovery */
			break;
		}
		/* deliberate fall through! */

	case RESUME_ALWAYS:
		/* resume and free */
		session_resume (session, FALSE);
		xmlFreeDoc (session);
		return TRUE;
	}

	/* never reached */
	g_assert_not_reached ();
	return FALSE;
}

/**
 * session_save_to: record the session state to the give location
 */
void
session_save_to (const gchar *filename)
{
	GList *w, *e;
	xmlNodePtr root_node;
	xmlNodePtr window_node;
	xmlNodePtr embed_node;
	xmlDocPtr doc;
	gint x = 0, y = 0, width = 0, height = 0, depth = 0;
	gchar buffer[32];

	/* version doesn't really make sense, but... */
        doc = xmlNewDoc ("2.0");

	/* create and set the root node for the session */
        root_node = xmlNewDocNode (doc, NULL, "session", NULL);
        xmlDocSetRootElement (doc, root_node);

	/* iterate through all the windows */
	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		/* get this item */
		GaleonWindow *window = (GaleonWindow *)(w->data);
		if (window->magic != GALEON_WINDOW_MAGIC)
		{
			g_warning ("not a valid galeon window");
			continue;
		}

		/* skip if for some reason it doesn't contain any embeds */
		if (window->embed_list == NULL)
		{
			g_warning ("window with no GaleonEmbeds");
			continue;
		}

		/* make a new XML node */
		window_node = xmlNewDocNode (doc, NULL, "window", NULL);

		/* get window geometry */
		gdk_window_get_geometry (GTK_WIDGET (window->WMain)->window,
					 &x, &y, &width, &height, &depth);
		gdk_window_get_position (GTK_WIDGET (window->WMain)->window,
					 &x, &y);

		/* set window properties */
		snprintf(buffer, 32, "%d", x);
		xmlSetProp (window_node, "x", buffer);
		snprintf(buffer, 32, "%d", y);
		xmlSetProp (window_node, "y", buffer);
		snprintf(buffer, 32, "%d", width);
		xmlSetProp (window_node, "width", buffer);
		snprintf(buffer, 32, "%d", height);
		xmlSetProp (window_node, "height", buffer);

		/* iterate through all the embeds */
		for (e = g_list_last (window->embed_list);
		     e != NULL; e = g_list_previous (e))
		{
			/* get this item */
			GaleonEmbed *embed = (GaleonEmbed *)(e->data);

			/* skip if it's a XUL dialog */
			if (embed->is_chrome)
			{
				continue;
			}

			/* skip if the wrapper is NULL, can't do anything
			 * with such a crippled embed */
			if (embed->wrapper == NULL)
			{
				g_warning ("GaleonEmbed with no GaleonWrapper");
				continue;
			}

			if (embed->site_location != NULL &&
			    strlen (embed->site_location) > 0)
			{
			/* make a new XML node */
			embed_node = xmlNewDocNode (doc, NULL, "embed", NULL);

			/* store url and title in the node */
			xmlSetProp (embed_node, "url", embed->site_location);
			xmlSetProp (embed_node, "title", 
				    embed->site_title_utf8);

			/* insert node into the tree */
			xmlAddChild (window_node, embed_node);
			}
		}

		/* add window into tree */
		if (g_list_length (window->embed_list) != 0)
		{
			xmlAddChild (root_node, window_node);
		}
	}

	/* save it all out to disk */
        xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);
}

/**
 * session_load: load a session into a new XML document
 */
static xmlDocPtr
session_load (const gchar *filename)
{
	xmlDocPtr doc;

	/* read the session file */
        doc = xmlParseFile (filename);
        if (doc == NULL) 
	{
		g_warning("unable to parse session file `%s'", filename);
		/* NOTE: fall through to return NULL */
	}

	return doc;
}

/**
 * session_resume: open browsers to resume the session state
 */
static void
session_resume (xmlDocPtr doc, gboolean override_behaviour)
{
	xmlNodePtr window_node;
	xmlNodePtr embed_node;
	gint behaviour;
	gchar *url;
	gchar *title;
	BookmarkItem *cat = NULL; 
	BookmarksEditorControls *controls;
	gint x = 0, y = 0, width = 0, height = 0;
	GdkWindow *gdk_window;
	GaleonWindow *window = NULL;
	GaleonEmbed *embed;
	gint embed_count = 0;

	/* so we don't open saved sessions into temporary bookmarks) */
	if (override_behaviour)
	{
		behaviour = 0;
	}
	else
	{
		/* decide whether to put into bookmarks or windows */
		behaviour = gnome_config_get_int
			(CONF_ADVANCED_CRASH_BEHAVIOUR);
	}
	
	if (behaviour == 1)
	{
		cat = add_bookmark_default (BM_FOLDER, 
					    _("URLs opened before crashing"),
	                                    NULL, NULL);
	}

	/* iterate over session windows */
        for (window_node = doc->root->childs; window_node != NULL;
		window_node = window_node->next)
	{
		if (window_node->childs == NULL)
		{
			g_warning ("window with no GaleonEmbeds");
			continue;
		}

		if (behaviour != 1)
		{
			/* create a default window */
			window = window_create (GTK_MOZ_EMBED_FLAG_ALLCHROME);

			/* get the size of the window */
			x      = xmlGetIntProp (window_node, "x");
			y      = xmlGetIntProp (window_node, "y");
			width  = xmlGetIntProp (window_node, "width");
			height = xmlGetIntProp (window_node, "height");
			
			/* set the size and position of the window */
			gtk_window_set_default_size  
				(GTK_WINDOW (window->WMain), width, height);
			gtk_widget_set_uposition (GTK_WIDGET (window->WMain),
						  x, y);
			window->set_size = TRUE;
		}
		
		/* iterate over session embeds */
		for (embed_node = window_node->childs; embed_node != NULL;
		     embed_node = embed_node->next)
		{
			/* get the properties */
			url = xmlGetProp (embed_node, "url");
			title = xmlGetPropLocale (embed_node, "title");
			
			if (behaviour == 1)
			{
				/* add the bookmark */
				add_bookmark_default (BM_SITE, title, 
						      url, cat);
			}
			else
			{
				/* open a browser pointing to this URL */
				embed = embed_create_in_window (window);
				embed_set_visibility (embed, TRUE);
				embed_load_url (embed, url);
				embed_count++;
			}

			/* free allocated strings */
			if (url != NULL) xmlFree (url);
			if (title != NULL) xmlFree (title);
		}

		if (behaviour != 1)
		{
			gdk_window = GTK_WIDGET (window->WMain)->window;
			gdk_window_move (gdk_window, x, y);
		}
	}
 
	if (embed_count == 0)
	{
		g_warning ("having to create default window");
		embed_create_default (NULL, FALSE);
	}

	/* Add an empty window bookmark to the temp bookmarks window, just
	   in case all the saved bookmarks are crashers */
	if (behaviour == 1)
	{
		controls = bookmarks_editor_show_dock (all_windows->data);
		bookmarks_editor_select_bookmark (controls, cat);
	}
}

void 
session_add_window (GaleonWindow *window)
{
	/* add it to the list of windows */
	all_windows = g_list_prepend (all_windows, window);
}

void 
session_remove_window (GaleonWindow *window)
{
	/* remove window from list */
	all_windows = g_list_remove (all_windows, window);

	/* quit if this was the last window to go and
	 * we're not in server mode */
	if (g_list_length (all_windows) == 0 && !galeon_server_mode)
	{
		galeon_exit ();
	}
}

void
session_save (void)
{
	gchar *filename;
	
	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (), 
				"/.galeon/session_saved.xml", NULL);

	/* save it out */
	session_save_to (filename);
	
	/* set config marker */
	gnome_config_set_int ("/galeon/General/session_saved", 1);
	gnome_config_sync ();
	
	g_free (filename);
}

void 
session_quit (gboolean auto_save)
{
	/* save the session, if it was requested or if the user has the
	 * "always save" option enabled */
	if (auto_save ||
	    gnome_config_get_bool (CONF_GENERAL_ALWAYS_SAVE_SESSION))

	{
		session_save ();
	}

	if (all_windows != NULL)
	{
		/* close all windows, this should quit */
		session_close_all_windows ();
	}
	else
	{
		/* quit, even if there was no window opened */
		galeon_exit ();
	}
}

/* argh, this is stupid */
void
session_quit_cb (gpointer auto_save_ptr)
{
	session_quit (GPOINTER_TO_INT (auto_save_ptr));
}

static
void galeon_exit (void)
{

	gnome_config_set_int("/galeon/General/crashed", FALSE);

	/* FIXME: uncatch GDK errors which happen on theme changes */
	gdk_error_trap_pop ();

	/* shut down galeon subsystems */
	preferences_save ();
	mime_db_shutdown ();
	bookmarks_exit ();
	history_exit ();
	mozilla_save_prefs ();

	/* check... */
	g_assert (g_list_length (all_embeds) == 0);
	g_assert (g_list_length (all_windows) == 0);

	/* shut down GNOME subsystmes */
	gnome_vfs_shutdown ();

	/* absolutely no more mozembeds */
	gtk_moz_embed_pop_startup ();

	gtk_main_quit();
}

static void
session_close_all_windows (void)
{
	GList *copy;

	/* force ourselves out of server mode -- this means that closing
	 * the last window really will quit out of galeon */
	galeon_server_mode = FALSE;

	/* close all windows: this in turn will close all embeds */
	/* at the closing of the last window galeon_exit will be called */
	copy = g_list_copy (all_windows);
	g_list_foreach (copy, (GFunc)window_close, NULL);
	g_list_free (copy);

	/* wait for pending destruction events */
	while (gtk_events_pending ())
	{
		gtk_main_iteration ();
	}

	/* check we've been successful in shutting everything down */
	if (g_list_length (all_windows) != 0)
	{
		/* FIXME: I think this bug is fixed now, remove this code in
		 * some later release and replace with more brutal assets */
		/* this should never happen, but apparently can do */
		g_warning ("shutdown wasn't successful! still %d window(s) "
			   "and %d embed(s)", g_list_length (all_windows),
			   g_list_length (all_embeds));
		all_windows = NULL;
		all_embeds = NULL;

		/* force death */
		galeon_exit ();
	}
}

/**
 * Restore a session from a file
 * Closes all current windows and opens the ones saved in the file
 */
void
session_load_from (const gchar *filename)
{
	xmlDocPtr session = session_load (filename);
	if (session != NULL) 
	{
		GList *copy;
		gboolean old_server_mode;
		/* Switch to serveer mode to avoid exiting from galeon
		 * when all windows are closed. This is a bit hacky... 
		 */
		old_server_mode = galeon_server_mode;
		galeon_server_mode = TRUE;
		/* close all windows */
		copy = g_list_copy (all_windows);
		g_list_foreach (copy, (GFunc) window_close, NULL);
		g_list_free (copy);
		/* reopen windows... */
		session_resume (session, TRUE);
		xmlFreeDoc (session);
		/* restore server mode */
		galeon_server_mode = old_server_mode;
	} 
	else
	{
		gchar *s = g_strdup_printf 
			(_("Couldn't load session from %s"), filename);
		GtkWidget *d = gnome_error_dialog (s);
		g_free (s);
		gnome_dialog_run_and_close (GNOME_DIALOG (d));
		gtk_widget_destroy (d);
	}
}

/**
 * gets the list of recently used sessions
 */
GList *
session_history_get (void)
{
	GList *ret = NULL;
	gint i, count;
	gnome_config_push_prefix ("/galeon/SessionHistory/");
	count = gnome_config_get_int ("count=0");
	for (i = count; i > 0; i--) 
	{
		gchar *key = g_strdup_printf ("File%d", i);
		gchar *file = gnome_config_get_string (key);
		if (file != NULL)
			ret = g_list_prepend (ret, file);
		g_free (key);
	}
	gnome_config_pop_prefix ();
	return ret;
}

/**
 * modifies the list of recently used sessions, addid the filename at the top
 */
void
session_history_add (const gchar *filename)
{
	gint max, count;
	GList *li;
	GList *current_list = session_history_get ();
	/* remove the file if already present */
	for (li = current_list; li != NULL; li = li->next) 
	{
		if (!strcmp (li->data, filename))
		{
			g_free (li->data);
			current_list = g_list_remove_link (current_list, li);
			break;
		}		
	}
	/* prepend the new item */
	current_list = g_list_prepend (current_list, g_strdup (filename));
	/* save the new list */
	gnome_config_push_prefix ("/galeon/SessionHistory/");
	max = gnome_config_get_int ("maxItems=4");
	count = 0;
	for (li = current_list; li != NULL; li = li->next) 
	{
		gchar *key = g_strdup_printf ("File%d", ++count);
		gchar *file = li->data;
		gnome_config_set_string (key, file);
		g_free (file);
		g_free (key);
		if (count == max) break;
	}
	gnome_config_set_int ("count", count);
	gnome_config_pop_prefix ();
	gnome_config_sync ();
	g_list_free (current_list);
	/* update the GUI */
	window_session_history_update_all ();
}

void
session_history_load_index (gint index)
{
	gint i = 1;
	GList *l = session_history_get ();
	GList *li;
	for (li = l; li != NULL; li = li->next)
	{
		if (i == index) 
		{
			gchar *fname = li->data;
			session_load_from (fname);
			/* put the item at the top*/
			session_history_add (fname);
			break;
		}
		i++;
	}
	for (li = l; li != NULL; li = li->next)
	{
		g_free (li->data);
	}
	g_list_free (l);
}
