/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "toolbar.h"
#include "spinner.h"
#include "misc.h"
#include "window.h"
#include "bookmarks.h"

#include <string.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkdnd.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-dock.h>
#include <libgnomeui/gnome-entry.h>

/* GNOME includes */
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

/* local function prototypes */
static void toolbar_update (GaleonWindow *window);
static void toolbar_create_widgets (GaleonWindow *window, 
				    GtkWidget *hbox, GtkWidget *toolbar,
				    gboolean location_on_main_toolbar);
static void toolbar_create_location_toolbar (GaleonWindow *window);
static void toolbar_create_location_widgets (GaleonWindow *window, 
					     GtkWidget *hbox, 
					     GtkWidget *toolbar);
static void toolbar_create_zoom_spin (GaleonWindow *window, 
				      GtkWidget *toolbar);
static void toolbar_create_drag_handle (GaleonWindow *window, 
					GtkWidget *toolbar);
static void toolbar_create_button (GaleonWindow *window, GtkWidget *toolbar,
				   ToolbarItemType type);
static GtkWidget *toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
				      gchar *name, gint band_num);
static void toolbar_set_style (GtkWidget *toolbar, GtkToolbarStyle final_style,
			       GtkReliefStyle relief);
static gchar *toolbar_get_theme_path (void);
static GtkWidget *toolbar_get_theme_pixmap (const gchar *pixmap, 
					    gboolean is_nautilus);
static gboolean toolbar_theme_use_standard_icons (void);

/* theme settings -- setup once per toolbar creation */
static gchar *theme_path;
static gchar *theme_name;

/* path to Nautilus theme pref in config space */
#define NAUTILUS_THEME_PREF "/apps/nautilus/preferences/theme"

/* array of available toolbar items */
const ToolbarItem toolbar_items[TOOLBAR_ITEM_COUNT] =
{
	{
		N_("New"), 
		N_("Open new browser"), 
		GNOME_STOCK_PIXMAP_NEW, 
		"New",
		FALSE, /* priority? */
		FALSE,
		100,
		NULL, /* clicked */
		window_new_button_press_cb,
		window_new_button_release_cb,
		window_new_button_drag_data_received_cb
	},
	{
		N_("Back"), 
		N_("Go back - Right button: History"), 
		GNOME_STOCK_PIXMAP_BACK, 
		"Back",
		TRUE, /* priority? */
		FALSE,
		200,
		NULL, /* clicked */
		window_back_forward_button_press_cb,
		window_back_forward_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Up"), 
		N_("Go up a level from the current location"), 
		GNOME_STOCK_PIXMAP_UP, 
		"Up",
		FALSE, /* priority? */
		FALSE,
		300,
		NULL, /* clicked */
		window_up_button_press_event_cb,
		window_up_button_release_event_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Forward"), 
		N_("Go forward - Right button: History"), 
		GNOME_STOCK_PIXMAP_FORWARD,
		"Forward",
		FALSE, /* priority? */
		FALSE,
		400,
		NULL, /* clicked */
		window_back_forward_button_press_cb,
		window_back_forward_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Reload"), 
		N_("Display the latest content of the current page"), 
		GNOME_STOCK_PIXMAP_REFRESH, 
		"Refresh",
		FALSE, /* priority? */
		FALSE,
		700,
		window_refresh_button_clicked_cb,
		window_refresh_button_press_event_cb,
		NULL, /* release_event */
		NULL /* drag_data_received */
	},
	{
		N_("Home"), 
		N_("Go to your home page"), 
		GNOME_STOCK_PIXMAP_HOME, 
		"Home",
		TRUE, /* priority? */
		FALSE,
		500,
		NULL, /* clicked */
		window_home_button_press_event_cb,
		window_home_button_release_event_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Stop"), 
		N_("Stop current data transfer"), 
		GNOME_STOCK_PIXMAP_STOP, 
		"Stop",
		TRUE, /* priority? */
		FALSE,
		600,
		window_stop_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL /* release_event */,
		NULL /* drag_data_received */
	},
	{
		N_("Go"), 
		N_("Go to specified location"), 
		GNOME_STOCK_PIXMAP_JUMP_TO, 
		"Go",
		FALSE, /* priority? */
		FALSE,
		1400,
		NULL /* clicked */,
		window_generic_button_press_event_cb,
		window_go_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Zoom control"), 
		NULL, 
		NULL, /* pixmap */ 
		NULL,
		FALSE, /* priority? */
		FALSE,
		1100,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Drag handle"), 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1200,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Location entry"),
		NULL,
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1300,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Spinner"), 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1500,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		"------", 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1600,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Find"), 
		N_("Find in page"), 
		GNOME_STOCK_PIXMAP_SEARCH, 
		"Search",
		FALSE, /* priority? */
		FALSE,
		800,
		window_find_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Bookmarks"), 
		N_("Toggle Bookmarks Sidebar"), 
		GNOME_STOCK_PIXMAP_BOOK_OPEN, 
		NULL,
		FALSE, /* priority? */
		TRUE,
		1000,
		window_bookmarks_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("History"), 
		N_("Toggle History Sidebar"), 
		GNOME_STOCK_PIXMAP_INDEX, 
		NULL,
		FALSE, /* priority? */
		TRUE,
		900,
		window_history_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Print"), 
		N_("Print Current Page"), 
		GNOME_STOCK_PIXMAP_PRINT, 
		NULL,
		FALSE, /* priority? */
		FALSE,
		550,
		window_print_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	}
};

/**
 * toolbar_create: create main toolbar
 */
void
toolbar_create (GaleonWindow *window)
{
	gboolean location_on_main_toolbar;
	ToolbarIcons toolbar_icons;
	GtkWidget *hbox, *toolbar;

	/* which icons to use */
	toolbar_icons = gnome_config_get_int (CONF_APPEARANCE_TOOLBAR_ICONS);
	if (toolbar_icons == TOOLBAR_ICONS_NAUTILUS)
	{
		/* get path or fail */
		theme_path = toolbar_get_theme_path ();
		if (theme_path == NULL || toolbar_theme_use_standard_icons ())
		{
			toolbar_icons = TOOLBAR_ICONS_NORMAL;
		}
	}

	/* read prefs */
	location_on_main_toolbar = gnome_config_get_bool 
		(CONF_APPEARANCE_URL_LOCATION);

	/* build the overall box to pack the toolbar into */
	hbox = gtk_hbox_new (FALSE, 2);

	/* dock the box into the window */
	window->main_dockitem = 
		toolbar_add_docked (window, hbox, GNOME_APP_TOOLBAR_NAME, 1);

	/* create items toolbar */
	window->main_toolbar = toolbar =
		gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);

	/* pack it into the box */
	gtk_box_pack_start (GTK_BOX (hbox), toolbar, FALSE, FALSE, 0);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (window->main_dockitem), 
			    "button-press-event", GTK_SIGNAL_FUNC
			    (window_generic_button_press_event_cb), window);

	/* create all the buttons and widgets */
	toolbar_create_widgets (window, hbox, toolbar, 
				location_on_main_toolbar);

	/* show the box and all its contents */
	gtk_widget_show_all (window->main_dockitem);

	/* create location toolbar, if necessary */
	if (!location_on_main_toolbar)
	{
		/* create it's own toolbar */
		toolbar_create_location_toolbar (window);
	}

	/* update the toolbar appearance */
	toolbar_update (window);

	/* free theme info */
	if (toolbar_icons == TOOLBAR_ICONS_NAUTILUS)
	{
		g_free (theme_path);
		g_free (theme_name);
	}
}

/**
 * toolbar_recreate: recreate main toolbar (e.g. on prefs apply)
 */
void
toolbar_recreate (GaleonWindow *window)
{
	gchar *text = NULL;

	/* get current text in url entry if any */
	/* NB: this MUST be strdup'd here since it's only a reference
	 * and we're about to destroy the toolbar (D'Oh! -- MattA) */
	if (window->toolBarOn)
	{
		text = g_strdup (gtk_entry_get_text 
				 (GTK_ENTRY (window->location_entry)));

		/* also store history */
		gnome_entry_save_history 
			(GNOME_ENTRY (window->location_gnomeentry));
	}

	/* destroy spinner, this prevents crashes with timeouts */
	spinner_destroy (window);

	/* destroy main toolbar */
	if (window->main_dockitem != NULL)
	{
		gtk_widget_destroy (window->main_dockitem->parent);
		window->main_dockitem = NULL;
		window->main_toolbar = NULL;
		window->right_toolbar = NULL;
	}

	/* destroy location toolbar */
	if (window->location_dockitem != NULL)
	{
		gtk_widget_destroy (window->location_dockitem->parent);
		window->location_dockitem = NULL;
		window->location_toolbar = NULL;
	}

	/* zero out the toolbar widgets, these should have already
	 * been destroyed by the destruction of the toolbars */
	window->new_button          = NULL;
	window->back_button         = NULL;
	window->up_button           = NULL;
	window->forward_button      = NULL;
	window->refresh_button      = NULL;
	window->home_button         = NULL;
	window->stop_button         = NULL;
	window->go_button           = NULL;
	window->find_button         = NULL;
	window->zoom_spin           = NULL;
	window->drag_event_box      = NULL;
	window->drag_pixmap         = NULL;
	window->location_gnomeentry = NULL;
	window->location_entry      = NULL;
	window->history_button      = NULL;
	window->bookmarks_button    = NULL;

	/* create new toolbar */
	toolbar_create (window);
	
	/* set the controls properly for this window */
	window_update_nav_controls (window);

	/* restore location entry text and history */
	if (window->toolBarOn)
	{
		gnome_entry_load_history  
			(GNOME_ENTRY (window->location_gnomeentry));
		if (text != NULL)
		{
			window_set_url_entry (window, text);
			g_free (text);
		}
	}

	/* set bookmarks and history togglebuttons */
	window_update_bm_and_hist_buttons (window,
		(window->dock_type == DOCK_BOOKMARKS),
		(window->dock_type == DOCK_HISTORY));
}

/**
 * toolbar_update: update the toolbar appearance
 */
static void
toolbar_update (GaleonWindow *window)
{
	ToolbarStyle style;
	GtkReliefStyle relief;
	GtkToolbarStyle final_style;
	gboolean toolbar_button_relief;

	/* skip if no toolbar */
	if (!(window->toolBarOn))
	{
		return;
	}

	/* get relief */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();
	relief = (toolbar_button_relief ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);

	/* get real style */
	style = gnome_config_get_int (CONF_APPEARANCE_TOOLBAR_STYLE);

	/* convert to GTK+ style for setting in buttons */
	final_style = GTK_TOOLBAR_BOTH; /* only to stop a warning */
	switch (style)
	{
	case TOOLBAR_STYLE_HORIZONTAL:
	case TOOLBAR_STYLE_TEXT_ONLY:
		final_style = GTK_TOOLBAR_TEXT;
		break;

	case TOOLBAR_STYLE_ICONS_ONLY:
		final_style = GTK_TOOLBAR_ICONS;
		break;

	case TOOLBAR_STYLE_VERTICAL:
		final_style = GTK_TOOLBAR_BOTH;
		break;
	}

	/* set toolbar styles */
	toolbar_set_style (window->main_toolbar,     final_style, relief);
	toolbar_set_style (window->right_toolbar,    final_style, relief);
	toolbar_set_style (window->location_toolbar, final_style, relief);
}

/**
 * toolbar_create_widgets: create the widgets on the main toolbar
 */
static void
toolbar_create_widgets (GaleonWindow *window, 
			GtkWidget *hbox, GtkWidget *toolbar,
			gboolean location_on_main_toolbar)
{
	GtkWidget *spinner;
	gint i, count;

	/* create defaults when no configuration exists */
	gnome_config_push_prefix ("/galeon/Toolbar/");
	count = gnome_config_get_int ("count=-1");
	if (count == -1)
	{
		/* default toolbar layout */
		count = 9;
		gnome_config_set_int ("count", count);
		gnome_config_set_int ("item0", TOOLBAR_ITEM_BACK);
		gnome_config_set_int ("item1", TOOLBAR_ITEM_FORWARD);
		gnome_config_set_int ("item2", TOOLBAR_ITEM_REFRESH);
		gnome_config_set_int ("item3", TOOLBAR_ITEM_START);
		gnome_config_set_int ("item4", TOOLBAR_ITEM_STOP);
		gnome_config_set_int ("item5", TOOLBAR_ITEM_ZOOM_SPIN);
		gnome_config_set_int ("item6", TOOLBAR_ITEM_DRAG_HANDLE);
		gnome_config_set_int ("item7", TOOLBAR_ITEM_LOCATION_ENTRY);
		gnome_config_set_int ("item8", TOOLBAR_ITEM_SPINNER);
		gnome_config_sync ();
	}

	/* create all toolbar items on the main toolbar */
	for (i = 0; i < count; i++)
	{
		ToolbarItemType type;
		gchar *key;

		/* get toolbar item type */
		key = g_strdup_printf ("item%d=-1", i);
		type = gnome_config_get_int (key);
		g_assert (type != -1);
		g_free (key);
	
		/* check for illegal values */
		if (i < 0 || i >= TOOLBAR_ITEM_COUNT)
		{
			g_warning (_("illegal toolbar item type\n"));
			continue;
		}

		/* build the right thing */
		switch (type)
		{
		case TOOLBAR_ITEM_ZOOM_SPIN:
			/* create zoom spin entry */
			toolbar_create_zoom_spin (window, toolbar);
			break;

		case TOOLBAR_ITEM_DRAG_HANDLE:
			/* create drag handle if location widgets here */
			if (location_on_main_toolbar)
			{
				toolbar_create_drag_handle (window, toolbar);
			}
			break;

		case TOOLBAR_ITEM_LOCATION_ENTRY:
			/* create location widgets if on this toolbar */
			if (location_on_main_toolbar)
			{
				toolbar_create_location_widgets 
					(window, hbox, toolbar);

				/* use continuation toolbar */
				toolbar = window->right_toolbar;
			}
			break;

		case TOOLBAR_ITEM_SPINNER:
			/* create the spinner */
			spinner = spinner_create (window);
			
			/* if not the last item... */
			if (i != count - 1)
			{
				/* pack it normally */
				gtk_toolbar_append_element 
					(GTK_TOOLBAR (toolbar), 
					 GTK_TOOLBAR_CHILD_WIDGET,
					 GTK_WIDGET (spinner),
					 NULL, NULL, NULL, NULL, NULL, NULL);
			}
			else
			{
				/* otherwise use hbox to right justify */
				gtk_box_pack_end (GTK_BOX (hbox), spinner,
						  FALSE, FALSE, 0);
			}
			break;

		case TOOLBAR_ITEM_SEPARATOR:
			/* append some space */
			gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
			break;

		case TOOLBAR_ITEM_GO:
			/* create go button if location widgets here */
			if (location_on_main_toolbar)
			{
				toolbar_create_button (window, toolbar, type);
			}
			break;

		default:
			/* create a normal button */
			toolbar_create_button (window, toolbar, type);
			break;
		}
	}

	/* done parsing config */
	gnome_config_pop_prefix ();
}

/**
 * toolbar_create_drag_handle: create a "drag handle", where you can
 * drag the current URL from
 */
static void
toolbar_create_drag_handle (GaleonWindow *window, GtkWidget *toolbar)
{
	PixmapData *pixmap = NULL;
	GtkWidget *event_box;

	/* build suitable pixmap */
	pixmap = pixmap_data_new_from_file (SHARE_DIR "/i-bookmark.xpm");
	if (pixmap != NULL)
	{
		window->drag_pixmap = gtk_pixmap_new (pixmap->pixmap, 
						      pixmap->mask);
		g_free (pixmap);
	}

	/* build event box (for capturing drag signals */
	event_box = window->drag_event_box = gtk_event_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (event_box), 4);

	/* connect drag signals */
	gtk_signal_connect (GTK_OBJECT (event_box), "drag_data_get",
			    (window_drag_pixmap_drag_data_get_cb), window);
	gtk_drag_source_set (GTK_WIDGET (event_box), GDK_BUTTON1_MASK, 
			     url_drag_types, url_drag_types_num_items,
			     GDK_ACTION_COPY | GDK_ACTION_LINK);

	/* add pixmap into box */
	gtk_container_add (GTK_CONTAINER (event_box), window->drag_pixmap);

	/* add to toolbar */
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (event_box),
				    NULL, _("Drag page location from here"),
				    NULL, NULL, NULL, NULL);
}

/**
 * toolbar_create_location_widgets: create location related widgets
 */
static void
toolbar_create_location_widgets (GaleonWindow *window, 
				 GtkWidget *hbox, GtkWidget *toolbar) 
{
	GtkWidget *entry;
	GtkWidget *popwin;
	GtkWidget *gnomeentry;
	gboolean location_on_main_toolbar;

	/* read prefs */
	location_on_main_toolbar = gnome_config_get_bool 
		(CONF_APPEARANCE_URL_LOCATION);

	/* create drag handle if on separate toolbar */
	if (!location_on_main_toolbar)
	{
		toolbar_create_drag_handle (window, toolbar);
	}

	/* create location entry and access components */
	gnomeentry = gnome_entry_new ("LocationEntry");
	gnome_entry_load_history (GNOME_ENTRY (gnomeentry));
	entry = gnome_entry_gtk_entry (GNOME_ENTRY (gnomeentry));
	popwin = GTK_COMBO (gnomeentry)->popwin;
	window->location_gnomeentry = gnomeentry;
	window->location_entry = entry;

	/* connect the various signals */
	gtk_signal_connect (GTK_OBJECT (popwin), "show", GTK_SIGNAL_FUNC 
			    (window_location_gnomeentry_popwin_cb), window);
	gtk_signal_connect (GTK_OBJECT (popwin), "hide", GTK_SIGNAL_FUNC
			    (window_location_gnomeentry_popwin_cb), window);
	gtk_signal_connect_after (GTK_OBJECT (entry), "key_press_event", 
				  GTK_SIGNAL_FUNC 
				  (window_location_entry_key_press_cb), 
				  window);
	gtk_signal_connect (GTK_OBJECT (entry), "drag_data_received", 
			    GTK_SIGNAL_FUNC 
			    (window_location_entry_drag_data_received_cb), 
			    window);
	gtk_signal_connect_after (GTK_OBJECT (entry), "button_press_event", 
				  GTK_SIGNAL_FUNC 
				  (window_location_entry_button_press_cb), 
				  window);

	/* set url entry drop destination */
	gtk_drag_dest_set (gnomeentry, GTK_DEST_DEFAULT_ALL, drop_types, 
			   drop_types_num_items, GDK_ACTION_COPY);

	/* pack into toplevel box */
	gtk_box_pack_start (GTK_BOX (hbox), gnomeentry, TRUE, TRUE, 1);

	/* create continutation toolbar */
	window->right_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
						 GTK_TOOLBAR_BOTH);

        /* pack it in */
	gtk_box_pack_start (GTK_BOX (hbox), window->right_toolbar,
			    FALSE, FALSE, 0);

	/* add in Go button if on separate toolbar */
	if (!location_on_main_toolbar)
	{
		toolbar_create_button (window, window->right_toolbar,
				       TOOLBAR_ITEM_GO);
	}
}

/*
 * toolbar_create_location_toolbar: create location toolbar
 */
static void
toolbar_create_location_toolbar (GaleonWindow *window)
{
	GtkWidget *hbox;
	
	/* build the overall box to pack the toolbar into */
	hbox = gtk_hbox_new (FALSE, 2);

	/* dock the box into the window */
	window->location_dockitem = toolbar_add_docked (window, hbox, 
							"location", 2);

	/* create toolbar */
	window->location_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
						    GTK_TOOLBAR_BOTH);

	/* pack it into the box */
	gtk_box_pack_start (GTK_BOX (hbox), window->location_toolbar,
			    FALSE, FALSE, 0);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (window->location_dockitem), 
			    "button-press-event", GTK_SIGNAL_FUNC
			    (window_generic_button_press_event_cb), window);

	/* insert the location widgets */
	toolbar_create_location_widgets (window, hbox, 
					 window->location_toolbar);

	/* show everything */
	gtk_widget_show_all (window->location_dockitem);
}

/**
 * toolbar_create_zoom_spin: create zoom spin button
 */
static void
toolbar_create_zoom_spin (GaleonWindow *window, GtkWidget *toolbar)
{
	GtkObject *adj;

	/* create the adjustment */
	adj = gtk_adjustment_new (100, 1, 999, 10, 25, 10);

	/* build the spin button */
	window->zoom_spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);

	/* setup properties */
	gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (window->zoom_spin),
					 GTK_SHADOW_NONE);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed", 
			    window_zoom_spin_changed_cb, window);
	gtk_signal_connect (GTK_OBJECT (window->zoom_spin),
			    "key_press_event", GTK_SIGNAL_FUNC 
			    (window_zoom_spin_key_press_cb), window);
	gtk_signal_connect_after (GTK_OBJECT (window->zoom_spin),
				  "button_press_event", GTK_SIGNAL_FUNC 
				  (window_zoom_spin_button_press_cb), window);

	/* append to toolbar */
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (window->zoom_spin),
				    NULL, NULL, NULL, NULL, NULL, NULL);
}

/*
 * toolbar_button: create and append a button to a toolbar
 */
static void
toolbar_create_button (GaleonWindow *window, GtkWidget *toolbar, 
		       ToolbarItemType type)
{
	ToolbarIcons toolbar_icons;
	GtkWidget *button, *label, *box;
	GtkWidget *pixmap = NULL;
	const ToolbarItem *item;

	/* get item from array */
	item = &(toolbar_items[type]);
	
	/* which icons to use */
	toolbar_icons = gnome_config_get_int (CONF_APPEARANCE_TOOLBAR_ICONS);

	/* try getting large icon */
	if (toolbar_icons == TOOLBAR_ICONS_LARGE)
	{
		pixmap = toolbar_get_theme_pixmap (item->nautilus_icon, 
						   FALSE);
	}

	/* try getting nautilus button */
	if (toolbar_icons == TOOLBAR_ICONS_NAUTILUS)
	{
		pixmap = toolbar_get_theme_pixmap (item->nautilus_icon, 
						   TRUE);
	}

	/* use GNOME stock icon if not found or not requested */
	if (pixmap == NULL)
	{
		if (toolbar_icons == TOOLBAR_ICONS_LARGE)
		{ 
			pixmap = gnome_stock_pixmap_widget_at_size
				(window->WMain, item->stock_icon, 36, 36);
		}
		else
		{
			pixmap = gnome_stock_pixmap_widget (window->WMain,
							    item->stock_icon);
		}
	}

	/* make the toolbar button and add it to the toolbar */
	if (gnome_config_get_int (CONF_APPEARANCE_TOOLBAR_STYLE) ==
	    TOOLBAR_STYLE_HORIZONTAL)
	{
		/* we create a horizontal toolbar style ourselves */
		box = gtk_hbox_new (FALSE, 2);

		/* pack the pixmap first */
		gtk_box_pack_start (GTK_BOX (box), pixmap, FALSE, FALSE, 0);

		/* if priority text for this button, pack that too */
		if (item->priority)
		{
			label = gtk_label_new (_(item->label));
			gtk_box_pack_start (GTK_BOX (box), label, 
					    TRUE, TRUE, 0);
		}

		/* build button */
		if (item->togglebutton)
		{
			button = gtk_toggle_button_new ();
		}
		else
		{
			button = gtk_button_new ();
		}

		/* insert box into button */
		gtk_container_add (GTK_CONTAINER (button), box);

		/* append button to toolbar */
		gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), button,
					   _(item->tooltip), NULL);

		/* set it up */
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	}
	else
	{
		/* insert plain button */
		button = gtk_toolbar_append_element 
			(GTK_TOOLBAR (toolbar), 
			 (item->togglebutton ? 
			  GTK_TOOLBAR_CHILD_TOGGLEBUTTON : 
			  GTK_TOOLBAR_CHILD_BUTTON),
			 NULL, _(item->label), _(item->tooltip),
			 NULL, pixmap, NULL, NULL);
	}

	/* connect appropriate signals */
	if (item->clicked_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "clicked", 
				    item->clicked_cb, window);
	}
	if (item->button_press_event_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "button_press_event", 
				    GTK_SIGNAL_FUNC
				    (item->button_press_event_cb), window);
	}
	if (item->button_release_event_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button),
				    "button_release_event", GTK_SIGNAL_FUNC
				    (item->button_release_event_cb), window);
	}
	if (item->drag_data_received_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "drag_data_received", 
				    GTK_SIGNAL_FUNC
				    (item->drag_data_received_cb), window);

		/* set drop types */
		gtk_drag_dest_set (GTK_WIDGET (button), GTK_DEST_DEFAULT_ALL,
				   drop_types, drop_types_num_items, 
				   GDK_ACTION_COPY);
	}

	/* set the window->XXX_button appropiately */
	switch (type)
	{
	case TOOLBAR_ITEM_NEW:
		window->new_button = button;
		break;
	case TOOLBAR_ITEM_BACK:
		window->back_button = button;
		break;
	case TOOLBAR_ITEM_UP:
		window->up_button = button;
		break;
	case TOOLBAR_ITEM_FORWARD:
		window->forward_button = button;
		break;
	case TOOLBAR_ITEM_REFRESH:
		window->refresh_button = button;
		break;
	case TOOLBAR_ITEM_START:
		window->home_button = button;
		break;
	case TOOLBAR_ITEM_STOP:
		window->stop_button = button;
		break;
	case TOOLBAR_ITEM_GO:
		window->go_button = button;
		break;
	case TOOLBAR_ITEM_FIND:
		window->find_button = button;
		break;
	case TOOLBAR_ITEM_HISTORY:
		window->history_button = button;
		break;
	case TOOLBAR_ITEM_BOOKMARKS:
		window->bookmarks_button = button;
		break;
	default:
	}
}

/**
 * toolbar_add_docked: add an horizontal box to the toolbar dock
 */
static GtkWidget *
toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
		    gchar *name, gint band_number)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	gboolean location_on_main_toolbar;
	GnomeDockItemBehavior props;
	GtkReliefStyle relief;

	/* read prefs */
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	relief = (toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();
	location_on_main_toolbar = gnome_config_get_bool 
		(CONF_APPEARANCE_URL_LOCATION);

	/* default properties */
	props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
		 GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL);

	/* lock according to gnome prefs */
	if (!toolbar_detachable)
	{
		props |= GNOME_DOCK_ITEM_BEH_LOCKED;
	}

	/* dock into app */
	gnome_app_add_docked (GNOME_APP (window->WMain), hbox, name,
			      props, GNOME_DOCK_TOP, band_number, 0, 0);

	/* set border appropriate to whether or not there's relief */
	gtk_container_set_border_width (GTK_CONTAINER(hbox->parent),
					toolbar_relief ? 2 : 0);

	/* set shadowing follwing gnome pref */
	gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (hbox->parent), 
					 relief);

	/* this is a GnomeDockItem */
	return hbox->parent;
}

/**
 * toolbar_set_style: set overall relief and button relief styles
 */	
static void
toolbar_set_style (GtkWidget *toolbar, GtkToolbarStyle final_style,
		   GtkReliefStyle relief)
{
	/* skip if not there */
	if (toolbar == NULL)
	{
		return;
	}

	/* otherwise set style */
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), final_style);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar), relief);
}

/**
 * toolbar_theme_use_standard_icons: FIXME
 */
static gboolean 
toolbar_theme_use_standard_icons (void)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	char *filename;
	gchar *icon_theme;
	gboolean result = FALSE;

	filename = g_strconcat (theme_path, theme_name, ".xml", NULL);
	if (g_file_exists (filename))
	{
		doc = xmlParseFile (filename);
		g_return_val_if_fail (doc != NULL, FALSE);
		g_return_val_if_fail (doc->root != NULL, FALSE);
		item = doc->root->childs;
		while (item != NULL)
		{
			if (!strcmp (item->name, "toolbar"))
			{
				icon_theme = xmlGetProp (item, "icon_theme");
				result = (strcmp (icon_theme, "standard")==0);
				xmlFree (icon_theme);
			}
			item = item->next;
		}
		xmlFreeDoc (doc);
	}

	g_free (filename);
	return result;
}

/**
 * toolbar_get_theme_pixmap: FIXME
 */
static GtkWidget *
toolbar_get_theme_pixmap (const gchar *pixmap, gboolean is_nautilus)
{
	gchar *filename;
	PixmapData *icon;
	GtkWidget *widget;

	if (is_nautilus)
	{
		filename = g_strconcat (theme_path, pixmap, ".png", NULL);
	}
	else
	{
		filename = g_strconcat (SHARE_DIR, "/", pixmap, ".png", NULL);
	}

	if (!g_file_exists (filename) || pixmap == NULL)
	{
		gchar *dir; 

		g_free (filename);
	
		if (is_nautilus)
		{
			/* fall back to the default nautilus icon */
			dir = gnome_datadir_file("pixmaps/nautilus/");
			filename = g_strconcat (dir, pixmap, ".png", NULL);
			g_free (dir);
			if (!g_file_exists (filename))
			{
				g_free (filename);
				return NULL;
			}
		}
		else
		{
			return NULL;
		}
	}

	icon = pixmap_data_new_from_file (filename);
	g_free (filename);

	if (icon != NULL)
	{
		widget = gtk_pixmap_new (icon->pixmap, icon->mask);
		g_free (icon); /* FIXME: better to cache these? */
		return GTK_WIDGET (widget);
	}
	else
	{
		return NULL;
	}
}

/**
 * toolbar_get_theme_path: FIXME
 */
static char *
toolbar_get_theme_path (void)
{
	extern GConfClient *gconf_client;
	char *aa_pixmap_path;
	char *pixmap_path;
	char *theme;
	char *dir;

	theme = gconf_client_get_string (gconf_client, 
					 NAUTILUS_THEME_PREF, NULL);
	if (!theme)
	{
		return NULL;
	}

	theme_name = theme;

	dir = gnome_datadir_file ("pixmaps/nautilus/");

	if (strcmp (theme_name, "default") == 0)
	{
		pixmap_path = g_strdup (dir);
	} 
	else
	{
		pixmap_path = g_strconcat (dir, theme, "/", NULL);
	}
	g_free (dir);

	aa_pixmap_path = g_concat_dir_and_file (pixmap_path, 
						"fake-aa-navibar/");
	if (g_file_exists (aa_pixmap_path))
	{
		g_free (pixmap_path);
		return aa_pixmap_path;
	}
	else
	{
		g_free (aa_pixmap_path);
	}
	
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	/* try with the user directory */
	/* FIXME: look for AA path here too */
	pixmap_path = g_strconcat (g_get_home_dir (), "/.nautilus/themes/", 
		theme, "/", NULL);
	if (g_file_exists (pixmap_path))
	{
		return pixmap_path;
	}
	else
	{
		g_free (pixmap_path);
	}

	return NULL;
}
