/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "embed.h"
#include "window.h"
#include "misc.h"
#include "autocompletion.h"
#include "history.h"
#include "bookmarks.h"
#include "mozilla.h"
#include "context.h"
#include "toolbar.h"
#include "find.h"
#include "session.h"

#include <string.h>
#include <gtkmozembed.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-app-helper.h>
#include <gdk/gdkkeysyms.h>

/* local function prototypes */
gboolean window_delete_cb (GtkWidget *widget, GdkEventAny *event, 
			   GaleonWindow *window);
gint window_zoom_spin_timeout_cb (GaleonWindow *window);
void window_entry_changed_cb (GtkEditable *editable, gboolean *changed); 
void window_location_gnomeentry_popwin_cb (GtkWidget *widget, 
					   GaleonWindow *window);
static void window_handle_go_button (GaleonWindow *window, LinkState state);

GTimer *zoom_timer = NULL;
#define ZOOM_DELAY 0.20

/**
 * mozembed_selection_get_cb: get selection on copy link location
 * FIXME HYBRID: move this elsewhere
 */
void
window_selection_get_cb (GtkWidget *widget, GtkSelectionData *selection_data,
			 guint info, guint time_stamp, gpointer data)
{
	gchar *text;

	text = gtk_object_get_data (GTK_OBJECT(widget), "selection");

	g_return_if_fail (text != NULL);

	gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
				8, text, strlen (text));
}

/**
 * window_selection_received_cb: load url if received on middle button click
 */
void
window_selection_received_cb (GtkWidget *widget,
			      GtkSelectionData *selection_data,
			      guint time, GaleonWindow *window)
{
	GdkModifierType modifier;
	
	return_if_not_window (window);

	if (selection_data->data)
	{
		gdk_window_get_pointer (window->WMain->window,
			NULL, NULL, &modifier);

		embed_activate_link_keyboard (window->active_embed, NULL, 
					      selection_data->data, modifier);
	}
}

/**
 * window_delete_cb: deleting toplevel window
 */
gboolean
window_delete_cb (GtkWidget *widget, GdkEventAny *event, GaleonWindow *window)
{
	/* this can cause one of two things; if this is one of many windows
	 * then we want to close this window but keep the others going. If
	 * this is the last window, we want to behave as though the user
	 * chose to exit Galeon (and do automatic session saving if the
	 * pref is set */

	/* FIXME: I think this is broken. Should call always window_close, 
	 * window_close already takes care of this (ricardo) 
	 */
	if (g_list_length (all_windows) == 1 && 
	    !galeon_server_mode && !galeon_panel_mode)
	{
		/* exit the session */
		session_quit (FALSE);
	}
	else
	{
		/* close the window */
		window_close (window);
	}

	return TRUE;
}

/** 
 * window_back_forward_button_press_cb:
 */
gboolean
window_back_forward_button_press_cb (GtkButton *button,
				     GdkEventButton *event,
				     GaleonWindow *window)
{
	GtkMenu *menu = NULL;
	GaleonEmbed *embed;
	
	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	if (event->button == 3)
	{
		if (GTK_WIDGET (button) == window->forward_button)
		{
			menu = embed_create_forward_menu (embed);
		}
		else if (GTK_WIDGET (button) == window->back_button)
		{
			menu = embed_create_back_menu (embed);
		}
		else
		{
			g_assert_not_reached ();
		}
#if 0		/* FIXME: removed this since people objected to it --
		 * MattA 12/07/01 */
		context_menu_add_seperator (GTK_MENU (menu));
		context_show_appearance_menu (window, GTK_MENU (menu), 
					      NULL, TRUE, TRUE);
#endif
		gnome_popup_menu_do_popup_modal (GTK_WIDGET (menu), NULL,
						 NULL, event, NULL);
		gtk_widget_unref (GTK_WIDGET(menu));
		return TRUE;
	}	
	return FALSE;
}

/** 
 * window_home_button_release_event_cb: home button clicked
 */
gboolean
window_home_button_release_event_cb (GtkButton *button, GdkEventButton *event,
				     GaleonWindow *window)
{
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	return_val_if_not_window (window, FALSE);
	if (event->button != 3)
	{
		window_go_home (window, mouse_state_to_link_state 
						(event->button, event->state));
		return TRUE;
	}
	return FALSE;
}

/** 
 * window_home_button_press_event_cb:  Open home page in new window on
 * middle click, or display a context menu on right-click.
 */
gboolean
window_home_button_press_event_cb (GtkButton *button, GdkEventButton *event,
				   GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *menu;
	gint action;

	/* homepage context menu */
	/* although there's only one entry I think it's still necessary
	 * to have a context menu for UI consistency -- MattA */
	static GnomeUIInfo menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Set current page as home page"),
					NULL, NULL, GNOME_STOCK_MENU_HOME),
		GNOMEUIINFO_END
	};
	
	/* check args */
	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	/* handle button accordingly */	
	if (event->button == 3)
	{
		/* show context menu */
		menu = gnome_popup_menu_new (menu_uiinfo);
		context_menu_add_seperator (GTK_MENU (menu));
		context_show_appearance_menu (window, GTK_MENU (menu), 
					      NULL, TRUE, TRUE);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* perform set homepage action if chosen */
		if (action == 0 && embed->site_location != NULL &&
		    strlen (embed->site_location) != 0)
		{
			gnome_config_set_string (CONF_GENERAL_HOMEPAGE,
						 embed->site_location);
			gnome_config_sync ();
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		return TRUE;
	}
	return FALSE;
}

/** 
 * window_new_button_press_cb: new browser button pressed
 */
gboolean
window_new_button_press_cb (GtkButton *button, GdkEventButton *event,
			    GaleonWindow *window)
{
	GaleonEmbed *embed;
	GtkWidget *menu;
	gint action;

	/* new button context menu */
	static GnomeUIInfo menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Open new window"),
					NULL, NULL, GNOME_STOCK_MENU_NEW),
		GNOMEUIINFO_ITEM_STOCK (N_("Open new tab"),
					NULL, NULL, GNOME_STOCK_MENU_NEW),
		GNOMEUIINFO_END
	};

	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	switch (event->button)
	{
	case 3:
		/* show context menu */
		menu = gnome_popup_menu_new (menu_uiinfo);
		context_menu_add_seperator (GTK_MENU (menu));
		context_show_appearance_menu (window, GTK_MENU (menu), 
					      NULL, TRUE, TRUE);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);
		
		/* do action */
		if (action != -1)
		{
			embed_create_default (embed, action == 0);
		}
		gtk_widget_unref (menu);
		return TRUE;

	default:
		return FALSE;
	}
}
/** 
 * window_new_button_release_cb: new browser button clicked
 */
gboolean
window_new_button_release_cb (GtkButton *button, GdkEventButton *event,
			    GaleonWindow *window)
{
	gint tabbed_mode;
	gint shift_modifier;
	GaleonEmbed *embed;
	GaleonEmbed *new_embed;

	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED) ? 1 : 0;
	shift_modifier = (event->state & GDK_SHIFT_MASK) ? 0 : 1;
	
	switch (event->button)
	{
	case 1:
		new_embed = embed_create_default (embed, (tabbed_mode ^ 
							  shift_modifier));
		/* FIXME: why this, why here? */
		window_update_nav_controls (window);
		if (new_embed->parent_window->location_entry != NULL)
		{
			gtk_widget_grab_focus (GTK_WIDGET (
				new_embed->parent_window->location_entry));
		}
		return TRUE;
		
	case 2:
		/* hmmm, this is probably a bit hacky ;-) -- MattA */
		embed = embed_create (embed, (tabbed_mode ^ shift_modifier), 
				      TRUE, GTK_MOZ_EMBED_FLAG_ALLCHROME);
		gtk_selection_convert (embed->parent_window->WMain,
				       GDK_SELECTION_PRIMARY,
				       GDK_SELECTION_TYPE_STRING,
				       GDK_CURRENT_TIME);
		return FALSE;
	default:
		return FALSE;
	}
}

/**
 * window_new_button_drag_data_received_cb: called when data is dropped
 * on the "New" button
 */
void
window_new_button_drag_data_received_cb (GtkWidget *widget, 
					 GdkDragContext *drag_context,
					 gint x, gint y,
					 GtkSelectionData *selection_data,
					 guint info, guint time,
					 GaleonWindow *window)
{
	gboolean tabbed_mode;

	/* check args */
	g_return_if_fail (selection_data != NULL);
	g_return_if_fail (selection_data->data != NULL);
	return_if_not_window (window);
	return_if_not_embed (window->active_embed);

	/* get state */
	tabbed_mode = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

	/* check drop data */
	switch (info)
	{
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
	case DND_TARGET_STRING:
		embed_create_from_url (window->active_embed,
				       selection_data->data,
				       TRUE, !tabbed_mode);
		window_update_nav_controls (window);
		break;

	default:
		g_warning ("Unknown DND type");
		break;
	}
}

/** 
 * window_back_forward_button_release_cb: back/forward button clicked
 */
gboolean
window_back_forward_button_release_cb (GtkButton *button,
				     GdkEventButton *event,
				     GaleonWindow *window)
{
	char *url = NULL;
	GaleonEmbed *embed;
	
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	return_val_if_not_window (window, FALSE);
	return_val_if_not_embed (window->active_embed, FALSE);

	if (event->button == 3)
		return FALSE;
	
	embed = window->active_embed;
	if (event->button == 1 && !(event->state & GDK_SHIFT_MASK))
	{
		if (GTK_WIDGET (button) == window->forward_button)
		{
			gtk_moz_embed_go_forward 
				(GTK_MOZ_EMBED (embed->mozEmbed));
		}
		else if (GTK_WIDGET (button) == window->back_button)
		{
			gtk_moz_embed_go_back 
				(GTK_MOZ_EMBED (embed->mozEmbed));
		}
		else
		{
			g_assert_not_reached ();
		}
		return TRUE;
	}
	
	if (GTK_WIDGET (button) == window->forward_button)
	{
		url = mozilla_session_history_get_url_relative (embed, 1);
	}
	else if (GTK_WIDGET (button) == window->back_button)
	{
		url = mozilla_session_history_get_url_relative (embed, -1);
	}
	else
	{
		g_assert_not_reached ();
	}
	embed_activate_link_mouse (window->active_embed, NULL, url, event);
	if (url != NULL) free (url);
	
	return TRUE;
}

/** 
 * window_new_button_clicked_cb: new browser button pressed
 */
gboolean
window_up_button_press_event_cb (GtkButton *button, GdkEventButton *event,
				 GaleonWindow *window)
{
	GtkMenu *menu;

	/* check arguments */
	return_val_if_not_window (window, FALSE);
	return_val_if_not_embed (window->active_embed, FALSE);

	/* check button */
	if (event->button != 3)
	{
		return TRUE;
	}

	/* make menu */
	menu = embed_create_up_menu (window->active_embed);
	context_menu_add_seperator (GTK_MENU (menu));
	context_show_appearance_menu (window, GTK_MENU (menu), 
				      NULL, TRUE, TRUE);
	if (menu == NULL)
	{
		return TRUE;
	}

	/* show popup menu */
	gnome_popup_menu_do_popup_modal (GTK_WIDGET (menu),
					 NULL, NULL, event, NULL);

	/* destroy the popup menu */
	gtk_object_unref (GTK_OBJECT (menu));

	/* success */
	return FALSE;
}

/** 
 * window_up_button_release_event_cb: up browser button clicked
 */
gboolean
window_up_button_release_event_cb (GtkButton *button, GdkEventButton *event,
				   GaleonWindow *window)
{
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	return_val_if_not_window (window, FALSE);
	return_val_if_not_embed (window->active_embed, FALSE);
	if (event->button == 3)
		return FALSE;
	embed_go_up (window->active_embed, 0, 
		mouse_state_to_link_state (event->button, event->state));
	return TRUE;
}

/** 
 * window_refresh_button_clicked_cb: refresh button clicked
 */
void
window_refresh_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	embed_reload (window->active_embed);
}

/** 
 * window_refresh_button_press_event_cb: refresh button pressed
 */
gboolean
window_refresh_button_press_event_cb (GtkButton *button, GdkEventButton *event,
				      GaleonWindow *window)
{
	GtkWidget *menu;
	gint action;
	GList *l;

	/* reload context menu */
	static GnomeUIInfo menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Reload all tabs"),
					NULL, NULL, GNOME_STOCK_MENU_REFRESH),
		GNOMEUIINFO_ITEM_STOCK (N_("Reload all windows"),
					NULL, NULL, GNOME_STOCK_MENU_REFRESH),
		GNOMEUIINFO_END
	};
	
	/* check args */
	return_val_if_not_window (window, FALSE);

	/* handle button accordingly */	
	switch (event->button)
	{
	case 1:
		/* nothing -- handled by click callback */
		break;

	case 2:
		/* nothing on middle click */
		break;

	case 3:
		/* show context menu */
		menu = gnome_popup_menu_new (menu_uiinfo);
		context_menu_add_seperator (GTK_MENU (menu));
		context_show_appearance_menu (window, GTK_MENU (menu), 
					      NULL, TRUE, TRUE);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* perform set homepage action if chosen */
		switch (action)
		{
		case 0: /* reload all tabs */
			window_reload_all (window);
			break;

		case 1: /* reload all windows */
			for (l = all_windows; l != NULL; l = g_list_next (l))
			{
				window_reload_all ((GaleonWindow *)(l->data));
			}
			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		break;
	}

	return TRUE;
}

/** 
 * window_stop_button_clicked_cb: stop button clicked
 */
void
window_stop_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	return_if_not_window (window);
	gtk_moz_embed_stop_load (GTK_MOZ_EMBED(window->active_embed->mozEmbed));
}

/** 
 * window_generic_button_press_event_cb: some button clicked, show appearance
 * menu
 */
gboolean
window_generic_button_press_event_cb (GtkButton *button, GdkEventButton *event,
				      GaleonWindow *window)
{
	GtkWidget *menu;

	/* check args */
	return_val_if_not_window (window, FALSE);

	/* handle button accordingly */	
	switch (event->button)
	{
	case 1:
		/* nothing -- handled by click callback */
		break;

	case 2:
		/* nothing on middle click */
		break;

	case 3:
		/* show context menu */
		menu = gtk_menu_new ();
		context_show_appearance_menu (window, GTK_MENU(menu), 
					      NULL, TRUE, FALSE);
		gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		break;
	}

	return TRUE;
}

/** 
 * window_location_entry_key_press_cb: key pressed in the url entry
 */
gboolean
window_location_entry_key_press_cb (GtkWidget *widget, GdkEventKey *event,
				    GaleonWindow *window)
{
	static gchar *before_completion = NULL;
	GtkEntry *entry = GTK_ENTRY (widget);
	GtkEditable *editable = GTK_EDITABLE (widget);
	gboolean url_dialog = FALSE;
	GaleonEmbed *embed;
	guint keyval;

	g_return_val_if_fail (GTK_IS_ENTRY (widget), TRUE);
	return_val_if_not_window (window, TRUE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, TRUE);

	auto_completion_reset();

	/* don't process the standard GtkEntry editing shortcuts */
	if (event->keyval > GDK_A && event->keyval < GDK_Z)
		keyval = event->keyval + 0x20;
	else
		keyval = event->keyval;

	if (((event->state & GDK_Control_L || event->state & GDK_Control_R) &&
	     (keyval == GDK_a || keyval == GDK_b || keyval == GDK_c ||
	      keyval == GDK_d || keyval == GDK_e || keyval == GDK_f ||
	      keyval == GDK_h || keyval == GDK_k || keyval == GDK_u ||
	      keyval == GDK_v || keyval == GDK_w || keyval == GDK_x )) ||
	    (event->state == 0 && event->keyval == GDK_BackSpace))
	{
		return TRUE;
	}

	/* don't grab alt combos, thus you can still access the menus. */
	if (event->state & GDK_MOD1_MASK)
		return FALSE;

	/* check to see if key press is in the window's location bar or
	   the open url dialog */
	if (widget != window->location_entry)
	{
		url_dialog = TRUE;
	}

	if ((event->keyval == GDK_Return) || (event->keyval == GDK_KP_Enter))
	{
		if (url_dialog)
		{
			open_url_handle_link (window, 
				key_state_to_link_state (event->state));
		}
		else
		{
			window_handle_go_button (window, 
				key_state_to_link_state (event->state));
		}
		return TRUE;
	}

	/* make sure the end key works at all times */ 
	if ( ( ! ( (event->state & GDK_SHIFT_MASK) ||
		   (event->state & GDK_CONTROL_MASK) ||
		   (event->state & GDK_MOD1_MASK) ) 
	      && (event->keyval == GDK_End)))
	{
		gtk_editable_select_region(editable, 0, 0); 
		gtk_editable_set_position(editable, -1);
		return TRUE;
	}

	switch (event->keyval)
	{
	case GDK_Left:
	case GDK_Right:
		return TRUE;
	case GDK_Tab:
	{
		gchar *common_prefix = NULL;
		gchar *text;

		gtk_editable_delete_selection (editable);
		text = gtk_editable_get_chars (editable, 0, -1);
		common_prefix = auto_completion_complete_url_extended (text);
		if (common_prefix)
		{
			if (!before_completion) 
				before_completion = g_strdup(text);

			gtk_entry_set_text (entry, common_prefix);
			auto_completion_display_alternatives (window, widget);
			g_free (common_prefix);
		}
		return TRUE;
	}
	case GDK_Escape:
		auto_completion_reset();

		if (before_completion)
		{
			gtk_entry_set_text (entry, before_completion);
			g_free (before_completion);
			before_completion = NULL;
		}
		else
		{
			gtk_entry_set_text (entry, embed->site_location);
		}
		return TRUE;
	default:
		break;
	}

	if (event->string[0] > 32 && event->string[0] < 126) 
	{
		if (before_completion != NULL) 
		{
			g_free (before_completion);
			before_completion = NULL;
		}
		auto_completion_display (window, widget);
		return TRUE;
	}

	/* The URL dialog connects to this handler with *_connect_after,
	   so we need to return TRUE by default */
	if (url_dialog)
		return TRUE;

	return FALSE;
}

/**
 * window_location_entry_button_press_cb: select the whole location field
 * when the user double clicks on the entry. This is preferable to the
 * GNOME and X standard way of selecting since it's not selecting english
 * text but instead a URL.
 */
gboolean
window_location_entry_button_press_cb (GtkWidget *entry, 
				       GdkEventButton *event, 
				       GaleonWindow *window)
{
	if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
	{
		gtk_editable_set_position (GTK_EDITABLE (entry), -1);
		gtk_entry_select_region (GTK_ENTRY (entry), 0, -1);

		/* Ugh, more GTK+ goofiness. Returning TRUE doesn't stop
		   emission of the signal, so we have to do it ourselves */
		gtk_signal_emit_stop_by_name (GTK_OBJECT (entry),
		                              "button-press-event");
		return TRUE;
	}
	return TRUE;
}

static void
window_handle_go_button (GaleonWindow *window, LinkState state)
{
	gchar *text;
	gchar *text2;
	GList *wl;
	GnomeEntry *ge;

	return_if_not_window (window);

	if (window->location_entry == NULL) return;
	if (window->location_gnomeentry == NULL) return;

	text = gtk_editable_get_chars (GTK_EDITABLE (window->location_entry),
				       0, -1);

	/* make sure the string isn't empty */
	if (strcmp (text, "") != 0)
	{
		/* forcibly save contents of entry */
		ge = GNOME_ENTRY (window->location_gnomeentry);
		gnome_entry_save_history (ge);
		gnome_config_sync ();

		/* add item to location bars in all other
		 * open browser windows */
		for (wl = all_windows; wl != NULL; wl = wl->next)
		{
			GaleonWindow *awindow = (GaleonWindow *)(wl->data);

			/* skip if no locatio entry to fill */
			if (awindow->location_gnomeentry == NULL)
			{
				continue;
			}

			/* reload entry */
			gnome_entry_load_history 
				(GNOME_ENTRY (awindow->location_gnomeentry));
		}

		text2 = bookmarks_parse_nick (text, NULL);

		if (text2) 
		{
			g_free (text);
			text = text2;
		}
	}
	/* empty string -- load a blank page */
	else
	{
		g_free (text);
		text = g_strdup ("about:blank");
	}

	embed_activate_link (window->active_embed, NULL, text, state);

	g_free (text);
}

/** 
 * window_go_button_release_cb: go button released
 */
gboolean
window_go_button_release_cb (GtkButton *button, GdkEventButton *event,
			     GaleonWindow *window)
{
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	return_val_if_not_window (window, FALSE);

	/* open location in correct place */
	window_handle_go_button (window,
		mouse_state_to_link_state (event->button, event->state));

	return TRUE;
}

/**
 * Changes the zoom if enough time time has passed 
 */
gint 
window_zoom_spin_timeout_cb (GaleonWindow *window)
{
	GaleonEmbed *embed;
	gint zoom;

	return_val_if_not_window (window, FALSE);
	embed = window->active_embed;
	return_val_if_not_embed (embed, FALSE);

	/* timer still valid? */
	if (zoom_timer == NULL)
	{
		return FALSE;
	}

	/* okay, we're ready to set */
	if (g_timer_elapsed (zoom_timer, NULL) >= ZOOM_DELAY)
	{
		/* kill off the timer */
		g_timer_destroy (zoom_timer);
		zoom_timer = NULL;

		/* get the value of the spin */
		zoom = gtk_spin_button_get_value_as_int 
			(GTK_SPIN_BUTTON (window->zoom_spin));

		/* set */
		embed_set_zoom (embed, zoom);

		/* store for this site */
		history_set_zoom (embed->site_location, zoom);

		/* done now */
		return FALSE;
	}

	/* call me again */
	return TRUE;
}

/** 
 * window_zoom_spin_changed_cb: zoom spin value changed. Starts the zoom timer
 */
void 
window_zoom_spin_changed_cb (GtkEditable *editable, GaleonWindow *window)
{
	gint zoom;

	/* get the value of the spin */
	zoom = gtk_spin_button_get_value_as_int 
		(GTK_SPIN_BUTTON (window->zoom_spin));

	/* check we haven't already registered the change */
	if (window->active_embed != NULL && window->active_embed->zoom == zoom)
	{
		return;
	}

	/* destroy any existing timer */
	if (zoom_timer != NULL)
	{
		g_timer_destroy (zoom_timer);
	}

	/* start the new one */
	zoom_timer = g_timer_new();
	g_timer_start (zoom_timer);
	g_timeout_add (50, (GSourceFunc) window_zoom_spin_timeout_cb, window);
}

/** 
 * window_zoom_spin_button_press_cb: stop context menu from appearing
 * (context menu here will make galeon go nuts)
 */
gboolean
window_zoom_spin_button_press_cb (GtkButton *button,
				     GdkEventButton *event,
				     GaleonWindow *window)
{
	return TRUE;
}

/** 
 * window_drag_pixmap_drag_data_get_cb:
 */
void
window_drag_pixmap_drag_data_get_cb (GtkWidget *widget, 
				     GdkDragContext *context,
				     GtkSelectionData *selection_data, 
				     guint info, guint time, 
				     GaleonWindow *window)
{
	GaleonEmbed *embed;
	BookmarkItem *b;
	gchar *mem;

	return_if_not_window (window);
	embed = window->active_embed;
	return_if_not_embed (embed);

	switch (info)
	{
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_new_bookmark (BM_SITE, TRUE, embed->site_title,
					    embed->site_location,
					    NULL, NULL, NULL);
		mem = bookmarks_item_to_string (b);
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, mem, strlen (mem));
		g_free (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, embed->site_location, 
			 strlen (embed->site_location));
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
}

/** 
 * window_location_entry_drag_data_received_cb:
 */
void
window_location_entry_drag_data_received_cb (GtkWidget *widget, 
				 GdkDragContext *drag_context, gint x, gint y,
				 GtkSelectionData *selection_data, guint info,
				 guint time, GaleonWindow *window)
{
	gchar *url = selection_data->data;
	gint i;

	switch (info) {
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
	case DND_TARGET_TEXT_URI_LIST:/* FIXME: won't work right for a list */
		window_clear_url_entry (window);
		gtk_editable_insert_text (GTK_EDITABLE (window->location_entry),
					  url, strlen (url), &i);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
}

/** 
 * window_zoom_spin_key_press_cb: Ignore up/down key presses 
 */
gboolean
window_zoom_spin_key_press_cb (GtkWidget *widget, GdkEventKey *event, GaleonWindow *window)
{
	if ((event->keyval == GDK_Up) || (event->keyval == GDK_Down)) {
		event->keyval = GDK_VoidSymbol;
	}
	return FALSE;
}

void
window_entry_changed_cb (GtkEditable *editable, gboolean *changed)
{
	*changed = TRUE;
}

/* 
 * window_location_gnomeentry_popwin_cb:
 * Handler used to see if a selection was made in the location entry.
 * If a selection is made and the go toolbar is hidden, it will load
 * the URL.  It will also clear out the focus box that the GtkCombo
 * widget leaves behind after making a selection. 
 */
void
window_location_gnomeentry_popwin_cb (GtkWidget *widget, GaleonWindow *window)
{
	GtkCombo *combo;
	GtkContainer *container;
	GtkList *list;
	static gint selection_made;

	g_assert(widget!=NULL);
	g_assert(window->location_gnomeentry!=NULL);

	if (GTK_WIDGET_VISIBLE(widget))
	{
		combo = GTK_COMBO(window->location_gnomeentry);
		list = GTK_LIST(combo->list);
		container = GTK_CONTAINER(list);

		if (container->focus_child)
		{
			if (!GTK_LIST(container)->selection)
			{
				gtk_window_set_focus (GTK_WINDOW (GTK_COMBO 
						      (combo)->popwin), NULL);
				container->focus_child = NULL;
				list->last_focus_child = NULL;
			}

		}

		selection_made = FALSE;

		/* connect a handler to the entry's "changed" signal */
		gtk_signal_connect(GTK_OBJECT(window->location_entry),
		                   "changed",
		                   GTK_SIGNAL_FUNC(window_entry_changed_cb),
		                   &selection_made);
	}
	else
	{
		/* connect a handler to the entry's "changed" signal */
		gtk_signal_disconnect_by_func
			(GTK_OBJECT (window->location_entry),
			 GTK_SIGNAL_FUNC (window_entry_changed_cb),
			 &selection_made);

		if (window->go_button != NULL && selection_made)
		{
			window_handle_go_button (window, 0);
		}
	}
}

/**
 * window_drag_data_get_cb:
 */
void
window_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
		    GtkSelectionData *selection_data, 
		    guint info, guint time, 
		    GaleonWindow *window)
{
	gchar *link;
	g_return_if_fail (window->WMain != NULL);

	link = gtk_object_get_data (GTK_OBJECT (widget), "dragging_link");

	if (link) {
		switch (info) {
		case DND_TARGET_STRING:
		case DND_TARGET_NETSCAPE_URL:
		case DND_TARGET_GALEON_URL:
			gtk_selection_data_set 
				(selection_data, selection_data->target,
				 8, link, strlen (link));
			break;
		default:
			g_warning ("Unknown DND type");
			break;
		}
	} 
}

/**
 * window_progress_action: update the status bar progress
 */
gboolean
window_progress_action (GaleonWindow *window)
{
	GtkProgress *progress;
	gfloat val;

	/* check window */
	return_val_if_not_window (window, FALSE);

	/* check window */
	g_return_val_if_fail (window->appbar != NULL, FALSE);

	/* check we're still meant to be running */
	if (!(window->progress_timeout))
	{
		return FALSE;
	}
	
	/* find the progress widget */
	progress = gnome_appbar_get_progress (GNOME_APPBAR (window->appbar));

	/* update progressive loader "activity" indicator */
	val = gtk_progress_get_value (progress);
	if (++val > 100)
	{
		val = 0;
	}
	gtk_progress_set_value (progress, val/100);

	/* continue to be called */
	return TRUE;
}

/**
 * window_find_button_clicked_cb: Find button on toolar clicked; open find
 * dialog
 */
void
window_find_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	find_show_dialog (window);
}

/**
 * window_drag_data_received_cb: handle dropped data
 */
void
window_drag_data_received_cb (GtkWidget *widget, GdkDragContext *context,
			      gint x, gint y, GtkSelectionData *selection_data,
			      guint info, guint time, GaleonWindow *window)
{
	switch (info)
	{
		case DND_TARGET_GALEON_EMBED:
			{
				GaleonEmbed *embed;

				return_if_not_window (window);

				/* get embed address */
				memcpy (&embed, selection_data->data,
					sizeof (GaleonEmbed *));
				return_if_not_embed (embed);

				embed_move_tab (embed, window->active_embed);
				break;
			}
	}
}

/**
 * window_history_button_clicked_cb: history button clicked, show docked
 * history view unless already shown, in which case hide it.
 */
void
window_history_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
	{
		if (window->dock_type != DOCK_HISTORY)
		{
			history_show_dock (window);
		}
	}
	else
	{
		if (window->dock_type != DOCK_NONE)
		{
			window_undock (window);
		}
	}
}

/**
 * window_bookmarks_button_clicked_cb: bookmarks button clicked, show docked
 * bookmarks view unless already shown, in which case hide it.
 */
void
window_bookmarks_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button)))
	{
		if (window->dock_type != DOCK_BOOKMARKS)
		{
			bookmarks_editor_show_dock (window);
		}
	}
	else
	{
		if (window->dock_type != DOCK_NONE)
		{
			window_undock (window);
		}
	}
}

/**
 * window_print_button_clicked_cb: print button clicked, show dialog
 */
void
window_print_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	print_show_dialog (window);
}

void 
window_session_history_menuitem_activate_cb (GtkMenuItem *menuitem, 
					     gint index)
{
	session_history_load_index (index);
}


