/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "misc.h"
#include "embed.h"
#include "autocompletion.h"
#include "bookmarks.h"
#include "window.h"
#include "prefs.h"
#include "favicon.h"
#include "mozilla.h"
#include "themes.h"

/* GNOME includes */
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtkdnd.h>
#include <gtkmozembed.h>

#define NUM_BOOKMARK_BACKUPS 3

/** The pixmaps for the bookmarks */
PixmapData *folder_pixmap_data = NULL;
PixmapData *folder_open_pixmap_data = NULL;
PixmapData *default_folder_pixmap_data = NULL;
PixmapData *default_folder_open_pixmap_data = NULL;
PixmapData *site_pixmap_data = NULL;
PixmapData *smart_bm_tray_pixmap_data = NULL;

/** The root of the bookmark tree */
BookmarkItem *bookmarks_root = NULL;

/** The root of the temporary bookmark tree */
BookmarkItem *default_bookmarks_root = NULL;

/** The bookmarks need to be saved */
gboolean bookmarks_dirty = FALSE;

/** Do we need to recreate the bookmarks toolbars? */
gboolean bookmarks_tb_dirty = FALSE;

/** The next id to assign (for alias saving) */
static long maxid = 0;

/** 
 * Aliases that have been loaded, but whose real bookmark has not been loaded 
 */
GList *unresolved_aliases = NULL;

/** The favicon pixmap data cache */
GHashTable *favicons_cache = NULL;

/* the filename of the bookmarks */
gchar *bookmarks_filename = NULL;

/* DnD */
const GtkTargetEntry bookmarks_dnd_targets [] = {
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "text/uri-list",   0, DND_TARGET_TEXT_URI_LIST   },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint bookmarks_dnd_targets_num_items = (sizeof (bookmarks_dnd_targets) /
					      sizeof (GtkTargetEntry));

/* Private prototypes */
static void bookmarks_save (void);
static BookmarkItem *bookmarks_xml_read_item (xmlNodePtr item);
static BookmarkItem *bookmarks_xml_read (xmlNodePtr item);
static void bookmarks_add_completion_recursively (BookmarkItem *b);
static void bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root,
					BookmarkItem *b);
static void bookmarks_remove_all_alias (BookmarkItem *b);
static void bookmarks_new_alias_do (BookmarkItem *b, BookmarkItem *bookmark);
static BookmarkItem *bookmarks_find_by_id (long id,
					   BookmarkItem *root);
static BookmarkItem *bookmarks_merge_trees_look_for_equal 
	(BookmarkItem *folder, BookmarkItem *bookmark);
static gint bookmarks_compare_name_and_type (gconstpointer a, gconstpointer b);
static gint bookmarks_compare_name (gconstpointer a, gconstpointer b);
static gchar *bookmarks_get_bookmark_path (BookmarkItem *b);
static void bookmarks_ensure_unique_ids (BookmarkItem *root);
static void bookmarks_ensure_unique_ids_rec (BookmarkItem *b, gulong *maxid);

/**
 * bookmarks_init: intialise bookmarks for the first time
 */
void
bookmarks_init (void)
{
	bookmarks_filename = g_strconcat (g_get_home_dir (),
			                  "/.galeon/bookmarks.xml", NULL);
	bookmarks_load_icons ();
	bookmarks_load ();
}

/**
 * bookmarks_exit: shut down bookmarks
 */
void
bookmarks_exit (void)
{
	GList *l = g_list_copy (bookmarks_editors);
	GList *li;

	bookmarks_save ();

	/* close all editors */
	for (li = l; li != NULL; li = li->next) 
	{
		bookmarks_editor_hide_dialog (li->data);
	}
	g_list_free (l);

	g_free (folder_open_pixmap_data);
	g_free (folder_pixmap_data);
	g_free (default_folder_open_pixmap_data);
	g_free (default_folder_pixmap_data);
	g_free (site_pixmap_data);
	g_free (smart_bm_tray_pixmap_data);

	folder_open_pixmap_data = NULL;
	folder_pixmap_data = NULL;
	default_folder_open_pixmap_data = NULL;
	default_folder_pixmap_data = NULL;
	site_pixmap_data = NULL;
	smart_bm_tray_pixmap_data = NULL;

	bookmarks_remove_recursively (bookmarks_root);
}

/**
 * bookmarks_load: load bookmarks from xml file
 */
void
bookmarks_load (void)
{
	default_bookmarks_root = NULL;

	bookmarks_set_root (bookmarks_load_from (bookmarks_filename));
}

/**
 * Sets the root of the bookmarks tree, freeing the previous tree and updating
 * everything
 */
void
bookmarks_set_root (BookmarkItem *new_root)
{
	GList *li;
	/* iterate throught the list of all opened editors */
	for (li = bookmarks_editors; li != NULL; li = li->next) {
		BookmarksEditorControls *current_controls = li->data;
		gtk_clist_freeze (GTK_CLIST (current_controls->ctree));
	}
	
	/* delete the old bookmarks */
	bookmarks_remove_recursively (bookmarks_root);
	
	bookmarks_root = new_root;
	if (!bookmarks_root) {
		gchar* utfstr = mozilla_locale_to_utf8(_("Bookmarks"));
		bookmarks_root = bookmarks_new_bookmark 
			(BM_FOLDER, TRUE, utfstr, NULL, NULL,
			 NULL, NULL);
		g_free(utfstr);
	}
	
	if (default_bookmarks_root == NULL) {
		default_bookmarks_root = bookmarks_root;
	}

	/* generate the autobookmarks folder */
	autobookmarks_generate();
	
	bookmarks_add_completion_recursively (bookmarks_root);

	/* view new bookmarks */
	for (li = bookmarks_editors; li != NULL; li = li->next) {
		/* FIXME: this sets the root bookmark of every editor
		   to the root bookmark. This is incorrect. 
		   (but not a big deal) */
		BookmarksEditorControls *current_controls = li->data;
		bookmarks_editor_set_root (current_controls, bookmarks_root);
		current_controls->selection = NULL;
		current_controls->last_pressed = NULL;
		gtk_clist_thaw (GTK_CLIST (current_controls->ctree));
	}
}

/**
 * bookmarks_load_icons: load the bookmarks icons
 */
void 
bookmarks_load_icons (void) 
{
	/* load pixmaps */
	folder_pixmap_data = get_theme_pixmap ("dir.xpm", FALSE);
	folder_open_pixmap_data = get_theme_pixmap ("dir_open.xpm", FALSE);
	default_folder_pixmap_data = get_theme_pixmap ("default.xpm", FALSE);
	default_folder_open_pixmap_data = get_theme_pixmap ("default_open.xpm",
							    FALSE);
	site_pixmap_data = get_theme_pixmap ("i-bookmark.xpm", FALSE);
	smart_bm_tray_pixmap_data = get_theme_pixmap ("smart-bm-tray.xpm",
						      FALSE);

	/* TODO: add icons for alias */
}

/**
 * bookmarks_load_from: Read a bookmarks tree
 * @file: The file to read, it should exist
 * Return value: The new allocated tree of bookmarks or NULL if error
 **/
BookmarkItem *
bookmarks_load_from (char *file)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	if (!(g_file_exists (file))) {
		/* no bookmarks */
		return NULL;
	}

	doc = xmlParseFile (file);	
	
	if (doc) {
		item = doc->root;
		b = bookmarks_xml_read (item);
		xmlFreeDoc (doc);
		return b;
	} else {
		g_warning ("unable to parse bookmarks file: %s", file);
		return NULL;
	}
}

/**
 * bookmarks_xml_read: read all items from xml file, doing alias resolution
 */
static BookmarkItem * 
bookmarks_xml_read (xmlNodePtr item) 
{
	static BookmarkItem *error_bookmark = NULL;
	BookmarkItem *result;
	item = item->childs;
	unresolved_aliases = NULL;
	
	if (!error_bookmark) {
		gchar* eutfstr = mozilla_locale_to_utf8(_("Error"));
		gchar* nutfstr = mozilla_locale_to_utf8(
			_("An error ocurred loading your bookmarks."));
		error_bookmark = bookmarks_new_bookmark 
			(BM_SITE, TRUE, eutfstr, "about:blank", NULL,
			 nutfstr, NULL);
		g_free(eutfstr);
		g_free(nutfstr);
	}
	
	/* old format, no longer supported */
	/*
	result = bookmarks_new_bookmark (BM_FOLDER, TRUE, _("Bookmarks"),
					 NULL, NULL, NULL, NULL);
	if (item->next) 
		while (item != NULL) { 
			xmlNodePtr item2 = item; 
			BookmarkItem *b = bookmarks_xml_read_item (item2); 
			result->list = g_list_append (result->list, b);
			b->parent = result; 
			item = item->next; 
		} 
	*/

	result = bookmarks_xml_read_item (item);
	
	/* Finish creating the aliases */
	while (unresolved_aliases) {
		BookmarkItem *alias = unresolved_aliases->data;
		BookmarkItem *real = bookmarks_find_by_id (alias->id, result);
		unresolved_aliases = g_list_remove (unresolved_aliases, alias);
		if (real) {
			bookmarks_new_alias_do (real, alias);
		} else {
			g_warning 
				("Aliased bookmark not found: %ld", alias->id);
			bookmarks_new_alias_do (error_bookmark, alias);
		}
	}

	return result;
}

/**
 * bookmarks_xml_read_item: read an item from xml file. Does not resolve 
 * aliases. Use with care.
 */
static BookmarkItem *
bookmarks_xml_read_item (xmlNodePtr item)
{
	BookmarkItem *b;
	BookmarkItem *b2;
	gchar *ct, *expanded, *ccm, *default_root, *tbs, *ak, *am;

	gchar *name = xmlGetRawProp (item, "name");	
	gchar *notes = xmlGetRawProp (item, "notes");
	gchar *nick = xmlGetRawProp (item, "nick");
	gchar *pixmap_file = xmlGetProp (item, "pixmap");
	gchar *idstr = xmlGetProp (item, "id");
	gchar *time_added_str = xmlGetProp (item, "time_added");
	gchar *time_modified_str = xmlGetProp (item, "time_modified");
	gchar *time_visited_str = xmlGetProp (item, "time_visited");
  
	g_return_val_if_fail(item != NULL, NULL);

	if (strcmp (item->name, "folder") == 0 ||
	    strcmp (item->name, "category") == 0) 
	{
		b = bookmarks_new_bookmark (BM_FOLDER, FALSE, name, NULL,
					    nick, notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
	        default_root = xmlGetProp (item, "default_bookmarks_root");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE") ? 
				     TRUE : FALSE);
		if (tbs)
			b->toolbar_style = atoi(tbs);
		if (b->toolbar_style > 2) /* backwards compatiblity FIXME */
			b->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);
		b->expanded = (expanded && !strcmp (expanded, "TRUE") ? 
			       TRUE : FALSE);
		if (default_root && !strcmp (default_root, "TRUE"))
		{
			default_bookmarks_root = b;
		}
	        xmlFree (ct);		
		xmlFree (tbs);
		xmlFree (ccm);
		xmlFree (expanded);
		xmlFree (default_root);
		item = item->childs;
		while (item != NULL)
		{
			xmlNodePtr item2 = item;
			b2 = bookmarks_xml_read_item (item2);
			b->list = g_list_append (b->list, b2);
			b2->parent = b;
			item = item->next;
		}
	} 
	else if (!strcmp (item->name, "separator"))
	{
		b = bookmarks_new_bookmark (BM_SEPARATOR, FALSE, name, NULL,
					    nick, notes, pixmap_file);
	} 
	else if (!strcmp (item->name, "autobookmarks"))
	{
		/* FIXME: remove code duplication */
		b = bookmarks_new_bookmark (BM_AUTOBOOKMARKS, FALSE, name, 
					    NULL, nick, notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE 
			: FALSE;
		if (tbs)
			b->toolbar_style = atoi(tbs);
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE")) ? TRUE
			: FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE
			: FALSE;
		xmlFree (ct);
		xmlFree (tbs);
		xmlFree (ccm);
		xmlFree (expanded);
		autobookmarks_root = b;
	} 
	else if (!strcmp (item->name, "alias")) 
	{
		b = bookmarks_new_alias (NULL);
		unresolved_aliases = g_list_prepend (unresolved_aliases, b);
		ak = xmlGetProp (item, "accel_key");
		if (ak)
			b->accel_key = atoi (ak);
		xmlFree (ak);
		am = xmlGetProp (item, "accel_mods");
		if (am)
			b->accel_mods = atoi (am);
		xmlFree (am);

		ct = xmlGetProp (item, "create_toolbar");
		tbs = xmlGetProp (item, "toolbar_style");
		ccm = xmlGetProp (item, "create_context_menu");

		b->create_toolbar = (ct && !strcmp (ct, "TRUE") ? 
				     TRUE : FALSE);
		if (tbs)
			b->toolbar_style = atoi(tbs);
		if (b->toolbar_style > 2) /* backwards compatiblity FIXME */
			b->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);

		xmlFree (ct);
		xmlFree (tbs);
		xmlFree (ccm);

	} 
	else 
	{
		/* site */
		gchar *url = xmlGetProp (item, "url");
		b = bookmarks_new_bookmark (BM_SITE, FALSE, name, url, 
					    nick, notes, pixmap_file);
		ccm = xmlGetProp (item, "create_context_menu");
		b->create_context_menu = (ccm && !strcmp (ccm, "TRUE") ? 
					  TRUE : FALSE);
		xmlFree (ccm);
		if (url) g_free (url);
		ak = xmlGetProp (item, "accel_key");
		if (ak)
			b->accel_key = atoi (ak);
		xmlFree (ak);
		am = xmlGetProp (item, "accel_mods");
		if (am)
			b->accel_mods = atoi (am);
		xmlFree (am);
	}
	if (time_added_str)
	{
		b->time_added = strtol (time_added_str, NULL, 10);
		g_free (time_added_str);
	}
	if (time_modified_str)
	{
		b->time_modified = strtol (time_modified_str, NULL, 10);
		g_free (time_modified_str);
	}
	if (time_visited_str)
	{
		b->time_visited = strtol (time_visited_str, NULL, 10);
		g_free (time_visited_str);
	}
	if (idstr) b->id = atol (idstr);
	if (maxid <= b->id) maxid = b->id + 1;
	if (name) g_free (name);
	if (notes) g_free (notes);
	if (nick) g_free (nick);
	if (pixmap_file) g_free (pixmap_file);
	if (idstr) g_free (idstr);
	
	return b;
}

/**
 * bookmarks_updated: called when one or more change has taken place in 
 * the bookmarks which we wish to record, i.e. save to disk and display
 * in the menus and toolbars.
 */
void
bookmarks_updated (void)
{
	GList *l;

	/* force a save */
	bookmarks_dirty = TRUE;
	/* FIXME save is very expensive because of the backups ! */
	bookmarks_save ();

	/* update each window */
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *)(l->data);

		if (!window || (window->magic != GALEON_WINDOW_MAGIC))
		{
			g_warning ("bookmaks_updated found unexpected things "
				   " in window list!");
			continue;
		}
		
		/* save toolbar layout */
		window_save_layout (window);
	
		/* update bookmarks menubar */
		bookmarks_menu_recreate (window);
	
		/* update bookmarks toolbars */
		if (bookmarks_tb_dirty)
			bookmarks_toolbars_recreate (window);

		/* restore toolbar layout */
		window_restore_layout (window);
	}

	bookmarks_tb_dirty = FALSE;

	/* reload any myportals */
	for (l = all_embeds; l; l = g_list_next (l))
	{
		GaleonEmbed *embed = (GaleonEmbed *) l->data;

		if (embed->site_location 
		    && strncmp (embed->site_location, "myportal:", 9) == 0)
		{
			embed_reload (embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
		}
	}

	/* FIXME: update applet... */
}

/**
 * bookmarks_save: save all bookmarks to bookmarks_file, after backing up
 * the previous versions 
 */
void
bookmarks_save (void)
{
	gint size;
	gchar *temp_filename;
	GnomeVFSURI *temp_uri;

	/* make sure the bookmarks have been changed before we do anything */
	if (!bookmarks_dirty)
	{
		return;
	}

	/* write the temp file */
	temp_filename = g_strconcat (bookmarks_filename, ".temp", NULL);
	size = bookmarks_save_as (temp_filename);
	temp_uri = gnome_vfs_uri_new (temp_filename);

	/* if the file was written... */
	if (size > 0 && gnome_vfs_uri_exists (temp_uri))
	{
		gint count;
		GnomeVFSResult result;

		/* backup/rotate the bookmarks */
		for (count = NUM_BOOKMARK_BACKUPS - 1; count >= 0; count--)
		{
			gchar *src_ext, *dest_ext;
			gchar *src_filename, *dest_filename;
			GnomeVFSURI *src_uri, *dest_uri;

			/* get the extensions */
			if (count > 0) src_ext = g_strdup_printf (
							".%d", count - 1);
			else src_ext = NULL;
			dest_ext = g_strdup_printf (".%d", count);

			/* create the filenames */
			src_filename = g_strconcat (bookmarks_filename,
						    src_ext, NULL);
			dest_filename = g_strconcat (bookmarks_filename,
						     dest_ext, NULL);

			/* create uris from the filenames */
			src_uri = gnome_vfs_uri_new (src_filename);
			dest_uri = gnome_vfs_uri_new (dest_filename);

			/* move the file */
			if (gnome_vfs_uri_exists (src_uri))
			{
				gnome_vfs_move_uri (src_uri,
						    dest_uri, TRUE);
			}

			/* free stuff */
			g_free (src_ext);
			g_free (dest_ext);
			g_free (src_filename);
			g_free (dest_filename);
			gnome_vfs_uri_unref (src_uri);
			gnome_vfs_uri_unref (dest_uri);
		}

		/* move the temp file to the normal location */
		result = gnome_vfs_move (temp_filename,
					 bookmarks_filename, TRUE);

		/* make sure the move went okay */
		if (result == GNOME_VFS_OK)
		{
			/* bookmarks have been synched now */
			bookmarks_dirty = FALSE;
		}
		else
		{
			gnome_error_dialog (
				_("Unable to save bookmark file."));
		}
	}

	/* otherwise... */
	else
	{
		gnome_error_dialog (_("Unable to save bookmark file."));
	}

	/* free the temp file stuff */
	gnome_vfs_uri_unref (temp_uri);
	g_free (temp_filename);
}

/**
 * bookmarks_save_as: attempts to write the bookmarks to the given
 * filename, and returns the number of bytes actually written */
int
bookmarks_save_as (const gchar *filename)
{
	xmlNodePtr node;
	xmlDocPtr doc;
	int size;
	
	/* build an XML document and save it to the filename */
	doc = xmlNewDoc ("1.0");
	node = xmlNewDocNode (doc, NULL, "bookmarks", NULL);
	xmlDocSetRootElement (doc, node);
	bookmarks_ensure_unique_ids (bookmarks_root);
	bookmarks_save_recursively (doc, node, bookmarks_root);
	size = xmlSaveFile (filename, doc);
	xmlFreeDoc (doc);     

	return size;
}

/**
 * bookmarks_save_recursively: recursively save a bookmarks tree to 
 * the xml file 
 */
static void
bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root, BookmarkItem *b)
{
	BookmarkItem *b2;
	xmlNodePtr node;
	GList *li;
	gchar *idstr;
	GString *s = g_string_new ("");
	gchar *tmp;

	if (b->alias_of)
	{
		idstr = g_strdup_printf ("%ld", b->id);
		node = xmlNewDocNode (doc, NULL, "alias", NULL);
		xmlSetProp (node, "id", idstr);
		if (b->create_toolbar)
			xmlSetProp (node, "create_toolbar", "TRUE");
		if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
		{
			tmp = g_strdup_printf("%d", b->toolbar_style);
			xmlSetProp (node, "toolbar_style", tmp);
			g_free (tmp);
		}
		if (b->create_context_menu)
			xmlSetProp (node, "create_context_menu", "TRUE");
		if (b->accel_key) {
			g_string_sprintf (s, "%d", b->accel_key);
			xmlSetProp (node, "accel_key", s->str);
		}
		if (b->accel_mods) {
			g_string_sprintf (s, "%d", b->accel_mods);
			xmlSetProp (node, "accel_mods", s->str);
		}
		if (b->time_added) {
			g_string_sprintf (s, "%d", b->time_added);
			xmlSetProp (node, "time_added", s->str);
		}
		xmlAddChild (root, node);
	} 
	else
	{
		idstr = g_strdup_printf ("%ld", b->id);
		switch (b->type)
		{
		case BM_FOLDER:
			node = xmlNewDocNode (doc, NULL, "folder", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			if (b->create_toolbar)
				xmlSetProp (node, "create_toolbar", "TRUE");
			if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
			{
				tmp = g_strdup_printf("%d", b->toolbar_style);
				xmlSetProp (node, "toolbar_style", tmp);
				g_free (tmp);
			}
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu", 
					    "TRUE");
			if (b->expanded)
				xmlSetProp (node, "expanded", "TRUE");
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (default_bookmarks_root == b) {
				xmlSetProp (node, "default_bookmarks_root", 
					    "TRUE");
			}
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp (node, "pixmap", b->pixmap_file);
			if (b->time_added) {
				g_string_sprintf (s, "%d", b->time_added);
				xmlSetProp (node, "time_added", s->str);
			}
			if (b->time_modified) {
				g_string_sprintf (s, "%d", b->time_modified);
				xmlSetProp (node, "time_modified", s->str);
			}
			if (b->time_visited) {
				g_string_sprintf (s, "%d", b->time_visited);
				xmlSetProp (node, "time_visited", s->str);
			}
			xmlAddChild (root, node);
			
			for (li = b->list; li != NULL; li = li->next) {
				b2 = li->data;
				bookmarks_save_recursively (doc, node, b2);
		}
			break;
			
		case BM_SEPARATOR:
			node = xmlNewDocNode(doc, NULL, "separator", NULL);
			xmlAddChild(root, node);
			break;

		case BM_SITE:
			node = xmlNewDocNode(doc, NULL, "site", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			xmlSetProp (node, "url", b->url);    
			if (*b->nick)
				xmlSetRawProp (node, "nick", b->nick);
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu",
					    "TRUE");
			if (b->accel_key) {
				g_string_sprintf (s, "%d", b->accel_key);
				xmlSetProp (node, "accel_key", s->str);
			}
			if (b->accel_mods) {
				g_string_sprintf (s, "%d", b->accel_mods);
				xmlSetProp (node, "accel_mods", s->str);
			}
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp (node, "pixmap", b->pixmap_file);
			if (b->time_added) {
				g_string_sprintf (s, "%d", b->time_added);
				xmlSetProp (node, "time_added", s->str);
			}
			if (b->time_modified) {
				g_string_sprintf (s, "%d", b->time_modified);
				xmlSetProp (node, "time_modified", s->str);
			}
			if (b->time_visited) {
				g_string_sprintf (s, "%d", b->time_visited);
				xmlSetProp (node, "time_visited", s->str);
			}
			xmlAddChild(root, node);
			break;
			
		case BM_AUTOBOOKMARKS:
			node = xmlNewDocNode(doc, NULL, "autobookmarks", NULL);
			xmlSetProp (node, "id", idstr);
			xmlSetRawProp (node, "name", b->name);
			if (b->create_toolbar)
				xmlSetProp (node, "create_toolbar", "TRUE");
			if (b->toolbar_style != TOOLBAR_STYLE_HORIZONTAL)
			{
				tmp = g_strdup_printf("%d", b->toolbar_style);
				xmlSetProp (node, "toolbar_style", tmp);
				g_free (tmp);
			}
			if (b->create_context_menu)
				xmlSetProp (node, "create_context_menu", 
					    "TRUE");
			if (b->expanded)
				xmlSetProp (node, "expanded", "TRUE");
			if (*b->notes)
				xmlSetRawProp (node, "notes", b->notes);
			if (b->pixmap_file && *b->pixmap_file)
				xmlSetProp(node, "pixmap", b->pixmap_file);
			xmlAddChild(root, node);
			break;
		}
	}
        g_string_free (s, TRUE);
	g_free (idstr);
}

/**
 * bookmarks_remove_recursively: recursively remove a tree of bookmarks items
 */
void 
bookmarks_remove_recursively (BookmarkItem *b) {
	GList *position;
	
	if (b == NULL) return;

	if (b->parent) {
		position = g_list_find (b->parent->list, b);
		if (position) {
			b->parent->list = 
				g_list_remove_link (b->parent->list, position);
			bookmarks_update_alias (b->parent);
		}
		b->parent = NULL;
	}
	
	if (!b->alias_of) 
		/* When removing a bookmark, we remove all its alias
		   Instead, we should check before removing if it has alias 
		   and warn the user */
		bookmarks_remove_all_alias (b);
	else 
		/* we are removing an alias, connect the aliased bookmark with 
		   it's next, if any */
		b->alias_of->alias = b->alias;
	
	if (BOOKMARK_ITEM_IS_FOLDER (b) && !b->alias_of)
	{
		while (b->list)
		{
			bookmarks_remove_recursively (b->list->data);
		}
	}
	
	if (b == default_bookmarks_root) 
		default_bookmarks_root = bookmarks_root;
	bookmarks_editor_remove_tree_items (b);
	bookmarks_free_bookmark (b);
}

/**
 * bookmarks_new_bookmark: Creates a new allocated BookmarkItem
 * @type: the type of the bookmark
 * @escape_name: TRUE if the name needs to have _'s escaped, 
 *                       FALSE if they already are
 * @name: the name, if NULL the url is used
 * @url: the url (ignored if type != BM_SITE)
 * @nick: the nickname for the bookmark, if any
 * @notes: the comments aout the bookmark
 * 
 * This function allocates a BookmarkItem and initializes its fields to sane 
 * values.
 * Use it instead of g_new (BookmarkItem, 1) and manually initializing the 
 * fields. 
 * The returned BookmarkItem has it's parent set to NULL, so you have to add it
 * to the bookmarks list yourself (if you need it)
 * All the strings passed to this function are g_strduped
 * 
 * Return value: The new BookmarkItem
 **/
BookmarkItem *
bookmarks_new_bookmark (BookmarkType type, gboolean escape_name,
			const char *name, const char *url, 
			const char *nick, const char *notes, 
			const char *pixmap_file)
{
	BookmarkItem *bookmark;

	g_return_val_if_fail (!((type == BM_SITE) && (url == NULL)), NULL);
	g_return_val_if_fail (!((type == BM_FOLDER) && (name == NULL)), NULL);

	bookmark = g_new0 (BookmarkItem, 1);
	bookmark->type = type;

	switch (bookmark->type) {
	case BM_AUTOBOOKMARKS:
	case BM_FOLDER:
		if (escape_name)
			bookmark->name = escape_uline_accel (name);
		else
			bookmark->name = g_strdup (name);
		bookmark->url = NULL;
		break;
	case BM_SITE:
		if (name)
		{
			if (escape_name)
				bookmark->name = escape_uline_accel (name);
			else
				bookmark->name = g_strdup (name);
		}
		else
			bookmark->name = escape_uline_accel (url);
		bookmark->url = g_strdup (url);
		auto_completion_add_url (bookmark->url);
		break;
	default:
		bookmark->name = bookmark->url = NULL;
		break;
	}
	bookmark->list = NULL;
	bookmark->tree_items = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	bookmark->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
	bookmark->create_context_menu = FALSE;
	if (nick) 
		bookmark->nick = g_strdup (nick);
	else
		bookmark->nick = g_strdup ("");
	if (notes)
		bookmark->notes = g_strdup (notes);
	else
		bookmark->notes = g_strdup ("");
	if (pixmap_file != NULL && strlen(pixmap_file) != 0)
	{
		bookmark->pixmap_file = g_strdup (pixmap_file);
		bookmark->pixmap_data = pixmap_data_new_from_file
			(pixmap_file, FALSE);
	}
	else
	{
		bookmark->pixmap_file = g_strdup ("");
		bookmark->pixmap_data = NULL;
	}
	bookmark->expanded = TRUE;
	bookmark->alias = NULL;
	bookmark->alias_of = NULL;
	bookmark->id = maxid++;
	bookmark->time_added = 0;
	bookmark->time_modified = 0;
	bookmark->time_visited = 0;
	
	return bookmark;
}

/**
 * Creates a an alias of a bookmark. If the parameter is NULL, returns bookmark
 * with all fields NULLs, for creating an alias later.
 */
BookmarkItem *
bookmarks_new_alias (BookmarkItem *b)
{
	BookmarkItem *bookmark = g_new0 (BookmarkItem, 1);
	bookmark->tree_items = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	bookmark->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
	bookmark->create_context_menu = FALSE;
	if (b) bookmarks_new_alias_do (b, bookmark);
	return bookmark;
}

/**
 * Fill the fields of and alias
 */
static void
bookmarks_new_alias_do (BookmarkItem *b, BookmarkItem *bookmark)
{
	/* Most are the same as the original bookmark */
	/* actually, they point to the same string, so be cautious */
	bookmark->type = b->type;
	bookmark->name = b->name;
	bookmark->url = b->url;
	bookmark->list = b->list;
	bookmark->nick = b->nick;
	bookmark->notes = b->notes;
	bookmark->pixmap_file = b->pixmap_file;
	bookmark->pixmap_data = b->pixmap_data;
	bookmark->expanded = TRUE; /* does not have sense here */
	/* insert in the list */
	bookmark->alias = b->alias;
	bookmark->alias_of = b;
	bookmark->id = b->id;
	b->alias = bookmark;
}

/**
 * Updates all aliases of a bookmark
 */
void
bookmarks_update_alias (BookmarkItem *b)
{
	BookmarkItem *alias = b->alias;
	if (!alias) return;
	alias->name = b->name;
	alias->url = b->url;
	alias->list = b->list;
	alias->nick = b->nick;
	alias->notes = b->notes;
	alias->pixmap_file = b->pixmap_file;
	alias->pixmap_data = b->pixmap_data;
	alias->id = b->id;
	alias->time_visited = b->time_visited;
	alias->time_modified = b->time_modified;
	bookmarks_editor_update_tree_items (alias);
	bookmarks_update_alias (alias);
}

/**
 * Finds the real bookmark of an alias
 */
BookmarkItem *
bookmarks_find_real_bookmark (BookmarkItem *b)
{
	if (!b->alias_of) 
		return b;
	else
		return bookmarks_find_real_bookmark (b->alias_of);
}


/**
 * Sets the real bookmark visited date and ensures it is saved.
 */
void bookmarks_set_visited (BookmarkItem *b)
{
	bookmarks_find_real_bookmark (b)->time_visited = time (NULL);
	bookmarks_dirty = TRUE;
}

/**
 * bookmarks_copy_bookmark: copy a bookmark
 * @b: the bookmark
 * 
 * copy a bookmarks and its children
 * 
 * Return value: the new bookmark
 **/
/* FIXME: This fails miserably wrt aliases */
BookmarkItem *
bookmarks_copy_bookmark (BookmarkItem *b)
{
	GList *l;
	/* TODO: if it is an alias... */
	BookmarkItem *new = bookmarks_new_bookmark (b->type, FALSE, b->name,
						    b->url, b->nick, b->notes,
						    b->pixmap_file);
	
	new->create_toolbar = b->create_toolbar;
	new->toolbar_style = b->toolbar_style;
	new->create_context_menu = b->create_context_menu;
	new->expanded = b->expanded;
	/* should these times just copied here? 
	   should they be updated? */
	new->time_added = b->time_added;
	new->time_modified = b->time_modified;
	new->time_visited = b->time_visited;

	if (BOOKMARK_ITEM_IS_FOLDER (b)) 
	{
		for (l = b->list; l != NULL; l = g_list_next (l)) {
			BookmarkItem *child = l->data;
			BookmarkItem *new_child;
			if (BOOKMARK_ITEM_IS_FOLDER (b) && (b->alias_of))
				continue;
			new_child = bookmarks_copy_bookmark (child);
			new_child->parent = new;
			new->list = g_list_append (new->list, new_child);
		}
	}
	return new;
}

/**
 * bookmarks_free_bookmark: free a bookmark
 * @b: the bookmark to be freed, if NULL thenb do nothing
 * 
 * Frees the bookmarks and all the strings referenced by it and the list 
 * of children if it's a BM_FOLDER (but does not free the children). Use it 
 * instead of g_free. 
 **/
void
bookmarks_free_bookmark (BookmarkItem *b)
{
	if (! b) return;
	
	/* aliases fields point to the real bookmark fields */
	if (!b->alias_of) {
		if (b->name) g_free (b->name);
		if (b->url) g_free (b->url);
		if (b->nick) g_free (b->nick);
		if (b->pixmap_file) g_free (b->pixmap_file);
		if (b->pixmap_data)
		{
			if (b->pixmap_data->pixmap)
				gdk_pixmap_unref (b->pixmap_data->pixmap);
			if (b->pixmap_data->mask)
				gdk_bitmap_unref (b->pixmap_data->mask);

			/* scrub and free pixmap */
			memset (b->pixmap_data, 0, sizeof (PixmapData));
			g_free (b->pixmap_data);
		}
		if (b->notes) g_free (b->notes);
		if (b->list) g_list_free (b->list);
	}
	g_free (b);
}

/**
 * bookmarks_insert_bookmark: insert a bookmark
 * @b: the bookmark to insert
 * @near: a bookmark that should be near @b (a brother or its parent)
 **/
void
bookmarks_insert_bookmark (BookmarkItem *b, BookmarkItem *near,
			   GtkCListDragPos insert_pos)
{
	BookmarkItem *parent;
	GList *position_node;
	gint position;
	
	if (near == NULL)
	{
		parent = bookmarks_root;
	}
	else {
		if (insert_pos == GTK_CLIST_DRAG_INTO 
		    && BOOKMARK_ITEM_IS_FOLDER (near) 
		    && !near->alias_of)
			parent = near;
		else
			parent = near->parent;
	}
	
	g_assert (parent != NULL);
	b->parent = parent;
	
	position_node = g_list_find (parent->list, near);
	if (!position_node)
		parent->list = g_list_prepend (parent->list, b);
	else {
		position = g_list_position (parent->list, position_node);
		
		switch (insert_pos) {
		case GTK_CLIST_DRAG_NONE:
		case GTK_CLIST_DRAG_BEFORE:
			break;
		case GTK_CLIST_DRAG_INTO:
			if (near == NULL)
				break;
			if (BOOKMARK_ITEM_IS_FOLDER (near))
				position++;
			break;				
		case GTK_CLIST_DRAG_AFTER:
			position++;
			break;
		default:
			break;
		}
		parent->list = g_list_insert (parent->list, b, position);
	}
	bookmarks_update_alias (parent);

	bookmarks_toolbars_check_update (b);
}

/**
 * bookmarks_move_bookmark: Moves a bookmark
 * @controls: the controls of the editor
 * @b: the bookmark to move
 * @where: if 0 then move up, else move down
 * Returns: 1 if bookmark is now first or last in the list, 0 otherwise
 *
 * Moves a bookmark up or down and updates both the edition ctree and the 
 * bookmarks structure 
 **/
gint
bookmarks_move_bookmark (BookmarksEditorControls *controls, 
			 BookmarkItem *b, int where)
{
	BookmarkItem *parent, *grandparent, *other;
	GList *pos, *other_pos;
	
	parent = b->parent;
	if (!parent)
		return 1;

	grandparent = parent->parent;
	pos = g_list_find (parent->list, b);

	if (!where) /* up */
		other_pos = g_list_previous(pos);
	else /* down */
		other_pos = g_list_next(pos);
	
	if (other_pos) {
		other = other_pos->data;
		if (other->type == BM_FOLDER) {
			/* expand the folder and move b into it */
			parent->list = g_list_remove_link (parent->list, pos);
			bookmarks_update_alias (parent);
			gtk_ctree_expand (GTK_CTREE (controls->ctree), 
					bookmarks_get_tree_item
					  (other, controls));
			
			if (!where) { 
				/* up */
				other->list = g_list_append (other->list, b);
				b->parent = other;
			} else {
				/* down */
				b->parent = other;
				other->list = g_list_prepend (other->list, b);
			}
		} else {
			gint newindex = g_list_position
				(parent->list, other_pos);
			b->parent->list = g_list_remove_link 
				(b->parent->list, pos);
			b->parent->list = g_list_insert 
				(parent->list, b, newindex);
		}
	} else {
		/* move b to its grandparent*/
		if  (!grandparent)
			return 1;
		
		parent->list = g_list_remove_link(parent->list, pos);
		bookmarks_update_alias (parent);
		
		if (!where) {
			/* up */
			grandparent->list = g_list_insert 
				(grandparent->list, b, g_list_position 
				 (grandparent->list, g_list_find(grandparent->list, 
								 parent)));
			b->parent = grandparent;
		} else {
			/* down */
			GList *next = g_list_find
				(grandparent->list, parent)->next;
			grandparent->list = g_list_insert 
				(grandparent->list, b, 
				 g_list_position 
				 (grandparent->list, next));
			b->parent = grandparent;
		}
	}
	bookmarks_editor_place_tree_items (b);
	bookmarks_update_alias (b->parent);
	gtk_ctree_select ( GTK_CTREE (controls->ctree),
			   bookmarks_get_tree_item (b, controls));

	return 0;
}

/**
 * Renames a bookmark (which has just been added) or deletes it if the 
 * user cancelled
 */
void 
bookmarks_string_request_callback (gchar *string, BookmarkItem *b) 
{
	if (string == NULL) 
	{ 
                /* user canceled */
		bookmarks_remove_recursively (b);
	} 
	else 
	{ 
		g_free (b->name);
		b->name = mozilla_locale_to_utf8 (string);
		g_free (string);
		
		if (strlen (b->name) == 0)
		{
			g_free (b->name);
			b->name = escape_uline_accel ( _("Untitled"));
		}

		if (b->url &&
		    eel_gconf_get_boolean (CONF_BOOKMARKS_FAVICONS_ENABLED))
		{
			if (b->url)
				favicon_get_siteicon (NULL, b, NULL);
		}
		
		bookmarks_editor_place_tree_items (b);
		bookmarks_toolbars_check_update (b);
		bookmarks_updated (); 
	}
}

/**
 * add_bookmark_default: Adds a bookmark to the default bookmark folder
 * @type: The type of the bookmark
 * @name: The name of the bookmark, if NULL or "" the url will be used instead.
 * @url: The url of the bookmark
 * @parent: The parent bookmark or NULL if the bookmark should be added to the 
 * main temp folder
 * Return value: The new bookmark
 **/
BookmarkItem *
add_bookmark_default (BookmarkType type, const gchar *name, const gchar *url, 
		      BookmarkItem *parent)
{
	BookmarkItem *bookmark;

	bookmark = bookmarks_new_bookmark (type, TRUE, name, url, NULL, NULL, NULL);

	if (!parent) parent = default_bookmarks_root;
	
	bookmark->parent = parent;
	parent->list = g_list_append (parent->list, bookmark);
	bookmark->time_added = time (NULL);

	bookmarks_editor_place_tree_items (bookmark);
	bookmarks_updated ();

	return bookmark;
}

/**
 * add_temp_link_bookmark: Adds a temporal bookmark of a link
 * @name: The link text
 * @url: The url of the bookmark
 * @parent: The parent bookmark or NULL if the bookmark should be added to the 
 * main temp folder
 * Return value: The new bookmark
 **/
BookmarkItem *
add_temp_link_bookmark (const gchar *name, const gchar *url, BookmarkItem *parent)
{
	BookmarkItem *bookmark;
	gchar *newname;
	gint shorttext = eel_gconf_get_integer (CONF_BOOKMARKS_SHORTTEXT);

	/**
	 * if shorttext is < 0, use old url only behavior,
	 * if >=0, it is the length above which only the link text will be used
	 */
	
	if (shorttext < 0)
		newname = g_strdup (url);
	else if (strlen(name) <= shorttext)
		newname = g_strdup_printf ("%s - %s", name, url);
	else
		newname = g_strdup (name);

	bookmark = add_bookmark_default (BM_SITE, newname, url, parent);
	g_free (newname);

	return bookmark;
}

/**
 * bookmarks_list_all_urls:
 * -----------------------
 * get a list of all bookmarked URLs
 */
GSList *
bookmarks_list_all_urls (BookmarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == BM_FOLDER, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookmarkItem *b = l->data;
		/* skip alias */
		if (b->alias_of) continue;
		if (b->type == BM_FOLDER) {
			g_slist_concat(result, bookmarks_list_all_urls(b));
		} else
			if (b->type == BM_SITE) {
				g_slist_append(result, b->url);
			}
	}
	return result;
}

/**
 * get a singly linked list of all the folder/folder bookmarks 
 */
GSList * 
bookmarks_list_all_folders (BookmarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == BM_FOLDER, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookmarkItem *b = l->data;
		if (b->type == BM_FOLDER) {
			g_slist_append(result, b);
			if (!b->alias_of) 
				g_slist_concat (result, 
						bookmarks_list_all_folders(b));
		}
	}
	return result;
}


/**
 * bookmarks_parse_nick: returns the url of a given nick with one arg
 * @text: the text to try to parse
 * @root: the root bookmark for start the search, if NULL the usual tree will
 * be used
 * 
 * Tries to find a bookmarkitem wich has a nick like the first word of @text 
 * and if found return the URL result of substituting each %s in the 
 * BookmarkItem url by the 
 * text after the first " " in @text
 * 
 * Return value: the resulting URL or NULL if not matching nick found
 **/
gchar *
bookmarks_parse_nick (gchar *text, BookmarkItem *root)
{
	gchar *nick;
	gchar *url;
	gchar *space;
	BookmarkItem *b;

	/* remove any leading spaces */
	while (*text == ' ')
		text++;
	if (*text == '\0') 
		return NULL;

	/* find the nick */
	space = strstr (text, " ");
	if (space)  {
		/* split out nickname and arguments */
		nick = g_strndup (text, space - text);

		/* find smart bookmark */
		b = bookmarks_find_by_nick (nick, root, TRUE);
		if (b == NULL) /* try for a normal if no smartbm found*/
			b = bookmarks_find_by_nick (nick, root, FALSE);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* do substitution */		
		url = bookmarks_substitute_argument (b, space + 1);

		/* return completed url */
		return url;
	} else {
		/* the whole thing is the nick */
		nick = g_strdup (text);

		/* find normal bookmark */
		b = bookmarks_find_by_nick (nick, root, FALSE);
		if (b == NULL) /* try for a smartbm if no normal found*/
			b = bookmarks_find_by_nick (nick, root, TRUE);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* return copy of the url */
		return g_strdup(b->url);
	}
}

/**
 * bookmarks_substitute_argument:
 **/
gchar *
bookmarks_substitute_argument (BookmarkItem *b, const gchar *arguments)
{
	gchar *ret = NULL;
	const gchar *s = b->url;
	int expected_args = 0;

	/* count %s in the URL */
	while ((s = strstr(s, "%s"))) {
	    ++expected_args;
	    ++s;
	}
	
	ret = g_strdup (b->url);

	if (expected_args <= 1) {
	    /*  do the substitution */
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, arguments, s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	}
	else {
	    /* try to substitute matching arguments */
	    gchar **args;
	    int arg = 0;
	    args = g_strsplit(arguments, " ", -1);
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		if (!args[arg])
		    break;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, args[arg++], s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	    g_strfreev (args);
	}

	return ret;
}

/**
 * bookmarks_find_by_nick: search for a BookmarkItem with a nick
 * @nick: the nick to search for.
 * @r: the root bookmark for start the search, if NULL the usual tree will be
 * used
 * @wantsmart: if we are trying to find a smart bookmark or not
 * 
 * Search a bookmarks folder recursivley for a bookmark with the given nick
 * 
 * Return value: the found BookmarkItem or NULL if nothing found
 **/
BookmarkItem *
bookmarks_find_by_nick (const gchar *nick, BookmarkItem *r, gboolean wantsmart)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;

	if (((r->nick) && r->nick[0])
	    && (wantsmart ^ (r->url == NULL || strstr(r->url,"%s") == NULL)))
	{
		/* maybe the bookmark defines several nicks, separated
		 * by commas or spaces */
		gchar **nicks = g_strsplit (r->nick, ", ", 999);
		gint i;
		gboolean found = FALSE;
		for (i = 0; nicks[i] != NULL; i++)
			if (!strcmp (nicks[i], nick))
			{
				found = TRUE;
				break;
			}
			g_strfreev (nicks);
			if (found) return r;
	}

	if ((r->type == BM_FOLDER) && !r->alias_of)
	{
		for (l = r->list; l != NULL; l = g_list_next (l))
		{
			b = bookmarks_find_by_nick (nick, l->data, wantsmart);
			if (b) return b;
		}
	}
	return NULL;
}

/**
 * bookmarks_save_as: convert a bookmark to a string
 * @bookmark: the bookmark to be saved (can be a folder)
 *
 * Return value: the bookmarks as a string.
 **/
gchar * 
bookmarks_item_to_string (BookmarkItem *bookmark)
{
	xmlNodePtr node;
	xmlDocPtr doc;
	xmlChar *mem;
	int size;

	doc = xmlNewDoc ("1.0");
	node = xmlNewDocNode (doc, NULL, "single_bookmark", NULL);
	bookmarks_ensure_unique_ids (bookmark);
	bookmarks_save_recursively (doc, node, bookmark);
	xmlDocSetRootElement (doc, node);
	xmlDocDumpMemory (doc, &mem, &size);
	xmlFreeDoc (doc);

	return mem;
}

/**
 * bookmarks_item_from_string: Read a bookmark from a string
 * @string: the string to parse.
 * Return value: The new allocated bookmark or NULL if error
 **/
BookmarkItem *
bookmarks_item_from_string (char *string)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookmarkItem *b = NULL;

	doc = xmlParseMemory (string, strlen (string));	
	
	if (doc)
	{
		item = doc->root;
		b = bookmarks_xml_read (item);
		xmlFreeDoc (doc);
	} 
	else
	{
		g_warning ("Error parsing: %s", string);
	}
	return b;
}

/**
 * bookmarks_translate_string
 *   Converts non-alphanumeric characters in a smart bookmark string
 *   to the form %{hex key code}
 *   The returned string should be freed.
 **/
/* FIXME: audit this for utf8 */
gchar *
bookmarks_translate_string (gchar *smart_bm_string)
{
	gchar *translated_string, *tstring_ptr;
	guchar curr_char;
	gchar key_code[4];

	/* each translated character will have the form %XX, so we'll
	   allocatedenough space for 3*current string length */
	translated_string = g_malloc0(3*strlen(smart_bm_string)+1);
	tstring_ptr = translated_string;

	while (*smart_bm_string)
	{
		curr_char = *(unsigned char *)smart_bm_string;
		if ( (curr_char >= 'a' && curr_char <= 'z') ||
		     (curr_char >= 'A' && curr_char <= 'Z') ||
		     (curr_char >= '0' && curr_char <= '9') )
			*tstring_ptr++ = curr_char;
		else if (curr_char == ' ')
			*tstring_ptr++ = '+';
		else
		{
			sprintf(key_code, "%%%02X", curr_char);
			strcat(translated_string, key_code);
			tstring_ptr += 3;
		}
		smart_bm_string++;
	}

	return (translated_string);
}

/**
 * bookmarks_folder_open_all_items: Loads all bookmarks in a folder 
 **/
void
bookmarks_folder_open_all_items (GaleonEmbed *embed, BookmarkItem *bi,
				 gboolean new_window, gboolean reuse)
{
	GList *l = bi->list;
	BookmarkItem *child = NULL;

	while (l) {
		child = l->data;
		if (child->url)
			break;
		l = l->next;
	}

	/* reuse the first tab/window if required */
	if (reuse && l && child->url)
	{
		embed_load_url (embed, child->url);
		l = l->next;
	}

	/* open all the rest of the items */
	while (l)
	{
		BookmarkItem *child = l->data;
		if (child->url)
		{
			embed_create_from_url (embed, child->url, TRUE,
					       new_window);
		}
		l = l->next;
	}
}

/**
 * bookmarks_add_completion_recursively: recursively add bookmarks to the
 * auto completions table
 */
static void bookmarks_add_completion_recursively (BookmarkItem *b)
{
	GList *item;
  
	if (b->alias_of)
		return; /* the real one should have been added already */
	
	switch (b->type)
	{
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		for (item = b->list; item != NULL; item = item->next)
		{
			bookmarks_add_completion_recursively (item->data);
		}
		break;

	case BM_SITE:
		auto_completion_add_url (b->url);
		break;

	case BM_SEPARATOR:
		break;
	}

}

/**
 * Removes all alias of a bookmark 
 */
static void 
bookmarks_remove_all_alias (BookmarkItem *b)
{
	BookmarkItem *alias = b->alias;
	if (alias) {
		bookmarks_remove_all_alias (alias);
		bookmarks_remove_recursively (alias);
		b->alias = NULL;
	}
}

/**
 * Return the bookmark with the given id
 */
static BookmarkItem *bookmarks_find_by_id (long id, BookmarkItem *r)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;
	
	/* Don't return bookmarks that may be not yet created aliases */
	if ((r->id == id) && (r->name != NULL)) 
		return bookmarks_find_real_bookmark (r);
	if (BOOKMARK_ITEM_IS_FOLDER (r) && !r->alias_of)
		for (l = r->list; l != NULL; l = g_list_next (l)) {
			b = bookmarks_find_by_id (id, l->data);
			if (b) return b;
		}
	return NULL;
}

/**
 * bookmarks_get_siteicon: Parse an http URL ans return the appropriate icon 
 */
const PixmapData *
bookmarks_get_siteicon (const gchar *url)
{
	GnomeVFSURI *vfs_uri, *vfs_parent_uri, *favicon_uri;
	gchar *favicon_url, *favicon_path;
	PixmapData *result = NULL;

	if (url == NULL)
	{
		return site_pixmap_data;
	}
	
	if (favicons_cache != NULL)
	{
		result = g_hash_table_lookup (favicons_cache, url);
		/* perhaps we need to add a reference to the GdkPixmap
		   and the GdkBitmap? */
		if (result != NULL) return result;
	}
	
	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri == NULL)
	{
		return site_pixmap_data;
	} 

	if (gnome_vfs_uri_has_parent (vfs_uri))
	{
		vfs_parent_uri = gnome_vfs_uri_get_parent (vfs_uri);
		gnome_vfs_uri_unref (vfs_uri);
	} 
	else
	{
		vfs_parent_uri = vfs_uri;
	}
	favicon_uri = gnome_vfs_uri_append_file_name (vfs_parent_uri, 
						      "favicon.ico");
	favicon_url = gnome_vfs_uri_to_string (vfs_parent_uri, 0);

	gnome_vfs_uri_unref (vfs_parent_uri);
	if (favicon_uri) gnome_vfs_uri_unref (favicon_uri);
	
	favicon_path = favicon_filename (favicon_url);
	
	if (favicon_path && g_file_exists (favicon_path))
	{
		GdkPixbuf *pixbuf;
		
		pixbuf = gdk_pixbuf_new_from_file (favicon_path);
		if (pixbuf != NULL)
		{
			if (gdk_pixbuf_get_width (pixbuf) > 16 || 
			    gdk_pixbuf_get_height (pixbuf) > 16)
			{
				GdkPixbuf *scaled_icon = 
					gdk_pixbuf_scale_simple (pixbuf, 
						16, 16, GDK_INTERP_NEAREST);
				gdk_pixbuf_unref (pixbuf);
				pixbuf = scaled_icon;
			}

			result = g_new (PixmapData, 1);
			gdk_pixbuf_render_pixmap_and_mask (pixbuf,
							   &(result->pixmap),
							   &(result->mask),
							   100);
			gdk_pixbuf_unref (pixbuf);
	
			if (favicons_cache == NULL) 
			{
				favicons_cache = g_hash_table_new (g_str_hash,
								   g_str_equal);
			}
			g_hash_table_insert (favicons_cache, (gchar *)url, 
					     result);
		}
		else
		{
			result = site_pixmap_data;
		}
	}
	else
	{
		result = site_pixmap_data;
	}

	g_free (favicon_url);
	g_free (favicon_path);
	
	return result;
}

GtkCTreeNode *bookmarks_get_tree_item (BookmarkItem *b,
				       BookmarksEditorControls *c)
{
	GList *l;
	for (l = b->tree_items; l; l = l->next) {
		BookmarkTreeItem *ti = l->data;
		if (ti->controls == c) return ti->tree_item;
	}
	return NULL;
}

void bookmarks_set_tree_item (BookmarkItem *b, BookmarksEditorControls *c,
			      GtkCTreeNode *n)
{
	GList *l;
	BookmarkTreeItem *ti;		
	for (l = b->tree_items; l; l = l->next) {
		ti = l->data;
		if (ti->controls == c) {
			if (n)
				ti->tree_item = n;
			else  {
				b->tree_items = g_list_remove_link 
					(b->tree_items, l);
				g_free (ti);
			}	
			return;
		}
	}
	/* not found, add if needed */
	if (n)
	{
		ti = g_new0 (BookmarkTreeItem, 1);
		ti->tree_item = n;
		ti->controls = c;
		b->tree_items = g_list_prepend (b->tree_items, ti);
	}
}

/**
 * Returns TRUE if "a" is an ancestor of "b"
 */
gboolean
bookmarks_is_ancestor (BookmarkItem *a, BookmarkItem *b)
{
	for (; b; b = b->parent)
		if (a == b) return TRUE;
	return FALSE;
}


/**
 * Merges two trees of bookmarks, adding bookmarks in root2 to root1
 * root1 is modified, and root2 is not.
 * This function is intended to be used when importing bookmarks
 * The policies are very simple, they are open to discussion but I wthink 
 * they cover most cases and are asy to implement. They are:
 *    - for categories:
 *            - two categories with the same name at the same level are merged
 *              recursively
 *            - categories in root2 not present in root1 are copied to root1
 *    - for items
 *            - two items are considered equal if they have the same url
 *            - items not present in root1 are copied
 *    - for separators
 *            - they are added only if the previous item was added too, unless
 *              a whole folder is imported
 *    - for aliases
 *            - aliases are copied as if they were normal bookmarks 
 *              (for simplicity)
 */
void
bookmarks_merge_trees (BookmarkItem *root1, BookmarkItem *root2)
{
	GList *li;
	gboolean last_one_added = FALSE;
	for (li = root2->list; li != NULL; li = li->next) {
		BookmarkItem *item = li->data;
		BookmarkItem *item2 = bookmarks_merge_trees_look_for_equal
			(root1, item);
		if (item2 == NULL) {
			if ((item->type != BM_SEPARATOR) || last_one_added) {
				BookmarkItem *b = bookmarks_copy_bookmark
					(item);
				root1->list = g_list_append (root1->list, b);
				b->parent = root1;
				last_one_added = TRUE;
			}
		} else if (BOOKMARK_ITEM_IS_FOLDER (item2)) {
			if (!(item2->alias_of || item->alias_of))
				bookmarks_merge_trees (item2, item);
			last_one_added = FALSE;
		} else {
			last_one_added = FALSE;
		}
	}	
}

static BookmarkItem *
bookmarks_merge_trees_look_for_equal (BookmarkItem *folder, 
				      BookmarkItem *bookmark)
{
	GList *li;
	if (bookmark->type == BM_SEPARATOR) return NULL;
	for (li = folder->list; li != NULL; li = li->next) {
		BookmarkItem *item = li->data;
		if ((item->type == BM_SITE) 
		    && (bookmark->type == BM_SITE)) {
			if (!strcmp (item->url, bookmark->url)) 
				return item;
		} else if ((BOOKMARK_ITEM_IS_FOLDER (bookmark))
			   && (BOOKMARK_ITEM_IS_FOLDER (item))) {
			if (!strcmp (item->name, bookmark->name)
			    && (!(item->alias_of || bookmark->alias_of)
				|| (item->alias_of && bookmark->alias_of))) 
				return item;
		}
	}
	return NULL;
}


void bookmarks_sort_norec (BookmarkItem *b, gboolean folders_first, 
			   gboolean update_edit)
{
	if (BOOKMARK_ITEM_IS_FOLDER (b)) {
		if (!b->alias_of) {
			b->list = g_list_sort 
				(b->list, folders_first 
				 ? bookmarks_compare_name_and_type
				 : bookmarks_compare_name);
			bookmarks_update_alias (b);
		}
		if (update_edit)
			bookmarks_editor_place_tree_items (b);
	}
}

void bookmarks_sort_rec (BookmarkItem *b, gboolean folders_first,
			 gboolean update_edit)
{
	GList *l;
	bookmarks_sort_norec (b, folders_first, FALSE);
	if (BOOKMARK_ITEM_IS_FOLDER (b))
		for (l = b->list; l != NULL; l = l->next) {
			BookmarkItem *b2 = l->data;
			if (!b2->alias_of) {
				bookmarks_sort_rec (b2, folders_first, FALSE);
			}
		}
	if (update_edit)
		bookmarks_editor_place_tree_items (b);

}

static gint 
bookmarks_compare_name (gconstpointer a, gconstpointer b)
{
	BookmarkItem *b1 = (BookmarkItem *) a;
	BookmarkItem *b2 = (BookmarkItem *) b;

	return g_strcasecmp (b1->name, b2->name);
}

static gint 
bookmarks_compare_name_and_type (gconstpointer a, gconstpointer b)
{
	BookmarkItem *b1 = (BookmarkItem *) a;
	BookmarkItem *b2 = (BookmarkItem *) b;

	if (BOOKMARK_ITEM_IS_FOLDER (b1) && !BOOKMARK_ITEM_IS_FOLDER (b2))
		return -1;
	if (!BOOKMARK_ITEM_IS_FOLDER (b1) && BOOKMARK_ITEM_IS_FOLDER (b2))
		return 1;

	return g_strcasecmp (b1->name, b2->name);
}

/**
 * bookmarks_folder_open_in_myportal: Open the MyPortal page for a category
 **/
void
bookmarks_folder_open_in_myportal (GaleonEmbed *embed, BookmarkItem *bi,
				   gboolean new_window, gboolean reuse)
{
	gchar *url;
	gchar *path;

	g_return_if_fail (bi != NULL);

	if (!BOOKMARK_ITEM_IS_FOLDER (bi))
	{
		bi = bi->parent;
	}

	path = bookmarks_get_bookmark_path (bi);
	url = g_strconcat ("myportal:", path, NULL);
	g_free (path);

	if (reuse)
	{
		embed_load_url (embed, url);
	}
	else 
	{
		embed_create_from_url (embed, url, TRUE,
				new_window);
	}
	g_free (url);
}

/**
 * Returns the path of the bookmark (for MyPortal)
 */
static gchar *
bookmarks_get_bookmark_path (BookmarkItem *b)
{
	gchar *stripped = strip_uline_accel (b->name);
	gchar *escaped = escape_path (stripped);
	gchar *path;
	if ((b->parent == NULL) || (b->parent == bookmarks_root)) 
	{
		path = g_strconcat ("/", escaped, NULL);
	}
	else
	{
		gchar *parent = bookmarks_get_bookmark_path (b->parent);
		path = g_strconcat (parent, "/", escaped, NULL);
		g_free (parent);
	}
	g_free (escaped);
	g_free (stripped);
	return path;
}

/**
 * Makes sure that all the bookmarks descending from root have unique ids.
 * The ids may change even if not necesary (because they are assigned in 
 * growing order). Aliases are updated.
 */
static void
bookmarks_ensure_unique_ids (BookmarkItem *root)
{
	gulong maxid = root->id;
	bookmarks_ensure_unique_ids_rec (root, &maxid);
}

static void
bookmarks_ensure_unique_ids_rec (BookmarkItem *b, gulong *maxid)
{
	GList *li;
	for (li = b->list; li != NULL; li = li->next)
	{
		BookmarkItem *bi = li->data;

		/* aliases are updated when they real bookmarks is updated */
		if (bi->alias_of) continue;
		
		if (bi->id <= *maxid)
		{
			bi->id = ++(*maxid);
			bookmarks_update_alias (bi);
		}
		else
		{
			*maxid = bi->id;
		}

		if (BOOKMARK_ITEM_IS_FOLDER (bi))
		{
			bookmarks_ensure_unique_ids_rec (bi, maxid);
		}
	}
}

