/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "window.h"
#include "embed.h"
#include "misc.h"
#include "prefs.h"
#include "dialog.h"
#include "favicon.h"
#include "filepicker.h"
#include "mozilla.h"

#include <string.h>
#include <time.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gdk/gdkkeysyms.h>

void bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window);
void bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, 
					 BookmarksEditorControls *controls);
static void bookmarks_import_netscape_style (GaleonWindow *window,
					     GString *fname,
					     gboolean use_locale);
void bookmark_export_netscape_message (GaleonWindow *window, gchar *fname);
static void bookmark_export_netscape_do (gchar *fname, gboolean use_locale);
static void bookmarks_import_xbel_style (GaleonWindow *window,
					 const gchar *fname);
static void bookmarks_import_own_style (GaleonWindow *window, 
					const gchar *fname);

/* from bookmarks_menus.c */
extern gint bookmarks_block_accel_update;

static void 
bookmarks_handle_link_clicked (GaleonWindow *window, BookmarkItem *bi,
				LinkState state)
{
	g_return_if_fail (bi != NULL);
	g_return_if_fail (bi->type == BM_SITE);
	bookmarks_set_visited (bi);
	if (strstr(bi->url, "%s"))
	{
		bookmarks_smart_bm_dialog (window->active_embed, NULL,
					   bi, state);
	}
	else
	{
		embed_activate_link (window->active_embed, NULL, 
				     bi->url, state);
	}
	embed_grab_focus (window->active_embed);
}


/**
 * Open bookmark in same window if it's a site and the user
 * clicked with button 1
 * Open the bookmark in a new window/tab if it's a site and the user 
 * clicked with button 2
 * Open all the bookmarks in the folder if user clicked with button 2
 * on a folder
 */
gboolean
bookmarks_button_release_event_cb (GtkWidget *widget, GdkEventButton *event,
				   BookmarkItem *bi)
{
	GaleonWindow *window =  window_from_widget (widget);
	static gboolean handled = FALSE;
	GtkWidget *menu;

	return_val_if_not_window (window, FALSE);

	/* don't handle button 3 */
	if (event->button == 3) return FALSE;

	/* Check to see if we're supposed to handle this event */
	/* These checks are here just to make sure tearoff menus get
	   activated correctly */
	if (handled)
	{
		handled = FALSE;
		return FALSE;
	}

	/* If bi is NULL, don't process the event */
	if (!bi)
	{
		handled = TRUE;
		return FALSE;
	}

	/* Get the submenu widget */
	if (GTK_IS_MENU(widget))
		menu = widget;
	else
		menu = widget->parent;

	if (bi->type == BM_SITE)
	{
		if (GTK_IS_MENU_SHELL (menu))
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
		bookmarks_handle_link_clicked (window, bi, 
				mouse_state_to_link_state (event->button,
							   event->state));
	}
	else if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_FOLDER)
	{
		/* FIXME: I've had to remove the following behaviour
		 * because of an odd problem: when selecting a bookmarks
		 * folder, and then clicking the middle button *not* on
		 * folder or any item in it, it was still getting the
		 * callback and ending up here. This was causing people
		 * some major grief, so I've disabled it for now until
		 * someone can debug it properly. -- MattA 26/04/2001 */
#if 0
		/* A middle click is handled the same way when clicking on
		   the menuitem or directly on a bookmark folder. In tabbed
		   mode, a new window will be opened with new tabs for each
		   bookmark. In new window mode, a new window will be opened
		   for each bookmark.  */
		if ((event->button == 2))
		{
			gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
			if (event->state & GDK_SHIFT_MASK)
				tabbed_mode = !tabbed_mode;
			bookmarks_folder_open_all_items (window->active_embed,
							 bi, !tabbed_mode,
							 FALSE);
		}
#endif
	}
	return TRUE;
}

/**
 * Handle bookmarks menus right click context menus
 */
gboolean
bookmarks_button_press_event_cb (GtkWidget *widget, GdkEventButton *event,
				 BookmarkItem *bi)
{
	if (event->button == 3)
	{
		GaleonWindow *window = window_from_widget (widget);
		GtkWidget *popup = NULL;
		return_val_if_not_window (window, FALSE);
		g_return_val_if_fail (bi != NULL, FALSE);

		if (bi->type == BM_AUTOBOOKMARKS || bi->type == BM_FOLDER)
		{
			popup = gtk_menu_new ();
			bookmarks_menu_create_extra_items (GTK_MENU (popup),
					bi, EXTRA_ITEM_LOCATION_EXCLUSIVE);
		}
		else if (bi->type == BM_SITE)
		{
			popup = gtk_menu_new ();
			bookmarks_item_create_extra_items (GTK_MENU (popup),
					bi);
		}
		if (popup)
		{
/* This is a hack to see if we are a descendant of the main bm menu, or of a bm
 * toolbar.  If we are from the bm menu, we must deactivate the root bookmark
 * menu since the current one may be destroyed by some of the context menu
 * actions.  If we are from the bm toolbar, we must deactivate the current
 * menu, since it isn't a real menu but a context, it won't get destroyed if
 * the bm toolbar updates anyway - donut */
			GtkWidget *root = widget;
			while (root){
				root = (GTK_IS_MENU (root) ? 
					(gtk_menu_get_attach_widget 
							(GTK_MENU (root))) :
					(root->parent));
				if (root == window->bookmarks_menu)
					break;
			}
			if (!root)
			{
				if (GTK_IS_MENU(widget))
					root = widget;
				else
					root = widget->parent;
			}

			/* warning: ugly code */
			gtk_object_set_data(GTK_OBJECT(popup), "GaleonWindow",
					    window);
			gnome_popup_menu_do_popup_modal (popup, NULL,
							 NULL, event, NULL);
			gtk_widget_unref (GTK_WIDGET(popup));
			gtk_menu_shell_deactivate (GTK_MENU_SHELL (root));

			/* deactivate the bookmark menu too, may be neccesary
			   in some cases if root != window->bookmarks_menu */
			gtk_menu_shell_deactivate (GTK_MENU_SHELL 
						   (window->bookmarks_menu));

			/* stop event from causing ui freezage/weirdness */
			gtk_signal_emit_stop_by_name (GTK_OBJECT(widget),
					"button-press-event");
			return TRUE;
		}
	}
	return FALSE;
}

/**
 * handle "standard" activation of bookmarks (eg, with keyboard)
 */
void
bookmarks_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, 0);
}

void
bookmarks_open_in_new_tab_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, LINKSTATE_NEWTAB);
}

void
bookmarks_open_in_new_window_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);

	bookmarks_handle_link_clicked (window, bi, LINKSTATE_NEWWIN);
}

void
bookmarks_copy_link_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	g_return_if_fail (bi->type == BM_SITE);

	copy_to_clipboard (bi->url, window->active_embed);
}

void
bookmarks_add_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	g_return_if_fail (bi != NULL);
	bookmarks_file_bookmark_cb (widget, bi);
}

void
bookmarks_add_folder_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	BookmarkItem *new;
	GaleonWindow *par_gwin = window_from_widget (widget);
	GtkWidget *par_win = par_gwin ? par_gwin->WMain : NULL;
	gchar* new_folder_name = NULL;
	
	g_return_if_fail (bi != NULL);
	g_return_if_fail ((bi->type == BM_FOLDER) 
			  || (bi->type == BM_AUTOBOOKMARKS));

	new_folder_name = mozilla_locale_to_utf8(_("New folder"));
	new = bookmarks_new_bookmark (BM_FOLDER, TRUE, new_folder_name,
				      NULL, NULL, NULL, NULL);
	g_free(new_folder_name);
	new->time_added = time (NULL);

	bi->list = g_list_append (bi->list, new);
	new->parent = bi;

	bookmarks_dirty = TRUE;
	bookmarks_toolbars_check_update (new);

	gnome_request_dialog 
		(FALSE, _("Please enter the name of the folder"),
		 _("New folder"), 100,
		 GTK_SIGNAL_FUNC (bookmarks_string_request_callback),
		 new, par_win ? GTK_WINDOW (par_win) : NULL);
}

void
bookmarks_open_all_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, FALSE, TRUE);
}

void
bookmarks_open_all_release_cb (GtkWidget *widget, GdkEventButton *event,
			       BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);
	GtkMenu *menu = GTK_MENU (widget->parent);

	return_if_not_window (window);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
	
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, FALSE, (event->button != 2));
}

void
bookmarks_open_all_in_wins_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, TRUE, TRUE);
}

void
bookmarks_open_all_in_wins_release_cb (GtkWidget *widget, GdkEventButton *event,
			               BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);
	GtkMenu *menu = GTK_MENU (widget->parent);

	return_if_not_window (window);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
	
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_all_items (window->active_embed,
					 bi, TRUE, (event->button != 2));
}

void
bookmarks_create_toolbar_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	g_return_if_fail (bi != NULL);
	bookmarks_toolbars_check_update (bi);
	bi->create_toolbar = !bi->create_toolbar;
	/* called twice intentionally */
	bookmarks_toolbars_check_update (bi);
	bookmarks_updated ();
}

void
bookmarks_set_as_default_activate_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GList * tree_items;
	gint i = 2;
	g_return_if_fail (bi != NULL);
	
	bookmarks_toolbars_check_update (default_bookmarks_root);
	tree_items = default_bookmarks_root->tree_items;
	while (i > 0)
	{
		PixmapData *pd = (i == 1? default_folder_pixmap_data :
					  folder_pixmap_data);
		PixmapData *pdo = (i == 1? default_folder_open_pixmap_data :
					   folder_open_pixmap_data);
		while (tree_items != NULL)
		{
			BookmarkTreeItem *bm_tree_item = tree_items->data;
			GtkCTreeRow *tree_item;
			PixmapData *pd_draw;
				
			g_return_if_fail (bm_tree_item != NULL);
			tree_item = bm_tree_item->tree_item->list.data;

			tree_item->pixmap_opened = pdo->pixmap;
			tree_item->mask_opened = pdo->mask;
			tree_item->pixmap_closed = pd->pixmap;
			tree_item->mask_closed = pd->mask;
			
			pd_draw = (tree_item->expanded ? pdo : pd);
		
			gtk_ctree_node_set_pixtext (
				GTK_CTREE (bm_tree_item->controls->ctree),
					   bm_tree_item->tree_item, 0,
					   default_bookmarks_root->name, 1,
					   pd_draw->pixmap, pd_draw->mask);
			tree_items = tree_items->next;
		}
		tree_items = bi->tree_items;
		i--;
		default_bookmarks_root = bi;
		bookmarks_toolbars_check_update (bi);
	}
	bookmarks_updated ();
}

void
bookmarks_regenerate_autobookmarks_cb (GtkWidget *widget, BookmarkItem *bi)
{
	autobookmarks_generate ();
}

/**
 * bookmarks_toolbar_menu_edit_cb: called by the context menu on the toolbar
 * and menus.  Open the edit bookmarks dialog and select the rigth folder.
 */
void
bookmarks_edit_activate_cb (GtkWidget *menuitem, BookmarkItem *bm)
{
	BookmarksEditorControls *controls;
	GtkWidget *popup;
	GaleonWindow *window;

	/* get GaleonWindow */
	popup = gtk_widget_get_toplevel (menuitem); 
	window = (GaleonWindow *) gtk_object_get_data (GTK_OBJECT(popup), 
						       "GaleonWindow");

	controls = bookmarks_editor_show_dialog (window);
	bookmarks_editor_select_bookmark (controls, bm);
}

/**
 * bookmarks_toolbar_menu_remove_cb: called by the remove context menu on the
 * toolbar and menus. Remove the item.
 */
void
bookmarks_remove_activate_cb (GtkWidget *menuitem, 
			      BookmarkItem *bm)
{
	gboolean autobookmarks_deleted = FALSE;
	
	GaleonWindow *window = window_from_widget (menuitem);

	/* Make sure that the bookmarks menu has disapeared from the
	   screen, otherwise the menu can hide the dialog (and it would be 
	   very bad) */
	if (window) 
	{
		GtkWidget *item = gtk_menu_get_attach_widget 
			(GTK_MENU (window->bookmarks_menu));		
		gtk_menu_popdown (GTK_MENU (window->bookmarks_menu));
		gtk_menu_item_deselect (GTK_MENU_ITEM (item));
		/* the menubar */ 
		if (GTK_IS_MENU_SHELL (item->parent))
			gtk_menu_shell_deactivate 
				(GTK_MENU_SHELL (item->parent));
	}

	if (bm->list /* don't ask about deleting an empty folder or an alias */
	    && !bm->alias_of
	    && !dialog_ask_question 
	    (_("Removing this category will remove all its children.\n"
	       "Are you sure that you want to continue?"),
	     window->WMain))
	{
		return;
	}

	bookmarks_toolbars_check_update (bm);
	if (bm->type == BM_AUTOBOOKMARKS)
	{
		autobookmarks_deleted = TRUE;
	}			
	bookmarks_editor_remove_tree_items (bm);
	bookmarks_remove_recursively (bm);
	if (autobookmarks_deleted)
	{
		autobookmarks_root = NULL;
	}
	bookmarks_updated ();
}

void
bookmarks_file_bookmark_cb (GtkWidget *widget, BookmarkItem *p)
{
	gchar *loc_str;
	BookmarkItem *new;	
	GaleonEmbed *embed;
	GaleonWindow *window;

	window = window_from_widget (widget);
	return_if_not_window (window);

	embed = window->active_embed;
	return_if_not_embed (embed);

	new = bookmarks_new_bookmark (BM_SITE, TRUE, embed->site_title_utf8, 
				      embed->site_location, NULL, NULL, NULL);
	new->time_added = time (NULL);

	p->list = g_list_append (p->list, new);
	new->parent = p;
	bookmarks_dirty = TRUE;
	bookmarks_toolbars_check_update (new);

	if (eel_gconf_get_integer(CONF_BOOKMARKS_TITLE) == 
	    BOOKMARK_PAGE_TITLE)
	{
		if (new->url && 
		    eel_gconf_get_boolean (CONF_BOOKMARKS_FAVICONS_ENABLED))
			favicon_get_siteicon (embed, new, NULL);

		bookmarks_editor_place_tree_items (new);
		bookmarks_toolbars_check_update (new);
		bookmarks_updated ();
	} else {
		loc_str = mozilla_utf8_to_locale (new->name);
		gnome_request_dialog 
			(FALSE, _("Please enter the name of the bookmark"),
			 loc_str, 100,
			 GTK_SIGNAL_FUNC (bookmarks_string_request_callback),
			 new, GTK_WINDOW (window->WMain));
		g_free (loc_str);
	}
}

/* import given netscape-compatible bookmark filename with pass/fail dialog */
static void
bookmarks_import_netscape_style (GaleonWindow *window, GString *fname,
				 gboolean use_locale)
{
	/* NB: used to use gnome_app_message here, but there's not a 
	 * good choice of window to associate the message with, so 
	 * forcing a dialog seems like a better idea -- MattA 30-12-2000 */
	if (netscape_import_bookmarks(fname, use_locale))
	{
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	else
	{
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	}
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
}

static void
bookmarks_import_xbel_style (GaleonWindow *window, const gchar *fname)
{
	BookmarkItem *imported = xbel_import_bookmarks (fname);
	if (imported) {
		bookmarks_merge_trees (bookmarks_root, imported);
		bookmarks_free_bookmark (imported);
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	} else {
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
	bookmarks_editors_refresh ();
}

static void
bookmarks_import_own_style (GaleonWindow *window, const gchar *fname)
{
	BookmarkItem *imported = bookmarks_load_from ((gchar *) fname);
	if (imported) {
		bookmarks_merge_trees (bookmarks_root, imported);
		bookmarks_free_bookmark (imported);
		gnome_ok_dialog (_("Bookmarks successfully imported/merged"));
	} else {
		gnome_ok_dialog (_("Importing bookmarks failed"));
	}
	bookmarks_updated ();
	bookmarks_editors_refresh ();
}

void
bookmarks_import_netscape_activate_cb (GtkMenuItem *menuitem,
				       GaleonWindow *window)
{
	GString *fname = g_string_new (NULL);

	g_string_assign (fname, g_get_home_dir ());
	g_string_append (fname, NETSCAPE_BOOKMARKS);
       	
	bookmarks_import_netscape_style (window, fname, TRUE);

	g_string_free (fname, TRUE);
}

void
bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GaleonWindow *window)
{
	GString * fname = g_string_new (NULL);
	char *prefs_dir = get_mozilla_prefs_directory ();

	if (!prefs_dir) return;

	g_string_assign (fname, prefs_dir);
	g_string_append (fname, "/bookmarks.html");

	bookmarks_import_netscape_style (window, fname, FALSE);

	g_string_free (fname, TRUE);
	g_free (prefs_dir);
}

void
bookmarks_import_other_activate_cb (GtkMenuItem *menuitem, 
				    BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file, NULL)) {
		GString *temp = g_string_new (file);
		bookmarks_import_netscape_style (NULL, temp, FALSE);
		g_string_free (temp, TRUE);
	}

	g_free (file);
}

void
bookmarks_import_xbel_activate_cb (GtkMenuItem *menuitem, 
				   BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file, NULL))
		bookmarks_import_xbel_style (NULL, file);
	
	g_free (file);
}

void
bookmarks_import_own_activate_cb (GtkMenuItem *menuitem, 
				  BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Import from..."),
			      g_get_home_dir(), NULL, modeOpen,
			      &file, NULL))
		bookmarks_import_own_style (NULL, file);
	
	g_free (file);
}

void
bookmarks_export_netscape_activate_cb (GtkMenuItem *menuitem,
				       GtkWidget *widget)
{
	gchar *fname = g_strconcat (g_get_home_dir(), NETSCAPE_BOOKMARKS,
				    NULL);
       	
	if (dialog_check_overwrite_file (fname, widget))
		bookmark_export_netscape_do (fname, TRUE);

	g_free (fname);
}

void
bookmarks_export_mozilla_activate_cb (GtkMenuItem *menuitem,
				      GtkWidget *widget)
{
	char *fname;
	char *prefs_dir = get_mozilla_prefs_directory();

	fname = g_strconcat (prefs_dir,"/bookmarks.html", NULL);
	if (dialog_check_overwrite_file (fname, widget))
		bookmark_export_netscape_do (fname, FALSE);

	g_free (fname);
	g_free (prefs_dir);
}

void
bookmarks_export_other_activate_cb (GtkMenuItem *menuitem, 
				    BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Export to..."),
			      g_get_home_dir(), NULL, modeSave,
			      &file, NULL))
	{
		if (dialog_check_overwrite_file (file, NULL))
			bookmark_export_netscape_do (file, FALSE);
		g_free (file);
	}
}

void
bookmarks_export_xbel_activate_cb (GtkMenuItem *menuitem, 
				   BookmarksEditorControls *controls)
{
	gchar *file = NULL;

	if (show_file_picker (controls->dialog, _("Export in XBEL as..."),
			      g_get_home_dir (), NULL, modeSave,
			      &file, NULL))
	{
		if (dialog_check_overwrite_file (file, NULL))
			xbel_export_bookmarks (file, bookmarks_root);
		g_free (file);
	}
}

static void 
bookmark_export_netscape_do (gchar *fname, gboolean use_locale)
{
	if (netscape_export_bookmarks (fname, bookmarks_root, use_locale))
		gnome_ok_dialog (_("Exporting bookmarks failed"));
	else 
		gnome_ok_dialog (_("Bookmarks successfully exported"));
}

void
bookmarks_open_in_myportal_cb (GtkWidget *widget, BookmarkItem *bi)
{
	GaleonWindow *window = window_from_widget (widget);

	return_if_not_window (window);
	g_return_if_fail (bi != NULL);
	bookmarks_folder_open_in_myportal (window->active_embed,
			bi, TRUE, TRUE);
}

void
bookmarks_add_accel_cb (GtkWidget *widget, guint accel_signal_id,
			GtkAccelGroup *accel_group, guint accel_key,
			GdkModifierType accel_mods, GtkAccelFlags accel_flags,
			BookmarkItem *bi)
{
	if (bookmarks_block_accel_update > 0) return;

	/* if nothing is to be added, don't do anything */
	if (accel_key == GDK_VoidSymbol || accel_key == 0) return;

	/* check whether bookmarks will need to be updated */
	if (!(bi->accel_key == accel_key && bi->accel_mods == accel_mods))
		bookmarks_dirty = TRUE;

	/* update bookmark */
	bi->accel_key = accel_key;
	bi->accel_mods = accel_mods;

	/* sync */
	bookmarks_menu_sync_accel_labels ();
}

void
bookmarks_remove_accel_cb (GtkWidget *widget, GtkAccelGroup *accel_group, 
			   guint accel_key, GdkModifierType accel_mods, 
			   BookmarkItem *bi)
{
	if (bookmarks_block_accel_update > 0) return;

	/* bookmarks will need to be updated */
	bookmarks_dirty = TRUE;

	/* update bookmark */
	bi->accel_key = 0;
	bi->accel_mods = 0;

	/* sync */
	bookmarks_menu_sync_accel_labels ();
}

void
bookmarks_ensure_accel_cb (GtkWidget *widget, BookmarkItem *bm)
{
	GtkMenu *menu;
	GtkWidget *l;
	gchar *str;

	if (bookmarks_block_accel_update > 0) return;

	g_return_if_fail (GTK_IS_MENU_ITEM (widget));
	g_return_if_fail (GTK_IS_MENU (widget->parent));

	menu = GTK_MENU (widget->parent);

	bookmarks_block_accel_update++;
	gtk_widget_add_accelerator (widget, "activate", menu->accel_group, 
				    bm->accel_key, bm->accel_mods, 
				    GTK_ACCEL_VISIBLE);
	bookmarks_block_accel_update--;
	
	/* update accel label too, 
	 * FIXME: this is not really neccesary 90% of time */
	
	l = gtk_object_get_data (GTK_OBJECT (widget), "accel_label");
	g_return_if_fail (l != NULL);

	str = string_from_accel_data (widget, bm->accel_key, bm->accel_mods);
	gtk_label_set_text (GTK_LABEL (l), str);
	g_free (str);
}

void
bookmarks_remove_dummy_accel_cb (GtkWidget *widget, GtkAccelGroup *accel_group, 
			         guint accel_key, GdkModifierType accel_mods, 
			         BookmarkItem *bi)
{
	if (bookmarks_block_accel_update > 0) return;

	/* update bookmark */
	bi->accel_key = 0;
	bi->accel_mods = 0;
}
