/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "galeon_types.h"
#include "favicon.h"
#include "mozilla.h"
#include "file-operations.h"
#include "bookmarks.h"

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

char *messages [5] = {NULL, N_("Looking for a bookmarks icon"), NULL};

static void favicon_download_completed_cb (gpointer data);

typedef struct
{
	BookmarkItem *b;
	gchar *favicon_path;
	BookmarksEditorControls *editor;
} FaviconInfo;

/**
 * favicon_get_siteicon: Parse an http URL ans get its favicon if there is one
 */
void
favicon_get_siteicon (GaleonEmbed *embed, BookmarkItem *b,
		      BookmarksEditorControls *editor)
{
	gchar *url = b->url;
	GnomeVFSURI *vfs_uri;
	gchar *favicon_url, *favicon_path;
	GList *source_uris = NULL, *target_uris = NULL;
	FaviconInfo *info;

	g_return_if_fail (url != NULL);
	
	vfs_uri = gnome_vfs_uri_new (url);
	if (vfs_uri == NULL) return;
	
	if (embed == NULL || !mozilla_get_favicon_location (embed, &favicon_url))
	{
		GnomeVFSURI *vfs_favicon_uri;
		
		while (gnome_vfs_uri_has_parent (vfs_uri))
		{
			GnomeVFSURI *vfs_parent_uri;
			vfs_parent_uri = gnome_vfs_uri_get_parent (vfs_uri);
			gnome_vfs_uri_unref (vfs_uri);
			vfs_uri = vfs_parent_uri;
		} 
		vfs_favicon_uri = gnome_vfs_uri_append_file_name (vfs_uri, 
								 "favicon.ico");
		favicon_url = gnome_vfs_uri_to_string (vfs_favicon_uri, 0);
		gnome_vfs_uri_unref (vfs_favicon_uri);
	}
	else if (favicon_url == NULL)
	{
		gnome_vfs_uri_unref (vfs_uri);
		return;
	}

	gnome_vfs_uri_unref (vfs_uri);

	favicon_path = favicon_filename (favicon_url);

	source_uris = g_list_append(source_uris, favicon_url);
        target_uris = g_list_append(target_uris, favicon_path);

	info = g_new0 (FaviconInfo, 1);
	info->b = b;
	info->favicon_path = favicon_path;
	info->editor = editor;
	
        file_operations_copy_move (source_uris, target_uris,
                                   FALSE,
                                   messages, 
				   favicon_download_completed_cb,
				   info);
	g_free (favicon_url);
}

static void 
favicon_download_completed_cb (gpointer data)
{
	GdkPixbuf *test;
	FaviconInfo *info = (FaviconInfo *) data;
	BookmarkItem *b = info->b;
	gchar *favicon_path = info->favicon_path;
	
	test = gdk_pixbuf_new_from_file (favicon_path);
	if (test == NULL)
	{
		gnome_vfs_unlink (favicon_path);
	}
	else
	{
		gdk_pixbuf_unref (test);
	}

	bookmarks_dirty = TRUE;
	bookmarks_toolbars_check_update (b);
	if (info->editor != NULL)
		bookmarks_editor_update_tree_item (info->editor, b);
	else
		bookmarks_updated ();

	g_free (favicon_path);
	g_free (info);
}

gchar *
favicon_filename (gchar *favicon_url)
{
	GnomeVFSURI *vfs_uri = gnome_vfs_uri_new (favicon_url);
	gchar  *result;

	if (vfs_uri == NULL) return NULL;
	
	/* this is the RIGHT thing to do, but it breaks for the link rel icons
	gchar *slashpos, *filename;
	filename = gnome_vfs_uri_to_string (vfs_uri, 
					    GNOME_VFS_URI_HIDE_USER_NAME |
					    GNOME_VFS_URI_HIDE_PASSWORD |
					    GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);

	while ((slashpos = strstr (filename, "/")) != NULL) *slashpos = '_';

	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      filename, NULL);
	g_free (filename);
	*/
	
	/* some braindamaged handling until the situation is fixed. FIXME!! 
	 * this leaves the filename as tld_favicon.ico, which isnt right
	 * but will work until we decide to store the favicon remote location
	 * in the bookmarks file */
	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      gnome_vfs_uri_get_host_name (vfs_uri), "_",
			      "favicon.ico", NULL);
	gnome_vfs_uri_unref (vfs_uri);

	return result;
}

