/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "window.h"
#include "embed.h"
#include "mime.h"
#include "bookmarks.h"
#include "mozilla.h"
#include "prefs.h"
#include "session.h"
#include "toolbar.h"
#include "prefs.h"
#include "themes.h"

#include <gtk/gtkfilesel.h>
#include <gtk/gtkvseparator.h>
#include <gtk/gtkhseparator.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-dialog-util.h>

/* saved open URL dialog string */
gchar *open_url_str = NULL;

/* local function prototypes */
static gint radio_menuitem_index (GtkRadioMenuItem *item);

/**
 * closes a popup menu with a popup menus child item is selected
 */
gint popup_selection_done_cb (GtkMenuShell *menushell, GtkWidget *popup)
{
	gtk_menu_shell_deactivate (GTK_MENU_SHELL (popup));
	return FALSE;
}

void			
open_url_handle_link (GaleonWindow *window, LinkState state)
{
	GtkWidget *entry;
	gchar *string;

	return_if_not_window (window);
	entry = window->open_url_entry;
	g_return_if_fail (entry != NULL);

	string = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (string)
	{
		gchar *string2;

		g_free (open_url_str);
		open_url_str = string;

		string2 = bookmarks_parse_nick (string, NULL);
		if (string2) 
		{
			string = string2;
		} 
		embed_activate_link (window->active_embed, NULL, 
				     string, state);
		g_free(string2);
	}

	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (entry)));
}

/** 
 * open_url_ok_button_release_event_cb: open url dialog ok button pressed
 */
gboolean
open_url_ok_button_release_event_cb (GtkButton *button, GdkEventButton *event,
				     GaleonWindow *window)
{
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	open_url_handle_link(window,
		     mouse_state_to_link_state (event->button, event->state));
	return TRUE;
}

/** 
 * open_url_clear_button_clicked_cb: open url dialog clear button clicked
 */
void
open_url_clear_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *entry = window->open_url_entry;

	g_return_if_fail (entry != NULL);

	gtk_editable_delete_text (GTK_EDITABLE (entry), 0, -1);
}

/** 
 * open_url_cancel_button_clicked_cb : open url dialog cancel button clicked
 */
void
open_url_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/**
 * history_menu_menuitem_activate_cb: go to a selected item in the 
 * back/forward history menu 
 */
void
history_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint index;

	/* fetch data */
	index = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	mozilla_session_history_go (embed, index);
}
/**
 * history_menu_menuitem_button_release_cb: go to a selected item in the 
 * up directory level menu 
 */
gboolean
history_menu_menuitem_button_release_cb (GtkMenuItem *menuitem,
					 GdkEventButton *event, 
					 gpointer user_data)
{
	GaleonEmbed *embed;
	gint index;
	char *url;
	/* Get the submenu widget */
	GtkWidget *menu = GTK_WIDGET (menuitem)->parent; 

	if ((event->button == 1 && !(event->state & GDK_SHIFT_MASK)) ||
			event->button == 3)
		return FALSE;

	/* fetch data */
	index = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_val_if_not_embed (embed, FALSE);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
	url = mozilla_session_history_get_url (embed, index);
	embed_activate_link_mouse (embed, NULL, url, event);
	if (url != NULL) free (url);
	return TRUE;
}

/**
 * up_menu_menuitem_activate_cb: go to a selected item in the 
 * up directory level menu 
 */
void
up_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint levels;

	/* fetch data */
	levels = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	embed_go_up (embed, levels, 0);
}

/**
 * up_menu_menuitem_activate_cb: go to a selected item in the 
 * up directory level menu 
 */
gboolean
up_menu_menuitem_button_release_cb (GtkMenuItem *menuitem,
				GdkEventButton *event, gpointer user_data)
{
	GaleonEmbed *embed;
	gint levels;
	/* Get the submenu widget */
	GtkWidget *menu = GTK_WIDGET (menuitem)->parent;

	if ((event->button == 1 && !(event->state & GDK_SHIFT_MASK)) ||
			event->button == 3)
		return FALSE;
	/* fetch data */
	levels = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_val_if_not_embed (embed, FALSE);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
	embed_go_up (embed, levels,
		mouse_state_to_link_state (event->button, event->state));
	return TRUE;
}

/**
 * client_save_yourself_cb: called during window manager session shutdown,
 * we use our internal exit-with-session functionality to restore next
 * time the user starts Galeon.
 */
gboolean
client_save_yourself_cb (GnomeClient *client, gint phase, 
			 GnomeSaveStyle save_style, gint shutdown,
			 GnomeInteractStyle interact_style, gint fast)
{
	/* save session */
	session_save ();

	/* we're done */
	return TRUE;
}

/**
 * context_bm_toolbar_appearance_menu_cb: called by the toolbar 
 * appearance context menu. Set toolbar style.
 */
void
context_bm_toolbar_appearance_menu_cb (GtkRadioMenuItem *menuitem, 
				       BookmarkItem *bm)
{
	/* check being activated */
	if (!(GTK_CHECK_MENU_ITEM (menuitem)->active))
	{
		return;
	}

	/* button group is in reverse order! */
	bm->toolbar_style = radio_menuitem_index (menuitem);
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
}

/**
 * context_main_toolbar_appearance_menu_cb: called by the toolbar 
 * appearance context menu. Set toolbar style.
 */
void
context_main_toolbar_appearance_menu_cb (GtkRadioMenuItem *menuitem, 
					 BookmarkItem *bm)
{
	/* check being activated */
	if (!(GTK_CHECK_MENU_ITEM (menuitem)->active))
	{
		return;
	}

	/* get new value */
	eel_gconf_set_integer (CONF_TOOLBAR_STYLE, 
			      radio_menuitem_index (menuitem));
	window_update_ui ();
}

/**
 * context_main_toolbar_edit_toolbar_cb:
 */
void
context_main_toolbar_edit_toolbar_cb (GtkMenuItem *menuitem, 
				      GaleonWindow *window)
{
	return_if_not_window (window);
	preferences_show_page (window, PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS);
}

/**
 * radio_menuitem_index: find the index of a give radio menuitem within
 * its group. FIXME: refactor this to elsewhere.
 */
static gint
radio_menuitem_index (GtkRadioMenuItem *menuitem)
{
	GSList *list;
	gint index, length;

	list = GTK_RADIO_MENU_ITEM (menuitem)->group;
	index = g_slist_index (list, menuitem);
	length = g_slist_length (list);

	/* reverse order! */
	return ((length - 1) - index);
}

void
separator_toolbar_orientation_changed_cb (GtkToolbar *tb,
					  GtkOrientation o,
					  GtkWidget *sep)
{
	gboolean vsep = GTK_IS_VSEPARATOR (sep);
	GtkRequisition r;

	gtk_widget_size_request (GTK_WIDGET (tb), &r);
	
	if (o == GTK_ORIENTATION_HORIZONTAL)
	{
		if (vsep)
		{
			gtk_widget_set_usize (sep, 5, r.height / 2);
			gtk_widget_show (sep);
		}
		else
			gtk_widget_hide (sep);
	}
	else
	{
		if (!vsep)
		{
			gtk_widget_set_usize (sep, r.width / 2, 5);
			gtk_widget_show (sep);
		}
		else
			gtk_widget_hide (sep);
	}
}

/**
 * gtk_theme_changed_cb: called when a Gtk+ theme change is detected, update the
 * tab label styles and aa pixmaps
 * FIXME this seems to be called too many times
 */
GdkFilterReturn
gtk_theme_changed_cb (GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	GList *l;
	
	/* reload colour styles, reload pixmaps */
	initialise_colours ();
	reset_theme_pixmap_cache ();
	bookmarks_load_icons ();
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *) l->data;

		window_save_layout (window);
		toolbar_recreate (window);
		window_restore_layout (window);
		window_update_tabs_appearance (window);
		window_statusbar_set_security_icon (window,
			window->active_embed->secure);
		bookmarks_menu_recreate (window);
	}

	/* process other GDK events */
	return GDK_FILTER_CONTINUE;
}
