/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The functioning of the download architecture, as described by Philip
 * on 28 May 2001 and updated on 28 June 2001:
 * 
 * When mozilla runs into a file it cannot render internally or that it
 * does not have a plugin for, it calls the
 * nsIExternalHelperAppService. This service will then either attempt to
 * save the file or run it with a helper app depending on what the
 * mozilla mime database returns.
 * 
 * nsIExternalHelperAppService then calls out to the nsIHelperAppDialog
 * interface which handles the UI for the service. This is the interface
 * which we have reimplemented. Therefore, with a major caveat, we have
 * put a GNOME/GTK frontend on an unmodified mozilla backend.
 * 
 * Now for the caveat. With respect to saving files to disk, the mozilla
 * backend works exactly the same as it does in
 * mozilla-the-browser. However, for dealing with helper apps, we do not
 * use the mozilla backend at all. This is because we want to use the
 * gnome-vfs database to retrieve helper-app info, rather than the
 * mozilla helper app database.
 * 
 * How it works:
 * 
 * a) The user clicks on a link or follows a redirect to a file of a type
 * that mozilla cannot handle. Mozilla passes the link to the
 * ExternalHelperAppService which in turn calls the Show() method of
 * nsIHelperAppDialog.
 * 
 * b) In our implementation of Show() we first compare the passed mime
 * type to galeon's mime list. If the mime type is in the list, we then
 * lookup the Action associated with the mime type. Currently, the
 * possible mime-actions are:
 * 
 * Save to Disk
 * Run with Helper App
 * Ask User
 * 
 * The default action is Ask User, and if the mime-type is not in our
 * list, this is what will be assumed.
 * 
 * c) If Ask User is the chosen action, a dialog will be shown to the
 * user allowing the user to choose from the other two possible actions
 * as well as a checkbox to let the user set the default action to the
 * chosen action for the future.
 * 
 * d-1) The "Save to Disk" action. We first check galeon preferences to
 * see if the user wants to use the built-in mozilla downloader, gtm or
 * a command-line executed downloader.
 *
 * d-2a) The built-in downloader.  This action is handled by the mozilla
 * backend. Our nsIHelperAppDialog does the same thing that the
 * mozilla-the-browser one does, which is to call the SaveToDisk() method
 * of nsIExternalHelperAppService. This in turn calls the
 * PromptForSaveToFile() method of nsIHelperAppDialog putting the ball
 * back in our court.
 * 
 * d-2b) Now, if galeon is configured to always ask for a download
 * directory, it will pop up a file selector so that the user can select
 * the directory and filename to save the file to.  Otherwise, it will
 * use galeon's default download directory and proceed without
 * interaction.
 * 
 * d-2c) When PromptForSaveToFile() returns, nsIExternalHelperAppService
 * will then call the ShowProgressDialog() method of
 * nsIHelperAppDialog. This progress dialog, obviously, tracks the
 * progress of the download. It is worth noting that mozilla starts the
 * actual download as soon as the user clicks on the link or follows the
 * redirect. While the user is deciding what action to take, the file is
 * downloading. Often, for small files, the file is already downloaded
 * when the user decides what directory to put it in. The progress dialog
 * does not appear in these cases. Also, we currently have a small
 * problem where our progress dialog times the download from the point
 * the dialog appears, not from the time the download starts. This is due
 * to the timestamp that is passed to us is just plain weird, and I
 * haven't worked out how to turn it into a useable time. The fact that
 * the download starts early means that the file is actually downloaded
 * to a temp file and only at the end is it moved to it's final location.
 * 
 * d-3a) The two external downloader options.  These options are
 * handled completely by galeon. The first thing that we do is call the
 * Cancel() method of nsIExternalHelperAppService to cancel the mozilla
 * download. We then pass the url to our own LaunchExternalDownloader()
 * method. This method will ask for a download directory as appropriate
 * as with the "Save to disk" action.
 * 
 * d-3b) Finally, depending on whether GTM or a command line handler was
 * selected in prefs, the external handler will be called with the url
 * passed and the directory selected.
 * 
 * e-1) The "Run with Helper App" action.  This action is currently only
 * working with a minimal implementation.  First, we explicitly call
 * ShowProgressDialog() so the user knows that the file is being
 * downloaded. We also need this so that we only run the helper after the
 * file is completely downloaded. The file will download to temp location
 * that it would be moved from if the action was "Save to Disk".  We have
 * to call ShowProgressDialog() ourselves because we are not using
 * mozilla's helper mechanism which would usually make the call for us.
 * 
 * e-2) If there is a default helper app in our mime database and alwaysAsk
 * is false, galeon will run the default helper automatically. Otherwise it
 * will pop up a helper chooser dialog which lists the helpers that gnome-vfs
 * knows about as well as providing a GnomeFileEntry to allow the user to
 * select and arbitrary application. The default value of the GnomeFileEntry
 * is the helper stored in our database if one exits.
 * 
 * f) General notes.  We cannot use this infrastructure to override
 * native mozilla types. mozilla will attempt to render these types and
 * never call out to us. We are at the end of the chain as the handler of
 * last resort, so native and plugin types will never reach us. This also
 * means that a file with an incorrect mime-type ( eg: .tar.bz2 marked as
 * text/plain ) will be incorrectly rendered by mozilla. We cannot help
 * this.
 * 
 * Despite the apparent user-side similarity with explicit downloads by
 * a shift-click or context-menu item, there is actually none at all.
 * Explicit downloads are handled by the nsIStreamTransfer manager which
 * we use as is. Currently the progress dialog for the stream transfer
 * manager is un-overridable, so it appears in XUL. This will change in
 * due course.
 * 
 * Matt would like the modifiy the progress dialog so each file currently
 * being downloaded becomes a clist entry in a master dialog rather than
 * causing a separate progress dialog. a lot of progress dialogs gets
 * really messy.
 */

#include "galeon.h"
#include "mime.h"
#include "window.h"
#include "mozilla.h"
#include "prefs.h"
#include "misc.h"
#include "glade.h"
#include "downloader.h"
#include "dialog.h"
#include "ProgressCallbacks.h"

extern "C" {
#include "libgnomevfs/gnome-vfs-mime-handlers.h"
}

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkoptionmenu.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include "FilePicker.h"

#include "nsIHelperAppLauncherDialog.h"
#include "nsIWebProgressListener.h"
#include "nsIExternalHelperAppService.h"
#include "nsCExternalHandlerService.h"

#include "nsCRT.h"
#include "nsCOMPtr.h"
#include "nsIFactory.h"
#include "nsISupportsArray.h"
#include "nsIServiceManager.h"
#include "nsWeakReference.h"
#include "nsXPComFactory.h"

#include "nsString.h"
#include "nsIURI.h"
#include "nsIMIMEInfo.h"
#include "nsIChannel.h"
#include "nsIFTPChannel.h"
#include "nsILocalFile.h"
#include "nsIPrefService.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"

class GContentHandler;
class GDownloadProgressListener;
struct MimeAskActionDialog;
struct HelperAppChooserDialog;

extern "C" {
void mime_ask_dialog_ok_clicked_cb (GtkButton *button,
				    MimeAskActionDialog *dialog);
gint mime_ask_dialog_cancel_clicked_cb (GtkButton *button,
					MimeAskActionDialog *dialog);
void helper_chooser_ok_clicked_cb (GtkButton *button,
				   HelperAppChooserDialog *dialog);
gint helper_chooser_cancel_clicked_cb (GtkButton *button,
				       HelperAppChooserDialog *dialog);
void helper_chooser_auto_default_toggled (GtkToggleButton *button,
					  HelperAppChooserDialog *dialog);
void helper_chooser_radio_list_toggled (GtkToggleButton *button,
					HelperAppChooserDialog *dialog);
void helper_chooser_radio_other_toggled (GtkToggleButton *button,
					 HelperAppChooserDialog *dialog);
gint progress_dialog_cancel_cb (GtkButton *button,
				GIProgressCallbacks *aProgress);
void progress_dialog_pause_cb (GtkButton *button,
			       GIProgressCallbacks *aProgress);
}

/**
 * MimeAskActionDialog: the representation of dialogs used to ask
 * about actions on MIME types
 */
struct MimeAskActionDialog
{
	gchar *url;
	gchar *mime_type;
	GContentHandler *mContentHandler;
	GladeXML *gxml;
};

struct HelperAppChooserDialog
{
	HelperAppChooserDialog (void) : mOtherHelperApp(FALSE) {};
	~HelperAppChooserDialog () {};

	GList *mHelperApps;
	GContentHandler *mContentHandler;
	GladeXML *mGXML;
	gboolean mOtherHelperApp;

	GtkWidget *mDialogWidget;
	GtkWidget *mHelperGnomeEntry;
	GtkWidget *mHelperEntry;
	GtkWidget *mOptionMenu;
	GtkWidget *mCheckUseTerminal;
	GtkWidget *mCheckUseUrl;
	GtkWidget *mCheckAlwaysUse;
};

class GContentHandler : public nsIHelperAppLauncherDialog
{
  public:
 	NS_DECL_ISUPPORTS
	NS_DECL_NSIHELPERAPPLAUNCHERDIALOG

	GContentHandler();
	virtual ~GContentHandler();

	NS_IMETHODIMP LaunchExternalDownloader (void);
	NS_IMETHODIMP FindHelperApp (void);
	NS_IMETHODIMP LaunchHelperApp (void);
	NS_IMETHODIMP ShowHelperProgressDialog (void);

	NS_IMETHODIMP GetLauncher (nsIHelperAppLauncher * *_retval);
	NS_IMETHODIMP GetContext (nsISupports * *_retval);
	NS_IMETHODIMP SetHelperApp(GnomeVFSMimeApplication *mHelperApp,
				   PRBool alwaysUse);

  private:
	/* additional members */
	NS_IMETHODIMP Init (void);
	NS_IMETHODIMP ChooseHelperApp (void);
	NS_IMETHODIMP ProcessMimeInfo (void);
	NS_IMETHODIMP MIMEAskAction (void);
	
	nsCOMPtr<nsIHelperAppLauncher> mLauncher;
	nsCOMPtr<nsISupports> mContext;

	nsCOMPtr<nsIURI> mUri;
	PRInt64 mTimeDownloadStarted;
	nsCOMPtr<nsIFile> mTempFile;

	MimeItem *mMimeItem;

	char *mUrl;
	char *mMimeType;

	PRBool mDownloadCanceled;
	PRBool mHelperProgress;

	nsCOMPtr<nsIWebProgressListener> mListener;
	
};

class GDownloadProgressListener : public nsIWebProgressListener,
				  public nsSupportsWeakReference,
				  public GIProgressCallbacks
{
 public:
 	NS_DECL_ISUPPORTS
	NS_DECL_NSIWEBPROGRESSLISTENER
	G_DECL_GIPROGRESSCALLBACKS

	GDownloadProgressListener (nsIHelperAppLauncher *aLauncher,
				   nsISupports *aContext,
				   GContentHandler *aHandler,
				   PRBool aUsedByHelper);
	virtual ~GDownloadProgressListener ();
 private:

	char * FormatTime (PRUint32 aTime);
	
	nsCOMPtr<nsIHelperAppLauncher> mLauncher;
	nsCOMPtr<nsISupports> mContext;
	nsCOMPtr<GContentHandler> mHandler;
	nsCOMPtr<nsIRequest> mRequest;
	PRBool mUsedByHelper;
	
	nsCOMPtr<nsIURI> mUri;
	PRInt64 mTimeDownloadStarted;
	nsCOMPtr<nsIFile> mFile;
	
	PRInt64 mStartTime;
	PRInt64 mElapsed;
	
	PRInt64 mLastUpdate;
	PRInt32 mInterval;

	PRFloat64 mPriorKRate;
	PRInt32 mRateChanges;
	PRInt32 mRateChangeLimit;

	PRBool mCheckedCanPause;
	PRBool mCanPause;
	PRBool mIsPaused;

	GtkWidget *mProgressDialog;
	GtkWidget *mProgressBar;
	GtkWidget *mLocation;
	GtkWidget *mFileName;
	GtkWidget *mStatus;
	GtkWidget *mTimeElapsed;
	GtkWidget *mTimeRemaining;
	GtkWidget *mPauseButton;
};


/* Implementation file */
NS_IMPL_ISUPPORTS1(GContentHandler, nsIHelperAppLauncherDialog)

NS_IMPL_ISUPPORTS3(GDownloadProgressListener, nsIWebProgressListener,
		   GIProgressCallbacks, nsISupportsWeakReference)

GContentHandler::GContentHandler()
{
	NS_INIT_ISUPPORTS();
	/* member initializers and constructor code */
	mUri = nsnull;
	mMimeType = nsnull;
	mMimeItem = nsnull;
	mDownloadCanceled = PR_FALSE;
	mHelperProgress = PR_FALSE;
}

GContentHandler::~GContentHandler()
{
	/* destructor code */
	g_free (mUri);
	g_free (mMimeType);
	mime_item_free (mMimeItem);
}

////////////////////////////////////////////////////////////////////////////////
// begin nsIHelperAppLauncher impl
////////////////////////////////////////////////////////////////////////////////

/* void show (in nsIHelperAppLauncher aLauncher, in nsISupports aContext); */
NS_IMETHODIMP GContentHandler::Show(nsIHelperAppLauncher *aLauncher,
					      nsISupports *aContext)
{
	nsresult rv;

	mLauncher = aLauncher;
	mContext = aContext;
	rv = Init ();
	
	switch (mMimeItem->action)
	{
	 case MIME_UNKNOWN:
		mime_set_action (mMimeType, MIME_ASK_ACTION);
	 case MIME_ASK_ACTION:
		MIMEAskAction ();
		break;
	 case MIME_IGNORE:
		mLauncher->Cancel ();
		delete mMimeItem;
		break;
	 case MIME_SAVE_TO_DISK:
	 	if (eel_gconf_get_integer (CONF_DOWNLOADING_DOWNLOADER)
	 	    == DOWNLOADER_BUILTIN)
	 	{
			mLauncher->SaveToDisk (nsnull,PR_FALSE);
		}
		else
		{
			mLauncher->Cancel ();
			LaunchExternalDownloader ();
		}
	 	break;
	 case MIME_RUN_PROGRAM:
		FindHelperApp ();
	 	break;
	 default:
		/* shouldn't get here */
		g_warning ("Unexpected MIME action");
		break;
	};
	return NS_OK;
}

/* nsILocalFile promptForSaveToFile (in nsISupports aWindowContext, in wstring aDefaultFile, in wstring aSuggestedFileExtension); */
NS_IMETHODIMP GContentHandler::
		PromptForSaveToFile(nsISupports *aWindowContext,
				    const PRUnichar *aDefaultFile,
				    const PRUnichar *aSuggestedFileExtension,
				    nsILocalFile **_retval)
{
	nsresult rv;

	mContext = aWindowContext;

	nsCOMPtr<nsIDOMWindowInternal> aWindowInternal = 
					do_QueryInterface (aWindowContext);
	
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("browser.download.", getter_AddRefs (pref));
	nsCOMPtr<nsILocalFile> saveDir;
	rv = pref->GetComplexValue ("dir", NS_GET_IID(nsILocalFile),
				    getter_AddRefs(saveDir));

	if (NS_FAILED(rv))
	{
		char *dirName = eel_gconf_get_string (CONF_DOWNLOADING_DIR);

		if (!dirName)
		{
			dirName = g_strdup (g_get_home_dir());
		}
		saveDir = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
		saveDir->InitWithPath (dirName);
		g_free (dirName);
	}

	nsCOMPtr<nsIFilePicker> aFilePicker =
				do_CreateInstance (G_FILEPICKER_CONTRACTID);

	nsString title = NS_LITERAL_STRING (N_("Save As:"));

	aFilePicker->Init (aWindowInternal, title.get(), 
			   nsIFilePicker::modeSave);
	aFilePicker->SetDefaultString (aDefaultFile);
	aFilePicker->SetDisplayDirectory (saveDir);

	nsCOMPtr <nsILocalFile> aFile (do_CreateInstance(NS_LOCAL_FILE_CONTRACTID));

	PRInt16 okToSave;
	if (eel_gconf_get_boolean (CONF_DOWNLOADING_ASK_DIR))
	{
		aFilePicker->Show (&okToSave);

		if (okToSave == nsIFilePicker::returnOK)
		{
			aFilePicker->GetFile (getter_AddRefs(aFile));
		};
	}
	else
	{
		char *cDirName;
		saveDir->GetPath (&cDirName);
		char *cDefaultFile = mozilla_unicode_to_locale (aDefaultFile);
		char *cFileName = g_strconcat (cDirName, "/", cDefaultFile, NULL);
		aFile->InitWithPath (cFileName);
		g_free (cDefaultFile);
		g_free (cFileName);
		nsMemory::Free (cDirName);
		okToSave = nsIFilePicker::returnOK;
	};


	if (okToSave == nsIFilePicker::returnCancel)
		return NS_ERROR_FAILURE;
	else
	{
		nsCOMPtr<nsIFile> aDirectory;
		rv = aFile->GetParent (getter_AddRefs(aDirectory));
		nsCOMPtr<nsILocalFile> aLocalDirectory 
					(do_QueryInterface(aDirectory));
		rv = pref->SetComplexValue ("dir", NS_GET_IID(nsILocalFile),
					    aLocalDirectory);
		NS_IF_ADDREF (*_retval = aFile);
		return NS_OK;
	};
}

/* void showProgressDialog (in nsIHelperAppLauncher aLauncher, in nsISupports aContext); */
NS_IMETHODIMP GContentHandler::
			ShowProgressDialog(nsIHelperAppLauncher *aLauncher,
					   nsISupports *aContext)
{
	GDownloadProgressListener *aListener =
				new GDownloadProgressListener (aLauncher,
							       aContext,
							       this,
							       mHelperProgress);
#if MOZILLA_VERSION < VERSION4(0,9,4,1)
	NS_ADDREF (aListener);
	mListener = NS_STATIC_CAST (nsIWebProgressListener *, aListener);
	NS_RELEASE (aListener);
#endif
	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local public methods impl
////////////////////////////////////////////////////////////////////////////////

NS_IMETHODIMP GContentHandler::LaunchExternalDownloader (void)
{
	downloader_save_url (mUrl);
	return NS_OK;
}


NS_IMETHODIMP GContentHandler::FindHelperApp (void)
{
	if (mMimeItem->alwaysUse)
		return ShowHelperProgressDialog ();
	else
		return ChooseHelperApp ();
}

NS_IMETHODIMP GContentHandler::LaunchHelperApp (void)
{
	gchar *tmp_dir = g_get_tmp_dir ();

	if (mMimeItem)
	{
		nsresult rv;
		nsCOMPtr<nsIExternalHelperAppService> helperService =
			do_GetService (NS_EXTERNALHELPERAPPSERVICE_CONTRACTID);

		nsCOMPtr<nsPIExternalAppLauncher> appLauncher =
			do_QueryInterface (helperService, &rv);
		if (NS_SUCCEEDED(rv))
		{
			appLauncher->DeleteTemporaryFileOnExit(mTempFile);
		}

		char *aFileName, *document(NULL);
		mTempFile->GetPath(&aFileName);

		if (mMimeItem->urlHelper)
		{
			document = mUrl;
		}
		else
		{
			document = aFileName;
		}
		char *full_command = g_strconcat (mMimeItem->helperApp,
						  " ", document, NULL);

		if (mMimeItem->terminalHelper == FALSE)
		{
			char **argv = g_strsplit (full_command, " ", 100);
			int i;
			for (i = 0; argv[i] != NULL; i++);
			gnome_execute_async (tmp_dir, i, argv);
			g_strfreev (argv);
		}
		else
		{
			char *argv[3] = { "gnome-terminal", "-e",
					  full_command };
			gnome_execute_async (tmp_dir, 3, argv);
		}
		g_free (full_command);
		g_free (aFileName);
	}
	else
	{
		mLauncher->Cancel ();
	}

	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ShowHelperProgressDialog (void)
{
	mHelperProgress = PR_TRUE;
	return ShowProgressDialog (mLauncher,mContext);
}

NS_IMETHODIMP GContentHandler::GetLauncher (nsIHelperAppLauncher * *_retval)
{
	NS_IF_ADDREF (*_retval = mLauncher);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::GetContext (nsISupports * *_retval)
{
	NS_IF_ADDREF (*_retval = mContext);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::SetHelperApp(GnomeVFSMimeApplication *aHelperApp,
					    PRBool alwaysUse)
{
	mMimeItem->helperApp = g_strdup (aHelperApp->command);
	mMimeItem->alwaysUse = alwaysUse ? TRUE : FALSE;
	mMimeItem->terminalHelper = aHelperApp->requires_terminal;
	if (aHelperApp->expects_uris ==
				GNOME_VFS_MIME_APPLICATION_ARGUMENT_TYPE_PATHS)
	{
		mMimeItem->urlHelper = FALSE;
	}
	else
	{
		mMimeItem->urlHelper = TRUE;
	}

	if (nsCRT::strcmp(aHelperApp->id,"OtHeR") == 0)
	{
		g_free (aHelperApp->command);
		g_free (aHelperApp->id);
		delete aHelperApp;
	}

	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local private methods impl
////////////////////////////////////////////////////////////////////////////////
NS_IMETHODIMP GContentHandler::Init (void)
{
	nsresult rv;

	nsCOMPtr<nsIMIMEInfo> aMIMEInfo;
	rv = mLauncher->GetMIMEInfo (getter_AddRefs(aMIMEInfo));
	rv = aMIMEInfo->GetMIMEType (&mMimeType);
	
	ProcessMimeInfo ();

	rv = mLauncher->GetDownloadInfo(getter_AddRefs(mUri),
					&mTimeDownloadStarted,
					getter_AddRefs(mTempFile));

	rv = mUri->GetSpec (&mUrl);
#if 0
	/* GetSource seems redundant and isn't in 0.9 This code is here while
	   it remains unclear what GetSource is for. --phil */
	nsCOMPtr<nsIURI> aUri;
	rv = mLauncher->GetSource(getter_AddRefs(aUri));
	rv = aUri->GetSpec (&mUrl);
#endif	
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ProcessMimeInfo (void)
{
	/* FIXME: hack for 
           http://www.xref-tech.com/xrefactory/ftp/xref-i86pcLinux.tar.gz?Cache
	*/
	if (mMimeType == NULL)
	{
		mMimeType = g_strdup ("unknown/unknown");
	}

	/* FIXME this it to work around a mozilla bug, when closing popups
	   sometimes mime types handled internally by mozilla are passed
	   to the content handler. Force the correct behavior */
	if (!strcasecmp(mMimeType, "text/html")   ||
	    !strcasecmp(mMimeType, "text/plain")  ||
	    !strcasecmp(mMimeType, "application/vnd.mozilla.xul+xml")    ||
	    !strcasecmp(mMimeType, "text/rdf")    ||
	    !strcasecmp(mMimeType, "text/xml")    ||
	    !strcasecmp(mMimeType, "application/xml")    ||
	    !strcasecmp(mMimeType, "application/xhtml+xml")    ||
	    !strcasecmp(mMimeType, "text/css")    ||
	    !strcasecmp(mMimeType, "image/gif")   ||
	    !strcasecmp(mMimeType, "image/jpeg")  ||
	    !strcasecmp(mMimeType, "image/png")   ||
	    !strcasecmp(mMimeType, "application/http-index-format"))
	{
		mMimeItem = new MimeItem;
		mMimeItem->action = MIME_IGNORE;
		g_warning ("wrong mime type detection !");
		return NS_OK;
	}

	mMimeItem = mime_item_get (mMimeType);	

	if (!mMimeItem)
	{
		mime_item_create (mMimeType, MIME_UNKNOWN, _("None"),
				  FALSE, FALSE, FALSE);
		mMimeItem = mime_item_get (mMimeType);
	}

	return NS_OK;
}

NS_IMETHODIMP GContentHandler::MIMEAskAction (void)
{
	MimeAskActionDialog *dialog = new MimeAskActionDialog;
	GtkWidget *label;
	GtkWidget *dialogWidget;
	gchar *msg;
		
	dialog->url = g_strdup (mUrl);
	dialog->mime_type = g_strdup (mMimeType);
	dialog->mContentHandler = this;
	dialog->gxml = glade_widget_new ("galeon.glade", "mime_ask_action_dialog", 
					 &dialogWidget, dialog);

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	dialog_set_parent (dialogWidget, aParentWidget);

	label = glade_xml_get_widget (dialog->gxml, "mime_ask_action_message");
	msg = g_strdup_printf (_("The URL %s has MIME type %s"),
			       mUrl, mMimeType);
	gtk_label_set_text (GTK_LABEL (label), msg);
	g_free (msg);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ChooseHelperApp (void)
{
	HelperAppChooserDialog *dialog = new HelperAppChooserDialog;

	dialog->mHelperApps = gnome_vfs_mime_get_all_applications (mMimeType);
	dialog->mContentHandler = this;
	dialog->mGXML = glade_widget_new ("galeon.glade",
					  "helper_chooser_dialog", 
					  &(dialog->mDialogWidget), dialog);

	dialog->mOptionMenu = glade_xml_get_widget (dialog->mGXML,
						    "option_menu");
	dialog->mHelperGnomeEntry = 
		glade_xml_get_widget (dialog->mGXML, 
				      "other_helper_gnomefileentry");
	dialog->mHelperEntry = glade_xml_get_widget (dialog->mGXML,
						     "other_helper_entry");
	dialog->mCheckUseTerminal = glade_xml_get_widget (dialog->mGXML,
							  "check_use_terminal");
	dialog->mCheckUseUrl = glade_xml_get_widget (dialog->mGXML,
						     "check_use_url");
	dialog->mCheckAlwaysUse = glade_xml_get_widget (dialog->mGXML,
							"check_make_default");

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	dialog_set_parent (dialog->mDialogWidget, aParentWidget);

	GnomeVFSMimeApplication *app;
	app = gnome_vfs_mime_get_default_application (mMimeType);

	GtkWidget *menu = gtk_menu_new ();
	int k = 0;
	for (GList *helperApp = g_list_first(dialog->mHelperApps) ;
	     helperApp!=NULL ;
	     helperApp = g_list_next(helperApp))
	{
		GtkWidget *menuItem = gtk_menu_item_new_with_label (
			((GnomeVFSMimeApplication *)(helperApp->data))->name);
		gtk_menu_append (GTK_MENU(menu),menuItem);

		char *id = ((GnomeVFSMimeApplication *)(helperApp->data))->id;
		if (strcmp (id, app->id) == 0)
		{
			gtk_menu_set_active ((GtkMenu*)menu, k);
		}
		k++;

		gtk_widget_show (menuItem);
	}

	g_free (app);

	gtk_option_menu_remove_menu (GTK_OPTION_MENU(dialog->mOptionMenu));
	gtk_option_menu_set_menu (GTK_OPTION_MENU(dialog->mOptionMenu),
				  menu);

	gtk_entry_set_text (GTK_ENTRY(dialog->mHelperEntry),
			    mMimeItem->helperApp);
	gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON(dialog->mCheckUseTerminal),
			mMimeItem->terminalHelper);
	gtk_toggle_button_set_active (
			GTK_TOGGLE_BUTTON(dialog->mCheckUseUrl),
			mMimeItem->urlHelper);

	gtk_widget_set_sensitive (dialog->mOptionMenu, TRUE);
	gtk_widget_set_sensitive (dialog->mHelperGnomeEntry, FALSE);
	gtk_widget_set_sensitive (dialog->mHelperEntry, FALSE);
	gtk_widget_set_sensitive (dialog->mCheckUseTerminal, FALSE);
	gtk_widget_set_sensitive (dialog->mCheckUseUrl, FALSE);

	GtkWidget *label = glade_xml_get_widget (dialog->mGXML, "mime_label");
	char *msg = g_strdup_printf (_("MIME type: %s"), mMimeType);
	gtk_label_set_text (GTK_LABEL(label), msg);
	g_free (msg);

	return NS_OK;
}

//------------------------------------------------------------------------------

NS_DEF_FACTORY (GContentHandler, GContentHandler);

/**
 * NS_NewContentHandlerFactory:
 */ 
nsresult NS_NewContentHandlerFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsGContentHandlerFactory *result = new nsGContentHandlerFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	}
    
	NS_ADDREF(result);
	*aFactory = result;

	return NS_OK;
}

//------------------------------------------------------------------------------

GDownloadProgressListener::
	GDownloadProgressListener(nsIHelperAppLauncher *aLauncher,
				  nsISupports *aContext,
				  GContentHandler *aHandler,
				  PRBool aUsedByHelper)
{
	NS_INIT_ISUPPORTS();
	
	/* member initializers and constructor code */
	nsresult rv;
	
	mInterval = 500000; //in microsecs == 500ms == 0.5s
	mPriorKRate = 0;
	mRateChanges = 0;
	mRateChangeLimit = 2; //Only update rate every second
	
	mLauncher = aLauncher;
	mContext = aContext;
	mHandler = aHandler;
	mUsedByHelper = aUsedByHelper;

	rv = mLauncher->GetDownloadInfo(getter_AddRefs(mUri),
					&mTimeDownloadStarted,
					getter_AddRefs(mFile));

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	nsCOMPtr<nsIWebProgressListener> progress =
			NS_STATIC_CAST(nsIWebProgressListener *, this);
	nsCOMPtr<GIProgressCallbacks> callbacks =
			do_QueryInterface (progress);
	GladeXML *gxml = glade_widget_new ("galeon.glade", 
					   "download_progress_dialog",
					   &mProgressDialog, callbacks);

	mProgressBar = glade_xml_get_widget (gxml, "progressbar");
	mLocation = glade_xml_get_widget (gxml, "location_entry");
	mFileName = glade_xml_get_widget (gxml, "filename_entry");
	mStatus = glade_xml_get_widget (gxml, "status_entry");
	mTimeElapsed = glade_xml_get_widget (gxml, "elapsed_entry");
	mTimeRemaining = glade_xml_get_widget (gxml, "remaining_entry");
	mPauseButton = glade_xml_get_widget (gxml, "pause_button");
	gtk_object_unref (GTK_OBJECT (gxml));

	mCheckedCanPause = PR_FALSE;
	mCanPause = PR_FALSE;
	mIsPaused = PR_FALSE;

	PRInt64 now = PR_Now ();
	mLastUpdate = now;
#if MOZILLA_VERSION > VERSION4(0,9,4,1)
	mStartTime = mTimeDownloadStarted;
#else
	mStartTime = now;
#endif
	mElapsed = now - mStartTime;

	char *text;

	rv = mUri->GetSpec (&text);
	gtk_entry_set_text (GTK_ENTRY(mLocation),text);
	g_free (text);
	
	rv = mFile->GetPath (&text);
	gtk_entry_set_text (GTK_ENTRY(mFileName),text);
	g_free (text);
	
	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));
	gtk_label_set_text (GTK_LABEL(mTimeRemaining),FormatTime(0));

	dialog_set_parent (mProgressDialog, aParentWidget);

	aLauncher->SetWebProgressListener (this);

	// Increment window count while progress is under way.
	window_count++;
}

GDownloadProgressListener::~GDownloadProgressListener()
{
	/* destructor code */
}

/* void onStateChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aStateFlags, in unsigned long aStatus); */
NS_IMETHODIMP GDownloadProgressListener::
			OnStateChange(nsIWebProgress *aWebProgress,
				      nsIRequest *aRequest,PRInt32 aStateFlags,
				      PRUint32 aStatus)
{
	if (aStateFlags & nsIWebProgressListener::STATE_STOP)
	{
		if (GTK_IS_WIDGET(mProgressDialog))
			gtk_widget_destroy (mProgressDialog);
		mLauncher->CloseProgressWindow ();
		if (mUsedByHelper)
			mHandler->LaunchHelperApp ();

		// Decrement window count as this progess is finished.
		window_count--;

		// If there are no more windows left, exit galeon.
		if (g_list_length (all_windows) == 0 && !galeon_server_mode
		    && window_count ==0)
		{
			galeon_exit ();
		}
	}

	return NS_OK;
}

/* void onProgressChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aCurSelfProgress, in long aMaxSelfProgress, in long aCurTotalProgress, in long aMaxTotalProgress); */
NS_IMETHODIMP GDownloadProgressListener::
			OnProgressChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest,
					 PRInt32 aCurSelfProgress,
					 PRInt32 aMaxSelfProgress,
					 PRInt32 aCurTotalProgress,
					 PRInt32 aMaxTotalProgress)
{
	char *text, *rateStr, *totalStr;

	if (!mCheckedCanPause)
	{
		mCheckedCanPause = PR_TRUE;

		nsresult rv;
		nsCOMPtr<nsIFTPChannel> aChannel = do_QueryInterface (aRequest,
								      &rv);
		if (NS_SUCCEEDED(rv))
		{
			mCanPause = PR_TRUE;
			gtk_widget_show (mPauseButton);
		}
		else
		{
			mCanPause = PR_FALSE;
			gtk_widget_hide (mPauseButton);
		}
	}
	mRequest = aRequest;

	PRInt64 now = PR_Now ();

	if ((now - mLastUpdate < mInterval) && 
	     (aMaxTotalProgress != -1) &&  
	     (aCurTotalProgress < aMaxTotalProgress))
		return NS_OK;

	mLastUpdate = now;
	mElapsed = now - mStartTime;

	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));

	PRInt32 currentKBytes = (PRInt32)(aCurTotalProgress/1024 +.5);

	PRInt32 totalKBytes = (PRInt32)(aMaxTotalProgress/1024 +.5);
	if (aMaxTotalProgress != -1)
	{
		gfloat progress = (gfloat)aCurTotalProgress/
				  (gfloat)aMaxTotalProgress;
		gchar *strper = g_strdup_printf (_("%.2f%% - Downloading"), 
						 progress*100);

		if (progress > 0 && progress < 1)
			gtk_window_set_title (GTK_WINDOW(mProgressDialog), 
					      strper);
		gtk_progress_set_percentage (GTK_PROGRESS(mProgressBar),
					     progress);
		totalStr = g_strdup_printf ("%d",totalKBytes);
		g_free (strper);
	}
	else
	{
		gtk_progress_set_format_string (GTK_PROGRESS(mProgressBar),
						"?? %%");
		totalStr = g_strdup ("??");
	}

	PRInt64 currentRate;
	if (mElapsed)
		currentRate = ((PRInt64)(aCurTotalProgress))*1000000 / mElapsed;
	else
		currentRate = 0;
		
	if (currentRate)
	{
		PRFloat64 currentKRate = ((PRFloat64)currentRate)/1024;
		if (currentKRate != mPriorKRate)
		{
			if (mRateChanges++ == mRateChangeLimit)
			{
				mPriorKRate = currentKRate;
				mRateChanges = 0;
			}
			else
			{
				currentKRate = mPriorKRate;
			}
		}
		else
		{
			mRateChanges = 0;
		}
		rateStr = g_strdup_printf("%.2f",currentKRate);
	}
	else
	{
		rateStr = g_strdup ("??.??");
	}

	if (mIsPaused)
	{
		text = g_strdup (_("Download Paused"));
	}
	else
	{
		text = g_strdup_printf (_("%dK of %sK bytes at %sK bytes/sec"),
					currentKBytes, totalStr, rateStr);
	}
	gtk_label_set_text (GTK_LABEL(mStatus),text);
	g_free (text);
	g_free (totalStr);
	g_free (rateStr);

	if (currentRate && (aMaxTotalProgress != -1))
	{
		PRInt32 remaining = 
			(PRInt32)((aMaxTotalProgress - aCurTotalProgress)
				   /currentRate +.5);
		gtk_label_set_text (GTK_LABEL(mTimeRemaining),
				    FormatTime(remaining));
	}
	else
	{
		gtk_label_set_text (GTK_LABEL(mTimeRemaining), _("Unknown"));
	}

	return NS_OK;
}

/* void onLocationChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsIURI location); */
NS_IMETHODIMP GDownloadProgressListener::
			OnLocationChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, nsIURI *location)
{
    return NS_OK;
}

/* void onStatusChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsresult aStatus, in wstring aMessage); */
NS_IMETHODIMP GDownloadProgressListener::
			OnStatusChange(nsIWebProgress *aWebProgress,
				       nsIRequest *aRequest, nsresult aStatus,
				       const PRUnichar *aMessage)
{
    return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long state); */
NS_IMETHODIMP GDownloadProgressListener::
			OnSecurityChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, PRInt32 state)
{
    return NS_OK;
}

NS_IMETHODIMP GDownloadProgressListener::CancelHelperProgress (void)
{
	mUsedByHelper = PR_FALSE;
	if (mIsPaused)
		TogglePause ();
	return mLauncher->Cancel ();
}

NS_IMETHODIMP GDownloadProgressListener::TogglePause (void)
{
	if (!mCheckedCanPause) return NS_ERROR_FAILURE;

	nsresult rv;
	if (mIsPaused)
	{
		rv = mRequest->Resume ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Pause"));
		}
	}
	else
	{
		rv = mRequest->Suspend ();
		if (NS_SUCCEEDED(rv))
		{
			gtk_label_set_text (GTK_LABEL(mStatus),
					    _("Download Paused"));
			gtk_label_set_text (GTK_LABEL(GTK_BIN(mPauseButton)
					    ->child),
					    _("Resume"));
		}
	}
	mIsPaused = !mIsPaused;
	return rv;
}

char *GDownloadProgressListener::FormatTime (PRUint32 aTime)
{
	PRUint32 secs = (PRUint32)(aTime+.5);
	PRUint32 hours = secs/3600;
	secs -= hours*3600;
	PRUint32 mins = secs/60;
	secs -= mins*60;
	char *result;
	if (hours)
		result = g_strdup_printf ("%u:%02u.%02u",hours,mins,secs);
	else
		result = g_strdup_printf ("%02u.%02u",mins,secs);
	return result;
}

////////////////////////////////////////////////////////////////////////////////
// begin MIMEAskActionDialog callbacks.
////////////////////////////////////////////////////////////////////////////////

void
mime_ask_dialog_ok_clicked_cb (GtkButton *button, MimeAskActionDialog *dialog)
{
	MimeAction action;
	GtkWidget *save_check = glade_xml_get_widget 
		(dialog->gxml, "mime_action_save");
	GtkWidget *run_program_check = 
		glade_xml_get_widget (dialog->gxml, "mime_action_run_program");
	GtkWidget *remember_check =
		glade_xml_get_widget (dialog->gxml, "mime_remember_check");

	gtk_widget_hide (glade_xml_get_widget (dialog->gxml, 
					       "mime_ask_action_dialog"));

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (save_check)))
		action = MIME_SAVE_TO_DISK;
	else if (gtk_toggle_button_get_active 
			(GTK_TOGGLE_BUTTON (run_program_check)))
		action = MIME_RUN_PROGRAM;
	else
		action = MIME_UNKNOWN;

	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));
	
	switch (action)
	{
	 case MIME_SAVE_TO_DISK:
	 	if (eel_gconf_get_integer (CONF_DOWNLOADING_DOWNLOADER)
	 	    == DOWNLOADER_BUILTIN)
	 	{
			aLauncher->SaveToDisk (nsnull,PR_FALSE);
		}
		else
		{
			aLauncher->Cancel ();
			dialog->mContentHandler->LaunchExternalDownloader ();
		}
		break;
	 case MIME_RUN_PROGRAM:
		dialog->mContentHandler->FindHelperApp ();
		break;
	 case MIME_ASK_ACTION:
	 case MIME_UNKNOWN:
	 case MIME_IGNORE:
	 default:
		/* shouldn't get here */
		g_warning ("Unexpected MIME action");
		break;
	};
	
	/* check if this should be the new default action */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (remember_check))) 
		mime_set_action (dialog->mime_type, action);

	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	delete dialog;
}

gint
mime_ask_dialog_cancel_clicked_cb (GtkButton *button,
				   MimeAskActionDialog *dialog)
{
	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));

	aLauncher->Cancel ();
	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	delete dialog;
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

////////////////////////////////////////////////////////////////////////////////
// begin HelperAppChooserDialog callbacks.
////////////////////////////////////////////////////////////////////////////////

void
helper_chooser_ok_clicked_cb (GtkButton *button,
			      HelperAppChooserDialog *dialog)
{
	GnomeVFSMimeApplication *helperApp;
	if (dialog->mOtherHelperApp == FALSE)
	{
		GtkWidget *menu = GTK_OPTION_MENU (dialog->mOptionMenu)->menu;
		GList *menuItems = GTK_MENU_SHELL (menu)->children;
		if (!menuItems)
		{
			helper_chooser_cancel_clicked_cb (button, dialog);
			return;
		}
		gpointer item = gtk_menu_get_active (GTK_MENU(menu));
		gint index = g_list_index (menuItems,item);
		GList *helperApps = g_list_nth(dialog->mHelperApps,index);
		helperApp = (GnomeVFSMimeApplication *)(helperApps->data);
	}
	else
	{
		helperApp = new GnomeVFSMimeApplication;
		helperApp->command = g_strdup (gtk_entry_get_text(
						GTK_ENTRY(dialog->mHelperEntry)));
		if (!helperApp->command)
		{
			helper_chooser_cancel_clicked_cb (button, dialog);
			return;
		}
		helperApp->id = g_strdup ("OtHeR");
		helperApp->requires_terminal = gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON(dialog->mCheckUseTerminal));
		helperApp->expects_uris = gtk_toggle_button_get_active (
					GTK_TOGGLE_BUTTON(dialog->mCheckUseUrl)) ?
					GNOME_VFS_MIME_APPLICATION_ARGUMENT_TYPE_URIS :
					GNOME_VFS_MIME_APPLICATION_ARGUMENT_TYPE_PATHS;
	}

	PRBool alwaysUse = gtk_toggle_button_get_active (
				GTK_TOGGLE_BUTTON(dialog->mCheckAlwaysUse)) ?
			   PR_TRUE : PR_FALSE;
	dialog->mContentHandler->SetHelperApp (helperApp, alwaysUse);
	dialog->mContentHandler->ShowHelperProgressDialog ();

	gnome_vfs_mime_application_list_free (dialog->mHelperApps);

	gtk_widget_destroy (dialog->mDialogWidget);
	gtk_object_unref (GTK_OBJECT (dialog->mGXML));
	delete dialog;
}

gint
helper_chooser_cancel_clicked_cb (GtkButton *button,
				  HelperAppChooserDialog *dialog)
{
	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));

	aLauncher->Cancel ();

	gnome_vfs_mime_application_list_free (dialog->mHelperApps);

	gtk_widget_destroy (dialog->mDialogWidget);
	gtk_object_unref (GTK_OBJECT (dialog->mGXML));
	delete dialog;
	
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

#if 0 /* philipl: I've changed it over to using a GnomeFileEntry for
       * this, is this okay? -- MattA */
void helper_chooser_browse_clicked_cb (GtkButton *button,
				       HelperAppChooserDialog *dialog)
{
	char *fileName = g_strdup (g_get_home_dir());

	nsCOMPtr<nsISupports> aContext;
	dialog->mContentHandler->GetContext (getter_AddRefs(aContext));
	nsCOMPtr<nsIDOMWindowInternal> aWindowInternal = 
				do_QueryInterface (aContext);

	nsCOMPtr<nsILocalFile> aSaveDir = 
			do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	aSaveDir->InitWithPath (fileName);

	nsCOMPtr<nsIFilePicker> aFilePicker =
			do_CreateInstance (G_FILEPICKER_CONTRACTID);

	nsString title = NS_LITERAL_STRING (N_("Choose Helper"
					       " Application"));

	aFilePicker->Init (aWindowInternal, title.get(), 
			   nsIFilePicker::modeOpen);
	aFilePicker->SetDefaultString (nsnull);
	aFilePicker->SetDisplayDirectory (aSaveDir);
		
	PRInt16 okToSave;
	aFilePicker->Show (&okToSave);

	g_free (fileName);
	if (okToSave == nsIFilePicker::returnOK)
	{
		nsCOMPtr<nsILocalFile> aFile;
		aFilePicker->GetFile (getter_AddRefs(aFile));
		aFile->GetPath (&fileName);
		gtk_entry_set_text (GTK_ENTRY(dialog->mHelperEntry),
				    fileName);
		nsMemory::Free (fileName);
	};
}
#endif

void helper_chooser_radio_list_toggled (GtkToggleButton *button,
					HelperAppChooserDialog *dialog)
{
	gboolean active = gtk_toggle_button_get_active (button);
	if (active == TRUE)
		dialog->mOtherHelperApp = FALSE;
	gtk_widget_set_sensitive (dialog->mOptionMenu, active);
}

void helper_chooser_radio_other_toggled (GtkToggleButton *button,
					 HelperAppChooserDialog *dialog)
{
	gboolean active = gtk_toggle_button_get_active (button);
	if (active == TRUE)
		dialog->mOtherHelperApp = TRUE;
	gtk_widget_set_sensitive (dialog->mHelperGnomeEntry, active);
	gtk_widget_set_sensitive (dialog->mHelperEntry, active);
	gtk_widget_set_sensitive (dialog->mCheckUseTerminal, active);
	gtk_widget_set_sensitive (dialog->mCheckUseUrl, active);
}

//////////////////////////////////////////////////////////////////////////////
// begin ProgressDialog callbacks.
//////////////////////////////////////////////////////////////////////////////

gint
progress_dialog_cancel_cb (GtkButton *button,
			   GIProgressCallbacks *aProgress)
{
	aProgress->CancelHelperProgress ();
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

void progress_dialog_pause_cb (GtkButton *button,
			       GIProgressCallbacks *aProgress)
{
	aProgress->TogglePause ();
}
