/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#include "ContentHandler.h"
#include "ExternalProtocolService.h"
#include "FilePicker.h"
#include "FtpProtocolHandler.h"
#include "IRCProtocolHandler.h"
#include "MailtoProtocolHandler.h"
#include "MyportalProtocolHandler.h"
#include "PromptService.h"
#include "npl/gNplCID.h"

#include "GnomeHelpProtocolHandler.h"
#include "GHelpProtocolHandler.h"
#include "InfoProtocolHandler.h"
#include "ManProtocolHandler.h"
#include "TOCProtocolHandler.h"

#include "nsIFactory.h"
#include "nsIComponentManager.h"
#include "nsCOMPtr.h"
#include "nsILocalFile.h"

static NS_DEFINE_CID(kContentHandlerCID, G_CONTENTHANDLER_CID);
static NS_DEFINE_CID(kProtocolServiceCID, G_EXTERNALPROTOCOLSERVICE_CID);
static NS_DEFINE_CID(kFilePickerCID, G_FILEPICKER_CID);
static NS_DEFINE_CID(knsFtpProtocolHandlerCID, NS_FTPPROTOCOLHANDLER_CID);
static NS_DEFINE_CID(kFtpHandlerCID, G_FTP_PROTOCOL_CID);
static NS_DEFINE_CID(kIRCHandlerCID, G_IRC_PROTOCOL_CID);
static NS_DEFINE_CID(kMailtoHandlerCID, G_MAILTO_PROTOCOL_CID);
static NS_DEFINE_CID(kMyportalHandlerCID, G_MYPORTAL_PROTOCOL_CID);
static NS_DEFINE_CID(kPromptServiceCID, G_PROMPTSERVICE_CID);
static NS_DEFINE_CID(kGnomeHelpUrlCID, G_GNOMEHELPURL_CID);
static NS_DEFINE_CID(kMailtoUrlCID, NS_MAILTOURL_CID);

static NS_DEFINE_CID(kGnomeHelpHandlerCID, G_GNOMEHELP_PROTOCOL_CID);
static NS_DEFINE_CID(kGHelpHandlerCID, G_GHELP_PROTOCOL_CID);
static NS_DEFINE_CID(kInfoHandlerCID, G_INFO_PROTOCOL_CID);
static NS_DEFINE_CID(kManHandlerCID, G_MAN_PROTOCOL_CID);
static NS_DEFINE_CID(kTOCHandlerCID, G_TOC_PROTOCOL_CID);

//RegisterFactory is local
NS_METHOD RegisterFactory (nsresult (aFactoryFunc)(nsIFactory** aFactory),
			   const nsCID & aClass, const char *aClassName,
			   const char *aContractID, PRBool aReplace);

NS_METHOD RegisterComponent (const nsCID & aClass, const char *aClassName,
			     const char *aContractID, const char *aDLLPath,
			     PRBool aReplace);

//Annoying globals to track the mozilla ftp handler so it can be restored.
static PRBool ftpRegistered = PR_FALSE;
static nsCOMPtr<nsIFactory> nsFtpFactory;

extern "C" gboolean
mozilla_register_components (void)
{
	gboolean ret = TRUE;
	nsresult rv;

	rv = RegisterFactory (NS_NewContentHandlerFactory, kContentHandlerCID,
			      NS_IHELPERAPPLAUNCHERDLG_CLASSNAME,
			      NS_IHELPERAPPLAUNCHERDLG_CONTRACTID, PR_TRUE);
	if (NS_FAILED(rv)) ret = FALSE;

#ifdef XPCOM_SHARED_LIB
	rv = RegisterComponent (kProtocolServiceCID,
				G_EXTERNALPROTOCOLSERVICE_CLASSNAME,
				NS_EXTERNALPROTOCOLSERVICE_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
#else
	rv = RegisterFactory (NS_NewExternalProtocolServiceFactory,
			      kProtocolServiceCID,
			      G_EXTERNALPROTOCOLSERVICE_CLASSNAME,
			      NS_EXTERNALPROTOCOLSERVICE_CONTRACTID, PR_TRUE);
#endif
	if (NS_FAILED(rv)) ret = FALSE;

	rv = RegisterFactory (NS_NewFilePickerFactory, kFilePickerCID,
			      G_FILEPICKER_CLASSNAME, G_FILEPICKER_CONTRACTID,
			      PR_TRUE);
	if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewFtpHandlerFactory, kFtpHandlerCID,
			      G_FTP_CONTENT_CLASSNAME, G_FTP_CONTENT_CONTRACTID,
			      PR_TRUE);
	if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewIRCHandlerFactory, kIRCHandlerCID,
			      G_IRC_PROTOCOL_CLASSNAME,   
			      G_IRC_PROTOCOL_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewIRCHandlerFactory, kIRCHandlerCID,
			      G_IRC_CONTENT_CLASSNAME,   
			      G_IRC_CONTENT_CONTRACTID, PR_TRUE);        
        if (NS_FAILED(rv)) ret = FALSE;

	rv = RegisterFactory (NS_NewMyportalHandlerFactory,
			      kMyportalHandlerCID, G_ABOUT_MYPORTAL_CLASSNAME,
			      G_ABOUT_MYPORTAL_CONTRACTID, PR_TRUE);
	if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewMyportalHandlerFactory,
        		      kMyportalHandlerCID, G_MYPORTAL_HANDLER_CLASSNAME,
			      G_MYPORTAL_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewPromptServiceFactory, kPromptServiceCID,
			      G_PROMPTSERVICE_CLASSNAME,   
			      G_PROMPTSERVICE_CONTRACTID, PR_TRUE);        
        if (NS_FAILED(rv)) ret = FALSE;

	rv = RegisterComponent (kGnomeHelpUrlCID,
				G_GNOMEHELPURL_CLASSNAME,
				G_GNOMEHELPURL_CONTRACTID,
				SHARE_DIR"/components/libgaleon-npl.so",
				PR_TRUE);
	if (NS_FAILED(rv)) ret = FALSE;

//We only register this mailto handler if mozilla mailnews hasn't already
//done so, so we pass PR_FALSE for aReplace

	rv = RegisterComponent (kMailtoUrlCID,
				NS_MAILTOURL_CLASSNAME,
				NS_MAILTOURL_CONTRACTID,
				SHARE_DIR"/components/libgaleon-npl.so",
				PR_FALSE);
	if (NS_FAILED(rv)) ret = FALSE;

	return ret;
}

NS_METHOD RegisterFactory (nsresult (aFactoryFunc)(nsIFactory** aFactory),
			   const nsCID & aClass, const char *aClassName,
			   const char *aContractID, PRBool aReplace)
{
	if (!pushed_startup) return NS_ERROR_FAILURE;

	nsresult rv = NS_OK;

	nsCOMPtr<nsIFactory> aFactory;
	rv = aFactoryFunc(getter_AddRefs(aFactory));
	if (NS_FAILED(rv)) return rv;
	rv = nsComponentManager::RegisterFactory(aClass, aClassName,
						 aContractID,
						 aFactory, aReplace);
	return rv;
}

NS_METHOD RegisterComponent (const nsCID & aClass, const char *aClassName,
			     const char *aContractID, const char *aDLLPath,
			     PRBool aReplace)
{
	if (!pushed_startup) return NS_ERROR_FAILURE;

	nsresult rv = NS_OK;

	nsCOMPtr<nsILocalFile> dllFile;
	rv = NS_NewLocalFile (aDLLPath, PR_TRUE, getter_AddRefs (dllFile));
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	rv = nsComponentManager::RegisterComponentSpec (aClass,
							aClassName,
							aContractID,
						 	dllFile,
						 	aReplace,
						 	PR_FALSE);
	return rv;
}

/**
 * mozilla_register_FtpProtocolHandler: Register Ftp Protocol Handler
 */
extern "C" gboolean mozilla_register_FtpProtocolHandler (void)
{
	if (!pushed_startup) return FALSE;
	if (ftpRegistered == PR_TRUE) return TRUE;

	nsresult rv = NS_OK;
     
        rv = nsComponentManager::FindFactory (knsFtpProtocolHandlerCID,
                                              getter_AddRefs(nsFtpFactory));
        if (NS_FAILED(rv)) return FALSE;

	rv = RegisterFactory (NS_NewFtpHandlerFactory, kFtpHandlerCID,
			      G_FTP_PROTOCOL_CLASSNAME, 
			      G_FTP_PROTOCOL_CONTRACTID, PR_TRUE);  

	if (NS_FAILED(rv)) return FALSE;

	ftpRegistered = PR_TRUE;
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_unregister_FtpProtocolHandler: Unregister Ftp Protocol Handler
 */
extern "C" gboolean mozilla_unregister_FtpProtocolHandler (void)
{
        if (!pushed_startup) return FALSE;
	if (ftpRegistered == PR_FALSE) return FALSE;
        
        nsresult rv = NS_OK;
	
	rv = nsComponentManager::RegisterFactory(knsFtpProtocolHandlerCID,
						 NS_FTPPROTOCOLHANDLER_CLASSNAME,
						 G_FTP_PROTOCOL_CONTRACTID,
						 nsFtpFactory, PR_TRUE);

	ftpRegistered = PR_FALSE;
        return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_register_MailtoProtocolHandler: Register Mailto Protocol Handler
 */
extern "C" gboolean mozilla_register_MailtoProtocolHandler (void)
{
        if (!pushed_startup) return FALSE;
        
        nsresult rv = NS_OK;

        rv = RegisterFactory (NS_NewMailtoHandlerFactory, kMailtoHandlerCID,
			      G_MAILTO_PROTOCOL_CLASSNAME,   
			      G_MAILTO_PROTOCOL_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) return FALSE;

        rv = RegisterFactory (NS_NewMailtoHandlerFactory, kMailtoHandlerCID,
			      G_MAILTO_CONTENT_CLASSNAME,   
			      G_MAILTO_CONTENT_CONTRACTID, PR_TRUE);        
        return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_register_gHelpProtocolHandlers: Register gHelp Protocol Handlers
 * eg: man, info, etc.
 */
extern "C" gboolean mozilla_register_gHelpProtocolHandlers (void)
{
        if (!pushed_startup) return FALSE;

	gboolean ret = TRUE;
	nsresult rv;

#ifdef XPCOM_SHARED_LIB
        rv = RegisterComponent (kTOCHandlerCID, G_TOC_HANDLER_CLASSNAME,
				G_TOC_HANDLER_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterComponent (kManHandlerCID, G_MAN_HANDLER_CLASSNAME,
				G_MAN_HANDLER_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterComponent (kInfoHandlerCID, G_INFO_HANDLER_CLASSNAME,
				G_INFO_HANDLER_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterComponent (kGnomeHelpHandlerCID,
        			G_GNOMEHELP_HANDLER_CLASSNAME,
				G_GNOMEHELP_HANDLER_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterComponent (kGHelpHandlerCID,
        			G_GHELP_HANDLER_CLASSNAME,
				G_GHELP_HANDLER_CONTRACTID,
				SHARE_DIR"/components/libgaleon-xpcom.so",
				PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;
#else
        rv = RegisterFactory (NS_NewTOCHandlerFactory,
        		      kTOCHandlerCID, G_TOC_HANDLER_CLASSNAME,
			      G_TOC_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewManHandlerFactory,
        		      kManHandlerCID, G_MAN_HANDLER_CLASSNAME,
			      G_MAN_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewInfoHandlerFactory,
        		      kInfoHandlerCID, G_INFO_HANDLER_CLASSNAME,
			      G_INFO_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewGnomeHelpHandlerFactory,
        		      kGnomeHelpHandlerCID,
        		      G_GNOMEHELP_HANDLER_CLASSNAME,
			      G_GNOMEHELP_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;

        rv = RegisterFactory (NS_NewGHelpHandlerFactory,
        		      kGHelpHandlerCID,
        		      G_GHELP_HANDLER_CLASSNAME,
			      G_GHELP_HANDLER_CONTRACTID, PR_TRUE);
        if (NS_FAILED(rv)) ret = FALSE;
#endif
	return ret;
}

/**
 * mozilla_unregister_gHelpProtocolHandlers: Unregister gHelp Protocol Handlers
 */
extern "C" gboolean mozilla_unregister_gHelpProtocolHandlers (void)
{
        if (!pushed_startup) return FALSE;

	gboolean ret = TRUE;
	nsresult rv;
#ifdef XPCOM_SHARED_LIB
	rv = nsComponentManager::UnregisterComponent 
		(kTOCHandlerCID, SHARE_DIR"/components/libgaleon-xpcom.so");
	if (NS_FAILED(rv)) ret = FALSE;

	rv = nsComponentManager::UnregisterComponent
		(kManHandlerCID, SHARE_DIR"/components/libgaleon-xpcom.so");
	if (NS_FAILED(rv)) ret = FALSE;

	rv = nsComponentManager::UnregisterComponent
		(kInfoHandlerCID, SHARE_DIR"/components/libgaleon-xpcom.so");
	if (NS_FAILED(rv)) ret = FALSE;

	rv = nsComponentManager::UnregisterComponent
		(kGnomeHelpHandlerCID, SHARE_DIR"/components/libgaleon-xpcom.so");
	if (NS_FAILED(rv)) ret = FALSE;

	rv = nsComponentManager::UnregisterComponent
		(kGHelpHandlerCID, SHARE_DIR"/components/libgaleon-xpcom.so");
	if (NS_FAILED(rv)) ret = FALSE;
#else
	nsCOMPtr<nsIFactory> factory;
        rv = nsComponentManager::FindFactory (kTOCHandlerCID,
                                              getter_AddRefs(factory));
	if (NS_SUCCEEDED(rv))
	{
		rv = nsComponentManager::UnregisterFactory (kTOCHandlerCID,
							    factory);
		if (NS_FAILED(rv)) ret = FALSE;
	}

        rv = nsComponentManager::FindFactory (kManHandlerCID,
                                              getter_AddRefs(factory));
	if (NS_SUCCEEDED(rv))
	{
		rv = nsComponentManager::UnregisterFactory (kManHandlerCID,
							    factory);
		if (NS_FAILED(rv)) ret = FALSE;
	}

        rv = nsComponentManager::FindFactory (kInfoHandlerCID,
                                              getter_AddRefs(factory));
	if (NS_SUCCEEDED(rv))
	{
		rv = nsComponentManager::UnregisterFactory (kInfoHandlerCID,
							    factory);
		if (NS_FAILED(rv)) ret = FALSE;
	}

        rv = nsComponentManager::FindFactory (kGnomeHelpHandlerCID,
                                              getter_AddRefs(factory));
	if (NS_SUCCEEDED(rv))
	{
		rv = nsComponentManager::UnregisterFactory (kGnomeHelpHandlerCID,
							    factory);
		if (NS_FAILED(rv)) ret = FALSE;
	}

        rv = nsComponentManager::FindFactory (kGHelpHandlerCID,
                                              getter_AddRefs(factory));
	if (NS_SUCCEEDED(rv))
	{
		rv = nsComponentManager::UnregisterFactory (kGHelpHandlerCID,
							    factory);
		if (NS_FAILED(rv)) ret = FALSE;
	}
#endif
	return ret;
}

