/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mime.h"
#include "prefs.h"
#include "misc.h"
#include "spinner.h"
#include "history.h"
#include "window.h"
#include "mozilla.h"
#include "dialog.h"

#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-icon-list.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-file-entry.h>

/* from nsICache.h */
enum
{
	STORE_ANYWHERE	      = 0,
	STORE_IN_MEMORY	      = 1,
	STORE_ON_DISK	      = 2,
	STORE_ON_DISK_AS_FILE = 3
};

/* local function prototypes */
void prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_cancel_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
gboolean prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd);
void prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);

/* mime prototypes */
void prefs_mime_list_select_row_cb (GtkCList *clist, gint row, gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd);
void prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				      GdkEventButton *event, 
				      PreferencesDialog *pd);
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd);
void prefs_mime_type_changed_cb (GtkWidget *entry, PreferencesDialog *pd);
void prefs_mime_action_clicked_cb (GtkWidget *button, PreferencesDialog *pd); 
void prefs_mime_helper_changed_cb (GtkWidget *entry, PreferencesDialog *pd); 
void prefs_mime_helper_buttons_toggled_cb (GtkWidget *button,
					   PreferencesDialog *pd); 
void prefs_mime_new_clicked_cb (GtkButton *button, 
				PreferencesDialog *pd);
void prefs_mime_delete_clicked_cb (GtkButton *button, 
				   PreferencesDialog *pd);

/* other prototypes */
void prefs_homepage_my_portal_button_clicked_cb (GtkButton *button,
						 PreferencesDialog *pd);
void prefs_homepage_current_button_clicked_cb (GtkButton *button,
					       PreferencesDialog *pd);
void prefs_lang_add_button_clicked_cb (GtkButton *button, 
				       PreferencesDialog *pd);
void prefs_lang_remove_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd);
void prefs_lang_up_button_clicked_cb (GtkButton *button, 
				      PreferencesDialog *pd);
void prefs_lang_down_button_clicked_cb (GtkButton *button, 
					PreferencesDialog *pd);
void prefs_clear_disk_cache_button_clicked_cb (GtkButton *button,
					       PreferencesDialog *pd);
void prefs_clear_memory_cache_button_clicked_cb (GtkButton *button,
						 PreferencesDialog *pd);
void prefs_clear_history_button_clicked_cb (GtkButton *button,
					    PreferencesDialog *pd);
void prefs_clear_history_question_cb (gint reply, gpointer data);
void prefs_browse_clicked_cb (GnomeFileEntry *fileentry, 
			      PreferencesDialog *pd);
void prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
				   GdkEvent *event, PreferencesDialog *pd);
void  prefs_smart_bm_entry_width_spin_changed_cb (GtkEditable *editable,
						  PreferencesDialog *pd);
void preferences_switch_page_cb (GtkWidget *notebook, 
				 GtkNotebookPage *notebook_page,
				 guint page_num,
				 PreferencesDialog *pd);
void prefs_image_sites_button_clicked_cb (GtkWidget *button, 
					  PreferencesDialog *pd);
void prefs_cookies_button_clicked_cb (GtkWidget *button, 
				      PreferencesDialog *pd);
void prefs_cookie_sites_button_clicked_cb (GtkWidget *button, 
					   PreferencesDialog *pd);
void prefs_passwords_button_clicked_cb (GtkWidget *button, 
					PreferencesDialog *pd);
void prefs_password_sites_button_clicked_cb (GtkWidget *button, 
					     PreferencesDialog *pd);
void prefs_download_use_command_toggled_cb (GtkWidget *button, 
					    PreferencesDialog *pd);
void prefs_cookie_policy_clicked_cb (GtkWidget *button,
				     PreferencesDialog *pd);
void prefs_theme_select_icon_cb (GnomeIconList *list, gint row,
				 GdkEvent *event, PreferencesDialog *pd);
void prefs_javascript_checkbutton_toggled_cb (GtkWidget *button,
					      PreferencesDialog *pd);
void
prefs_autobookmarks_enabled_check_toggled_cb (GtkWidget *button, 
					      PreferencesDialog *pd);
void
prefs_use_external_source_viewer_check_toggled_cb (GtkWidget *button, 
						   PreferencesDialog *pd);
void
prefs_shorten_target_toggled_cb (GtkWidget *button, 
				 PreferencesDialog *pd);
void
prefs_proxy_radiogroup_toggle_cb (GtkRadioButton *radioButton,
				  PreferencesDialog *pd);
void
prefs_proxy_auto_url_reload_cb (GtkButton *button,
				PreferencesDialog *pd);
void prefs_enable_history_dock_clicked_cb (GtkToggleButton *check,
					   PreferencesDialog *pd);
void prefs_enable_bookmarks_dock_clicked_cb (GtkToggleButton *check,
					     PreferencesDialog *pd);

static gboolean block_dock_toggle = FALSE;

/**
 * prefs_ok_button_clicked_cb: preferences dialog ok button clicked
 */
void
prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	preferences_apply (pd);
	preferences_close (pd);
}

/**
 * prefs_apply_button_clicked_cb: preferences dialog apply button clicked
 */
void
prefs_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	preferences_apply (pd);
}

/**
 * prefs_cancel_button_clicked_cb: preferences dialog cancel button clicked
 */
void
prefs_cancel_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	// This is ok for now, but we also need a 'revert' feature
	preferences_close (pd);
}

/**
 * prefs_close_cb: dialog has been closed
 */
gboolean
prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd)
{
	preferences_close (pd);	
	return TRUE;
}

void
prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
        preferences_help ();
}

/**
 * prefs_clear_disk_cache_button_clicked_cb: "Clear disk cache" clicked
 */
void
prefs_clear_disk_cache_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_ON_DISK);
}

/**
 * prefs_clear_memory_cache_button_clicked_cb: "Clear memory cache" clicked
 */
void
prefs_clear_memory_cache_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_IN_MEMORY);
}

/* new mime type button clicked */
void
prefs_mime_new_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	MimeItem *newitem = g_new0 (MimeItem, 1);
	GtkCList *clist = GTK_CLIST (pd->mime_list);
	gchar *text[5];
	gint row;

	newitem->name = g_strdup (_("New mime item"));
	newitem->action = MIME_UNKNOWN;
	newitem->helperApp = g_strdup (_("None"));
	newitem->terminalHelper = FALSE;
	newitem->urlHelper = FALSE;
	newitem->alwaysUse = FALSE;
	text[0] = newitem->name;
	text[1] = _("Unknown");
	text[2] = newitem->helperApp;
	text[3] = _("False");
	text[4] = NULL;
	row = gtk_clist_append (clist, text);
	gtk_clist_set_row_data (clist, row, newitem);
	gtk_clist_select_row (clist, row, 0);
	gtk_clist_moveto (clist, row, 0, 0, 0);
}

/* mime delete button clicked */
void
prefs_mime_delete_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkCList *clist = GTK_CLIST (pd->mime_list);
	gint row;
	
	if (pd->selected_mime == NULL)
	{
		return;
	}

	row = gtk_clist_find_row_from_data (clist, pd->selected_mime);
	gtk_clist_remove (clist, row);
	mime_item_free (pd->selected_mime);
	pd->selected_mime = NULL;

	if (row >= clist->rows) row = clist->rows - 1;
	gtk_clist_select_row (clist, row, 0);
	gtk_clist_moveto (clist, row, 0, 0, 0);
}

/* mime type selected */
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd)
{
	gint act_num = 0;
	
	pd->enable_edit_mime = FALSE;
	pd->selected_mime = (MimeItem*) gtk_clist_get_row_data (clist, row);
		
	switch (pd->selected_mime->action)
	{
	case MIME_SAVE_TO_DISK:
		act_num = 0;	/* save */
		break;
	case MIME_RUN_PROGRAM:	/* run with helper app */
		act_num = 1;
		break;
	case MIME_ASK_ACTION:	/* Ask */
	case MIME_UNKNOWN:	/* default to ask */
		act_num = 2;
		break;
	case MIME_IGNORE:
		break;
	}

	gtk_widget_set_sensitive (pd->mime_action_frame, TRUE);
	gtk_widget_set_sensitive (pd->mime_delete_button, TRUE);
	gtk_widget_set_sensitive (pd->mime_type_label, TRUE);
	gtk_widget_set_sensitive (pd->mime_type_entry, TRUE);
	gtk_widget_set_sensitive (pd->mime_helper_frame, act_num);

	gtk_radio_button_set (GTK_RADIO_BUTTON (pd->mime_save_to_disk),
			      act_num);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_type_entry), 
			    pd->selected_mime->name);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry),
			    pd->selected_mime->helperApp);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      pd->selected_mime->alwaysUse);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      pd->selected_mime->terminalHelper);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      pd->selected_mime->urlHelper);

	pd->enable_edit_mime = TRUE;
}

/* mime type unselected */
void 
prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				 GdkEventButton *event, PreferencesDialog *pd)
{
	pd->selected_mime = NULL;
	pd->enable_edit_mime = FALSE;

	/* The default is Ask */
	gtk_radio_button_set (GTK_RADIO_BUTTON (pd->mime_save_to_disk), 2);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_type_entry), "");
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry), "");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      FALSE);
	gtk_widget_set_sensitive (pd->mime_action_frame, FALSE);
	gtk_widget_set_sensitive (pd->mime_helper_frame, FALSE);
	gtk_widget_set_sensitive (pd->mime_type_entry, FALSE);
	gtk_widget_set_sensitive (pd->mime_delete_button, FALSE);
	gtk_widget_set_sensitive (pd->mime_type_label, FALSE);
}

/* sort the mime listing */
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd)
{
	if (column == pd->mime_last_sort_column)
		pd->mime_last_sort_type = !pd->mime_last_sort_type;
	else
		pd->mime_last_sort_type = GTK_SORT_ASCENDING;
	gtk_clist_set_sort_type (clist, pd->mime_last_sort_type);
	pd->mime_last_sort_column = column;
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);
}

/* handles changes in the mime type entry */
void
prefs_mime_type_changed_cb (GtkWidget *entry, PreferencesDialog *pd)
{
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	if (pd->selected_mime->name)
		g_free (pd->selected_mime->name);
	pd->selected_mime->name = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_type_entry), 0, -1);

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 0,
			    pd->selected_mime->name);
}

/* handles changes in the mime action */
void
prefs_mime_action_clicked_cb (GtkWidget *button, PreferencesDialog *pd)
{
	gint state;
	gchar *name;
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	state = gtk_radio_button_get (GTK_RADIO_BUTTON (
				      pd->mime_save_to_disk));
	switch (state)
	{
		case 0:
			pd->selected_mime->action = MIME_SAVE_TO_DISK;
			name = _("Save to disk");
			break;
		case 1:
			pd->selected_mime->action = MIME_RUN_PROGRAM;
			name = _("Run with Helper App");
			break;
		case 2:
			pd->selected_mime->action = MIME_ASK_ACTION;
			name = _("Ask the user");
			break;
		default:
			pd->selected_mime->action = MIME_UNKNOWN;
			name = _("Unknown");
			break;
	}

	/* update the clist */
	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 1, name);

	/* set the helper frame sensitive if the type is run or ask */
	gtk_widget_set_sensitive (pd->mime_helper_frame, state);
}

/* handles changes in the helper app entry */
void
prefs_mime_helper_changed_cb (GtkWidget *entry, PreferencesDialog *pd)
{
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	if (pd->selected_mime->helperApp)
	{
		g_free (pd->selected_mime->helperApp);
	}
	pd->selected_mime->helperApp = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_helper_entry), 0, -1);

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 2,
			    pd->selected_mime->helperApp);
}

/* handles changes in the mime helper buttons' states */
void
prefs_mime_helper_buttons_toggled_cb (GtkWidget *button,
				      PreferencesDialog *pd)
{
	gchar *text;
	gint row;

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
		return;

	pd->selected_mime->terminalHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_terminal));
	pd->selected_mime->urlHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_url_helper));
	pd->selected_mime->alwaysUse = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_always_use));

	if (pd->selected_mime->alwaysUse)
		text = _("True");
	else
		text = _("False");

	/* update clist */
	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 3, text);
}

/**
 * prefs_homepage_my_portal_button_clicked_cb: set homepage URL to "My Portal"
 */
void
prefs_homepage_my_portal_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry), MYPORTAL_URL);
}


/**
 * prefs_homepage_current_button_clicked_cb: set homepage URL to current page
 */
void
prefs_homepage_current_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd)
{
	const gchar *url;
	
	url = history_get_last_url ();
	gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry), url);
}

void
prefs_lang_add_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GtkWidget *entry = pd->lang_entry;
	GList *items = NULL;
	GtkWidget *list_item;
	char *entry_text;

	entry_text = gtk_editable_get_chars (GTK_EDITABLE (entry),0,-1);
	list_item = gtk_list_item_new_with_label (entry_text);
	gtk_object_set_data (GTK_OBJECT (list_item), "string", entry_text);

	items = g_list_append (items, list_item);
	gtk_list_append_items (list, items);
	gtk_widget_show (list_item);
}

void
prefs_lang_remove_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		gtk_list_remove_items (list, selected); 
	}
}

void
prefs_lang_up_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		if (pos!=0)
		{
			gtk_list_remove_items_no_unref (list, selected); 
			pos--;
			gtk_list_insert_items (list, selected, pos);
			gtk_list_select_item (list, pos);
		}
	}
}

void
prefs_lang_down_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		gtk_list_remove_items_no_unref (list, selected); 
		pos++;
		gtk_list_insert_items (list, selected, pos);
		gtk_list_select_item (list, pos);
	}
	
}

void
prefs_clear_history_button_clicked_cb (GtkButton *button,
				       PreferencesDialog *pd)
{
	GtkWidget *dialog;
	
	dialog = gnome_question_dialog_modal_parented
		(_("This will delete all items stored in your history.\n"
		   "Are you sure you want to do this?"),
		 (GnomeReplyCallback)prefs_clear_history_question_cb, pd,
		 GTK_WINDOW (pd->dialog));

	dialog_set_parent (dialog, pd->dialog);
	gnome_dialog_run (GNOME_DIALOG (dialog));
}

void
prefs_clear_history_question_cb (gint reply, gpointer data)
{
	if (reply)
	{
		return;
	}

	/* clear the hash table */
	history_clear ();
}

void
prefs_browse_clicked_cb (GnomeFileEntry *fileentry, PreferencesDialog *pd)
{
	if (fileentry->fsw != NULL)
	{
		dialog_set_parent (fileentry->fsw, pd->dialog);
	}
}

void
prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_FONT_PICKER (widget)->font_dialog,
			   pd->dialog);
}

void
prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_COLOR_PICKER (widget)->cs_dialog,
			   pd->dialog);
}

void 
prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
			      GdkEvent *event, PreferencesDialog *pd)
{
	const gchar *path;

	path = spinner_get_path_from_index (num);
	gtk_entry_set_text (GTK_ENTRY (pd->spinner_dir_entry), path);
}

void
prefs_theme_select_icon_cb (GnomeIconList *list, gint row, GdkEvent *event, 
			    PreferencesDialog *pd)
{
	const gchar *path;

	path = gnome_icon_list_get_icon_data (list, row);
	gtk_entry_set_text (GTK_ENTRY (pd->theme_dir_entry), path);
}

void 
prefs_smart_bm_entry_width_spin_changed_cb (GtkEditable *editable,
					    PreferencesDialog *pd)
{
	gint width;

	width = gtk_spin_button_get_value_as_int 
		(GTK_SPIN_BUTTON (pd->smart_bm_entry_width_spin));
	gtk_widget_set_usize (GTK_WIDGET (pd->smart_bm_example_entry),
			      width, -2);
}

/**
 * preferences_switch_page_cb: called when the user jumps to a new page
 */
void
preferences_switch_page_cb (GtkWidget *notebook, 
			    GtkNotebookPage *notebook_page,
			    guint page_num,
			    PreferencesDialog *pd)
{
	/* if it's the themes page, load spinners into the list */
	/* this is quite slow, so we only do it if the user selects
	 * this specific page */
	if (notebook_page->child == pd->themes_page)
	{
		spinner_fill_iconlist (GNOME_ICON_LIST (pd->spinner_iconlist));
	}
}

/**
 * prefs_image_sites_button_clicked_cb
 */
void
prefs_image_sites_button_clicked_cb (GtkWidget *button, 
				     PreferencesDialog *pd)
{
	pdm_show_page (NULL, IMAGESITES);
}

/**
 * prefs_cookies_button_clicked_cb
 */
void
prefs_cookies_button_clicked_cb (GtkWidget *button, 
				 PreferencesDialog *pd)
{
	pdm_show_page (NULL, COOKIES);
}

/**
 * prefs_cookie_sites_button_clicked_cb
 */
void
prefs_cookie_sites_button_clicked_cb (GtkWidget *button, 
				      PreferencesDialog *pd)
{
	pdm_show_page (NULL, COOKIESITES);
}

/**
 * prefs_passwords_button_clicked_cb
 */
void
prefs_passwords_button_clicked_cb (GtkWidget *button, 
				   PreferencesDialog *pd)
{
	pdm_show_page (NULL, PASSWORDS);
}

/**
 * prefs_password_sites_button_clicked_cb
 */
void
prefs_password_sites_button_clicked_cb (GtkWidget *button, 
				        PreferencesDialog *pd)
{
	pdm_show_page (NULL, PASSWORDSITES);
}

/**
 * prefs_downloader_use_command_toggled_cb: called when the use command
 * radiobutton is toggled; we have to set the command entry and terminal
 * widgets to sensitive or not, appropriately.
 */
void
prefs_download_use_command_toggled_cb (GtkWidget *button, 
				       PreferencesDialog *pd)
{
	gboolean active;

	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->download_command_combo, active);
	gtk_widget_set_sensitive (pd->download_command_label, active);
	gtk_widget_set_sensitive (pd->download_run_in_terminal, active);
}

/* toggle the sensitivity of the "warn before accepting cookie"
 * checkbutton, depending on whether all cookies are being denied or not */
void
prefs_cookie_policy_clicked_cb (GtkWidget *button, PreferencesDialog *pd)
{
	gint state;

	state = gtk_radio_button_get (GTK_RADIO_BUTTON (button));
	gtk_widget_set_sensitive (pd->warn_cookie_checkbutton,
				  (state == 0 || state == 1));
}

void
prefs_autobookmarks_enabled_check_toggled_cb (GtkWidget *button, 
					      PreferencesDialog *pd)
{
	gboolean active;
	
	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->autobookmarks_count_spin, active);
	gtk_widget_set_sensitive (pd->autobookmarks_shorten_spin, active);
}

void
prefs_use_external_source_viewer_check_toggled_cb (GtkWidget *button, 
						   PreferencesDialog *pd)
{
	gboolean active;
	
	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->external_source_viewer_entry, active);
}

void
prefs_shorten_target_toggled_cb (GtkWidget *button, 
				 PreferencesDialog *pd)
{
	gboolean active;
	
	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->tabbed_shorten_spin, active);
	gtk_widget_set_sensitive (pd->tabbed_shorten_spin2, !active);
	gtk_widget_set_sensitive (pd->tabbed_shorten_name_vbox, active);
}

void
prefs_encoding_activated_cb (GtkWidget *button, 
			     PreferencesDialog *pd)
{
	prefs_fonts_remember (pd);
	prefs_fonts_language_init (pd);
}

void
prefs_javascript_checkbutton_toggled_cb (GtkWidget *button,
					 PreferencesDialog *pd)
{
	gboolean active;

	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->statusbar_rewrite_check, active);
}



void
prefs_wheel_changed_cb (GtkWidget *widget,  PreferencesDialog *pd)
{
	mouse_wheel_controls_set_sensitive (pd);
}

void
prefs_proxy_radiogroup_toggle_cb (GtkRadioButton *radioButton,
				  PreferencesDialog *pd)
{
	/* get value from radio button group */
	gint index = gtk_radio_button_get (radioButton);
	switch (index)
	{
	 case 0:gtk_widget_set_sensitive (pd->proxy_auto_url_entry, FALSE);
		gtk_widget_set_sensitive (pd->proxy_auto_reload_button, FALSE);
		gtk_widget_set_sensitive (pd->http_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->ssl_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->ftp_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->http_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->ssl_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->ftp_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->no_proxies_for_entry, FALSE);
		break;
	 case 1:gtk_widget_set_sensitive (pd->proxy_auto_url_entry, FALSE);
		gtk_widget_set_sensitive (pd->proxy_auto_reload_button, FALSE);
		gtk_widget_set_sensitive (pd->http_proxy_entry, TRUE);
		gtk_widget_set_sensitive (pd->ssl_proxy_entry, TRUE);
		gtk_widget_set_sensitive (pd->ftp_proxy_entry, TRUE);
		gtk_widget_set_sensitive (pd->http_proxy_port_spin, TRUE);
		gtk_widget_set_sensitive (pd->ssl_proxy_port_spin, TRUE);
		gtk_widget_set_sensitive (pd->ftp_proxy_port_spin, TRUE);
		gtk_widget_set_sensitive (pd->no_proxies_for_entry, TRUE);
		break; 
	 case 2:gtk_widget_set_sensitive (pd->proxy_auto_url_entry, TRUE);
		gtk_widget_set_sensitive (pd->proxy_auto_reload_button, TRUE);
		gtk_widget_set_sensitive (pd->http_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->ssl_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->ftp_proxy_entry, FALSE);
		gtk_widget_set_sensitive (pd->http_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->ssl_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->ftp_proxy_port_spin, FALSE);
		gtk_widget_set_sensitive (pd->no_proxies_for_entry, FALSE);
		break; 
	}
}
void
prefs_proxy_auto_url_reload_cb (GtkButton *button,
				PreferencesDialog *pd)
{
	gchar *url = gtk_entry_get_text (GTK_ENTRY(pd->proxy_auto_url_entry));
	mozilla_reload_proxy_autoconfiguration (url);
}

void
prefs_enable_history_dock_clicked_cb (GtkToggleButton *check,
				      PreferencesDialog *pd)
{
	if (block_dock_toggle) return;

	if (GTK_TOGGLE_BUTTON (pd->show_bookmarks_dock_check)->active)
	{
		block_dock_toggle = TRUE;
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (pd->show_bookmarks_dock_check),
			 FALSE);
		block_dock_toggle = FALSE;
	}
}

void
prefs_enable_bookmarks_dock_clicked_cb (GtkToggleButton *check,
					PreferencesDialog *pd)
{
	if (block_dock_toggle) return;

	if (GTK_TOGGLE_BUTTON (pd->show_history_dock_check)->active)
	{
		block_dock_toggle = TRUE;
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (pd->show_history_dock_check),
			 FALSE);
		block_dock_toggle = FALSE;
	}	
}
