/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc.h"
#include "themes.h"
#include "eel-gconf-extensions.h"

#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-util.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* cache of theme pixmaps */
static GHashTable *pixmaps_cache = NULL;
static GHashTable *prelight_pixmaps_cache = NULL;

/* Local prototypes */
static gchar *get_nautilus_theme_path (void);
static gboolean theme_pixmap_free_cb (gpointer key, gpointer value,
				      gpointer user_data);

/**
 * get_nautilus_theme_path: get the path of the 
 * current nautilus theme
 */
static gchar *
get_nautilus_theme_path (void)
{
	int i;
	char *theme;
	char *icon, *tmp;
	char *result = NULL;

	static gchar *path[] =
	{
		SHARE_DIR "/../pixmaps/nautilus/%s",
		SHARE_DIR "/../pixmaps/nautilus/%s/fake-aa-navibar",
		NULL,
		NULL,
		SHARE_DIR "/../pixmaps/nautilus/",
		NULL
	};

	theme = eel_gconf_get_string(CONF_NAUTILUS_THEME);

	if (!theme || 
	    strcmp (theme, "default") == 0)
	{
		g_free (theme);
		theme = g_strdup ("");
	}
	else
	if (strcmp (theme, "gnome") == 0) /* use standard icons */
	{
		g_free (theme);
		return NULL;
	}

	path[2] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/%s", NULL);
	path[3] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/fake-aa-navibar/%s",
			       NULL);

	for (i = 0; path[i] != NULL && !result; i++)
	{
		tmp = g_strdup_printf (path[i], theme);
		icon = g_strconcat (tmp, "/Refresh.png", NULL);

		if (g_file_exists (icon))
		{
			result = g_strdup (tmp);
		}
		g_free (tmp);
		g_free (icon);
	}

	g_free (path [2]);
	g_free (path [3]);
	g_free (theme);

	return result;
}

/**
 * theme_pixmap_free_cb: free one theme pixmap
 */
static gboolean
theme_pixmap_free_cb (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);
	return TRUE;
}

/**
 * reset_theme_pixmap_cache: Reset the theme pixmap cache, used when changing
 * themes
 */
void
reset_theme_pixmap_cache (void)
{
	if (prelight_pixmaps_cache != NULL)
	{
		g_hash_table_freeze (prelight_pixmaps_cache);
		g_hash_table_foreach_remove (prelight_pixmaps_cache,
			(GHRFunc) theme_pixmap_free_cb, NULL);
		g_hash_table_thaw (prelight_pixmaps_cache);
		g_hash_table_destroy (prelight_pixmaps_cache);
		prelight_pixmaps_cache = NULL;
	}

	if (pixmaps_cache != NULL)
	{
		g_hash_table_freeze (pixmaps_cache);
		g_hash_table_foreach_remove (pixmaps_cache,
			(GHRFunc) theme_pixmap_free_cb, NULL);
		g_hash_table_thaw (pixmaps_cache);
		g_hash_table_destroy (pixmaps_cache);
		pixmaps_cache = NULL;
	}
}

/**
 * get_theme_pixmap: get the PixmapData of a given file, taking themes and
 * caching into account.
 */
PixmapData *
get_theme_pixmap (const gchar *filename, gboolean prelight)
{
	PixmapData *pixmap = NULL;
	gchar *theme, *tmp;

	/* lookup in cache */
	if (prelight && prelight_pixmaps_cache != NULL)
		pixmap = g_hash_table_lookup (prelight_pixmaps_cache, filename);
	else if (!prelight && pixmaps_cache != NULL)
		pixmap = g_hash_table_lookup (pixmaps_cache, filename);

	if (pixmap != NULL)
		return pixmap;

	theme = get_theme_dir ();
	
	if (theme)
	{
		tmp = g_concat_dir_and_file (theme, filename);

		if (g_file_exists (tmp))
		{
			pixmap = pixmap_data_new_from_file (tmp, prelight);
		}

		g_free (tmp);
		g_free (theme);
	}
	
	if (pixmap == NULL)
	{
		tmp = g_concat_dir_and_file (SHARE_DIR, filename);
		pixmap = pixmap_data_new_from_file (tmp, prelight);
		g_free (tmp);
	}

	/* if we didn't manage to load the icon, make an empty structure
	 * anyway, this allows the browser to run even when not fully 
	 * installed.... */
	if (pixmap == NULL)
	{
		pixmap = g_new0 (PixmapData, 1);
	}

	/* cache the created PixmapData */
	if (prelight)
	{
		if (prelight_pixmaps_cache == NULL)
		{
			prelight_pixmaps_cache = g_hash_table_new (g_str_hash,
							  	   g_str_equal);
		}
		g_hash_table_insert (prelight_pixmaps_cache,
				     g_strdup (filename), pixmap);
	}
	else
	{
		if (pixmaps_cache == NULL)
		{
			pixmaps_cache = g_hash_table_new (g_str_hash,
							  g_str_equal);
		}
		g_hash_table_insert (pixmaps_cache, g_strdup (filename),
				     pixmap);
	}

	return pixmap;
}

/**
 * get_theme_pixmap_height: get the pixmap height of the current toolbar theme
 */
gint
get_theme_pixmap_height ()
{
	gchar *theme = get_theme_dir ();
	gchar *filename = NULL;
	GdkPixbuf *pixbuf;
	gint icon_height;

	if (theme && strcmp (theme, "") != 0)
	{
		/* Refresh seems to be the highest pixmap */
		filename = g_concat_dir_and_file (theme, "Refresh.png");
	}

	if (filename && g_file_exists (filename))
	{
		pixbuf = gdk_pixbuf_new_from_file (filename);
		icon_height = gdk_pixbuf_get_height (pixbuf);
		gdk_pixbuf_unref (pixbuf);
	}
	else
	{
		icon_height = 24; /* GNOME stock pixmap height */
	}

	if (filename) g_free (filename);
	if (theme) g_free (theme);

	return icon_height;
}

/**
 * get_theme_dir: retreive current theme dir
 */
gchar *
get_theme_dir ()
{
	gchar *theme = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);
	
	if (theme && strcmp (theme, "NAUTILUS_THEME") == 0)
	{
		g_free (theme);
		theme = get_nautilus_theme_path ();
	}

	return theme;
}
