/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla.h"

#include <unistd.h>
#include <time.h>
#include <gtkmozembed.h>
#include <gtkmozembed_internal.h>

#include "regex.h"
#include "nsIGlobalHistory.h"
#include "nsIDocShellHistory.h"
#include "nsIDocShellTreeItem.h"
#include "nsIDocShellTreeNode.h"
#include "nsIDocShellTreeOwner.h"
#include "nsIDiskDocument.h"
#include "nsIWebBrowserFind.h"
#include "nsIWebBrowserFocus.h"
#include "nsIDocument.h"
#include "nsFileSpec.h"
#include "nsISHEntry.h"
#include "nsISHistoryInternal.h"
#include "nsIWebBrowserPrint.h"
#include "nsIURI.h"
#include "nsIURL.h"
#include "nsIPresShell.h"
#include "nsTextServicesCID.h"
#include "nsIMarkupDocumentViewer.h"
#include "nsIDOMHTMLElement.h"
#include "nsIDOMHTMLInputElement.h"
#include "nsIDOMNSHTMLElement.h"
#include "nsIDOMHTMLDocument.h"
#include "nsIDOMNodeList.h"
#include "nsIDOMNamedNodeMap.h"
#include "nsIScriptGlobalObject.h"
#include "nsIDOMWindowInternal.h"
#include "nsIDOMHTMLAnchorElement.h"
#include "nsIDOMHTMLAreaElement.h"
#include "nsIContentViewerFile.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIInterfaceRequestor.h"
#include "nsIChannel.h"
#include "nsIFocusController.h"
#include "nsILoadGroup.h"
#include "nsIWebBrowserPersist.h"
#include "nsIStreamTransfer.h"
#include "nsNetUtil.h"
#include "nsIDOMParser.h"
#include "nsIChromeEventHandler.h"
#include "nsIClipboardCommands.h"
#include "GaleonWrapper.h"
#include "PersistProgress.h"
#include "PrintListener.h"
#include "GlobalHistory.h"
#include "nsIGlobalHistory.h"
#include "nsIDOMHTMLImageElement.h"

/* type of drag and drop for links */
static const GtkTargetEntry link_drag_types[] =
{
        { "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
        { "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
        { "STRING",        0, DND_TARGET_STRING       }
};
static const gint link_drag_types_num_items = (sizeof (link_drag_types) / 
					       sizeof (GtkTargetEntry));
static GtkTargetList *link_drag_types_tl = NULL;

static NS_DEFINE_CID(kCTextServicesDocumentCID, NS_TEXTSERVICESDOCUMENT_CID);
static NS_DEFINE_CID(kStandardURLCID, NS_STANDARDURL_CID);

GaleonWrapper::GaleonWrapper ()
{
}

GaleonWrapper::~GaleonWrapper ()
{
}

nsresult GaleonWrapper::Init (GaleonEmbed *galeon_embed)
{
	nsresult result;

	return_val_if_not_embed (galeon_embed, NS_ERROR_FAILURE);

	embed = galeon_embed;
	mGtkMozEmbed = GTK_MOZ_EMBED(embed->mozEmbed);

	gtk_moz_embed_get_nsIWebBrowser (mGtkMozEmbed,
					 getter_AddRefs(mWebBrowser));
	if (!mWebBrowser) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShellHistory> dsHistory = do_QueryInterface (DocShell);
	if (!dsHistory) return NS_ERROR_FAILURE;

	static NS_DEFINE_CID(kGlobalHistoryCID, GALEON_GLOBALHISTORY_CID);

	nsCOMPtr<nsIFactory> GHFactory;
	result = NS_NewGlobalHistoryFactory(getter_AddRefs(GHFactory));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	result = nsComponentManager::RegisterFactory(kGlobalHistoryCID,
						     "Global history",
						     NS_GLOBALHISTORY_CONTRACTID,
						     GHFactory,
						     PR_TRUE);

	nsCOMPtr<nsIGlobalHistory> inst =  
		do_GetService(NS_GLOBALHISTORY_CONTRACTID, &result);
	dsHistory->SetGlobalHistory(inst);

	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;
  
        nsCOMPtr<nsIChromeEventHandler> chromeHandler;
        piWin->GetChromeEventHandler(getter_AddRefs(chromeHandler));
  
	mEventReceiver = do_QueryInterface(chromeHandler);

        mEventListener = new GaleonEventListener();
        mEventListener->Init(this);

	return AddListener ();

	return NS_OK;
}

nsresult GaleonWrapper::GetPIDOMWindow(nsPIDOMWindow **aPIWin)
{
  *aPIWin = nsnull;

  nsCOMPtr<nsIDOMWindow> domWindow;
  mWebBrowser->GetContentDOMWindow(getter_AddRefs(domWindow));
  if (!domWindow)
    return NS_ERROR_FAILURE;

  nsCOMPtr<nsPIDOMWindow> domWindowPrivate = do_QueryInterface(domWindow);
  nsCOMPtr<nsIDOMWindowInternal> rootWindow;
  domWindowPrivate->GetPrivateRoot(getter_AddRefs(rootWindow));
  
  nsCOMPtr<nsIChromeEventHandler> chromeHandler;
  nsCOMPtr<nsPIDOMWindow> piWin(do_QueryInterface(rootWindow));

  *aPIWin = piWin.get();

  if (*aPIWin) {
    NS_ADDREF(*aPIWin);
    return NS_OK;
  }

  return NS_ERROR_FAILURE;
}

/* This is how gtkmozembed is focusing childs,
   I'm not using nsIWebBrowserFocus because gtkmozembed
   doesnt, so I guess something could be buggy there */

nsresult GaleonWrapper::Activate ()
{
	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;

	return piWin->Activate();
}

nsresult GaleonWrapper::Deactivate ()
{
	nsCOMPtr<nsPIDOMWindow> piWin;
	GetPIDOMWindow(getter_AddRefs(piWin));
	if (!piWin) return NS_ERROR_FAILURE;

	return piWin->Deactivate();
}

nsresult GaleonWrapper::SetOffline(PRBool aOffline)
{
	nsresult rv;
	nsCOMPtr<nsIIOService> io = do_GetService(NS_IOSERVICE_CONTRACTID, &rv);
	if (NS_FAILED(rv))
		return NS_ERROR_FAILURE;

	return io->SetOffline(aOffline);
}

nsresult GaleonWrapper::Print (nsIPrintOptions *options, PRBool preview)
{
#if MOZILLA_VERSION >= VERSION3(0,9,3)
	nsresult result;

	nsCOMPtr<nsIWebBrowserPrint> print(do_GetInterface(mWebBrowser, &result));
	if (NS_FAILED(result) || !print) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetMainDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	if (!preview)
	{
		result = print->Print (DOMWindow, options, NULL);
	}
	else
	{
		char *filename;

		filename = g_strconcat(g_get_tmp_dir(),"/galeon-print-XXXXXX", NULL);
		/* get a name for the temporary file */
		int res;
		res = mkstemp (filename);
		if (res == -1)
			return NS_ERROR_FAILURE;
		close (res);
		mozilla_delete_temp_file_on_exit (filename);

		nsCString fileString(filename);
		options->SetToFileName (fileString.ToNewUnicode());
		options->SetPrintToFile (TRUE);

		PrintListener *listener = 
			new PrintListener(filename);
		result = print->Print (DOMWindow, options, listener); 
	}

	return result;
#else
	nsresult rv = NS_OK;

	nsCOMPtr<nsIDocShell> DocShell;
	rv = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(rv) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;
	DocShell->GetContentViewer(getter_AddRefs(contentViewer));
	if (contentViewer) {
		nsCOMPtr<nsIContentViewerFile> contentViewerFile(do_QueryInterface(contentViewer));
		if (contentViewerFile) {
			rv = contentViewerFile->Print(PR_TRUE, nsnull, nsnull);
		}
	}

	return rv;
#endif
}

nsresult GaleonWrapper::GetSHistory (nsISHistory **aSHistory)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = ContentNav->GetSessionHistory (getter_AddRefs (SessionHistory));
	if (!SessionHistory) return NS_ERROR_FAILURE;

	*aSHistory = SessionHistory.get();
	NS_IF_ADDREF (*aSHistory);

	return NS_OK;
}

nsresult GaleonWrapper::GetDocShell (nsIDocShell **aDocShell)
{
	nsCOMPtr<nsIDocShellTreeItem> browserAsItem;
	browserAsItem = do_QueryInterface(mWebBrowser);
	if (!browserAsItem) return NS_ERROR_FAILURE;

	// get the owner for that item
	nsCOMPtr<nsIDocShellTreeOwner> treeOwner;
	browserAsItem->GetTreeOwner(getter_AddRefs(treeOwner));
	if (!treeOwner) return NS_ERROR_FAILURE;

	// get the primary content shell as an item
	nsCOMPtr<nsIDocShellTreeItem> contentItem;
	treeOwner->GetPrimaryContentShell(getter_AddRefs(contentItem));
	if (!contentItem) return NS_ERROR_FAILURE;

	// QI that back to a docshell
	nsCOMPtr<nsIDocShell> DocShell;
	DocShell = do_QueryInterface(contentItem);
	if (!DocShell) return NS_ERROR_FAILURE;

	*aDocShell = DocShell.get();

	NS_IF_ADDREF(*aDocShell);
	
	return NS_OK;
}

nsresult GaleonWrapper::Destroy ()
{
  	RemoveListener ();
      	mWebBrowser = nsnull;
	mChromeNav = nsnull;
	
	return NS_OK;
}

nsresult GaleonWrapper::AddListener ()
{
	nsresult rv = NS_OK;

        if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;

        nsCOMPtr<nsIDOMEventListener> eventListener =
                NS_STATIC_CAST(nsIDOMEventListener *,
                               NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

        rv = mEventReceiver->AddEventListenerByIID(eventListener,
                                                 NS_GET_IID(nsIDOMDragListener));
        if (NS_FAILED(rv)) return NS_ERROR_FAILURE;
	
	return NS_OK;
}

nsresult GaleonWrapper::RemoveListener ()
{
	nsresult rv = NS_OK;

	if (!mEventReceiver || !mEventListener) return NS_ERROR_FAILURE;
        
        nsCOMPtr<nsIDOMEventListener> eventListener =
                NS_STATIC_CAST(nsIDOMEventListener *,
                               NS_STATIC_CAST(nsIDOMDragListener *, mEventListener));

        rv = mEventReceiver->RemoveEventListenerByIID(eventListener,
                                                      NS_GET_IID(nsIDOMDragListener));

	return rv;
}

nsresult GaleonWrapper::GoToHistoryIndex (PRInt16 index)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> ContentNav = do_QueryInterface (DocShell,
								   &result);
	if (!ContentNav) return NS_ERROR_FAILURE;

	return  ContentNav->GotoIndex (index);
}

nsresult GaleonWrapper::SetZoom (float aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->SetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetZoom (float *aZoom)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	return mdv->GetTextZoom (aZoom);
}

nsresult GaleonWrapper::GetDOMWindow (nsIDOMWindow **aDOMWindow)
{
	nsresult rv;

	nsCOMPtr<nsIWebBrowserFocus> focus = do_GetInterface(mWebBrowser, &rv);
	if (NS_FAILED(rv) || !focus) return NS_ERROR_FAILURE;

	rv = focus->GetFocusedWindow (aDOMWindow);
	if (NS_FAILED(rv))
		rv = GetMainDOMWindow (aDOMWindow);
	return rv;
}

nsresult GaleonWrapper::GetMainDOMWindow (nsIDOMWindow **aDOMWindow)
{
	return mWebBrowser->GetContentDOMWindow (aDOMWindow);
}

nsresult GaleonWrapper::GetDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = GetDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow)
	{
		return GetMainDOMDocument (aDOMDocument);
	}

	return DOMWindow->GetDocument (aDOMDocument);
}

nsresult GaleonWrapper::GetMainDOMDocument (nsIDOMDocument **aDOMDocument)
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs (DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	return contentViewer->GetDOMDocument (aDOMDocument);
}

nsresult GaleonWrapper::SaveURI (const char *url, char *filename, int action)
{
        nsresult result;

	nsCOMPtr<nsIWebBrowserPersist> persist(do_GetInterface(mWebBrowser));
	nsCOMPtr<nsIWebBrowserPersistProgress> progress;

	if (action)
	{
		progress = new PersistProgress(filename, action);
		persist->SetProgressListener (progress);
	}

	nsString s;
	s.AssignWithConversion(url);
	nsCOMPtr<nsIURI> linkURI;
	result = NS_NewURI(getter_AddRefs(linkURI), s);
	if (NS_FAILED(result))
		return NS_ERROR_FAILURE;
	return persist->SaveURI(linkURI, nsnull, filename);
}

nsresult GaleonWrapper::SaveLink (const char *url)
{
	nsresult rv;

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	rv = mWebBrowser->GetContentDOMWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(rv) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMWindowInternal> DOMWInternal =
						do_QueryInterface (DOMWindow);
	if (!DOMWInternal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIStreamTransfer> xfer = 
				do_GetService (NS_ISTREAMTRANSFER_CONTRACTID);
	if (!xfer) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,2)
	return xfer->SelectFileAndTransferLocationSpec 
		(url, DOMWInternal, "", "", PR_TRUE, nsnull, nsnull);
#else
	return xfer->SelectFileAndTransferLocationSpec (url, DOMWInternal, "",
							  "", PR_TRUE, nsnull);
#endif
}

nsresult GaleonWrapper::SaveDocument (char *filename, const char *datapath,
				      int action, PRBool mainDoc)
{
	nsresult rv;
	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	if (mainDoc)
		rv = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	else
		rv = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(rv) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersist> persist = do_GetInterface (mWebBrowser,
								  &rv);
	if (NS_FAILED(rv) || !persist) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebBrowserPersistProgress> progress;
	if (action)
	{
		progress = new PersistProgress(filename, action);
		persist->SetProgressListener (progress);
	}

	return persist->SaveDocument (aDOMDocument, filename, datapath);
}

nsresult GaleonWrapper::GetSHInfo (PRInt32 *count, PRInt32 *index)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	SessionHistory->GetCount (count);
	SessionHistory->GetIndex (index);	

	return NS_OK;
}

nsresult GaleonWrapper::GetSHTitleAtIndex (PRInt32 index, PRUnichar **title)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	result = he->GetTitle (title);
	if (!NS_SUCCEEDED(result) || (!title)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetSHUrlAtIndex (PRInt32 index, char **url)
{
	nsresult result;

	nsCOMPtr<nsISHistory> SessionHistory;
	result = GetSHistory (getter_AddRefs(SessionHistory));
	if (NS_FAILED(result) || ! SessionHistory) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIHistoryEntry> he;
	result = SessionHistory->GetEntryAtIndex (index, PR_FALSE,
						  getter_AddRefs (he));
	if (!NS_SUCCEEDED(result) || (!he)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIURI> uri;
	result = he->GetURI (getter_AddRefs(uri));
	if (!NS_SUCCEEDED(result) || (!uri)) return NS_ERROR_FAILURE;

	result = uri->GetSpec(url);
	if (!NS_SUCCEEDED(result) || (!*url)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::Find (const PRUnichar *search_string, 
			      PRBool matchcase, PRBool search_backwards,
			      PRBool *didFind)
{
	nsCOMPtr<nsIWebBrowserFind> finder (do_GetInterface(mWebBrowser));

	finder->SetSearchString (search_string);
	finder->SetFindBackwards (search_backwards);
	finder->SetMatchCase (matchcase);
	return finder->FindNext(didFind);
}

nsresult GaleonWrapper::ReloadDocument ()
{
	nsresult result;

	nsCOMPtr<nsIWebBrowserFocus> focus(do_GetInterface(mWebBrowser));

	nsCOMPtr<nsIDOMWindow> DOMWindow;
	result = focus->GetFocusedWindow (getter_AddRefs(DOMWindow));
	if (NS_FAILED(result) || !DOMWindow) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIScriptGlobalObject> scriptGlobal = do_QueryInterface(DOMWindow);
	if (!scriptGlobal) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> docshell;
	if (NS_FAILED(scriptGlobal->GetDocShell(getter_AddRefs(docshell))))
        return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn = do_QueryInterface (docshell, &result);
	if (!wn || !NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	result = wn->Reload (nsIWebNavigation::LOAD_FLAGS_BYPASS_CACHE | 
			     nsIWebNavigation::LOAD_FLAGS_BYPASS_PROXY);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;

	return NS_OK;
}

nsresult GaleonWrapper::GetMainDocumentUrl (char **url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,1)
	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));
#else
	nsCOMPtr<nsIURI> uri = getter_AddRefs(doc->GetDocumentURL());
#endif
	uri->GetSpec (url);

	return NS_OK;
}

nsresult GaleonWrapper::GetDocumentUrl (char **url)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;

	result = GetDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(aDOMDocument);
	if(!doc) return NS_ERROR_FAILURE;

#if MOZILLA_VERSION > VERSION3(0,9,1)
	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));
#else
	nsCOMPtr<nsIURI> uri = getter_AddRefs(doc->GetDocumentURL());
#endif
	uri->GetSpec (url);

	return NS_OK;
}

nsresult GaleonWrapper::GetKeyEventContext (nsIDOMKeyEvent *keyEvent, WrapperKeyEventInfo *info)
{
	PRBool mod_key;
	nsresult res;

	/* if we are in a text area do not process keys */
	nsCOMPtr<nsIDOMEventTarget>  targetNode;
	res = keyEvent->GetTarget(getter_AddRefs(targetNode));
	if (NS_FAILED(res) || !targetNode) return NS_ERROR_FAILURE;
	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(targetNode);
	/* don't grab alt combos, thus you can still access the menus and
	 *  other alt functions.  */
	keyEvent->GetAltKey (&mod_key);
	if (node && !mod_key)
	{
		nsCOMPtr<nsIDOMHTMLElement> element;

		element = do_QueryInterface(node);
		if (element)
		{
			nsAutoString tag;
			element->GetTagName(tag);

			nsCOMPtr<nsIDOMHTMLInputElement> inputelement;

			inputelement = do_QueryInterface(element);
			if (inputelement) {
				nsAutoString type;
				inputelement->GetType(type);
				if (type.EqualsWithConversion 
						("text", PR_TRUE) ||
				    type.EqualsWithConversion 
						("password", PR_TRUE) ||
				    type.EqualsWithConversion 
						("file", PR_TRUE))
				{
					return NS_ERROR_FAILURE;
				}
			}
			else if (!tag.EqualsWithConversion ("a", PR_TRUE) &&
				 !tag.EqualsWithConversion ("html", PR_TRUE))
			{
				return NS_ERROR_FAILURE;
			}

		}
	}

	/* Get the context */

	res = GetEventContext (targetNode, &info->ctx);
	if (NS_FAILED(res)) return res;
	
	/* Get the key/modifier */

	info->modifier = 0;

	keyEvent->GetCharCode((PRUint32*)&info->key);

	if (info->key==0) 
	{
		keyEvent->GetKeyCode((PRUint32*)&info->key);
		info->modifier |= KEY_CODE;
	}

	keyEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	keyEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	keyEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	keyEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult GaleonWrapper::GetDOMAttribute (nsIDOMNode *node, const char *tag,
					 char **attribute)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMNamedNodeMap> attributes;
	result = node->GetAttributes(getter_AddRefs (attributes));
	if (!NS_SUCCEEDED (result) || !attributes) return NS_ERROR_FAILURE;

	nsAutoString attr; 

	attr.AssignWithConversion (tag);
                                        
	nsCOMPtr<nsIDOMNode> attrNode;
	result = attributes->GetNamedItem (attr, getter_AddRefs (attrNode));
	if (!NS_SUCCEEDED (result) || !attrNode) return NS_ERROR_FAILURE;

	nsAutoString nodeValue;
			
	result = attrNode->GetNodeValue (nodeValue);
	if (!NS_SUCCEEDED (result)) return NS_ERROR_FAILURE;
                                                
	char *cstr = nodeValue.ToNewCString();
	*attribute = g_strdup (cstr);
			
	nsMemory::Free (cstr);

	return NS_OK;
}

nsresult GaleonWrapper::GetEventContext (nsIDOMEventTarget *EventTarget,
					 WrapperContextInfo *info)
{
	nsresult rv;

	info->context = CONTEXT_DOCUMENT;

	nsCOMPtr<nsIDOMNode> node = do_QueryInterface(EventTarget, &rv);
	if (NS_FAILED(rv) || !node) return NS_ERROR_FAILURE;

        /* Is page xul ? then do not display context menus
	 * FIXME I guess there is an easier way ... */
	/* From philipl: This test needs to be here otherwise we
	 * arrogantly assume we can QI to a HTMLElement, which is
	 * not true for xul content. */ 

	nsCOMPtr<nsIDOMDocument> domDoc;
	rv = node->GetOwnerDocument(getter_AddRefs(domDoc));
	if (NS_FAILED(rv) || !domDoc) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface(domDoc, &rv);
	if (NS_FAILED(rv) || !doc) return NS_ERROR_FAILURE;

	nsString mime;  
	rv = doc->GetContentType(mime);
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	// Return NS_OK for xul as this is the correct point to exit for
	// the xul case. Nothing had failed.

	if (mime.EqualsWithConversion ("text/xul"))
		return NS_OK;

	// Now we know that the page isn't a xul window, we can try and
	// do something useful with it.

	PRUint16 type;
	rv = node->GetNodeType(&type);
	if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

	if (nsIDOMNode::ELEMENT_NODE == type)
	{
		nsCOMPtr<nsIDOMHTMLElement> element = do_QueryInterface(node,
									&rv);
		if (NS_FAILED(rv) || !element) return NS_ERROR_FAILURE;

		nsAutoString tag;
		rv = element->GetTagName(tag);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		if (tag.EqualsWithConversion("img", PR_TRUE))
		{
			info->context |= CONTEXT_IMAGE;

			nsAutoString img;
			nsCOMPtr <nsIDOMHTMLImageElement> image = 
						do_QueryInterface(node, &rv);
			if (NS_FAILED(rv) || !image) return NS_ERROR_FAILURE;			

			rv = image->GetSrc (img);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

			info->img = img.ToNewCString();
		}
		else if (tag.EqualsWithConversion("input", PR_TRUE))
		{
			info->context |= CONTEXT_INPUT;
		}
	}


	/* Is page framed ? */
	PRBool framed;
	IsPageFramed (node, &framed);
	info->framed_page = (gboolean)framed;

	/* Bubble out, looking for items of interest */
	while (node)
	{
		PRUint16 type;
		rv = node->GetNodeType(&type);
		if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

		if (nsIDOMNode::ELEMENT_NODE == type)
		{
			nsCOMPtr<nsIDOMHTMLElement> element = 
					do_QueryInterface(node, &rv);
			if (NS_FAILED(rv) || !element) return NS_ERROR_FAILURE;

			nsAutoString tag;
			rv = element->GetTagName(tag);
			if (NS_FAILED(rv)) return NS_ERROR_FAILURE;

			/* Link */
			if (tag.EqualsWithConversion("a", PR_TRUE))
			{
				PRBool isHref;
				rv = element->HasAttribute (NS_LITERAL_STRING("href"),
							    &isHref);
				if (NS_FAILED(rv))
					return NS_ERROR_FAILURE;
				if (isHref == PR_FALSE)
					return NS_OK;

				info->context |= CONTEXT_LINK;

				nsCOMPtr <nsIDOMHTMLAnchorElement> anchor =
					do_QueryInterface(node, &rv);
				if (NS_SUCCEEDED(rv) && anchor) 
				{
					nsAutoString href;
					rv = anchor->GetHref (href);
					if (NS_FAILED(rv))
						return NS_ERROR_FAILURE;
					
					info->link = href.ToNewCString();
					
					/* Get the text of the link */
					nsCOMPtr<nsIDOMNSHTMLElement> nsElement;
					nsElement = do_QueryInterface (element,
								       &rv);
					if (NS_SUCCEEDED(rv) && nsElement)
					{
						nsAutoString linkhtml;
						rv = nsElement->GetInnerHTML
								(linkhtml);
						if (NS_SUCCEEDED(rv))
						{
							info->linktext =
								mozilla_unicode_to_locale
							(linkhtml.get());
						}
					}
				}
			}
			if (tag.EqualsWithConversion("area", PR_TRUE))
			{
				info->context |= CONTEXT_LINK;
				nsCOMPtr <nsIDOMHTMLAreaElement> area =
						do_QueryInterface(node, &rv);
				if (NS_SUCCEEDED(rv) && area)
				{
					nsAutoString href;
					rv = area->GetHref (href);
					if (NS_FAILED(rv))
						return NS_ERROR_FAILURE;
					
					info->link = href.ToNewCString();
				}
			}
		}
		
		nsCOMPtr<nsIDOMNode> parentNode;
		node->GetParentNode (getter_AddRefs(parentNode));
		node = parentNode;
	}
	
	return NS_OK;
}

nsresult GaleonWrapper::GetMouseEventContext (nsIDOMMouseEvent *event, WrapperMouseEventInfo *info)
{
	nsresult result;
	DOMTimeStamp ts;
	nsIDOMMouseEvent *aMouseEvent = (nsIDOMMouseEvent*)event;

	aMouseEvent->GetButton ((PRUint16*)&info->button);

	aMouseEvent->GetTimeStamp(&ts);
	info->timestamp = ts;

	/* be sure we are not clicking on the scroolbars */
	/* FIXME this need to be moved in GetEventContext and cleaned */

	nsCOMPtr<nsIDOMEventTarget> OriginalTarget;
	result = aMouseEvent->GetOriginalTarget(getter_AddRefs(OriginalTarget));
	if (NS_FAILED(result) || !OriginalTarget) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMNode> OriginalNode = do_QueryInterface(OriginalTarget);
	if (!OriginalNode) return NS_ERROR_FAILURE;

	nsString nodename;
	OriginalNode->GetNodeName(nodename);

	if (nodename.EqualsWithConversion ("xul:thumb") ||
	    nodename.EqualsWithConversion ("xul:slider"))
		return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDOMEventTarget> EventTarget;
	result = aMouseEvent->GetTarget(getter_AddRefs(EventTarget));
	if (NS_FAILED(result) || !EventTarget) return NS_ERROR_FAILURE;

	result = GetEventContext (EventTarget, &info->ctx);
	if (NS_FAILED(result)) return result;

	/* Get the modifier */

	PRBool mod_key;

	info->modifier = 0;

	aMouseEvent->GetAltKey(&mod_key);
	if (mod_key) info->modifier |= ALT_KEY;

	aMouseEvent->GetShiftKey(&mod_key);
	if (mod_key) info->modifier |= SHIFT_KEY;

	aMouseEvent->GetMetaKey(&mod_key);
	if (mod_key) info->modifier |= META_KEY;
	
	aMouseEvent->GetCtrlKey(&mod_key);
	if (mod_key) info->modifier |= CTRL_KEY;

	return NS_OK;
}

nsresult  GaleonWrapper::CopyHistoryTo (GaleonWrapper *dest)
{
	nsresult result;
	int count,index;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_src = do_QueryInterface (DocShell,
							       &result);
	if (!wn_src) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_src;
	result = wn_src->GetSessionHistory (getter_AddRefs (h_src));
	if (!NS_SUCCEEDED(result) || (!h_src)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocShell> destDocShell;
	result = dest->GetDocShell (getter_AddRefs(destDocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIWebNavigation> wn_dest = do_QueryInterface (destDocShell,
								&result);
	if (!wn_dest) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsISHistory> h_dest;
	result = wn_dest->GetSessionHistory (getter_AddRefs (h_dest));
	if (!NS_SUCCEEDED (result) || (!h_dest)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsISHistoryInternal> hi_dest = do_QueryInterface (h_dest);
	if (!hi_dest) return NS_ERROR_FAILURE;

	h_src->GetCount (&count);
	h_src->GetIndex (&index);

	if (count) {
		nsCOMPtr<nsIHistoryEntry> he;
		nsCOMPtr<nsISHEntry> she;

		for (PRInt32 i = 0; i < count; i++) {

			result = h_src->GetEntryAtIndex (i, PR_FALSE,
							 getter_AddRefs (he));
			if (!NS_SUCCEEDED(result) || (!he))
				return NS_ERROR_FAILURE;

			she = do_QueryInterface (he);
			if (!she) return NS_ERROR_FAILURE;

			result = hi_dest->AddEntry (she, PR_TRUE);
			if (!NS_SUCCEEDED(result) || (!she))
				return NS_ERROR_FAILURE;
		}

		result = wn_dest->GotoIndex(index);
		if (!NS_SUCCEEDED(result)) return NS_ERROR_FAILURE;
	}

	return NS_OK;
}


nsresult GaleonWrapper::SetSitePermission (gboolean block, PRInt32 type)
{
	nsresult result;

	nsCOMPtr<nsIDOMDocument> DOMDocument;
	result = GetDOMDocument (getter_AddRefs(DOMDocument));
	if (NS_FAILED(result) || !DOMDocument) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIDocument> doc = do_QueryInterface (DOMDocument);
	if (!doc) return NS_ERROR_FAILURE;
	
#if MOZILLA_VERSION > VERSION3(0,9,1)
	nsCOMPtr<nsIURI> uri;
	doc->GetDocumentURL(getter_AddRefs(uri));
#else
	nsCOMPtr<nsIURI> uri = getter_AddRefs(doc->GetDocumentURL());
#endif
	char *url;
	uri->GetSpec (&url);
	
	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance (PERMISSIONMANAGER_ID);
	
	result = permissionManager->Add (url,block ? PR_TRUE : PR_FALSE, type);

	return result;
}

nsresult GaleonWrapper::ForceCharacterSet (char *charset) 
{
	nsresult result;

	nsCOMPtr<nsIDocShell> DocShell;
	result = GetDocShell (getter_AddRefs(DocShell));
	if (NS_FAILED(result) || !DocShell) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIContentViewer> contentViewer;	
	result = DocShell->GetContentViewer (getter_AddRefs(contentViewer));
	if (!NS_SUCCEEDED (result) || !contentViewer) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIMarkupDocumentViewer> mdv = do_QueryInterface(contentViewer,
								  &result);
	if (NS_FAILED(result) || !mdv) return NS_ERROR_FAILURE;

	nsAutoString charset_str;
	charset_str.AssignWithConversion (charset);
	result = mdv->SetForceCharacterSet (charset_str.ToNewUnicode());

	return result;
}

nsresult GaleonWrapper::IsPageFramed (nsIDOMNode *node, PRBool *Framed)
{
	nsresult result;
	
	nsCOMPtr<nsIDOMDocument> mainDocument;
	result = GetMainDOMDocument (getter_AddRefs(mainDocument));
	if (NS_FAILED(result) || !mainDocument) return NS_ERROR_FAILURE;
	
	nsCOMPtr<nsIDOMDocument> nodeDocument;
	result = node->GetOwnerDocument (getter_AddRefs(nodeDocument));
	if (NS_FAILED(result) || !nodeDocument) return NS_ERROR_FAILURE;
 
	*Framed = (mainDocument != nodeDocument);

        return NS_OK;
}

nsresult GaleonWrapper::CanCutSelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCutSelection (result);
}

nsresult GaleonWrapper::CanCopySelection(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanCopySelection (result);
}

nsresult GaleonWrapper::CanPaste(PRBool *result)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CanPaste (result);
}

nsresult GaleonWrapper::CutSelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CutSelection ();
}

nsresult GaleonWrapper::CopySelection(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->CopySelection ();
}

nsresult GaleonWrapper::Paste(void)
{
	nsCOMPtr<nsIClipboardCommands> clipboard (do_GetInterface(mWebBrowser));
	return clipboard->Paste ();
}

nsresult GaleonWrapper::GetFaviconURL (char **favicon_url)
{
	nsresult result;
	PRUint32 links_count;
	char *relstr;

	nsCOMPtr<nsIDOMDocument> aDOMDocument;
	result = GetMainDOMDocument (getter_AddRefs(aDOMDocument));
	if (NS_FAILED(result) || !aDOMDocument) return NS_ERROR_FAILURE;

	nsString strname;
	strname.AssignWithConversion("LINK");

	nsCOMPtr<nsIDOMNodeList> aLinks;
	result = aDOMDocument->GetElementsByTagName (strname, 
						     getter_AddRefs (aLinks));
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	result = aLinks->GetLength (&links_count);
	if (NS_FAILED (result)) return NS_ERROR_FAILURE;

	for (PRUint32 i = 0; i < links_count; i++)
	{
		nsCOMPtr<nsIDOMNode> aLink;
		result = aLinks->Item (i, getter_AddRefs (aLink));
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;
		
		result = GetDOMAttribute (aLink, "rel", &relstr);
		if (NS_FAILED (result)) return NS_ERROR_FAILURE;
		
		if (relstr == NULL) continue;

		if (strcmp(relstr, "SHORTCUT ICON") == 0)
		{
			char *link = NULL;
			
			g_free (relstr);
			
			result = GetDOMAttribute (aLink, "href", &link);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			if (link == NULL)
			{
				*favicon_url = NULL;
				return NS_OK;
			}
			
			nsCOMPtr<nsIDocument> doc = 
				do_QueryInterface (aDOMDocument);
			if(!doc) return NS_ERROR_FAILURE;
			
#if MOZILLA_VERSION > VERSION3(0,9,1)
			nsCOMPtr<nsIURI> uri;
			doc->GetDocumentURL(getter_AddRefs(uri));
#else
			nsCOMPtr<nsIURI> uri = getter_AddRefs
						(doc->GetDocumentURL());
#endif
			result = uri->Resolve (link, favicon_url);
			if (NS_FAILED (result)) return NS_ERROR_FAILURE;
			
			g_free (link);
			return NS_OK;
		}
	}

	return NS_ERROR_FAILURE;
}

GaleonEventListener::GaleonEventListener(void)
{
	NS_INIT_REFCNT();
	wrapper = NULL;
}

GaleonEventListener::~GaleonEventListener(void)
{
}


NS_IMPL_ADDREF(GaleonEventListener)
NS_IMPL_RELEASE(GaleonEventListener)
NS_INTERFACE_MAP_BEGIN(GaleonEventListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIDOMEventListener, nsIDOMDragListener)
  NS_INTERFACE_MAP_ENTRY(nsIDOMDragListener)
NS_INTERFACE_MAP_END

void
GaleonEventListener::Init(GaleonWrapper *galeon_wrapper)
{
	wrapper = galeon_wrapper;
}

NS_IMETHODIMP
GaleonEventListener::HandleEvent(nsIDOMEvent* aEvent)
{
	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragEnter(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragExit(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragDrop(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
NS_IMETHODIMP GaleonEventListener::DragGesture(nsIDOMEvent* aMouseEvent)
{
	gchar *old_dragging_link;
	WrapperMouseEventInfo *info;
	gchar *url = NULL;

	info = g_new0 (WrapperMouseEventInfo,1);

	mozilla_get_mouse_event_info(wrapper->embed, aMouseEvent, info);


	if (info->ctx.context & CONTEXT_LINK)
	{
		url = info->ctx.link;
	}
	else if (info->ctx.context & CONTEXT_IMAGE)
	{	
		url = info->ctx.img;
	}
	
	if (url)
	{
		if (!link_drag_types_tl)
			link_drag_types_tl = gtk_target_list_new 
				(link_drag_types, link_drag_types_num_items);

		old_dragging_link = (gchar*)gtk_object_get_data (
					GTK_OBJECT (wrapper->mGtkMozEmbed),
					"dragging_link");
		if (old_dragging_link) g_free (old_dragging_link);
		gtk_object_set_data (GTK_OBJECT(wrapper->mGtkMozEmbed),
				     "dragging_link", g_strdup (url));


		GdkEvent gdk_event;
		gdk_event.type = GDK_BUTTON_PRESS;
		gdk_event.button.window = NULL;
		gdk_event.button.send_event = 0;
		gdk_event.button.time = time (NULL);
		gdk_event.button.x = 0;
		gdk_event.button.y = 0;
		gdk_event.button.pressure = 0;
		gdk_event.button.xtilt = 0;
		gdk_event.button.ytilt = 0;
		gdk_event.button.state = 0;
		gdk_event.button.button = 0;
		gdk_event.button.source = (GdkInputSource)0;
		gdk_event.button.deviceid = 0;
		gdk_event.button.x_root = 0;
		gdk_event.button.y_root = 0;

		gtk_drag_begin (GTK_WIDGET(wrapper->mGtkMozEmbed),
				link_drag_types_tl, 
				(GdkDragAction)(GDK_ACTION_COPY | 
						GDK_ACTION_LINK),
				1, &gdk_event);
	}

	mozilla_free_context_info_sub (&info->ctx);
	g_free (info);

	return NS_OK;
}

NS_IMETHODIMP GaleonEventListener::DragOver(nsIDOMEvent* aMouseEvent)
{
	return NS_OK;
}
