/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "bookmarks.h"
#include "bookmarks_editor_callbacks.h"
#include "misc.h"
#include "glade.h"
#include "window.h"
#include "state.h"
#include "prefs.h"
#include "context.h"
#include "dialog.h"

#include <string.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkdnd.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-pixmap-entry.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <gdk/gdkkeysyms.h>

extern GtkTargetEntry bookmarks_dnd_targets [];

/** The bookmarks editing controls list */
GList *bookmarks_editors = NULL;

/* Function prototypes */
static BookmarksEditorControls *bookmarks_editor_controls_init
                               (BookmarkItem *root,
				BookmarksEditorControlsType type);
static void bookmarks_editor_remove_tree_item_null
                               (BookmarkItem *b,
				BookmarksEditorControls *controls);
static void bookmarks_editor_add_menubar (BookmarksEditorControls *controls);
static void
bookmarks_editor_fill_menu (GtkMenuShell *menushell, 
			    GtkAccelGroup *accel, 
			    BookmarksEditorControls *controls,
			    GtkWidget **edit_menuitem,
			    GtkWidget **set_default_folder_menuitem);


static BookmarksEditorControls *
bookmarks_editor_controls_init (BookmarkItem *root, 
				BookmarksEditorControlsType type)
{
	BookmarksEditorControls *editor;
	GtkWidget *close, *detach, *show_menu;
	GladeXML *gxml = NULL;
	GList *toolbar_items, *tlist;
		
	editor = g_new0 (BookmarksEditorControls, 1);

	editor->type = type;
	editor->root_bookmark = root;
	editor->clipboard = NULL;

	switch (type)
	{
	case STANDARD_BOOKMARKS_EDITOR:
		gxml = glade_widget_new ("bookmarks.glade", "bookmarks_editor", 
					 &(editor->dialog), editor);
		editor->menubar = glade_xml_get_widget 
			(gxml, "bookmarks_editor_menubar");
		editor->toolbar = glade_xml_get_widget 
			(gxml, "bookmarks_editor_toolbar");
		editor->menubar_dock = glade_xml_get_widget 
			(gxml, "bookmarks_editor_menubar_dock");
		editor->toolbar_dock = glade_xml_get_widget 
			(gxml, "bookmarks_editor_toolbar_dock");
		editor->ctree = glade_xml_get_widget
			(gxml, "bookmarks_editor_ctree");
		editor->vpane = NULL;
		editor->edit_frame = glade_xml_get_widget
			(gxml, "bookmarks_editor_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_url_entry");
		editor->nick_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_nick_entry");
		editor->pixmap_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_entry");
		editor->pixmap_file_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_file_entry");
		editor->create_toolbar_toggle = glade_xml_get_widget
			(gxml, "bookmarks_editor_create_toolbar_toggle");
		editor->create_context_menu_toggle = glade_xml_get_widget
			(gxml, "bookmarks_editor_create_context_menu_toggle");
		editor->notes_text = glade_xml_get_widget
			(gxml, "bookmarks_editor_notes_text");
		editor->time_added_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_added_entry");
		editor->time_modified_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_modified_entry");
		editor->time_visited_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_time_visited_entry");
		bookmarks_editor_add_menubar (editor);
		g_assert (editor->dialog != NULL);
		g_assert (editor->toolbar != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry != NULL);
		g_assert (editor->pixmap_entry != NULL);
		g_assert (editor->pixmap_file_entry != NULL);
		g_assert (editor->create_toolbar_toggle != NULL);
		g_assert (editor->create_context_menu_toggle != NULL);
		g_assert (editor->notes_text != NULL);
		g_assert (editor->time_added_entry != NULL);
		g_assert (editor->time_modified_entry != NULL);
		g_assert (editor->time_visited_entry != NULL);

		break;
	case DOCKED_BOOKMARKS_EDITOR:
		gxml = glade_widget_new ("bookmarks.glade", "bookmarks_dock", 
					 &(editor->dialog), editor);
		editor->toolbar = glade_xml_get_widget 
			(gxml, "bookmarks_dock_toolbar");
		editor->ctree = glade_xml_get_widget
			(gxml, "bookmarks_dock_ctree");
		editor->vpane = NULL;
		editor->edit_frame = glade_xml_get_widget
			(gxml, "bookmarks_dock_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_url_entry");
		editor->nick_entry = glade_xml_get_widget
			(gxml, "bookmarks_dock_nick_entry");
		editor->notes_text = glade_xml_get_widget
			(gxml, "bookmarks_dock_notes_text");
		editor->pixmap_entry = NULL;
		editor->pixmap_file_entry = NULL;
		editor->create_toolbar_toggle = NULL;
		editor->create_context_menu_toggle = NULL;
		editor->toolbar_dock = NULL;
		editor->menubar_dock = NULL;
		editor->time_added_entry = NULL;
		editor->time_modified_entry = NULL;
		editor->time_visited_entry = NULL;

		g_assert (editor->dialog != NULL);
		g_assert (editor->toolbar != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry != NULL);
		g_assert (editor->notes_text != NULL);

		/* fix up broken button relief (bug in libglade) */
		close = glade_xml_get_widget 
			(gxml, "bookmarks_dock_close_button");
		detach = glade_xml_get_widget 
			(gxml, "bookmarks_dock_detach_button");
		show_menu = glade_xml_get_widget 
			(gxml, "bookmarks_dock_menu_button");
		gtk_button_set_relief (GTK_BUTTON (close),  GTK_RELIEF_NONE);
		gtk_button_set_relief (GTK_BUTTON (detach), GTK_RELIEF_NONE);
		gtk_button_set_relief (GTK_BUTTON (show_menu),
				       GTK_RELIEF_NONE);
		break;
	default:
		g_error ("Incorrect bookmarks editor dialog type");
	}
	g_assert (editor->ctree != NULL);
	
	editor->selection = NULL;
	editor->last_pressed = NULL;
	editor->lock_edit_controls = FALSE;
	editor->window = NULL;

	/* get the widgets we need to work with later*/
	editor->edit_button = glade_xml_get_widget
			      (gxml, "bookmarks_editor_edit_button");

	/* unref the GladeXML object, we don't use it anymore */
	gtk_object_unref (GTK_OBJECT (gxml));

	gtk_drag_source_set (editor->ctree, GDK_BUTTON1_MASK, 
			     bookmarks_dnd_targets, 
			     bookmarks_dnd_targets_num_items, 
			     GDK_ACTION_COPY | GDK_ACTION_MOVE |
			     GDK_ACTION_LINK);
	gtk_drag_dest_set (editor->ctree,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);

	gtk_selection_add_target (GTK_WIDGET (editor->ctree),
				  GDK_SELECTION_PRIMARY,
				  GDK_SELECTION_TYPE_STRING, /* FIXME, use a correct type*/
				  1);
	/* copy & paste callbacks */
	gtk_signal_connect 
		(GTK_OBJECT (editor->ctree), "selection_get",
		 GTK_SIGNAL_FUNC (bookmarks_editor_selection_get_cb), editor);
	gtk_signal_connect 
		(GTK_OBJECT (editor->ctree), "selection_received",
		 GTK_SIGNAL_FUNC (bookmarks_editor_selection_received_cb),
		 NULL);
	/* need this because selection_received does not take any user data */
	gtk_object_set_data (GTK_OBJECT (editor->ctree), "controls", editor);

	if (editor->pixmap_entry != NULL)
	{
		gnome_pixmap_entry_set_pixmap_subdir 
			(GNOME_PIXMAP_ENTRY (editor->pixmap_entry), SHARE_DIR);
	}

	/* setup the context menu callback for the toolbar */
	toolbar_items = gtk_container_children 
		(GTK_CONTAINER (editor->toolbar));
	tlist = toolbar_items;
	while (tlist != NULL)
	{
		gtk_signal_connect 
			(GTK_OBJECT (tlist->data), "button-press-event",
			GTK_SIGNAL_FUNC 
			 (bookmarks_editor_toolbar_button_press_event_cb),
			editor);
		tlist = tlist->next;
	}
	g_list_free (toolbar_items);

	/* callback to make sure that everything is cleaned before destroying 
	   the widgets */
	gtk_signal_connect (GTK_OBJECT (editor->ctree), "destroy",
			    bookmarks_editor_destroy_cb, editor);

	bookmarks_editor_place_tree_item (editor, root);
	return editor;
}

BookmarksEditorControls *
bookmarks_editor_show_dialog (GaleonWindow *window)
{
	gboolean toolbar_relief;
	gboolean menubar_relief;
	gboolean toolbar_detachable;
	gboolean menubar_detachable;
	BookmarksEditorControls *controls = NULL;

	/* read preferences */
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();
	menubar_detachable = gnome_preferences_get_menubar_detachable ();
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	menubar_relief = gnome_preferences_get_menubar_relief ();

	controls = bookmarks_editor_controls_init
		(bookmarks_root, STANDARD_BOOKMARKS_EDITOR);
	bookmarks_editors = g_list_prepend (bookmarks_editors, controls);
	
	/* 630 is chosen since thats wide enough for the toolbar in
	 * with all the standard GNOME defaults */
	state_load_window_state (controls->dialog, "bookmarks_editor");
	state_load_column_widths (controls->ctree, "bookmarks_editor");
	state_load_pane_pos (controls->vpane,
			     "bookmarks_editor_vpane=-1");
	
	g_return_val_if_fail (controls != NULL, NULL);
	
	/* setup toolbar*/
	gtk_toolbar_set_style 
		(GTK_TOOLBAR (controls->toolbar),
		eel_gconf_get_integer
			(CONF_TOOLBAR_BOOKMARKS_EDITOR_STYLE));
	gnome_dock_item_set_shadow_type
		(GNOME_DOCK_ITEM (controls->toolbar_dock), 
		 toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	(GNOME_DOCK_ITEM (controls->toolbar_dock))->behavior &=
		~(toolbar_detachable ? GNOME_DOCK_ITEM_BEH_LOCKED : 0);
	if (toolbar_relief)
	{
		gtk_container_set_border_width 
			(GTK_CONTAINER (controls->toolbar_dock), 2);
	}
	
	(GNOME_DOCK_ITEM (controls->menubar_dock))->behavior &=
		~(menubar_detachable ? GNOME_DOCK_ITEM_BEH_LOCKED : 0);
	gnome_dock_item_set_shadow_type
		(GNOME_DOCK_ITEM (controls->menubar_dock),
		 menubar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
		 
	gtk_widget_show (controls->dialog);

	if (window != NULL)
	{
		controls->window = window;
		dialog_set_parent (controls->dialog, window->WMain);
	}
	return controls;
}

void
bookmarks_editor_hide_dialog (BookmarksEditorControls *controls)
{
	g_return_if_fail (controls != NULL);

	/* the destroy callback of the ctree calls this function.
	   Disconnect that to acoid recursion */
	gtk_signal_disconnect_by_func (GTK_OBJECT (controls->ctree),
				       bookmarks_editor_destroy_cb,
				       controls);

	/* must sync because the editor really does change bookmarks
	 * in memory, only way out is to use revert */
	bookmarks_updated ();

	if (controls->type != DOCKED_BOOKMARKS_EDITOR) 
	{
		// FIXME
		// I think that we should not save window state 
		// at all... - ricardo
		if (controls->root_bookmark == bookmarks_root)
		{
			state_save_window_state (controls->dialog->window,
						 "bookmarks_editor");
			state_save_column_widths (controls->ctree, 
						  "bookmarks_editor");
			state_save_pane_pos (controls->vpane,
					     "bookmarks_editor_vpane");
		}
		else if (controls->root_bookmark == default_bookmarks_root)
		{
			state_save_window_state (controls->dialog->window,
						 "compact_bookmarks_editor");
		}
	}
	
	gtk_clist_freeze (GTK_CLIST(controls->ctree));
	bookmarks_editor_remove_tree_item (controls, 
					   controls->root_bookmark);
	gtk_clist_thaw (GTK_CLIST(controls->ctree));
	bookmarks_editors = g_list_remove (bookmarks_editors, controls);

	if (controls->clipboard) g_free (controls->clipboard);

	if (controls->type == DOCKED_BOOKMARKS_EDITOR) 
	{
		window_undock (controls->window);
	} 
	else 
	{
		gtk_widget_destroy (controls->dialog);
	}

	/* scrub control structure and free */
	memset (controls, 0, sizeof (BookmarksEditorControls));
	g_free (controls);
}

BookmarksEditorControls *
bookmarks_editor_show_dock (GaleonWindow *window)
{
	BookmarksEditorControls *controls;

	controls = bookmarks_editor_controls_init (bookmarks_root, 
						   DOCKED_BOOKMARKS_EDITOR);
	/* hide the URL column */
	gtk_clist_set_column_visibility (GTK_CLIST (controls->ctree), 1,
					 FALSE);
	bookmarks_editors = g_list_append (bookmarks_editors, controls);
	controls->window = window;
	
	gtk_toolbar_set_style 
		(GTK_TOOLBAR (controls->toolbar), 
		eel_gconf_get_integer 
		       (CONF_TOOLBAR_BOOKMARKS_EDITOR_DOCKED_STYLE));

	window_dock (window, controls->dialog, 
		     CONF_STATE_BOOKMARKS_DOCK_WIDTH);

	/* set dock type */
	window->dock_type = DOCK_BOOKMARKS;
	
	/* update toolbar buttons */
	window_update_bm_and_hist_buttons (window, TRUE, FALSE);

	return controls;
}
				
/**
 * bookmarks_editor_place_tree_item: place a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Create the tree items for a bookmark and its children. If it has already
 * been placed, removes the old item.
 * 
 **/
void
bookmarks_editor_place_tree_item (BookmarksEditorControls *controls,
				  BookmarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	const PixmapData *bm_icon;
	GdkPixmap *pixmap;
	GdkPixmap *pixmap_open;
	GdkPixmap *mask_open;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
	GtkCTreeNode *sibling_node;

	text[2] = NULL;
	switch (b->type) {
	case BM_SEPARATOR:
		text[0] = g_strdup ("------");
		text[1] = g_strdup ("------");
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	case BM_SITE:
		/* b->name is nicely escaped, but we need to unescape
		 * it here, as GtkCLists and GtkEntries don't know about 
		 * accels */
		text[0] = strip_uline_accel (b->name);
		text[1] = g_strdup (b->url);
		isleaf = TRUE;

		bm_icon = bookmarks_get_siteicon (b->url);
		pixmap = bm_icon->pixmap;
		mask = bm_icon->mask;

		pixmap_open = pixmap;
		mask_open = mask;
		break;
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		/* b->name is nicely escaped, but we need to unescape
		 * it here, as GtkCLists and GtkEntries don't know about 
		 * accels */
		text[0] = strip_uline_accel (b->name);
		text[1] = g_strdup ("");
		isleaf = FALSE;
		if (b == default_bookmarks_root) 
		{
			pixmap = default_folder_pixmap_data->pixmap; 
			mask = default_folder_pixmap_data->mask;
			pixmap_open = default_folder_open_pixmap_data->pixmap;
			mask_open = default_folder_open_pixmap_data->mask;
		} else {
			pixmap = folder_pixmap_data->pixmap; 
			mask = folder_pixmap_data->mask;
			pixmap_open = folder_open_pixmap_data->pixmap;
			mask_open = folder_open_pixmap_data->mask;
		}
		break;
	default:
		text[0] = g_strdup("");
		text[1] =  g_strdup("");
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = bookmarks_get_tree_item (b->parent, controls);

	sibling_node = NULL;
	if (b->parent != NULL) {
		GList *list_node = g_list_find (b->parent->list, b);
		if (list_node) {
			GList *sibling_list_node = g_list_next (list_node);
			if (sibling_list_node) {
				sibling_node = bookmarks_get_tree_item 
					(((BookmarkItem *) 
					  (sibling_list_node->data)),
					 controls);
			}
		}
	}
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	bookmarks_editor_remove_tree_item (controls, b);
	if (b != controls->root_bookmark) {
		if (b->alias_of) {
			/* TODO: use a diferent pixmap for aliases 
			   for now, just leave it w/o pixmap */
			pixmap = mask = pixmap_open = mask_open = NULL;
		}
		bookmarks_set_tree_item 
			(b, controls, gtk_ctree_insert_node
			 (GTK_CTREE (controls->ctree), parent_node,
			  sibling_node, text, 2, pixmap, mask, pixmap_open,
			  mask_open, isleaf, b->expanded));
		gtk_ctree_node_set_row_data (GTK_CTREE (controls->ctree), 
					     bookmarks_get_tree_item 
					     (b,
					      controls), b);
	}

	if (!isleaf && !b->alias_of) /* don't display alias' children */ {
		GList *l;
		for (l = b->list; l != NULL; l = g_list_next (l))
			bookmarks_editor_place_tree_item (controls, l->data);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
	g_free (text[0]);
	g_free (text[1]);
}

void
bookmarks_editor_place_tree_items (BookmarkItem *b)
{
	GList *l;
	for (l = bookmarks_editors; l; l = l->next) {
		BookmarksEditorControls *controls = l->data;
		if (bookmarks_is_ancestor (controls->root_bookmark, b))
			bookmarks_editor_place_tree_item (controls, b);
	}
}

void
bookmarks_editor_remove_tree_items (BookmarkItem *b)
{
	GList *li;

	for (li = bookmarks_editors; li; li = li->next) 
	{
		BookmarksEditorControls *controls = li->data;
		if (controls->root_bookmark == b) {
			if (b != bookmarks_root) {
				bookmarks_editor_set_root 
					(controls, bookmarks_root);
			} else {
				controls->root_bookmark = NULL;
			}
		}
	}

	while (b->tree_items) 
		bookmarks_editor_remove_tree_item 
			(((BookmarkTreeItem *) 
			  (b->tree_items->data))->controls, b);
}

void
bookmarks_editor_update_tree_items (BookmarkItem *b)
{
	GList *l;
	for (l = b->tree_items; l; l = l->next) 
		bookmarks_editor_update_tree_item (((BookmarkTreeItem *) 
						    (l->data))->controls, b);
}


/**
 * bookmarks_editor_remove_tree_item: remove a bookmark from the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Removes bookmarks and it's childs from the bookmark editor ctree and sets
 * the tree_item fields to NULL. This does not delete the bookmarks
 * 
 **/
void
bookmarks_editor_remove_tree_item (BookmarksEditorControls *controls, BookmarkItem *b)
{
	if (b == controls->root_bookmark) {
		GList *l;
		for (l = b->list; l; l = l->next) 
			bookmarks_editor_remove_tree_item (controls, l->data);
	} else {
		GtkCTreeNode *node = bookmarks_get_tree_item (b, controls);
		if (node) {
			/* this removes the treeitem and its children */
			gtk_ctree_remove_node (GTK_CTREE (controls->ctree),
					       GTK_CTREE_NODE (node));
			/* now, let's NULL the fields */
			bookmarks_editor_remove_tree_item_null (b, controls);
		}
	}
}

/**
 * bookmarks_editor_remove_tree_item_null: Sets to NULL the tree_item field of
 * a bookmarks and its children
 * @b: the bookmark
 * 
 * This is an auxiliar function for bookmarks_editor_remove_tree_item.
 **/
static void
bookmarks_editor_remove_tree_item_null (BookmarkItem *b, BookmarksEditorControls *controls)
{
	GList *l;
	bookmarks_set_tree_item (b, controls, NULL);
	if ((b->type == BM_FOLDER || b->type == BM_AUTOBOOKMARKS) 
	    && ! b->alias_of) /* alias don't have children */
	{
		for (l = b->list; l != NULL; l = g_list_next (l)) 
			bookmarks_editor_remove_tree_item_null
				(l->data, controls);
	}
}

/**
 * bookmarks_editor_update_tree_item: update a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Updates the bookmark (or adds it if not yet added)
 * 
 **/
void
bookmarks_editor_update_tree_item (BookmarksEditorControls *controls, BookmarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	const PixmapData *bm_icon;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
		
	if (!bookmarks_get_tree_item (b, controls)) {
		bookmarks_editor_place_tree_item (controls, b);
		return;
	}

	text[2] = NULL;
	switch (b->type) {
	case BM_SEPARATOR:
		text[0] = g_strdup ("------"); 
		text[1] = g_strdup ("------");
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	case BM_SITE:
		/* b->name is nicely escaped, but we need to unescape
		 * it here, as GtkCLists and GtkEntries don't know about 
		 * accels */
		text[0] = strip_uline_accel (b->name);
		text[1] = g_strdup (b->url);
		isleaf = TRUE;
		bm_icon = bookmarks_get_siteicon (b->url);
		pixmap = bm_icon->pixmap;
		mask = bm_icon->mask;;
		break;
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		/* b->name is nicely escaped, but we need to unescape
		 * it here, as GtkCLists and GtkEntries don't know about 
		 * accels */
		text[0] = strip_uline_accel (b->name);
		text[1] = g_strdup ("");
		isleaf = FALSE;
		if (b == default_bookmarks_root)
		{
			pixmap = default_folder_pixmap_data->pixmap; 
			mask = default_folder_pixmap_data->mask;
		} else 
		{
			pixmap = folder_pixmap_data->pixmap; 
			mask = folder_pixmap_data->mask;
		}
		break;
	default:
		text[0] = g_strdup (""); 
		text[1] = g_strdup ("");
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = bookmarks_get_tree_item (b->parent, controls);

	if (b != controls->root_bookmark) {
		if (b->alias_of) {
			/* TODO: use a diferent pixmap for aliases 
			   for now, just leave it w/o pixmap */
			pixmap = mask = NULL;
		}
		gtk_ctree_node_set_pixtext
			(GTK_CTREE (controls->ctree), bookmarks_get_tree_item (b, controls), 0, text[0],
			2, pixmap, mask);
		gtk_ctree_node_set_text
			(GTK_CTREE (controls->ctree), bookmarks_get_tree_item (b, controls), 1, text[1]);
	}
	g_free (text[0]);
	g_free (text[1]);
}

/**
 * bookmarks_editor_select_bookmark : position the bookmark editor 
 * 	at a given bookmark item & focus the name entry
 * @b : bookmark item to focus
 */
void bookmarks_editor_select_bookmark (BookmarksEditorControls *controls,
				       BookmarkItem *bi)
{
	GtkCTreeNode *node, *parent;
	gint row;

	/* freeze the view */
	gtk_clist_freeze (GTK_CLIST (controls->ctree));

	/* make sure this node is visible in the bookmarks editor */
	if (bi->parent == NULL)
	{
		bookmarks_editor_set_root (controls, bi);
		/* thaw the view */
		gtk_clist_thaw (GTK_CLIST (controls->ctree));
		/* return now, we can't actually focus the item since it
		 * is the root, and the root isn't shown */
		return;
	}

	/* make sure the selection is shown */
	node = bookmarks_get_tree_item (bi, controls);
	parent = GTK_CTREE_ROW (node)->parent;
	while (parent != NULL)
	{
		gtk_ctree_expand (GTK_CTREE (controls->ctree),
				  GTK_CTREE_NODE (parent));
		parent = GTK_CTREE_ROW (parent)->parent;
	}

	/* select it */
	node = bookmarks_get_tree_item (bi, controls);
	gtk_ctree_node_moveto (GTK_CTREE (controls->ctree), 
			       GTK_CTREE_NODE (node), 0, 0, 0);
	gtk_clist_unselect_all (GTK_CLIST (controls->ctree));
	gtk_ctree_select (GTK_CTREE (controls->ctree), GTK_CTREE_NODE (node));

	/* find the position (FIXME: necessary?) */
	row = g_list_position (GTK_CLIST (controls->ctree)->row_list, 
			       (GList *)node);
	GTK_CLIST (controls->ctree)->focus_row = row;

	/* put focus into the name entry */
	gtk_widget_grab_focus (GTK_WIDGET (controls->name_entry));

	/* thaw the view */
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

static void 
bookmarks_editor_add_menubar (BookmarksEditorControls *controls)
{
	GtkAccelGroup *accel = GNOME_APP (controls->dialog)->accel_group;

	bookmarks_editor_fill_menu (GTK_MENU_SHELL (controls->menubar),
				    accel, controls,
				    &(controls->edit_menuitem),
				    &(controls->set_default_folder_menuitem));

	gtk_check_menu_item_set_active 
				(GTK_CHECK_MENU_ITEM (controls->edit_menuitem),
				 TRUE);
	gtk_widget_set_sensitive 
				(controls->set_default_folder_menuitem, FALSE);
}

void 
bookmarks_editor_popup_menu (BookmarksEditorControls *controls)
{
	gint ret;
	GtkWidget *popup = gtk_menu_new ();

	bookmarks_editor_fill_menu (GTK_MENU_SHELL (popup),
				    NULL, controls,
				    NULL, NULL);

	ret = gnome_popup_menu_do_popup_modal (popup, NULL, NULL,
					       NULL, NULL);

	/* Wait for the menu to be destroyed if a menu item is selected */
	if (ret != -1) 
		while (gtk_events_pending()) 
			gtk_main_iteration();

	gtk_widget_unref (popup);
}

/* 
 * a nonsense needed to get some of the GNOMEUINFO defined strings translated 
 * blame xgettext 
 */
#if 0
static char *ugh[] = {
	N_("_File"),
		N_("_Open..."),
		N_("_Revert"),
		N_("_Save"),
		N_("_Close"),
	N_("_Edit"),
		N_("C_ut"),
		N_("_Copy"),
		N_("_Paste"),
	N_("_View")
}
#endif

static void 
bookmarks_editor_fill_menu (GtkMenuShell *menushell, 
			    GtkAccelGroup *accel, 
			    BookmarksEditorControls *controls,
			    GtkWidget **edit_menuitem,
			    GtkWidget **set_default_folder_menuitem)
{
	GnomeUIInfo file_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_ITEM, N_("New _item"), NULL,
		  bookmarks_editor_new_item_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_item_small.xpm",
		  GDK_n, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _separator"), NULL,
		  bookmarks_editor_new_separator_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_FILENAME,
		  "../galeon/new_separator_small.xpm",
		  GDK_e, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _folder"), NULL,
		  bookmarks_editor_new_folder_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_folder_small.xpm",
		  GDK_F, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("New _alias"), NULL,
		  bookmarks_editor_add_alias_cb, controls, NULL,
		  GNOME_APP_PIXMAP_FILENAME, "../galeon/new_alias_small.xpm",
		  GDK_a, GDK_CONTROL_MASK, NULL },
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_OPEN_ITEM
                        (bookmarks_editor_open_clicked_cb, controls),
		GNOMEUIINFO_MENU_REVERT_ITEM 
		        (bookmarks_editor_revert_clicked_cb, controls),
		GNOMEUIINFO_MENU_SAVE_ITEM 
		        (bookmarks_editor_apply_clicked_cb, controls),
		GNOMEUIINFO_MENU_SAVE_AS_ITEM
		        (bookmarks_editor_save_as_clicked_cb, controls),
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_CLOSE_ITEM 
		        (bookmarks_editor_accept_clicked_cb, controls),
		GNOMEUIINFO_END
	};
	GnomeUIInfo sort_folder_menu_uiinfo[] = 
	{
		{ GNOME_APP_UI_ITEM, N_("_One Level"), NULL,
		  bookmarks_editor_sort_norec_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("_Recursive"), NULL,
		  bookmarks_editor_sort_rec_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_END
	};
	GnomeUIInfo edit_menu_uiinfo[] =
	{
		GNOMEUIINFO_MENU_CUT_ITEM (bookmarks_editor_cut_cb, controls),
		GNOMEUIINFO_MENU_COPY_ITEM
  		        (bookmarks_editor_copy_cb, controls),
		GNOMEUIINFO_MENU_PASTE_ITEM
		        (bookmarks_editor_paste_cb,controls),
		{ GNOME_APP_UI_ITEM, N_("_Remove bookmark"), NULL,
		  bookmarks_editor_remove_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_TRASH,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Go to bookmark target"), NULL,
		  bookmarks_editor_go_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_JUMP_TO,
		  GDK_g, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("_Set as default folder"), NULL,
		  bookmarks_editor_set_default_folder_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("Move _up"), NULL,
		  bookmarks_editor_move_up_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_UP,
		  GDK_u, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("Move _down"), NULL,
		  bookmarks_editor_move_down_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_DOWN,
		  GDK_d, GDK_CONTROL_MASK, NULL },
		{ GNOME_APP_UI_ITEM, N_("_Refresh icon"), NULL,
		  bookmarks_editor_fetch_icon_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("Add alias for r_oot bookmark"), NULL,
		  bookmarks_editor_add_alias_root_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_SUBTREE (N_("Sort folder"), 
				     sort_folder_menu_uiinfo),
		GNOMEUIINFO_END
	};
	GnomeUIInfo view_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_TOGGLEITEM, N_("_Show editing controls"), NULL,
		  bookmarks_editor_edit_toggled_cb , controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Expand all categories"), NULL,
		  bookmarks_editor_expand_all_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_OPEN,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("_Collapse all categories"), NULL,
		  bookmarks_editor_collapse_all_clicked_cb, controls, NULL,
		  GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_RED,
		  0, 0, NULL },
		/* TODO
		   GNOMEUIINFO_SEPARATOR,
		   { GNOME_APP_UI_TOGGLEITEM, 
		   N_("Show _date / time information"), NULL, 
		   bookmarks_editor_time_show_cb, controls, NULL, 
		   GNOME_APP_PIXMAP_NONE, NULL,
		   0, 0, NULL },
		*/
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("Show only _this folder"), NULL,
		  bookmarks_editor_show_only_this_folder_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Show _all bookmarks"), NULL,
		  bookmarks_editor_show_all_bookmarks_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("_Dock"), NULL,
		  bookmarks_editor_show_attach_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_END
	};
	GnomeUIInfo import_menu_uiinfo[] =
	{
		{ GNOME_APP_UI_ITEM, N_("Import from _XBEL..."), NULL,
		  bookmarks_import_xbel_activate_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Import from _Netscape bookmarks"),
		  NULL,
		  bookmarks_import_netscape_activate_cb, NULL, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Import from _Mozilla bookmarks"),
		  NULL,
		  bookmarks_import_mozilla_activate_cb, NULL, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Import from _other location..."), 
		  NULL,
		  bookmarks_import_other_activate_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_SEPARATOR,
		{ GNOME_APP_UI_ITEM, N_("Export bookmarks in X_BEL..."), NULL,
		  bookmarks_export_xbel_activate_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Export to Net_scape bookmarks..."), 
		  NULL,
		  bookmarks_export_netscape_activate_cb, controls->dialog, 
		  NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Export to Mo_zilla bookmarks..."), 
		  NULL,
		  bookmarks_export_mozilla_activate_cb, controls->dialog, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		{ GNOME_APP_UI_ITEM, N_("Export to o_ther location..."), NULL,
		  bookmarks_export_other_activate_cb, controls, NULL,
		  GNOME_APP_PIXMAP_NONE, NULL,
		  0, 0, NULL },
		GNOMEUIINFO_END
	};

	/* TODO
	   GnomeUIInfo help_menu_uiinfo[] =
	   {
	   GNOMEUIINFO_END
	   };
	*/
	GnomeUIInfo menubar_uiinfo[] =
	{
		GNOMEUIINFO_SUBTREE (N_("_File"), file_menu_uiinfo),
		GNOMEUIINFO_SUBTREE (N_("_Edit"), edit_menu_uiinfo),
		GNOMEUIINFO_SUBTREE (N_("_View"), view_menu_uiinfo),
		GNOMEUIINFO_SUBTREE (N_("I_mport / Export"), 
				     import_menu_uiinfo),
		/*TODO: GNOMEUIINFO_SUBTREE (N_("_Help"), help_menu_uiinfo),*/
		GNOMEUIINFO_END
	};
	gnome_app_fill_menu (menushell,
			     menubar_uiinfo, accel, TRUE, 0);
	if (edit_menuitem)
		*edit_menuitem = 
			GTK_WIDGET (view_menu_uiinfo[0].widget);
	if (set_default_folder_menuitem)
		*set_default_folder_menuitem = 
			GTK_WIDGET (edit_menu_uiinfo[6].widget);
}

void
bookmarks_create_edit_items (GtkMenu *popup, BookmarksEditorControls *controls)
{
	context_menu_add_item (popup, _("C_ut"), bookmarks_editor_cut_cb,
			controls, GNOME_STOCK_MENU_CUT, TRUE);
	context_menu_add_item (popup, _("_Copy"), bookmarks_editor_copy_cb,
			controls, GNOME_STOCK_MENU_COPY, TRUE);
	context_menu_add_item (popup, _("_Paste"), bookmarks_editor_paste_cb,
			controls, GNOME_STOCK_MENU_PASTE, TRUE);
	context_menu_add_item (popup, _("_Remove bookmark"),
			bookmarks_editor_remove_clicked_cb,
			controls, GNOME_STOCK_MENU_TRASH, TRUE);
}

void 
bookmarks_editor_set_root (BookmarksEditorControls *controls, BookmarkItem *b)
{
	BookmarkItem *oldroot = controls->root_bookmark;
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	controls->root_bookmark = NULL;
	if (oldroot != NULL)
		bookmarks_editor_remove_tree_item_null 
			(oldroot, controls);
	gtk_clist_clear (GTK_CLIST (controls->ctree));
	controls->root_bookmark = b;
	bookmarks_editor_place_tree_item (controls, controls->root_bookmark);
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

void
bookmarks_editors_refresh (void)
{
	GList *l;
	for (l = bookmarks_editors; l; l = l->next) {
		BookmarksEditorControls *controls = l->data;
		/* this forces recreating all tree items */
		bookmarks_editor_set_root (controls, controls->root_bookmark);
	}
}
