/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * This file contains all the global types used within Galeon.
 */

#ifndef __galeon_types_h
#define __galeon_types_h

#include <gtk/gtkwidget.h>

/* FIXME Remove these */
#include <gtk/gtktooltips.h>
#include <gtk/gtkctree.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkdnd.h>

/* forward declarations of types */
typedef struct _GaleonWindow GaleonWindow;
typedef struct _GaleonEmbed  GaleonEmbed;
typedef struct _BookmarkItem BookmarkItem;
typedef struct _HistoryHost  HistoryHost;
typedef struct _HistoryItem  HistoryItem;


/**
 * FocusType: what caused the switch to a embed object
 */
typedef enum
{
	FOCUS_ON_CREATE,	/* auto_jump or opened via menu */
	FOCUS_ON_REQUEST	/* user explicitly switched (clicked on tab) */
} FocusType;

/**
 * DockType: type of dock
 */
typedef enum
{
	DOCK_NONE,
	DOCK_BOOKMARKS,
	DOCK_HISTORY
} DockType;

/**
 * GaleonEmbed: this is the type of a single web view (e.g. a tab):
 * this is effectively Galeon's wrapping of a GtkMozEmbed with extra
 * state information.
 */ 
struct _GaleonEmbed
{
	guint32 magic;

	/* parent window in which this embed is visible */
	GaleonWindow *parent_window;

	/* embedding widget */
	GtkWidget *mozEmbed;

	/* tab widgets */
	GtkWidget *notebook_hbox;                /* packing box           */
	GtkWidget *notebook_label;               /* the label             */
	GtkWidget *notebook_close_button;        /* active close button   */

	/* C++ helper, a "GaleonWrapper" */
        gpointer wrapper;

	/* current site location and page title, if any */
	gchar *site_location;
	gchar *site_title_utf8;
	gchar *site_title;

	gint load_started; /* count of currently active connections */
	gint loadPercent;
	gint bytesLoaded;
	gint maxBytesLoaded;
	const gchar *statusMessage;

	/* time when the load was actually started */
	GTime when_started;

	/* currently set zoom level */
	gint zoom;

	/* true if the zoom was set automatically */
	gboolean zoom_auto_set : 1;

	/* true if this is the "active_embed" of the parent window */
	gboolean is_active : 1;

	/* is this browser actually visible */
	gboolean is_visible : 1;

	/* is this embed in the process of closing? */
	gboolean is_closing : 1;

	/* ignore mime processing of next url */
	gboolean mime_ignore_once : 1;

	/* has this browser content been viewed yet? */
	gboolean has_been_viewed : 1;

	/* is this a xul dialog ? */
	gboolean is_chrome : 1;

	/* how this embed was focused if applicable */
	FocusType focus_type;

	/* close timeout to ensure all mozembed callbacks have finished */
	gint close_timeout_id;
};

/**
 * GaleonWindow: this is the type of Galeon windows, i.e. a wrapping of
 * the toplevel GnomeApp window. A single window can contain many 
 * GaleonEmbeds.
 */
struct _GaleonWindow
{
	guint32 magic;

	/* window parts */
	GtkWidget *WMain;
	GaleonEmbed *active_embed;
	GList *embed_list; /* list of GaleonEmbeds */
	char *tempMessage;
	gint visible_embeds; /* number of visible GaleonEmbeds, used to
			      * track whether or not the window is visible */

	/* number of embeds which are running (for spinner) */
	gint n_running;

	/* assigned size */
	gboolean set_size : 1;

	/* still running the activity bar? */
	gboolean progress_timeout : 1;

	/* have we set the encodings menu yet? */
	gboolean encodings_menu_set : 1;

	/* GUI components */
	GtkWidget *notebook;
	GtkWidget *hpaned;
	GtkWidget *hbox;
	GtkWidget *docked;                /* currently docked control      */
	const gchar *docked_width_pref;   /* where width of dock is stored */

	/* Toolbar components */
	GtkWidget *dock;
	GtkWidget *appbar;
	GtkWidget *menubar;
	GtkWidget *main_toolbar;
	GtkWidget *main_dockitem;
	GtkWidget *right_toolbar;
	GtkWidget *location_toolbar;
	GtkWidget *location_dockitem;

	/* Toolbar buttons etc */
	GtkWidget *new_button;
	GtkWidget *back_button;
	GtkWidget *up_button;
	GtkWidget *forward_button;
	GtkWidget *refresh_button;
	GtkWidget *home_button;
	GtkWidget *stop_button;
	GtkWidget *go_button;
	GtkWidget *zoom_spin;
	GtkWidget *drag_event_box;
	GtkWidget *drag_pixmap;
	GtkWidget *location_gnomeentry;
	GtkWidget *location_entry;
	GtkWidget *spinner;               /* the GnomeAnimator spinner */
	GtkWidget *history_button;
	GtkWidget *bookmarks_button;
	
	/* menus */
	GtkWidget *file_menu;
	GtkWidget *bookmarks_menu;
	GtkWidget *go_menu;

	/* menu items */
	GtkWidget *file_session_history_separator_sup;
	GtkWidget *file_session_history_separator_inf;
	GtkWidget *view_toolbar;
	GtkWidget *view_bookmarks;
	GtkWidget *view_statusbar;
	GtkWidget *view_menubar;
	GtkWidget *view_fullscreen;
	GtkWidget *detach_tab;
	GtkWidget *move_tab_to_window_menu;
	GtkWidget *next_tab;
	GtkWidget *prev_tab;
	GtkWidget *move_tab_left;
	GtkWidget *move_tab_right;
	GtkWidget *load_images_always;
	GtkWidget *load_images_from_current_server_only;
	GtkWidget *load_images_never;
	GtkWidget *animate_always;
	GtkWidget *animate_once_through;
	GtkWidget *animate_never;
	GtkWidget *use_own_fonts;
	GtkWidget *use_own_colors;
	GtkWidget *enable_java;
	GtkWidget *enable_javascript;
	GtkWidget *enable_proxy;
	GtkWidget *autojump;
	GtkWidget *add_bookmark;
	GtkWidget *bookmarks_separator;
	GtkWidget *encoding;
	/* edit menu menuitems */
	GtkWidget *edit_cut;
	GtkWidget *edit_copy;
	GtkWidget *edit_paste;

	/* dialogs and views */
	GtkWidget *print_dialog;
	GtkWidget *find_dialog;
	GtkWidget *open_url_entry;

	/* list of toolbars */
	GList *bookmarks_toolbars; 
	GtkTooltips *bookmarks_tooltips;

	/* saved window geometry for fullscreen windows */
	gint x, y, width, height;

	/* visibility */
	gboolean menuBarOn : 1;
	gboolean toolBarOn : 1;
	gboolean locationBarOn : 1;
	gboolean statusBarOn : 1;

	/* is this a popup window ? */
	gboolean is_popup : 1;

	/* dock type */
	DockType dock_type;

	/* saved layout */
	gchar *layout;

	/* notifiers IDs to remove them when the window is destroyed */
	GList *notifiers_id;

	/* used for draggable tabs */
	gint drag_motion_signal;
	gint drag_release_signal;
};

/**
 * Magic numbers used to detect if the structure has been accidentally 
 * free'd or corrupted
 */
#define GALEON_WINDOW_MAGIC   (0x87de130f)
#define GALEON_EMBED_MAGIC    (0x45a0ebed)

/**
 * Checking macros used to validate window and embed pointers and
 * their magic. At some point we'll probably do all of this with
 * GObjects, but for now this is it. This will introduce some (very
 * small) overhead, and we should probably provide a #ifdef DEBUG
 * which disables these (around v1.0).
 */

#define return_if_not_window(window) \
  g_return_if_fail (window != NULL); \
  g_return_if_fail (window->magic == GALEON_WINDOW_MAGIC);

#define return_val_if_not_window(window, val) \
  g_return_val_if_fail (window != NULL, val); \
  g_return_val_if_fail (window->magic == GALEON_WINDOW_MAGIC, val);

#define return_if_not_embed(embed) \
  g_return_if_fail (embed != NULL); \
  g_return_if_fail (embed->magic == GALEON_EMBED_MAGIC);

#define return_val_if_not_embed(embed, val) \
  g_return_val_if_fail (embed != NULL, val); \
  g_return_val_if_fail (embed->magic == GALEON_EMBED_MAGIC, val);

/**
 * WidgetLookup: a mapping from strings to pointers, used to lookup
 * a group of widgets from a Glade description.
 */
typedef struct 
{ 
	const gchar *name; 
	GtkWidget **ptr;
} WidgetLookup;

/**
 * KeyCode: key pressed char codes, taken from mozilla
 */
typedef enum
{
	DOM_VK_CANCEL = 3,
	DOM_VK_BACK_SPACE = 8,
	DOM_VK_TAB = 9,
	DOM_VK_CLEAR = 12,
	DOM_VK_RETURN = 13,
	DOM_VK_ENTER = 14,
	DOM_VK_SHIFT = 16,
	DOM_VK_CONTROL = 17,
	DOM_VK_ALT = 18,
	DOM_VK_PAUSE = 19,
	DOM_VK_CAPS_LOCK = 20,
	DOM_VK_ESCAPE = 27,
	DOM_VK_SPACE = 32,
	DOM_VK_PAGE_UP = 33,
	DOM_VK_PAGE_DOWN = 34,
	DOM_VK_END = 35,
	DOM_VK_HOME = 36,
	DOM_VK_LEFT = 37,
	DOM_VK_UP = 38,
	DOM_VK_RIGHT = 39,
	DOM_VK_DOWN = 40,
	DOM_VK_PRINTSCREEN = 44,
	DOM_VK_INSERT = 45,
	DOM_VK_DELETE = 46,
	DOM_VK_0 = 48,
	DOM_VK_1 = 49,
	DOM_VK_2 = 50,
	DOM_VK_3 = 51,
	DOM_VK_4 = 52,
	DOM_VK_5 = 53,
	DOM_VK_6 = 54,
	DOM_VK_7 = 55,
	DOM_VK_8 = 56,
	DOM_VK_9 = 57,
	DOM_VK_SEMICOLON = 59,
	DOM_VK_EQUALS = 61,
	DOM_VK_A = 65,
	DOM_VK_B = 66,
	DOM_VK_C = 67,
	DOM_VK_D = 68,
	DOM_VK_E = 69,
	DOM_VK_F = 70,
	DOM_VK_G = 71,
	DOM_VK_H = 72,
	DOM_VK_I = 73,
	DOM_VK_J = 74,
	DOM_VK_K = 75,
	DOM_VK_L = 76,
	DOM_VK_M = 77,
	DOM_VK_N = 78,
	DOM_VK_O = 79,
	DOM_VK_P = 80,
	DOM_VK_Q = 81,
	DOM_VK_R = 82,
	DOM_VK_S = 83,
	DOM_VK_T = 84,
	DOM_VK_U = 85,
	DOM_VK_V = 86,
	DOM_VK_W = 87,
	DOM_VK_X = 88,
	DOM_VK_Y = 89,
	DOM_VK_Z = 90,
	DOM_VK_a = 97,
	DOM_VK_b = 98,
	DOM_VK_c = 99,
	DOM_VK_d = 100,
	DOM_VK_e = 101,
	DOM_VK_f = 102,
	DOM_VK_g = 103,
	DOM_VK_h = 104,
	DOM_VK_i = 105,
	DOM_VK_j = 106,
	DOM_VK_k = 107,
	DOM_VK_l = 108,
	DOM_VK_m = 109,
	DOM_VK_n = 110,
	DOM_VK_o = 111,
	DOM_VK_p = 112,
	DOM_VK_q = 113,
	DOM_VK_r = 114,
	DOM_VK_s = 115,
	DOM_VK_t = 116,
	DOM_VK_u = 117,
	DOM_VK_v = 118,
	DOM_VK_w = 119,
	DOM_VK_x = 120,
	DOM_VK_y = 121,
	DOM_VK_z = 122,
	DOM_VK_NUMPAD0 = 96,
	DOM_VK_NUMPAD1 = 97,
	DOM_VK_NUMPAD2 = 98,
	DOM_VK_NUMPAD3 = 99,
	DOM_VK_NUMPAD4 = 100,
	DOM_VK_NUMPAD5 = 101,
	DOM_VK_NUMPAD6 = 102,
	DOM_VK_NUMPAD7 = 103,
	DOM_VK_NUMPAD8 = 104,
	DOM_VK_NUMPAD9 = 105,
	DOM_VK_MULTIPLY = 106,
	DOM_VK_ADD = 107,
	DOM_VK_SEPARATOR = 108,
	DOM_VK_SUBTRACT = 109,
	DOM_VK_DECIMAL = 110,
	DOM_VK_DIVIDE = 111,
	DOM_VK_F1 = 112,
	DOM_VK_F2 = 113,
	DOM_VK_F3 = 114,
	DOM_VK_F4 = 115,
	DOM_VK_F5 = 116,
	DOM_VK_F6 = 117,
	DOM_VK_F7 = 118,
	DOM_VK_F8 = 119,
	DOM_VK_F9 = 120,
	DOM_VK_F10 = 121,
	DOM_VK_F11 = 122,
	DOM_VK_F12 = 123,
	DOM_VK_F13 = 124,
	DOM_VK_F14 = 125,
	DOM_VK_F15 = 126,
	DOM_VK_F16 = 127,
	DOM_VK_F17 = 128,
	DOM_VK_F18 = 129,
	DOM_VK_F19 = 130,
	DOM_VK_F20 = 131,
	DOM_VK_F21 = 132,
	DOM_VK_F22 = 133,
	DOM_VK_F23 = 134,
	DOM_VK_F24 = 135,
	DOM_VK_NUM_LOCK = 144,
	DOM_VK_SCROLL_LOCK = 145,
	DOM_VK_COMMA = 188,
	DOM_VK_PERIOD = 190,
	DOM_VK_SLASH = 191,
	DOM_VK_BACK_QUOTE = 192,
	DOM_VK_OPEN_BRACKET = 219,
	DOM_VK_BACK_SLASH = 220,
	DOM_VK_CLOSE_BRACKET = 221,
	DOM_VK_QUOTE = 222,
	DOM_VK_META = 224,
	DOM_VK_PLUS = 43,
	DOM_VK_MINUS = 45,

	DOM_VK_LEFT_2 = 65361,   /* added */
	DOM_VK_KP_RIGHT = 65363, /* added */
	DOM_VK_KP_LEFT = 65430   /* added */
} KeyCode;

/**
 * KeyModifier: key modifiers
 */
typedef enum
{
	ALT_KEY   = (1 << 0),
	CTRL_KEY  = (1 << 1),
	SHIFT_KEY = (1 << 2),
	META_KEY  = (1 << 3),
	KEY_CODE  = (1 << 4)
} KeyModifier;

/**
 * BookmarkTitle: type of behaviour when making a new bookmark
 */
typedef enum 
{ 
	BOOKMARK_PAGE_TITLE,
	BOOKMARK_ASK_TITLE
} BookmarkTitle;

/**
 * StartPageType: start site type
 */
typedef enum
{
	STARTPAGE_HOME,
	STARTPAGE_LAST,
	STARTPAGE_BLANK,
} StartPageType;

/**
 * HomePageType: home page type
 */
typedef enum
{
	HOMEPAGE_URL,
	HOMEPAGE_MYPORTAL
} HomePageType;

/**
 * MimeAction: types of actions to take for opening files
 */
typedef enum
{
	MIME_UNKNOWN,
	MIME_SAVE_TO_DISK,
	MIME_RUN_PROGRAM,      /* should run the GNOME association */
	MIME_ASK_ACTION,
	MIME_IGNORE
} MimeAction;

/**
 * FilePickerMode: What mode FilePicker should run in
 */

typedef enum
{
	modeOpen = 0,
	modeSave = 1,
	modeGetFolder =2
} FilePickerMode;

/**
 * AutoResume: how to handling resuming crashed sessions
 */
typedef enum
{
	RESUME_NEVER,
	RESUME_ASK,
	RESUME_ALWAYS
} AutoResume;

/**
 * NSItemType: netscape bookmark item type
 */
typedef enum
{
	NS_SITE,
	NS_NOTES,
	NS_FOLDER,
	NS_FOLDER_END,
	NS_SEPARATOR,
	NS_UNKNOWN
} NSItemType;

/**
 * ImageLoadingType: various image loading behaviours
 */
typedef enum
{
	IMAGE_LOADING_ALWAYS                   = 0,
	IMAGE_LOADING_FROM_CURRENT_SERVER_ONLY = 1,
	IMAGE_LOADING_NEVER                    = 2
} ImageLoadingType;

/**
 * AnimateType: various image animation behaviours
 */
typedef enum
{
	ANIMATE_ALWAYS       = 0,
	ANIMATE_ONCE_THROUGH = 1,
	ANIMATE_NEVER        = 2
} AnimateType;

/**
 * ViewState: destination for loading URLs
 */
typedef enum
{
	VIEWSTATE_NONE,
	VIEWSTATE_DEFAULT,
	VIEWSTATE_WINDOW,
	VIEWSTATE_TAB
} ViewState;

/**
 * LinkState: generalized URL activation types
 */
typedef enum
{
	LINKSTATE_BUTTON_MASK = 7,
	LINKSTATE_SHIFTED     = (1 << 3),
	LINKSTATE_CTRLED      = (1 << 4),
	LINKSTATE_ALTED       = (1 << 5),
	/* force a certain mode rather than basing off the pref */
	LINKSTATE_NEWTAB      = 1 + (1 << 9),
	LINKSTATE_NEWWIN      = 1 + (1 << 10),
} LinkState;

/**
 * ContextMenuType: various types of context menu
 */
typedef enum
{
	CONTEXT_NONE     = 0,
	CONTEXT_DEFAULT  = (1 << 1),
	CONTEXT_LINK     = (1 << 2),
	CONTEXT_IMAGE    = (1 << 3),
	CONTEXT_DOCUMENT = (1 << 4),
	CONTEXT_INPUT    = (1 << 5),
	CONTEXT_XUL      = (1 << 7),
} ContextMenuType;

/**
 * MimeItem: a particular MIME type binding
 */
typedef struct
{
	gchar *name; 
	MimeAction action;
	gchar *helperApp;
	gboolean terminalHelper;
	gboolean urlHelper;
	gboolean alwaysUse;
} MimeItem;

typedef struct
{
	gchar *serif;
	gchar *sansserif;
	gchar *cursive;
	gchar *fantasy;
	gchar *monospace;

	int min_size;
	
	gboolean changed;
} LanguageFontInfo;

/**
 * PreferencesDialog: structure representing Preferences dialog box
 */
typedef struct
{
	GtkWidget *dialog;
	GtkWidget *notebook;
	GtkWidget *sidebar;
	GtkWidget *themes_page;
	GtkWidget *spinner_page;
	GtkWidget *iconlist;
	GtkWidget *themelist;
	GtkWidget *theme_dir_entry;
	GtkWidget *smart_bm_example_entry;
	GtkWidget *lang_entry;/*FIXME*/
	GtkWidget *toolbar_style_optionmenu;
	GtkWidget *toolbar_share_main_toolbar;
	GtkWidget *startpage_entry;
	GtkWidget *startup_show_homepage;
	GtkWidget *newpage_show_homepage;
	GtkWidget *always_save_session_check;
	GtkWidget *bookmarks_use_page_title;
	GtkWidget *bookmarks_show_actions_top;
	GtkWidget *favicons_enabled_check;
	GtkWidget *autobookmarks_enabled_check;
	GtkWidget *autobookmarks_count_spin;
	GtkWidget *autobookmarks_shorten_spin;
	GtkWidget *smart_bm_entry_width_spin;
	GtkWidget *smart_bm_goto_button_checkbutton;
	GtkWidget *smart_bm_history_checkbutton;
	GtkWidget *java_checkbutton;
	GtkWidget *javascript_checkbutton;
	GtkWidget *user_agent_entry;

	/* toolbar editor */
	GtkWidget *add_button;
	GtkWidget *remove_button;
	GtkWidget *up_button;
	GtkWidget *down_button;
	GtkWidget *available_clist;
	GtkWidget *current_clist;

	/* mime page */
	MimeItem *selected_mime;
	gboolean enable_edit_mime;
	gint mime_last_sort_column;
	GtkSortType mime_last_sort_type;
	GtkWidget *mime_list;
	GtkWidget *mime_save_to_disk;
	GtkWidget *mime_helper_frame;
	GtkWidget *mime_name_entry;
	GtkWidget *mime_helper_entry;
	GtkWidget *mime_always_use;
	GtkWidget *mime_terminal;
	GtkWidget *mime_url_helper;

	/* downloader page */
	GtkWidget *download_use_builtin;
	GtkWidget *download_command_entry;
	GtkWidget *download_command_combo;
	GtkWidget *download_run_in_terminal;
	GtkWidget *download_autodownload;
	GtkWidget *download_ask_dir;
	GtkWidget *download_dir_entry;
	GtkWidget *download_ftp_use_builtin;

	/* tabbed browsing page */
	GtkWidget *tabbed_enable_check;
	GtkWidget *tabbed_popups_enable_check;
	GtkWidget *tabbed_autojump_check;
	GtkWidget *tabbed_insert_new_tabs_check;
	GtkWidget *tabbed_always_show_check;
	GtkWidget *tabbed_closebutton_check;
	GtkWidget *tabbed_closebutton_insensitive_check;
	GtkWidget *tabbed_shorten_spin;
	GtkWidget *tabbed_shorten_spin2;
	GtkWidget *tabbed_position_optionmenu;
	GtkWidget *tabbed_style_radiobutton1;
	GtkWidget *tabbed_separator_check;
	GtkWidget *tabbed_vowel_check;
	GtkWidget *tabbed_prefix_check;
	GtkWidget *tabbed_loading_color;
	GtkWidget *tabbed_new_color;

	/* ... */
	GtkWidget *autocompletion_enabled_checkbutton;
	GtkWidget *autocompletion_sort_whole;
        GtkWidget *proxy_use_checkbutton;
	GtkWidget *http_proxy_entry;
	GtkWidget *ssl_proxy_entry;
	GtkWidget *ftp_proxy_entry;
	GtkWidget *http_proxy_port_spin;
	GtkWidget *ssl_proxy_port_spin;
	GtkWidget *ftp_proxy_port_spin;
	GtkWidget *no_proxies_for_entry;
	GtkWidget *disk_cache_entry;
	GtkWidget *mem_cache_entry;
	GtkWidget *serif_font_fpicker;
	GtkWidget *sansserif_font_fpicker;
	GtkWidget *cursive_font_fpicker;
	GtkWidget *fantasy_font_fpicker;
	GtkWidget *monospace_font_fpicker;
	GtkWidget *minimum_font_size_spin;
	GtkWidget *default_font_optionmenu;
	GtkWidget *history_expire_spin;
	GtkWidget *use_own_fonts_checkbutton;
	GtkWidget *images_always_load;
	GtkWidget *show_toolbars_in_fullscreen_check;
	GtkWidget *show_bookmarks_in_fullscreen_check;
	GtkWidget *show_menubar_in_fullscreen_check;
	GtkWidget *show_statusbar_in_fullscreen_check;
	GtkWidget *use_external_source_viewer_check;
	GtkWidget *external_source_viewer_entry;
	GtkWidget *crash_optionmenu;
	GtkWidget *allow_popups_checkbutton;
	GtkWidget *statusbar_rewrite_check;
	GtkWidget *windows_title_entry;
	GtkWidget *fullscreen_stacking_checkbutton;
	GtkWidget *myportal_stylesheet_entry;
	GtkWidget *spinner_dir_entry;
	GtkWidget *autodetect_charset_optionmenu;
	GtkWidget *default_charset_combo;
	GtkWidget *default_charset_entry;
	GtkWidget *lang_list;
	GtkWidget *background_color;
	GtkWidget *text_color;
	GtkWidget *unvisited_link_color;
	GtkWidget *visited_link_color;
	GtkWidget *use_own_colors_checkbutton;
	GtkWidget *underline_links_checkbutton;
	GtkWidget *http_version_optionmenu;
	GtkWidget *proxy_keepalive_checkbutton;
	GtkWidget *history_popup_checkbutton;
	GtkWidget *middle_button_optionmenu;
	GtkWidget *right_button_optionmenu;
	GtkWidget *cookies_always_accept;
	GtkWidget *cookie_manager_button;
	GtkWidget *warn_cookie_checkbutton;
	GtkWidget *passwords_save_checkbutton;
	GtkWidget *wheel_nokey_action_optionmenu;
	GtkWidget *wheel_nokey_step_spinbutton;
	GtkWidget *wheel_nokey_default_step_checkbutton;
	GtkWidget *wheel_alt_action_optionmenu;
	GtkWidget *wheel_alt_step_spinbutton;
	GtkWidget *wheel_alt_default_step_checkbutton;
	GtkWidget *wheel_ctrl_action_optionmenu;
	GtkWidget *wheel_ctrl_step_spinbutton;
	GtkWidget *wheel_ctrl_default_step_checkbutton;
	GtkWidget *wheel_shift_action_optionmenu;
	GtkWidget *wheel_shift_step_spinbutton;
	GtkWidget *wheel_shift_default_step_checkbutton;
	GtkWidget *crash_behaviour_optionmenu;
	GtkWidget *links_drag_checkbutton;
	GtkWidget *cache_compare_optionmenu;
	GtkWidget *lang_encoding_optionmenu;
	GtkWidget *autozoom_checkbutton;

	/* keep the fonts changed in the prefs dialog */
	LanguageFontInfo *fonts;
	gboolean last_language;
} PreferencesDialog;

/**
 * HistoryView: a history view (either dialog or dockeD)
 */
typedef struct
{
	GaleonWindow *window;            /* parent window */
	GtkWidget *view;                 /* either the dock or the dialog */
	GtkWidget *ctree;
	GtkWidget *host_checkbutton;
	GtkWidget *time_optionmenu;
	GtkWidget *search_entry;
	GtkWidget *dock_close_button;
	GtkWidget *dock_detach_button;
	gboolean group;
	gchar *search;
	gint time;
	GtkWidget *sibling;              /* used in tree construction */
	GHashTable *host_to_node;        /* maps HistoryHosts to tree nodes */
	GHashTable *item_to_node;        /* maps HistoryItems to tree nodes */
	HistoryItem *selected_item;
} HistoryView;

/**
 * HistoryHost: a group of history elements assembled by host
 */
struct _HistoryHost
{
	/* NB: these must stay the same as HistoryItem */
	gchar *url; /* NOTE: unused! */
	gchar *title_locale;
	gchar *title_utf8;
	GTime first;
	GTime last;
	gint visits;

	/* tree specific data */
	GtkCTreeNode *node;
	GtkCTreeNode *sibling;

	/* stored zoom for this host */
	gint zoom;

	/* dominant child -- the location visited the most of this server */
	HistoryItem *dominant_item;
};

/**
 * HistoryItem: an item in the history
 */
struct _HistoryItem
{
	/* NB: these must stay the same as HistoryItem */
	gchar *url;
	gchar *title_locale;
	gchar *title_utf8;
	GTime first;
	GTime last;
	gint visits;

	/* parent host */
	HistoryHost *host;
};

/**
 * ToolbarItemType: various toolbar item types
 */
typedef enum
{
	TOOLBAR_ITEM_NEW = 0,
	TOOLBAR_ITEM_BACK,
	TOOLBAR_ITEM_UP,
	TOOLBAR_ITEM_FORWARD,
	TOOLBAR_ITEM_REFRESH,
	TOOLBAR_ITEM_START,
	TOOLBAR_ITEM_STOP,
	TOOLBAR_ITEM_GO,
	TOOLBAR_ITEM_ZOOM_SPIN,
	TOOLBAR_ITEM_DRAG_HANDLE,
	TOOLBAR_ITEM_LOCATION_ENTRY,
	TOOLBAR_ITEM_SPINNER,
	TOOLBAR_ITEM_SEPARATOR,
	TOOLBAR_ITEM_FIND,
	TOOLBAR_ITEM_BOOKMARKS,
	TOOLBAR_ITEM_HISTORY,
	TOOLBAR_ITEM_PRINT,

	/* must be last */
	TOOLBAR_ITEM_COUNT
} ToolbarItemType;	

/**
 * ToolbarItem: a toolbar item
 */
typedef struct
{
	const gchar *label;
	const gchar *tooltip;
	const gchar *stock_icon;
	const gchar *theme_icon;
	gboolean priority; /* i.e. show the text in horizontal mode */
	gboolean togglebutton;
	gint order;        /* order value for default insertion place */
	void (*clicked_cb) (GtkButton *button, 
			    GaleonWindow *window);
	gint (*button_press_event_cb) (GtkButton *button, 
				       GdkEventButton *event,
				       GaleonWindow *window);
	gint (*button_release_event_cb) (GtkButton *button, 
				         GdkEventButton *event,
				         GaleonWindow *window);
	void (*drag_data_received_cb) (GtkWidget *widget, 
				       GdkDragContext *drag_context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info, guint time,
				       GaleonWindow *window);
} ToolbarItem;

/**
 * PixmapData: a single pixmap (stored on the X server)
 */
typedef struct
{
	GdkPixmap *pixmap;
	GdkBitmap *mask;
} PixmapData;

/**
 * BookmarkType: various types of bookmarks
 */ 
typedef enum
{
	BM_FOLDER,
	BM_SITE,
	BM_SEPARATOR,
	BM_AUTOBOOKMARKS
} BookmarkType;

/**
 * ToolbarStyle: various toolbar styles
 */
typedef enum
{
	TOOLBAR_STYLE_HORIZONTAL,
	TOOLBAR_STYLE_VERTICAL,
	TOOLBAR_STYLE_TEXT_ONLY,
	TOOLBAR_STYLE_ICONS_ONLY,       /* not used for bookmark toolbars */
} ToolbarStyle;

/**
 * Forward declaration
 */
typedef struct _BookmarkTreeItem BookmarkTreeItem;

/**
 * BookmarkItem: a single entry in bookmarks
 */
struct _BookmarkItem
{
	/* the type of this bookmark */
	BookmarkType type;

	/* the user provided name of the bookmark */
	gchar *name;

	/* the site it points at */
	gchar *url;

	/* nickname used as a shorthand for this bookmark */
	gchar *nick;
	
	/* any user provided comments about the bookmark */
	gchar *notes; 

	/* the pixmap file used to represent it */
	gchar *pixmap_file;

	/* the in-memory binary data of the pixmap */
	PixmapData *pixmap_data;

	/* the list of children if this is a folder */
	GList *list;

	/* a handles to find the bookmark in the editor dialogs */
	/* this is a list of BookmarkTreeItem */
	GList *tree_items;

	/* parent folder (NULL for root bookmark) */
	BookmarkItem *parent; 

	/* if TRUE and this is a folder, create a toolbar from it */
	gboolean create_toolbar : 1; 

	/* if TRUE, we add this bookmark to the document context menu */
	gboolean create_context_menu : 1;

	/* if TRUE and this is a folder, then it is shown open in the editor */
	gboolean expanded : 1; 

	/* if not NULL, then this is an alias. The pointed bookmark can be an
	 * alias too */
	BookmarkItem *alias_of;

	/* Points to the first alias of this bookmark, if any. If the 
	 * bookmark is an alias, then it points to the next alias */
	BookmarkItem *alias;

	/* a unique identifier */
	glong id;

	/* time info (if they are 0, then they should be ignored) */
	GTime time_added;
	GTime time_modified;
	GTime time_visited;

	/* toolbar style */
	ToolbarStyle toolbar_style;
};

/**
 * BookmarksEditorControlsType: types of bookmark editor dialogs
 */
typedef enum
{
	STANDARD_BOOKMARKS_EDITOR,
	DOCKED_BOOKMARKS_EDITOR
} BookmarksEditorControlsType;

/**
 * BookmarksEditorControls: representation of a bookmarks editor.
 */
typedef struct
{
	BookmarksEditorControlsType type;

	/* the root folder being edited here */
	BookmarkItem *root_bookmark;

	/* the widgets of the editor */
	GtkWidget *dialog;
	GtkWidget *menubar;
	GtkWidget *toolbar;
	GtkWidget *menubar_dock;
	GtkWidget *toolbar_dock;
	GtkWidget *ctree;
	GtkWidget *vpane;
	GtkWidget *edit_frame;
	GtkWidget *name_entry;
	GtkWidget *url_entry;
	GtkWidget *nick_entry;
	GtkWidget *pixmap_entry;
	GtkWidget *pixmap_file_entry;
	GtkWidget *create_toolbar_toggle;
	GtkWidget *create_context_menu_toggle;
	GtkWidget *notes_text;
	GtkWidget *time_added_entry;
	GtkWidget *time_modified_entry;
	GtkWidget *time_visited_entry;

	BookmarkItem *selection;    /* The current selected bookmark */
	BookmarkItem *last_pressed; /* For dnd */
	gboolean lock_edit_controls; /* Don't update bookmark when editing 
				      * fields. Set to TRUE to avoid recursion
				      */
	/* widgets we need to update at various stages*/
	GtkWidget *edit_menuitem;
	GtkWidget *edit_button;

	GtkWidget *set_default_folder_menuitem;
	
	GaleonWindow *window;

	/* the clipboard data owned by this editor, if any */
	gchar *clipboard;

} BookmarksEditorControls;

/**
 * Pair of controls / tree_item for a bookmark
 */
struct _BookmarkTreeItem
{
	BookmarksEditorControls *controls;
	GtkCTreeNode *tree_item; 	
};

/**
 * The "folder extra items" actions
 */
typedef enum
{
	EXTRA_ITEM_LOCATION_EXCLUSIVE = -1,
	EXTRA_ITEM_LOCATION_TOP_SUBMENU = 0,
	EXTRA_ITEM_LOCATION_BOTTOM_SUBMENU,
	EXTRA_ITEM_LOCATION_CONTEXT,
} ExtraItemsLocation;

/**
 * DnDType: various types that can be dragged and dropped
 */
typedef enum
{
	DND_TARGET_GALEON_BOOKMARK, 
	DND_TARGET_GALEON_URL,
	DND_TARGET_NETSCAPE_URL, 
	DND_TARGET_TEXT_URI_LIST,
	DND_TARGET_STRING,
} DnDType;

/**
 * CookieBase: FIXME?
 */
typedef struct
{
	gchar *type;
	gchar *domain;
	gboolean remove;
	gboolean block;
} CookieBase;

/**
 * Cookie: the type of cookies
 */
typedef struct
{
	CookieBase base;
	gchar *name;
	gchar *value;
	gchar *path;
	gchar *secure;
	gchar *expire;
} Cookie;

/**
 * Password: a password manager entry
 */
typedef struct
{
	gchar *host;
	gchar *username;
	gboolean remove;
} Password;

/**
 * PasswordType: To distinguish actual passwords from blocked password sites
 */
typedef enum
{
	PASSWORD, /* FIXME: prefix  */
	REJECT
} PasswordType;

/**
 * PDMPage: For selecting which page in the persistent data manager is active
 */
typedef enum
{
	COOKIES = 0, /* FIXME: prefix */
	COOKIESITES = 1,
	IMAGESITES = 2,
	PASSWORDS = 3,
	PASSWORDSITES = 4
} PDMPage;

/**
 * WheelModKey: modifiers that can be used in conjuction with the mouse wheel
 */
typedef enum
{
	WHEEL_MOD_NOKEY,
	WHEEL_MOD_ALT,
	WHEEL_MOD_CTRL,
	WHEEL_MOD_SHIFT
} WheelModKey;

/**
 * Downloader: the choice between the builtin (mozilla) downloader, the
 * external GTM downloader (via CORBA), or using some external command.
 */
typedef enum
{
	DOWNLOADER_BUILTIN  = 0,
	DOWNLOADER_GTM      = 1,
	DOWNLOADER_EXTERNAL = 2
} Downloader;

/**
 * PreferencesPageId: different pages of the prefs dialog
 */
typedef enum
{
	PREFS_PAGE_ID_BROWSING_GENERAL = 1,
	PREFS_PAGE_ID_BROWSING_BOOKMARKS,
	PREFS_PAGE_ID_BROWSING_HISTORY,
	PREFS_PAGE_ID_USER_INTERFACE_TABS,
	PREFS_PAGE_ID_USER_INTERFACE_WINDOWS,
	PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS,
	PREFS_PAGE_ID_USER_INTERFACE_MOUSE,
	PREFS_PAGE_ID_HANDLERS_DOWNLOADING,
	PREFS_PAGE_ID_HANDLERS_PROGRAMS,
	PREFS_PAGE_ID_HANDLERS_MIME_TYPES,
	PREFS_PAGE_ID_RENDERING_FONTS_COLORS,
	PREFS_PAGE_ID_RENDERING_LANGUAGES,
	PREFS_PAGE_ID_ADVANCED_NETWORK,
	PREFS_PAGE_ID_ADVANCED_FILTERING,
	PREFS_PAGE_ID_ADVANCED_PERSISTENT_DATA,
	PREFS_PAGE_ID_ADVANCED_CRASH_RECOVERY,
	PREFS_PAGE_COUNT /* must be last */
} PreferencesPageId;

/**
 * FileCallbackFunc: the type of functions used in file callbacks
 */
typedef void (*FileCallbackFunc) (const gchar *file, gpointer data);

#endif /* __galeon_types_h */
