/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mozilla_i18n.h"
#include "mozilla.h"
#include "GaleonWrapper.h"

#include <time.h>
#include <gtkmozembed_internal.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>

#include "nsString.h"
#include "nsXPIDLString.h"
#include "nsIPrefService.h"
#include "nsICharsetConverterManager.h"
#include "nsICharsetConverterManager2.h"
#include "nsIUnicodeEncoder.h"
#include "nsISupportsPrimitives.h"
#include "nsIPrintOptions.h"
#include "nsGfxCIID.h"
#include "nsICacheService.h"
#include "nsIPassword.h"
#include "nsIPasswordManager.h" 
#include "nsIDOMWindow.h"
#include "nsIEmbeddingSiteWindow.h"
#include "nsIWindowWatcher.h"
#include "nsIWebBrowserChrome.h"
#include "nsIFontList.h"
#include "nsIFontEnumerator.h"
#include "nsIExternalHelperAppService.h"
#include "nsCExternalHandlerService.h"
#include "nsILocalFile.h"
#include "nsPIDOMWindow.h"
#include "nsIDOMWindowInternal.h"
#include "nsIJVMManager.h"
#include "nsIJSConsoleService.h"

static NS_DEFINE_CID(kPrintOptionsCID, NS_PRINTOPTIONS_CID);
static NS_DEFINE_CID(kJVMManagerCID, NS_JVMMANAGER_CID);

/* local function prototypes */
static char  *convert_ns_string_to_c_string (const nsString & ns_string);

extern "C" gboolean
mozilla_load_prefs (void)
{
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	g_return_val_if_fail (prefService != nsnull, FALSE);

	nsresult rv = prefService->ReadUserPrefs (nsnull);
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_save_prefs (void)
{
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	g_return_val_if_fail (prefService != nsnull, FALSE);

	nsresult rv = prefService->SavePrefFile (nsnull);
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_preference_set: set a string mozilla preference
 */
extern "C" gboolean
mozilla_preference_set(const char *preference_name, const char *new_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);
	g_return_val_if_fail (new_value != NULL, FALSE);
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult rv = pref->SetCharPref (preference_name, new_value);            
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_set_boolean: set a boolean mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_boolean (const char *preference_name,
				gboolean new_boolean_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);
  
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));
  
	if (pref)
	{
		nsresult rv = pref->SetBoolPref (preference_name,
				new_boolean_value ? PR_TRUE : PR_FALSE);
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_set_int: set an integer mozilla preference
 */
extern "C" gboolean
mozilla_preference_set_int (const char *preference_name, int new_int_value)
{
	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		nsresult rv = pref->SetIntPref (preference_name, new_int_value);
		return NS_SUCCEEDED (rv) ? TRUE : FALSE;
	}

	return FALSE;
}

/**
 * mozilla_preference_get_int: get an integer mozilla preference
 */
extern "C" gint
mozilla_preference_get_int (const char *preference_name)
{
	int value = -1;

	g_return_val_if_fail (preference_name != NULL, FALSE);

	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		pref->GetIntPref (preference_name, &value);
	}

	return value;
}

/**
 * mozilla_preference_get: get a string mozilla preference
 */
extern "C" gchar *
mozilla_preference_get(const char *preference_name)
{
	gchar *value = NULL;
	gchar *result = NULL;

	g_return_val_if_fail (preference_name != NULL, FALSE);
	nsCOMPtr<nsIPrefService> prefService = 
				do_GetService (NS_PREFSERVICE_CONTRACTID);
	nsCOMPtr<nsIPrefBranch> pref;
	prefService->GetBranch ("", getter_AddRefs(pref));

	if (pref)
	{
		pref->GetCharPref (preference_name, &value);            
	}

	/* it's allocated by mozilla, so I could not g_free it */
	if (value)
	{
		result = g_strdup (value);
		nsMemory::Free (value);
	}

	return result;
}

/**
 * mozilla_wrapper_init: initialize the mozilla wrapper
 * @attach_listener: attach event listener. Currently used for
 * links drag and drop only.
 */
extern "C" gpointer 
mozilla_wrapper_init (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper = new GaleonWrapper ();
	result = wrapper->Init (embed);
	if (NS_FAILED(result))
		g_warning ("Wrapper initialization failed");
	return wrapper;
}

/**
 * mozilla_wrapper_destroy: destroy the mozilla wrapper
 */
extern "C" gboolean
mozilla_wrapper_destroy (GaleonEmbed *embed)
{
	nsresult result;
	GaleonWrapper *wrapper;

	wrapper = (GaleonWrapper *)(embed->wrapper);
	result = wrapper->Destroy();
	embed->wrapper = NULL;
	delete wrapper;
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_find (GaleonEmbed *embed, 
				  WrapperSearchProperties *properties)
{
	GaleonWrapper *wrapper;
	PRUnichar *search_string;		
	PRBool didFind;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper = (GaleonWrapper *)embed->wrapper;
	search_string = mozilla_locale_to_unicode (properties->search_string);
	wrapper->Find (search_string, properties->match_case, 
		       properties->backwards, &didFind);
	g_free (search_string);

	return didFind;
}

extern "C" gboolean mozilla_save_url (GaleonEmbed *embed, const char *url, 
				      char *file_name, int action)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SaveURI (url, file_name, action);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_save_link (GaleonEmbed *embed, const char *url)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SaveLink (url);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_save_document (GaleonEmbed *embed, char *filename,
					   const char *datapath, int action,
					   gboolean mainDoc)
{
	nsresult rv;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	rv = wrapper->SaveDocument (filename, datapath, action,
				    mainDoc ? TRUE : FALSE);

	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

extern "C" gboolean mozilla_reload (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	nsresult result = wrapper->ReloadDocument ();
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_set_zoom: Sets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_set_zoom (GaleonEmbed *embed, float f)
{
	GaleonWrapper *wrapper;
	gfloat current_zoom;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (&current_zoom);
	if (NS_FAILED (result) || current_zoom == f)
		return FALSE;

	result = wrapper->SetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_zoom: Gets the zoom factor of a embed
 */
extern "C" gboolean
mozilla_get_zoom (GaleonEmbed *embed, float *f)
{
	GaleonWrapper *wrapper;

	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper = (GaleonWrapper *)embed->wrapper;

	nsresult result = wrapper->GetZoom (f);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_print: print a document
 */
extern "C" gboolean
mozilla_print (GaleonEmbed *embed, PrintInfo *info, gboolean preview)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)(embed->wrapper);

	nsCOMPtr<nsIPrintOptions> options(do_GetService(kPrintOptionsCID, &result));
	if (NS_FAILED(result)) return NS_ERROR_FAILURE;

	if (info->pages)
	{
		options->SetStartPageRange (info->from_page);
		options->SetEndPageRange (info->to_page);
	}

	options->SetMarginTop (info->top_margin);
	options->SetMarginBottom (info->bottom_margin);
	options->SetMarginLeft (info->left_margin);
	options->SetMarginRight (info->right_margin);

	nsCString printerString(info->printer);
	nsCString fileString(info->file);
	options->SetToFileName (fileString.ToNewUnicode());
	options->SetPrintCommand (printerString.ToNewUnicode());
	options->SetPrintToFile (info->print_to_file);

	options->SetPrintReversed (info->reversed);
	options->SetPaperSize (info->paper);

	options->SetPrintSilent (PR_TRUE);

	result = wrapper->Print(options, preview);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_session_history_get_url: Gets the session history and  ..
 * GtkMozWrapper. On return,  ..
 */
extern "C" char *
mozilla_session_history_get_url (GaleonEmbed *embed, int index)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	nsresult result;
	nsXPIDLCString url;

	result = wrapper->GetSHUrlAtIndex(index, getter_Copies(url));

	if (NS_FAILED(result))
	{
		return NULL;
	}
	
	return g_strdup(url.get());
}

extern "C" char *
mozilla_session_history_get_url_relative (GaleonEmbed *embed, int offset)
{
	int count, index;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 
	wrapper->GetSHInfo (&count, &index);
	return mozilla_session_history_get_url (embed, index+offset);
}

/**
 * mozilla_session_history: Gets the session history and current position of a 
 * GtkMozWrapper. On return, *titles will be an array of char * wich must be freed.
 */
extern "C" gboolean
mozilla_session_history_get_all_titles (GaleonEmbed *embed, char **titles[], 
					int *count, int *index)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	if (wrapper->GetSHInfo (count, index) == NS_ERROR_FAILURE)
		return FALSE;

	char **t = g_new(char *, *count);

	for (PRInt32 i = 0; i < *count; i++) {

		nsresult result;
		nsXPIDLString unicodeTitle;

		result = wrapper->GetSHTitleAtIndex(i,
						    getter_Copies(unicodeTitle));

		if (NS_FAILED(result))
		{
			return NS_OK;
		}
		
		t[i] = mozilla_unicode_to_locale (unicodeTitle.get());
	}
	*titles = t;

	return TRUE;
}

/**
 * mozilla_session_history_go: Goes to the SHEntry at the given index
 */
extern "C" gboolean
mozilla_session_history_go (GaleonEmbed *embed, int index)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->GoToHistoryIndex (index);
	
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_main_document_url: Get URL of main document
 * embed: Parent widget of document
 */
extern "C" gchar * 
mozilla_get_main_document_url (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (embed->wrapper != NULL, NULL); 

	nsXPIDLCString url;
	nsresult rv = wrapper->GetMainDocumentUrl (getter_Copies(url));

	return NS_SUCCEEDED (rv) ? g_strdup(url.get()) : NULL;
}

/**
 * mozilla_get_document_url: Get URL of focused document (frame)
 * embed: Parent widget of document
 */
extern "C" gchar * 
mozilla_get_document_url (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (embed->wrapper != NULL, NULL); 

	nsXPIDLCString url;
	nsresult rv = wrapper->GetDocumentUrl (getter_Copies(url));

	return NS_SUCCEEDED (rv) ? g_strdup(url.get()) : NULL;
}

/**
 * mozilla_get_document_title: get the document title in a locale specific
 * NULL terminated C string, using the private UniCode interface.
 */
extern "C" gchar * 
mozilla_get_document_title (GaleonEmbed *embed, gchar **title_utf8)
{
	PRUnichar *unicode_title;
	gchar *title;

	/* get the title in unicode */
	unicode_title = gtk_moz_embed_get_title_unichar 
		(GTK_MOZ_EMBED(embed->mozEmbed));

	/* do get utf8 version too */
	if (title_utf8 != NULL)
	{
		*title_utf8 = mozilla_unicode_to_utf8 (unicode_title);
	}

	/* attempt conversion */
	title = mozilla_unicode_to_locale (unicode_title);

	/* free unicode version */
	g_free (unicode_title);

	/* return it */
	return title;
}

/**
 * mozilla_get_key_event_info:
 */
extern "C" gboolean
mozilla_get_key_event_info (GaleonEmbed *embed, gpointer event, WrapperKeyEventInfo *info)
{
	nsresult result;
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	result = wrapper->GetKeyEventContext ((nsIDOMKeyEvent *)event, info);

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}

/**
 * 
 */
extern "C" gboolean
mozilla_get_mouse_event_info (GaleonEmbed *embed, gpointer event, 
			      WrapperMouseEventInfo *context)
{
	nsresult result;
	return_val_if_not_embed (embed, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (wrapper, FALSE);
	g_return_val_if_fail (event, FALSE);
	
	result = wrapper->GetMouseEventContext ((nsIDOMMouseEvent*)event, 
						context);

	/* for some reason we get different numbers on PPC, this fixes
	 * that up... -- MattA */
	if (context->button == 65536)
	{
		context->button = 1;
	}
	else if (context->button == 131072)
	{
		context->button = 2;
	}

	return NS_SUCCEEDED(result) ? TRUE : FALSE;
}


/**
 * mozilla_free_context_info_sub: free info contained in WrapperContextInfo 
 * struct.
 */
extern "C" void
mozilla_free_context_info_sub (WrapperContextInfo *ctx)
{
	g_free(ctx->linktext);

	/* created by mozilla: */
	if (ctx->link) nsMemory::Free (ctx->link);
	if (ctx->img) nsMemory::Free (ctx->img);
}

/**
 * mozilla_list_cookies: get a list of all saved cookies
 */
extern "C" GList *
mozilla_list_cookies (void)
{
	GList *cookies = NULL;
	nsresult result;

	nsCOMPtr<nsICookieManager> 
			cookieManager = do_CreateInstance (COOKIEMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> cookieEnumerator;
	result = 
	    cookieManager->GetEnumerator (getter_AddRefs(cookieEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	
	PRBool enumResult;
	for (cookieEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     cookieEnumerator->HasMoreElements(&enumResult))
	{
		Cookie *c;
	
		nsCOMPtr<nsICookie> nsCookie;
		result = cookieEnumerator->GetNext (getter_AddRefs(nsCookie));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		c = g_new0 (Cookie, 1);

		nsXPIDLCString transfer;

		nsCookie->GetHost (getter_Copies(transfer));
		c->base.domain = g_strdup (transfer.get());
		nsCookie->GetName (getter_Copies(transfer));
		c->name = g_strdup (transfer.get());
		nsCookie->GetValue (getter_Copies(transfer));
		c->value = g_strdup (transfer.get());
		nsCookie->GetPath (getter_Copies(transfer));
		c->path = g_strdup (transfer.get());
		
		PRBool isSecure;
		nsCookie->GetIsSecure (&isSecure);
		if (isSecure == PR_TRUE) 
			c->secure = g_strdup_printf ("Yes");
		else 
			c->secure = g_strdup_printf ("No");

		PRUint64 dateTime;
		nsCookie->GetExpires (&dateTime);
		c->expire = g_strdup_printf ("%s",ctime((time_t*)&dateTime));
		
		cookies = g_list_prepend (cookies, c);
	}	
	cookies = g_list_reverse (cookies);
	return cookies;
}

/**
 * mozilla_get_permissions:
 * @type: Type of permissions to get ( cookie or image )
 */

extern "C" GList *
mozilla_get_permissions (int type)
{
	GList *permissions = NULL;
	nsresult result;
	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance (PERMISSIONMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> permissionEnumerator;
	result = permissionManager->
			GetEnumerator (getter_AddRefs(permissionEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);
	PRBool enumResult;
	for (permissionEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     permissionEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPermission> nsPermission;
		result = permissionEnumerator->
				GetNext (getter_AddRefs(nsPermission));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);		

		PRInt32 cType;
		nsPermission->GetType (&cType);
		if (cType == type)
		{
			CookieBase *b = g_new0 (CookieBase, 1);

			nsPermission->GetHost (&(b->domain));

			PRBool cap;
			nsPermission->GetCapability (&cap);
			if (cap == PR_TRUE) 
				b->type = g_strdup_printf (_("Allowed"));
			else 
				b->type = g_strdup_printf (_("Blocked"));

			permissions = g_list_prepend (permissions, b);
		}
	}

	permissions = g_list_reverse (permissions);
	return permissions;
};

/**
 * mozilla_block_url:
 * @imgURL: url of site to block
 */
extern "C" void
mozilla_block_url (const char *imgURL)
{
	nsresult result;
	
	nsCOMPtr<nsIImgManager> 
		imageManager = do_CreateInstance (IMAGEMANAGER_ID);
	
	result = imageManager->Block (imgURL);

	g_return_if_fail (NS_SUCCEEDED(result));
}

/**
 * mozilla_set_permission:
 * @permit: whether to block or allow
 * @type: type of permission to set ( cookie or image )
 */
 
extern "C" void
mozilla_set_permission (GaleonEmbed *embed, gboolean permit, int type)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	if (!wrapper) return;

	wrapper->SetSitePermission (permit ? PR_TRUE : PR_FALSE, type);
}

/**
 * mozilla_remove_cookies:
 * @gone: list of cookies which are to be removed
 * @block: whether to block the originating servers in future
 */
extern "C" gboolean 
mozilla_remove_cookies (GList *gone)
{
	nsresult result;
	nsCOMPtr<nsICookieManager> 
		cookieManager = do_CreateInstance (COOKIEMANAGER_ID);

	for (GList *cookies = g_list_first(gone) ; cookies!=NULL ; 
	     cookies = g_list_next(cookies))
	{
		Cookie *c = (Cookie *)cookies->data;

		result = cookieManager->Remove (c->base.domain,c->name,
					 	c->path,
						c->base.block ? PR_TRUE : PR_FALSE);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_remove_permissions:
 * @gone: list of cookie or image permissions which are to be removed
 * @type: type of permissions ( cookies or images )
 */
extern "C" gboolean 
mozilla_remove_permissions (GList *gone, int type)
{
	nsresult result;
	nsCOMPtr<nsIPermissionManager> 
		permissionManager = do_CreateInstance( PERMISSIONMANAGER_ID );

	for (GList *permissions = g_list_first(gone) ; permissions!=NULL;
	     permissions = g_list_next(permissions))
	{
		CookieBase *b = (CookieBase *)permissions->data;
		result = permissionManager->Remove(b->domain,type);
		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_copy_session_history: copy the session history to another embed
 */
extern "C" gboolean
mozilla_copy_session_history (GaleonEmbed *embed, GaleonEmbed *dest)
{
	g_return_val_if_fail (embed != NULL, FALSE);
	g_return_val_if_fail (dest != NULL, FALSE);
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	GaleonWrapper *dest_wrapper = (GaleonWrapper *)dest->wrapper;
	g_return_val_if_fail (wrapper != NULL, FALSE);
	g_return_val_if_fail (dest_wrapper != NULL, FALSE);
	nsresult result = wrapper->CopyHistoryTo (dest_wrapper);
	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

/**
 * mozilla_get_charsets: get a list of charset supported by mozilla
 */
extern "C" void
mozilla_get_charsets (GHashTable **charsets, GList **sorted_charset_titles)
{
	nsresult rv;
	PRUint32 cscount;
	PRUint32 translated_cscount = get_translated_cscount ();
	char *charset_str, *charset_title_str;

	nsCOMPtr<nsIAtom> docCharsetAtom;
	nsCOMPtr<nsICharsetConverterManager2> ccm2 =
		do_GetService (NS_CHARSETCONVERTERMANAGER_CONTRACTID, &rv);
	if (!NS_SUCCEEDED(rv)) return;

	nsCOMPtr <nsISupportsArray> cs_list;
	rv = ccm2->GetDecoderList (getter_AddRefs(cs_list));
	if (!NS_SUCCEEDED(rv)) return;

	rv = cs_list->Count(&cscount);
	*charsets = g_hash_table_new (g_str_hash, g_str_equal);
	for (PRUint32 i = 0; i < cscount; i++)
	{
		nsCOMPtr<nsISupports> cssupports =
					(dont_AddRef)(cs_list->ElementAt(i));
		nsCOMPtr<nsIAtom> csatom ( do_QueryInterface(cssupports) );
		nsString charset_ns = nsnull, charset_title_ns = nsnull;

		/* charset name */
		rv = csatom->ToString(charset_ns);
		charset_str = convert_ns_string_to_c_string (charset_ns);
		if (charset_str == NULL || strlen (charset_str) == 0)
		{
			continue;
		}

		/* charset readable title */
		rv = ccm2->GetCharsetTitle2(csatom, &charset_title_ns);
		charset_title_str = convert_ns_string_to_c_string 
							(charset_title_ns);
		if (charset_title_str == NULL || 
		    strlen (charset_title_str) == 0)
		{
			g_free (charset_title_str);
			charset_title_str = g_strdup (charset_str);
		}
		
		for (PRUint32 j = 0; j < translated_cscount; j++)
		{
			if (g_strcasecmp (
				charset_str, 
				charset_trans_array[j].charset_name) == 0)
			{
				g_free (charset_title_str);
				charset_title_str = (char *) 
					_(charset_trans_array[j].charset_title);
				break;
			}
		}

		/* fill the hash and the sorted list */
		g_hash_table_insert (*charsets, charset_title_str, charset_str);
		*sorted_charset_titles = g_list_insert_sorted (
						(GList*)*sorted_charset_titles,
						(gpointer)charset_title_str,
						(GCompareFunc)g_strcasecmp); 
	}
}

/**
 * mozilla_force_character_set: force the embed to use the specified
 * character set
 */
extern "C" void
mozilla_force_character_set (GaleonEmbed *embed, char *force_character_set)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	wrapper->ForceCharacterSet (force_character_set);
}

/**
 * mozilla_clear_cache: Clear the global memory/disk cache
 */
extern "C" void
mozilla_clear_cache (unsigned long cache)
{
	nsCOMPtr<nsICacheService> 
		CacheService = do_CreateInstance (CACHESERVICE_ID);
	CacheService->EvictEntries(cache);
}

/* This nonsense is needed to get the allocators right */
static char *
convert_ns_string_to_c_string (const nsString & ns_string)
{
	char *c_string;
	nsCString ns_c_string;

	ns_c_string.AssignWithConversion (ns_string);

	c_string = g_strdup (ns_c_string.get());

	return c_string;
}

extern "C" gboolean
mozilla_get_favicon_location (GaleonEmbed *embed, gchar **url)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	nsXPIDLCString transfer;
	nsresult rv = wrapper->GetFaviconURL (getter_Copies(transfer));
	*url = g_strdup (transfer.get());
	
	return NS_SUCCEEDED (rv) ? TRUE : FALSE;
}

/**
 * mozilla_list_passwords: get a list of all saved username/password pairs
 */
extern "C" GList *
mozilla_list_passwords (PasswordType type)
{
	GList *passwords = NULL;
	nsresult result = NS_ERROR_FAILURE;

	nsCOMPtr<nsIPasswordManager> 
		passwordManager = do_CreateInstance (PASSWORDMANAGER_ID);
	nsCOMPtr<nsISimpleEnumerator> passwordEnumerator;
	if (type == PASSWORD)
		result = passwordManager->GetEnumerator 
				(getter_AddRefs(passwordEnumerator));
	else if (type == REJECT)
		result = passwordManager->GetRejectEnumerator 
				(getter_AddRefs(passwordEnumerator));
	g_return_val_if_fail (NS_SUCCEEDED(result), NULL);	

	PRBool enumResult;
	for (passwordEnumerator->HasMoreElements(&enumResult) ;
	     enumResult == PR_TRUE ;
	     passwordEnumerator->HasMoreElements(&enumResult))
	{
		nsCOMPtr<nsIPassword> nsPassword;
		result = passwordEnumerator->GetNext 
					(getter_AddRefs(nsPassword));
		g_return_val_if_fail (NS_SUCCEEDED(result), NULL);

		Password *p = g_new0 (Password, 1);

		nsXPIDLCString transfer;
		nsPassword->GetHost (getter_Copies(transfer));
		p->host = g_strdup (transfer.get());

		if (type==PASSWORD)
		{
			nsXPIDLString unicodeName;
			nsPassword->GetUser (getter_Copies(unicodeName));
			p->username = mozilla_unicode_to_locale (unicodeName.get());
		}
		
		passwords = g_list_prepend (passwords, p);
	}	
	passwords = g_list_reverse (passwords);
	return passwords;
}

/**
 * mozilla_remove_passwords:
 * @gone: list of passwords/rejcts which are to be removed
 * @type: whether list is passwords or rejects
 */
extern "C" gboolean 
mozilla_remove_passwords (GList *gone, PasswordType type)
{
	nsresult result = NS_ERROR_FAILURE;
	nsCOMPtr<nsIPasswordManager> 
		passwordManager = do_CreateInstance (PASSWORDMANAGER_ID);

	for (GList *passwords = g_list_first(gone) ; passwords!=NULL ; 
	     passwords = g_list_next(passwords))
	{
		Password *p = (Password *)passwords->data;
		if (type == PASSWORD)
		{
			PRUnichar *unicodeName = mozilla_locale_to_unicode
							(p->username);
			result = passwordManager->RemoveUser (p->host,
							      unicodeName);
			g_free (unicodeName);
		}
		else if (type == REJECT)
		{
			result = passwordManager->RemoveReject (p->host);
		};

		if (NS_FAILED(result)) return FALSE;
	};
	return TRUE;
}

/**
 * mozilla_unicode_to_utf8: Encodes unicode string to UTF-8
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_utf8 (const PRUnichar *uniStr)
{
	PRInt32 sSize,dSize;
	nsresult result;

	const nsString str (uniStr);

	sSize = str.Length ();

	nsCOMPtr<nsIUnicodeEncoder> 
		unicodeEncoder = do_CreateInstance (UTF8_ENCODER_ID);

	/* GetMaxLength returns a worst case prediction for the size
	   of the returned char*. Using it ensures that Convert will
	   not run out of space */
	result = unicodeEncoder->GetMaxLength (uniStr, sSize, &dSize);
	gchar *utfStr = g_new0 (gchar, dSize + 1);

	/* Convert must be passed the size of unicode string and
	   the size of the char* buffer so that bad things(tm)
	   won't happen. No null termination here. */
	result = unicodeEncoder->Convert (uniStr, &sSize, utfStr, &dSize);

	/* Finish ensures that the encoder is left in a clean state
	   for it's next use. */
	result = unicodeEncoder->Finish (utfStr, &dSize);

	/* Normally a null would need to appended at this point but
	   cStr was initialised to zero, so there's no need */
	return utfStr;
}

/**
 * mozilla_unicode_to_locale: Encodes unicode string to something
 * valid for the current locale (which can then be used in GTK+ labels).
 * @uniStr: The unicode string to encode
 */
extern "C" gchar *
mozilla_unicode_to_locale (const PRUnichar *uniStr)
{
	PRInt32 sSize;
	wchar_t *wide;
	gchar *output;
	gint i, count;

	/* sanity */
	if (uniStr == NULL)
	{
		return NULL;
	}

	const nsString str (uniStr);
	sSize = str.Length ();

	/* allocate a wide string big enough to hold the unicode string,
	 * this is necessary since wchar_t is 32-bits with glibc */
	wide = g_new0 (wchar_t, sSize + 1);
	for (i = 0; i < sSize + 1; i++)
	{
		wide[i] = uniStr[i];
	}

	/* use glibc function to determine the size of this string once
	 * encoded to a locale specfic multibyte string */
	count = wcstombs (NULL, wide, 0);

	/* check for success */
	if (count == -1)
	{
		/* let Mozilla do a (lossy) conversion then */
		nsCString str;
		str.AssignWithConversion(uniStr);
		g_free (wide);
		return g_strdup (str.get()); /* FIXME strdup needed? */
					     /* Phil: Oh Yes! indeed. */
	}

	/* allocate a string big enough and do the actual conversion */
	output = g_new0 (gchar, count + 1);
	count = wcstombs (output, wide, count + 1);
	g_assert (count != -1);

	/* free wide version and return */
	g_free (wide);
	return output;
}

/**
 * mozilla_utf8_to_unicode: Decodes UTF-8 string to unicode
 * @utfStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_utf8_to_unicode (const gchar *utfStr)
{
	PRInt32 sSize,dSize;
	nsresult result;
	
	for (sSize = 0; utfStr[sSize] != 0; sSize++);

	nsCOMPtr<nsIUnicodeDecoder> unicodeDecoder = 
					do_CreateInstance (UTF8_DECODER_ID);

	/* Unicode decoding is much the same as encoding as
	   described in mozilla_session_history comments, 
	   but there is no Finish function needed to complete
	   the process */
	result = unicodeDecoder->GetMaxLength (utfStr, sSize, &dSize);
	PRUnichar *uniStr = g_new0 (PRUnichar, dSize + 1);

	result = unicodeDecoder->Convert (utfStr, &sSize, uniStr, &dSize);

	return uniStr;
}

/**
 * mozilla_locale_to_unicode: Decodes a string encoded for the current
 * locale into unicode. Used for getting text entered in a GTK+ entry
 * into a form which mozilla can use.
 * @locStr: The unicode string to encode
 */
extern "C" PRUnichar *
mozilla_locale_to_unicode (const gchar *locStr)
{
	PRUnichar *uniStr;
	wchar_t *wide;
	gint i, count;

	/* sanity */
	if (locStr == NULL)
	{
		return NULL;
	}

	/* count the number of wide characters which will be produced */
	count = mbstowcs (NULL, locStr, 0);
	if (count == -1)
	{
		/* hmm, shouldnt happen but fallback to utf8 */
		g_warning ("unusual locale string: [%s]\n", locStr);
		return mozilla_utf8_to_unicode (locStr);
	}

	/* allocate and decode */
	wide = g_new0 (wchar_t, count + 1);
	mbstowcs (wide, locStr, count + 1);

	/* make a unicode string and copy into it */
	uniStr = g_new0 (PRUnichar, count + 1);
	for (i = 0; i < count + 1; i++)
	{
		uniStr[i] = wide[i];
	}

	/* free wide string and return the unicode one */
	g_free (wide);
	return uniStr;
}

/**
 * mozilla_find_gtk_parent: retrives gtkwidget pointer to parent
 * of domWindow. It will be the gtkmozembed that holds it.
 * @domWindow: nsIDOMWindow to find gtk parent of.
 */
extern "C" GtkWidget *
mozilla_find_gtk_parent (nsIDOMWindow *domWindow)
{
	nsresult result;
	
        nsCOMPtr<nsIWindowWatcher> wwatch 
		(do_GetService("@mozilla.org/embedcomp/window-watcher;1"));
        if (!wwatch) return nsnull;

        nsCOMPtr<nsIWebBrowserChrome> windowChrome;
        result = wwatch->GetChromeForWindow (domWindow,
        				     getter_AddRefs(windowChrome));
        if (NS_FAILED(result)) return nsnull;

        nsCOMPtr<nsIEmbeddingSiteWindow> window
        		(do_QueryInterface(windowChrome, &result));
        if (NS_FAILED(result)) return nsnull;

	GtkWidget *mozembed;
	result = window->GetSiteWindow ((void **)&mozembed);
	if (NS_FAILED(result)) return nsnull;

	GaleonEmbed *embed = (GaleonEmbed*)
		gtk_object_get_data (GTK_OBJECT (mozembed), "GaleonEmbed");
	g_assert (embed != NULL);
	g_assert (embed->magic == GALEON_EMBED_MAGIC);

	return embed->parent_window->WMain;
}

extern "C" gboolean
mozilla_set_offline (GaleonEmbed *embed, gboolean offline)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	nsresult result;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	result = wrapper->SetOffline (offline);

	return NS_SUCCEEDED (result) ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_can_cut_selection (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	gboolean result;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->CanCutSelection (&result);

	return result;
}

extern "C" gboolean
mozilla_can_copy_selection (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	gboolean result;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->CanCopySelection (&result);

	return result;
}

extern "C" gboolean
mozilla_can_paste (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;
	gboolean result;

	g_return_val_if_fail (embed->wrapper != NULL, FALSE); 

	wrapper->CanPaste (&result);

	return result;
}

extern "C" void
mozilla_cut_selection (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 

	wrapper->CutSelection ();
}

extern "C" void
mozilla_copy_selection (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 

	wrapper->CopySelection ();
}

extern "C" void
mozilla_activate (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 

	wrapper->Activate ();
}

extern "C" void
mozilla_deactivate (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 

	wrapper->Deactivate ();
}

extern "C" void
mozilla_paste (GaleonEmbed *embed)
{
	GaleonWrapper *wrapper = (GaleonWrapper *)embed->wrapper;

	g_return_if_fail (embed->wrapper != NULL); 

	wrapper->Paste ();
}

extern "C" void
mozilla_delete_temp_file_on_exit (const char *filename)
{
#if MOZILLA_VERSION > VERSION3(0,9,2)
	nsresult rv;
	nsCOMPtr<nsILocalFile> localFile;
	rv = NS_NewLocalFile (filename, PR_TRUE, getter_AddRefs(localFile));
	if (NS_FAILED(rv)) return;

	nsCOMPtr<nsIExternalHelperAppService> helperService =
			do_GetService (NS_EXTERNALHELPERAPPSERVICE_CONTRACTID);

	nsCOMPtr<nsPIExternalAppLauncher> appLauncher =
					do_QueryInterface (helperService, &rv);
	if (NS_SUCCEEDED(rv))
	{
		appLauncher->DeleteTemporaryFileOnExit (localFile);
	}
#endif
}

extern "C" gboolean
mozilla_show_java_console (void)
{
	nsresult rv;
	nsCOMPtr<nsIJVMManager> JVMManager = do_GetService (kJVMManagerCID,
							    &rv);
	if (NS_FAILED(rv)) return FALSE;

	rv = JVMManager->ShowJavaConsole ();

	return rv ? TRUE : FALSE;
}

extern "C" gboolean
mozilla_show_javascript_console (void)
{
	nsresult rv;
	nsCOMPtr<nsIJSConsoleService> JSConsoleService =
		do_GetService ("@mozilla.org/embedcomp/jsconsole-service;1",
			       &rv);
	if (NS_FAILED(rv)) return FALSE;

	rv = JSConsoleService->Open (nsnull);

	return rv ? TRUE : FALSE;
}
