/*
 *  Original Cookie Manager code
 *  Copyright (C) 2001 Daniel Stodden <stodden@in.tum.de>
 *  
 *  Updates for new nsICookieManager interface
 *  Password Handling
 *  General Cleanup
 *  Copyright (C) 2001 Philip Langdale <philipl@mail.utexas.edu>
 *
 *  GTK+based persistent data manager ( cookies, passwords, image blocks )
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "misc.h"
#include "mozilla.h"
#include "glade.h"
#include "dialog.h"

#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkentry.h>
#include <gdk/gdkkeysyms.h>
#include <libgnomeui/gnome-propertybox.h>
#include <libgnome/gnome-i18n.h>

static GtkWidget *dialog;
static GtkWidget *stored_name;
static GtkWidget *stored_value; 
static GtkWidget *stored_host;
static GtkWidget *stored_path;
static GtkWidget *stored_secure;
static GtkWidget *stored_expire;
static GtkWidget *stored_block;

#define foreach_list_elem(elem, list) \
	for (elem = list; elem != NULL; elem = g_list_next(elem))

typedef enum {
	PDM_COOKIE,
	PDM_COOKIEPERM,
	PDM_IMAGEPERM,
	PDM_PASSWORD,
	PDM_REJECT
} PDMViewType;

typedef struct {
	GList		*keep, *remove;
	GtkWidget	*clist;
	GtkWidget	*remove_button, *remove_all_button;
	gint		last_sort_column;
	GtkSortType	last_sort_type;
	PDMViewType     type;
} PDMView;

PDMView *stored, *permissions, *images;
PDMView *passwords, *rejects;

/* local function prototypes */
static void cookies_clear_stored_fields (void);
static void cleanup_cookie_struct (gpointer element, gpointer v);

static void
view_select_cb (GtkCList *clist, gint row, gint column,
		GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	CookieBase *c;

	if (entry)
	{
		c = (CookieBase*)entry->data;
		c->remove = TRUE;
	}
}

static void
view_unselect_cb (GtkCList *clist, gint row, gint column,
		  GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	CookieBase *c = (CookieBase*)entry->data;
	c->block = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(stored_block));
	c->remove = FALSE;
}

static void
view_remove_cb (GtkWidget *widget, gpointer data)
{
	PDMView *view = data;
	GList *entry, *next, *remove = NULL;
	gboolean block = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(stored_block));

	for (entry = view->keep; entry != NULL; entry = next)
	{
		CookieBase *b = (CookieBase*)entry->data;
		gint rownum;

		next = g_list_next (entry);

		if (b->remove == FALSE) continue;
		
		// This will be used by the cookie remover to know if it should be 
		// completely blocked next time.
		b->block = block;

		rownum = gtk_clist_find_row_from_data (GTK_CLIST(view->clist),
						       entry);

		remove = g_list_append (remove, GINT_TO_POINTER(rownum));

		view->keep = g_list_remove_link (view->keep, entry);
		view->remove = g_list_concat (view->remove, entry);
		
		gnome_property_box_changed (GNOME_PROPERTY_BOX(dialog));
	}

	cookies_clear_stored_fields ();

	foreach_list_elem (entry, remove)
		gtk_clist_remove (GTK_CLIST(view->clist),
				  GPOINTER_TO_INT(entry->data));
	g_list_free (remove);
}

static void
view_remove_all_cb (GtkWidget *widget, gpointer data)
{
	PDMView *view = data;

	view->remove = g_list_concat (view->remove, view->keep);
	view->keep = NULL;

	gtk_clist_clear (GTK_CLIST(view->clist));

	gnome_property_box_changed (GNOME_PROPERTY_BOX(dialog));

	cookies_clear_stored_fields ();
}

static void
cookies_clear_stored_fields (void)
{
	gtk_editable_delete_text (GTK_EDITABLE (stored_name),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_value),  0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_host),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_path),   0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_secure), 0, -1);
	gtk_editable_delete_text (GTK_EDITABLE (stored_expire), 0, -1);
}

static gboolean
view_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	if (event->state == 0)
 	{
 		if (event->keyval == GDK_Delete ||
 		    event->keyval == GDK_KP_Delete)
 			view_remove_cb (NULL, data);
 	}
	else if ((event->state & GDK_LOCK_MASK ||
		  event->state & GDK_MOD2_MASK ||
		  event->state & GDK_MOD5_MASK) &&
		 !(event->state & GDK_SHIFT_MASK ||
		   event->state & GDK_MOD1_MASK  ||
		   event->state & GDK_CONTROL_MASK))
 	{
 		if (event->keyval == GDK_Delete)
 			view_remove_cb (NULL, data);
 	}
	
 	return FALSE;
}

static void view_sort_cb (GtkCList *clist, gint column, PDMView *view)
{
	if (column == view->last_sort_column)
		view->last_sort_type = !view->last_sort_type;
	else
		view->last_sort_type = GTK_SORT_ASCENDING;
	gtk_clist_set_sort_type (clist, view->last_sort_type);
	view->last_sort_column = column;
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);
}

static void
cookie_view_init (PDMView *view)
{
	gtk_signal_connect (GTK_OBJECT(view->clist), "select-row",
			    view_select_cb, view);
	gtk_signal_connect (GTK_OBJECT(view->clist), "unselect-row",
			    view_unselect_cb, view);
	gtk_signal_connect (GTK_OBJECT(view->remove_button), "clicked",
			    GTK_SIGNAL_FUNC(view_remove_cb), view);
	gtk_signal_connect (GTK_OBJECT(view->remove_all_button), "clicked",
			    GTK_SIGNAL_FUNC(view_remove_all_cb), view);
	gtk_signal_connect (GTK_OBJECT(view->clist), "key_press_event",
			    GTK_SIGNAL_FUNC(view_key_press_cb), view);
	gtk_signal_connect (GTK_OBJECT(view->clist), "click_column",
			    GTK_SIGNAL_FUNC(view_sort_cb), view);
}

static void
stored_select_cb (GtkCList *clist, gint row, gint column,
		  GdkEventButton *event, gpointer data)
{
	GList *entry = gtk_clist_get_row_data (clist, row);
	Cookie *c; 

	if (entry)
	{
		c = (Cookie*)entry->data;

		gtk_entry_set_text (GTK_ENTRY(stored_name), c->name);
		gtk_entry_set_text (GTK_ENTRY(stored_value), c->value);
		gtk_entry_set_text (GTK_ENTRY(stored_host), c->base.domain);
		gtk_entry_set_text (GTK_ENTRY(stored_path), c->path);
		gtk_entry_set_text (GTK_ENTRY(stored_secure), c->secure);
		gtk_entry_set_text (GTK_ENTRY(stored_expire), c->expire);
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(stored_block),
					      c->base.block);
	}
}

static void
init_cookie_clist (PDMView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	/* 
	   set selection mode to single, to prevent
	   select/unselect-row emission without valid row data
	*/
	gtk_clist_set_selection_mode (GTK_CLIST(view->clist), 
				      GTK_SELECTION_SINGLE);

	foreach_list_elem (entry, view->keep) {
		Cookie *c = (Cookie*)entry->data;
		gint rownum;
		gchar *row[2];

		c->base.remove = FALSE;

		row[0] = c->base.domain;
		row[1] = c->name;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_set_selection_mode(GTK_CLIST(view->clist), 
				     GTK_SELECTION_BROWSE);

	gtk_clist_thaw (GTK_CLIST(view->clist));
}
		
static void
init_server_clist (PDMView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		CookieBase *b = (CookieBase*)entry->data;
		gint rownum;
		gchar *row[2];
		gchar *status;

		b->remove = FALSE;
		
		row[0] = b->domain;
		status = b->type;

		row[1] = status;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}

static void
init_password_clist (PDMView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		Password *p = (Password*)entry->data;
		gint rownum;
		gchar *row[2];

		p->remove = FALSE;
		
		row[0] = p->host;
		row[1] = p->username;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}

static void
init_reject_clist (PDMView *view, GList *list)
{
	GList *entry;

	gtk_clist_freeze (GTK_CLIST(view->clist));
	gtk_clist_clear (GTK_CLIST(view->clist));

	view->keep = list;

	foreach_list_elem (entry, list) {
		Password *p = (Password*)entry->data;
		gint rownum;
		gchar *row[1];

		p->remove = FALSE;
		
		row[0] = p->host;

		rownum = gtk_clist_append (GTK_CLIST(view->clist), row);
		gtk_clist_set_row_data (GTK_CLIST(view->clist), 
					rownum, entry);
	}

	gtk_clist_thaw (GTK_CLIST(view->clist));
}

static void 
init_all_clists (void)
{
	init_cookie_clist (stored, mozilla_list_cookies());
	init_server_clist (permissions,
			   mozilla_get_permissions(COOKIEPERMISSION));
	init_server_clist (images, mozilla_get_permissions(IMAGEPERMISSION));
	init_password_clist (passwords, mozilla_list_passwords(PASSWORD));
	init_reject_clist (rejects, mozilla_list_passwords(REJECT));
}

static void
pdm_view_cleanup (PDMView *view)
{
	if (view->keep) {
		g_list_foreach (view->keep, cleanup_cookie_struct, 
				(gpointer) view);
		g_list_free (view->keep);
	}

	if (view->remove) {
		g_list_foreach (view->remove, cleanup_cookie_struct, 
				(gpointer) view);
		g_list_free (view->remove);
	}

	g_free (view);
}
		
static void
free_cookiebase_data (CookieBase *cb)
{
	if (cb)
	{
		if (cb->type) g_free (cb->type);
		if (cb->domain) g_free (cb->domain);
	}
}

static void
free_cookie_data (Cookie *c)
{
	if (c)
	{
		free_cookiebase_data (&c->base);
		if (c->name) g_free (c->name);
	        if (c->value) g_free (c->value);
		if (c->path) g_free (c->path);
		if (c->secure) g_free (c->secure);
		if (c->expire) g_free (c->expire);
	}
}

static void
free_password_data (Password *pw)
{
	if (pw)
	{
		if (pw->host) g_free (pw->host);
		if (pw->username) g_free (pw->username);
	}
}

static void
cleanup_cookie_struct (gpointer element, gpointer v)
{
	PDMView *view = (PDMView *) v;
	Cookie *c = (Cookie *) element;
	CookieBase *cb = (CookieBase *) element;
	Password *pw = (Password *) element;
	
	switch (view->type)
	{
	case PDM_COOKIE:
		free_cookie_data (c);
		g_free (c);
		break;
	case PDM_COOKIEPERM:
	case PDM_IMAGEPERM:
		free_cookiebase_data (cb);
		g_free (cb);
		break;
	case PDM_PASSWORD:
	case PDM_REJECT:
		free_password_data (pw);
		g_free (pw);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}		
		
static void
dialog_destroy_cb (GtkWidget *widget, gpointer data)
{
	pdm_view_cleanup (stored);
	pdm_view_cleanup (permissions);
	pdm_view_cleanup (images);
	pdm_view_cleanup (passwords);
	pdm_view_cleanup (rejects);
	dialog = NULL;
}



static void
dialog_apply_cb (GtkWidget *widget, gint pagenum)
{
	if (pagenum != -1) return;

	mozilla_remove_cookies (stored->remove);
	mozilla_remove_permissions (permissions->remove, COOKIEPERMISSION);
	mozilla_remove_permissions (images->remove,IMAGEPERMISSION);
	mozilla_remove_passwords (passwords->remove, PASSWORD);
	mozilla_remove_passwords (rejects->remove, REJECT);
	passwords->remove = rejects-> remove = NULL;	
	stored->remove = permissions->remove = images->remove = NULL;

	init_all_clists ();
}

static void
dialog_help_cb (GtkWidget *widget, gpointer data)
{
}

GtkWidget *
pdm_show_manager (GaleonWindow *window)
{
	GladeXML *gxml;

	if (dialog) return dialog;

	gxml = glade_widget_new ("galeon.glade", "pdm_dialog",
				 &dialog, NULL);

	/*
	 * init stored cookies list
	 */
	stored = g_new0 (PDMView, 1);
	stored->clist = glade_lookup_widget (dialog, "stored_clist");
	stored->remove_button = glade_lookup_widget (dialog, "stored_remove");
	stored->remove_all_button = glade_lookup_widget (dialog, 
							 "stored_remove_all");
	stored->type = PDM_COOKIE;
	gtk_signal_connect (GTK_OBJECT(stored->clist), "select-row",
			    stored_select_cb, stored);

	stored_name = glade_lookup_widget (dialog, "stored_name");
	stored_value = glade_lookup_widget (dialog, "stored_value");
	stored_host = glade_lookup_widget (dialog, "stored_host");
	stored_path = glade_lookup_widget (dialog, "stored_path");
	stored_secure = glade_lookup_widget (dialog, "stored_secure");
	stored_expire = glade_lookup_widget (dialog, "stored_expire");
	stored_block = glade_lookup_widget (dialog, "stored_block");

	cookie_view_init (stored);

	/*
	 * init permissions list
	 */
	permissions = g_new0 (PDMView, 1);
	permissions->clist = glade_lookup_widget (dialog, "permissions_clist");
	permissions->remove_button = 
			glade_lookup_widget (dialog, "permissions_remove");
	permissions->remove_all_button = 
			glade_lookup_widget (dialog, "permissions_remove_all");
	permissions->type = PDM_COOKIEPERM;

	cookie_view_init (permissions);

	/*
	 * init images list
	 */
	images = g_new0 (PDMView, 1);
	images->clist = glade_lookup_widget (dialog, "images_clist");
	images->remove_button = glade_lookup_widget (dialog, "images_remove");
	images->remove_all_button = glade_lookup_widget (dialog, 
							 "images_remove_all");
	images->type = PDM_IMAGEPERM;
	
	cookie_view_init (images);

	/*
	 * init password list
	 */
	passwords = g_new0 (PDMView, 1);
	passwords->clist = glade_lookup_widget (dialog, "passwords_clist");
	passwords->remove_button = glade_lookup_widget (dialog, 
							"passwords_remove");
	passwords->remove_all_button = glade_lookup_widget 
					(dialog, "passwords_remove_all");
	passwords->type = PDM_PASSWORD;
					
	cookie_view_init (passwords);

	/*
	 * init rejected password sites list
	 */
	rejects = g_new0 (PDMView, 1);
	rejects->clist = glade_lookup_widget (dialog, "rejects_clist");
	rejects->remove_button = glade_lookup_widget (dialog, "rejects_remove");
	rejects->remove_all_button = glade_lookup_widget (dialog,
							  "rejects_remove_all");
	rejects->type = PDM_REJECT;
					
	cookie_view_init (rejects);

	/*
	 * connect lower buttons
	 */
	gtk_signal_connect (GTK_OBJECT(dialog), "apply",
			    dialog_apply_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(dialog), "help",
			    dialog_help_cb, NULL);
	gtk_signal_connect (GTK_OBJECT(dialog), "destroy", 
			    dialog_destroy_cb, NULL);
	
	gtk_window_set_title (GTK_WINDOW(dialog), 
			      _("Persistent Data Manager"));
	gtk_window_set_modal (GTK_WINDOW(dialog), TRUE);

	init_all_clists ();

	gtk_widget_show (dialog);
	if (window != NULL)
	{
		dialog_set_parent (dialog, window->WMain);
	}
	gtk_object_destroy (GTK_OBJECT(gxml));

	return dialog;
}

GtkWidget *
pdm_show_page (GaleonWindow *window, PDMPage page)
{
	GtkWidget *dialog, *notebook;

	dialog = pdm_show_manager (window);

	notebook = GNOME_PROPERTY_BOX (dialog)->notebook;
	gtk_notebook_set_page (GTK_NOTEBOOK(notebook), page);

	return dialog;
}
