/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "mime.h"
#include "misc.h"
#include "spinner.h"
#include "history.h"
#include "prefs.h"
#include "window.h"
#include "mozilla.h"
#include "dialog.h"

#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtknotebook.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-color-picker.h>
#include <libgnomeui/gnome-file-entry.h>

/* from nsICache.h */
enum
{
	STORE_ANYWHERE	      = 0,
	STORE_IN_MEMORY	      = 1,
	STORE_ON_DISK	      = 2,
	STORE_ON_DISK_AS_FILE = 3
};

/* local function prototypes */
void prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_cancel_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
gboolean prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd);
void prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd);
void prefs_mime_add_button_clicked_cb (GtkButton *button, 
				       PreferencesDialog *pd);
void prefs_mime_delete_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd);
void prefs_mime_edit_cb (GtkWidget *w, PreferencesDialog *pd);
void prefs_mime_list_select_row_cb (GtkCList *clist, gint row, gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd);
void prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				      GdkEventButton *event, 
				      PreferencesDialog *pd);
void prefs_homepage_my_portal_button_clicked_cb (GtkButton *button,
						 PreferencesDialog *pd);
void prefs_homepage_current_button_clicked_cb (GtkButton *button,
					       PreferencesDialog *pd);
void prefs_lang_add_button_clicked_cb (GtkButton *button, 
				       PreferencesDialog *pd);
void prefs_lang_remove_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd);
void prefs_lang_up_button_clicked_cb (GtkButton *button, 
				      PreferencesDialog *pd);
void prefs_lang_down_button_clicked_cb (GtkButton *button, 
					PreferencesDialog *pd);
void prefs_clear_disk_cache_button_clicked_cb (GtkButton *button,
					       PreferencesDialog *pd);
void prefs_clear_memory_cache_button_clicked_cb (GtkButton *button,
						 PreferencesDialog *pd);
void prefs_clear_history_button_clicked_cb (GtkButton *button,
					    PreferencesDialog *pd);
void prefs_clear_history_question_cb (gint reply, gpointer data);
void prefs_browse_clicked_cb (GnomeFileEntry *fileentry, 
			      PreferencesDialog *pd);
void prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd);
void prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
				   GdkEvent *event, PreferencesDialog *pd);
void  prefs_smart_bm_entry_width_spin_changed_cb (GtkEditable *editable,
						  PreferencesDialog *pd);
void preferences_switch_page_cb (GtkWidget *notebook, 
				 GtkNotebookPage *notebook_page,
				 guint page_num,
				 PreferencesDialog *pd);
void prefs_image_sites_button_clicked_cb (GtkWidget *button, 
					  PreferencesDialog *pd);
void prefs_cookies_button_clicked_cb (GtkWidget *button, 
				      PreferencesDialog *pd);
void prefs_cookie_sites_button_clicked_cb (GtkWidget *button, 
					   PreferencesDialog *pd);
void prefs_passwords_button_clicked_cb (GtkWidget *button, 
					PreferencesDialog *pd);
void prefs_password_sites_button_clicked_cb (GtkWidget *button, 
					     PreferencesDialog *pd);
void prefs_download_use_command_toggled_cb (GtkWidget *button, 
					    PreferencesDialog *pd);
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd);
void prefs_mime_save_to_disk_toggled_cb (GtkWidget *button, PreferencesDialog *pd); 
void prefs_theme_select_row_cb (GtkCList *list, gint row, gint column,
				GdkEvent *event, PreferencesDialog *pd);

/**
 * prefs_mime_list_sort_cb:
 */
void prefs_mime_list_sort_cb (GtkCList *clist, gint column,
			      PreferencesDialog *pd)
{
	if (column == pd->mime_last_sort_column)
		pd->mime_last_sort_type = !pd->mime_last_sort_type;
	else
		pd->mime_last_sort_type = GTK_SORT_ASCENDING;
	gtk_clist_set_sort_type (clist, pd->mime_last_sort_type);
	pd->mime_last_sort_column = column;
	gtk_clist_set_sort_column (clist, column);
	gtk_clist_sort (clist);
}

/**
 * prefs_ok_button_clicked_cb: preferences dialog ok button clicked
 */
void
prefs_ok_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	preferences_apply (pd);
	preferences_close (pd);
}

/**
 * prefs_apply_button_clicked_cb: preferences dialog apply button clicked
 */
void
prefs_apply_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	preferences_apply (pd);
}

/**
 * prefs_cancel_button_clicked_cb: preferences dialog cancel button clicked
 */
void
prefs_cancel_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	// This is ok for now, but we also need a 'revert' feature
	preferences_close (pd);
}

/**
 * prefs_close_cb: dialog has been closed
 */
gboolean
prefs_close_cb (GnomeDialog *dialog, PreferencesDialog *pd)
{
	preferences_close (pd);	
	return TRUE;
}

void
prefs_help_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
        preferences_help ();
}

/**
 * prefs_clear_disk_cache_button_clicked_cb: "Clear disk cache" clicked
 */
void
prefs_clear_disk_cache_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_ON_DISK);
	mozilla_clear_cache (STORE_ON_DISK_AS_FILE);
}

/**
 * prefs_clear_memory_cache_button_clicked_cb: "Clear memory cache" clicked
 */
void
prefs_clear_memory_cache_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	mozilla_clear_cache (STORE_IN_MEMORY);
}

/**
 * on_prefs_add_button_clicked: mime types add button clicked
 */
void
prefs_mime_add_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	MimeItem *newitem = g_new0 (MimeItem, 1);
	GtkWidget *clist = pd->mime_list;
	gchar *text[5];
	gint row;

	newitem->name = g_strdup (_("New mime item"));
	newitem->action = MIME_UNKNOWN;
	newitem->helperApp = g_strdup (_("None"));
	newitem->terminalHelper = FALSE;
	newitem->urlHelper = FALSE;
	newitem->alwaysUse = FALSE;
	text[0] = newitem->name;
	text[1] = _("Unknown");
	text[2] = newitem->helperApp;
	text[3] = _("False");
	text[4] = NULL;
	row = gtk_clist_append (GTK_CLIST (clist), text);
	gtk_clist_set_row_data (GTK_CLIST (clist), row, newitem);
	gtk_clist_select_row (GTK_CLIST (clist), row, 0);
}

/**
 * on_prefs_delete_button_clicked: mime types delete button clicked
 */
void
prefs_mime_delete_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	gint row;
	
	if (pd->selected_mime == NULL)
	{
		return;
	}

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);
	gtk_clist_remove (GTK_CLIST (pd->mime_list), row);
	mime_item_free (pd->selected_mime);
	pd->selected_mime = NULL;
	gtk_clist_select_row (GTK_CLIST (pd->mime_list), row, 0);
}


/**
 * prefs_mime_list_select_row_cb: mime type selected
 */
void prefs_mime_list_select_row_cb (GtkCList *clist, 
				    gint row, 
				    gint column, 
				    GdkEventButton *event, 
				    PreferencesDialog *pd)
{
	gint act_num = 0;
	
	pd->enable_edit_mime = FALSE;
	pd->selected_mime = (MimeItem*) gtk_clist_get_row_data (clist, row);
		
	switch (pd->selected_mime->action)
	{
	case MIME_SAVE_TO_DISK:
		act_num = 0;	/* save */
		break;
	case MIME_RUN_PROGRAM:	/* run with helper app */
		act_num = 1;
		break;
	case MIME_ASK_ACTION:	/* Ask */
	case MIME_UNKNOWN:	/* default to ask */
		act_num = 2;
		break;
	case MIME_IGNORE:
		break;
	}

	gtk_radio_button_set (GTK_RADIO_BUTTON (pd->mime_save_to_disk),
			      act_num);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_name_entry), 
			    pd->selected_mime->name);
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry),
			    pd->selected_mime->helperApp);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      pd->selected_mime->alwaysUse);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      pd->selected_mime->terminalHelper);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      pd->selected_mime->urlHelper);
	pd->enable_edit_mime = TRUE;
}

/**
 * prefs_mime_list_unselect_row_cb: mime type unselected
 */
void 
prefs_mime_list_unselect_row_cb (GtkCList *clist, gint row, gint column,
				 GdkEventButton *event, PreferencesDialog *pd)
{
	pd->selected_mime = NULL;
	pd->enable_edit_mime = FALSE;

	/* The default is Ask */
	gtk_entry_set_text (GTK_ENTRY (pd->mime_name_entry), "");
	gtk_entry_set_text (GTK_ENTRY (pd->mime_helper_entry), "");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_always_use),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_terminal),
				      FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(pd->mime_url_helper),
				      FALSE);
				      
}

/* FIXME! */
void
prefs_mime_edit_cb (GtkWidget *w, PreferencesDialog *pd)
{
        gint action_index;
	gchar *text[4];
	gint row;
	
        /* extract the action menu index */
	action_index = gtk_radio_button_get 
		(GTK_RADIO_BUTTON (pd->mime_save_to_disk));

	if (pd->selected_mime == NULL || !pd->enable_edit_mime)
	{
		return;
	}

	if (pd->selected_mime->name)
	{
		g_free (pd->selected_mime->name);
	}
	pd->selected_mime->name = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_name_entry), 0, -1);

	if (pd->selected_mime->helperApp)
	{
		g_free (pd->selected_mime->helperApp);
	}
	pd->selected_mime->helperApp = gtk_editable_get_chars 
		(GTK_EDITABLE (pd->mime_helper_entry), 0, -1);

	switch (action_index)
	{
	case 0:
		pd->selected_mime->action = MIME_SAVE_TO_DISK;
		break;
	case 1:
		pd->selected_mime->action = MIME_RUN_PROGRAM;
		break;
	case 2:
		pd->selected_mime->action = MIME_ASK_ACTION;
		break;
	default:
		pd->selected_mime->action = MIME_UNKNOWN;
		break;
	}
	pd->selected_mime->terminalHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_terminal));
	pd->selected_mime->urlHelper = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_url_helper));
	pd->selected_mime->alwaysUse = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(pd->mime_always_use));

	row = gtk_clist_find_row_from_data (GTK_CLIST (pd->mime_list),
					    pd->selected_mime);

	text[0] = pd->selected_mime->name;
	switch (pd->selected_mime->action)
	{
	case MIME_SAVE_TO_DISK:
		text[1] = _("Save to disk");
		break;
	case MIME_RUN_PROGRAM:  
		text[1] = _("Run with Helper App");
		break;
	case MIME_UNKNOWN:
		text[1] = _("Unknown");
		break;
	case MIME_ASK_ACTION:
		text[1] = _("Ask the user");
		break;
	case MIME_IGNORE:
		break;
	}
	text[2] = pd->selected_mime->helperApp;
	if (pd->selected_mime->alwaysUse)
		text[3] = _("True");
	else
		text[3] = _("False");

	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 0, text[0]);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 1, text[1]);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 2, text[2]);
	gtk_clist_set_text (GTK_CLIST (pd->mime_list), row, 3, text[3]);
}

/**
 * prefs_homepage_my_portal_button_clicked_cb: set homepage URL to "My Portal"
 */
void
prefs_homepage_my_portal_button_clicked_cb (GtkButton *button, 
					    PreferencesDialog *pd)
{
	gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry), MYPORTAL_URL);
}


/**
 * prefs_homepage_current_button_clicked_cb: set homepage URL to current page
 */
void
prefs_homepage_current_button_clicked_cb (GtkButton *button, 
					  PreferencesDialog *pd)
{
	const gchar *url;
	
	url = history_get_last_url ();
	gtk_entry_set_text (GTK_ENTRY (pd->startpage_entry), url);
}

void
prefs_lang_add_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GtkWidget *entry = pd->lang_entry;
	GList *items = NULL;
	GtkWidget *list_item;
	char *entry_text;

	entry_text = gtk_editable_get_chars (GTK_EDITABLE (entry),0,-1);
	list_item = gtk_list_item_new_with_label (entry_text);
	gtk_object_set_data (GTK_OBJECT (list_item), "string", entry_text);

	items = g_list_append (items, list_item);
	gtk_list_append_items (list, items);
	gtk_widget_show (list_item);
}

void
prefs_lang_remove_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		gtk_list_remove_items (list, selected); 
	}
}

void
prefs_lang_up_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		if (pos!=0)
		{
			gtk_list_remove_items_no_unref (list, selected); 
			pos--;
			gtk_list_insert_items (list, selected, pos);
			gtk_list_select_item (list, pos);
		}
	}
}

void
prefs_lang_down_button_clicked_cb (GtkButton *button, PreferencesDialog *pd)
{
	GtkList *list = GTK_LIST (pd->lang_list);
	GList *selected;
	int pos;

	if (list->selection)
	{
		selected = g_list_copy (list->selection);
		pos = gtk_list_child_position (list, 
					       GTK_WIDGET (selected->data));
		gtk_list_remove_items_no_unref (list, selected); 
		pos++;
		gtk_list_insert_items (list, selected, pos);
		gtk_list_select_item (list, pos);
	}
	
}

void
prefs_clear_history_button_clicked_cb (GtkButton *button,
				       PreferencesDialog *pd)
{
	GtkWidget *dialog;
	
	dialog = gnome_question_dialog_modal_parented
		(_("This will delete all items stored in your history.\n"
		   "Are you sure you want to do this?"),
		 (GnomeReplyCallback)prefs_clear_history_question_cb, pd,
		 GTK_WINDOW (pd->dialog));
	
	dialog_set_parent (dialog, pd->dialog);
	gnome_dialog_run (GNOME_DIALOG (dialog));
}

void
prefs_clear_history_question_cb (gint reply, gpointer data)
{
	if (reply)
	{
		return;
	}

	/* clear the hash table */
	history_clear ();
}

void
prefs_browse_clicked_cb (GnomeFileEntry *fileentry, PreferencesDialog *pd)
{
	if (fileentry->fsw != NULL)
	{
		dialog_set_parent (fileentry->fsw, pd->dialog);
	}
}

void
prefs_font_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_FONT_PICKER (widget)->font_dialog,
			   pd->dialog);
}

void
prefs_color_picker_clicked_cb (GtkWidget *widget, PreferencesDialog *pd)
{
	dialog_set_parent (GNOME_COLOR_PICKER (widget)->cs_dialog,
			   pd->dialog);
}

void 
prefs_spinner_select_icon_cb (GnomeIconList *gil, gint num, 
			      GdkEvent *event, PreferencesDialog *pd)
{
	const gchar *path;

	path = spinner_get_path_from_index (num);
	gtk_entry_set_text (GTK_ENTRY (pd->spinner_dir_entry), path);
}

void
prefs_theme_select_row_cb (GtkCList *list, gint row, gint column,
			   GdkEvent *event, PreferencesDialog *pd)
{
	const gchar *path;

	path = gtk_clist_get_row_data (list, row);
	gtk_entry_set_text (GTK_ENTRY (pd->theme_dir_entry), path);
}

void 
prefs_smart_bm_entry_width_spin_changed_cb (GtkEditable *editable,
					    PreferencesDialog *pd)
{
	gint width;

	width = gtk_spin_button_get_value_as_int 
		(GTK_SPIN_BUTTON (pd->smart_bm_entry_width_spin));
	gtk_widget_set_usize (GTK_WIDGET (pd->smart_bm_example_entry),
			      width, -2);
}

/**
 * preferences_switch_page_cb: called when the user jumps to a new page
 */
void
preferences_switch_page_cb (GtkWidget *notebook, 
			    GtkNotebookPage *notebook_page,
			    guint page_num,
			    PreferencesDialog *pd)
{
	/* if it's the themes page, load spinners into the list */
	/* this is quite slow, so we only do it if the user selects
	 * this specific page */
	if (notebook_page->child == pd->themes_page)
	{
		spinner_fill_iconlist (GNOME_ICON_LIST (pd->iconlist));
	}
}

/**
 * prefs_image_sites_button_clicked_cb
 */
void
prefs_image_sites_button_clicked_cb (GtkWidget *button, 
				     PreferencesDialog *pd)
{
	pdm_show_page (NULL, IMAGESITES);
}

/**
 * prefs_cookies_button_clicked_cb
 */
void
prefs_cookies_button_clicked_cb (GtkWidget *button, 
				 PreferencesDialog *pd)
{
	pdm_show_page (NULL, COOKIES);
}

/**
 * prefs_cookie_sites_button_clicked_cb
 */
void
prefs_cookie_sites_button_clicked_cb (GtkWidget *button, 
				      PreferencesDialog *pd)
{
	pdm_show_page (NULL, COOKIESITES);
}

/**
 * prefs_passwords_button_clicked_cb
 */
void
prefs_passwords_button_clicked_cb (GtkWidget *button, 
				   PreferencesDialog *pd)
{
	pdm_show_page (NULL, PASSWORDS);
}

/**
 * prefs_password_sites_button_clicked_cb
 */
void
prefs_password_sites_button_clicked_cb (GtkWidget *button, 
				        PreferencesDialog *pd)
{
	pdm_show_page (NULL, PASSWORDSITES);
}

/**
 * prefs_downloader_use_command_toggled_cb: called when the use command
 * radiobutton is toggled; we have to set the command entry and terminal
 * widgets to sensitive or not, appropriately.
 */
void
prefs_download_use_command_toggled_cb (GtkWidget *button, 
				       PreferencesDialog *pd)
{
	gboolean active;

	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->download_command_combo,   active);
	gtk_widget_set_sensitive (pd->download_run_in_terminal, active);
}

/**
 * prefs_mime_save_to_disk_toggled_cb: called whenever "save to disk" is
 * toggled, make the helper chooser frame sensitive or not accordingly
 */
void
prefs_mime_save_to_disk_toggled_cb (GtkWidget *button, PreferencesDialog *pd)
{
	gboolean active;

	active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));
	gtk_widget_set_sensitive (pd->mime_helper_frame, !active);
}

void
prefs_encoding_activated_cb (GtkWidget *button, 
			     PreferencesDialog *pd)
{
	prefs_fonts_remember (pd);
	prefs_fonts_language_init (pd);
}
