/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "toolbar.h"
#include "spinner.h"
#include "misc.h"
#include "window.h"
#include "bookmarks.h"
#include "eel-gconf-extensions.h"

#include <string.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktoolbar.h>
#include <gtk/gtkdnd.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gnome-dock.h>
#include <libgnomeui/gnome-entry.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

/* local function prototypes */
static void toolbar_update (GaleonWindow *window);
static void toolbar_create_widgets (GaleonWindow *window, 
				    GtkWidget *hbox, GtkWidget *toolbar,
				    gboolean location_on_main_toolbar);
static void toolbar_create_location_toolbar (GaleonWindow *window,
					     gboolean show);
static void toolbar_create_location_widgets (GaleonWindow *window, 
					     GtkWidget *hbox, 
					     GtkWidget *toolbar);
static void toolbar_create_find_entry (GaleonWindow *window,
				       GtkWidget *toolbar,
				       ToolbarItemType type);
static void toolbar_create_zoom_spin (GaleonWindow *window, 
				      GtkWidget *toolbar);
static void toolbar_create_drag_handle (GaleonWindow *window, 
					GtkWidget *toolbar);
static void toolbar_create_button (GaleonWindow *window, GtkWidget *toolbar,
				   ToolbarItemType type);
static GtkWidget *toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
				      gchar *name, gint band_num);
static void toolbar_set_style (GtkWidget *toolbar, GtkToolbarStyle final_style,
			       GtkReliefStyle relief);
static GtkWidget *toolbar_get_theme_pixmap (const gchar *pixmap);
static char *toolbar_get_nautilus_theme_path (void);

/* array of available toolbar items */
const ToolbarItem toolbar_items[TOOLBAR_ITEM_COUNT] =
{
	{
		N_("New"), 
		N_("Open new browser"), 
		GNOME_STOCK_PIXMAP_NEW, 
		"New",
		FALSE, /* priority? */
		FALSE,
		100,
		NULL, /* clicked */
		window_new_button_press_cb,
		window_new_button_release_cb,
		window_new_button_drag_data_received_cb
	},
	{
		N_("Back"), 
		N_("Go back - Right button: History"), 
		GNOME_STOCK_PIXMAP_BACK, 
		"Back",
		TRUE, /* priority? */
		FALSE,
		200,
		NULL, /* clicked */
		window_back_forward_button_press_cb,
		window_back_forward_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Up"), 
		N_("Go up a level from the current location"), 
		GNOME_STOCK_PIXMAP_UP, 
		"Up",
		FALSE, /* priority? */
		FALSE,
		300,
		NULL, /* clicked */
		window_up_button_press_event_cb,
		window_up_button_release_event_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Forward"), 
		N_("Go forward - Right button: History"), 
		GNOME_STOCK_PIXMAP_FORWARD,
		"Forward",
		FALSE, /* priority? */
		FALSE,
		400,
		NULL, /* clicked */
		window_back_forward_button_press_cb,
		window_back_forward_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Reload"), 
		N_("Display the latest content of the current page"), 
		GNOME_STOCK_PIXMAP_REFRESH, 
		"Refresh",
		FALSE, /* priority? */
		FALSE,
		700,
		window_refresh_button_clicked_cb,
		window_refresh_button_press_event_cb,
		NULL, /* release_event */
		NULL /* drag_data_received */
	},
	{
		N_("Home"), 
		N_("Go to your home page"), 
		GNOME_STOCK_PIXMAP_HOME, 
		"Home",
		TRUE, /* priority? */
		FALSE,
		500,
		NULL, /* clicked */
		window_home_button_press_event_cb,
		window_home_button_release_event_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Stop"), 
		N_("Stop current data transfer"), 
		GNOME_STOCK_PIXMAP_STOP, 
		"Stop",
		TRUE, /* priority? */
		FALSE,
		600,
		window_stop_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL /* release_event */,
		NULL /* drag_data_received */
	},
	{
		N_("Go"), 
		N_("Go to specified location"), 
		GNOME_STOCK_PIXMAP_JUMP_TO, 
		"Go",
		FALSE, /* priority? */
		FALSE,
		1400,
		NULL /* clicked */,
		window_generic_button_press_event_cb,
		window_go_button_release_cb,
		NULL /* drag_data_received */
	},
	{
		N_("Zoom control"), 
		NULL, 
		NULL, /* pixmap */ 
		NULL,
		FALSE, /* priority? */
		FALSE,
		1100,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Drag handle"), 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1200,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Location entry"),
		NULL,
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1300,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Spinner"), 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1500,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		"------", 
		NULL, 
		NULL, /* pixmap */
		NULL,
		FALSE, /* priority? */
		FALSE,
		1600,
		NULL /* clicked */,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Find"), 
		N_("Find in page"), 
		GNOME_STOCK_PIXMAP_SEARCH, 
		"Search",
		FALSE, /* priority? */
		FALSE,
		800,
		NULL,
		NULL,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Bookmarks"), 
		N_("Toggle Bookmarks Sidebar"), 
		GNOME_STOCK_PIXMAP_BOOK_OPEN, 
		NULL,
		FALSE, /* priority? */
		TRUE,
		1000,
		window_bookmarks_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("History"), 
		N_("Toggle History Sidebar"), 
		GNOME_STOCK_PIXMAP_INDEX, 
		NULL,
		FALSE, /* priority? */
		TRUE,
		900,
		window_history_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	},
	{
		N_("Print"), 
		N_("Print Current Page"), 
		GNOME_STOCK_PIXMAP_PRINT, 
		NULL,
		FALSE, /* priority? */
		FALSE,
		550,
		window_print_button_clicked_cb,
		window_generic_button_press_event_cb,
		NULL,
		NULL /* drag_data_received */
	}
};

/**
 * toolbar_create: create main toolbar
 */
void
toolbar_create (GaleonWindow *window, gboolean show)
{
	gboolean location_on_main_toolbar;
	GtkWidget *hbox, *toolbar;

	/* read prefs */
	location_on_main_toolbar = eel_gconf_get_boolean
		(CONF_TOOLBAR_URL_LOCATION);

	/* build the overall box to pack the toolbar into */
	hbox = gtk_hbox_new (FALSE, 2);

	/* dock the box into the window */
	window->main_dockitem = 
		toolbar_add_docked (window, hbox, GNOME_APP_TOOLBAR_NAME, 1);

	/* create items toolbar */
	window->main_toolbar = toolbar =
		gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);

	/* pack it into the box */
	gtk_box_pack_start (GTK_BOX (hbox), toolbar, FALSE, FALSE, 0);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (window->main_dockitem), 
			    "button-press-event", GTK_SIGNAL_FUNC
			    (window_generic_button_press_event_cb), window);

	/* create all the buttons and widgets */
	toolbar_create_widgets (window, hbox, toolbar, 
				location_on_main_toolbar);

	/* show the box and all its contents */
	gtk_widget_show_all (hbox);
	if (show) gtk_widget_show (window->main_dockitem);

	/* create location toolbar, if necessary */
	if (!location_on_main_toolbar)
	{
		/* create it's own toolbar */
		toolbar_create_location_toolbar (window, show);
	}

	/* update the toolbar appearance */
	toolbar_update (window);
}

/**
 * toolbar_recreate: recreate main toolbar (e.g. on prefs apply)
 */
void
toolbar_recreate (GaleonWindow *window)
{
	gchar *text = NULL;
	gboolean show = FALSE;

	/* get current text in url entry if any */
	/* NB: this MUST be strdup'd here since it's only a reference
	 * and we're about to destroy the toolbar (D'Oh! -- MattA) */
	if (window->toolBarOn && window->location_gnomeentry && 
	    window->location_entry)
	{
		text = g_strdup (gtk_entry_get_text 
				 (GTK_ENTRY (window->location_entry)));

		/* also store history */
		gnome_entry_save_history 
			(GNOME_ENTRY (window->location_gnomeentry));
	}

	/* destroy spinner, this prevents crashes with timeouts */
	spinner_destroy (window);

	/* destroy main toolbar */
	if (window->main_dockitem != NULL)
	{
		show = GTK_WIDGET_VISIBLE (window->main_dockitem);
		gtk_widget_destroy (window->main_dockitem->parent);
		window->main_dockitem = NULL;
		window->main_toolbar = NULL;
		window->right_toolbar = NULL;
	}

	/* destroy location toolbar */
	if (window->location_dockitem != NULL)
	{
		gtk_widget_destroy (window->location_dockitem->parent);
		window->location_dockitem = NULL;
		window->location_toolbar = NULL;
	}

	/* zero out the toolbar widgets, these should have already
	 * been destroyed by the destruction of the toolbars */
	window->new_button          = NULL;
	window->back_button         = NULL;
	window->up_button           = NULL;
	window->forward_button      = NULL;
	window->refresh_button      = NULL;
	window->home_button         = NULL;
	window->stop_button         = NULL;
	window->go_button           = NULL;
	window->zoom_spin           = NULL;
	window->drag_event_box      = NULL;
	window->drag_pixmap         = NULL;
	window->location_gnomeentry = NULL;
	window->location_entry      = NULL;
	window->history_button      = NULL;
	window->bookmarks_button    = NULL;

	/* create new toolbar */
	toolbar_create (window, show);
	
	/* set the controls properly for this window */
	window_update_nav_controls (window);

	/* restore location entry text and history */
	if (window->toolBarOn && window->location_gnomeentry && 
	    window->location_entry)
	{
		gnome_entry_load_history  
			(GNOME_ENTRY (window->location_gnomeentry));
		if (text != NULL)
		{
			window_set_url_entry (window, text);
			g_free (text);
		}
	}

	/* set bookmarks and history togglebuttons */
	window_update_bm_and_hist_buttons (window,
		(window->dock_type == DOCK_BOOKMARKS),
		(window->dock_type == DOCK_HISTORY));
}

/**
 * toolbar_update: update the toolbar appearance
 */
static void
toolbar_update (GaleonWindow *window)
{
	ToolbarStyle style;
	GtkReliefStyle relief;
	GtkToolbarStyle final_style;
	gboolean toolbar_button_relief;

	/* skip if no toolbar */
	if (!(window->toolBarOn))
	{
		return;
	}

	/* get relief */
	toolbar_button_relief = gnome_preferences_get_toolbar_relief_btn ();
	relief = (toolbar_button_relief ? GTK_RELIEF_NORMAL : GTK_RELIEF_NONE);

	/* get real style */
	style = eel_gconf_get_integer (CONF_TOOLBAR_STYLE);

	/* convert to GTK+ style for setting in buttons */
	final_style = GTK_TOOLBAR_BOTH; /* only to stop a warning */
	switch (style)
	{
	case TOOLBAR_STYLE_HORIZONTAL:
	case TOOLBAR_STYLE_TEXT_ONLY:
		final_style = GTK_TOOLBAR_TEXT;
		break;

	case TOOLBAR_STYLE_ICONS_ONLY:
		final_style = GTK_TOOLBAR_ICONS;
		break;

	case TOOLBAR_STYLE_VERTICAL:
		final_style = GTK_TOOLBAR_BOTH;
		break;
	}

	/* set toolbar styles */
	toolbar_set_style (window->main_toolbar,     final_style, relief);
	toolbar_set_style (window->right_toolbar,    final_style, relief);
	toolbar_set_style (window->location_toolbar, final_style, relief);
}

/**
 * toolbar_create_widgets: create the widgets on the main toolbar
 */
static void
toolbar_create_widgets (GaleonWindow *window, 
			GtkWidget *hbox, GtkWidget *toolbar,
			gboolean location_on_main_toolbar)
{
	GtkWidget *spinner;
	GSList *icon_list, *tmp;
	
	icon_list = eel_gconf_get_integer_list(CONF_TOOLBAR_ICON_LIST);
	tmp = icon_list;
	
	while (tmp!=NULL) 
	{
		ToolbarItemType type;

		/* get toolbar item type */
		type = GPOINTER_TO_INT(tmp->data);
	
		/* build the right thing */
		switch (type)
		{
		case TOOLBAR_ITEM_ZOOM_SPIN:
			/* create zoom spin entry */
			toolbar_create_zoom_spin (window, toolbar);
			break;

		case TOOLBAR_ITEM_DRAG_HANDLE:
			/* create drag handle if location widgets here */
			if (location_on_main_toolbar)
			{
				toolbar_create_drag_handle (window, toolbar);
			}
			break;

		case TOOLBAR_ITEM_LOCATION_ENTRY:
			/* create location widgets if on this toolbar */
			if (location_on_main_toolbar)
			{
				toolbar_create_location_widgets 
					(window, hbox, toolbar);

				/* use continuation toolbar */
				toolbar = window->right_toolbar;
			}
			break;

		case TOOLBAR_ITEM_SPINNER:
			/* create the spinner */
			spinner = spinner_create (window);
			if (!spinner) break;
			
			/* if not the last item... */
			if (g_slist_next(tmp)!=NULL)
			{
				/* pack it normally */
				gtk_toolbar_append_element 
					(GTK_TOOLBAR (toolbar), 
					 GTK_TOOLBAR_CHILD_WIDGET,
					 GTK_WIDGET (spinner),
					 NULL, NULL, NULL, NULL, NULL, NULL);
			}
			else
			{
				/* otherwise use hbox to right justify */
				gtk_box_pack_end (GTK_BOX (hbox), spinner,
						  FALSE, FALSE, 0);
			}
			break;

		case TOOLBAR_ITEM_SEPARATOR:
			/* append some space */
			gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));
			break;

		case TOOLBAR_ITEM_GO:
			/* create go button if location widgets here */
			if (location_on_main_toolbar)
			{
				toolbar_create_button (window, toolbar, type);
			}
			break;

		case TOOLBAR_ITEM_FIND:
			/* create find entry */
			toolbar_create_find_entry (window, toolbar, type);
			break;
			
		default:
			/* create a normal button */
			toolbar_create_button (window, toolbar, type);
			break;
		}
		tmp = g_slist_next(tmp);
	}

	g_slist_free(icon_list);
}

/**
 * toolbar_create_drag_handle: create a "drag handle", where you can
 * drag the current URL from
 */
static void
toolbar_create_drag_handle (GaleonWindow *window, GtkWidget *toolbar)
{
	PixmapData *pixmap = NULL;
	GtkWidget *event_box;

	/* build suitable pixmap */
	pixmap = pixmap_data_new_from_file (SHARE_DIR "/i-bookmark.xpm");
	if (pixmap != NULL)
	{
		window->drag_pixmap = gtk_pixmap_new (pixmap->pixmap, 
						      pixmap->mask);
		g_free (pixmap);
	}

	/* build event box (for capturing drag signals */
	event_box = window->drag_event_box = gtk_event_box_new ();
	gtk_container_set_border_width (GTK_CONTAINER (event_box), 4);

	/* connect drag signals */
	gtk_signal_connect (GTK_OBJECT (event_box), "drag_data_get",
			    (window_drag_pixmap_drag_data_get_cb), window);
	gtk_drag_source_set (GTK_WIDGET (event_box), GDK_BUTTON1_MASK, 
			     url_drag_types, url_drag_types_num_items,
			     GDK_ACTION_COPY | GDK_ACTION_LINK);

	/* add pixmap into box */
	gtk_container_add (GTK_CONTAINER (event_box), window->drag_pixmap);

	/* add to toolbar */
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (event_box),
				    NULL, _("Drag page location from here"),
				    NULL, NULL, NULL, NULL);
}

/**
 * toolbar_create_location_widgets: create location related widgets
 */
static void
toolbar_create_location_widgets (GaleonWindow *window, 
				 GtkWidget *hbox, GtkWidget *toolbar) 
{
	GtkWidget *entry;
	GtkWidget *popwin;
	GtkWidget *gnomeentry;
	gboolean location_on_main_toolbar;

	/* read prefs */
	location_on_main_toolbar = eel_gconf_get_boolean
		(CONF_TOOLBAR_URL_LOCATION);

	/* create drag handle if on separate toolbar */
	if (!location_on_main_toolbar)
	{
		toolbar_create_drag_handle (window, toolbar);
	}

	/* create location entry and access components */
	gnomeentry = gnome_entry_new ("LocationEntry");
	gnome_entry_load_history (GNOME_ENTRY (gnomeentry));
	entry = gnome_entry_gtk_entry (GNOME_ENTRY (gnomeentry));
	popwin = GTK_COMBO (gnomeentry)->popwin;
	window->location_gnomeentry = gnomeentry;
	window->location_entry = entry;

	/* connect the various signals */
	gtk_signal_connect (GTK_OBJECT (popwin), "show", GTK_SIGNAL_FUNC 
			    (window_location_gnomeentry_popwin_cb), window);
	gtk_signal_connect (GTK_OBJECT (popwin), "hide", GTK_SIGNAL_FUNC
			    (window_location_gnomeentry_popwin_cb), window);
	gtk_signal_connect_after (GTK_OBJECT (entry), "key_press_event", 
				  GTK_SIGNAL_FUNC 
				  (window_location_entry_key_press_cb), 
				  window);
	gtk_signal_connect (GTK_OBJECT (entry), "drag_data_received", 
			    GTK_SIGNAL_FUNC 
			    (window_location_entry_drag_data_received_cb), 
			    window);
	gtk_signal_connect_after (GTK_OBJECT (entry), "button_press_event", 
				  GTK_SIGNAL_FUNC 
				  (window_location_entry_button_press_cb), 
				  window);

	/* set url entry drop destination */
	gtk_drag_dest_set (gnomeentry, GTK_DEST_DEFAULT_ALL, drop_types, 
			   drop_types_num_items, GDK_ACTION_COPY);

	/* pack into toplevel box */
	gtk_box_pack_start (GTK_BOX (hbox), gnomeentry, TRUE, TRUE, 1);

	/* create continutation toolbar */
	window->right_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, 
						 GTK_TOOLBAR_BOTH);

        /* pack it in */
	gtk_box_pack_start (GTK_BOX (hbox), window->right_toolbar,
			    FALSE, FALSE, 0);

	/* add in Go button if on separate toolbar */
	if (!location_on_main_toolbar)
	{
		toolbar_create_button (window, window->right_toolbar,
				       TOOLBAR_ITEM_GO);
	}
}

/*
 * toolbar_create_location_toolbar: create location toolbar
 */
static void
toolbar_create_location_toolbar (GaleonWindow *window, gboolean show)
{
	GtkWidget *hbox;
	
	/* build the overall box to pack the toolbar into */
	hbox = gtk_hbox_new (FALSE, 2);

	/* dock the box into the window */
	window->location_dockitem = toolbar_add_docked (window, hbox, 
							"location", 2);

	/* create toolbar */
	window->location_toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
						    GTK_TOOLBAR_BOTH);

	/* pack it into the box */
	gtk_box_pack_start (GTK_BOX (hbox), window->location_toolbar,
			    FALSE, FALSE, 0);

	/* setup the toolbar context menu */
	gtk_signal_connect (GTK_OBJECT (window->location_dockitem), 
			    "button-press-event", GTK_SIGNAL_FUNC
			    (window_generic_button_press_event_cb), window);

	/* insert the location widgets */
	toolbar_create_location_widgets (window, hbox, 
					 window->location_toolbar);

	/* show everything */
	gtk_widget_show_all (hbox);
	if (show) gtk_widget_show (window->location_dockitem);
}

/**
 * toolbar_create_find_entry: create find in page entry
 */
static void
toolbar_create_find_entry (GaleonWindow *window, GtkWidget *toolbar,
			   ToolbarItemType type)
{
	const ToolbarItem *item;
	GtkWidget *entry;

	/* get item from array */
	item = &(toolbar_items[type]);

	/* create the entry, pack it */
	entry = gtk_entry_new ();
	gtk_widget_set_usize (entry, 100, -2);	
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar), 
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (entry), NULL, 
				    _(item->tooltip), NULL, NULL, NULL, NULL);

	/* connect signal (Return should search for inserted text) */
	gtk_signal_connect (GTK_OBJECT (entry), "activate",
			    GTK_SIGNAL_FUNC (window_find_entry_activate_cb), 
			    window);
}

/**
 * toolbar_create_zoom_spin: create zoom spin button
 */
static void
toolbar_create_zoom_spin (GaleonWindow *window, GtkWidget *toolbar)
{
	GtkObject *adj;

	/* create the adjustment */
	adj = gtk_adjustment_new (100, 1, 999, 10, 25, 10);

	/* build the spin button */
	window->zoom_spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 1, 0);

	/* setup properties */
	gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (window->zoom_spin),
					 GTK_SHADOW_NONE);

	/* connect signals */
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed", 
			    window_zoom_spin_changed_cb, window);
	gtk_signal_connect (GTK_OBJECT (window->zoom_spin),
			    "key_press_event", GTK_SIGNAL_FUNC 
			    (window_zoom_spin_key_press_cb), window);
	gtk_signal_connect_after (GTK_OBJECT (window->zoom_spin),
				  "button_press_event", GTK_SIGNAL_FUNC 
				  (window_zoom_spin_button_press_cb), window);

	/* append to toolbar */
	gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
				    GTK_TOOLBAR_CHILD_WIDGET,
				    GTK_WIDGET (window->zoom_spin),
				    NULL, NULL, NULL, NULL, NULL, NULL);
}

/*
 * toolbar_button: create and append a button to a toolbar
 */
static void
toolbar_create_button (GaleonWindow *window, GtkWidget *toolbar, 
		       ToolbarItemType type)
{
	GtkWidget *button, *label, *box;
	GtkWidget *pixmap;
	gchar *theme, *filename;
	const ToolbarItem *item;

	/* get item from array */
	item = &(toolbar_items[type]);
	
	pixmap = NULL;
	
	/* which theme to use */
	theme = eel_gconf_get_string (CONF_TOOLBAR_THEME_DIR);

	/* get default nautilus theme when needed */
	if (strcmp (theme, "NAUTILUS_THEME") == 0)
	{
		g_free (theme);
		theme = toolbar_get_nautilus_theme_path ();
	}

	if (theme && item->theme_icon)
	{
		filename = g_strconcat (theme, "/", item->theme_icon, ".png", 
					NULL);
		pixmap = toolbar_get_theme_pixmap (filename);
		g_free (filename);
	}

	/* use GNOME stock icon if not found or not requested */
	if (pixmap == NULL)
	{
		GdkPixbuf *pixbuf;
		gint height = 24;

		filename = NULL;

		/* we take Refresh because this is the highest icon in 
		 * nautilus themes... nautilus icons haven't got the same 
		 * height. Also, we only take height because they aren't
		 * square either :-/ */
		if (theme)
			filename = g_concat_dir_and_file (theme, "Refresh.png");
		if (filename && g_file_exists (filename))
		{
			pixbuf = gdk_pixbuf_new_from_file (filename);
			height = gdk_pixbuf_get_height (pixbuf);
			gdk_pixbuf_unref (pixbuf);
		}
		if (filename) g_free (filename);
		
		if (height != 24)
		{ 
			pixmap = gnome_stock_pixmap_widget_at_size 
				 (window->WMain, item->stock_icon, height, 
				  height); /* GNOME stock icons are square */
		}
		else
		{
			pixmap = gnome_stock_pixmap_widget (window->WMain,
							    item->stock_icon);
		}
	}

	if (theme) g_free (theme);

	/* make the toolbar button and add it to the toolbar */
	if (eel_gconf_get_integer (CONF_TOOLBAR_STYLE) ==
	    TOOLBAR_STYLE_HORIZONTAL)
	{
		/* we create a horizontal toolbar style ourselves */
		box = gtk_hbox_new (FALSE, 2);

		/* pack the pixmap first */
		gtk_box_pack_start (GTK_BOX (box), pixmap, FALSE, FALSE, 0);

		/* if priority text for this button, pack that too */
		if (item->priority)
		{
			label = gtk_label_new (_(item->label));
			gtk_box_pack_start (GTK_BOX (box), label, 
					    TRUE, TRUE, 0);
		}

		/* build button */
		if (item->togglebutton)
		{
			button = gtk_toggle_button_new ();
		}
		else
		{
			button = gtk_button_new ();
		}

		/* insert box into button */
		gtk_container_add (GTK_CONTAINER (button), box);

		/* append button to toolbar */
		gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), button,
					   _(item->tooltip), NULL);

		/* set it up */
		gtk_button_set_relief (GTK_BUTTON (button), GTK_RELIEF_NONE);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	}
	else
	{
		/* insert plain button */
		button = gtk_toolbar_append_element 
			(GTK_TOOLBAR (toolbar), 
			 (item->togglebutton ? 
			  GTK_TOOLBAR_CHILD_TOGGLEBUTTON : 
			  GTK_TOOLBAR_CHILD_BUTTON),
			 NULL, _(item->label), _(item->tooltip),
			 NULL, pixmap, NULL, NULL);
	}

	/* connect appropriate signals */
	if (item->clicked_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "clicked", 
				    item->clicked_cb, window);
	}
	if (item->button_press_event_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "button_press_event", 
				    GTK_SIGNAL_FUNC
				    (item->button_press_event_cb), window);
	}
	if (item->button_release_event_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button),
				    "button_release_event", GTK_SIGNAL_FUNC
				    (item->button_release_event_cb), window);
	}
	if (item->drag_data_received_cb != NULL)
	{
		gtk_signal_connect (GTK_OBJECT (button), "drag_data_received", 
				    GTK_SIGNAL_FUNC
				    (item->drag_data_received_cb), window);

		/* set drop types */
		gtk_drag_dest_set (GTK_WIDGET (button), GTK_DEST_DEFAULT_ALL,
				   drop_types, drop_types_num_items, 
				   GDK_ACTION_COPY);
	}

	/* set the window->XXX_button appropiately */
	switch (type)
	{
	case TOOLBAR_ITEM_NEW:
		window->new_button = button;
		break;
	case TOOLBAR_ITEM_BACK:
		window->back_button = button;
		break;
	case TOOLBAR_ITEM_UP:
		window->up_button = button;
		break;
	case TOOLBAR_ITEM_FORWARD:
		window->forward_button = button;
		break;
	case TOOLBAR_ITEM_REFRESH:
		window->refresh_button = button;
		break;
	case TOOLBAR_ITEM_START:
		window->home_button = button;
		break;
	case TOOLBAR_ITEM_STOP:
		window->stop_button = button;
		break;
	case TOOLBAR_ITEM_GO:
		window->go_button = button;
		break;
	case TOOLBAR_ITEM_HISTORY:
		window->history_button = button;
		break;
	case TOOLBAR_ITEM_BOOKMARKS:
		window->bookmarks_button = button;
		break;
	default:
		break;
	}
}

/**
 * toolbar_add_docked: add an horizontal box to the toolbar dock
 */
static GtkWidget *
toolbar_add_docked (GaleonWindow *window, GtkWidget *hbox, 
		    gchar *name, gint band_number)
{
	gboolean toolbar_relief;
	gboolean toolbar_detachable;
	gboolean location_on_main_toolbar;
	GnomeDockItemBehavior props;
	GtkReliefStyle relief;

	/* read prefs */
	toolbar_relief = gnome_preferences_get_toolbar_relief ();
	relief = (toolbar_relief ? GTK_SHADOW_OUT : GTK_SHADOW_NONE);
	toolbar_detachable = gnome_preferences_get_toolbar_detachable ();
	location_on_main_toolbar = eel_gconf_get_boolean 
		(CONF_TOOLBAR_URL_LOCATION);

	/* default properties */
	props = (GNOME_DOCK_ITEM_BEH_EXCLUSIVE |
		 GNOME_DOCK_ITEM_BEH_NEVER_VERTICAL);

	/* lock according to gnome prefs */
	if (!toolbar_detachable)
	{
		props |= GNOME_DOCK_ITEM_BEH_LOCKED;
	}

	/* dock into app */
	gnome_app_add_docked (GNOME_APP (window->WMain), hbox, name,
			      props, GNOME_DOCK_TOP, band_number, 0, 0);

	/* set border appropriate to whether or not there's relief */
	gtk_container_set_border_width (GTK_CONTAINER(hbox->parent),
					toolbar_relief ? 2 : 0);

	/* set shadowing follwing gnome pref */
	gnome_dock_item_set_shadow_type (GNOME_DOCK_ITEM (hbox->parent), 
					 relief);

	/* this is a GnomeDockItem */
	return hbox->parent;
}

/**
 * toolbar_set_style: set overall relief and button relief styles
 */	
static void
toolbar_set_style (GtkWidget *toolbar, GtkToolbarStyle final_style,
		   GtkReliefStyle relief)
{
	/* skip if not there */
	if (toolbar == NULL)
	{
		return;
	}

	/* otherwise set style */
	gtk_toolbar_set_style (GTK_TOOLBAR (toolbar), final_style);
	gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar), relief);
}

/**
 * toolbar_get_theme_pixmap: FIXME
 */
static GtkWidget *
toolbar_get_theme_pixmap (const gchar *pixmap)
{
	PixmapData *icon;
	GtkWidget *widget;

	if (!g_file_exists (pixmap))
	{
		return NULL;
	}

	icon = pixmap_data_new_from_file (pixmap);

	if (icon != NULL)
	{
		widget = gtk_pixmap_new (icon->pixmap, icon->mask);
		g_free (icon); /* FIXME: better to cache these? */
		return GTK_WIDGET (widget);
	}
	else
	{
		return NULL;
	}
}

/**
 * toolbar_get_nautilus_theme_path: get the path of the 
 * current nautilus theme
*/
static char *
toolbar_get_nautilus_theme_path (void)
{
	int i;
	char *theme;
	char *icon, *tmp;
	char *result = NULL;

	static gchar *path[] =
	{
		SHARE_DIR "/../pixmaps/nautilus/%s",
		SHARE_DIR "/../pixmaps/nautilus/%s/fake-aa-navibar",
		NULL,
		NULL,
		SHARE_DIR "/../pixmaps/nautilus/",
		NULL
	};

	theme = eel_gconf_get_string(CONF_NAUTILUS_THEME);

	if (!theme || 
	    strcmp (theme, "default") == 0)
	{
		g_free (theme);
		theme = g_strdup ("");
	}
	else
	if (strcmp (theme, "gnome") == 0) /* use standard icons */
	{
		g_free (theme);
		return NULL;
	}

	path[2] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/%s", NULL);
	path[3] = g_strconcat (g_get_home_dir (), 
			       "/.nautilus/themes/fake-aa-navibar/%s",
			       NULL);

	for (i = 0; path[i] != NULL && !result; i++)
	{
		tmp = g_strdup_printf (path[i], theme);
		icon = g_strconcat (tmp, "/Refresh.png", NULL);

		if (g_file_exists (icon))
		{
			result = g_strdup (tmp);
		}
		g_free (tmp);
		g_free (icon);
	}

	g_free (path [2]);
	g_free (path [3]);
	g_free (theme);

	return result;
}

