/*
 *  Copyright (C) 2001 Christophe Fergeau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window_notifiers.h"
#include "window.h"
#include "eel-gconf-extensions.h"
#include "prefs_strings.h"
#include "bookmarks.h"
#include "toolbar.h"
#include "embed.h"
#include "misc.h"

extern gboolean fullscreen_active;

void main_toolbar_notifier(GConfClient *client,
			   guint cnxn_id,
			   GConfEntry *entry,
			   GaleonWindow *window);
void bookmarks_toolbars_notifier(GConfClient *client,
				 guint cnxn_id,
				 GConfEntry *entry,
				 GaleonWindow *window);
void bookmarks_menus_notifier(GConfClient *client,
			      guint cnxn_id,
			      GConfEntry *entry,
			      GaleonWindow *window);
void tabs_position_notifier(GConfClient *client,
			   guint cnxn_id,
			   GConfEntry *entry,
			   GaleonWindow *window);
void
tabs_appeareance_notifier(GConfClient *client,
			  guint cnxn_id,
			  GConfEntry *entry,
			  GaleonWindow *window);
void
autobookmarks_notifier(GConfClient *client,
		       guint cnxn_id,
		       GConfEntry *entry,
		       GaleonWindow *window);

void generic_settings_bool_notifier(GConfClient *client,
				    guint cnxn_id,
				    GConfEntry *entry,
				    GtkWidget *widget);

void window_menubar_init_notifiers (GaleonWindow *window);

static void
window_menubar_install_notifier(GaleonWindow *window, 
				gchar *pref_name,
				GConfClientNotifyFunc notifier_cb,
				gpointer user_data);

void load_images_notifier(GConfClient *client,
			  guint cnxn_id,
			  GConfEntry *entry,
			  GaleonWindow *window);
void animate_notifier(GConfClient *client,
		      guint cnxn_id,
		      GConfEntry *entry,
		      GaleonWindow *window);
void
fullscreen_menubar_notifier (GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     GaleonWindow *window);
void
fullscreen_toolbar_notifier (GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     GaleonWindow *window);
void
fullscreen_bookmarks_notifier (GConfClient *client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       GaleonWindow *window);
void
fullscreen_statusbar_notifier (GConfClient *client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       GaleonWindow *window);

/*
 * window_menubar_install_notifier: installs the notifiers for one control 
 * in the settings menu
 */ 
static void
window_menubar_install_notifier(GaleonWindow *window, 
				gchar *pref_name,
				GConfClientNotifyFunc notifier_cb,
				gpointer user_data)
{	
	GConfClient* client = eel_gconf_client_get_global();
	GConfValue* value;

	galeon_notification_add(pref_name, 
				notifier_cb, 
				(gpointer) user_data, 
				&window->notifiers_id);

	/* set the control according to the current state */
	value = gconf_client_get (client, pref_name, NULL);
	gconf_client_set (client, pref_name, value, NULL);
	gconf_value_free (value);
}

typedef struct 
{
	char *gconf_key;
	gpointer func;
	gpointer user_data;
} SettingsNotifierInfo;

/* installs all the notifiers for the settings menu */
void
window_menubar_init_notifiers (GaleonWindow *window)
{
	SettingsNotifierInfo menubar_notifiers [] =
	{
		{CONF_FILTERING_JAVA_ENABLED, 
		 generic_settings_bool_notifier, window->enable_java},
		{CONF_FILTERING_JAVASCRIPT_ENABLED, 
		 generic_settings_bool_notifier, window->enable_javascript},
		{CONF_NETWORK_PROXY_ENABLED, 
		 generic_settings_bool_notifier, window->enable_proxy},
		{CONF_RENDERING_USE_OWN_COLORS, 
		 generic_settings_bool_notifier, window->use_own_colors},
		{CONF_RENDERING_USE_OWN_FONTS, 
		 generic_settings_bool_notifier, window->use_own_fonts},
		{CONF_TABS_TABBED_AUTOJUMP, 
		 generic_settings_bool_notifier, window->autojump},
		{CONF_FILTERING_ANIMATE_TYPE,
		 animate_notifier, window},
		{CONF_FILTERING_IMAGE_LOADING_TYPE,
		 load_images_notifier, window},
		{NULL, NULL, NULL}
	};
	int i;

	for (i = 0; menubar_notifiers[i].gconf_key != NULL; i++)
	{
		window_menubar_install_notifier
			(window, menubar_notifiers[i].gconf_key,
			 (GConfClientNotifyFunc)menubar_notifiers[i].func,
			 menubar_notifiers[i].user_data);
	}
}

static const struct 
{
	char *gconf_key;
	gpointer func;
}
window_notifiers [] =
{
	{CONF_TOOLBAR_STYLE, main_toolbar_notifier},
	{CONF_TOOLBAR_ICON_LIST, main_toolbar_notifier},
	{CONF_TOOLBAR_URL_LOCATION, main_toolbar_notifier},
	{CONF_TOOLBAR_SPINNER_DIR, main_toolbar_notifier},
	{CONF_TOOLBAR_THEME_DIR, main_toolbar_notifier},
	{CONF_BOOKMARKS_ENTRY_WIDTH, bookmarks_toolbars_notifier},
	{CONF_BOOKMARKS_GOTO_BUTTON, bookmarks_toolbars_notifier},
	{CONF_BOOKMARKS_HISTORY, bookmarks_toolbars_notifier},
	{CONF_BOOKMARKS_XITEMS, bookmarks_menus_notifier},
	{CONF_TABS_TABBED_POSITION, tabs_position_notifier},
	{CONF_TABS_TABBED_ALWAYS_SHOW, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_SHORTEN_STYLE, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_SHORTEN_CHARS, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_SHORTEN_POINTS, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_CLOSEBUTTON, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_NEW_COLOR, tabs_appeareance_notifier},
	{CONF_TABS_TABBED_LOADING_COLOR, tabs_appeareance_notifier},
	{CONF_BOOKMARKS_AB_COUNT, autobookmarks_notifier},
	{CONF_BOOKMARKS_AB_SHORTEN, autobookmarks_notifier},
	{CONF_BOOKMARKS_AB_ENABLED, autobookmarks_notifier},
	{CONF_WINDOWS_FS_SHOW_MENUBAR, fullscreen_menubar_notifier},
	{CONF_WINDOWS_FS_SHOW_BOOKMARKS, fullscreen_bookmarks_notifier},
	{CONF_WINDOWS_FS_SHOW_TOOLBARS, fullscreen_toolbar_notifier},
	{CONF_WINDOWS_FS_SHOW_STATUSBAR, fullscreen_statusbar_notifier},
	{NULL, NULL}
};

/* add gconf notifiers for the toolbars */
void
window_init_notifiers(GaleonWindow *window)
{
	int i;

	for (i = 0; window_notifiers[i].gconf_key != NULL; i++)
	{
		galeon_notification_add(
			window_notifiers[i].gconf_key,
			(GConfClientNotifyFunc)window_notifiers[i].func,
			(gpointer) window,
			&window->notifiers_id);
	}
}

static const struct 
{
	char *gconf_key;
	gpointer func;
}
toolbars_notifiers [] =
{
	{CONF_TOOLBAR_STYLE, main_toolbar_notifier},
	{CONF_NAUTILUS_THEME, main_toolbar_notifier},
	{CONF_TOOLBAR_ICON_LIST, main_toolbar_notifier},
	{CONF_TOOLBAR_URL_LOCATION, main_toolbar_notifier},
	{CONF_TOOLBAR_SPINNER_DIR, main_toolbar_notifier},
	{CONF_TOOLBAR_THEME_DIR, main_toolbar_notifier},
	{CONF_BOOKMARKS_ENTRY_WIDTH, bookmarks_toolbars_notifier},
	{NULL, NULL}
};

/* add gconf notifiers for the toolbars */
void
window_toolbars_init_notifiers(GaleonWindow *window)
{
	int i;

	for (i = 0; toolbars_notifiers[i].gconf_key != NULL; i++)
	{
		galeon_notification_add(
			toolbars_notifiers[i].gconf_key,
			(GConfClientNotifyFunc)toolbars_notifiers[i].func,
			(gpointer) window,
			&window->notifiers_id);
	}
}

void generic_settings_bool_notifier(GConfClient *client,
				    guint cnxn_id,
				    GConfEntry *entry,
				    GtkWidget *widget)
{
	gtk_check_menu_item_set_active
		(GTK_CHECK_MENU_ITEM (widget),
		 gconf_value_get_bool(gconf_entry_get_value(entry)));
}

void 
load_images_notifier(GConfClient *client,
		     guint cnxn_id,
		     GConfEntry *entry,
		     GaleonWindow *window)
{
	gint load_images = gconf_value_get_int(gconf_entry_get_value(entry));
	if (!window->menuBarOn) return;
	switch (load_images) {
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->load_images_from_current_server_only), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_FILTERING_IMAGE_LOADING_TYPE);
	}
}

void animate_notifier(GConfClient *client,
		     guint cnxn_id,
		     GConfEntry *entry,
		     GaleonWindow *window)
{
	gint animate = gconf_value_get_int(gconf_entry_get_value(entry));
	if (!window->menuBarOn) return;
	switch (animate)
	{
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->animate_once_through), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_FILTERING_ANIMATE_TYPE);
	}
}


/* Notifiers for the toolbars */

void main_toolbar_notifier(GConfClient *client,
			   guint cnxn_id,
			   GConfEntry *entry,
			   GaleonWindow *window)
{
	if (!(window->toolBarOn))
	{
		return;
	}
	window_save_layout (window);
	toolbar_recreate (window);
	window_restore_layout (window);
}

void bookmarks_toolbars_notifier(GConfClient *client,
				 guint cnxn_id,
				 GConfEntry *entry,
				 GaleonWindow *window)
{
	if (!(window->toolBarOn))
	{
		return;
	}
	window_save_layout (window);
	bookmarks_toolbars_recreate (window);	
	window_restore_layout (window);
}

void bookmarks_menus_notifier(GConfClient *client,
			      guint cnxn_id,
			      GConfEntry *entry,
			      GaleonWindow *window)
{
	bookmarks_menu_recreate (window);
}


void
tabs_position_notifier(GConfClient *client,
		      guint cnxn_id,
		      GConfEntry *entry,
		      GaleonWindow *window)
{
	gboolean pos = gconf_value_get_bool(gconf_entry_get_value(entry));
	gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
				  pos);
}

void
tabs_appeareance_notifier (GConfClient *client,
			   guint cnxn_id,
			   GConfEntry *entry,
			   GaleonWindow *window)
{
	gint n_embeds, always_show, popup = 0;
	GList *w, *e;

	initialise_colours ();

	always_show =
		eel_gconf_get_boolean (CONF_TABS_TABBED_ALWAYS_SHOW);

	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		GaleonWindow *window = (GaleonWindow *)(w->data);

		/* check if the tab(s) should be visible */
		n_embeds = g_list_length (window->embed_list);
		if (n_embeds == 1)
		{
			GaleonEmbed *embed;

			e = window->embed_list;
 			embed = (GaleonEmbed *) (e->data);
			popup = window->is_popup;
		}

		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
					    (n_embeds > 1) | 
					    (always_show && !popup));

		/* update notebook tabs */
		for (e = window->embed_list; e != NULL; e = g_list_next (e))
		{
			GaleonEmbed *embed = (GaleonEmbed *) (e->data);

			embed_update_tab_title (embed);
			embed_update_tab_close_button (embed);
			embed_update_tab_status (embed);
		}
	}
}

void
autobookmarks_notifier(GConfClient *client,
		       guint cnxn_id,
		       GConfEntry *entry,
		       GaleonWindow *window)
{
	/* FIXME this is called correctly but does not
	   seems to generate/remove them on the fly */
	autobookmarks_generate ();
}

void
fullscreen_menubar_notifier (GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     GaleonWindow *window)
{
	gboolean show;

	if (!window->menuBarOn || !fullscreen_active) return;

	show = eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_MENUBAR);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_menubar), show);
}

void
fullscreen_toolbar_notifier (GConfClient *client,
			     guint cnxn_id,
			     GConfEntry *entry,
			     GaleonWindow *window)
{
	gboolean show;

	if (!window->menuBarOn || !fullscreen_active) return;

	show = eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_TOOLBARS);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_toolbar), show);
}

void
fullscreen_bookmarks_notifier (GConfClient *client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       GaleonWindow *window)
{
	gboolean show;

	if (!window->menuBarOn || !fullscreen_active) return;

	show = eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_BOOKMARKS);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_bookmarks), show);
}

void
fullscreen_statusbar_notifier (GConfClient *client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       GaleonWindow *window)
{
	gboolean show;

	if (!window->menuBarOn || !fullscreen_active) return;

	show = eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_STATUSBAR);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->view_statusbar), show);
}
