/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/**
 * Total number of completions to offer
 */
#define MAX_ALTERNATIVES 200

/**
 * Avaible completions
 */
GCompletion *URLCompletion = NULL;

/** 
 * List of possible completions for the current entry
 */
GList *possible_completions = NULL;

/**
 * The alternatives window
 */ 
GtkWidget *alternatives_window = NULL;


/**
 * Common autocompletion prefixes
 * Please note that the order of this list is important
 * for simplifying urls : 
 * when we have http://www.foo.bar, 
 * we want to get foo.bar as simplified url 
 * so http://www. must come before http://
 */
const gchar *common_prefixes[] = 
{ 
	"", 
	"http://www.", 
	"http://", 
	"https://www.", 
	"https://", 
	"file://", 
	"www.", 
	NULL             /* terminator, must be last */
};


/* Private functions */
static GList *auto_completion_insert_sorted_nodup(GList *l,char *url, int type);
static GList *auto_completion_concat(GList *l1, GList *l2);
static void auto_completion_alternatives_clist_select_row_cb (GtkCList *clist, gint row,
		gint column, GdkEventButton *event, GtkWidget *entry);
gchar *auto_completion_truncate_url(gchar *url);
gchar *auto_completion_simplify_url(gchar *url);
static gboolean alternatives_win_button_press_cb (GtkWidget *widget,
						GdkEventButton *event,
						GtkWidget *alternatives_window);
static gboolean alternatives_win_key_press_cb (GtkWidget *widget,
					       GdkEventKey *event,
					       GtkWidget *location_entry);


/**
 * Adds an url to the list of completions
 */
void
auto_completion_add_url (gchar *url)
{
	if (strstr (url, "%s")) return;
	
	if (URLCompletion) {
		GList *list = g_list_append (NULL, url);
		g_completion_add_items (URLCompletion, list);
		g_list_free (list);	
	} else {
		URLCompletion = g_completion_new (NULL);
		auto_completion_add_url (url);
	}
}

/**
 * Destroys the alternatives window and reset the list of _current_ possible 
 * completions. Does not clear the whole completions list.
 */
void 
auto_completion_reset (void)
{
	g_list_free(possible_completions);
	possible_completions = NULL;
	if (alternatives_window != NULL)
		gtk_widget_destroy (alternatives_window);
	alternatives_window = NULL;
}

/** 
 * Tries to complete an url 
 * Returns a possible completion of the url being typed
 * You mustn't free the returned string
 */
gchar *
auto_completion_complete_url (gchar *prefix)
{
	gchar *common;
	GList *tmp = NULL;

	if (URLCompletion == NULL) return NULL;
	tmp = g_completion_complete(URLCompletion, prefix, &common);
	if (common) {
		g_free(common);
		return (gchar *)tmp->data;;
	} 
	return NULL;		
}

/**
  * Concatenate the two GLists l1 and a copy of l2
  */
static GList *auto_completion_concat(GList *l1, GList *l2)
{
	while (l2!=NULL) {
		l1 = g_list_prepend (l1, l2->data);
		l2 = g_list_next (l2);
	}
	return l1;
}


/** 
 * Like auto_completion_complete_url but tries to prepend "http://" and similar
 * returns the longest common prefix of all the possible completions
 * You must free the returned string
 */
gchar *
auto_completion_complete_url_extended (gchar *prefix)
{
	gchar *common = NULL;
	GList *tmp = NULL;
	gchar *common_prefix = NULL;
	int i;

	g_list_free(possible_completions);
	possible_completions = NULL;
	if (strcmp (prefix, "") == 0) return NULL;	
	if (URLCompletion == NULL) return NULL;

	for (i = 0; (common_prefixes[i] != NULL); i++) {
		gchar *extended_prefix = 
			g_strconcat (common_prefixes[i], prefix, NULL);
		tmp = g_completion_complete(URLCompletion, 
				      extended_prefix, &common);
		if (tmp != NULL) {
			possible_completions = auto_completion_concat(possible_completions, tmp);
		}
		if (common != NULL) {
			if (!common_prefix)
				common_prefix = common;
			else 
			       	g_free(common);
		}
		g_free(extended_prefix);
		common = NULL;
	}	
	return common_prefix;
}

/**
 * Suppress http://, http://www., www., ... from the url
 * The returned string mustn't be modified nor freed
 */

gchar *
auto_completion_simplify_url(gchar *url)
{
	unsigned char i = 0;
	unsigned char url_length = 0;

	for (i=1; common_prefixes[i]!=NULL; i++) {
		url_length = strlen(common_prefixes[i]);
		if (strncmp(url, common_prefixes[i], url_length) == 0) {
			return url+url_length;
		}
	}
	return url;
}


/**
 * REMOVED If someone wants to readd it, 
 * add a preference for disabling it -- ricardo
 *
 * Strips the url before the first /, ? or #
 * Deals with file:// (does nothing) and the xxx:// suffixes (skips them)
 * You must free the returned string
 */

gchar *
auto_completion_truncate_url(gchar *url)
{
	gchar *newurl;

	newurl = strstr(url, "file:");	
	if (newurl == NULL) {
		newurl = g_strdup(auto_completion_simplify_url (url));
		while ((*newurl != '\0') && (*newurl != '/') && (*newurl != '?') && (*newurl !='#'))
			newurl++;
		*newurl = '\0';
	} else {
		newurl = g_strdup (url);
	}
	return newurl;
}


/**
 * insert the string data into the given glist 
 * the list returned is sorted and without duplicates 
 * type indicate whether we should consider full urls or
 * simplified urls
 */
static GList *
auto_completion_insert_sorted_nodup(GList *l, char *url, int type)
{
	GList *tmp = l;
	int i = 0;
	int pos = 0;
	gchar *newurl;
	gchar *data = NULL;


	if (type==1) 
		newurl = auto_completion_simplify_url(url);
	else newurl = url;

	/** Inserts url into the GList l */
	while (tmp != NULL) {
		if (type==1) 
			data = auto_completion_simplify_url(tmp->data);
		else data = tmp->data;

		i = strcmp (newurl, data);
		if (i == 0) return l;
		if (i < 0) {
			break;
		} else {
			tmp = g_list_next(tmp);
			pos++;
		}
	}
	l = g_list_insert(l, url, pos);
	return l;
}


/** 
 * displays an completed url in the location bar
 * according to the preferences, appends a selected autocompleted 
 * string to the end of the url
 */
void
auto_completion_display(GaleonBrowser *browser, GtkWidget *entry)
{
	gchar *text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	const gchar *completion = auto_completion_complete_url (text);
	int completion_type = gnome_config_get_int
					(CONF_COMPLETION_AUTOCOMP_ENABLE);

	if (completion) {
		int text_len = strlen(text);

		if (completion_type == 0) {
			gtk_entry_set_text(GTK_ENTRY (entry), completion);
			text = gtk_editable_get_chars(GTK_EDITABLE (entry),
						      0, -1);
			gtk_editable_select_region(GTK_EDITABLE (entry),
						   text_len, -1);
			gtk_editable_set_position(GTK_EDITABLE (entry),  -1);
		}
	}
	g_free(text);
}

/**
 * Displays the list of possible completions under the location box of the given browser
 */
void 
auto_completion_display_alternatives (GaleonBrowser *browser, GtkWidget *entry)
{
	int sort = gnome_config_get_int (CONF_COMPLETION_SORT);
	gint x, y, height, width, depth;
	GtkWidget *scroll = gtk_scrolled_window_new (NULL, NULL);
	GtkWidget *clist = gtk_clist_new (1); 
	GtkRequisition r;
	GList *l;
	int count = 0;
	GList *labels = NULL;

	if (alternatives_window)
		gtk_widget_destroy (alternatives_window);
	alternatives_window = gtk_window_new (GTK_WINDOW_POPUP);

	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
		       			GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (alternatives_window), scroll);
	gtk_container_add (GTK_CONTAINER (scroll), clist);

	gtk_signal_connect (GTK_OBJECT(clist), "select_row", 
			    GTK_SIGNAL_FUNC(auto_completion_alternatives_clist_select_row_cb), 
			    entry);
	
	for (l = possible_completions; (l!=NULL) && (count < MAX_ALTERNATIVES);
	     l = g_list_next(l), count++) 
	{
		labels = auto_completion_insert_sorted_nodup(labels, l->data, sort);
	};

	for (l =  labels; (l != NULL); l = g_list_next(l)) 
	{
		char *text[] = { l->data, NULL };
		gtk_clist_append (GTK_CLIST (clist), text);
	}	
	g_list_free(labels);
	gdk_window_get_geometry (entry->window, &x, &y, &width, &height, &depth);
	gdk_window_get_deskrelative_origin (entry->window, &x, &y);
	y += height;
	gtk_widget_set_uposition (alternatives_window, x, y);

	/* For some reason, the requisition height won't be correct until
	 * the clist has been shown. I get it twice and set the size of the 
	 * window twice to avoid flickering -- ricardo
	 */
       	gtk_widget_size_request (clist, &r);
	gtk_widget_set_usize (alternatives_window, width, r.height);
	gtk_widget_show_all (alternatives_window);       
	gtk_widget_size_request (clist, &r);

	if ((y + r.height) > gdk_screen_height ()) {
		gtk_window_set_policy
			(GTK_WINDOW (alternatives_window), TRUE, FALSE, FALSE);
		gtk_widget_set_usize 
			(alternatives_window, width, gdk_screen_height () - y);
		
	}
	gtk_signal_connect (GTK_OBJECT(alternatives_window),
			    "button-press-event",
			    GTK_SIGNAL_FUNC(alternatives_win_button_press_cb),
			    alternatives_window);
	gtk_signal_connect (GTK_OBJECT(alternatives_window),
			    "key-press-event",
			    GTK_SIGNAL_FUNC(alternatives_win_key_press_cb),
			    entry);

	gdk_pointer_grab (alternatives_window->window, TRUE,
			  GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK |
			  GDK_BUTTON_RELEASE_MASK,
			  NULL, NULL, GDK_CURRENT_TIME);
	gtk_grab_add (alternatives_window);
}

static void 
auto_completion_alternatives_clist_select_row_cb (GtkCList *clist,
						  gint row, gint column,
						  GdkEventButton *event,
						  GtkWidget *entry)
{
	GdkEventKey tmp_event;

	if (GTK_IS_ENTRY (entry)) {
		gchar *text;
		gtk_clist_get_text (clist, row, column, &text);
		gtk_entry_set_text (GTK_ENTRY (entry), text);
		auto_completion_reset();

		/* send a synthetic return keypress to the entry */
		tmp_event.type = GDK_KEY_PRESS;
		tmp_event.window = entry->window;
		tmp_event.send_event = TRUE;
		tmp_event.time = GDK_CURRENT_TIME;
		tmp_event.state = 0;
		tmp_event.keyval = GDK_Return;
		gtk_widget_event(entry, (GdkEvent *)&tmp_event);
	}
}

/** 
 * Adds the history of a GnomeEntry to the list of completions 
 */
/* Actually, I think that this may be unnecessary. */
void 
auto_completion_add_from_entry (GnomeEntry *e) 
{
	static gboolean done = FALSE;
	if (!done) { /* do this only once */
		struct item {
			gboolean save;
			gchar *text;
		};
		GList *list;
		for (list = e->items; list != NULL; list = list->next) {
			struct item *i = list->data;
			auto_completion_add_url (i->text);
		}	
	}
}

static gboolean
alternatives_win_button_press_cb (GtkWidget *widget,
				  GdkEventButton *event,
				  GtkWidget *alternatives_window)
{
	GtkWidget *event_widget;

	event_widget = gtk_get_event_widget((GdkEvent *)event);

	/* Check to see if button press happened inside the alternatives
	   window.  If not, destroy the window. */
	if (event_widget != widget)
	{
		while (event_widget)
		{
			if (event_widget == widget)
				return FALSE;
			event_widget = event_widget->parent;
		}
	}
	auto_completion_reset();

	return TRUE;
}

static gboolean
alternatives_win_key_press_cb (GtkWidget *widget,
			       GdkEventKey *event,
			       GtkWidget *location_entry)
{
	GdkEventKey tmp_event;
	GtkCList *clist = GTK_CLIST (GTK_BIN (GTK_BIN (widget)->child)->child);

	/* allow keyboard navigation in the alternatives clist */
	if (event->keyval == GDK_Up || event->keyval == GDK_Down ||
	    event->keyval == GDK_Page_Up ||  event->keyval == GDK_Page_Down ||
	    event->keyval == GDK_space)
		return FALSE;
	
	if (event->keyval == GDK_Return)
	{
		if (!GTK_CLIST_CHILD_HAS_FOCUS (clist)) {
			event->keyval = GDK_space;
			return FALSE;
		}
		auto_completion_reset();
	}

	/* else send the key event to the location entry */
	tmp_event.type = event->type;
	tmp_event.window = location_entry->window;
	tmp_event.send_event = TRUE;
	tmp_event.time = event->time;
	tmp_event.state = event->state;
	tmp_event.keyval = event->keyval;
	tmp_event.length = event->length;
	tmp_event.string = event->string;
	gtk_widget_event(location_entry, (GdkEvent *)&tmp_event);

	return TRUE;
}
