/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/** The pixmaps for the bookmarks */
GnomePixmap *folder_pixmap = NULL;
GnomePixmap *folder_open_pixmap = NULL;
GnomePixmap *site_pixmap = NULL;

/** The root of the bookmark tree */
BookMarkItem *bookmarks_root = NULL;

/** The root of the temporary bookmark tree */
BookMarkItem *temp_bookmarks_root = NULL;

/** The temp bookmarks window */
bookmarks_editor_controls *temp_bookmarks_window = NULL;

/** The bookmarks editing controls */
bookmarks_editor_controls *bookmarks_editor = NULL;

/* Private prototypes */
ns_item_type ns_get_bookmark_item(FILE *f, GString *name, GString *url);
void bookmarks_save_as (BookMarkItem *root, char *file);
BookMarkItem *bookmarks_load_from (char *file);
static GtkWidget *bookmarks_create_menu_item (BookMarkItem *bm);
static void bookmarks_create_file_submenu (GtkMenuItem *file_menuitem, 
					   BookMarkItem *bookmark);
static void bookmarks_create_toolbars (GaleonBrowser *browser, BookMarkItem *bi);
static void bookmarks_create_toolbar_from (GaleonBrowser *browser, BookMarkItem *bi);
static void bookmarks_toolbar_clicked (GtkButton *button, gpointer user_data);
static gboolean bookmarks_button_release_event (GtkWidget *widget, 
						GdkEventButton *event,
						gpointer user_data);
static BookMarkItem *bookmarks_xml_read_item (xmlNodePtr item);
static BookMarkItem *bookmarks_xml_read (xmlNodePtr item);
static void bookmarks_load_icons (void);
void bookmarks_editor_remove_tree_item_null (BookMarkItem *b);


static void bookmarks_add_completion_recursively (BookMarkItem *b);
static void bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root,
					BookMarkItem *b);
static BookMarkItem *bookmarks_find_by_nick (const gchar *nick, BookMarkItem *root);

typedef enum {
	STANDARD_BOOKMARKS_EDITOR,
	COMPACT_BOOKMARKS_EDITOR
} bookmarks_editor_controls_type;		

bookmarks_editor_controls *
bookmarks_editor_controls_init (BookMarkItem *root,
				bookmarks_editor_controls_type type);

GtkMenu*
bookmarks_create_copy_menu_recursively (bookmarks_editor_controls *destination_window, 
					BookMarkItem *source, BookMarkItem *b);

/* dnd */
const GtkTargetEntry bookmarks_dnd_targets [] = {
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK},
	{ "GALEON_URL", 0, DND_TARGET_GALEON_URL },
	{ "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL},
	{ "text/uri-list", 0, DND_TARGET_TEXT_URI_LIST},
	{ "STRING", 0, DND_TARGET_STRING}
};
const gint bookmarks_dnd_targets_num_items = (
	sizeof (bookmarks_dnd_targets) / sizeof (GtkTargetEntry));


/**
 * bookmarks_xml_read_item: read an item from xml file
 */
static BookMarkItem *
bookmarks_xml_read_item (xmlNodePtr item)
{
	BookMarkItem *b;
	BookMarkItem *b2;
	gchar *ct, *expanded;

	gchar *name = xmlGetProp (item, "name");	
	gchar *notes = xmlGetProp (item, "notes");
	gchar *nick = xmlGetProp (item, "nick");
	gchar *pixmap_file = xmlGetProp (item, "pixmap");
  
	g_return_val_if_fail(item != NULL, NULL);

	if (!strcmp (item->name, "category")) {
		b = bookmarks_new_bookmark (CATEGORY, name, NULL, nick, notes,
					    pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE : FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE : FALSE;
		xmlFree (ct);
		xmlFree (expanded);
		item = item->childs;
		while (item != NULL) {
			xmlNodePtr item2 = item;
			b2 = bookmarks_xml_read_item (item2);
			b->list = g_list_append (b->list, b2);
			b2->parent = b;
			item = item->next;
		}
	} else if (!strcmp (item->name, "separator")) {
		b = bookmarks_new_bookmark (SEPARATOR, name, NULL, nick, notes,
					    pixmap_file);
	} else if (!strcmp (item->name, "autobookmarks")) {
		/* FIXME: remove code duplication */
		b = bookmarks_new_bookmark (AUTOBOOKMARKS, name, NULL, nick,
					    notes, pixmap_file);
		ct = xmlGetProp (item, "create_toolbar");
		expanded = xmlGetProp (item, "expanded");
		b->create_toolbar = (ct && !strcmp (ct, "TRUE")) ? TRUE : FALSE;
		b->expanded = (expanded && !strcmp (expanded, "TRUE")) ? TRUE : FALSE;
		xmlFree (ct);
		xmlFree (expanded);
		autobookmarks_root = b;
	} else { /* site */
		gchar *url = xmlGetProp (item, "url");
		b = bookmarks_new_bookmark (SITE, name, url, nick, notes,
					    pixmap_file);
		if (url) g_free (url);
	}

	if (name) g_free (name);
	if (notes) g_free (notes);
	if (nick) g_free (nick);
	if (pixmap_file) g_free (pixmap_file);

	return b;
}

/**
 * bookmarks_xml_read: read all items from xml file
 */
static BookMarkItem * 
bookmarks_xml_read (xmlNodePtr item) 
{
	BookMarkItem *b, *result;
	item = item->childs;

	result = bookmarks_new_bookmark (CATEGORY, _("Bookmarks"),
					 NULL, NULL, NULL, NULL);
  
	while (item != NULL) {
		xmlNodePtr item2 = item;
		b = bookmarks_xml_read_item(item2);
		result->list = g_list_append(result->list, b);
		b->parent = result;
		item = item->next;
	}
	
	return result;
}

/**
 * bookmarks_editor_place_tree_item: place a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Create the tree items for a bookmark and its children. If it has already
 * been placed, removes the old item.
 * 
 **/
void
bookmarks_editor_place_tree_item (bookmarks_editor_controls *controls, BookMarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	GdkPixmap *pixmap;
	GdkPixmap *pixmap_open;
	GdkPixmap *mask_open;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
	GtkCTreeNode *sibling_node;

	text[2] = NULL;
	switch (b->type) {
	case SEPARATOR:
		text[0] = text[1] = "------";
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	case SITE:
		text[0] = b->name;
		text[1] = b->url;
		isleaf = TRUE;
		pixmap = pixmap_open = site_pixmap->pixmap;
		mask = mask_open = site_pixmap->mask;
		break;
	case CATEGORY:
	case AUTOBOOKMARKS:
		text[0] = b->name;
		text[1] = "";
		isleaf = FALSE;
		pixmap = folder_pixmap->pixmap;
		mask = folder_pixmap->mask;
		pixmap_open = folder_open_pixmap->pixmap;
		mask_open = folder_open_pixmap->mask;
		break;
	default:
		text[0] = text[1] = "";
		isleaf = TRUE;
		pixmap = mask = pixmap_open = mask_open = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = b->parent->tree_item;

	sibling_node = NULL;
	if (b->parent != NULL) {
		GList *list_node = g_list_find (b->parent->list, b);
		if (list_node) {
			GList *sibling_list_node = g_list_next (list_node);
			if (sibling_list_node) {
				sibling_node = ((BookMarkItem *) 
					(sibling_list_node->data))->tree_item;
			}
		}
	}
	gtk_clist_freeze (GTK_CLIST (controls->ctree));
	if (b->tree_item)
		bookmarks_editor_remove_tree_item (controls, b);
	if (b != controls->root_bookmark) {
		b->tree_item = gtk_ctree_insert_node
			(GTK_CTREE (controls->ctree), parent_node, sibling_node, text, 
			 2, pixmap, mask, pixmap_open, mask_open, isleaf, b->expanded);
		gtk_ctree_node_set_row_data (GTK_CTREE (controls->ctree), 
					     b->tree_item, b);
	}

	if (!isleaf) {
		GList *l;
		for (l = b->list; l != NULL; l = g_list_next (l))
			bookmarks_editor_place_tree_item (controls, l->data);
	}
	gtk_clist_thaw (GTK_CLIST (controls->ctree));
}

/**
 * bookmarks_editor_remove_tree_item: remove a bookmark from the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Removes bookmarks and it's childs from the bookmark editor ctree and sets the 
 * tree_item fields to NULL. This does not delete the bookmarks
 * 
 **/
void
bookmarks_editor_remove_tree_item (bookmarks_editor_controls *controls, BookMarkItem *b)
{
	/* this removes the treeitem and its children */
	gtk_ctree_remove_node (GTK_CTREE (controls->ctree), b->tree_item);
	/* now, let's NULL the fields */
	bookmarks_editor_remove_tree_item_null (b);
}

/**
 * bookmarks_editor_remove_tree_item_null: Sets to NULL the tree_item field of
 * a bookmarks and its children
 * @b: the bookmark
 * 
 * This is an auxiliar function for bookmarks_editor_remove_tree_item.
 **/
void
bookmarks_editor_remove_tree_item_null (BookMarkItem *b)
{
	GList *l;
	b->tree_item = NULL;
	if (b->type == CATEGORY) 
		for (l = b->list; l != NULL; l = g_list_next (l)) 
			bookmarks_editor_remove_tree_item_null (l->data);
}

/**
 * bookmarks_editor_update_tree_item: update a bookmark in the editor ctree
 * @controls: the set of controls
 * @b: the bookmark
 * 
 * Updates the bookmark (or adds it if not yet added)
 * 
 **/
void
bookmarks_editor_update_tree_item (bookmarks_editor_controls *controls, BookMarkItem *b)
{
	gchar *text[3];
	gboolean isleaf;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
	GtkCTreeNode *parent_node;
		
	if (!b->tree_item) {
		bookmarks_editor_place_tree_item (controls, b);
		return;
	}

	text[2] = NULL;
	switch (b->type) {
	case SEPARATOR:
		text[0] = text[1] = "------";
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	case SITE:
		text[0] = b->name;
		text[1] = b->url;
		isleaf = TRUE;
		pixmap = site_pixmap->pixmap;
		mask = site_pixmap->mask;
		break;
	case CATEGORY:
	case AUTOBOOKMARKS:
		text[0] = b->name;
		text[1] = "";
		isleaf = FALSE;
		pixmap = folder_pixmap->pixmap;
		mask = folder_pixmap->mask;
		break;
	default:
		text[0] = text[1] = "";
		isleaf = TRUE;
		pixmap = mask = NULL;
		break;
	}

	if ((b->parent == NULL) || (b->parent == controls->root_bookmark))
		parent_node = NULL;
	else
		parent_node = b->parent->tree_item;

	if (b != controls->root_bookmark) {
		gtk_ctree_node_set_pixtext
			(GTK_CTREE (controls->ctree), b->tree_item, 0, text[0],
			2, pixmap, mask);
		gtk_ctree_node_set_text
			(GTK_CTREE (controls->ctree), b->tree_item, 1, text[1]);
	}
}

/**
 * bookmarks_load_icons: load the bookmarks icons
 */
static void 
bookmarks_load_icons (void) 
{
	static gboolean done = FALSE;
	if (done) return;
	folder_pixmap = GNOME_PIXMAP (
		gnome_pixmap_new_from_file (SHARE_DIR "/dir.xpm"));

	folder_open_pixmap = GNOME_PIXMAP (
		gnome_pixmap_new_from_file (SHARE_DIR "/dir_open.xpm"));
			
	site_pixmap = GNOME_PIXMAP (
		gnome_pixmap_new_from_file (SHARE_DIR "/i-bookmark.xpm"));
	done = TRUE;
}

/**
 * bookmarks_insert_bookmark: insert a bookmark
 * @b: the bookmark t insert
 * @near: a bookmark that should be near @b (a brother or its parent)
 **/
void
bookmarks_insert_bookmark (BookMarkItem *b, BookMarkItem *near,
			   GtkCListDragPos insert_pos)
{
	BookMarkItem *parent;
	GList *position_node;
	gint position;

	if (near == NULL)
		parent = bookmarks_root;
	else
	{
		if (insert_pos == GTK_CLIST_DRAG_INTO &&
		    (near->type == CATEGORY || near->type == AUTOBOOKMARKS))
			parent = near;
		else
			parent = near->parent;
	}

	g_assert (parent != NULL);
	b->parent = parent;

	position_node = g_list_find (parent->list, near);
	if (!position_node)
		parent->list = g_list_prepend (parent->list, b);
	else
	{
		position = g_list_position (parent->list, position_node);

		switch (insert_pos)
		{
		case GTK_CLIST_DRAG_NONE:
		case GTK_CLIST_DRAG_BEFORE:
			break;
		case GTK_CLIST_DRAG_INTO:
			if (near == NULL)
				break;
			if (near->type == SITE ||
			    near->type == SEPARATOR)
				position++;
			break;				
		case GTK_CLIST_DRAG_AFTER:
			position++;
			break;
		default:
			break;
		}
		parent->list = g_list_insert (parent->list, b, position);
	}
}

/**
 * bookmarks_remove_recursively: recursively remove a tree of bookmarks items
 * Doesn't update tree items
 */
void 
bookmarks_remove_recursively (BookMarkItem *b) {
	GList *position, *li;
  
	if (b->parent) {
		position = g_list_find (b->parent->list, b);
		b->parent->list = 
			g_list_remove_link (b->parent->list, position);
	}

	if (b->type == CATEGORY || b->type == AUTOBOOKMARKS)
		for (li = b->list; li != NULL; li = li->next) 
			bookmarks_remove_recursively (li->data);

	bookmarks_free_bookmark (b);
}

/**
 * bookmarks_add_completion_recursively: recursively add bookmarks to the
 * auto completions table
 */
static void bookmarks_add_completion_recursively (BookMarkItem *b)
{
	GList *item;
  
	switch (b->type)
	{
	case CATEGORY:
	case AUTOBOOKMARKS:
		for (item = b->list; item != NULL; item = item->next)
		{
			bookmarks_add_completion_recursively (item->data);
		}
		break;

	case SITE:
		auto_completion_add_url (b->url);
		if (strncmp(b->url, "http://", 7) == 0)
		{
			auto_completion_add_url (b->url + 7);
		}
		break;

	case SEPARATOR:
		break;
	}

}

/**
 * bookmarks_load_from: Read a bookmarks tree
 * @file: The file to read, it should exist
 * Return value: The new allocated tree of bookmarks or NULL if error
 **/
BookMarkItem *
bookmarks_load_from (char *file)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookMarkItem *b = NULL;

	doc = xmlParseFile (file);	
	
	if (doc)
	{
		item = doc->root;
		b = bookmarks_xml_read (item);
		xmlFreeDoc (doc);
		return b;
	}
	else 
	{
		g_warning ("unable to parse bookmarks file: %s", file);
		return NULL;
	}
}

/**
 * bookmarks_load: load bookmarks from xml file
 */
void
bookmarks_load (void)
{
	gchar *confdir;
	gchar *bookfile; 
	
	bookmarks_load_icons ();

	/* FIXME: we should use something like get_conf_dir() or get_bookmarks_file */
	/* this directory should have already been created */
	confdir = g_strconcat (g_get_home_dir(), "/.galeon", NULL);
	if (access (confdir, F_OK) == -1)
	{
		g_warning("unable to access %s\n", confdir);
		return;
	}
		
	bookfile = g_strconcat (confdir, "/bookmarks.xml", NULL);
	
	bookmarks_root = bookmarks_load_from (bookfile);

	if (!bookmarks_root) {
		bookmarks_root = bookmarks_new_bookmark 
			(CATEGORY, _("Bookmarks"), NULL, NULL, NULL, NULL);
	}

	/* generate the autobookmarks category */
	autobookmarks_generate();
  
	bookmarks_add_completion_recursively (bookmarks_root);

	g_free (bookfile);
}

void 
bookmarks_editor_init (void)
{
	bookmarks_editor = bookmarks_editor_controls_init
		(bookmarks_root, STANDARD_BOOKMARKS_EDITOR);
	bookmarks_editor->save_function = bookmarks_save;
}

/**
 * bookmarks_save_recursively: recursively save a bookmarks tree to the xml file 
 */
static void
bookmarks_save_recursively (xmlDocPtr doc, xmlNodePtr root, BookMarkItem *b)
{
	BookMarkItem *b2;
	xmlNodePtr node;
	GList *li;
  
	switch (b->type)
	{
	case CATEGORY:
		node = xmlNewDocNode (doc, NULL, "category", NULL);
		xmlSetProp (node, "name", b->name);
		xmlSetProp (node, "create_toolbar", 
			    b->create_toolbar ? "TRUE" : "FALSE");
		xmlSetProp (node, "expanded", 
			    b->expanded ? "TRUE" : "FALSE");
		xmlSetProp(node, "notes", b->notes);
		if (b->pixmap_file)
			xmlSetProp(node, "pixmap", b->pixmap_file);
		xmlAddChild (root, node);
    
		for (li = b->list; li != NULL; li = li->next) {
			b2 = li->data;
			bookmarks_save_recursively(doc, node, b2);
		}
		break;

	case SEPARATOR:
		node = xmlNewDocNode(doc, NULL, "separator", NULL);
		xmlAddChild(root, node);
		break;

	case SITE:
		node = xmlNewDocNode(doc, NULL, "site", NULL);
		xmlSetProp(node, "name", b->name);
		xmlSetProp(node, "url", b->url);    
		xmlSetProp(node, "nick", b->nick);
		xmlSetProp(node, "notes", b->notes);
		if (b->pixmap_file)
			xmlSetProp(node, "pixmap", b->pixmap_file);
		xmlAddChild(root, node);
		break;

	case AUTOBOOKMARKS:
		node = xmlNewDocNode(doc, NULL, "autobookmarks", NULL);
		xmlSetProp (node, "name", b->name);
		xmlSetProp (node, "create_toolbar", 
			    b->create_toolbar ? "TRUE" : "FALSE");
		xmlSetProp (node, "expanded", 
			    b->expanded ? "TRUE" : "FALSE");
		xmlSetProp(node, "notes", b->notes);
		if (b->pixmap_file)
			xmlSetProp(node, "pixmap", b->pixmap_file);
		xmlAddChild(root, node);
		break;
	}
}

/**
 * bookmarks_save: save all bookmarks to the xml file
 */
void
bookmarks_save (void) 
{
	bookmarks_update_menu ();
	bookmarks_update_tb ();
	if ((!bookmarks_editor) || (bookmarks_editor->dirty)) {
		char *savefile = g_strconcat (g_get_home_dir(), 
					      "/.galeon/bookmarks.xml", NULL);
		bookmarks_save_as (bookmarks_root, savefile);
		if (bookmarks_editor) bookmarks_editor->dirty = FALSE;
		g_free (savefile);
	}
}

/**
 * bookmarks_save_as: Save a set of bookmarks
 * @root: the root category to be saved
 * @file: the filename, if already exists will be overwriten
 **/
/* FIXME: check for errors (permisions...) */
void 
bookmarks_save_as (BookMarkItem *root, char *file)
{
	xmlNodePtr node;
	xmlDocPtr doc = xmlNewDoc ("1.0");
	GList *li;
	node = xmlNewDocNode (doc, NULL, "bookmarks", NULL);

  	for (li = root->list; li != NULL; li = li->next) {
		BookMarkItem *b = li->data;
		bookmarks_save_recursively (doc, node, b);
	}	

	xmlDocSetRootElement (doc, node);
	xmlSaveFile (file, doc);
	xmlFreeDoc (doc);
}

/**
 * bookmarks_smart_bm_dialog_single: prompt user to resolve smart bookmark with
 * exactly one parameter
 * Returns: return value of the dialog
 */
static gint
bookmarks_smart_bm_dialog_single(GtkWidget *dialog, GaleonBrowser *browser, 
				 BookMarkItem *bi)
{
	GtkWidget *entry;
	gchar *s, *tmp, *result;
	gint dlg_status;
			   
	entry = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
			    entry, TRUE, TRUE, 0);
	
	gtk_widget_grab_focus (GTK_WIDGET (entry));
			    
	gnome_dialog_editable_enters (GNOME_DIALOG(dialog), GTK_EDITABLE(entry));
	gtk_widget_show_all (dialog);
	if ((dlg_status = gnome_dialog_run (GNOME_DIALOG (dialog))) == 0) {
		s = strstr (bi->url, "%s");
		g_return_val_if_fail (s, 0);
		tmp = g_strndup(bi->url, s - bi->url);
		result = g_strconcat (tmp, gtk_entry_get_text (GTK_ENTRY (entry)),
				      s+2, NULL);
		g_free (tmp);
		browser_load_url (browser, result);
		g_free (result);
	}
	return dlg_status;
}

/**
 * bookmarks_smart_bm_dialog_single: prompt user to resolve smart bookmark with
 * 2 or more parameters
 * Returns: return value of the dialog
 */
static gint
bookmarks_smart_bm_dialog_multi(GtkWidget *dialog, GaleonBrowser *browser, 
				BookMarkItem *bi)
{
	gchar *start, *point, *tmp, *result;
	int i, dlg_status;
	GtkWidget *seperator, *entry, *label;
	GPtrArray *entries;

	seperator = gtk_hseparator_new();
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
			    seperator, TRUE, TRUE, 0);

	entries = g_ptr_array_new();

	start = bi->url;
	while(1)
	{
		point = strstr(start, "%s");
		if (point == start)
		{
			/* user-input part of URL */
			entry = gtk_entry_new ();
			gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
					    entry, TRUE, TRUE, 0);
			g_ptr_array_add(entries, (gpointer) entry);
			start = point + 2;
			continue;
		}
		if (point == NULL)
			point = start + strlen(start);
		if (point == start)
			break;
		
		/* static part of URL */
		tmp = g_strndup(start, point-start);
		label = gtk_label_new (tmp);
		g_free(tmp);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    label, TRUE, TRUE, 0);
		g_ptr_array_add(entries, (gpointer) label);
		start = point;
	}

	/* set focus on the first entry */
	for (i = 0; i < entries->len; ++i) {
		if (GTK_IS_ENTRY (entries->pdata[i]))
		{
			gtk_widget_grab_focus (GTK_WIDGET (entries->pdata[i]));
			break;
		}
	}

	/* activating (hitting enter) on the last entry should select "OK" */
	for (i = entries->len - 1; i >= 0; --i) {
		if (GTK_IS_ENTRY (entries->pdata[i]))
		{
			gnome_dialog_editable_enters
				(GNOME_DIALOG(dialog), 
				 GTK_EDITABLE(entries->pdata[i]));
			break;
		}
	}

	gtk_widget_show_all (dialog);

	if ((dlg_status = gnome_dialog_run (GNOME_DIALOG (dialog))) == 0)
	{
		/* user selected OK, construct URL and go there */
		result = g_strnfill(0,0);
		for (i = 0; i < entries->len; ++i)
		{
			gchar *text;
			if (GTK_IS_ENTRY (entries->pdata[i]))
			{
				text = gtk_entry_get_text 
					(GTK_ENTRY (entries->pdata[i]));
			}
			else
			{
				gtk_label_get 
					(GTK_LABEL (entries->pdata[i]), &text);
			}
			tmp = g_strconcat (result, text, NULL);
			g_free (result);
			result = tmp;
		}
		browser_load_url (browser, result);
		g_free(result);
	}
	g_ptr_array_free (entries, FALSE);
	return dlg_status;
}

/**
 * bookmarks_smart_bm_dialog: prompt user to complete the URL
 */
void
bookmarks_smart_bm_dialog (GaleonBrowser *browser, BookMarkItem *bi)
{	
	GtkWidget *dialog;
	gchar *s = bi->url;
	int num_parameters = 0;
	int dlg_status = 0;
	
	dialog = gnome_dialog_new (bi->name,
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG(dialog), 0);
	
	if (bi->pixmap == NULL)
	{
		GtkWidget *label;
		label = gtk_label_new(bi->name);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    label, TRUE, TRUE, 0);
	}
	else
	{
		GtkWidget *image;
		image = gtk_pixmap_new (bi->pixmap->pixmap, bi->pixmap->mask);
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), 
				    GTK_WIDGET (image), TRUE, FALSE, 0);
	}

	/* count %s in the URL */
	s = bi->url;
	while ((s = strstr(s, "%s"))) {
		++num_parameters;
		++s;
	}

	if (num_parameters == 1)
	{
		dlg_status = bookmarks_smart_bm_dialog_single(dialog, browser, bi);
	}
	else if (num_parameters > 1)
	{
		dlg_status = bookmarks_smart_bm_dialog_multi(dialog, browser, bi);
	}
	if (dlg_status != -1)
		gnome_dialog_close (GNOME_DIALOG (dialog));
}

/**
 * bookmarks_create_toolbars: create personal toolbars
 */
static void 
bookmarks_create_toolbars (GaleonBrowser *browser, BookMarkItem *bi)
{
	/* skip if we're not a category */
	if ((bi->type != CATEGORY) && (bi->type != AUTOBOOKMARKS))
		return;

	/* if we're meant to create a toolbar... */
	if (bi->create_toolbar) {
		/* do so */
		bookmarks_create_toolbar_from (browser, bi);
	} else {
		GList *l;

		/* otherwise iterate to find any subfolders to use */
		for (l = bi->list; l != NULL; l = l->next) {
			bookmarks_create_toolbars (browser, l->data);
		}
	}
}

/**
 * bookmarks_create_toolbar_from: create a toolbar from a folder
 */
static void 
bookmarks_create_toolbar_from (GaleonBrowser *browser, BookMarkItem *bi)
{
	/* this magic number is used to hack around a problem with
	 * gnome-libs (I think) which causes multiple toolbars with
	 * the same number not to display properly -- MattA */
	static gint magic_number = 99;
	GtkToolbar *tb;
	GList *l;

	/* make the toolbar */
	tb = GTK_TOOLBAR(gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
					  GTK_TOOLBAR_TEXT));

	/* add the toolbar it to the main window */
	/* this has to be done here for the relief to get set */
	gnome_app_add_toolbar (GNOME_APP (browser->WMain), tb, bi->name,
			       GNOME_DOCK_ITEM_BEH_EXCLUSIVE, GNOME_DOCK_TOP, 
			       magic_number++, 0, 0);

	/* override global setting */
	gtk_toolbar_set_space_style (tb, GTK_TOOLBAR_SPACE_EMPTY);
	gtk_toolbar_set_space_size (tb, 6);

	/* iterate over the contents of the folder */
	for (l = bi->list; l != NULL; l = l->next)
	{
		BookMarkItem *b = l->data;

		/* build the appropriate element */
		if (b->type != SEPARATOR)
		{
			bookmarks_create_toolbar_from_bm(browser, tb, b, -1);
		} 
		else 
		{
			gtk_toolbar_append_space(tb);
		}
	}

	/* show it */
	gtk_widget_show_all (GTK_WIDGET(tb));

	/* add it to the per-browser list of toolbars */
	browser->bookmarks_toolbars = 
		g_list_append (browser->bookmarks_toolbars, tb);
}

/**
 * bookmarks_create_toolbar_from_bm:
 *
 * drag_info should not be freed
 *
 */
void
bookmarks_create_toolbar_from_bm (GaleonBrowser *browser, GtkToolbar *tb,
				  BookMarkItem *b, gint insert_pos)
{
	GtkWidget *button, *hbox, *pixmap, *entry;
	GtkReliefStyle relief;
	GtkWidget *dnd_dest;
	BookMarkDragItem *drag_info = g_malloc0 (sizeof(BookMarkDragItem));

	/* get the button relief setting */
	relief = gtk_toolbar_get_button_relief (tb);

	/* make the right pixmap for the folder or bookmark */
	pixmap = ((b->type == CATEGORY) || (b->type == AUTOBOOKMARKS))
		? gtk_pixmap_new(folder_pixmap->pixmap, 
				 folder_pixmap->mask)
		: gtk_pixmap_new(site_pixmap->pixmap, 
				 site_pixmap->mask);

	/* build a box, and pack the pixmap and label inside */
	hbox = gtk_hbox_new (FALSE, 2);
	if (b->pixmap == NULL)
	{
		gtk_box_pack_start (GTK_BOX (hbox), pixmap, TRUE, FALSE, 0);
		gtk_box_pack_start (GTK_BOX (hbox), gtk_label_new (b->name),
				    TRUE, FALSE, 0);
	}
	else
	{
		GtkWidget *image;

		image = gtk_pixmap_new (b->pixmap->pixmap, b->pixmap->mask);
		gtk_box_pack_start (GTK_BOX (hbox), GTK_WIDGET (image),
				    TRUE, FALSE, 0);
	}

	/* is this a normal (i.e. non-smart) bookmark */
	if (b->url == NULL || strstr(b->url, "%s") == NULL)
	{
		/* yes, build a button and pack it */
		dnd_dest = button = gtk_button_new ();
		gtk_container_add(GTK_CONTAINER(button), hbox);
		if (insert_pos == -1)
			gtk_toolbar_append_widget(tb, button, b->url, NULL);
		else
			gtk_toolbar_insert_widget(tb, button, b->url, NULL,
						  insert_pos);
		gtk_signal_connect (GTK_OBJECT (button), "clicked", 
				    GTK_SIGNAL_FUNC 
				    (bookmarks_toolbar_clicked), b);
		gtk_signal_connect (GTK_OBJECT (button), 
				    "button_release_event", 
				    GTK_SIGNAL_FUNC 
				    (bookmarks_button_release_event), b);
		gtk_button_set_relief (GTK_BUTTON (button), relief);
		GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
	}
	else
	{
		GtkWidget *eb = gtk_event_box_new ();
		gtk_container_add (GTK_CONTAINER (eb), hbox);
		dnd_dest = eb;
		/* otherwise build an entry field and pack it */
		entry = gtk_entry_new ();
		gtk_toolbar_append_widget (tb, eb, b->url, NULL);
		gtk_toolbar_append_widget (tb, entry, b->url, NULL);
		gtk_toolbar_append_space (tb);
		gtk_object_set_data (GTK_OBJECT (entry), "mozilla", browser);
		gtk_signal_connect (GTK_OBJECT (entry), "activate",
				    bookmarks_nick_entry_activate, b);
	}

	/* dnd */
	drag_info->browser = browser;
	drag_info->bookmark = b;
	drag_info->toolbar = tb;
	gtk_signal_connect (GTK_OBJECT (dnd_dest), "drag_data_received",
		       GTK_SIGNAL_FUNC (bookmarks_toolbar_drag_data_received_cb),
		       drag_info);
	gtk_drag_dest_set (dnd_dest,
			   GTK_DEST_DEFAULT_ALL,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);
			   
	/* context menu */
	gtk_signal_connect (GTK_OBJECT (dnd_dest), "button-release-event",
		GTK_SIGNAL_FUNC (bookmarks_toolbar_item_button_release_event_cb),
		drag_info);				
}

void
bookmarks_create_menu_global (void)
{
	GList *sl;

	for (sl = all_windows; sl != NULL; sl = sl->next) 
	{
		/* Create the menus and the toolbars for this window */
		bookmarks_create_menu (sl->data);
	}
}

void
bookmarks_create_tb_global (void)
{
	GList *sl;

	for (sl = all_windows; sl != NULL; sl = sl->next) 
	{
		/* Create the menus and the toolbars for this window */
		bookmarks_create_tb (sl->data);
	}
}
	
void
bookmarks_remove_menu_global (void)
{
	GList *sl;

	/* Destroys the menus */
	for (sl = all_windows; sl != NULL; sl = sl->next) {
		GtkWidget *WMain = ( (GaleonBrowser*) sl->data)->WMain;
		GtkWidget *separator = glade_lookup_widget
			(WMain, "bookmarks_separator");
		GList *l = gtk_container_children (GTK_CONTAINER (
			glade_lookup_widget (WMain, "bookmarks_menu")));
		GList *li;
		/* first look for the separator */
		for (li = l; li != NULL; li = li->next) {
			if (li->data == separator) break;
		}
		/* then, destroy every widget after it */
		for (li = li->next; li != NULL; li = li->next) {
			if (GTK_IS_WIDGET (li->data))
			    gtk_widget_destroy (GTK_WIDGET (li->data));
		}
		g_list_free(l);
	}
}


void
bookmarks_remove_tb_global (void)
{
	GList *sl;
	GList *li;

	/* Destroys the toolbars */
	for (sl = all_windows; sl != NULL; sl = sl->next)
	{
		GList *toolbars = ( (GaleonBrowser*) sl->data)->bookmarks_toolbars;
		/* destroy every toolbar and its parent (the dock) */
		for (li = toolbars; li != NULL; li = li->next) {
			if (GTK_IS_WIDGET(li->data)) {
				GtkWidget *w = NULL;
				if (GTK_IS_WIDGET (GTK_WIDGET (li->data)->parent)) 
					w = GTK_WIDGET(li->data)->parent;
				gtk_widget_destroy (GTK_WIDGET (li->data));
				if (w) gtk_widget_destroy (w);
			}
		}
		g_list_free (toolbars);
		( (GaleonBrowser*) sl->data)->bookmarks_toolbars = NULL;
	}
}

/**
 * bookmarks_update_menu: destroy and recreate the bookmarks menu
 */
void 
bookmarks_update_menu (void) 
{
	bookmarks_remove_menu_global ();
	bookmarks_create_menu_global ();
}

/**
 * bookmarks_update_tb: destroy and recreate the boomarks toolbars
 */
void 
bookmarks_update_tb (void) 
{
	bookmarks_remove_tb_global ();
	bookmarks_create_tb_global ();
}

/**
 * Returns a menuitem with the appropiate icon
 */
/* This may be improved */
static GtkWidget *
bookmarks_create_menu_item (BookMarkItem *bm)
{
	GtkWidget *pixmap = (bm->type == CATEGORY || bm->type == AUTOBOOKMARKS)
		? gtk_pixmap_new (folder_pixmap->pixmap, 
				  folder_pixmap->mask)
		: gtk_pixmap_new (site_pixmap->pixmap, 
				  site_pixmap->mask);
	GtkWidget *m = gtk_pixmap_menu_item_new ();
	GtkWidget *l;
	GtkWidget *hb = gtk_hbox_new(FALSE, 0);

	g_return_val_if_fail (bm->name != NULL, NULL);

	if (bm->pixmap == NULL)
	{
		if (strlen (bm->name) > BOOKMARKS_MENU_MAX_LENGTH)
		{
			gchar *s, *t;

			s = g_strndup(bm->name, BOOKMARKS_MENU_MAX_LENGTH);
			t = g_strconcat (s, "...", NULL);
			l = gtk_label_new (t);
			g_free (t);
			g_free (s);
		} 
		else
		{
			l = gtk_label_new (bm->name);
		}
		gtk_box_pack_start (GTK_BOX (hb), l, FALSE, FALSE, 2);
	}
	else
	{
		GtkWidget *image;

		image = gtk_pixmap_new (bm->pixmap->pixmap, bm->pixmap->mask);
		gtk_box_pack_start (GTK_BOX (hb), GTK_WIDGET (image),
				    FALSE, FALSE, 2);
	}
	gtk_container_add (GTK_CONTAINER (m), hb);
	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (m), pixmap);
	gtk_widget_show_all (hb);
	if (gnome_preferences_get_menus_have_icons ())
		gtk_widget_show (pixmap);
	return m;
}

/**
 * Creates the file submenu for a bookmark catergory that has subcategories.
 * This functions does not fill the submenu.
 */
static void
bookmarks_create_file_submenu (GtkMenuItem *file_menuitem,
			       BookMarkItem *bookmark)
{
	GtkWidget *w;
	GtkWidget *file_menu = gtk_menu_new();

	gtk_menu_set_title (GTK_MENU (file_menu), bookmark->name);

	gtk_menu_item_set_submenu
		(GTK_MENU_ITEM (file_menuitem),
		 file_menu);
	
	w = bookmarks_create_menu_item (bookmark);
	gtk_signal_connect (GTK_OBJECT (w), "activate", 
			    bookmarks_file_bookmark_cb, bookmark);
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (file_menu), w);
	
	w = gtk_menu_item_new ();
	gtk_widget_show (w);
	gtk_menu_append (GTK_MENU (file_menu), w);

	if (gnome_preferences_get_menus_have_tearoff ()){
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_menu_prepend (GTK_MENU (file_menu), tearoff);
		gtk_widget_show (tearoff);
	}
}

/**
 * bookmarks_create_menu_recursively: Create menus for bookmarks
 * @bookmark: the root bookmark of the menu
 * @menu: the menu to fill with @bookmark children
 * @file_menuitem: the menu to fill with categories (to add bookmarks)
 * @tooltips: the tooltips object to use (or NULL if not needed)
 * 
 * Creates the bookmarks menu and the file submenu if necessary. If you don't
 * need to create a file submenu pass NULL as file_menuitem. This will also
 * disable tearoff menus (usefull for the toolbar menus).
 **/
void
bookmarks_create_menu_recursively (BookMarkItem *bookmark, GtkMenu *menu, 
				   GtkMenuItem *file_menuitem, GtkTooltips *tooltips)
{
	GList *l;
	
	g_return_if_fail (bookmark->type == CATEGORY || 
			  bookmark->type == AUTOBOOKMARKS);

	for (l = bookmark->list; l != NULL; l = l->next) 
	{
		BookMarkItem *b = l->data;
		GtkWidget *mi, *submenu, *file, *file_menu;
		switch (b->type) {
		case CATEGORY:
		case AUTOBOOKMARKS:
			submenu = gtk_menu_new ();
			gtk_menu_set_title (GTK_MENU (submenu), b->name);
			mi = bookmarks_create_menu_item (b);
			gtk_menu_item_set_submenu (GTK_MENU_ITEM(mi), submenu);
			if (file_menuitem != NULL) {
				file = bookmarks_create_menu_item (b);
				gtk_signal_connect (GTK_OBJECT(file), "activate", 
						    bookmarks_file_bookmark_cb, b);

				if (GTK_MENU_ITEM (file_menuitem) -> submenu == NULL) {
					gtk_signal_disconnect_by_data
						(GTK_OBJECT (file_menuitem), bookmark);
					bookmarks_create_file_submenu
						(file_menuitem, bookmark);
				}

				file_menu = GTK_MENU_ITEM (file_menuitem) -> submenu;
				gtk_widget_show (file);
				gtk_menu_append (GTK_MENU (file_menu), file);

				bookmarks_create_menu_recursively
					(b, GTK_MENU (submenu), GTK_MENU_ITEM (file), 
					 tooltips);
			} else {
				bookmarks_create_menu_recursively
					(b, GTK_MENU (submenu), NULL, tooltips);
			}
			break;
		case SITE:
			mi = bookmarks_create_menu_item (b);
			gtk_signal_connect (GTK_OBJECT (mi), "activate", 
					    window_menu_bookmark_activate_cb, b);
			gtk_signal_connect 
				(GTK_OBJECT (mi), "button_release_event", 
				 GTK_SIGNAL_FUNC (
					 bookmarks_button_release_event), b);
			if (tooltips)
				gtk_tooltips_set_tip 
					(tooltips, GTK_WIDGET (mi), b->url, NULL);
			break;
		case SEPARATOR:
		default:
			mi = gtk_menu_item_new ();
			break;
		}
		gtk_widget_show (mi);
		gtk_menu_append (menu, mi);
	}
	if (gnome_preferences_get_menus_have_tearoff ()
	    && (bookmark != bookmarks_root)
	    && (file_menuitem != NULL)) {
		GtkWidget *tearoff = gtk_tearoff_menu_item_new ();
		gtk_menu_prepend (GTK_MENU (menu), tearoff);
		gtk_widget_show (tearoff);
	}
}

void
bookmarks_create_menu (GaleonBrowser *browser) 
{
	GtkWidget *app = browser->WMain;
	GtkMenu *menu = GTK_MENU
		(glade_lookup_widget (app, "bookmarks_menu"));
	GtkMenuItem *file_menuitem = GTK_MENU_ITEM 
		(glade_lookup_widget (app, "file_bookmark"));
	if (browser->bookmarks_tooltips) 
		gtk_object_destroy (GTK_OBJECT (browser->bookmarks_tooltips));
	browser->bookmarks_tooltips = gtk_tooltips_new ();

	bookmarks_create_file_submenu (file_menuitem, bookmarks_root);
	bookmarks_create_menu_recursively (bookmarks_root, menu, file_menuitem,
					   browser->bookmarks_tooltips);
}

void 
bookmarks_create_tb (GaleonBrowser *browser)
{
	gboolean visible = GTK_CHECK_MENU_ITEM 
		(glade_lookup_widget (browser->WMain, "view_toolbar"))->active;
	bookmarks_create_toolbars (browser, bookmarks_root);
	bookmarks_toolbar_set_visible (browser, visible);
}

/**
 * Renames a bookmark (wich has just been added) or deletes it if the user cancelled 
 */
void 
bookmarks_string_request_callback (gchar *string, BookMarkItem *b) 
{
	if (!string) { /* User canceled */
		bookmarks_remove_recursively (b);
	} else { 
		g_free (b->name);
		b->name = string;
		
		if (strlen (b->name) == 0) {
			g_free (b->name);
			b->name = g_strdup ( _("Untitled"));
		}
		
		if (bookmarks_editor)
			bookmarks_editor_place_tree_item (bookmarks_editor, b);
		bookmarks_update_menu ();
		bookmarks_save (); 
	}
}

/**
 * bookmarks_move_bookmark: Moves a bookmark
 * @controls: the controls of the editor
 * @b: the bookmark to move
 * @where: if 0 then move up, else move down
 * Returns: 1 if bookmark is now first or last in the list, 0 otherwise
 *
 * Moves a bookmark up or down and updates both the edition ctree and the 
 * bookmarks structure 
 **/
gint
bookmarks_move_bookmark (bookmarks_editor_controls *controls, BookMarkItem *b, int where)
{
	BookMarkItem *parent, *grandparent, *other;
	GList *pos, *other_pos;
	
	parent = b->parent;
	if (!parent)
		return 1;

	grandparent = parent->parent;
	pos = g_list_find (parent->list, b);

	if (!where) /* up */
		other_pos = g_list_previous(pos);
	else /* down */
		other_pos = g_list_next(pos);
	
	if (other_pos) {
		other = other_pos->data;
		if (other->type == CATEGORY) {
			/* expand the category and move b into it */
			parent->list = g_list_remove_link (parent->list, pos);
			gtk_ctree_expand (GTK_CTREE (controls->ctree), other->tree_item);
			
			if (!where) { 
				/* up */
				other->list = g_list_append (other->list, b);
				b->parent = other;
			} else {
				/* down */
				b->parent = other;
				other->list = g_list_prepend (other->list, b);
			}
		} else {
			gint newindex = g_list_position (parent->list, other_pos);
			b->parent->list = g_list_remove_link (b->parent->list, pos);
			b->parent->list = g_list_insert (parent->list, b, newindex);
		}
	} else {
		/* move b to its grandparent*/
		if  (!grandparent)
			return 1;
		
		parent->list = g_list_remove_link(parent->list, pos);

		if (!where) {
			/* up */
			grandparent->list = g_list_insert 
				(grandparent->list, b, g_list_position 
				 (grandparent->list, g_list_find(grandparent->list, 
								 parent)));
			b->parent = grandparent;
		} else {
			/* down */
			GList *next = g_list_find(grandparent->list, parent)->next;
			grandparent->list = g_list_insert (grandparent->list, b, 
							   g_list_position 
							   (grandparent->list, next));
			b->parent = grandparent;
		}
	}
	bookmarks_editor_place_tree_item (controls, b);
	gtk_ctree_select ( GTK_CTREE (controls->ctree), b->tree_item);

	return 0;
}

/**
 * Go to the site of the clicked bookmark or open a popup menu if it's
 * a category
 */
static void 
bookmarks_toolbar_clicked (GtkButton *button, gpointer user_data)
{
	GtkWidget *WMain = glade_lookup_widget (GTK_WIDGET (button),"WMain");
	GaleonBrowser *browser =  gtk_object_get_data(GTK_OBJECT(WMain),"mozilla");
	BookMarkItem *bi = user_data;
	if (bi->type == SITE) { 
		browser_load_url(browser, bi->url);
	} else if ((bi->type == CATEGORY) || (bi->type == AUTOBOOKMARKS)) {
		GtkWidget *menu = gtk_menu_new();
		GtkTooltips *tooltips = gtk_tooltips_new ();
		
		bookmarks_create_menu_recursively(bi, GTK_MENU (menu), NULL, tooltips);
		
		/* see glade_lookup_widget for an explanation of the following */
		gtk_object_set_data (GTK_OBJECT(menu), "widget_tree", 
				     glade_get_widget_tree(WMain));
		gnome_popup_menu_do_popup_modal(menu, menu_position_under_widget,
						button, NULL, NULL);
		gtk_widget_destroy (menu);
		gtk_object_destroy (GTK_OBJECT (tooltips));
	}
}

/**
 * Open the bookmark in a new window if it's a site and the user clicked with button 2
 **/
static gboolean
bookmarks_button_release_event (GtkWidget *widget, GdkEventButton *event,
				gpointer user_data)
{
	if (event->button == 2) {
		BookMarkItem *bi = user_data;
		if (bi->type == SITE) {
			browser_create_from_url (bi->url);
			return TRUE;
		}
	}
	return FALSE;
}

/**
 * add_temp_bookmark: Adds a temporal bookmark
 * @type: The type of the bookmark
 * @name: The name of the bookmark, if NULL or "" the url will be used instead.
 * @url: The url of the bookmark
 * @parent: The parent bookmark or NULL if the bookmark should be added to the 
 * main temp category
 * Return value: The new bookmark
 **/
BookMarkItem *
add_temp_bookmark (BookMarkType type, const gchar *name, const gchar *url, 
		   BookMarkItem *parent)
{
	BookMarkItem *bookmark;

	bookmark = bookmarks_new_bookmark (type, name, url, NULL, NULL, NULL);

	if (!temp_bookmarks_root) {
		temp_bookmarks_init ();
	}
	if (!parent) parent = temp_bookmarks_root;
	
	bookmark->parent = parent;
	parent->list = g_list_append (parent->list, bookmark);
	bookmarks_editor_place_tree_item (temp_bookmarks_window, bookmark);
	temp_bookmarks_window->dirty = TRUE;

	/* Save the temp bookmarks, for safety in case of a crash */
	temp_bookmarks_save ();

	return bookmark;
}

/**
 * Inits the temp bookmarks
 */
void
temp_bookmarks_init (void)
{
	char *file = g_strconcat (g_get_home_dir(), "/.galeon/temp_bookmarks.xml", NULL);
	temp_bookmarks_root = bookmarks_load_from (file);
	if (!temp_bookmarks_root) 
		temp_bookmarks_root = bookmarks_new_bookmark 
			(CATEGORY, _("Temporary bookmarks"), 
			 NULL, NULL, NULL, NULL);
	temp_bookmarks_window = bookmarks_editor_controls_init 
		(temp_bookmarks_root, COMPACT_BOOKMARKS_EDITOR);
	temp_bookmarks_window->save_function = temp_bookmarks_save;
}

void
temp_bookmarks_save (void)
{
	char *file = g_strconcat (g_get_home_dir(), "/.galeon/temp_bookmarks.xml", NULL);
	if (temp_bookmarks_root && temp_bookmarks_window 
	    && temp_bookmarks_window->dirty) {
		bookmarks_save_as (temp_bookmarks_root, file);
		temp_bookmarks_window->dirty = FALSE;
	}
	g_free (file);
}

/**
 * bookmarks_list_all_urls:
 * -----------------------
 * get a list of all bookmarked URLs
 */
GSList *
bookmarks_list_all_urls (BookMarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == CATEGORY, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookMarkItem *b = l->data;
		if (b->type == CATEGORY) {
			g_slist_concat(result, bookmarks_list_all_urls(b));
		} else
			if (b->type == SITE) {
				g_slist_append(result, b->url);
			}
	}
	return result;
}


/**
 * get a singly linked list of all the category/folder bookmarks 
 */
GSList * 
bookmarks_list_all_folders (BookMarkItem *bookmark)
{
	GList *l;
	GSList * result;

	g_return_val_if_fail(bookmark != NULL, NULL);
	g_return_val_if_fail(bookmark->type == CATEGORY, NULL);

	result = g_slist_alloc();

	for (l = bookmark->list; l != NULL; l = l->next)
	{
		BookMarkItem *b = l->data;
		if (b->type == CATEGORY) {
			g_slist_append(result, b);
			g_slist_concat(result, bookmarks_list_all_folders(b));
		}
	}
	return result;
}

/**
 * bookmarks_new_bookmark: Creates a new allocated BookMarkItem
 * @type: the type of the bookmark
 * @name: the name, if NULL the url is used
 * @url: the url (ignored if type != SITE)
 * @nick: the nickname for the bookmark, if any
 * @notes: the comments aout the bookmark
 * 
 * This function allocates a BookMarkItem and initializes its fields to sane values.
 * Use it instead of g_new (BookMarkItem, 1) and manually initializing the fields. 
 * The returned BookMarkItem has it's parent set to NULL, so you have to add it to 
 * the bookmarks list yourself (if you need it)
 * All the strings passed to this function are g_strduped
 * 
 * Return value: The new BookMarkItem
 **/
BookMarkItem *
bookmarks_new_bookmark (BookMarkType type, const char *name, const char *url, 
			const char *nick, const char *notes, 
			const char *pixmap_file)
{
	BookMarkItem *bookmark;

	g_return_val_if_fail (!((type == SITE) && (url == NULL)), NULL);
	g_return_val_if_fail (!((type == CATEGORY) && (name == NULL)), NULL);

	bookmark = g_new0 (BookMarkItem, 1);
	bookmark->type = type;

	switch (bookmark->type) {
	case AUTOBOOKMARKS:
	case CATEGORY:
		bookmark->name = g_strdup (name);
		bookmark->url = NULL;
		break;
	case SITE:
		if (name) 
			bookmark->name = g_strdup (name);
		else
			bookmark->name = g_strdup (url);
		bookmark->url = g_strdup (url);
		break;
	default:
		bookmark->name = bookmark->url = NULL;
		break;
	}
	bookmark->list = NULL;
	bookmark->tree_item = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	if (nick) 
		bookmark->nick = g_strdup (nick);
	else
		bookmark->nick = g_strdup ("");
	if (notes)
		bookmark->notes = g_strdup (notes);
	else
		bookmark->notes = g_strdup ("");
	if (pixmap_file != NULL && strlen(pixmap_file) != 0)
	{
		bookmark->pixmap_file = g_strdup (pixmap_file);
		bookmark->pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_file 
						 (pixmap_file));
		if (bookmark->pixmap->pixmap == NULL)
		{
			gtk_widget_destroy (GTK_WIDGET (bookmark->pixmap));
			bookmark->pixmap = NULL;
		}
	}
	else
	{
		bookmark->pixmap_file = g_strdup ("");
		bookmark->pixmap = NULL;
	}
	bookmark->expanded = TRUE;
	return bookmark;
}

/**
 * bookmarks_copy_bookmark: copy a bookmark
 * @b: the bookmark
 * 
 * copy a bookmarks and its children
 * 
 * Return value: the new bookmark
 **/
BookMarkItem *
bookmarks_copy_bookmark (BookMarkItem *b)
{
	GList *l;
	BookMarkItem *new = bookmarks_new_bookmark (b->type, b->name, b->url,
						    b->nick, b->notes, 
						    b->pixmap_file);
	
	new->create_toolbar = b->create_toolbar;
	new->expanded = b->expanded;
	if (b->pixmap != NULL)
	{
		new->pixmap = GNOME_PIXMAP (gnome_pixmap_new_from_gnome_pixmap 
					    (b->pixmap));
	}
	if (b->type == CATEGORY) {
		for (l = b->list; l != NULL; l = g_list_next (l)) {
			BookMarkItem *child = l->data;
			BookMarkItem *new_child = bookmarks_copy_bookmark (child);
			new_child->parent = new;
			new->list = g_list_append (new->list, new_child);
		}
	}
	return new;
}

/**
 * bookmarks_free_bookmark: free a bookmark
 * @b: the bookmark to be freed, if NULL thenb do nothing
 * 
 * Frees the bookmarks and all the strings referenced by it and the list of children 
 * if it's a CATEGORY (but does not free the children). Use it instead of g_free. 
 **/
void
bookmarks_free_bookmark (BookMarkItem *b)
{
	if (! b) return;

	if (b->name) g_free (b->name);
	if (b->url) g_free (b->url);
	if (b->nick) g_free (b->nick);
	if (b->pixmap_file) g_free (b->pixmap_file);
	/* FIXME: free pixmap, although I'm not sure about ref counting... */
	if (b->notes) g_free (b->notes);
	if (b->list) g_list_free (b->list);

	g_free (b);
}

/**
 * bookmarks_parse_nick: returns the url of a given nick with one arg
 * @text: the text to try to parse
 * @root: the root bookmark for start the search, if NULL the usual tree will be used
 * 
 * Tries to find a bookmarkitem wich has a nick like the first word of @text and if 
 * found return the URL result of substituting each %s in the BookmarkItem url by the 
 * text after the first " " in @text
 * 
 * Return value: the resulting URL or NULL if not matching nick found
 **/
gchar *
bookmarks_parse_nick (gchar *text, BookMarkItem *root)
{
	gchar *nick;
	gchar *args;
	gchar *url;
	gchar *space;
	BookMarkItem *b;

	/* remove any leading spaces */
	while (*text == ' ')
		text++;
	if (*text == '\0') 
		return NULL;

	/* find the nick */
	space = strstr (text, " ");
	if (space) 
	{
		/* split out nickname and arguments */
		nick = g_strndup (text, space - text);
 		args = g_strdup (space + 1);

		/* find bookmark */
		b = bookmarks_find_by_nick (nick, root);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* do substitution */		
		url = bookmarks_substitute_argument (b, args);
		g_free (args);

		/* return completed url */
		return url;
	}
	else
	{
		/* the whole thing is the nick */
		nick = g_strdup (text);

		/* find bookmark */
		b = bookmarks_find_by_nick (nick, root);
		g_free (nick);

		/* check we found it */
		if (b == NULL)
			return NULL;

		/* return copy of the url */
		return g_strdup(b->url);
	}
}

/**
 * bookmarks_substitute_argument:
 **/
gchar *
bookmarks_substitute_argument (BookMarkItem *b, gchar *arguments)
{
	gchar *ret = NULL;
	gchar *s = b->url;
	int expected_args = 0;

	/* count %s in the URL */
	while ((s = strstr(s, "%s"))) {
	    ++expected_args;
	    ++s;
	}
	
	ret = g_strdup (b->url);

	if (expected_args <= 1) {
	    /*  do the substitution */
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, arguments, s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	}
	else {
	    /* try to substitute matching arguments */
	    gchar **args;
	    int arg = 0;
	    args = g_strsplit(arguments, " ", -1);
	    while ( (s = strstr (ret, "%s")) ) {
		gchar *new1, *new2;
		if (!args[arg])
		    break;
		new1 = g_strndup (ret, s - ret);
		new2 = g_strconcat (new1, args[arg++], s + 2, NULL);
		g_free (new1);
		g_free (ret);
		ret = new2;
	    }
	    g_strfreev(args);
	}

	return ret;
}

/**
 * bookmarks_find_by_nick: search for a BookMarkItem with a nick
 * @nick: the nick to search for.
 * @r: the root bookmark for start the search, if NULL the usual tree will be used
 * 
 * Search a bookmarks category recursivley for a bookmark with the given nick
 * 
 * Return value: the found BookMarkItem or NULL if nothing found
 **/
static BookMarkItem *
bookmarks_find_by_nick (const gchar *nick, BookMarkItem *r)
{
	GList *l;
	BookMarkItem *b;
	if (!r) r = bookmarks_root;

	if ((r->nick) && !strcmp (r->nick, nick)) return r;
	if (r->type == CATEGORY)
		for (l = r->list; l != NULL; l = g_list_next (l)) {
			b = bookmarks_find_by_nick (nick, l->data);
			if (b) return b;
		}
	return NULL;
}

bookmarks_editor_controls *
bookmarks_editor_controls_init (BookMarkItem *root, bookmarks_editor_controls_type type)
{
	GladeXML *gxml = NULL;
	GtkWidget *bookmarks_editor_edit_controls;
	bookmarks_editor_controls *editor = g_new0 (bookmarks_editor_controls, 1);	

	bookmarks_load_icons ();
	
	editor->root_bookmark = root;

	switch (type) {
	case STANDARD_BOOKMARKS_EDITOR:
		gxml = glade_xml_new (glade_file (), "bookmarks_editor");
		editor->dialog = glade_xml_get_widget 
			(gxml, "bookmarks_editor");
		editor->ctree = glade_xml_get_widget
			(gxml, "bookmarks_editor_ctree");
		editor->vpane = glade_xml_get_widget
			(gxml, "vpaned1");
		editor->edit_frame = glade_xml_get_widget
			(gxml, "bookmarks_editor_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_url_entry");
		editor->nick_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_nick_entry");
		editor->pixmap_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_entry");
		editor->pixmap_file_entry = glade_xml_get_widget
			(gxml, "bookmarks_editor_pixmap_file_entry");
		editor->create_toolbar_toggle = glade_xml_get_widget
			(gxml, "bookmarks_editor_create_toolbar_toggle");
		editor->notes_text = glade_xml_get_widget
			(gxml, "bookmarks_editor_notes_text");
		g_assert (editor->dialog != NULL);
		g_assert (editor->edit_frame != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry != NULL);
		g_assert (editor->pixmap_entry != NULL);
		g_assert (editor->pixmap_file_entry != NULL);
		g_assert (editor->create_toolbar_toggle != NULL);
		g_assert (editor->notes_text != NULL);
		break;
	case COMPACT_BOOKMARKS_EDITOR:
		gxml = glade_xml_new (glade_file (), "compact_bookmarks_editor");
		editor->dialog = glade_xml_get_widget 
			(gxml, "compact_bookmarks_editor");
		editor->ctree = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_ctree");
		editor->edit_frame = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_edit_frame");
		editor->name_entry = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_name_entry");
		editor->url_entry = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_url_entry");
		editor->nick_entry = NULL;
		editor->pixmap_entry = NULL;
		editor->pixmap_file_entry = NULL;
		editor->create_toolbar_toggle = NULL;
		editor->notes_text = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_notes_text");
		g_assert (editor->dialog != NULL);
		g_assert (editor->edit_frame != NULL);
		g_assert (editor->name_entry != NULL);
		g_assert (editor->url_entry != NULL);
		g_assert (editor->nick_entry == NULL);
		g_assert (editor->pixmap_entry == NULL);
		g_assert (editor->pixmap_file_entry == NULL);
		g_assert (editor->create_toolbar_toggle == NULL);
		g_assert (editor->notes_text != NULL);
		bookmarks_editor_edit_controls = glade_xml_get_widget
			(gxml, "compact_bookmarks_editor_control_buttons");
		gtk_toolbar_set_style (GTK_TOOLBAR (bookmarks_editor_edit_controls),
				       GTK_TOOLBAR_ICONS);
		break;
	default:
		g_error ("Incorrect bookmarks editor dialog type");
	}
	g_assert (editor->ctree != NULL);
	
	editor->selection = NULL;
	editor->last_pressed = NULL;
	editor->lock_edit_controls = FALSE;
	editor->dirty = FALSE;
	editor->browser = NULL;
	editor->save_function = NULL;

	glade_xml_signal_autoconnect_full 
		(gxml, (GladeXMLConnectFunc) glade_signal_connect_func, editor);

	/* unref the GladeXML object, we don't use it anymore */
	gtk_object_unref (GTK_OBJECT (gxml));

	gtk_drag_source_set (editor->ctree, GDK_BUTTON1_MASK, 
			     bookmarks_dnd_targets, 
			     bookmarks_dnd_targets_num_items, 
			     GDK_ACTION_COPY | GDK_ACTION_MOVE |
			     GDK_ACTION_LINK);
	gtk_drag_dest_set (editor->ctree,
			   GTK_DEST_DEFAULT_DROP,
			   bookmarks_dnd_targets,
			   bookmarks_dnd_targets_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK);

	if (editor->pixmap_entry != NULL)
	{
		gnome_pixmap_entry_set_pixmap_subdir 
			(GNOME_PIXMAP_ENTRY (editor->pixmap_entry), SHARE_DIR);
	}

	bookmarks_editor_place_tree_item (editor, root);
	return editor;
}

GtkMenu*
bookmarks_create_copy_menu (bookmarks_editor_controls *destination_window, 
			    BookMarkItem *source)
{
	GtkMenu *res;
	g_assert (destination_window->root_bookmark->type == CATEGORY);
	res = bookmarks_create_copy_menu_recursively (destination_window, source,
						      destination_window->root_bookmark);
	if (! res) {
		BookMarkItem *item = destination_window->root_bookmark;
		GtkWidget *menu_item = bookmarks_create_menu_item (item);
		res = GTK_MENU (gtk_menu_new ());
		gtk_widget_show (menu_item);
		gtk_menu_prepend (res, menu_item);
		gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
				    bookmarks_editor_bookmark_copy_cb, item);
		gtk_object_set_user_data (GTK_OBJECT (menu_item), 
					  source);
	} else {
		BookMarkItem *item = destination_window->root_bookmark;
		GtkWidget *menu_item = bookmarks_create_menu_item (item);
		GtkWidget *sep = gtk_menu_item_new ();
		gtk_widget_show (menu_item);
		gtk_widget_show (sep);
		gtk_menu_prepend (res, sep);
		gtk_menu_prepend (res, menu_item);
		gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
				    bookmarks_editor_bookmark_copy_cb, item);
		gtk_object_set_user_data (GTK_OBJECT (menu_item), 
					  source);
	}
	return res;
}

GtkMenu*
bookmarks_create_copy_menu_recursively (bookmarks_editor_controls *destination_window, 
					BookMarkItem *source, BookMarkItem *b)
{
	GtkMenu *res = GTK_MENU (gtk_menu_new ());
	GList *l;
	gboolean has_sub_categories = FALSE;
	g_assert (b->type == CATEGORY);
	for (l = b->list; l != NULL; l = g_list_next (l)) {
		BookMarkItem *item = l->data;
		GtkWidget *menu_item, *sub_menu_item, *sep;
		if (item->type == CATEGORY) {
			GtkMenu *submenu = 
				bookmarks_create_copy_menu_recursively 
				(destination_window, source, item);
			has_sub_categories = TRUE;
			sub_menu_item = bookmarks_create_menu_item (item);
			if (submenu) {
				gtk_menu_item_set_submenu 
					(GTK_MENU_ITEM (sub_menu_item), 
					 GTK_WIDGET (submenu));
				menu_item = bookmarks_create_menu_item (item);
				gtk_widget_show (menu_item);
				sep = gtk_menu_item_new ();
				gtk_widget_show (sep);
				gtk_menu_prepend (submenu, sep);
				gtk_menu_prepend (submenu, menu_item);
				gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
						    bookmarks_editor_bookmark_copy_cb, item);
				gtk_object_set_user_data (GTK_OBJECT (menu_item), 
							  source);
			} else {
				gtk_signal_connect (GTK_OBJECT (sub_menu_item), 
						    "activate", bookmarks_editor_bookmark_copy_cb, item);
				gtk_object_set_user_data (GTK_OBJECT (sub_menu_item), 
							  source);
			}
			gtk_widget_show (sub_menu_item);
			gtk_menu_append (res, sub_menu_item);
		}

	}
	if (has_sub_categories) {
		return res;
	} else {
		gtk_widget_destroy (GTK_WIDGET (res));
		return NULL;
	}
}

void 
bookmarks_toolbar_set_visible (GaleonBrowser *browser, gboolean visible)
{
	GList *l;
	for (l = browser->bookmarks_toolbars; l != NULL; l = l->next) {
		if (GTK_IS_WIDGET (l->data)) {
			GtkWidget *w = NULL;
			if (GTK_IS_WIDGET (GTK_WIDGET (l->data)->parent)) {
				w = GTK_WIDGET (l->data)->parent;
				if (visible) 
					gtk_widget_show (w);
				else
					gtk_widget_hide (w);
			}
		}
	}
}	

/**
 * bookmarks_save_as: convert a bookmark to a string
 * @bookmark: the bookmark to be saved (can be a category)
 *
 * Return value: the bookmarks as a string.
 **/
gchar * 
bookmarks_item_to_string (BookMarkItem *bookmark)
{
	xmlNodePtr node;
	xmlDocPtr doc = xmlNewDoc ("1.0");
	xmlChar *mem;
	int size;

	node = xmlNewDocNode (doc, NULL, "single_bookmark", NULL);

	bookmarks_save_recursively (doc, node, bookmark);

	xmlDocSetRootElement (doc, node);

	xmlDocDumpMemory (doc, &mem, &size);

	xmlFreeDoc (doc);

	return mem;
}

/**
 * bookmarks_load_from_string: Read a bookmark from a string
 * @string: the string to parse.
 * Return value: The new allocated bookmark or NULL if error
 **/
BookMarkItem *
bookmarks_item_from_string (char *string)
{
	xmlDocPtr doc;
	xmlNodePtr item;
	BookMarkItem *b = NULL;

	doc = xmlParseMemory (string, strlen (string));	
	
	if (doc) {
		item = doc->root->childs;
		b = bookmarks_xml_read_item (item);
		xmlFreeDoc (doc);
	} else {
		g_warning ("Error parsing: %s", string);
	}
	return b;
}
