/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* local function prototyprs */
gint window_back_forward_button_down_cb(GaleonBrowser *browser);
void window_menu_select_dir_activate_cb (GtkMenuItem *menuitem, GaleonBrowser *browser);
void window_menu_prefs_activate_cb (GtkMenuItem *menuitem, GaleonBrowser *browser);
gint window_zoom_spin_timeout_cb (GaleonBrowser *browser);
void window_entry_changed_cb (GtkEditable *editable, gboolean *changed); 
void window_location_gnomeentry_popwin_cb (GtkWidget *widget, GaleonBrowser *browser);
void window_menu_back_activate_cb (GtkMenuItem *menuitem, GaleonBrowser *browser);
gboolean window_delete_cb (GtkWidget *widget, GdkEventAny *event, GaleonBrowser *browser);
void window_menu_view_source_activate_cb (GtkMenuItem *menuitem, GaleonBrowser *browser);
void window_menu_history_activate_cb (GtkMenuItem *menuitem, GaleonBrowser *browser);
void window_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
			      GtkSelectionData *selection_data, 
			      guint info, guint time, 
			      GaleonBrowser *browser);



GTimer *zoom_timer = NULL;
#define ZOOM_DELAY 0.75 

/**
 * window_selection_get_cb: get selection on copy link location
 */
void 
window_selection_get_cb (GtkWidget *widget, GtkSelectionData *selection_data,
			 guint info, guint time_stamp,
			 gpointer data )
{
	gchar *text;

	text = gtk_object_get_data (GTK_OBJECT(widget), "selection");

	g_return_if_fail (text != NULL);

	gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
				8, text, strlen (text));
}

/**
 * window_selection_received_cb: load url if received on middle button click
 */
void
window_selection_received_cb (GtkWidget *widget,
			      GtkSelectionData *selection_data,
			      guint time, GaleonBrowser *browser)
{
	if (selection_data->data)
		browser_load_url(browser, selection_data->data);
}

/**
 * window_delete_cb: deleting toplevel window
 */
gboolean
window_delete_cb (GtkWidget *widget, GdkEventAny *event, 
		  GaleonBrowser *browser)
{
#ifdef DEBUG_SIGNALS
	g_print("WMain_delete\n");
#endif
	/* location bar history is saved automatically when the widget is
	   destroyed. we don't want to save the text currently in the entry */
	gtk_editable_delete_text (GTK_EDITABLE (browser->toolbar_entry),
				  0, -1);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!GTK_CHECK_MENU_ITEM (browser->view_fullscreen)->active
	    && !browser->is_popup)
	{	
		gnome_config_set_int("/galeon/Appearance/winwidth", 
				     browser->WMain->allocation.width);
		gnome_config_set_int("/galeon/Appearance/winheight", 
				     browser->WMain->allocation.height);
	}

	/* everything else is handled the same way as a menu close */
	browser_close(browser);

	return TRUE;
}


/** 
 * window_back_forward_button_down_cb: Gtk+ interval function used for timer handling
 * This timeout handler is currently only enabled when the Back/Forward
 * buttons are held down 
 */
gint window_back_forward_button_down_cb(GaleonBrowser *browser)
{
#ifdef B_HISTORY
        if (BTimer != NULL) /* holding down the "Back" button? */
	{
                if (g_timer_elapsed(BTimer, NULL) >= B_HISTORY_SECS) {
			GtkMenu *menu;
			g_timer_stop(BTimer);
			g_timer_destroy(BTimer);
			BTimer = NULL;
			timertag = -1;

			menu = create_back_menu (browser);

			if (menu)
			{
				gnome_popup_menu_do_popup_modal
					(GTK_WIDGET(menu), menu_position_under_widget, 
					 browser->BBack,
					 NULL, NULL);
				gtk_widget_destroy(GTK_WIDGET(menu));
			}
			
			return TRUE;

                }
        }

        if (FTimer != NULL) /* holding down the "Forward" button? */
	{
	        if (g_timer_elapsed(FTimer, NULL) >= B_HISTORY_SECS) 
		{
			GtkMenu *menu;
			g_timer_stop(FTimer);
			g_timer_destroy(FTimer);
			FTimer = NULL;
			timertag = -1;

			menu = create_forward_menu (browser);

			if (menu)
			{
				gnome_popup_menu_do_popup_modal
					(GTK_WIDGET(menu), menu_position_under_widget, 
					 browser->BForward,
					 NULL, NULL);
				gtk_widget_destroy(GTK_WIDGET(menu));
			}

			return TRUE;
                }
	}
#endif
        return TRUE;  /* return FALSE to also disable this callback */
}


/** 
 * window_back_forward_button_press_cb:
 */
gboolean
window_back_forward_button_press_cb (GtkWidget *widget, GdkEventButton  *event, GaleonBrowser *browser)
{
	if (event->button == 3) {
		GtkMenu *menu = NULL;
		if (widget == browser->BForward) menu = create_forward_menu (browser);
		if (widget == browser->BBack) menu = create_back_menu (browser);
		g_assert (menu != NULL);
		gnome_popup_menu_do_popup_modal
			(GTK_WIDGET (menu), menu_position_under_widget, 
			 widget, NULL, NULL);
		gtk_widget_destroy (GTK_WIDGET (menu));
		return FALSE;
	}	
	return TRUE;
}

/** 
 * window_back_button_leave_cb: back button left
 */
void
window_back_button_leave_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
	if (BTimer != NULL)
	{
	       g_timer_stop(BTimer);
	       g_timer_destroy(BTimer);
	       BTimer = NULL;
	}

        if (timertag != -1)
	{
	        gtk_timeout_remove(timertag);
	        timertag = -1;
  	}
#endif
}

/** 
 * window_forward_button_leave_cb: forward button left
*/
void
window_forward_button_leave_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
       if (FTimer != NULL)
       {
               g_timer_stop(FTimer);
               g_timer_destroy(FTimer);
               FTimer = NULL;
       }

       if (timertag != -1)
       {
               gtk_timeout_remove(timertag);
               timertag = -1;
       }
#endif
}


/** 
 * window_start_button_clicked_cb: start button clicked
*/
void
window_start_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	browser_go_home(browser, FALSE);
}

/** 
 * window_start_button_press_cb:  Open home page in new window on middle click
 */
gboolean
window_start_button_press_cb (GtkWidget *widget, GdkEventButton  *event, GaleonBrowser *browser)
{
	if (event->button == 2)
		browser_go_home(browser, TRUE);
	
	return TRUE;
}


/*
 * How the Back/Forward history works - Nate Case <nd@kracked.com>
 *
 * This is implemented using some standard gtk+ callbacks and timers
 * with a pop-up menu.
 * 
 * The Gtk+ event "pressed" is activated as soon as the mouse clicks it,
 * without caring if they release the button somewhere else like "clicked".
 *
 * The Gtk+ event "leave" is activated when the mouse cursor leaves the
 * button area.
 * 
 * When "pressed" is activated, we immediately start a timer.  This timer
 * is reset and disabled when either "clicked" or "leave" signals.  If
 * the timer reaches X ms, then they have been holding the button down
 * for that amount of time and we pop-up a menu containing the history.
*/

/** 
 * window_back_button_clicked_cb: back button clicked
*/
void
window_back_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
	if (BTimer != NULL)
	{
		g_timer_stop(BTimer);
		g_timer_destroy(BTimer);
		BTimer = NULL;
	}

	if (timertag != -1)
	{
		gtk_timeout_remove(timertag);
		timertag = -1;
	}
#endif
	gtk_moz_embed_go_back(browser->mozEmbed);
}


/** 
 * window_forward_button_clicked_cb: forward button clicked
*/
void
window_forward_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
	if (FTimer != NULL)
	{
		g_timer_stop(FTimer);
		g_timer_destroy(FTimer);
		FTimer = NULL;
	}
	
	if (timertag != -1)
	{
		gtk_timeout_remove(timertag);
		timertag = -1;
	}
#endif

	gtk_moz_embed_go_forward(browser->mozEmbed);
}

/** 
 * window_back_button_pressed_cb: back button pressed
*/
void
window_back_button_pressed_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
	if (BTimer == NULL)
	{
	        BTimer = g_timer_new();
	        g_timer_start(BTimer);
	        if (timertag == -1)
	                timertag = gtk_timeout_add(100, window_back_forward_button_down_cb, (gpointer) browser);
	}
#endif
}

/** 
 * window_forward_button_pressed_cb: forward button pressed
*/
void
window_forward_button_pressed_cb (GtkButton *button, GaleonBrowser *browser)
{
#ifdef B_HISTORY
	if (FTimer == NULL)
	{
	        FTimer = g_timer_new();
	        g_timer_start(FTimer);
 	        if (timertag == -1)
	        	timertag = gtk_timeout_add(100, window_back_forward_button_down_cb, (gpointer) browser);
	}
#endif
}

/** 
 * window_refresh_button_clicked_cb: refresh button clicked
*/
void
window_refresh_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	browser_reload (browser);
}


/** 
 * window_stop_button_clicked_cb: stop button clicked
 */
void
window_stop_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
  gtk_moz_embed_stop_load(browser->mozEmbed);
}


/** 
 * window_location_entry_key_press_cb: key pressed in the url entry
 */
gboolean
window_location_entry_key_press_cb (GtkWidget *widget, GdkEventKey *event,
				    GaleonBrowser *browser)
{
	static gchar *before_completion = NULL;
	GtkEntry *entry = GTK_ENTRY(widget);
	GtkEditable *editable = GTK_EDITABLE(widget);
	gboolean url_dialog = FALSE;

	g_return_val_if_fail(GTK_IS_ENTRY(widget), TRUE);

	/* check to see if key press is in the window's location bar or
	   the open url dialog */
	if (widget != browser->toolbar_entry)
	{
//		g_print("url dialog\n");
		url_dialog = TRUE;
	}
	auto_completion_reset();
	if ((event->keyval == GDK_Return) || (event->keyval == GDK_KP_Enter))
	{
		if (!url_dialog)
			window_go_button_clicked_cb (NULL, browser);
		return TRUE;
	}

	if ((event->keyval == GDK_Left) || (event->keyval == GDK_Right)) {
		return TRUE;
	}

	if ( ( ! ( (event->state & GDK_SHIFT_MASK) ||
		   (event->state & GDK_CONTROL_MASK) ||
		   (event->state & GDK_MOD1_MASK) ) 
	      && (event->keyval == GDK_End))) {
		gtk_editable_select_region(editable, 0, 0); 
		gtk_editable_set_position(editable, -1);
		return TRUE;
	}
	
	if (event->keyval == GDK_Tab) {
		gchar *common_prefix = NULL;
		gchar *text;

		gtk_editable_delete_selection(editable);
		text = gtk_editable_get_chars(editable, 0, -1);
		common_prefix = auto_completion_complete_url_extended(text);
		if (common_prefix) {
			if (!before_completion) 
				before_completion = g_strdup(text);

			gtk_entry_set_text(entry, common_prefix);
//			auto_completion_display(browser);			
			auto_completion_display_alternatives(browser, widget);
			g_free(common_prefix);
		}
		return TRUE;
	}

	if (event->keyval == GDK_Escape) {
		auto_completion_reset();
		if (before_completion) {
			gtk_entry_set_text(entry, before_completion);
			g_free(before_completion);
			before_completion = NULL;
		} else
			gtk_entry_set_text(entry,
				gtk_moz_embed_get_location(browser->mozEmbed));
		return TRUE;
	}

	/* do not toggle statusbar on Ctrl+U, since this is an editing
	   shortcut in GtkEntry */
	if ((event->state & GDK_Control_L || event->state & GDK_Control_R) 
	    && (event->keyval == GDK_U || event->keyval == GDK_u))
		return TRUE;

	if (event->string[0] > 32 && event->string[0] < 126) 
	{
		if (before_completion != NULL) 
		{
			g_free(before_completion);
			before_completion = NULL;
		}
		auto_completion_display(browser, widget);
		return TRUE;
	}

	return FALSE;
}

/** 
 * window_go_button_clicked_cb: go button clicked
*/
void
window_go_button_clicked_cb (GtkButton *button, GaleonBrowser *browser)
{
	gchar *text = gtk_editable_get_chars (GTK_EDITABLE (browser->toolbar_entry), 0, -1);
	gchar *text2;
	GList *wl;
	GnomeEntry *ge;

	if (strcmp(text,"")!=0)
	{
		/* don't save the text in the entry - avoids duplicate entries */
		gtk_editable_delete_text(GTK_EDITABLE(browser->toolbar_entry),
					 0, -1);
		ge = GNOME_ENTRY(browser->toolbar_gnomeentry);
		gnome_entry_save_history(ge);
		gtk_entry_set_text(GTK_ENTRY(browser->toolbar_entry), text);

		/* add item to location bars in all other open browser windows */
		for (wl = all_windows; wl != NULL; wl = wl->next)
		{
			GtkList *list;
			GList *items;
			gchar *label;
			gboolean dup = FALSE;

			if (wl->data == browser)
				continue;

			ge = GNOME_ENTRY(((GaleonBrowser *)
					   (wl->data))->toolbar_gnomeentry);
			list = GTK_LIST(ge->combo.list);
			for (items = list->children; items != NULL; items = items->next)
			{
				gtk_label_get(GTK_LABEL(
						GTK_BIN(items->data)->child),
					      &label);
				if (!strcmp(label, text))
					dup = TRUE;
			}
			if (!dup)
				gnome_entry_prepend_history(ge, TRUE, text);
		}

		text2 = bookmarks_parse_nick (text, NULL);

		if (text2) 
		{
			g_free (text);
			text = text2;
		}
	
		browser_load_url (browser, text);
		g_free (text);
	}
}

/**
 * Changes the zoom if enough time time has passed 
 */
gint 
window_zoom_spin_timeout_cb (GaleonBrowser *browser)
{
	if (zoom_timer == NULL) return FALSE;
	if (g_timer_elapsed (zoom_timer, NULL) >= ZOOM_DELAY) {
		gint value = atoi (gtk_editable_get_chars (GTK_EDITABLE (browser->zoom_spin), 0, -1));
		g_timer_destroy (zoom_timer);
		zoom_timer = NULL;
		mozilla_set_zoom (browser, (float) value / 100);
		return FALSE;
	}
	return TRUE;
}

/** 
 * window_zoom_spin_changed_cb: zoom spin value changed. Starts the zoom timer
 */
void 
window_zoom_spin_changed_cb (GtkEditable *editable, GaleonBrowser *browser)
{
	if (zoom_timer != NULL) g_timer_destroy (zoom_timer);
	zoom_timer = g_timer_new();
	g_timer_start (zoom_timer);
	g_timeout_add (50, (GSourceFunc) window_zoom_spin_timeout_cb, browser);
}

/** 
 * window_drag_data_received:
*/
void
window_drag_data_received(GtkWidget * widget, GdkDragContext * context,
			  gint x, gint y,
			  GtkSelectionData * selection_data,
			  guint info, guint time, GaleonBrowser *browser)
{
	switch(info)
	{
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
	case DND_TARGET_STRING:
		browser_load_url (browser, selection_data->data);
		break;
	default:
		break;
	}
}

/** 
 * window_drag_pixmap_drag_data_get_cb:
 */
void
window_drag_pixmap_drag_data_get_cb (GtkWidget *widget, 
				     GdkDragContext *context,
				     GtkSelectionData *selection_data, 
				     guint info, guint time, 
				     GaleonBrowser *browser)
{
	gchar *url = gtk_editable_get_chars 
		(GTK_EDITABLE (browser->toolbar_entry), 0, -1);
	gchar *name = g_strdup (gtk_moz_embed_get_title (browser->mozEmbed));
	BookMarkItem *b;
	gchar *mem;
	if (strlen (name) == 0) name = g_strdup (_("Untitled"));
	switch (info) {
	case DND_TARGET_GALEON_BOOKMARK:
		b = bookmarks_new_bookmark (SITE, name, url, NULL, NULL, NULL);
		mem = bookmarks_item_to_string (b);
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, mem, strlen (mem));
		g_free (mem);
		break;
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		gtk_selection_data_set 
			(selection_data, selection_data->target,
			 8, url, strlen (url));
				break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
	g_free (url);
	g_free (name);
}

/** 
 * window_location_entry_drag_data_received_cb:
 */
void
window_location_entry_drag_data_received_cb (GtkWidget *widget, 
				 GdkDragContext *drag_context, gint x, gint y,
				 GtkSelectionData *selection_data, guint info,
				 guint time, GaleonBrowser *browser)
{
	gchar *url = selection_data->data;
	gint i;
	
	switch (info) {
	case DND_TARGET_STRING:
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_GALEON_URL:
		gtk_editable_delete_text (GTK_EDITABLE (browser->toolbar_entry), 0, -1);
		gtk_editable_insert_text (GTK_EDITABLE (browser->toolbar_entry), url, 
					  strlen (url), &i);
		break;
	default:
		g_warning ("Unknown DND type");
		break;
	}
}

/** 
 * window_zoom_spin_key_press_cb: Ignore up/down key presses 
 */
gboolean
window_zoom_spin_key_press_cb (GtkWidget *widget, GdkEventKey *event, GaleonBrowser *browser)
{
	if ((event->keyval == GDK_Up) || (event->keyval == GDK_Down)) {
		event->keyval = GDK_VoidSymbol;
	}
	return FALSE;
}

void
window_entry_changed_cb (GtkEditable *editable, gboolean *changed)
{
	*changed = TRUE;
}

/* 
 * window_location_gnomeentry_popwin_cb:
 * Handler used to see if a selection was made in the location entry.
 * If a selection is made and the go toolbar is hidden, it will load
 * the URL.  It will also clear out the focus box that the GtkCombo
 * widget leaves behind after making a selection. 
 */
void
window_location_gnomeentry_popwin_cb (GtkWidget *widget, GaleonBrowser *browser)
{
	GtkCombo *combo;
	GtkContainer *container;
	GtkList *list;
	static gint selection_made;

	g_assert(widget!=NULL);
	g_assert(browser->toolbar_gnomeentry!=NULL);

	if (GTK_WIDGET_VISIBLE(widget))
	{
		combo = GTK_COMBO(browser->toolbar_gnomeentry);
		list = GTK_LIST(combo->list);
		container = GTK_CONTAINER(list);

		if (container->focus_child)
		{
			if (!GTK_LIST(container)->selection)
			{
				gtk_window_set_focus(GTK_WINDOW(GTK_COMBO(combo)->popwin), NULL);
				container->focus_child = NULL;
				list->last_focus_child = NULL;
			}

		}

		selection_made = FALSE;

		/* connect a handler to the entry's "changed" signal */
		gtk_signal_connect(GTK_OBJECT(browser->toolbar_entry),
		                   "changed",
		                   GTK_SIGNAL_FUNC(window_entry_changed_cb),
		                   &selection_made);
	}
	else
	{
		gboolean go_toolbar_hidden = FALSE;

		/* check if the go toolbar is hidden */
		if (!browser->go_toolbar)
			go_toolbar_hidden = TRUE;

		/* connect a handler to the entry's "changed" signal */
		gtk_signal_disconnect_by_func(GTK_OBJECT(browser->toolbar_entry),
	 	                              GTK_SIGNAL_FUNC(window_entry_changed_cb),
		                              &selection_made);

		if (go_toolbar_hidden && selection_made)
			window_go_button_clicked_cb(NULL, browser);
	}
}

/**
 * window_drag_data_get_cb:
 */
void
window_drag_data_get_cb (GtkWidget *widget, GdkDragContext *context,
		    GtkSelectionData *selection_data, 
		    guint info, guint time, 
		    GaleonBrowser *browser)
{
	gchar *link;
	g_return_if_fail (browser->WMain != NULL);

	link = gtk_object_get_data (GTK_OBJECT (widget), "dragging_link");

	if (link) {
		switch (info) {
		case DND_TARGET_STRING:
		case DND_TARGET_NETSCAPE_URL:
		case DND_TARGET_GALEON_URL:
			gtk_selection_data_set 
				(selection_data, selection_data->target,
				 8, link, strlen (link));
			break;
		default:
			g_warning ("Unknown DND type");
			break;
		}
	} 
}

/**
 * browser_notebook_switch_page: called in tabbed mode when the user
 * selects a different browser tab
 */
void
browser_notebook_switch_page_cb (GtkNotebook *notebook, 
				 GtkNotebookPage *page, guint page_num)
{
	GtkMozEmbed *mozEmbed;
	GaleonBrowser *browser;

	g_return_if_fail (notebook != NULL);
	g_return_if_fail (page != NULL);

	/* set the master browser to point at the right embedding widget */
	mozEmbed = (GtkMozEmbed *)page->child;
	g_return_if_fail (mozEmbed != NULL);
	master_browser->mozEmbed = mozEmbed;

	/* ARGH. This mess is due to this extra structure from
	 * the rewrite of mozilla.cpp, grrrr -- MattA 21/01/2001 */
	browser = gtk_object_get_data (GTK_OBJECT (mozEmbed), "browser");
	g_return_if_fail (browser != NULL);
	master_browser->embed = browser->embed;

	/* set the global title and location -- this is the lazy way! */
	mozembed_title_changed_cb (mozEmbed, master_browser);
	mozembed_location_changed_cb (mozEmbed, master_browser);

	/* FIXME: much else to set! */
}

/**
 * browser_notebook_close_clicked_cb: close a tab
 */
void 
browser_notebook_close_clicked_cb (GtkWidget *b, GaleonBrowser *browser)
{
	browser_close (browser);
}


