/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

/* type of drag and drop for links */
const GtkTargetEntry link_drag_types[] =
{
        { "GALEON_URL",    0, DND_TARGET_GALEON_URL   },
        { "_NETSCAPE_URL", 0, DND_TARGET_NETSCAPE_URL },
        { "STRING",        0, DND_TARGET_STRING       }
};
const gint link_drag_types_num_items = (sizeof (link_drag_types) / 
					sizeof (GtkTargetEntry));
GtkTargetList *link_drag_types_tl = NULL;

/**
 * mozembed_new_window_cb: GTKMOZEMBED SIGNAL, emitted any time a new window is requested by the document 
 */
void 
mozembed_new_window_cb (GtkMozEmbed *embed, GtkMozEmbed **retval, 
			guint chromemask, GaleonBrowser *browser) 
{
	GaleonBrowser *gb;

#ifdef DEBUG_SIGNALS 
	g_print("mozembed_new_window_cb\n");
#endif
	g_return_if_fail(browser != NULL);

	/* create a new browser */
	gb = browser_create ();

	/* set the popup flag */
	gb->is_popup = TRUE;

	/* set chrome settings (not when tabbed) */
	if (!tabbed_mode)
	{
		gtk_moz_embed_set_chrome_mask(gb->mozEmbed, chromemask);
	}

	/* return the new browser to gtkmozembed */
	*retval = GTK_MOZ_EMBED (gb->mozEmbed);
}

/**
 * mozembed_visibility_cb: GTKMOZEMBED SIGNAL, emitted when the toplevel window need to be showed or hidden
 */
void 
mozembed_visibility_cb (GtkMozEmbed *embed, gboolean visibility, GaleonBrowser *browser) 
{
	GtkMozEmbedChromeFlags flags;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_visibility_cb\n");
#endif
	g_return_if_fail(browser != NULL);

	/* show or hide browser toplevel window */
	/* check we're viewing the right embedding widget */
	/* FIXME Matt please check this. It was not showing the window in normal mode. 
	   Now I dont know if it is ok for tabbed mode */
	if (!tabbed_mode || master_browser->mozEmbed == embed)
	{
		browser_set_visibility (browser, visibility);
	}

	if (visibility && !browser->embed)
	{
		browser_embed_init (browser);
	}

	/* FIXME: This is ugly, since the resizing of the mozembed
	   widget is visible. */
	/* get and apply the requested chrome settings */
	flags = gtk_moz_embed_get_chrome_mask(embed);
	if (!(flags & GTK_MOZ_EMBED_FLAG_MENUBARON))
		browser_menubar_hide(browser);
	if (!(flags & GTK_MOZ_EMBED_FLAG_TOOLBARON))
		browser_toolbar_hide(browser);
	if (!(flags & GTK_MOZ_EMBED_FLAG_STATUSBARON))
		browser_statusbar_hide(browser);
}

/**
 * mozembed_destroy_brsr_cb: GTKMOZEMBED SIGNAL, emitted when the document has requested that the toplevel window be closed
 */
void 
mozembed_destroy_brsr_cb (GtkMozEmbed *embed, GaleonBrowser *browser) 
{
#ifdef DEBUG_SIGNALS
	g_print("mozembed_destroy_brsr_cb\n");
#endif

	g_return_if_fail(browser != NULL);

	/* FIXME: is this necessary? */
	if (browser->timeout)
	{
		gtk_timeout_remove(browser->timeout);
		browser->timeout = 0;
	}

	/* close the browser */
	browser_close (browser);
}

/**
 * mozembed_location_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that the location of the document has changed
 */
void
mozembed_location_changed_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	char *newLocation;
	int   newPosition = 0;

#ifdef DEBUG_SIGNALS
 	g_print("mozembed_location_changed_cb\n");
#endif
	g_return_if_fail(browser != NULL);

	/* autosave the updated session */
	session_autosave();

	/* check we're viewing the right embedding widget */
	if (tabbed_mode && master_browser->mozEmbed != embed)
		return;

	if (tabbed_mode)
	{
		browser = master_browser;
	}

	newLocation = gtk_moz_embed_get_location (embed);

	if (newLocation)
	{
		/* change the url entry text */
		gtk_editable_delete_text (GTK_EDITABLE (browser->toolbar_entry), 0, -1);
		if (strcmp(newLocation, "about:blank"))
			gtk_editable_insert_text (GTK_EDITABLE (browser->toolbar_entry), newLocation, 
						  strlen (newLocation), &newPosition);
		g_free (newLocation);
	}
	else 
	{
		browser_update_temp_message (browser, 0);
	}
	
	browser_update_nav_buttons (browser);
}

/**
 * mozembed_title_changed_cb: GTKMOZEMBED SIGNAL, emitted any time that the title of the document has changed
 */
void
mozembed_title_changed_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	char *title;
	char *full_title;
	char *title_string;

#ifdef DEBUG_SIGNALS	
	g_print("mozembed_title_changed_cb\n");
#endif
	g_return_if_fail(browser != NULL);

	/* get the new document title */
	title = gtk_moz_embed_get_title (embed);

	/* if the document has no title set it to Untitled */
	if (!title || (strlen (title) == 0)) 
	{ 
		title = _("Untitled");
	}

	/* tabbed mode? */
	if (tabbed_mode)
	{
		browser_set_notebook_label (browser, title);
	}

	/* check we're viewing the right embedding widget */
	if (tabbed_mode && master_browser->mozEmbed != embed)
		return;

	/* get the format string */
	title_string = gnome_config_get_string (CONF_APPEARANCE_WINDOWS_TITLE);

	/* format the full title */
	full_title = g_strdup_printf (title_string, title);

	/* set the toplevel window title to the document title */
	if (tabbed_mode)
	{
		gtk_window_set_title (GTK_WINDOW(master_browser->WMain),
				      full_title);
	}
	else
	{
		gtk_window_set_title (GTK_WINDOW(browser->WMain), full_title);
	}

	if (full_title) g_free (full_title);
	if (title_string) g_free (title_string);  
}

/**
 * mozembed_load_started_cb: GTKMOZEMBED SIGNAL, emitted any time that the load of a document has been started
 */
void
mozembed_load_started_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	GtkWidget *progress;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_load_started_cb\n");
#endif
	if (tabbed_mode)
	{
		browser = master_browser;
	}
	g_return_if_fail(browser != NULL);
		
	progress = GTK_WIDGET(gnome_appbar_get_progress (GNOME_APPBAR(browser->appbar1)));

	/* update stop and refresh toolbar buttons */
	gtk_widget_set_sensitive (browser->BStop, TRUE);
	gtk_widget_set_sensitive (browser->BRefresh, FALSE);

	/* start the spinner animation */
	if (browser->spinner)
		gnome_animator_start (GNOME_ANIMATOR(browser->spinner));

	/* initialize loading progress */
	browser_progress_clear (browser);

	/* show the progess bar */
	gtk_progress_set_activity_mode (GTK_PROGRESS(progress), TRUE);
	gtk_widget_show(GTK_WIDGET(progress));

	/* indicate this browser has started loading */
	if (!browser->timeout)
		browser->timeout = gtk_timeout_add (100,
					(GSourceFunc) browser_progress_action,
					 browser);

	browser_update_status_bar_text (browser);
}

/**
 * mozembed_load_finished_cb: GTKMOZEMBED SIGNAL, emitted any time that the load of a document has finished
 */
void
mozembed_load_finished_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	gchar *title, *url;
	history_item *hi;
	GtkWidget *progress;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_load_finished_cb\n");
#endif
	if (tabbed_mode)
	{
		browser = master_browser;
	}
	g_return_if_fail(browser != NULL);

	progress = GTK_WIDGET (gnome_appbar_get_progress 
			       (GNOME_APPBAR(browser->appbar1)));

	/* disable the progress bar */
	gtk_progress_set_activity_mode (GTK_PROGRESS(progress), FALSE);
	gtk_widget_hide (GTK_WIDGET(progress));

	/* update stop and refresh toolbar buttons */
	gtk_widget_set_sensitive (browser->BStop, FALSE);
	gtk_widget_set_sensitive (browser->BRefresh, TRUE);

	/* stop the spinner animator */
	if(browser->spinner) 
	{
		gnome_animator_stop (GNOME_ANIMATOR(browser->spinner));
		gnome_animator_goto_frame (GNOME_ANIMATOR(browser->spinner), 0);
	}

	if (browser->timeout)
	{
		gtk_timeout_remove(browser->timeout);
		browser->timeout = 0;
	}

	/* add the url to the history */
	if (browser->bytesLoaded)
	{
		title = gtk_moz_embed_get_title (embed);
		url = gtk_moz_embed_get_location (embed);
		hi = history_visited (url, title);

		/* add url to recent menu */
		if (hi) {
			history_add_recent_menu (hi, GNOME_APP(browser->WMain));
		}
	}

	/* clear progress infos */
	browser_progress_clear (browser);
	browser_update_status_bar_text (browser);
	gnome_appbar_set_progress (GNOME_APPBAR(browser->appbar1), 0);
}

/**
 * mozembed_net_status_change_cb: GTKMOZEMBED SIGNAL, emitted any time that there is a change in the status of network loading
 */
void mozembed_net_status_change_cb (GtkMozEmbed *embed, gint flags, guint status, GaleonBrowser *browser) 
{
	GaleonBrowser *window;
#ifdef DEBUG_SIGNALS
	g_print("mozembed_net_status_change_cb\n");
#endif
	
	window = tabbed_mode ? master_browser : browser;
	g_return_if_fail (browser != NULL);
	g_return_if_fail (window->WMain != NULL);

	if (flags & GTK_MOZ_EMBED_FLAG_IS_REQUEST) {
		if (flags & GTK_MOZ_EMBED_FLAG_REDIRECTING)
			window->statusMessage = _("Redirecting to site...");
		else if (flags & GTK_MOZ_EMBED_FLAG_TRANSFERRING)
			window->statusMessage = _("Transferring data from site...");
		else if (flags & GTK_MOZ_EMBED_FLAG_NEGOTIATING)
			window->statusMessage = _("Waiting for authorization...");
	}

	if (status == GTK_MOZ_EMBED_STATUS_FAILED_DNS)
		window->statusMessage = _("Site not found.");
	else if (status == GTK_MOZ_EMBED_STATUS_FAILED_CONNECT)
		window->statusMessage = _("Failed to connect to site.");
	else if (status == GTK_MOZ_EMBED_STATUS_FAILED_TIMEOUT)
		window->statusMessage = _("Failed due to connection timeout.");

	if (flags & GTK_MOZ_EMBED_FLAG_IS_DOCUMENT) {
		if (flags & GTK_MOZ_EMBED_FLAG_START)
		{
			window->statusMessage = _("Loading site...");
			if (tabbed_mode)
			{
				browser_set_notebook_label_status (browser, 
								   LOADING);
			}
		}
		else if (flags & GTK_MOZ_EMBED_FLAG_STOP)
		{
			window->statusMessage = _("Done.");
			if (tabbed_mode)
			{
				browser_set_notebook_label_status (browser, 
								   NORMAL);
			}
		}
	}

	browser_update_status_bar_text (window);
}

/**
 * mozembed_progress_change_cb: GTKMOZEMBED SIGNAL, emitted any time that there is a change in the progress of
 * loading a document.
 */
void mozembed_progress_change_cb (GtkMozEmbed *embed, gint cur, gint max,
				  GaleonBrowser *browser)
{
#ifdef DEBUG_SIGNALS
	g_print("mozembed_progress_change_cb\n");
#endif
	if (tabbed_mode)
	{
		browser = master_browser;
	}
	g_return_if_fail(browser != NULL);

	/* length of the document cannot be determined */
	if (max == -1)
	{
		browser->loadPercent = 0;
		browser->bytesLoaded = cur;
		browser->maxBytesLoaded = 0;
	}
	/* sometimes length of the downloaded document is > than the length of the document */
	else if (cur > max) {
		browser->loadPercent = 100;
		browser->bytesLoaded = cur;
		browser->maxBytesLoaded = max;
	} 
	else {
		browser->bytesLoaded = cur;
		browser->maxBytesLoaded = max;
		browser->loadPercent = (!max) ? 0 : (cur * 100) / max;
	}

	browser_update_status_bar_text (browser);

	/* FIXME:  these need tidying up */
	if (browser->loadPercent == 0) {
		GtkWidget *progress = GTK_WIDGET (gnome_appbar_get_progress (GNOME_APPBAR(browser->appbar1)));
		
		if (!(browser->timeout)) {
			browser->timeout = gtk_timeout_add (100,
				(GSourceFunc) browser_progress_action, browser);
		}

		gtk_progress_set_activity_mode (GTK_PROGRESS(progress), TRUE);
	} 
	else
	{
		GtkWidget *progress = GTK_WIDGET(gnome_appbar_get_progress(GNOME_APPBAR(browser->appbar1)));
		if (browser->timeout)
		{
			gtk_timeout_remove(browser->timeout);
			browser->timeout = 0;
		}
		gtk_progress_set_activity_mode (GTK_PROGRESS(progress), FALSE);
		gnome_appbar_set_progress (GNOME_APPBAR(browser->appbar1), browser->loadPercent / 100.0);
	}
}

/**
 * mozembed_link_message_cb: GTKMOZEMBED SIGNAL, emitted when the link message change
 */
void
mozembed_link_message_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	char *message;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_link_message_cb\n");
#endif
	if (tabbed_mode)
	{
		browser = master_browser;
	}
	g_return_if_fail(browser != NULL);

	/* get the link message */
	message = gtk_moz_embed_get_link_message (embed);

	/* update browser message */
	if (message)
	{
		if  (strlen (message) == 0) 
		{
			browser_update_temp_message (browser, NULL);
		}
		else
		{
			browser_update_temp_message (browser, message);
		}

		g_free (message);
	}
}

/**
 * mozembed_js_status_cb: GTKMOZEMBED SIGNAL, emitted when the Javascript message status change
 */
void
mozembed_js_status_cb (GtkMozEmbed *embed, GaleonBrowser *browser)
{
	char *message;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_js_status_cb\n");
#endif
	if (tabbed_mode)
	{
		browser = master_browser;
	}
	g_return_if_fail(browser != NULL);

	/* get the javascript message */
	message = gtk_moz_embed_get_js_status (embed);

	/* update the status bar message */
	if (message)
	{
		if (strlen (message) == 0)
		{
			browser_update_temp_message (browser, 0);
		}
		else
		{
			browser_update_temp_message (browser, message);
		}

		g_free (message);
	}
}

/**
 * mozembed_open_uri_cb: GTKMOZEMBED SIGNAL, emitted when the document tries to open a new document
 */
gint mozembed_open_uri_cb (GtkMozEmbed *embed, const char *uri, 
			   GaleonBrowser *browser)
{
#ifdef DEBUG_SIGNALS
	g_print("mozembed_open_uri_cb\n");
#endif
	g_return_val_if_fail(browser != NULL, FALSE);
	
	/* handle foreign protocols, things like "mailto:" */
	if (handle_foreign_protocols(uri))
		return TRUE;

	/* handle foreign mime types, e.g. RPMS (if needed) */
	if (handle_foreign_mime_types (uri, browser))
		return TRUE;

	/* we haven't handled it, so let mozilla try to load it */
	return FALSE;
}

/**
 * mozembed_dom_mouse_click_cb: GTKMOZEMBED SIGNAL, emitted when user click on the document
 */
gint mozembed_dom_mouse_click_cb (GtkMozEmbed *embed, gpointer dom_event, GaleonBrowser *browser)
{
	int type;
	gchar *img, *href;
	gpointer target;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_dom_mouse_click_cb\n");
#endif
	g_return_val_if_fail(browser != NULL, FALSE);

	type = mozilla_get_event_context (browser, dom_event,
					  &img, &target, &href);

	switch (mozilla_get_mouse_button(dom_event))
	{
	case 3:
		if ((type & CONTEXT_IMAGE) && (type & CONTEXT_LINK))
		{
			context_show_link_img_menu(browser, img, href);
		}	
		else if (type & CONTEXT_IMAGE)
		{
			context_show_img_menu(browser, img);
		}		
		else if (type & CONTEXT_LINK)
		{
			context_show_link_menu(browser, href);
		}		
		else if (type & CONTEXT_DOCUMENT)
		{
			if (!gnome_config_get_bool("/galeon/Appearance/min_doc_context_checkbutton=FALSE"))
			{
				context_show_doc_menu (browser, target);
			}
			else
			{
				context_show_minimal_doc_menu (browser, target);
			}
		}	
		else if (type  & CONTEXT_OTHER)
		{
			g_print("clicked other components");
		}	
		return TRUE;
		break;

	case 2:
		if ((type & CONTEXT_LINK) && href) {
			browser_create_from_url (href);
			return TRUE;
		} else {
			if (gnome_config_get_int(
				"/galeon/Mouse/middle_button_action") == 0)
				/* popup a bookmarks menu */
				context_show_bookmark_menu (browser);
			else {
				gtk_selection_convert(browser->WMain,
						      GDK_SELECTION_PRIMARY,
						      GDK_SELECTION_TYPE_STRING,
						      GDK_CURRENT_TIME);
			}
		}
		break;

	case 1: 
		if ((type & CONTEXT_LINK) && href)
		{
			return handle_foreign_protocols (href);
		}
		break;

	default:
		break;
	}

	return FALSE;
}

/**
 * mozembed_dom_mouse_click_cb: GTKMOZEMBED SIGNAL, emitted when a key is pressed
 */
gint mozembed_dom_key_press_cb (GtkMozEmbed *embed, gpointer dom_event, GaleonBrowser *browser)
{
	int modifier;
	long key;

#ifdef DEBUG_SIGNALS
	g_print("mozembed_dom_key_press_cb\n");
#endif
	g_return_val_if_fail(browser != NULL, FALSE);

	key = mozilla_get_key_pressed(browser, dom_event, &modifier);

	/* g_print("key =%ld\n",key); 
	   g_print("modifier = %d\n", modifier); */

	if (modifier==0)
	{
		switch(key)
		{
		default:
			break;
		}
	}
	else if ((modifier & ALT_KEY) && (modifier & CTRL_KEY))
	{
		gboolean b;

		switch(key)
		{
		case DOM_VK_v:
		case DOM_VK_V:
			browser_view_source (browser, NULL);
			return TRUE;
			break;
		case DOM_VK_h:
		case DOM_VK_H:
			browser_show_history (browser);
			return TRUE;
			break;
		case DOM_VK_s:
		case DOM_VK_S:
			b = !GTK_CHECK_MENU_ITEM(browser->view_source_mode)->active;
			gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(browser->view_source_mode), b); 
							
			mozilla_view_source (browser, b);
			return TRUE;
			break;
		default:
			break;
		}
	}
	else if ((modifier & KEY_CODE) && ((modifier & CTRL_KEY) ||
					   (modifier & ALT_KEY)))
	{
		switch(key)
		{
		case DOM_VK_KP_LEFT:
		case DOM_VK_LEFT:
			gtk_moz_embed_go_back(browser->mozEmbed);
			return TRUE;
			break;
		case DOM_VK_KP_RIGHT:
		case DOM_VK_RIGHT:
			gtk_moz_embed_go_forward(browser->mozEmbed);
			return TRUE;
			break;
		case DOM_VK_UP:
			if (!tabbed_mode)
			{
				return FALSE;
			}
			gtk_notebook_prev_page (GTK_NOTEBOOK(master_notebook));
			return TRUE;
			break;
		case DOM_VK_DOWN:
			if (!tabbed_mode)
			{
				return FALSE;
			}
			gtk_notebook_next_page (GTK_NOTEBOOK(master_notebook));
			return TRUE;
			break;
		default:
			break;
		}
	}
	else if ((modifier & ALT_KEY) && !(modifier & (CTRL_KEY | SHIFT_KEY)))
	{
		switch(key)
		{
		case DOM_VK_1:
		case DOM_VK_2:
		case DOM_VK_3:
		case DOM_VK_4:
		case DOM_VK_5:
		case DOM_VK_6:
		case DOM_VK_7:
		case DOM_VK_8:
		case DOM_VK_9:
			if (!tabbed_mode)
				return FALSE;
			gtk_notebook_set_page (GTK_NOTEBOOK(master_notebook),
					       key - DOM_VK_1);
			return TRUE;
			break;
		case DOM_VK_0:
			if (!tabbed_mode)
				return FALSE;
			gtk_notebook_set_page (GTK_NOTEBOOK(master_notebook),
					       9);
			return TRUE;
			break;
		case DOM_VK_EQUALS:
		case DOM_VK_PLUS:
			gtk_spin_button_spin (GTK_SPIN_BUTTON (browser->zoom_spin), GTK_SPIN_STEP_FORWARD, 10);
			return TRUE;
			break;
		case DOM_VK_MINUS:
			gtk_spin_button_spin (GTK_SPIN_BUTTON (browser->zoom_spin), GTK_SPIN_STEP_BACKWARD, 10);
			return TRUE;
			break;
		case DOM_VK_f:
		case DOM_VK_F:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "file1")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_e:
		case DOM_VK_E:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "edit2")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_v:
		case DOM_VK_V:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "view1")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_s:
		case DOM_VK_S:
                        gtk_signal_emit_by_name(GTK_OBJECT(
                                glade_lookup_widget(browser->WMain, "settings_menu")),
                                "activate_item");
                        return TRUE;
                        break;
		case DOM_VK_o:
		case DOM_VK_O:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "bookmarks")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_g:
		case DOM_VK_G:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "go1")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_h:
		case DOM_VK_H:
			gtk_signal_emit_by_name(GTK_OBJECT(
				glade_lookup_widget(browser->WMain, "help1")),
				"activate_item");
			return TRUE;
			break;
		case DOM_VK_n:
		case DOM_VK_N:
			browser_create_default(browser);
			return TRUE;
			break;
		case DOM_VK_r:
		case DOM_VK_R:
			browser_reload (browser);
			return TRUE;
			break;
		case DOM_VK_t:
		case DOM_VK_T:
			browser_show_temporary_bookmarks (browser);
			return TRUE;
			break;
		default:
			break;

		}
	}
	else if (modifier & KEY_CODE)
	{
		GtkCheckMenuItem *mi;

		switch(key)
		{
		case DOM_VK_ESCAPE:
			gtk_moz_embed_stop_load (browser->mozEmbed);
			return TRUE;
			break;
		case DOM_VK_F3:
			browser_show_open_dialog(browser);
			return TRUE;
			break;
		case DOM_VK_F6:
			browser_show_find_dialog(browser, NULL);
			return TRUE;
			break;
		case DOM_VK_F11:
			mi = GTK_CHECK_MENU_ITEM(browser->view_fullscreen);
			gtk_check_menu_item_set_active(mi, !mi->active);
			return TRUE;
			break;		
		default:
			break;
		}
	}
	else if (modifier & CTRL_KEY)
	{		
		GtkCheckMenuItem *mi;

		switch(key)
		{
		case DOM_VK_f:
		case DOM_VK_F:
			browser_show_find_dialog(browser, NULL);
			return TRUE;
			break;
		case DOM_VK_EQUALS:
		case DOM_VK_PLUS:
			gtk_spin_button_spin (GTK_SPIN_BUTTON (browser->zoom_spin), GTK_SPIN_STEP_FORWARD, 10);
			return TRUE;
			break;
		case DOM_VK_MINUS:
			gtk_spin_button_spin (GTK_SPIN_BUTTON (browser->zoom_spin), GTK_SPIN_STEP_BACKWARD, 10);
			return TRUE;
			break;
		case DOM_VK_a:
		case DOM_VK_A:
			browser_add_temporary_bookmark (browser);
			return TRUE;
			break;
		case DOM_VK_g:
		case DOM_VK_G:
			gtk_editable_select_region
				(GTK_EDITABLE(browser->toolbar_entry), 0, -1);
			gtk_window_set_focus (GTK_WINDOW(browser->WMain),
					      browser->toolbar_entry);
			return TRUE;
			break;   
		case DOM_VK_o:
		case DOM_VK_O:
			browser_show_openurl_dialog(browser);
			return TRUE;
			break;		
		case DOM_VK_s:
		case DOM_VK_S:
			browser_show_save_dialog(browser, NULL);
			return TRUE;
			break;		
		case DOM_VK_w:
		case DOM_VK_W:
			browser_close(browser);
			return TRUE;
			break;		
		case DOM_VK_q:
		case DOM_VK_Q:
			gtk_idle_add((GtkFunction) galeon_quit, NULL);
			return TRUE;
			break;		
		case DOM_VK_n:
		case DOM_VK_N:
			browser_create_default(browser);
			return TRUE;
			break;
		case DOM_VK_m:
		case DOM_VK_M:
			mi = GTK_CHECK_MENU_ITEM(browser->view_menubar);
			gtk_check_menu_item_set_active(mi, !mi->active);
			return TRUE;
			break;		
		case DOM_VK_t:
		case DOM_VK_T:
			mi = GTK_CHECK_MENU_ITEM(browser->view_toolbar);
			gtk_check_menu_item_set_active(mi, !mi->active);
			return TRUE;
			break;		
		case DOM_VK_u:
		case DOM_VK_U:
			mi = GTK_CHECK_MENU_ITEM(browser->view_statusbar);
			gtk_check_menu_item_set_active(mi, !mi->active);
			return TRUE;
			break;		
		case DOM_VK_r:
		case DOM_VK_R:
			browser_reload (browser);
			return TRUE;
			break;
		case DOM_VK_b:
		case DOM_VK_B:
			browser_edit_bookmarks (browser);
			return TRUE;
			break;		
		default:
			break;

		}
	}

	return FALSE;

}

/**
 * mozembed_dom_mouse_click_cb: GTKMOZEMBED SIGNAL, emitted when a size change is requested
 */
void 
mozembed_size_to_cb (GtkMozEmbed *embed, gint width, gint height, GaleonBrowser *browser)
{
#ifdef DEBUG_SIGNALS
	g_print("mozembed_size_to_cb\n");
#endif
	/* set window size (not when tabbed) */
	if (!tabbed_mode)
	{
		gtk_widget_set_usize (browser->WMain, width, height);
	}
}

/**
 * mozembed_destroy_cb: gtkmozembed component destroying
 */
void
mozembed_destroy_cb (GtkObject *object, GaleonBrowser *browser)
{
#ifdef DEBUG_SIGNALS
	g_print("mozembed_destroy_cb\n");
#endif

	if (browser->embed)
		mozilla_embed_destroy (browser);

	/* remove from list of browsers */
	all_browsers = g_list_remove (all_browsers, browser);

	/* autosave the updated session */
	session_autosave();

	/* free the list of toolbars */
	if (!tabbed_mode)
	{
		g_list_free (browser->bookmarks_toolbars);
	}

	/* scrub and free the GaleonBrowser structure */
	memset(browser, 0, sizeof(GaleonBrowser));
	g_free(browser);

	/* FIXME: ugh */
	if (g_list_length (all_browsers) == 0)
	{
		if (tabbed_mode)
		{
			all_windows = g_list_remove (all_windows, 
						     master_browser);
			g_list_free (master_browser->bookmarks_toolbars);
			gtk_widget_destroy (master_browser->WMain);
		}
		galeon_exit ();
	}
}

/** 
 * mozembed_drag_drop_cb:
 */
gboolean
mozembed_drag_drop_cb(GtkWidget * widget, GdkDragContext *context, 
		      gint x, gint y, GtkSelectionData *selection_data, 
		      guint info, guint time)
{
	return FALSE;
}

