/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

static char *autodetect_charset_prefs[] =
{
	"",
	"zh_parallel_state_machine",
	"cjk_parallel_state_machine",
	"ja_parallel_state_machine",
	"ko_parallel_state_machine",
	"ruprob",
	"zhcn_parallel_state_machine",
	"zhtw_parallel_state_machine",
	"ukprob"
};

void mozilla_prefs_set_autodetect_charset (int charset);
void mozilla_prefs_set_default_charset (void);
void mozilla_prefs_set_languages (void);
void mozilla_prefs_set_proxy (void);
void mozilla_prefs_set_cache (void);
void mozilla_prefs_set_fonts (void);
void mozilla_prefs_set_images (void);
void mozilla_prefs_set_colors (void);
void mozilla_prefs_set_java (void);
void mozilla_prefs_set_network (void);
void mozilla_prefs_set_wheel_mouse (void);
char * get_bracketed_string (char *str);

/**
 * mozilla_prefs_set: set mozilla preferences
 */
void
mozilla_prefs_set (void)
{
       	mozilla_prefs_set_proxy ();
	mozilla_prefs_set_cache ();
	mozilla_prefs_set_fonts ();
	mozilla_prefs_set_images ();
	mozilla_prefs_set_colors ();
	mozilla_prefs_set_java ();
	mozilla_prefs_set_network ();
	mozilla_prefs_set_autodetect_charset (gnome_config_get_int ("/galeon/Advanced/autodetect_charset=0"));
	mozilla_prefs_set_default_charset ();
	mozilla_prefs_set_languages ();
	mozilla_prefs_set_wheel_mouse ();
}

/**
 * mozilla_prefs_set_wheel_mouse: set wheel mouse preferences
 */
void mozilla_prefs_set_wheel_mouse (void)
{
	/* set mouse wheel actions */
	mozilla_preference_set_int ("mousewheel.withnokey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_nokey_action=0"));
	mozilla_preference_set_int ("mousewheel.withshiftkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_shift_action=0"));
	mozilla_preference_set_int ("mousewheel.withcontrolkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_ctrl_action=0"));
	mozilla_preference_set_int ("mousewheel.withaltkey.action",
				    gnome_config_get_int ("/galeon/Advanced/wheel_alt_action=2"));

	/* set mouse wheel to use system default step size */
	mozilla_preference_set_boolean ("mousewheel.withnokey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_nokey_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withcontrolkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_ctrl_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withaltkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_alt_default_step=TRUE"));
	mozilla_preference_set_boolean ("mousewheel.withshiftkey.sysnumlines",
				    gnome_config_get_bool ("/galeon/Advanced/wheel_shift_default_step=TRUE"));

	/* set mouse wheel step size */
	mozilla_preference_set_int ("mousewheel.withnokey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_nokey_step=1"));
	mozilla_preference_set_int ("mousewheel.withaltkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_alt_step=1"));
	mozilla_preference_set_int ("mousewheel.withcontrolkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_ctrl_step=1"));
	mozilla_preference_set_int ("mousewheel.withshiftkey.numlines",
				    gnome_config_get_int ("/galeon/Advanced/wheel_shift_step=1"));
}

/**
 * mozilla_prefs_set_java: set java and javascript preferences
 */
void 
mozilla_prefs_set_java (void)
{
	gboolean java_enabled = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
	gboolean javascript_enabled = gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);

	mozilla_preference_set_boolean ("security.enable_java", java_enabled); 
	mozilla_preference_set_boolean ("javascript.enabled", javascript_enabled); 
}

/**
 * mozilla_prefs_set_network: set network preferences
 */
void 
mozilla_prefs_set_network (void)
{
	gchar *http_version = NULL;
	gint cookieBeh = gnome_config_get_int (CONF_ADVANCED_COOKIES_BEHAVIOR);
	gboolean warn_cookie = gnome_config_get_bool ("/galeon/Advanced/cookie_warn=FALSE");

	switch (gnome_config_get_int ("/galeon/Advanced/http_version=1"))
	{
		case 0: http_version = g_strdup ("1.0");
		case 1: http_version = g_strdup ("1.1");
	}
      	
	mozilla_preference_set_boolean ("security.checkloaduri", FALSE); 

  
	mozilla_preference_set ("general.useragent.misc", "Galeon");

	mozilla_preference_set_boolean ("network.cookie.warnAboutCookies", warn_cookie);

	mozilla_preference_set_int ("network.cookie.cookieBehavior", cookieBeh);
	mozilla_preference_set ("network.http.version", http_version);
	g_free (http_version);
}

/**
 * mozilla_prefs_set: set images behavior preferences
 */
void 
mozilla_prefs_set_images (void)
{
	int image_loading = gnome_config_get_int ("/galeon/Advanced/image_loading_type=0");

	mozilla_preference_set_int ("network.image.imageBehavior", image_loading);
} 

/**
 * mozilla_prefs_set: set fonts preferences
 */
void 
mozilla_prefs_set_fonts (void)
{
	gchar *serif_font, *sansserif_font, *monospace_font, *default_font[2] = {"sans-serif","serif"};
	gchar **font;
	int size_font,default_font_idx;

	serif_font = gnome_config_get_string ("/galeon/Appearance/serif_font=-adobe-times-medium-r-normal-*-14-*-*-*-p-*-iso8859-1");
	sansserif_font = gnome_config_get_string ("/galeon/Appearance/sansserif_font=-adobe-times-medium-r-normal-*-14-*-*-*-p-*-iso8859-1");
	monospace_font = gnome_config_get_string ("/galeon/Appearance/monospace_font=-adobe-times-medium-r-normal-*-14-*-*-*-p-*-iso8859-1");
	size_font = gnome_config_get_int ("/galeon/Appearance/size_font=14");
	default_font_idx = gnome_config_get_int("/galeon/Appearance/default_font=0");

	font = g_strsplit (serif_font,"-",-1);
	mozilla_preference_set ("font.name.serif.x-western",g_strconcat(font[1],"-",font[2],"-",font[13],
								       "-",font[14],NULL));
	g_strfreev(font);

	font = g_strsplit (sansserif_font,"-",-1);
	mozilla_preference_set ("font.name.sans-serif.x-western",g_strconcat(font[1],"-",font[2],"-",font[13],
									    "-",font[14],NULL));
	g_strfreev(font);

	font = g_strsplit (monospace_font,"-",-1);
	mozilla_preference_set ("font.name.monospace.x-western",g_strconcat(font[1],"-",font[2],"-",font[13],
									   "-",font[14],NULL));
	mozilla_preference_set_int ("font.size.fixed.x-western",atoi(font[7]));
	g_strfreev(font);

	mozilla_preference_set_int ("font.size.variable.x-western",size_font);

	mozilla_preference_set ("font.default",default_font[default_font_idx]);
}

/**
 * mozilla_prefs_set: set colors preferences
 */
void 
mozilla_prefs_set_colors (void)
{
	gchar *background_color, *text_color, *visited_link_color;
	gchar *unvisited_link_color;
	gboolean use_provided_colors, underline_links;
	gboolean enable_syntax_highlighting;
	gint use_provided_fonts;

	background_color = gnome_config_get_string ("/galeon/Appearance/background_color=#FFFFFF");
	text_color = gnome_config_get_string ("/galeon/Appearance/text_color=#000000");
	visited_link_color = gnome_config_get_string ("/galeon/Appearance/visited_link_color=#FF0000");
	unvisited_link_color = gnome_config_get_string ("/galeon/Appearance/unvisited_link_color=#0000FF");
	use_provided_colors = (gnome_config_get_int
			       (CONF_APPEARANCE_USE_PROVIDED_COLORS) == 0) ? TRUE : FALSE;
	use_provided_fonts = gnome_config_get_int (CONF_APPEARANCE_USE_PROVIDED_FONTS);
	underline_links = gnome_config_get_bool ("/galeon/Appearance/underline_links=TRUE");
	enable_syntax_highlighting = gnome_config_get_bool
					(CONF_HANDLER_VIEW_SOURCE_COLORS);
	mozilla_preference_set ("browser.display.background_color", background_color);
	mozilla_preference_set ("browser.display.foreground_color", text_color);
	mozilla_preference_set ("browser.anchor_color", unvisited_link_color);
	mozilla_preference_set ("browser.visited_color", visited_link_color);
	mozilla_preference_set_boolean ("browser.underline_anchors", underline_links);
	mozilla_preference_set_boolean ("browser.display.use_document_colors",
					use_provided_colors);
	mozilla_preference_set_int ("browser.display.use_document_fonts", 
				    use_provided_fonts);

	/* syntax highlighting for view source mode */
	mozilla_preference_set_boolean ("browser.view_source.syntax_highlight",
					enable_syntax_highlighting);
	g_free (background_color);
	g_free (text_color);
	g_free (visited_link_color);
	g_free (unvisited_link_color);
 }

/**
 * mozilla_prefs_set: set proxy preferences
 */
void 
mozilla_prefs_set_proxy (void)
{
	gchar *http_proxy, *ftp_proxy;
	int http_proxy_port, ftp_proxy_port;
	gchar *no_proxies_for;
	gboolean proxy_keepalive;
	

	http_proxy = gnome_config_get_string ("/galeon/Advanced/http_proxy=");
	http_proxy_port = gnome_config_get_int ("/galeon/Advanced/http_proxy_port=8080");
	ftp_proxy = gnome_config_get_string ("/galeon/Advanced/ftp_proxy=");
	ftp_proxy_port = gnome_config_get_int ("/galeon/Advanced/ftp_proxy_port=8080");
	no_proxies_for = gnome_config_get_string ("/galeon/Advanced/no_proxies_for=localhost");
	proxy_keepalive = gnome_config_get_bool ("/galeon/Advanced/proxy_keepalive=TRUE");


	if (strlen (http_proxy) != 0)
	{
		mozilla_preference_set_int ("network.proxy.type", 1);
		mozilla_preference_set ("network.proxy.http", http_proxy);
		mozilla_preference_set_int ("network.proxy.http_port", http_proxy_port);
	}

	if (strlen (ftp_proxy) != 0)
	{
		mozilla_preference_set_int ("network.proxy.type", 1);
		mozilla_preference_set ("network.proxy.ftp", ftp_proxy);
		mozilla_preference_set_int ("network.proxy.ftp_port", ftp_proxy_port);
	}

	if ((strlen (http_proxy) != 0) || (strlen (ftp_proxy) != 0))
	{
	        if (strlen (no_proxies_for) != 0) 
		{
		  mozilla_preference_set ("network.proxy.no_proxies_on", no_proxies_for);
		}
		else
		{
			/* Fix problem with changing to blank entry on the fly in preferences */
		  mozilla_preference_set ("network.proxy.no_proxies_on", " ");
		}
	}

	mozilla_preference_set_boolean("network.http.proxy.keep-alive", proxy_keepalive);

	g_free (http_proxy);
	g_free (ftp_proxy);
	g_free (no_proxies_for);
}

/**
 * mozilla_prefs_set_cache: set cache preferences
 */
void 
mozilla_prefs_set_cache (void)
{
	int disk_cache_size = gnome_config_get_int("/galeon/Advanced/disk_cache_size=7680");
	int mem_cache_size = gnome_config_get_int("/galeon/Advanced/mem_cache_size=1024");

	if (disk_cache_size > 0)
	{
		gchar *cachedir = init_cache_dir();
		mozilla_preference_set("browser.cache.directory", cachedir);
		mozilla_preference_set_boolean("browser.enable.cache", TRUE);
		mozilla_preference_set_int ("browser.cache.disk_cache_size", disk_cache_size);
		g_free(cachedir);
	}
	else
	{
		mozilla_preference_set_boolean("browser.enable.cache", FALSE);
	}

	mozilla_preference_set_int ("browser.cache.memory_cache_size", mem_cache_size);
}

/**
 * mozilla_prefs_set_autodetect_charset: 
 */
void 
mozilla_prefs_set_autodetect_charset (int charset)
{
	mozilla_preference_set ("intl.charset.detector", autodetect_charset_prefs[charset]);
}

/**
 * mozilla_prefs_set_default_charset:
 */
void 
mozilla_prefs_set_default_charset (void)
{
	char *charset; 
	char *charset_code;

	charset = gnome_config_get_string ("/galeon/Advanced/default_charset=Western (ISO-8859-1)");
	charset_code = get_bracketed_string (charset);
	
	if (charset_code)
	{
		mozilla_preference_set ("intl.charset.default", charset_code);
		g_free(charset_code);
	}

	g_free(charset);
}

/**
 * mozilla_prefs_set_languages:
 */
void 
mozilla_prefs_set_languages (void)
{
	int size;
	gchar **strings = NULL;
	gchar *languages = NULL;
	int k;

	gnome_config_get_vector ("/galeon/Advanced/lang=English\\ (en)", &size, &strings);

	for (k=0; k<size ; k++)
	{	
		char *lang;

		lang = get_bracketed_string (strings[k]);

		if (!lang)
		{
			lang = strings[k];
		}

		if (k==0)
		{
			languages = lang;
		}
		else
		{
			languages = g_strconcat (languages, g_strdup(","), lang, NULL);
		}

	}	

	if (strings)
	{
		mozilla_preference_set ("intl.accept_languages", languages);
		g_free (languages);
	}
}

/**
 * get_bracketed_string: get the string between brackets. ex: language (string)
 */
char * 
get_bracketed_string (char *str)
{
	char *begin;
	char *end;

	begin = strchr (str,'(');
	end = strchr (str,')');

	if (begin && end)
	{
		begin ++;
		return g_strndup(begin, end - begin);
	}
	else 
	{
		return NULL;
	}
}

char* special_languages[] = {"bg_BG", "en_GB", "pt_BR", "zh_CN", "zh_TW" , NULL};

/**
 * mozilla_prefs_import_language:
 */
void mozilla_prefs_import_language (void)
{
	char *lc_messages, *lang = NULL;
	int i;

	lc_messages = setlocale(LC_MESSAGES, NULL);

	if (strcmp (lc_messages, "C")!=0)
	{
		for (i = 0; special_languages[i] != NULL; i++)
		{	
			if (strcasecmp (lc_messages, special_languages[i])==0)
			{
			    lang = g_strdup (lc_messages);
			    lang[2]='-';
			}
		}

		if (!lang)
		{
			lang = g_strndup (lc_messages, 2);
			gnome_config_set_vector ("/galeon/Advanced/lang", 1, (const char**) &lang);	
		}
	}
}

