/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef GALEON_EMBED_SHELL_H
#define GALEON_EMBED_SHELL_H

#include "global-history.h"
#include "downloader-view.h"
#include "galeon-embed.h"
#include "cookie-info.h"

#include <glib-object.h>
#include <glib.h>

G_BEGIN_DECLS

typedef struct GaleonEmbedShellClass GaleonEmbedShellClass;

#define GALEON_EMBED_SHELL_TYPE             (galeon_embed_shell_get_type ())
#define GALEON_EMBED_SHELL(obj)             (GTK_CHECK_CAST ((obj), GALEON_EMBED_SHELL_TYPE, GaleonEmbedShell))
#define GALEON_EMBED_SHELL_CLASS(klass)     (GTK_CHECK_CLASS_CAST ((klass), GALEON_EMBED_SHELL_TYPE, GaleonEmbedShellClass))
#define IS_GALEON_EMBED_SHELL(obj)          (GTK_CHECK_TYPE ((obj), GALEON_EMBED_SHELL_TYPE))
#define IS_GALEON_EMBED_SHELL_CLASS(klass)  (GTK_CHECK_CLASS_TYPE ((klass), GALEON_EMBED_SHELL))
#define GALEON_EMBED_SHELL_GET_CLASS(obj)   (G_TYPE_INSTANCE_GET_CLASS ((obj), GALEON_EMBED_SHELL_TYPE, GaleonEmbedShellClass))

typedef struct GaleonEmbedShell GaleonEmbedShell;
typedef struct GaleonEmbedShellPrivate GaleonEmbedShellPrivate;

extern GaleonEmbedShell *embed_shell;

/**
 * FilePickerMode: What mode FilePicker should run in
 */

typedef enum
{
        modeOpen = 0,
        modeSave = 1,
        modeGetFolder =2
} FilePickerMode;

typedef struct 
{
        /* description of the file format */
        gchar *description;
        /* tipical sufixes, NULL terminated */
        gchar **extensions;
} FileFormat;

/**
 * BlockedHost: a blocked host
 */
typedef struct
{
        char *domain;
	gboolean status;
} PermissionInfo;

/**
 * Password: a password manager entry
 */
typedef struct
{
        gchar *host;
        gchar *username;
} PasswordInfo;

typedef struct
{
	const char *name;
	const char *title;
} CharsetInfo;

/**
 * PasswordType: To distinguish actual passwords from blocked password sites
 */
typedef enum
{
        PASSWORD_PASSWORD,
        PASSWORD_REJECT
} PasswordType;

typedef enum
{
	COOKIES_PERMISSION,
	IMAGES_PERMISSION
} PermissionType;

typedef enum
{
	DISK_CACHE = 2,
	MEMORY_CACHE = 1
} CacheType;

typedef enum
{
	CACHE_CLEAR_CAP = 1 << 0,
	OFFLINE_CAP = 1 << 1,
	PROXY_AUTOCONF_CAP = 1 << 2,
	JAVA_CONSOLE_CAP = 1 << 3,
	JS_CONSOLE_CAP = 1 << 4,
	CHARSETS_CAP = 1 << 5,
	PERMISSIONS_CAP = 1 << 6,
	COOKIES_CAP = 1 << 7,
	PASSWORDS_CAP = 1 << 8,
	FILEPICKER_CAP = 1 << 9
} EmbedShellCapabilities;

struct GaleonEmbedShell 
{
	GObject parent;
        GaleonEmbedShellPrivate *priv;
};

struct GaleonEmbedShellClass
{
        GObjectClass parent_class;

	void 		(* new_window)          (GaleonEmbedShell *shell,
                        		         GaleonEmbed **new_embed,
                              			 EmbedChromeMask chromemask);

	/* Methods */
	
	void 		(* get_capabilities)    (GaleonEmbedShell *shell,
						 EmbedShellCapabilities *caps);
	GlobalHistory * (* get_global_history)  (GaleonEmbedShell *shell);
	DownloaderView* (* get_downloader_view) (GaleonEmbedShell *shell);
	gresult         (* clear_cache)         (GaleonEmbedShell *shell,
						 CacheType type);
	gresult         (* set_offline_mode)    (GaleonEmbedShell *shell,
						 gboolean offline);
	gresult         (* load_proxy_autoconf) (GaleonEmbedShell *shell,
					         const char* url);
	gresult         (* show_java_console)   (GaleonEmbedShell *shell);
	gresult         (* show_js_console)     (GaleonEmbedShell *shell);
	gresult		(* get_charset_groups)  (GaleonEmbedShell *shell,
						 GList **groups);
	gresult         (* get_charset_titles)  (GaleonEmbedShell *shell,
						 const char *group,
						 GList **charsets);
	gresult		(* get_font_list)	(GaleonEmbedShell *shell,
						 const char *langGroup,
						 const char *fontType,
						 GList **fontList,
						 char **default_font);
	gresult         (* set_permission)      (GaleonEmbedShell *shell,
						 const char *url, 
					         PermissionType type,
				 	         gboolean allow);
	gresult         (* list_permissions)    (GaleonEmbedShell *shell,
					    	 PermissionType type, 
						 GList **permissions);
	gresult         (* remove_permissions)  (GaleonEmbedShell *shell,
						 PermissionType type,
						 GList *permissions);
	gresult         (* list_cookies)        (GaleonEmbedShell *shell,
						 GList **cokies);
	gresult         (* remove_cookies)      (GaleonEmbedShell *shell,
						 GList *cookies);
	gresult         (* list_passwords)      (GaleonEmbedShell *shell,
						 PasswordType type, 
						 GList **passwords);
	gresult         (* remove_passwords)    (GaleonEmbedShell *shell,
						 GList *passwords, 
						 PasswordType type);
	gresult         (* show_file_picker)    (GaleonEmbedShell *shell,
						 GtkWidget *parentWidget, 
						 const char* title,
                  				 const char* directory, 
						 const char* file, 
						 FilePickerMode mode,
                  				 char **ret_fullpath, 
						 gboolean *ret_save_content, 
                  				 FileFormat *file_formats, 
						 gint *ret_file_format);
};

GType             galeon_embed_shell_get_type            (void);

GaleonEmbedShell *galeon_embed_shell_new                 (const char *type);

void              galeon_embed_shell_add_embed           (GaleonEmbedShell *ges,
                                                    	  GaleonEmbed *embed);

void              galeon_embed_shell_remove_embed        (GaleonEmbedShell *ges,
                                                    	  GaleonEmbed *embed);

GaleonEmbed      *galeon_embed_shell_get_active_embed    (GaleonEmbedShell *ges);

GList            *galeon_embed_shell_get_embeds          (GaleonEmbedShell *ges);

const char      **galeon_embed_shell_get_supported       (void);    

void              galeon_embed_shell_get_capabilities    (GaleonEmbedShell *shell,
							  EmbedShellCapabilities *caps);

/* Services provided by galeon. They can be overriden. */
GlobalHistory    *galeon_embed_shell_get_global_history  (GaleonEmbedShell *shell);

DownloaderView   *galeon_embed_shell_get_downloader_view (GaleonEmbedShell *shell);

/* Services provided by the renderer */
gresult           galeon_embed_shell_clear_cache         (GaleonEmbedShell *shell,
							  CacheType type);

gresult           galeon_embed_shell_set_offline_mode    (GaleonEmbedShell *shell,
							  gboolean offline);

gresult           galeon_embed_shell_load_proxy_autoconf (GaleonEmbedShell *shell,
							  const char* url);

gresult           galeon_embed_shell_show_java_console   (GaleonEmbedShell *shell);

gresult           galeon_embed_shell_show_js_console     (GaleonEmbedShell *shell);

/* Charsets */
gresult		  galeon_embed_shell_get_charset_groups  (GaleonEmbedShell *shell,
							  GList **groups);

gresult           galeon_embed_shell_get_charset_titles  (GaleonEmbedShell *shell,
							  const char *group,
							  GList **charsets);
gresult           galeon_embed_shell_get_font_list	 (GaleonEmbedShell *shell,
							  const char *langGroup,
							  const char *fontType,
							  GList **fontList,
							  char **default_font);

/* Permissions */
gresult           galeon_embed_shell_set_permission      (GaleonEmbedShell *shell,
							  const char *url, 
					                  PermissionType type,
				 	                  gboolean allow);

gresult           galeon_embed_shell_list_permissions    (GaleonEmbedShell *shell,
							  PermissionType type, 
							  GList **permissions);

gresult		  galeon_embed_shell_free_permissions    (GaleonEmbedShell *shell,
							  GList *permissions);

gresult           galeon_embed_shell_remove_permissions  (GaleonEmbedShell *shell,
							  PermissionType type,
							  GList *permissions);

/* Cookies */
gresult           galeon_embed_shell_list_cookies        (GaleonEmbedShell *shell,
							  GList **cookies);

gresult           galeon_embed_shell_remove_cookies      (GaleonEmbedShell *shell,
							  GList *cookies);

gresult		  galeon_embed_shell_free_cookies        (GaleonEmbedShell *shell,
							  GList *cookies);

/* Passwords */
gresult           galeon_embed_shell_list_passwords      (GaleonEmbedShell *shell,
							  PasswordType type, 
							  GList **passwords);

gresult		  galeon_embed_shell_free_passwords      (GaleonEmbedShell *shell,
							  GList *passwords);

gresult           galeon_embed_shell_remove_passwords    (GaleonEmbedShell *shell,
							  GList *passwords, 
							  PasswordType type);

gresult           galeon_embed_shell_show_file_picker    (GaleonEmbedShell *shell,
							  GtkWidget *parentWidget, 
							  const char *title,
                  					  const char *directory, 
							  const char *file, 
							  FilePickerMode mode,
                  					  char **ret_fullpath, 
							  gboolean *ret_save_content, 
                  					  FileFormat *file_formats, 
							  int *ret_file_format);

G_END_DECLS

#endif
