/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-tb-button.h"
#include "gul-gui.h"
#include "gul-string.h"
#include "galeon-navigation-button.h"
#include <gtk/gtkstock.h>
#include <string.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
#define DEBUG_MSG(x) g_print x
//#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GaleonNavigationButtonPrivate 
{
	GulTbButton *widget;
	GaleonNavigationDirection direction;
	gboolean show_arrow;
	gboolean sensitive;
};

/**
 * Private functions, only availble from this file
 */
static void		galeon_navigation_button_class_init		(GaleonNavigationButtonClass *klass);
static void		galeon_navigation_button_init			(GaleonNavigationButton *tb);
static void		galeon_navigation_button_finalize_impl		(GObject *o);
static GtkWidget *	galeon_navigation_button_get_widget_impl	(GulTbItem *i);
static GdkPixbuf *	galeon_navigation_button_get_icon_impl		(GulTbItem *i);
static gchar *		galeon_navigation_button_get_name_human_impl	(GulTbItem *i);
static gchar *		galeon_navigation_button_to_string_impl		(GulTbItem *i);
static GulTbItem *	galeon_navigation_button_clone_impl		(GulTbItem *i);
static void		galeon_navigation_button_parse_properties_impl	(GulTbItem *i, const gchar *props);
static void		galeon_navigation_button_menu_activated_cb	(GulTbButton *w, GaleonNavigationButton *b);
static void		galeon_navigation_button_clicked_cb		(GtkWidget *w, GaleonNavigationButton *b);


static gpointer gul_tb_item_class;

/**
 * TbiZoom object
 */

MAKE_GET_TYPE (galeon_navigation_button, "GaleonNavigationButton", GaleonNavigationButton,
	       galeon_navigation_button_class_init, 
	       galeon_navigation_button_init, GALEON_TYPE_TBI);

static void
galeon_navigation_button_class_init (GaleonNavigationButtonClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_navigation_button_finalize_impl;
	
	GUL_TB_ITEM_CLASS (klass)->get_widget = galeon_navigation_button_get_widget_impl;
	GUL_TB_ITEM_CLASS (klass)->get_icon = galeon_navigation_button_get_icon_impl;
	GUL_TB_ITEM_CLASS (klass)->get_name_human = galeon_navigation_button_get_name_human_impl;
	GUL_TB_ITEM_CLASS (klass)->to_string = galeon_navigation_button_to_string_impl;
	GUL_TB_ITEM_CLASS (klass)->clone = galeon_navigation_button_clone_impl;
	GUL_TB_ITEM_CLASS (klass)->parse_properties = galeon_navigation_button_parse_properties_impl;

	gul_tb_item_class = g_type_class_peek_parent (klass);
}

static void 
galeon_navigation_button_init (GaleonNavigationButton *tbi)
{
	GaleonNavigationButtonPrivate *p = g_new0 (GaleonNavigationButtonPrivate, 1);
	tbi->priv = p;

	p->direction = GALEON_NAVIGATION_DIRECTION_UP;
	p->show_arrow = TRUE;
	p->sensitive = TRUE;
}

GaleonNavigationButton *
galeon_navigation_button_new (void)
{
	GaleonNavigationButton *ret = g_object_new (GALEON_TYPE_NAVIGATION_BUTTON, NULL);
	return ret;
}

static void
galeon_navigation_button_finalize_impl (GObject *o)
{
	GaleonNavigationButton *it = GALEON_NAVIGATION_BUTTON (o);
	GaleonNavigationButtonPrivate *p = it->priv;

	if (p->widget)
	{
		g_object_unref (p->widget);
	}

	g_free (p);
	
	DEBUG_MSG (("GaleonNavigationButton finalized\n"));
	
	G_OBJECT_CLASS (gul_tb_item_class)->finalize (o);
}

static void
galeon_navigation_button_setup_widget (GaleonNavigationButton *b)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	const gchar *label;
	const gchar *tip;
	gboolean prio;

	if (!p->widget)
	{
		galeon_navigation_button_get_widget_impl (GUL_TB_ITEM (b));
	}
	g_assert (GUL_IS_TB_BUTTON (p->widget));

	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_UP:
		label = "gtk-go-up";
		tip = _("Go up");
		prio = FALSE;
		break;
	case GALEON_NAVIGATION_DIRECTION_BACK:
		label = "gtk-go-back";
		tip = _("Go back");
		prio = TRUE;
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
		label = "gtk-go-forward";
		tip = _("Go forward");
		prio = FALSE;
		break;
	default:
		g_assert_not_reached ();
		label = NULL;
		tip = NULL;
		prio = FALSE;
		break;
	}

	gul_tb_button_set_label (p->widget, label);
	gul_tb_button_set_tooltip_text (p->widget, tip);
	gul_tb_button_set_priority (p->widget, prio);
	gul_tb_button_set_show_arrow (p->widget, p->show_arrow);
	gul_tb_button_set_sensitivity (p->widget, p->sensitive);
}

static GtkWidget *
galeon_navigation_button_get_widget_impl (GulTbItem *i)
{
	GaleonNavigationButton *iz = GALEON_NAVIGATION_BUTTON (i);
	GaleonNavigationButtonPrivate *p = iz->priv;

	if (!p->widget)
	{
		p->widget = gul_tb_button_new ();
		g_object_ref (p->widget);
		gtk_object_sink (GTK_OBJECT (p->widget));

		gul_tb_button_set_use_stock (p->widget, TRUE);
		gul_tb_button_set_enable_menu (p->widget, TRUE);
		
		galeon_navigation_button_setup_widget (iz);

		gtk_widget_show (GTK_WIDGET (p->widget));
		
		g_signal_connect (p->widget, "menu-activated", 
				  G_CALLBACK (galeon_navigation_button_menu_activated_cb), i);
		g_signal_connect (gul_tb_button_get_button (p->widget), "clicked",
				  G_CALLBACK (galeon_navigation_button_clicked_cb), i);
	}

	return GTK_WIDGET (p->widget);
}

static GdkPixbuf *
galeon_navigation_button_get_icon_impl (GulTbItem *i)
{
	GaleonNavigationButtonPrivate *p = GALEON_NAVIGATION_BUTTON (i)->priv;

	static GdkPixbuf *pb_up = NULL;
	static GdkPixbuf *pb_back = NULL;
	static GdkPixbuf *pb_forward = NULL;

	if (!pb_up)
	{
		/* what's the easier way? */
		GtkWidget *b = gtk_button_new ();
		pb_up = gtk_widget_render_icon (b,
						GTK_STOCK_GO_UP,
						GTK_ICON_SIZE_SMALL_TOOLBAR,
						NULL);
		pb_back = gtk_widget_render_icon (b,
						  GTK_STOCK_GO_BACK,
						  GTK_ICON_SIZE_SMALL_TOOLBAR,
						  NULL);
		pb_forward = gtk_widget_render_icon (b,
						     GTK_STOCK_GO_FORWARD,
						     GTK_ICON_SIZE_SMALL_TOOLBAR,
						     NULL);
		gtk_widget_destroy (b);
	}

	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_BACK:
		return g_object_ref (pb_back);
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
		return g_object_ref (pb_forward);
		break;
	case GALEON_NAVIGATION_DIRECTION_UP:
		return g_object_ref (pb_up);
		break;
	default:
		g_assert_not_reached ();
		return NULL;
	}
}

static gchar *
galeon_navigation_button_get_name_human_impl (GulTbItem *i)
{
	GaleonNavigationButtonPrivate *p = GALEON_NAVIGATION_BUTTON (i)->priv;
	const gchar *ret;

	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_BACK:
		ret = p->show_arrow 
			? _("Back (with menu)")
			: _("Back");
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
		ret = p->show_arrow 
			? _("Forward (with menu)")
			: _("Forward");
		break;
	case GALEON_NAVIGATION_DIRECTION_UP:
		ret = p->show_arrow 
			? _("Up (with menu)")
			: _("Up");
		break;
	default:
		g_assert_not_reached ();
		ret = "Error: unexpected direction";
		break;
	}

	return g_strdup (ret);
}

static gchar *
galeon_navigation_button_to_string_impl (GulTbItem *i)
{
	GaleonNavigationButtonPrivate *p = GALEON_NAVIGATION_BUTTON (i)->priv;

	/* if it had any properties, the string should include them */
	const char *sdir;

	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_BACK:
		sdir = "back";
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
		sdir = "forward";
		break;
	case GALEON_NAVIGATION_DIRECTION_UP:
		sdir = "up";
		break;
	default:
		g_assert_not_reached ();
		sdir = "unknown";
	}

	return g_strdup_printf ("%s=navigation_button(direction=%s,arrow=%s)", 
				i->id, sdir, p->show_arrow ? "TRUE" : "FALSE");
}

static GulTbItem *
galeon_navigation_button_clone_impl (GulTbItem *i)
{
	GulTbItem *ret = GUL_TB_ITEM (galeon_navigation_button_new ());
	GaleonNavigationButtonPrivate *p = GALEON_NAVIGATION_BUTTON (i)->priv;
	
	gul_tb_item_set_id (ret, i->id);

	galeon_navigation_button_set_direction (GALEON_NAVIGATION_BUTTON (ret), p->direction);
	galeon_navigation_button_set_show_arrow (GALEON_NAVIGATION_BUTTON (ret), p->show_arrow);
	
	return ret;
}

static void
galeon_navigation_button_parse_properties_impl (GulTbItem *it, const gchar *props)
{
	GaleonNavigationButton *b = GALEON_NAVIGATION_BUTTON (it);

	/* we have two properties, the direction and the arrow */
	const gchar *direc_prop;
	const gchar *show_arrow_prop;

	direc_prop = strstr (props, "direction=");
	if (direc_prop)
	{
		direc_prop += strlen ("direction=");
		if (!strncmp (direc_prop, "back", 4))
		{
			galeon_navigation_button_set_direction (b, GALEON_NAVIGATION_DIRECTION_BACK);
		}
		else if (!strncmp (direc_prop, "forward", 4))
		{
			galeon_navigation_button_set_direction (b, GALEON_NAVIGATION_DIRECTION_FORWARD);
		}
		else if (!strncmp (direc_prop, "up", 2))
		{
			galeon_navigation_button_set_direction (b, GALEON_NAVIGATION_DIRECTION_UP);
		}
	}

	show_arrow_prop = strstr (props, "arrow=");
	if (show_arrow_prop)
	{
		show_arrow_prop += strlen ("arrow=");
		if (show_arrow_prop[0] == 'T')
		{
			galeon_navigation_button_set_show_arrow (b, TRUE);
		}
		else
		{
			galeon_navigation_button_set_show_arrow (b, FALSE);
		}
	}
}


void
galeon_navigation_button_set_direction (GaleonNavigationButton *b,
					GaleonNavigationDirection d)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	p->direction = d;
	galeon_navigation_button_setup_widget (b);
}

void
galeon_navigation_button_set_show_arrow	(GaleonNavigationButton *b,
					 gboolean value)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	p->show_arrow = value;
	if (p->widget)
	{
		gul_tb_button_set_show_arrow (p->widget, p->show_arrow);
	}
	else
	{
		galeon_navigation_button_get_widget_impl (GUL_TB_ITEM (b));
	}
}

GaleonNavigationDirection
galeon_navigation_button_get_direction (GaleonNavigationButton *b)
{
	return b->priv->direction;
}

void
galeon_navigation_button_set_sensitive (GaleonNavigationButton *b, gboolean s)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	p->sensitive = s;
	if (p->widget)
	{
		gul_tb_button_set_sensitivity (p->widget, s);
	}
	else
	{
		galeon_navigation_button_get_widget_impl (GUL_TB_ITEM (b));
	}
}

static void
galeon_navigation_button_clicked_cb (GtkWidget *w, GaleonNavigationButton *b)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	GaleonWindow *window;
	GaleonEmbed *embed;

	window = galeon_tbi_get_window (GALEON_TBI (b));
	g_return_if_fail (window != NULL);

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_UP:
		galeon_embed_go_up (embed);
		break;
	case GALEON_NAVIGATION_DIRECTION_BACK:
		galeon_embed_go_back (embed);
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
		galeon_embed_go_forward (embed);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}

/* TODO: clean all this, it came from toolbar.c and is messy */

static GtkWidget *
new_num_accel_menu_item (gint num, gchar *origtext, gboolean lettersok, const GdkPixbuf *ico)
{
        gchar *text = gul_string_new_num_accel (num, origtext, lettersok);
        GtkWidget *item = gtk_image_menu_item_new ();
        GtkWidget *hb = gtk_hbox_new (FALSE, 0);
        GtkWidget *label = gtk_label_new_with_mnemonic (text);

        gtk_box_pack_start (GTK_BOX (hb), label, FALSE, FALSE, 0);
        gtk_container_add (GTK_CONTAINER (item), hb);

        gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item),
                                       gtk_image_new_from_pixbuf ((GdkPixbuf *) ico));

        gtk_widget_show_all (item);

        return item;
}

static void
activate_back_or_forward_menu_item_cb (GtkWidget *menu, GaleonWindow *window)
{
	GaleonEmbed *embed;
	int go_nth;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	go_nth = (int)g_object_get_data (G_OBJECT(menu), "go_nth");

	galeon_embed_shistory_go_nth (embed, go_nth);
}

static void
activate_up_menu_item_cb (GtkWidget *menu, GaleonWindow *window)
{
	GaleonEmbed *embed;
	int go_nth;
	GSList *l;
	gchar *url;
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	go_nth = (int)g_object_get_data (G_OBJECT(menu), "go_nth");

	galeon_embed_get_go_up_list (embed, &l);

	url = g_slist_nth_data (l, go_nth);
	if (url)
	{
		galeon_embed_load_url (embed, url);
	}

	g_slist_foreach (l, (GFunc) g_free, NULL);
	g_slist_free (l);
}

static void
setup_back_or_forward_menu (GaleonWindow *window, GtkMenuShell *ms, GaleonNavigationDirection dir)
{
	int pos, count;
	GaleonEmbed *embed;
	int start, end, accell_count = 0;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);
	
	galeon_embed_shistory_get_pos (embed, &pos);
	galeon_embed_shistory_count (embed, &count);

	if (count == 0) return;
	
	if (dir == GALEON_NAVIGATION_DIRECTION_BACK)
	{
		start = pos - 1;
		end = -1;
	}
	else
	{
		start = pos + 1;
		end = count;
	}

	while (start != end)
	{
		char *title, *url, *tmp;
		GtkWidget *item;
		galeon_embed_shistory_get_nth (embed, start, FALSE, &url, &title);
		tmp = gul_string_escape_uline_accel (title);
		item = new_num_accel_menu_item (accell_count, tmp, TRUE, NULL);
        	gtk_menu_shell_append (ms, item);
		g_object_set_data (G_OBJECT (item), "go_nth", GINT_TO_POINTER (start));
		g_signal_connect (item, "activate",
                                  G_CALLBACK (activate_back_or_forward_menu_item_cb), window);
		gtk_widget_show_all (item);

		g_free (url);
		g_free (title);
		g_free (tmp);

		accell_count++;
		if (start < end)
		{
			start++;
		}
		else
		{
			start--;
		}
	}
}

static void
setup_up_menu (GaleonWindow *window, GtkMenuShell *ms)
{
	GaleonEmbed *embed;
	int accell_count = 0;
	GSList *l;
	GSList *li;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);
	
	galeon_embed_get_go_up_list (embed, &l);
	
	for (li = l; li; li = li->next)
	{
		char *url = li->data;
		char *tmp;
		GtkWidget *item;
		tmp = gul_string_escape_uline_accel (url);
		item = new_num_accel_menu_item (accell_count, tmp, TRUE, NULL);
        	gtk_menu_shell_append (ms, item);
		g_object_set_data (G_OBJECT(item), "go_nth", GINT_TO_POINTER (accell_count));
		g_signal_connect (item, "activate",
                                  G_CALLBACK (activate_up_menu_item_cb), window);
		gtk_widget_show_all (item);

		accell_count++;
		g_free (tmp);
	}
	
	g_slist_foreach (l, (GFunc) g_free, NULL);
	g_slist_free (l);
}

static void
galeon_navigation_button_menu_activated_cb (GulTbButton *w, GaleonNavigationButton *b)
{
	GaleonNavigationButtonPrivate *p = b->priv;
	GtkMenuShell *ms = gul_tb_button_get_menu (p->widget);
	GaleonWindow *win = galeon_tbi_get_window (GALEON_TBI (b));
	GList *children;
	GList *li;

	children = gtk_container_get_children (GTK_CONTAINER (ms));
	for (li = children; li; li = li->next)
	{
		gtk_container_remove (GTK_CONTAINER (ms), li->data);
	}
	g_list_free (children);
	
	switch (p->direction)
	{
	case GALEON_NAVIGATION_DIRECTION_UP:
		setup_up_menu (win, ms);
		break;
	case GALEON_NAVIGATION_DIRECTION_FORWARD:
	case GALEON_NAVIGATION_DIRECTION_BACK:
		setup_back_or_forward_menu (win, ms, p->direction);
		break;
	default:
		g_assert_not_reached ();
		break;
	}
}
