/*
 *  Copyright (C) 2000, 2001, 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "galeon-window.h"
#include "galeon-sidebar.h"
#include "gul-state.h"
#include "gul-gobject-misc.h"
#include "gul-notebook.h"
#include "statusbar.h"
#include "toolbar.h"
#include "ppview-toolbar.h"
#include "window-commands.h"
#include "find-dialog.h"
#include "history-dialog.h"
#include "popup-commands.h"
#include "galeon-shell.h"
#include "bookmarks-bonoboui-menu.h"
#include "bookmarks-bonoboui-tb.h"
#include "gul-bonobo-extensions.h"
#include "eel-gconf-extensions.h"
#include "prefs-strings.h"
#include "galeon-embed-utils.h"
#include "galeon-embed-prefs.h"
#include "bookmarks-editor-dockable.h"
#include "window-recent-history-menu.h"
#include "galeon-embed-prefs.h"

#include <stdlib.h>
#include <string.h>
#include <bonobo/bonobo-window.h>
#include <bonobo/bonobo-i18n.h>
#include <bonobo/bonobo-ui-util.h>
#include <bonobo/bonobo-ui-component.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gtk/gtk.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>

#define CHARSET_MENU_PATH "/menu/View/EncodingMenuPlaceholder"
#define GO_RECENT_HISTORY_PATH "/menu/Go/RecentHistory"

#define GO_BACK_CMD_PATH "/commands/GoBack"
#define GO_FORWARD_CMD_PATH "/commands/GoForward"
#define GO_UP_CMD_PATH "/commands/GoUp"
#define GO_STOP_CMD_PATH "/commands/GoStop"
#define EDIT_FIND_NEXT_CMD_PATH "/commands/EditFindNext"
#define EDIT_FIND_PREV_CMD_PATH "/commands/EditFindPrev"
#define VIEW_SIDEBAR_PATH "/commands/View Sidebar"
#define VIEW_MENUBAR_PATH "/commands/View Menubar"
#define VIEW_STATUSBAR_PATH "/commands/View Statusbar"
#define VIEW_TOOLBAR_PATH "/commands/View Toolbar"
#define VIEW_BOOKMARKS_TOOLBAR_PATH "/commands/View Bookmarks Toolbars"
#define VIEW_FULLSCREEN_PATH "/commands/View Fullscreen"
#define ALLOW_POPUPS_TOGGLE_PATH "/commands/Allow Popups"
#define ALLOW_JAVA_PATH "/commands/Allow Java"
#define ALLOW_JAVASCRIPT_PATH "/commands/Allow JavaScript"
#define ALLOW_POPUPS_CMD_PATH "/commands/AllowPopupSite"
#define ALLOW_COOKIES_CMD_PATH "/commands/AllowCookieSite"
#define BLOCK_COOKIES_CMD_PATH "/commands/BlockCookieSite"
#define ALLOW_IMAGES_CMD_PATH "/commands/AllowImageSite"
#define BLOCK_IMAGES_CMD_PATH "/commands/BlockImageSite"
#define ALLOW_POPUPS_MENU_PATH "/menu/Web/Popups/AllowPopupSite"
#define ALLOW_COOKIES_MENU_PATH "/menu/Web/Cookies/AllowCookieSite"
#define BLOCK_COOKIES_MENU_PATH "/menu/Web/Cookies/BlockCookieSite"
#define ALLOW_IMAGES_MENU_PATH "/menu/View/Images/AllowImageSite"
#define BLOCK_IMAGES_MENU_PATH "/menu/View/Images/BlockImageSite"


#define ID_VIEW_SIDEBAR "View Sidebar"
#define ID_VIEW_MENUBAR "View Menubar"
#define ID_VIEW_STATUSBAR "View Statusbar"
#define ID_VIEW_TOOLBAR "View Toolbar"
#define ID_VIEW_BOOKMARKS_TOOLBAR "View Bookmarks Toolbars"
#define ID_VIEW_FULLSCREEN "View Fullscreen"
#define ID_ALLOW_POPUPS "Allow Popups"
#define ID_ALLOW_JAVA "Allow Java"
#define ID_ALLOW_JAVASCRIPT "Allow JavaScript"

typedef enum {
	IMAGES,
	COOKIES
}MenuRadioGroup;

#define VIEW_IMAGES_ALWAYS	0
#define VIEW_IMAGES_CURRENT	1
#define VIEW_IMAGES_NEVER	2
#define ACCEPT_COOKIES_ALWAYS	0
#define ACCEPT_COOKIES_CURRENT	1
#define ACCEPT_COOKIES_NEVER	2

static BonoboUIVerb galeon_verbs [] = {
        BONOBO_UI_VERB ("EditFind", (BonoboUIVerbFn)window_cmd_edit_find),
	BONOBO_UI_VERB ("FilePrint", (BonoboUIVerbFn)window_cmd_file_print),
	BONOBO_UI_VERB ("GoStop", (BonoboUIVerbFn)window_cmd_go_stop),
	BONOBO_UI_VERB ("GoReload", (BonoboUIVerbFn)window_cmd_go_reload),
	BONOBO_UI_VERB ("GoBack", (BonoboUIVerbFn)window_cmd_go_back),
	BONOBO_UI_VERB ("GoForward", (BonoboUIVerbFn)window_cmd_go_forward),
	BONOBO_UI_VERB ("GoGo", (BonoboUIVerbFn)window_cmd_go_go),
	BONOBO_UI_VERB ("GoUp", (BonoboUIVerbFn)window_cmd_go_up),
	BONOBO_UI_VERB ("GoHome", (BonoboUIVerbFn)window_cmd_go_home),
	BONOBO_UI_VERB ("GoMyportal", (BonoboUIVerbFn)window_cmd_go_myportal),
	BONOBO_UI_VERB ("GoLocation", (BonoboUIVerbFn)window_cmd_go_location),
	BONOBO_UI_VERB ("FileNew", (BonoboUIVerbFn)window_cmd_new),
	BONOBO_UI_VERB ("FileNewWindow", (BonoboUIVerbFn)window_cmd_new_window),
	BONOBO_UI_VERB ("FileNewTab", (BonoboUIVerbFn)window_cmd_new_tab),
	BONOBO_UI_VERB ("FileOpen", (BonoboUIVerbFn)window_cmd_file_open),
	BONOBO_UI_VERB ("FileSaveAs", (BonoboUIVerbFn)window_cmd_file_save_as),
	BONOBO_UI_VERB ("FileQuit", (BonoboUIVerbFn)window_cmd_file_quit),
	BONOBO_UI_VERB ("FileCloseTab", (BonoboUIVerbFn)window_cmd_file_close_tab),
	BONOBO_UI_VERB ("FileSendTo", (BonoboUIVerbFn)window_cmd_file_send_to),
	BONOBO_UI_VERB ("EditCut", (BonoboUIVerbFn)window_cmd_edit_cut),
	BONOBO_UI_VERB ("EditCopy", (BonoboUIVerbFn)window_cmd_edit_copy),
	BONOBO_UI_VERB ("EditPaste", (BonoboUIVerbFn)window_cmd_edit_paste),
	BONOBO_UI_VERB ("EditSelectAll", (BonoboUIVerbFn)window_cmd_edit_select_all),
	BONOBO_UI_VERB ("EditPrefs", (BonoboUIVerbFn)window_cmd_edit_prefs),
	BONOBO_UI_VERB ("SettingsToolbarEditor", (BonoboUIVerbFn)window_cmd_settings_toolbar_editor),
	BONOBO_UI_VERB ("Zoom In", (BonoboUIVerbFn)window_cmd_view_zoom_in),
	BONOBO_UI_VERB ("EditFindNext", (BonoboUIVerbFn)window_cmd_edit_find_next),
	BONOBO_UI_VERB ("EditFindPrev", (BonoboUIVerbFn)window_cmd_edit_find_prev),
	BONOBO_UI_VERB ("Zoom Out", (BonoboUIVerbFn)window_cmd_view_zoom_out),
	BONOBO_UI_VERB ("Zoom Normal", (BonoboUIVerbFn)window_cmd_view_zoom_normal),
	BONOBO_UI_VERB ("ViewPageSource", (BonoboUIVerbFn)window_cmd_view_page_source),
	BONOBO_UI_VERB ("AllowPopupSite", (BonoboUIVerbFn)window_cmd_set_permission),
	BONOBO_UI_VERB ("BlockImageSite", (BonoboUIVerbFn)window_cmd_set_permission),
	BONOBO_UI_VERB ("AllowImageSite", (BonoboUIVerbFn)window_cmd_set_permission),
	BONOBO_UI_VERB ("BlockCookieSite", (BonoboUIVerbFn)window_cmd_set_permission),
	BONOBO_UI_VERB ("AllowCookieSite", (BonoboUIVerbFn)window_cmd_set_permission),
	BONOBO_UI_VERB ("ShowJavaConsole", (BonoboUIVerbFn)window_cmd_show_java_console),
	BONOBO_UI_VERB ("ShowJSConsole", (BonoboUIVerbFn)window_cmd_show_js_console),
	BONOBO_UI_VERB ("BookmarksAddDefault", (BonoboUIVerbFn)window_cmd_bookmarks_add_default),
	BONOBO_UI_VERB ("BookmarksEdit", (BonoboUIVerbFn)window_cmd_bookmarks_edit),
	BONOBO_UI_VERB ("ToolsHistory", (BonoboUIVerbFn)window_cmd_tools_history),
	BONOBO_UI_VERB ("ToolsPDM", (BonoboUIVerbFn)window_cmd_tools_pdm),
	BONOBO_UI_VERB ("TabsNext", (BonoboUIVerbFn)window_cmd_tabs_next),
	BONOBO_UI_VERB ("TabsPrevious", (BonoboUIVerbFn)window_cmd_tabs_previous),
	BONOBO_UI_VERB ("TabsMoveLeft", (BonoboUIVerbFn)window_cmd_tabs_move_left),
	BONOBO_UI_VERB ("TabsMoveRight", (BonoboUIVerbFn)window_cmd_tabs_move_right),
	BONOBO_UI_VERB ("TabsDetach", (BonoboUIVerbFn)window_cmd_tabs_detach),
	BONOBO_UI_VERB ("TabsClone", (BonoboUIVerbFn)window_cmd_tabs_clone),
	BONOBO_UI_VERB ("HelpContents", (BonoboUIVerbFn)window_cmd_help_manual),
	BONOBO_UI_VERB ("About", (BonoboUIVerbFn)window_cmd_help_about),

        BONOBO_UI_VERB_END
};

static BonoboUIVerb galeon_popup_verbs [] = {
        BONOBO_UI_VERB ("EPOpenInNewWindow", (BonoboUIVerbFn)popup_cmd_new_window),
	BONOBO_UI_VERB ("EPOpenInNewTab", (BonoboUIVerbFn)popup_cmd_new_tab),
	BONOBO_UI_VERB ("EPAddBookmark", (BonoboUIVerbFn)popup_cmd_add_bookmark),
	BONOBO_UI_VERB ("EPOpenImageInNewWindow", (BonoboUIVerbFn)popup_cmd_image_in_new_window),
	BONOBO_UI_VERB ("EPOpenImageInNewTab", (BonoboUIVerbFn)popup_cmd_image_in_new_tab),
	BONOBO_UI_VERB ("EPBlockImageSite", (BonoboUIVerbFn)popup_cmd_image_block_site),
	BONOBO_UI_VERB ("DPOpenFrameInNewWindow", (BonoboUIVerbFn)popup_cmd_frame_in_new_window),
	BONOBO_UI_VERB ("DPOpenFrameInNewTab", (BonoboUIVerbFn)popup_cmd_frame_in_new_tab),
	BONOBO_UI_VERB ("DPAddFrameBookmark", (BonoboUIVerbFn)popup_cmd_add_frame_bookmark),
	BONOBO_UI_VERB ("DPViewSource", (BonoboUIVerbFn)popup_cmd_view_source),
	
        BONOBO_UI_VERB_END
};

struct GaleonWindowPrivate
{
	PPViewToolbar *ppview_toolbar;
	Toolbar *toolbar;
	Statusbar *statusbar;
	GbBonoboUIMenu *bookmarks_menu;
	GbBonoboUIMenu *bookmarks_context_menu;
	GbBonoboUIToolbar *bookmarks_toolbar;
	GtkNotebook *notebook;
	GtkWidget *hpaned;
	GaleonTab *active_tab;
	GtkWidget *sidebar;
	GaleonEmbedPopupBW *embed_popup;
	GaleonDialog *find_dialog;
	GaleonDialog *history_dialog;
	GaleonDialog *history_sidebar;
	GbEditorDockable *bookmarks_sidebar;
	EmbedChromeMask chrome_mask;
	gboolean ignore_layout_toggles;
	gboolean has_default_size;
	gboolean closing;
	GaleonWindowRecentHistory *recent_history;
	GaleonWindowRecentHistoryMenu *recent_history_menu;
};

static void
galeon_window_class_init (GaleonWindowClass *klass);
static void
galeon_window_gb_location_source_init (GbLocationSourceIface *iface);
static void
galeon_window_init (GaleonWindow *gs);
static void
galeon_window_finalize (GObject *object);
static void
galeon_window_show (GtkWidget *widget);
static void
galeon_window_notebook_switch_page_cb (GtkNotebook *notebook,
				       GtkNotebookPage *page,
				       guint page_num,
				       GaleonWindow *window);
static void
galeon_window_bookmark_activated_cb (GObject *sender,
				     GbBookmarkEventActivated *ev,
				     GaleonWindow *w);

static void             
galeon_window_tab_detached_cb    (GulNotebook *notebook, gint page,
				  gint x, gint y, GaleonWindow *window);

void /* yes, this is not static */
galeon_window_bookmark_activate (GaleonWindow *w, GbBookmarkEventActivated *ev);

static char *
galeon_window_gb_location_source_get_location (GbLocationSource *src);
static char *
galeon_window_gb_location_source_get_title (GbLocationSource *src);

static void
galeon_window_recent_history_menu_activated_cb (GaleonWindowRecentHistoryMenu *wrhm, 
						const char *url, const char *title, 
						GaleonWindow *w);

/* static class variables */
static GdkColor _galeon_window_loading_tab_color;
static GdkColor _galeon_window_new_tab_color;

static GObjectClass *parent_class = NULL;

MAKE_GET_TYPE_IFACE (galeon_window, "GaleonWindow", GaleonWindow,
		     galeon_window_class_init, galeon_window_init, BONOBO_TYPE_WINDOW,
		     galeon_window_gb_location_source_init, GB_TYPE_LOCATION_SOURCE);

static void
galeon_window_class_init (GaleonWindowClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);
	char *tmp;
	
        parent_class = g_type_class_peek_parent (klass);

        object_class->finalize = galeon_window_finalize;
	
	widget_class->show = galeon_window_show;

	/* initialize colors */
	tmp = eel_gconf_get_string (CONF_TABS_TABBED_LOADING_COLOR);
	gdk_color_parse (tmp, &_galeon_window_loading_tab_color);
	g_free (tmp);

	tmp = eel_gconf_get_string (CONF_TABS_TABBED_NEW_COLOR);
	gdk_color_parse (tmp, &_galeon_window_new_tab_color);
	g_free (tmp);
}

static void
galeon_window_gb_location_source_init (GbLocationSourceIface *iface)
{
	iface->get_location = galeon_window_gb_location_source_get_location;
	iface->get_title = galeon_window_gb_location_source_get_title;
}

static
gboolean 
galeon_window_key_press_event_cb (GtkWidget *widget,
				  GdkEventKey *event,
				  GaleonWindow *window)
{
        int page;

	g_print("Window keypress\n");

	if ((event->state & GDK_Shift_L) || (event->state & GDK_Shift_R))
                return FALSE;

        if ((event->state & GDK_Alt_L) || (event->state & GDK_Alt_R))
        {
                page = event->keyval - GDK_0 -1;

                if (page == -1) page = 9;

                if (page>=-1 && page<=9)
                {
                        gtk_notebook_set_current_page 
				(GTK_NOTEBOOK (window->priv->notebook),
                                 page == -1 ? -1 : page);
                        return TRUE;
                }
        }
	
        return FALSE;
}

static void
galeon_window_selection_received_cb (GtkWidget *widget,
				     GtkSelectionData *selection_data,
				     guint time, GaleonWindow *window)
{
	GaleonTab *tab;
	gint action;

	if (selection_data->length <= 0 || selection_data->data == NULL)
		return;

	tab = galeon_window_get_active_tab(window);
	action = eel_gconf_get_integer(CONF_MOUSE_MIDDLE_BUTTON_ACTION);
	
	if (action == 6) //Paste into current tab
	{
		galeon_window_load_url(window, selection_data->data);
	}
	else //Paste into new tab (action == 1 or default)
	{
		galeon_shell_new_tab(galeon_shell, window, tab,
 				     selection_data->data, 0);
	}
}
	
static void
save_window_state (GtkWidget *win)
{
	GaleonWindow *window = GALEON_WINDOW (win);
	
	if (!(window->priv->chrome_mask & EMBED_CHROME_OPENASPOPUP) &&
	    !(window->priv->chrome_mask & EMBED_CHROME_OPENASFULLSCREEN))
	{
		gul_state_save_window (win, "main_window");
	}
}

static gboolean
galeon_window_configure_event_cb (GtkWidget *widget,
				  GdkEventConfigure *event,
				  gpointer data)
{	
	save_window_state (widget);
	return FALSE;
}

static gboolean
galeon_window_state_event_cb (GtkWidget *widget,
			      GdkEventWindowState *event,
			      gpointer data)
{
	save_window_state (widget);
	return FALSE;
}

static void
setup_bonobo_window (GaleonWindow *window,
		     BonoboUIComponent **ui_component)
{
	BonoboWindow *win = BONOBO_WINDOW(window);
	BonoboUIContainer *container;
	Bonobo_UIContainer corba_container;

	container = bonobo_window_get_ui_container (win);
        
        bonobo_ui_engine_config_set_path (bonobo_window_get_ui_engine (win),
                                          "/apps/galeon/UIConfig/kvps");

        corba_container = BONOBO_OBJREF (container);

	*ui_component = bonobo_ui_component_new_default ();
	
	bonobo_ui_component_set_container (*ui_component, 
					   corba_container, 
					   NULL);

	bonobo_ui_util_set_ui (*ui_component,
			       DATADIR,
#ifdef ENABLE_SIMPLE_MENUBAR
			       "galeon-simple-ui.xml",
#else
			       "galeon-ui.xml",
#endif
			       "galeon", NULL);

	/* Key handling */
	g_signal_connect(G_OBJECT(win), 
			 "key-press-event",
                         G_CALLBACK(galeon_window_key_press_event_cb),
                         window);

	g_signal_connect (G_OBJECT(win), "configure_event",
			  G_CALLBACK (galeon_window_configure_event_cb), NULL);
	g_signal_connect (G_OBJECT(win), "window_state_event",
			  G_CALLBACK (galeon_window_state_event_cb), NULL);
	
	g_signal_connect (G_OBJECT(win),
			  "selection-received",
			  G_CALLBACK (galeon_window_selection_received_cb), 
			  window);
}

static void
galeon_window_popup_before_show_cb (GaleonEmbedPopup *popup, GaleonWindow *window)
{
	/* we need this, because bonoboui sucks */
	gb_bonobo_ui_menu_rebuild (window->priv->bookmarks_context_menu);
}

static GaleonEmbedPopupBW *
setup_popup_factory (GaleonWindow *window, 
		     BonoboUIComponent *ui_component)
{
	GaleonEmbedPopupBW *popup;
	
	popup = galeon_window_get_popup_factory (window);
	g_object_set_data (G_OBJECT(popup), "GaleonWindow", window);
	bonobo_ui_component_add_verb_list_with_data (ui_component, 
                                		     galeon_popup_verbs, 
						     popup);

	g_signal_connect (popup, "before-show", 
			  G_CALLBACK (galeon_window_popup_before_show_cb), window);

	return popup;
}

static void
tabbed_position_gconf_changed_cb(GConfClient *client,
                             guint cnxn_id,
                             GConfEntry *entry,
                             GtkNotebook *notebook)
{
	GtkPositionType position = eel_gconf_get_integer (CONF_TABS_TABBED_EDGE);    
	gtk_notebook_set_tab_pos (notebook, position);
}

static GtkNotebook *
setup_notebook (GaleonWindow *window)
{
	GtkNotebook *notebook;
	GtkPolicyType policy;
	GtkPositionType position = eel_gconf_get_integer (CONF_TABS_TABBED_EDGE);

	policy = eel_gconf_get_boolean (CONF_TABS_TABBED_ALWAYS_SHOW)
		 ? GTK_POLICY_ALWAYS
		 : GTK_POLICY_AUTOMATIC;
	
	notebook = GTK_NOTEBOOK (gul_notebook_new ());
	gtk_notebook_set_scrollable (notebook, TRUE);
	gtk_notebook_set_show_border (notebook, FALSE);
	gtk_notebook_set_tab_pos (notebook, position);
	gul_notebook_set_policy (GUL_NOTEBOOK (notebook), policy);
	gtk_notebook_popup_enable (notebook);

	g_signal_connect_after (G_OBJECT (notebook), "switch_page",
				G_CALLBACK (
				galeon_window_notebook_switch_page_cb),
				window);

	g_signal_connect (G_OBJECT (notebook), "tab_detached", 
			  G_CALLBACK (galeon_window_tab_detached_cb), 
			  window);

	eel_gconf_notification_add(CONF_TABS_TABBED_EDGE,
				   (GConfClientNotifyFunc)tabbed_position_gconf_changed_cb,
				   notebook);
    
	gtk_widget_show (GTK_WIDGET (notebook));

	return notebook;
}

static GbBonoboUIMenu *
setup_bookmarks_menu (GaleonWindow *window, GbBookmarkSet *bs)
{
	GbBonoboUIMenu *menu;
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT (window->ui_component);
	
	menu = gb_bonobo_ui_menu_new (bs->root, ui_component, 
				      "/menu/Bookmarks/BookmarksPlaceholder", FALSE);
	gb_bonobo_ui_menu_set_location_source (menu, GB_LOCATION_SOURCE (window));

	g_signal_connect (menu, "bookmark-activated", 
			  G_CALLBACK (galeon_window_bookmark_activated_cb), window);

	return menu;
}

static GbBonoboUIMenu *
setup_bookmarks_context_menu (GaleonWindow *window, GbBookmarkSet *bs)
{
	GbBonoboUIMenu *menu;
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT (window->ui_component);
	
	menu = gb_bonobo_ui_menu_new (bs->root, ui_component, 
				      "/popups/GaleonEmbedDocumentPopup/BookmarksInContext", TRUE);
	gb_bonobo_ui_menu_set_location_source (menu, GB_LOCATION_SOURCE (window));

	g_signal_connect (menu, "bookmark-activated", 
			  G_CALLBACK (galeon_window_bookmark_activated_cb), window);

	return menu;
}

static GbBonoboUIToolbar *
setup_bookmarks_toolbar (GaleonWindow *window, GbBookmarkSet *bs)
{
	GbBonoboUIToolbar *toolbar;
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT (window->ui_component);
	
	toolbar = gb_bonobo_ui_toolbar_new (bs, ui_component);
	gb_bonobo_ui_toolbar_set_location_source (toolbar, GB_LOCATION_SOURCE (window));
	
	g_signal_connect (toolbar, "bookmark-activated", 
			  G_CALLBACK (galeon_window_bookmark_activated_cb), window);

	return toolbar;
}

static void
view_toolbar_changed_cb (BonoboUIComponent *component,
                         const char *path,
                         Bonobo_UIComponent_EventType type,
                         const char *state,
                         GaleonWindow *window)
{
	EmbedChromeMask mask;

	if (window->priv->ignore_layout_toggles) return;
	
        mask = galeon_window_get_chrome (window);
        mask ^= EMBED_CHROME_TOOLBARON;
        galeon_window_set_chrome (window, mask);
}

static void
view_bookmarks_toolbar_changed_cb (BonoboUIComponent *component,
				   const char *path,
				   Bonobo_UIComponent_EventType type,
				   const char *state,
				   GaleonWindow *window)
{
	EmbedChromeMask mask;
	
	if (window->priv->ignore_layout_toggles) return;

        mask = galeon_window_get_chrome (window);
        mask ^= EMBED_CHROME_PERSONALTOOLBARON;
        galeon_window_set_chrome (window, mask);
}

static void
view_sidebar_changed_cb (BonoboUIComponent *component,
                         const char *path,
                         Bonobo_UIComponent_EventType type,
                         const char *state,
                         GaleonWindow *window)
{
	EmbedChromeMask mask;

	if (window->priv->ignore_layout_toggles) return;
	
        mask = galeon_window_get_chrome (window);
        mask ^= EMBED_CHROME_SIDEBARON;
        galeon_window_set_chrome (window, mask);
}

static void
view_statusbar_changed_cb (BonoboUIComponent *component,
                           const char *path,
                           Bonobo_UIComponent_EventType type,
                           const char *state,
                           GaleonWindow *window)
{
	EmbedChromeMask mask;

	if (window->priv->ignore_layout_toggles) return;
	
        mask = galeon_window_get_chrome (window);
        mask ^= EMBED_CHROME_STATUSBARON;
        galeon_window_set_chrome (window, mask);
}

static void
view_fullscreen_changed_cb (BonoboUIComponent *component,
                            const char *path,
                            Bonobo_UIComponent_EventType type,
                            const char *state,
                            GaleonWindow *window)
{	
	if (window->priv->ignore_layout_toggles) return;

	galeon_window_toggle_fullscreen (window);
}

static void
update_layout_toggles(GaleonWindow *window)
{
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT(window->ui_component);
	EmbedChromeMask mask = window->priv->chrome_mask;
	
	window->priv->ignore_layout_toggles = TRUE;

	gul_bonobo_set_toggle_state(ui_component, VIEW_SIDEBAR_PATH, 
				    mask & EMBED_CHROME_SIDEBARON);
	gul_bonobo_set_toggle_state(ui_component, VIEW_TOOLBAR_PATH, 
				    mask & EMBED_CHROME_TOOLBARON);
	gul_bonobo_set_toggle_state(ui_component, VIEW_BOOKMARKS_TOOLBAR_PATH, 
				    mask & EMBED_CHROME_PERSONALTOOLBARON);
	gul_bonobo_set_toggle_state(ui_component, VIEW_STATUSBAR_PATH, 
				    mask & EMBED_CHROME_STATUSBARON);
	gul_bonobo_set_toggle_state(ui_component, VIEW_FULLSCREEN_PATH, 
				    mask & EMBED_CHROME_OPENASFULLSCREEN);

	window->priv->ignore_layout_toggles = FALSE;
}

static void
setup_layout_menus(GaleonWindow *window)
{
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT(window->ui_component);
	
	bonobo_ui_component_add_listener(ui_component, ID_VIEW_SIDEBAR, 
					 (BonoboUIListenerFn)view_sidebar_changed_cb, 
					 window);
	bonobo_ui_component_add_listener(ui_component, ID_VIEW_TOOLBAR, 
					 (BonoboUIListenerFn)view_toolbar_changed_cb, 
					 window);
	bonobo_ui_component_add_listener(ui_component, ID_VIEW_BOOKMARKS_TOOLBAR, 
					 (BonoboUIListenerFn)view_bookmarks_toolbar_changed_cb, 
					 window);
	bonobo_ui_component_add_listener(ui_component, ID_VIEW_STATUSBAR, 
					 (BonoboUIListenerFn)view_statusbar_changed_cb, 
					 window);
	bonobo_ui_component_add_listener(ui_component, ID_VIEW_FULLSCREEN, 
					 (BonoboUIListenerFn)view_fullscreen_changed_cb, 
					 window);
}

static void
radio_menu_group_set_from_value(BonoboUIComponent *ui_component,
				MenuRadioGroup group,
				GConfValue *value)
{
	const char *path;

	if (group == IMAGES)
	{
		switch (gconf_value_get_int(value))
		{
		  case VIEW_IMAGES_ALWAYS:
		  default:
			path = "/commands/ViewImagesAlways";
			break;
		  case VIEW_IMAGES_CURRENT:
			path = "/commands/ViewImagesCurrent";
			break;
		  case VIEW_IMAGES_NEVER:
			path = "/commands/ViewImagesNever";
			break;
		}
	}
	else //if (group == COOKIES)
	{
		switch (gconf_value_get_int(value))
		{
		  case ACCEPT_COOKIES_ALWAYS:
		  default:
			path = "/commands/AcceptCookiesAlways";
			break;
		  case ACCEPT_COOKIES_CURRENT:
			path = "/commands/AcceptCookiesCurrent";
			break;
		  case ACCEPT_COOKIES_NEVER:
			path = "/commands/AcceptCookiesNever";
			break;
		}
	}

	gul_bonobo_set_toggle_state(ui_component, path, TRUE);
}

static void
allow_setting_bonoboui_changed_cb(BonoboUIComponent *component,
				  const char *path,
				  Bonobo_UIComponent_EventType type,
				  const char *state,
				  const char *prefs_string)
{
	if (state == NULL || *state == '\0') return;

	eel_gconf_set_boolean(prefs_string, strcmp(state, "1") == 0);
}

static void
allow_setting_gconf_changed_cb(GConfClient *client,
			       guint cnxn_id,
			       GConfEntry *entry,
			       BonoboUIComponent *ui_component)
{
	gul_bonobo_set_toggle_state(ui_component, ALLOW_POPUPS_TOGGLE_PATH,
				    eel_gconf_get_boolean(CONF_FILTERING_ALLOW_POPUPS));
	gul_bonobo_set_toggle_state(ui_component, ALLOW_JAVASCRIPT_PATH,
				    eel_gconf_get_boolean(CONF_FILTERING_JAVASCRIPT_ENABLED));
	gul_bonobo_set_toggle_state(ui_component, ALLOW_JAVA_PATH,
				    eel_gconf_get_boolean(CONF_FILTERING_JAVA_ENABLED));
}

//Prototype needed as implemented later
static void update_permission_control(GaleonWindow *window);

static void
allow_popups_bonoboui_changed_cb(BonoboUIComponent *component,
				 const char *path,
				 Bonobo_UIComponent_EventType type,
				 const char *state,
				 GaleonWindow *window)
{
	update_permission_control(window);
}

static void
allow_popups_gconf_changed_cb(GConfClient *client,
			      guint cnxn_id,
			      GConfEntry *entry,
			      GaleonWindow *window)
{
	update_permission_control(window);
}

static void
accept_cookies_bonoboui_changed_cb(BonoboUIComponent *component,
				   const char *id,
				   Bonobo_UIComponent_EventType type,
				   const char *state,
				   gpointer user_data)
{
	if (state == NULL || *state == '\0') return;

	/* we only care when an item being set */
	if (!strcmp(state, "0")) return;

	eel_gconf_set_integer(CONF_PERSISTENT_COOKIES_BEHAVIOR, 
			      GPOINTER_TO_INT(user_data));
}

static void
accept_cookies_gconf_changed_cb(GConfClient *client,
				guint cnxn_id,
				GConfEntry *entry,
				BonoboUIComponent *ui_component)
{
	radio_menu_group_set_from_value(ui_component, COOKIES, entry->value);
}

static void
setup_web_menu (GaleonWindow *window)
{
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT (window->ui_component);

	radio_menu_group_set_from_value(ui_component, COOKIES,
					eel_gconf_get_value(CONF_PERSISTENT_COOKIES_BEHAVIOR));

	bonobo_ui_component_add_listener(ui_component, "AcceptCookiesAlways",
					 accept_cookies_bonoboui_changed_cb,
					 GINT_TO_POINTER(ACCEPT_COOKIES_ALWAYS));
	bonobo_ui_component_add_listener(ui_component, "AcceptCookiesCurrent",
					 accept_cookies_bonoboui_changed_cb,
					 GINT_TO_POINTER(ACCEPT_COOKIES_CURRENT));
	bonobo_ui_component_add_listener(ui_component, "AcceptCookiesNever",
					 accept_cookies_bonoboui_changed_cb,
					 GINT_TO_POINTER(ACCEPT_COOKIES_NEVER));

	eel_gconf_notification_add(CONF_PERSISTENT_COOKIES_BEHAVIOR, 
				   (GConfClientNotifyFunc)accept_cookies_gconf_changed_cb, 
				   ui_component);

	gul_bonobo_set_toggle_state(ui_component, ALLOW_POPUPS_TOGGLE_PATH,
				    eel_gconf_get_boolean (CONF_FILTERING_ALLOW_POPUPS));
	gul_bonobo_set_toggle_state(ui_component, ALLOW_JAVASCRIPT_PATH,
				    eel_gconf_get_boolean (CONF_FILTERING_JAVASCRIPT_ENABLED));
	gul_bonobo_set_toggle_state(ui_component, ALLOW_JAVA_PATH,
				    eel_gconf_get_boolean (CONF_FILTERING_JAVA_ENABLED));

	bonobo_ui_component_add_listener(ui_component, ID_ALLOW_POPUPS,
					 (BonoboUIListenerFn)allow_popups_bonoboui_changed_cb,
					 window);
	bonobo_ui_component_add_listener(ui_component, ID_ALLOW_POPUPS,
					 (BonoboUIListenerFn)allow_setting_bonoboui_changed_cb,
					 CONF_FILTERING_ALLOW_POPUPS);
	bonobo_ui_component_add_listener(ui_component, ID_ALLOW_JAVASCRIPT,
					 (BonoboUIListenerFn)allow_setting_bonoboui_changed_cb,
					 CONF_FILTERING_JAVASCRIPT_ENABLED);
	bonobo_ui_component_add_listener(ui_component, ID_ALLOW_JAVA,
					 (BonoboUIListenerFn)allow_setting_bonoboui_changed_cb,
					 CONF_FILTERING_JAVA_ENABLED);

	eel_gconf_notification_add(CONF_FILTERING_ALLOW_POPUPS,
				   (GConfClientNotifyFunc)allow_popups_gconf_changed_cb,
				   window);
	eel_gconf_notification_add(CONF_FILTERING_ALLOW_POPUPS,
				   (GConfClientNotifyFunc)allow_setting_gconf_changed_cb,
				   ui_component);
	eel_gconf_notification_add(CONF_FILTERING_JAVASCRIPT_ENABLED,
				   (GConfClientNotifyFunc)allow_setting_gconf_changed_cb,
				   ui_component);
	eel_gconf_notification_add(CONF_FILTERING_JAVA_ENABLED,
				   (GConfClientNotifyFunc)allow_setting_gconf_changed_cb,
				   ui_component);
}

static void
view_images_bonoboui_changed_cb(BonoboUIComponent *component,
			        const char *id,
			        Bonobo_UIComponent_EventType type,
			        const char *state,
			        gpointer user_data)
{
	if (state == NULL || *state == '\0') return;

	/* we only care when an item being set */
	if (!strcmp(state, "0")) return;

	eel_gconf_set_integer(CONF_FILTERING_IMAGE_LOADING_TYPE, 
			      GPOINTER_TO_INT(user_data));
}

static void
view_images_gconf_changed_cb(GConfClient *client,
		             guint cnxn_id,
			     GConfEntry *entry,
			     BonoboUIComponent *ui_component)
{
	radio_menu_group_set_from_value(ui_component, IMAGES, entry->value);
}

static void
setup_images_menu(GaleonWindow *window)
{
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT(window->ui_component);

	radio_menu_group_set_from_value(ui_component, IMAGES,
					eel_gconf_get_value(CONF_FILTERING_IMAGE_LOADING_TYPE));

	bonobo_ui_component_add_listener(ui_component, "ViewImagesAlways",
					 view_images_bonoboui_changed_cb,
					 GINT_TO_POINTER(VIEW_IMAGES_ALWAYS));
	bonobo_ui_component_add_listener(ui_component, "ViewImagesCurrent",
					 view_images_bonoboui_changed_cb,
					 GINT_TO_POINTER(VIEW_IMAGES_CURRENT));
	bonobo_ui_component_add_listener(ui_component, "ViewImagesNever",
					 view_images_bonoboui_changed_cb,
					 GINT_TO_POINTER(VIEW_IMAGES_NEVER));

	eel_gconf_notification_add(CONF_FILTERING_IMAGE_LOADING_TYPE, 
				   (GConfClientNotifyFunc)view_images_gconf_changed_cb, 
				   ui_component);
}

static void
sidebar_close_requested_cb (GaleonSidebar *sidebar, 
			    GaleonWindow *window)
{
	BonoboUIComponent *ui_component = BONOBO_UI_COMPONENT (window->ui_component);

	view_sidebar_changed_cb (ui_component,
		VIEW_SIDEBAR_PATH,
		BONOBO_TYPE_UI_COMPONENT,
		NULL,
		window);
}

static void
sidebar_page_changed_cb (GaleonSidebar *sidebar, 
			 const char *page_id, 
			 GaleonWindow *window)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);

	if (strcmp (page_id, "history") == 0)
	{
		window->priv->history_sidebar = 
			history_dialog_new (embed, TRUE);
	
		galeon_sidebar_set_content 
			(sidebar, 
			 G_OBJECT(window->priv->history_sidebar));
	}
	else if (strcmp (page_id, "bookmarks") == 0)
	{
		GtkWidget *w;
		if (!window->priv->bookmarks_sidebar)
		{
			GbEditorDockable *e;
			GbBookmarkSet *set;
			
			set = galeon_shell_get_bookmark_set (galeon_shell);
			if (!set) return;
		
			e = gb_editor_dockable_new_for_set (set);
			
			gb_editor_set_location_source (GB_EDITOR (e), GB_LOCATION_SOURCE (window));

			g_signal_connect (e, "bookmark-activated", 
					  G_CALLBACK (galeon_window_bookmark_activated_cb), window);
			window->priv->bookmarks_sidebar = e;
		}
		w = gb_editor_get_main_widget (GB_EDITOR (window->priv->bookmarks_sidebar));
		gtk_widget_show (w);
		galeon_sidebar_set_content (sidebar, G_OBJECT (w));
	}
}

static GtkWidget *
setup_sidebar (GaleonWindow *window)
{
	GtkWidget *hpaned;
	GtkWidget *sidebar;

	hpaned = gtk_hpaned_new ();
	gtk_widget_show (hpaned);

	sidebar = galeon_sidebar_new ();

	g_signal_connect (G_OBJECT(sidebar),
			  "close_requested",
			  G_CALLBACK (sidebar_close_requested_cb),
			  window);
	
	galeon_sidebar_add_page (GALEON_SIDEBAR(sidebar), 
				 _("Bookmarks"), "bookmarks", FALSE);
	galeon_sidebar_add_page (GALEON_SIDEBAR(sidebar), 
				 _("History"), "history", FALSE);

	g_signal_connect (sidebar, "page_changed",
			  G_CALLBACK (sidebar_page_changed_cb),
			  window);

	gtk_paned_add1 (GTK_PANED(hpaned), 
			sidebar);
	gtk_paned_add2 (GTK_PANED(hpaned),
			GTK_WIDGET(window->priv->notebook));

	window->priv->sidebar = sidebar;
	
	return hpaned;
}

static void
galeon_window_init (GaleonWindow *window)
{
	BonoboUIComponent *ui_component;
	GbBookmarkSet *bs;
	Session *session;

	session = galeon_shell_get_session (galeon_shell);

        window->priv = g_new0 (GaleonWindowPrivate, 1);
	window->priv->embed_popup = NULL;
	window->priv->active_tab = NULL;
	window->priv->chrome_mask = 0;
	window->priv->ignore_layout_toggles = FALSE;
	window->priv->closing = FALSE;
	window->priv->has_default_size = FALSE;
	window->priv->recent_history = galeon_window_recent_history_new ();
	window->priv->recent_history_menu = galeon_window_recent_history_menu_new ();

	galeon_window_recent_history_menu_set_history (window->priv->recent_history_menu,
						       window->priv->recent_history);
	g_signal_connect (window->priv->recent_history_menu, "activated",
			  G_CALLBACK (galeon_window_recent_history_menu_activated_cb), window);

	/* Setup the window and connect verbs */
	setup_bonobo_window (window, &ui_component);
	window->ui_component = G_OBJECT (ui_component);
	bonobo_ui_component_add_verb_list_with_data (ui_component, 
                                		     galeon_verbs, 
						     window);
	setup_layout_menus (window);
	setup_images_menu (window);
	setup_web_menu(window);
	
	/* Setup the embed popups factory */
	window->priv->embed_popup = setup_popup_factory (window, 
							 ui_component);

	bonobo_ui_component_freeze (ui_component, NULL);
	
	/* Setup menubar */
	galeon_embed_utils_build_charsets_submenu (ui_component,
                                                   CHARSET_MENU_PATH,
                                                   (BonoboUIVerbFn)window_cmd_set_charset,
                                                   window);
	galeon_window_recent_history_menu_set_path (window->priv->recent_history_menu, 
						    ui_component, GO_RECENT_HISTORY_PATH);

	/* Setup toolbar and statusbar */
	window->priv->toolbar = toolbar_new (window);
	window->priv->statusbar = statusbar_new (window);
	window->priv->ppview_toolbar = ppview_toolbar_new (window);

	/* Setup bookmarks ui elements */
	bs = galeon_shell_get_bookmark_set (galeon_shell);
	window->priv->bookmarks_menu = setup_bookmarks_menu (window, bs); 
	window->priv->bookmarks_context_menu = setup_bookmarks_context_menu (window, bs); 
	window->priv->bookmarks_toolbar = setup_bookmarks_toolbar (window, bs); 	
	
	/* Setup window contents */
	window->priv->notebook = setup_notebook (window);
	window->priv->hpaned = setup_sidebar (window);
	bonobo_window_set_contents (BONOBO_WINDOW (window), 
				    GTK_WIDGET (window->priv->hpaned));
	
	bonobo_ui_component_thaw (ui_component, NULL);
	
	g_object_ref (galeon_shell);

	/*Once window is fully created, add it to the session list*/
	session_add_window (session, window);
}

static void  
save_window_chrome (GaleonWindow *window)
{
	EmbedChromeMask flags = window->priv->chrome_mask;
	
	if (flags & EMBED_CHROME_OPENASPOPUP)
	{
	}
	else if (flags & EMBED_CHROME_PPVIEWTOOLBARON)
	{
	}
	else if (flags & EMBED_CHROME_OPENASFULLSCREEN)
	{
		eel_gconf_set_boolean (CONF_WINDOWS_FS_SHOW_SIDEBAR,
				       flags & EMBED_CHROME_SIDEBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_FS_SHOW_TOOLBARS,
				       flags & EMBED_CHROME_TOOLBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_FS_SHOW_BOOKMARKS,
				       flags & EMBED_CHROME_PERSONALTOOLBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_FS_SHOW_STATUSBAR,
				       flags & EMBED_CHROME_STATUSBARON);
	}
	else
	{
		eel_gconf_set_boolean (CONF_WINDOWS_SHOW_SIDEBAR,
				       flags & EMBED_CHROME_SIDEBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_SHOW_TOOLBARS,
				       flags & EMBED_CHROME_TOOLBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_SHOW_BOOKMARKS,
				       flags & EMBED_CHROME_PERSONALTOOLBARON);
		eel_gconf_set_boolean (CONF_WINDOWS_SHOW_STATUSBAR,
				       flags & EMBED_CHROME_STATUSBARON);
	}
}

static void
remove_from_session (GaleonWindow *window)
{
	Session *session;

	session = galeon_shell_get_session (galeon_shell);
	g_return_if_fail (session != NULL);
	
	session_remove_window (session, window);
}

static void
galeon_window_finalize (GObject *object)
{
        GaleonWindow *window;

        g_return_if_fail (IS_GALEON_WINDOW (object));

	window = GALEON_WINDOW (object);

        g_return_if_fail (window->priv != NULL);

	remove_from_session (window);

	if (window->priv->embed_popup)
	{
		g_object_unref (G_OBJECT (window->priv->embed_popup));
	}

	g_object_unref (G_OBJECT (window->priv->bookmarks_menu));
	g_object_unref (G_OBJECT (window->priv->bookmarks_context_menu));
	g_object_unref (G_OBJECT (window->priv->bookmarks_toolbar));
	g_object_unref (G_OBJECT (window->priv->toolbar));
	g_object_unref (G_OBJECT (window->priv->statusbar));
	
	if (window->priv->find_dialog)
	{
		g_object_unref (G_OBJECT (window->priv->find_dialog));
	}

	if (window->priv->history_dialog)
	{
		g_object_remove_weak_pointer 
                        (G_OBJECT(window->priv->history_dialog),
                         (gpointer *)&window->priv->history_dialog);
	}

	if (window->priv->bookmarks_sidebar)
	{
		g_object_unref (window->priv->bookmarks_sidebar);
	}
	
	g_signal_handlers_disconnect_matched (window->priv->recent_history_menu, 
					      G_SIGNAL_MATCH_DATA, 0, 0, NULL, NULL, window);

	g_object_unref (window->priv->recent_history);
	g_object_unref (window->priv->recent_history_menu);

        g_free (window->priv);

        G_OBJECT_CLASS (parent_class)->finalize (object);

#ifdef DEBUG_MARCO
	g_print ("Galeon Window finalized %p\n", window);
#endif
	
	g_object_unref (galeon_shell);
}

GaleonWindow *
galeon_window_new (void)
{
	return GALEON_WINDOW (g_object_new (GALEON_WINDOW_TYPE, NULL));
}

static void
dock_item_set_visibility (GaleonWindow *window,
			  const char *path,
			  gboolean visibility)
{
	gul_bonobo_set_hidden (BONOBO_UI_COMPONENT(window->ui_component),
                               path,
                               !visibility);
}

EmbedChromeMask  
galeon_window_get_chrome (GaleonWindow *window)
{
	return window->priv->chrome_mask;
}

static void
wmspec_change_state (gboolean   add,
                     GdkWindow *window,
                     GdkAtom    state1,
                     GdkAtom    state2)
{
	XEvent xev;

	#define _NET_WM_STATE_REMOVE        0    /* remove/unset property */
	#define _NET_WM_STATE_ADD           1    /* add/set property */
	#define _NET_WM_STATE_TOGGLE        2    /* toggle property  */  
  
	xev.xclient.type = ClientMessage;
  	xev.xclient.serial = 0;
  	xev.xclient.send_event = True;
  	xev.xclient.display = gdk_display;
  	xev.xclient.window = GDK_WINDOW_XID (window);
  	xev.xclient.message_type = gdk_x11_get_xatom_by_name ("_NET_WM_STATE");
  	xev.xclient.format = 32;
  	xev.xclient.data.l[0] = add ? _NET_WM_STATE_ADD : _NET_WM_STATE_REMOVE;
  	xev.xclient.data.l[1] = gdk_x11_atom_to_xatom (state1);
  	xev.xclient.data.l[2] = gdk_x11_atom_to_xatom (state2);
  
  	XSendEvent (gdk_display, GDK_WINDOW_XID (gdk_get_default_root_window ()),
                    False,
                    SubstructureRedirectMask | SubstructureNotifyMask,
                    &xev);
}

static void
window_set_fullscreen_mode (GaleonWindow *window, gboolean active)
{
        GdkWindow *gdk_window;

        gdk_window = GTK_WIDGET(window)->window;

        if (gdk_net_wm_supports (gdk_atom_intern ("_NET_WM_STATE_FULLSCREEN",
                                                  FALSE)))
        {
                wmspec_change_state (active,
                                     gdk_window,
                                     gdk_atom_intern ("_NET_WM_STATE_FULLSCREEN",
                                                      FALSE),
                                     GDK_NONE);
        }
        else
        {
                g_warning ("NET_WM_STATE_FULLSCREEN not supported");
	}
}

static void
translate_default_chrome (EmbedChromeMask *chrome_mask)
{	
	/* keep only not layout flags */
	*chrome_mask &= (EMBED_CHROME_WINDOWRAISED |
        		 EMBED_CHROME_WINDOWLOWERED |
        		 EMBED_CHROME_CENTERSCREEN |
        		 EMBED_CHROME_OPENASDIALOG |
        		 EMBED_CHROME_OPENASCHROME |
        		 EMBED_CHROME_OPENASPOPUP |
			 EMBED_CHROME_OPENASFULLSCREEN);

	/* Load defaults */
	if (*chrome_mask & EMBED_CHROME_OPENASFULLSCREEN)
	{
		if (eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_SIDEBAR))
		{
			*chrome_mask |= EMBED_CHROME_SIDEBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_STATUSBAR))
		{
			*chrome_mask |= EMBED_CHROME_STATUSBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_TOOLBARS))
		{
			*chrome_mask |= EMBED_CHROME_TOOLBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_FS_SHOW_BOOKMARKS))
		{
			*chrome_mask |= EMBED_CHROME_PERSONALTOOLBARON;
		}
	}
	else
	{
		if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_SIDEBAR))
		{
			*chrome_mask |= EMBED_CHROME_SIDEBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_STATUSBAR))
		{
			*chrome_mask |= EMBED_CHROME_STATUSBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_TOOLBARS))
		{
			*chrome_mask |= EMBED_CHROME_TOOLBARON;
		}
		if (eel_gconf_get_boolean (CONF_WINDOWS_SHOW_BOOKMARKS))
		{
			*chrome_mask |= EMBED_CHROME_PERSONALTOOLBARON;
		}
		
		*chrome_mask |= EMBED_CHROME_MENUBARON;
	}
}

void  
galeon_window_set_chrome (GaleonWindow *window, 
			  EmbedChromeMask flags)
{
	gboolean toolbar, ppvtoolbar, statusbar;
	
	if (flags & EMBED_CHROME_DEFAULT)
	{
		translate_default_chrome (&flags);
	}
		
	dock_item_set_visibility (window, "/menu", 
				  flags & EMBED_CHROME_MENUBARON);	

	if ((flags & EMBED_CHROME_SIDEBARON) != FALSE)
	{
		gtk_widget_show (window->priv->sidebar);
	}
	else
	{
		gtk_widget_hide (window->priv->sidebar);
	}
	
	toolbar = (flags & EMBED_CHROME_TOOLBARON) != FALSE;
	toolbar_set_visibility (window->priv->toolbar,
				toolbar);

	statusbar = (flags & EMBED_CHROME_STATUSBARON) != FALSE;
	statusbar_set_visibility (window->priv->statusbar,
				  statusbar);

	ppvtoolbar = (flags & EMBED_CHROME_PPVIEWTOOLBARON) != FALSE;
	ppview_toolbar_set_old_chrome (window->priv->ppview_toolbar,
				       window->priv->chrome_mask);
	ppview_toolbar_set_visibility (window->priv->ppview_toolbar, 
				       ppvtoolbar);
	
	/* set fullscreen only when it's really changed */
	if ((window->priv->chrome_mask & EMBED_CHROME_OPENASFULLSCREEN) != 
	    (flags & EMBED_CHROME_OPENASFULLSCREEN))
	{
		save_window_chrome (window);
		window_set_fullscreen_mode (window,
					    flags & EMBED_CHROME_OPENASFULLSCREEN);
	}

	gb_bonobo_ui_toolbar_set_visible (window->priv->bookmarks_toolbar,
					  flags & EMBED_CHROME_PERSONALTOOLBARON);
	
	window->priv->chrome_mask = flags;

	update_layout_toggles (window);

	save_window_chrome (window);
}

void
galeon_window_toggle_fullscreen (GaleonWindow *window)
{
	EmbedChromeMask mask;
	
	mask = galeon_window_get_chrome (window);
	mask ^= EMBED_CHROME_OPENASFULLSCREEN;
	mask |= EMBED_CHROME_DEFAULT;
	galeon_window_set_chrome (window, mask);
}

void 
galeon_window_add_tab (GaleonWindow *window, 
		       GaleonTab *tab,
		       gboolean jump_to)
{
	GtkWidget *widget;
	
	g_return_if_fail (IS_GALEON_WINDOW (window));
	g_return_if_fail (IS_GALEON_TAB (tab));

	galeon_tab_set_window (tab, window);
	
	widget = GTK_WIDGET(galeon_tab_get_embed (tab));

	gul_notebook_insert_page (GUL_NOTEBOOK (window->priv->notebook), 
				  widget,
				  GUL_NOTEBOOK_INSERT_GROUPED,
				  jump_to);
}

void
galeon_window_next_tab (GaleonWindow *window)
{
	gtk_notebook_next_page (GTK_NOTEBOOK(window->priv->notebook));
}

void
galeon_window_prev_tab (GaleonWindow *window)
{
	gtk_notebook_prev_page (GTK_NOTEBOOK(window->priv->notebook));
}

void 
galeon_window_jump_to_tab (GaleonWindow *window, 
			   GaleonTab *tab)
{
	GtkWidget *widget;	
	int page;
	
	widget = GTK_WIDGET(galeon_tab_get_embed (tab));

	page = gtk_notebook_page_num 
		(window->priv->notebook, widget);
	gtk_notebook_set_current_page 
		(window->priv->notebook, page);
}

static GaleonTab *
get_tab_from_page_num (GtkNotebook *notebook, gint page_num) 
{
	GaleonTab *tab;
	GtkWidget *embed_widget;

	if (page_num < 0) return NULL;
	
	embed_widget = gtk_notebook_get_nth_page (notebook, page_num);
	
	g_return_val_if_fail (GTK_IS_WIDGET (embed_widget), NULL);
	tab = g_object_get_data (G_OBJECT (embed_widget), "GaleonTab");
	g_return_val_if_fail (IS_GALEON_TAB (G_OBJECT (tab)), NULL);
	
	return tab;
}

static GaleonTab *
real_get_active_tab (GaleonWindow *window, int page_num)
{
	if (page_num == -1)
	{
		page_num = gtk_notebook_get_current_page (window->priv->notebook);
	}

	return get_tab_from_page_num (window->priv->notebook, page_num);
}

void 
galeon_window_remove_tab (GaleonWindow *window, 
		          GaleonTab *tab)
{
	GtkWidget *embed;
	
	g_return_if_fail (IS_GALEON_WINDOW (window));
	g_return_if_fail (IS_GALEON_TAB (tab));

	window->priv->active_tab = NULL;
	
	embed  = GTK_WIDGET (galeon_tab_get_embed (tab));
	
	gul_notebook_remove_page (GUL_NOTEBOOK (window->priv->notebook), 
				  embed);
}

void 
galeon_window_move_tab (GaleonWindow *window, 
			GaleonTab *tab,
			GaleonTab *new_sibling)
{
	int move_to;
	GtkNotebook *nb = window->priv->notebook;
	GtkWidget *w;
	GtkWidget *embed;
	
	embed = GTK_WIDGET(galeon_tab_get_embed (tab)); 
	w = GTK_WIDGET (galeon_tab_get_embed (new_sibling));
	move_to = gtk_notebook_page_num (nb, w);

	if (embed && move_to != -1)
	{
		gtk_notebook_reorder_child (nb, embed, move_to);
	}
}

void 
galeon_window_load_url (GaleonWindow *window,
			const char *url)
{
	GaleonEmbed *embed;
	GbBookmarkSet *set;
	gchar *real_url;

        g_return_if_fail (IS_GALEON_WINDOW(window));
	embed = galeon_window_get_active_embed (window);
        g_return_if_fail (embed != NULL);
        g_return_if_fail (url != NULL);
	
	set = galeon_shell_get_bookmark_set (galeon_shell);
	if (set)
	{
		real_url = gb_bookmark_set_get_url_by_nick_and_args (set, url);
		if (!real_url)
		{
			/* no matching nick found */
			real_url = g_strdup (url);
		}
	}
	else
	{
		real_url = g_strdup (url);
	}

        galeon_embed_load_url (embed, real_url);
	g_free (real_url);
}

void galeon_window_edit_location (GaleonWindow *window)
{
	toolbar_edit_location (window->priv->toolbar);
}

void
galeon_window_show (GtkWidget *widget)
{
	GaleonWindow *window = GALEON_WINDOW(widget);
	GaleonTab *tab;
	int w = -1, h = -1;
	
	if (!window->priv->chrome_mask)
	{
		galeon_window_set_chrome (window, EMBED_CHROME_DEFAULT);
	}

	tab = galeon_window_get_active_tab (window);
	if (tab) galeon_tab_get_size (tab, &w, &h);
	
	if (!window->priv->has_default_size && w == -1 && h == -1) 
	{
		gul_state_load_window (GTK_WIDGET(window), 
				       "main_window", 
			               600, 500, TRUE);
		window->priv->has_default_size = TRUE;
	}
	
	GTK_WIDGET_CLASS (parent_class)->show (widget);
}

static void
update_status_message (GaleonWindow *window)
{
	const char *message;
	GaleonTab *tab;

	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);
	
	message = galeon_tab_get_status_message (tab);
	g_return_if_fail (message != NULL);
	
	statusbar_set_message (STATUSBAR(window->priv->statusbar),
			       message);
}

static void
update_progress (GaleonWindow *window)
{
	GaleonTab *tab;
	int load_percent;
	
	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);
	
	load_percent = galeon_tab_get_load_percent (tab);

	statusbar_set_progress (STATUSBAR(window->priv->statusbar),
			        load_percent);
}

static void
update_security (GaleonWindow *window)
{
	GaleonEmbed *embed;
	EmbedSecurityLevel level;
	char *description;
	char *state = NULL;
	gboolean secure;
	char *tooltip;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	if (galeon_embed_get_security_level (embed, &level, &description) != G_OK)
	{
		level = STATE_IS_UNKNOWN;
		description = NULL;
	}

	secure = FALSE;
	switch (level)
	{
	case STATE_IS_UNKNOWN:
		state = _("Unknown");
		break;
	case STATE_IS_INSECURE:
		state = _("Insecure");
		break;
	case STATE_IS_BROKEN:
		state = _("Broken");
		break;
	case STATE_IS_SECURE_MED:
		state = _("Medium");
		secure = TRUE;
		break;
	case STATE_IS_SECURE_LOW:
		state = _("Low");
		secure = TRUE;
		break;
	case STATE_IS_SECURE_HIGH:
		state = _("High");
		secure = TRUE;
		break;
	default:
		g_assert_not_reached ();
		break;
	}

	if (description != NULL)
	{
		tooltip = g_strdup_printf (_("Security level: %s\n%s"),
					   state, description);
		g_free (description);
	}
	else
	{
		tooltip = g_strdup_printf (_("Security level: %s"), state);

	}

	statusbar_set_security_state (STATUSBAR (window->priv->statusbar),
			              secure, tooltip);
	g_free (tooltip);
}

static void
update_popup_block(GaleonWindow *window)
{
	GaleonTab *tab;
	const char *uri = NULL;

	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);

	uri = galeon_tab_get_blocked_popup_uri(tab);

	statusbar_set_popup_block(STATUSBAR(window->priv->statusbar),
				  uri ? TRUE : FALSE, uri);
}

static void
update_nav_control (GaleonWindow *window)
{
	/* the zoom control is updated at the same time than the navigation
	   controls. This keeps it synched most of the time, but not always,
	   because we don't get a notification when zoom changes */

	gresult back, forward, up, stop;
	GaleonEmbed *embed;
	GaleonTab *tab;
	gint zoom;

	g_return_if_fail (window != NULL);
	
	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);
	
	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);
	
	back = galeon_embed_can_go_back (embed);
	forward = galeon_embed_can_go_forward (embed);
	up = galeon_embed_can_go_up (embed);
	stop = galeon_tab_get_load_status (tab) & TAB_LOAD_STARTED;

	toolbar_button_set_sensitive (window->priv->toolbar, 
				      TOOLBAR_BACK_BUTTON,
				      back == G_OK ? TRUE : FALSE);
	toolbar_button_set_sensitive (window->priv->toolbar, 
				      TOOLBAR_FORWARD_BUTTON,
				      forward == G_OK ? TRUE : FALSE);
	toolbar_button_set_sensitive (window->priv->toolbar, 
				      TOOLBAR_UP_BUTTON,
				      up == G_OK ? TRUE : FALSE);
	toolbar_button_set_sensitive (window->priv->toolbar, 
				      TOOLBAR_STOP_BUTTON,
				      stop);
	if (galeon_embed_zoom_get (embed, &zoom) == G_OK)
	{
		toolbar_set_zoom (window->priv->toolbar, zoom);
	}

	gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
			          GO_BACK_CMD_PATH, !back);
	gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
			          GO_FORWARD_CMD_PATH, !forward);
	gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
			          GO_UP_CMD_PATH, !up);
	gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
			          GO_STOP_CMD_PATH, stop);
}

static void
update_title_control (GaleonWindow *window)
{
	GaleonTab *tab;
	const char *title;
	
	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);

	title = galeon_tab_get_title (tab);

	if (title)
	{
		gtk_window_set_title (GTK_WINDOW(window), 
				      title);
	}
}

static void
update_location_control (GaleonWindow *window)
{
	GaleonTab *tab;
	const char *location, *host = NULL;
	char *allow_images_str, *block_images_str,
	     *allow_cookies_str, *block_cookies_str,
	     *allow_popups_str;
	GnomeVFSURI *uri = NULL;

	tab = galeon_window_get_active_tab (window);
	g_return_if_fail (tab != NULL);

	location = galeon_tab_get_location (tab);

	/* don't display about:blank in the location entry, it's uninteresting
	 * and prevents the user from pasting a new url after opening a new
	 * window/tab */
	if (!location || !strcmp (location, "about:blank")) location = "";

	toolbar_set_location (window->priv->toolbar,
			      location);

	/* Update cookie and image allow/block menu items to refer to
	 * the current host. Makes things clearer for the user
	 */
	if (location)
		uri = gnome_vfs_uri_new(location);

	if (uri)
		host = gnome_vfs_uri_get_host_name(uri);


	if (!host)
		host = _("This Site");

	allow_cookies_str = g_strdup_printf(_("_Allow Cookies From %s"), host);
	block_cookies_str = g_strdup_printf(_("_Block Cookies From %s"), host);
	allow_images_str = g_strdup_printf(_("_Allow Images From %s"), host);
	block_images_str = g_strdup_printf(_("_Block Images From %s"), host);
	allow_popups_str = g_strdup_printf(_("_Allow Popups From %s"), host);

	if (uri)
		gnome_vfs_uri_unref(uri);

	gul_bonobo_set_label(BONOBO_UI_COMPONENT(window->ui_component),
			     ALLOW_COOKIES_MENU_PATH, allow_cookies_str);
	gul_bonobo_set_label(BONOBO_UI_COMPONENT(window->ui_component),
			     BLOCK_COOKIES_MENU_PATH, block_cookies_str);
	gul_bonobo_set_label(BONOBO_UI_COMPONENT(window->ui_component),
			     ALLOW_IMAGES_MENU_PATH, allow_images_str);
	gul_bonobo_set_label(BONOBO_UI_COMPONENT(window->ui_component),
			     BLOCK_IMAGES_MENU_PATH, block_images_str);
	gul_bonobo_set_label(BONOBO_UI_COMPONENT(window->ui_component),
			     ALLOW_POPUPS_MENU_PATH, allow_popups_str);

	g_free(allow_cookies_str);
	g_free(block_cookies_str);
	g_free(allow_images_str);
	g_free(block_images_str);
	g_free(allow_popups_str);
}

static void
update_favicon_control (GaleonWindow *window)
{
	toolbar_update_favicon (window->priv->toolbar);
}

static void
update_find_control (GaleonWindow *window)
{
/* We're not updating the item sensitivity right now as it only follows the
 * find dialog status. If you try to type-ahead-find with no valid search
 * string in the find dialog, it won't let you Ctrl+G for the type-ahead-find
 * even if there is a valid type-ahead-find result.
 */
#if MADE_TYPE_AHEAD_FIND_AWARE
	gboolean can_go_next, can_go_prev;

	if (window->priv->find_dialog)
	{
		can_go_next = find_dialog_can_go_next 
			(FIND_DIALOG(window->priv->find_dialog));
		can_go_prev = find_dialog_can_go_prev 
			(FIND_DIALOG(window->priv->find_dialog));
		
		gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
					  EDIT_FIND_NEXT_CMD_PATH, can_go_next);
		gul_bonobo_set_sensitive (BONOBO_UI_COMPONENT(window->ui_component),
					  EDIT_FIND_PREV_CMD_PATH, can_go_prev);
	}
#endif
}

static void
update_window_visibility (GaleonWindow *window)
{
	GList *tabs, *l;
	
	tabs = galeon_window_get_tabs (window);
	for (l = tabs; l != NULL; l = l->next)
	{
		GaleonTab *tab = GALEON_TAB(l->data);
		g_return_if_fail (IS_GALEON_TAB(tab));
		
		if (galeon_tab_get_visibility (tab))
		{
			gtk_widget_show (GTK_WIDGET(window));
			g_list_free (tabs);
			return;
		}
	}
	g_list_free (tabs);

	if (GTK_WIDGET_VISIBLE (GTK_WIDGET (window)))
	{
		gtk_widget_hide (GTK_WIDGET (window));
	}
}

static void
update_spinner_control (GaleonWindow *window)
{
	GList *tabs, *l;

	tabs = galeon_window_get_tabs (window);
	for (l = tabs; l != NULL; l = l->next)
	{
		GaleonTab *tab = GALEON_TAB(l->data);
		g_return_if_fail (IS_GALEON_TAB(tab));
		
		if (galeon_tab_get_load_status (tab) & TAB_LOAD_STARTED)
		{
			toolbar_spinner_start (window->priv->toolbar);
			g_list_free (tabs);
			return;
		}
	}
	g_list_free (tabs);
	
	toolbar_spinner_stop (window->priv->toolbar);
}

static void
update_permission_control (GaleonWindow *window)
{
	GaleonEmbedShell *embed_shell;
	gresult result;
	PermissionActionType image_action, cookie_action, popup_action;
	GaleonTab *tab = galeon_window_get_active_tab(window);
	if (tab == NULL) return;

	embed_shell = galeon_shell_get_embed_shell(galeon_shell);
	g_return_if_fail(embed_shell != NULL);

	result = galeon_embed_shell_test_permission(embed_shell,
						    galeon_tab_get_location(tab),
						    IMAGES_PERMISSION,
						    &image_action);
	if (result == G_OK)
	{
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 ALLOW_IMAGES_CMD_PATH,
					 (image_action != ALLOW_ACTION));
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 BLOCK_IMAGES_CMD_PATH,
					 (image_action != DENY_ACTION));
	}

	result = galeon_embed_shell_test_permission(embed_shell,
						    galeon_tab_get_location(tab),
						    COOKIES_PERMISSION,
						    &cookie_action);
	if (result == G_OK)
	{
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 ALLOW_COOKIES_CMD_PATH,
					 (cookie_action != ALLOW_ACTION));
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 BLOCK_COOKIES_CMD_PATH,
					 (cookie_action != DENY_ACTION));
	}

	//If global allow popups, then the site specific menu item
	//is always insensitive
	if (eel_gconf_get_boolean(CONF_FILTERING_ALLOW_POPUPS))
	{
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 ALLOW_POPUPS_CMD_PATH,
					 FALSE);
		galeon_tab_clear_blocked_popup_uri(tab);
		update_popup_block(window);
		return;
	}
	
	result = galeon_embed_shell_test_permission(embed_shell,
						    galeon_tab_get_location(tab),
						    POPUPS_PERMISSION,
						    &popup_action);
	if (result == G_OK)
	{
		gul_bonobo_set_sensitive(BONOBO_UI_COMPONENT(window->ui_component),
					 ALLOW_POPUPS_CMD_PATH,
					 (popup_action != ALLOW_ACTION));
		if (popup_action == ALLOW_ACTION)
		{
			galeon_tab_clear_blocked_popup_uri(tab);
			update_popup_block(window);
		}
	}
}

void  
galeon_window_update_control (GaleonWindow *window,
			      ControlID control)
{
	g_return_if_fail (IS_GALEON_WINDOW (window));
	
	switch (control)
	{
	case StatusbarMessageControl:
		update_status_message (window);
		break;
	case StatusbarProgressControl:
		update_progress (window);
		break;
	case StatusbarSecurityControl:
		update_security (window);
		break;
	case StatusbarPopupBlockControl:
		update_popup_block(window);
		break;
	case FindControl:
		update_find_control (window);
		break;
	case ZoomControl:
		 /* the zoom control is updated at the same time than the navigation
		    controls. This keeps it synched most of the time, but not always,
		    because we don't get a notification when zoom changes */
	case NavControl:
		update_nav_control (window);
		break;
	case TitleControl:
		update_title_control (window);
		break;
	case WindowVisibilityControl:
		update_window_visibility (window);
		break;
	case SpinnerControl:
		update_spinner_control (window);
		break;
	case LocationControl:
		update_location_control (window);
		break;
	case FaviconControl:
		update_favicon_control (window);
		break;
	case PermissionControl:
		update_permission_control(window);
		break;
	default:
		g_warning ("unknown control specified for updating");
		break;
	}
}

void
galeon_window_update_all_controls (GaleonWindow *window)
{
	g_return_if_fail (IS_GALEON_WINDOW (window));
	
	if (galeon_window_get_active_tab (window) != NULL)
	{
		update_nav_control (window);
		update_title_control (window);
		update_location_control (window);
		update_favicon_control (window);
		update_status_message (window);
		update_progress (window);
		update_security (window);
		update_popup_block(window);
		update_find_control (window);
		update_spinner_control (window);
		update_permission_control(window);
	}
}


GaleonSidebar *
galeon_window_get_sidebar (GaleonWindow *window)
{
	g_return_val_if_fail (IS_GALEON_WINDOW (window), NULL);

	if(!IS_GALEON_SIDEBAR(G_OBJECT(window->priv->sidebar)))
		setup_sidebar(window);

	return GALEON_SIDEBAR(window->priv->sidebar);
}	

GaleonTab *
galeon_window_get_active_tab (GaleonWindow *window)
{
	g_return_val_if_fail (IS_GALEON_WINDOW (window), NULL);

	return window->priv->active_tab;
}

GaleonEmbed *
galeon_window_get_active_embed (GaleonWindow *window)
{
	GaleonTab *tab;

	tab = galeon_window_get_active_tab (window);

	if (tab)
	{
		g_return_val_if_fail (IS_GALEON_WINDOW (G_OBJECT (window)),
				      NULL);
		return galeon_tab_get_embed (tab);
	}
	else return NULL;
}

GaleonEmbedPopupBW *
galeon_window_get_popup_factory (GaleonWindow *window)
{
	if (!window->priv->embed_popup)
	{
		window->priv->embed_popup = galeon_embed_popup_bw_new 
			(BONOBO_WINDOW(window));
		galeon_embed_popup_connect_verbs
			(GALEON_EMBED_POPUP (window->priv->embed_popup),
			 BONOBO_UI_COMPONENT (window->ui_component));
	}
	
	return window->priv->embed_popup;
}

GList *
galeon_window_get_tabs (GaleonWindow *window)
{
	GList *tabs = NULL;
	GtkWidget *w;
	int i = 0;
	
	while ((w = gtk_notebook_get_nth_page (window->priv->notebook, i)) != NULL)
	{
		GaleonTab *tab;
	
		tab = g_object_get_data (G_OBJECT (w), "GaleonTab");
		g_return_val_if_fail (IS_GALEON_TAB (G_OBJECT (tab)), NULL);
		
		tabs = g_list_append (tabs, tab);
		i++;
	}

	return tabs;
}

static void
save_old_embed_status (GaleonTab *tab, GaleonWindow *window)
{
	/* save old tab location status */
	galeon_tab_set_location (tab, 
				 toolbar_get_location (window->priv->toolbar));
}

static void
update_embed_dialogs (GaleonWindow *window,
		      GaleonTab *tab)
{
	GaleonEmbed *embed;
	GaleonDialog *find_dialog = window->priv->find_dialog;
	GaleonDialog *history_dialog = window->priv->history_dialog;
	GaleonDialog *history_sidebar = window->priv->history_sidebar;

	embed = galeon_tab_get_embed (tab);

	if (find_dialog)
	{
		galeon_embed_dialog_set_embed 
			(GALEON_EMBED_DIALOG(find_dialog),
			 embed);
	}

	if (history_dialog)
	{
		galeon_embed_dialog_set_embed 
			(GALEON_EMBED_DIALOG(history_dialog),
			 embed);
	}

	if (history_sidebar)
	{
		galeon_embed_dialog_set_embed 
			(GALEON_EMBED_DIALOG(history_sidebar),
			 embed);
	}
}

static void
galeon_window_notebook_switch_page_cb (GtkNotebook *notebook,
				       GtkNotebookPage *page,
				       guint page_num,
				       GaleonWindow *window)
{
	GaleonTab *tab, *old_tab;

	g_return_if_fail (IS_GALEON_WINDOW (window));
	if (window->priv->closing) return;

	/* get the new tab */
	tab = real_get_active_tab (window, page_num);
	
	/* update old tab */
	old_tab = window->priv->active_tab;
	if (old_tab && tab != old_tab)
	{
		g_return_if_fail (IS_GALEON_TAB (G_OBJECT (old_tab)));
		galeon_tab_set_is_active (old_tab, FALSE);
		save_old_embed_status (old_tab, window);
	}

	/* update new tab */
	window->priv->active_tab = tab;
	galeon_tab_set_is_active (tab, TRUE);

	update_embed_dialogs (window, tab);
	
	/* update window controls */
	galeon_window_update_all_controls (window);
}

/**
 * Prompt the user for a smart bookmark arguments, and return the
 * completed URL to load, or NULL if cancelled.
 */
static char *
handle_smartsite (GaleonWindow *w, GbSmartSite *s)
{
	GtkWidget *dialog, *vbox, *label, *entry;
	int response;
	char *args[2]; /* TODO: match this with get_num_fields() */
	char *url;
	
	dialog = gtk_dialog_new_with_buttons (_("Galeon"), GTK_WINDOW(w),
					      GTK_DIALOG_MODAL,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
					      GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					      NULL);
	gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

	vbox = gtk_vbox_new (FALSE, 8);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 8);

	label = gtk_label_new (_("Enter the arguments for this smart site:"));
	gtk_box_pack_start_defaults (GTK_BOX (vbox), label);

	entry = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);
	gtk_box_pack_start_defaults (GTK_BOX (vbox), entry);

	gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
	gtk_widget_show_all (dialog);

	response = gtk_dialog_run (GTK_DIALOG (dialog));
	if (response != GTK_RESPONSE_ACCEPT) {
		gtk_widget_destroy (dialog);
		return NULL;
	}

	args[0] = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	args[1] = NULL;
	gtk_widget_destroy (dialog);
	url = gb_smart_site_subst_args(s, args);
	g_free(args[0]);
	return url;
}

/**
 * Handle the activation of a bookmark. This function is not static because it's 
 * used in galeon-window-bookmark-activated-proxy.c (and only there: it's not public)
 */
void
galeon_window_bookmark_activate (GaleonWindow *w, GbBookmarkEventActivated *ev)
{
	GaleonTab *tab = galeon_window_get_active_tab (w);
	GbBookmark *b = ev->bookmark;
	char *url = g_strdup (ev->url);
	GbBookmarkActivatedFlags f = ev->flags;
	GaleonNewTabFlags flags = 0;

	if (GB_IS_SMART_SITE(b) && GB_SMART_SITE(b)->smarturl != NULL &&
	    (GB_SITE(b)->url == NULL || GB_SITE(b)->url[0]=='\0')) {
		url = handle_smartsite (w, GB_SMART_SITE (b));
		if (url == NULL)
		{
			return;
		}
	}

	if (GB_IS_SITE (b) && f == GB_BAF_DEFAULT)
	{
		galeon_window_load_url (w, url);
		g_free (url);
		return;
	}

	if (GB_IS_FOLDER (b) && f == GB_BAF_DEFAULT)
	{
		f = GB_BAF_NEW_TAB_OR_WINDOW;
	}

	if (f == GB_BAF_NEW_WINDOW)
	{
		flags = GALEON_NEW_TAB_IN_NEW_WINDOW;
	}
	else if (f == GB_BAF_NEW_TAB)
	{
		flags = GALEON_NEW_TAB_IN_EXISTING_WINDOW;
	}
	else if (f == GB_BAF_NEW_TAB_OR_WINDOW)
	{
		GdkModifierType mod;
		gdk_event_get_state (ev->event, &mod);
		flags = galeon_shell_modifier_flags (mod);
	}
	else
	{
		g_assert_not_reached ();
	}

	if (GB_IS_SITE (b))
	{
		galeon_shell_new_tab (galeon_shell, w, tab, url, flags);
	}
	else if (GB_IS_FOLDER (b))
	{
		GbBookmark *bi;
		int count = 0;
		
		/* count the number of activated bookmarks */
		for (bi = GB_FOLDER (b)->child; bi; bi = bi->next)
		{
			if (GB_IS_SITE (bi))
			{
				count++;
			}
		}
		
#define MANY_BOOKMARKS_WARNING_THRESHOLD 5
		if (count > MANY_BOOKMARKS_WARNING_THRESHOLD)
		{
			gint response;
			GtkWidget *dialog = gtk_message_dialog_new 
				(GTK_WINDOW (w),
				 GTK_DIALOG_MODAL,
				 GTK_MESSAGE_QUESTION,
				 GTK_BUTTONS_NONE,
				 _("You have requested to show %d bookmarked sites in different "
				   "windows or tabs. Are you sure?"),
				 count);

			gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
			gtk_dialog_add_button (GTK_DIALOG (dialog), 
					       _("Open the bookmarks"), 1);
			gtk_dialog_set_default_response (GTK_DIALOG (dialog), 1);
			
			response = gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);

			if (response != 1)
			{
				return;
			}
		}
		
		if (flags & GALEON_NEW_TAB_IN_NEW_WINDOW)
		{
			for (bi = GB_FOLDER (b)->child; bi; bi = bi->next)
			{
				if (GB_IS_SITE (bi))
				{
					galeon_shell_new_tab (galeon_shell, w, tab, 
							      GB_SITE (bi)->url, flags);
				}
			}
		}
		else
		{
			GaleonTab *jump_to_tab;
			gboolean jump_to;
			
			jump_to = flags & GALEON_NEW_TAB_JUMP;
			if (flags & GALEON_NEW_TAB_DONT_JUMP_TO) jump_to = FALSE;
			
			/* Open all bookmarks in order in the background.  Jumping
			 * to them automatically is a major performance hit with
			 * >3 bookmarks in a folder.  Once all bookmarks are opened,
			 * we can efficiently jump to the first one if 'jump to
			 * new tabs automatically' is enabled.
			 */
			flags |= GALEON_NEW_TAB_DONT_JUMP_TO;
			
			jump_to_tab = NULL;
			for (bi = GB_FOLDER (b)->child; bi; bi = bi->next)
			{
				if (GB_IS_SITE (bi))
				{
					tab = galeon_shell_new_tab (galeon_shell, w, tab,
								    GB_SITE (bi)->url, 
								    flags);
					
					if (jump_to_tab == NULL) jump_to_tab = tab;
				}
			}

			if (jump_to && jump_to_tab != NULL)
			{
				galeon_window_jump_to_tab (w, jump_to_tab);
			}
		}
	}
	else
	{
		g_warning ("don't know what to do...");
	}
}

static void
galeon_window_bookmark_activated_cb (GObject *sender,
				     GbBookmarkEventActivated *ev,
				     GaleonWindow *w)
{
	galeon_window_bookmark_activate (w, ev);
}

static void
find_dialog_search_cb (FindDialog *dialog, GaleonWindow *window)
{
	galeon_window_update_control (window, FindControl);
}

GaleonDialog *
galeon_window_get_find_dialog (GaleonWindow *window)
{			       
	GaleonDialog *dialog;
	GaleonEmbed *embed;

	if (window->priv->find_dialog)
	{
		return window->priv->find_dialog;
	}
	
	embed = galeon_window_get_active_embed (window);
	g_return_val_if_fail (GTK_IS_WINDOW(window), NULL);
	dialog = find_dialog_new_with_parent (GTK_WIDGET(window),
					      embed);

	g_signal_connect (dialog, "search", 
			  G_CALLBACK (find_dialog_search_cb),
			  window);
	
	window->priv->find_dialog = dialog;
	
	return dialog;
}

void
galeon_window_show_history (GaleonWindow *window)
{
	GaleonEmbed *embed;

	embed = galeon_window_get_active_embed (window);
	g_return_if_fail (embed != NULL);

	window->priv->history_dialog = history_dialog_new_with_parent 
						(GTK_WIDGET(window),
			 		     	 embed,
						 FALSE);
	g_object_add_weak_pointer 
                        (G_OBJECT(window->priv->history_dialog),
                         (gpointer *)&window->priv->history_dialog);

	galeon_dialog_show (window->priv->history_dialog);
}

static char *
galeon_window_gb_location_source_get_location (GbLocationSource *src)
{
	GaleonWindow *window;
	GaleonTab *tab;
	const gchar *location;

	window = GALEON_WINDOW (src);
	g_return_val_if_fail (window, NULL);

	tab = galeon_window_get_active_tab (window);
	g_return_val_if_fail (tab, NULL);

	location = galeon_tab_get_location (tab);
	if (!location) location = "";

	return g_strdup (location);
}

static char *
galeon_window_gb_location_source_get_title (GbLocationSource *src)
{
	GaleonWindow *window;
	GaleonTab *tab;
	const gchar *title;

	window = GALEON_WINDOW (src);
	g_return_val_if_fail (window, NULL);

	tab = galeon_window_get_active_tab (window);
	g_return_val_if_fail (tab, NULL);

	title = galeon_tab_get_title (tab);
	if (!title) 
	{
		return galeon_window_gb_location_source_get_location (src);
	}

	return g_strdup (title);
}

void
galeon_window_set_zoom (GaleonWindow *window, 
			gint zoom)
{
	GaleonEmbed *embed;

        g_return_if_fail (IS_GALEON_WINDOW (window));
        
	embed = galeon_window_get_active_embed (window);
        g_return_if_fail (embed != NULL);
	
        galeon_embed_zoom_set (embed, zoom, TRUE);
}

Toolbar *
galeon_window_get_toolbar (GaleonWindow *window)
{
	return window->priv->toolbar;
}

/**
 * Adds / updates an item to the window recent history menu 
 */
void
galeon_window_site_visited (GaleonWindow *window, const gchar *url, const gchar *title)
{
	g_return_if_fail (IS_GALEON_WINDOW (window));
	galeon_window_recent_history_visited (window->priv->recent_history, url, title);
}

static void
galeon_window_recent_history_menu_activated_cb (GaleonWindowRecentHistoryMenu *wrhm, 
						const char *url, const char *title, 
						GaleonWindow *w)
{
	galeon_window_load_url (w, url);
}

void
galeon_window_tab_detached_cb (GulNotebook *notebook, gint page, 
			        gint x, gint y, GaleonWindow *window)
{
	GaleonWindow *new_win;
	GaleonTab *tab;

	tab = get_tab_from_page_num (GTK_NOTEBOOK (notebook), page);
	new_win = galeon_window_new ();

	galeon_window_reparent_tab (new_win, window, tab);

	gtk_widget_show (GTK_WIDGET (new_win));
}

void
galeon_window_reparent_tab (GaleonWindow *window, GaleonWindow *source,
		            GaleonTab *tab)
{
	GtkWidget *src_page;

	src_page = GTK_WIDGET (galeon_tab_get_embed (tab));
	gul_notebook_move_page (GUL_NOTEBOOK (source->priv->notebook),
				GUL_NOTEBOOK (window->priv->notebook),
				src_page, 0);
	galeon_tab_set_window (tab, window);
}

void
galeon_window_set_tab_status (GaleonWindow	*window,
		              GaleonTab		*tab,
			      TabLoadStatus	 status)
{
	GdkColor *color;

	switch (status)
	{
	case TAB_LOAD_STARTED:
		color = &_galeon_window_loading_tab_color;
        	break;
	case TAB_LOAD_COMPLETED:
		color = &_galeon_window_new_tab_color;
		break;
	case TAB_LOAD_NONE:
	default:
		color = NULL;
		break;
	}

	gul_notebook_set_page_color (GUL_NOTEBOOK (window->priv->notebook),
			             GTK_WIDGET (galeon_tab_get_embed (tab)),
				     color);
}

void
galeon_window_set_tab_title (GaleonWindow	*window,
		             GaleonTab		*tab,
			     const char 	*title)
{
	gul_notebook_set_page_title (GUL_NOTEBOOK (window->priv->notebook),
				     GTK_WIDGET (galeon_tab_get_embed (tab)),
				     title);

	if (galeon_window_get_active_tab (window) == tab)
	{
		galeon_window_update_control (window, TitleControl);
	}
}
