/* entry.cc
 *
 * Copyright (C) 2000-2002 GConfmm Development Team
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gconfmm/entry.h"

namespace Gnome
{

namespace Conf
{

Entry::Entry()
{
  m_pEntry = 0;
}

Entry::Entry(const Glib::ustring& key, const Value& val)
{
  m_pEntry = gconf_entry_new(key.c_str(), const_cast<GConfValue*>(val.gobj()));
}

Entry::Entry(const GConfEntry* pEntry)
{
  m_pEntry = 0;

  //There is no gconf_entry_copy,
  //so we copy the fields manually:

  if(pEntry)
  {
    m_pEntry = gconf_entry_new(
                 gconf_entry_get_key(pEntry),
                 gconf_entry_get_value(pEntry) );

    gconf_entry_set_schema_name( m_pEntry, gconf_entry_get_schema_name(pEntry) );

    gconf_entry_set_is_default( m_pEntry, gconf_entry_get_is_default(pEntry) );
    gconf_entry_set_is_writable(m_pEntry, gconf_entry_get_is_writable(pEntry) );
  }
}

Entry::Entry(const Entry& Entry)
{
  m_pEntry = 0;

  operator=(Entry);
}

Entry::~Entry()
{
  clear();
}

Entry& Entry::operator=(const Entry& src)
{
  if(&src != this)
  {
    clear();

    //There is no gconf_entry_copy,
    //so we copy the fields manually:

    const GConfEntry* pEntry = src.gobj();
    if(pEntry)
    {
      m_pEntry = gconf_entry_new(
                   gconf_entry_get_key(pEntry),
                   gconf_entry_get_value(pEntry) );

      gconf_entry_set_schema_name( m_pEntry, gconf_entry_get_schema_name(pEntry) );

      gconf_entry_set_is_default( m_pEntry, gconf_entry_get_is_default(pEntry) );
      gconf_entry_set_is_writable( m_pEntry, gconf_entry_get_is_writable(pEntry) );
    }
  }

  return *this;
}

void Entry::clear()
{
  if(m_pEntry)
  {
    gconf_entry_free(m_pEntry);
    m_pEntry = 0;
  }
}

const GConfEntry* Entry::gobj() const
{
  return m_pEntry;
}

GConfEntry* Entry::gobj()
{
  return m_pEntry;
}

void Entry::set_value(const Value& val)
{
  gconf_entry_set_value(gobj(), const_cast<GConfValue*>(val.gobj()));
}

void Entry::set_schema_name(const Glib::ustring& name)
{
  gconf_entry_set_schema_name(gobj(), name.c_str());
}

void Entry::set_is_default(bool is_default = true)
{
  gconf_entry_set_is_default(gobj(), (gboolean)is_default);
}

void Entry::set_is_writable(bool is_writable = true)
{
  gconf_entry_set_is_writable(gobj(), (gboolean)is_writable);
}


Glib::ustring Entry::get_key() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_entry_get_key(gobj());
  if(pchVal)
    strResult = pchVal;

  return strResult;
}

Value Entry::get_value() const
{
  const GConfValue* pValue = gconf_entry_get_value(gobj());
  if(pValue)
  {
    return Value(pValue);
  }
  else
  {
    return Value();
  }
}

Glib::ustring Entry::get_schema_name() const
{
  Glib::ustring strResult;

  const gchar* pchVal = gconf_entry_get_schema_name(gobj());
  if(pchVal)
    strResult = pchVal;

  return strResult;
}

bool Entry::get_is_default() const
{
  return (bool)gconf_entry_get_is_default(gobj());
}

bool Entry::get_is_writable() const
{
  return (bool)gconf_entry_get_is_default(gobj());
}


} /* namespace Conf */
} /* namespace Gnome */
